/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author: Keith A. Shields (shields@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * IntervalFU.cc - A polymorophic functional unit for a dynamic interval plot
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/IntervalFU.C,v 1.22 1994/03/15 16:41:41 aydt Exp $
 */

#include "IntervalFU.h"
#include "IntervalWrapper.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "InputPort.h"
#include "SystemErrors.h"

#define Normalize(x)	( (((double) x) - minValue) / 		\
		  (maxValue - minValue + 1) )
#define Round(x)	(x >= 0.0 ? (int)(x + .5) : (int)(x - .5))


IntervalFU::IntervalFU()
{
        _setClassName( MY_CLASS );
	input = NULL;
	interval = NULL;
	dialog = NULL;
}

IntervalFU::~IntervalFU()
{
	delete input;
	delete interval;
	delete dialog;
}


void				/* virtual */ 
IntervalFU::configure()
{

#ifndef XtNminValue
#define XtNminValue "minValue"
#define XtNmaxValue "maxValue"
#endif

#ifndef XtNnumcolors
#define XtNnumColors "numColors"
#endif

	int fuColors, fuMin, fuMax;

 	if ( ! isConfigured ) {

	  	interval->getValue(XtNnumColors, (XtArgVal) &fuColors);
		interval->getValue(XtNminValue, (XtArgVal) &fuMin);
        	interval->getValue(XtNmaxValue, (XtArgVal) &fuMax);
		isConfigured = TRUE_;
	      }
        else {
	      fuColors = numColors;
	      fuMin = minValue;
	      fuMax = maxValue;
	    }

	FUParams params;

	params.addTextParam("Number of intervals", BaseFUParamEntry::Integer,
			      fuColors);
 	params.addTextParam("Minimum value", BaseFUParamEntry::Integer,
                              fuMin);
        params.addTextParam("Maximum value", BaseFUParamEntry::Integer,
                              fuMax);

	ParamConfig pc( Pablo::TopLevel(), params, getName() );
	pc.run();

	BaseFUParamEntry &colorEntry = params.getEntry("Number of intervals");
  	BaseFUParamEntry &minEntry = params.getEntry("Minimum value");
        BaseFUParamEntry &maxEntry = params.getEntry("Maximum value");

	if (colorEntry.valueIsValid()){
		fuColors = colorEntry.getValue().getInteger();
	} else {
	        warning( "Interval count was not valid\n" );
	}
	if (minEntry.valueIsValid()) {
              fuMin = minEntry.getValue().getInteger();
        } else {
                warning( "Minimum value was not valid\n" );
	}
        if (maxEntry.valueIsValid()) {
               fuMax = maxEntry.getValue().getInteger();
        } else {
                warning( "Maximum value was not valid\n" );
	}

	minValue = fuMin;
	maxValue = fuMax;
	numColors = fuColors;

	interval->setPerfWidgetColors(interval->getDisplay(),
				      interval->getScreen() );
}

FunctionalUnit *		/* virtual */
IntervalFU::copy()
{
	IntervalFU *copy = new IntervalFU();
	return copy;
}

void			/* virtual */ 
IntervalFU::init()
{
	if ( input != NULL ) {
	    delete input;
	}
	if ( interval != NULL ) {
	    delete interval;
	}
	if ( dialog != NULL ) {
	    delete dialog;
	}

	isConfigured = FALSE_;

	input = new InputPort( "Input" );
	input->addTraits( INTEGER, 1 );
	input->addTraits( FLOAT, 1 );
	input->addTraits( DOUBLE, 1 );
	_addInputPort( input );

	dialog = _getTitledFormDialog( "FUDialog", getName() );
	interval = new IntervalWrapper( dialog, NullArgs, "Contents" );

	dialog->manage();
	XtVaSetValues( dialog->getWidget(), XmNdefaultPosition, False, NULL );
}


Boolean_		/* virtual */
IntervalFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
        } else {
	    int x, y, width, height;
	    fscanf( fp, "%*[^\n]\n" );

            if ( fscanf( fp, "%d %d %d %d %d %d %d\n",
                           &minValue, &maxValue, &numColors,
			   &x, &y, &width, &height ) == 7 ) {
                /*
		 * We use the parameters loaded from the file
		 * to set other resources.
                 * Finally, initialize the static color table.
                 */
		interval->setPerfWidgetPosition( x, y, width, height );
        	interval->setPerfWidgetColors( interval->getDisplay(),
					       interval->getScreen() );
                isConfigured = TRUE_;
            } else {
                warning( "Unable to read configuration information from %s\n",
                          fileName.getValue() );
            }
            fclose( fp );
        }

	if ( ! isConfigured ) {
	    configure() ;
	}

        return isConfigured;
}

Boolean_ 
IntervalFU::ready()
{
	return isConfigured;
}

void				/* virtual */ 
IntervalFU::run(  Boolean_& errorFlag )
{
	Assert( input -> valueAvailable() );
	const Value *valueP = input->getValueP();

	if ( valueP->getTraits().getDimension() != 1 ) {
	    warning( "Interval needs a vector as input" );
	    errorFlag = TRUE_;
	    return;
	}

	Array *aInputData = valueP->operator Array*();
	const int *dimSizes = aInputData->getDimSizes();
	
	int *low, *high, *index;
	int nx = dimSizes[0];
	low = new int[nx];
	high = new int[nx];
	index = new int[nx];

	double sum = 0.0, partial_sum = 0.0;
	int valueRange = maxValue - minValue;
	double *dataSet = new double [nx];

	int i;
	for (i = 0; i < nx; i++) {
           dataSet[i] = warnings.fuDataRangeWarningCheck( this,
		   (double)aInputData->getCellValue(i),minValue,maxValue);
	   sum += dataSet[i];
        }
	low[0] = minValue;
	if (sum == 0.0)
	   high[0] = minValue;
	else
	   partial_sum += valueRange * (dataSet[0]/sum);
	   high[0] = low[0] + Round(partial_sum);
	index[0] = 1;

	for (i = 1; i < nx; i++) {
           if (sum == 0.0) {
              low[i] = (int)minValue;
              high[i] = (int)minValue;
           } else { 
	      low[i] = high[i-1];
	      if (i == nx-1) {
		  high[i] = (int)maxValue;
	      } else {
		  partial_sum += valueRange * (dataSet[i]/sum);
	          high[i] = Round(partial_sum);
	      }
           }
	   index[i] = i+1; 
	}
	interval->setPerfWidgetValues(low, high, index, nx);

	delete[] low;
	delete[] high;
	delete[] index;
}


Boolean_		/* virtual */
IntervalFU::saveConfigToFile( const CString& fileName ) const
{
        int     x, y, width, height;
	interval->getPerfWidgetPosition( &x, &y, &width, &height );
	
        Boolean_ result;
        FILE *fp = fopen( fileName.getValue(), "w" );

        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            result = FAILURE_;
        } else {
            fprintf( fp, "# Min Max Colors X Y Width Height\n" );
            fprintf( fp, "%d %d %d %d %d %d %d\n", 
			 minValue, maxValue, numColors,
			 x, y, width, height );
            fclose( fp );
            result = SUCCESS_;
        }
        return result;
}
	
/*
 *      Initialize the static data.   Only executed once.
 */
const char *const IntervalFU::MY_CLASS = "IntervalFU";

