/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * StructureDescriptorResolver.cc - Methods for the StructureDescriptorResolver
 * class.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Dictionaries/RCS/StructureDescriptorResolver.C,v 1.14 1994/03/01 21:47:31 aydt Exp $
 */

#include "Pablo.h"
#include "StructureDescriptor.h"
#include "StructureDescriptorResolver.h"
#include "StructureDictionary.h"
#include "StructureDictionaryInquirer.h"
#include "TagMappingDictionary.h"
#include "TagMappingDictionaryInquirer.h"

StructureDescriptorResolver::StructureDescriptorResolver()
{
	_setClassName( MY_CLASS ) ;

	tagMappingDictInquirer = new TagMappingDictionaryInquirer( 
						*Pablo::TagMappingDict() );
	structureDictInquirer = new StructureDictionaryInquirer( 
						*Pablo::StructureDict() );
}

StructureDescriptorResolver::~StructureDescriptorResolver()
{
	delete structureDictInquirer;
	delete tagMappingDictInquirer;
}

Boolean_
StructureDescriptorResolver::_sameSource( const int theTag, const int srcId )
{
	int theSource;

	tagMappingDictInquirer->resetQuery();

	while( theSource = tagMappingDictInquirer->nextMatch( theTag ) ) {
		if ( theSource == srcId ) {
			return( TRUE_ );
		}
	}

	return( FALSE_ );
}

Boolean_
StructureDescriptorResolver::_treatSameAs( const int theTag )
{
	int theSourceId;
	int theSourceTag;
	char ANS[64];

	cout << "\n<Message> An identical StructureDescriptor is already "
	     << "in the system."
	     << "\n<Message> It's sourceId/sourceTag pairs are shown below.\n";

	tagMappingDictInquirer->resetQuery();

	while( theSourceId = tagMappingDictInquirer->nextMatch( theTag ) ) {
		theSourceTag = tagMappingDictInquirer->theSourceTag();
		cout << "<Message> Id: " << theSourceId 
		     << "\tTag: " << theSourceTag << NL;
	}
	cout << "<Question> "
	     << "Treat this as an identical StructureDescriptor? (y/n) ";
	cin >> ANS;

	if ( ANS[0] == 'y' || ANS[0] == 'Y' ) {
		return( TRUE_ );
	} else {
		return( FALSE_ );
	}
}

int
StructureDescriptorResolver::determineTag( const int srcId, 
			                   const int srcTag, 
				           StructureDescriptor& structDescr )
{
	int theTag;
	StructureDescriptor theStruct;

	/* 
	 * First, we look for an exact match to the (srcId, srcTag) pair in
	 * the TagMappingDictionary.  If one is found, we verify that the
	 * structDict here is identical to the one in the Global Structure
	 * Dictionary.  If it is, we return the matching system tag, if it
	 * isn't we return FAILURE_.
	 */
	theTag = Pablo::TagMappingDict()->fetch( srcId, srcTag );
	if ( theTag ) {
	    theStruct = Pablo::StructureDict()->fetch( theTag );
	    if ( theStruct.isMatchFor( structDescr, TRUE_, TRUE_, TRUE_ ) ){
		return( theTag );
	    } else {
		return( FAILURE_ );
	    }
	} 

	/* 
	 * We haven't seen this (srcId, srgTag) pair before.  We'll look
	 * for a match on Name in the StructureDictionary.  If a match
	 * was found, verify that it is for an identical structure.
	 * If so, add the new mapping to dictionary.  	If not, error!
	 */
	structureDictInquirer->resetQuery();
	theTag = structureDictInquirer->nextMatch( structDescr.getName() );
	if ( theTag ) {
	    theStruct = Pablo::StructureDict()->fetch( theTag );
 	    if ( theStruct.isMatchFor( structDescr, TRUE_, TRUE_, TRUE_ ) ){
	        Pablo::TagMappingDict()->insert( srcId, srcTag, theTag );
		return( theTag );
	    } else {
	        return( FAILURE_ );
	    }
	}

	/*
	 * There were no matches to our Descriptor in the StructureDictionary. 
	 * So, add it to the StructureDictionary as a new entry and add the 
	 * mapping to the TagMappingDictionary.  
	 */
	theTag = Pablo::StructureDict()->insert( structDescr );
	Pablo::TagMappingDict()->insert( srcId, srcTag, theTag );
	return( theTag );
}


int
StructureDescriptorResolver::resolve( const int srcId, 
			              const int srcTag, 
				      StructureDescriptor& structDescr,
				      ResolverQuery queryUser ) 
{
	int theTag;
	StructureDescriptor theStruct;

	/* 
	 * First, we look for an exact match to the (srcId, srcTag) pair in
	 * the TagMappingDictionary.  If one is found, we verify that the
	 * structDict here is identical to the one in the Global Structure
	 * Dictionary.  If it is, we return the matching system tag, if it
	 * isn't  we return FAILURE_.
	 */
	if ( ( theTag = Pablo::TagMappingDict()->fetch( srcId, srcTag ) ) ){
		theStruct = Pablo::StructureDict()->fetch( theTag );
		if ( theStruct.isMatchFor( structDescr, TRUE_, TRUE_, TRUE_ ) ){
			return( theTag );
		} else {
			return( FAILURE_ );
		}
	} 

	/* 
	 * We haven't seen this (srcId, srgTag) pair before.  So, unless
	 * queryUser is set to RespondDifferent, we'll look for a match 
	 * in the StructureDictionary.  For now, we look only for exact 
	 * matches to the StructureDescriptor.  Rule out those entries 
	 * that came from the same Source, even if they are identical.  
	 * If queryUser is set to RespondSame, the first such match we see
	 * will be considered the same as the current StructureDescriptor
	 * and an entry will be made for it in the TagMappingDictionary.
	 * If queryUser is set to QuestionUser, we ask to see which, if any, 
	 * matches from others sources should be considered the same as 
	 * the current StructureDescriptor. If one selected, add the new 
	 * mapping to the TagMappingDictionary.
	 */

	if ( queryUser != RespondDifferent) {
	    structureDictInquirer->resetQuery();
	    while( theTag = structureDictInquirer->nextMatch( structDescr, 
						      TRUE_, TRUE_, TRUE_ ) ) {
						
		if ( ! _sameSource( theTag, srcId ) ) {
		    if ( queryUser == RespondSame ) {
		       Pablo::TagMappingDict()->insert( srcId, srcTag, theTag );
		       return( theTag );
		    } else {			// queryUser = QuestionUser
		       if (  _treatSameAs( theTag ) )  {
			    Pablo::TagMappingDict()->insert( srcId, srcTag, 
							            theTag );
			    return( theTag );
		       }
		    }

		}
	    }
	}

	/*
	 * Either there were no matches in the StructureDictionary or of
	 * those found, the user elected not to treat this as "identical"
	 * to any of them.  So, add it to the StructureDictionary as a new
	 * entry and add the mapping to the TagMappingDictionary.  
	 */

	theTag = Pablo::StructureDict()->insert( structDescr );
	Pablo::TagMappingDict()->insert( srcId, srcTag, theTag );
	return( theTag );
}

/*
 * 	Initialize the static data.   Only executed once.
 */
const char *const StructureDescriptorResolver::MY_CLASS = 
						"StructureDescriptorResolver";

