/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * Attributes.cc -  methods for the Attributes class.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Base/RCS/Attributes.C,v 1.15 1994/03/15 19:47:29 aydt Exp $
 */

#include <string.h>

#include "Attributes.h"
#include "DataCharacteristics.h"
#include "GetPut.h"

static char tmpBuffer[4096];	// used to get/put if conversion needed

Attributes::Attributes()
{
	headEntry = NULL;
	tailEntry = NULL;
}

Attributes::Attributes( const Attributes& attributes )
{
	AttributeEntry *origEntry = attributes.headEntry; 

	if ( origEntry == NULL ) {		// an empty list

	   headEntry = NULL;
	   tailEntry = NULL;

	} else {				// a non-empty list

	   AttributeEntry *newEntry = new AttributeEntry;	

	   headEntry = newEntry;			
	   newEntry->key = origEntry->key;
	   newEntry->value = origEntry->value;

	   while ( origEntry->nextEntry != NULL ) {
	      origEntry = origEntry->nextEntry;
	      newEntry->nextEntry = new AttributeEntry;

	      newEntry = newEntry->nextEntry;
	      newEntry->key = origEntry->key;
	      newEntry->value = origEntry->value;
	   }

	   tailEntry = newEntry;

	}
}

Attributes::~Attributes()
{
	AttributeEntry *currentEntry = headEntry;
	AttributeEntry *followingEntry;

	while ( currentEntry != NULL ) {
	   followingEntry = currentEntry->nextEntry;
	   delete currentEntry;
	   currentEntry = followingEntry;
	}
}

/*
 *  Parse binary data into Attributes.
 *      <attribute count>{<attribute name><attribute value>}*
 *      attribute count ::= int
 *      attribute name  ::= string
 *      attribute value ::= string
 */
int
Attributes::bitsToObj( const char *bufPtr )
{
	clearEntries();

        CString theKey, theValue;
        int attrCount;
        int currentOffset;

        currentOffset = functionGet( bufPtr, &attrCount );

        while( attrCount > 0 ) {
                theKey = (char *)(bufPtr + currentOffset);
                currentOffset = currentOffset + theKey.length() + 1;
                theValue = (char *)(bufPtr + currentOffset);
                currentOffset = currentOffset + theValue.length() + 1;
                insert( theKey, theValue );
                attrCount--;
        }

        return( currentOffset );
}

int
Attributes::bytesNeeded( ) const
{
        int bytesRequired = sizeof( int );	// Attribute Count

	AttributeEntry *currentEntry = headEntry;
	while ( currentEntry != NULL ) {
		bytesRequired += currentEntry->key.length() + 1;
		bytesRequired += currentEntry->value.length() + 1;
		currentEntry = currentEntry->nextEntry;
	}

        return( bytesRequired );
}

int
Attributes::cbitsToObj( const char *bufPtr, const DataCharacteristics *cnvInfo )
{
	clearEntries();

        CString theKey, theValue;
        int attrCount;
        int currentOffset;
	char *tmpPtr;

        currentOffset = cnvInfo->functionGet( bufPtr, &attrCount );

        while( attrCount > 0 ) {

	    // First read the characters that make up the Key 
	    tmpPtr = tmpBuffer;
	    do {
		currentOffset += cnvInfo->functionGet( bufPtr+currentOffset,
						       tmpPtr );
	    } while ( *tmpPtr++ != '\0' );

            theKey = tmpBuffer;

	    // Next read the characters that make up the Value 
	    tmpPtr = tmpBuffer;
	    do {
		currentOffset += cnvInfo->functionGet( bufPtr+currentOffset,
						       tmpPtr );
	    } while ( *tmpPtr++ != '\0' );

            theValue = tmpBuffer;
            insert( theKey, theValue );

            attrCount--;
        }

        return( currentOffset );
}

int
Attributes::cbytesNeeded( const DataCharacteristics *cnvInfo ) const
{
        int bytesRequired = cnvInfo->intSize();	 // Attribute Count

	int charsInEntry;

	AttributeEntry *currentEntry = headEntry;
	while ( currentEntry != NULL ) {
	    charsInEntry = currentEntry->key.length() + 
			   currentEntry->value.length() + 2;      // count 0"s

	    bytesRequired += charsInEntry * cnvInfo->getAtomSize( CHARACTER );
	    currentEntry = currentEntry->nextEntry;
	}

        return( bytesRequired );
}

void
Attributes::clearEntries()
{
	AttributeEntry *currentEntry = headEntry;
	AttributeEntry *followingEntry;

	while( currentEntry != NULL ) {
		followingEntry = currentEntry->nextEntry;
		delete currentEntry;
		currentEntry = followingEntry;
	}
	headEntry = tailEntry = NULL;
}

Boolean_ 
Attributes::contains( const CString& key ) const
{
	AttributeEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( key == currentEntry->key )
	      return( TRUE_ );
	   else
	      currentEntry = currentEntry->nextEntry;
	}

	return( FALSE_ );
}

int
Attributes::entryCount() const
{
	int count = 0;
	AttributeEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
		count++;
		currentEntry = currentEntry->nextEntry;
	}

	return( count );
}
	

const CString& 
Attributes::fetch( const CString& key ) const
{
	AttributeEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( key == currentEntry->key )
	      return( currentEntry->value );
	   else
	      currentEntry = currentEntry->nextEntry;
	}

	return( CString::NOMATCH );			// Not in list
}

Boolean_
Attributes::insert( const CString& key, const CString& value )
{
	if ( contains( key ) )			// Verify it isn"t already there
	   return( FAILURE_ );

	AttributeEntry *newEntry = new AttributeEntry;
	newEntry->key = key;
	newEntry->value = value;

	if ( headEntry == NULL ) {
	   headEntry = newEntry;
	   tailEntry = newEntry;
	} else {
	   tailEntry->nextEntry = newEntry;
	   tailEntry = newEntry;
	}
	return( SUCCESS_ );
}

/*
 * Parse Attributes into binary data
 */
int
Attributes::objToBits( char *const bufPtr, int bufLen ) const
{
	if ( bytesNeeded() > bufLen ) {
	    return( -1 );
	} 

        int attrCount = entryCount();
        int bytesUsed = functionPut( bufPtr, &attrCount );

	AttributeEntry *currentEntry = headEntry;
	while ( currentEntry != NULL ) {
           strcpy( bufPtr+bytesUsed, currentEntry->key.getValue() );
           bytesUsed += currentEntry->key.length() + 1;

           strcpy( bufPtr+bytesUsed, currentEntry->value.getValue() );
           bytesUsed += currentEntry->value.length() + 1;

 	   currentEntry = currentEntry->nextEntry;
        }
        return( bytesUsed );
}

int
Attributes::objToCbits( char *const bufPtr, int bufLen,
			const DataCharacteristics *cnvInfo ) const
{
	if ( cbytesNeeded( cnvInfo ) > bufLen ) {
	    return( -1 );
	} 

        int attrCount = entryCount();
        int bytesUsed = cnvInfo->functionPut( bufPtr, &attrCount );

	char *tmpPtr;

	AttributeEntry *currentEntry = headEntry;
	while ( currentEntry != NULL ) {

	   strcpy( tmpBuffer, currentEntry->key.getValue() );
	   tmpPtr = tmpBuffer;
	   do {
	       bytesUsed += cnvInfo->functionPut( bufPtr+bytesUsed, tmpPtr );
	   } while ( *tmpPtr++ != '\0' );

	   strcpy( tmpBuffer, currentEntry->value.getValue() );
	   tmpPtr = tmpBuffer;
	   do {
	       bytesUsed += cnvInfo->functionPut( bufPtr+bytesUsed, tmpPtr );
	   } while ( *tmpPtr++ != '\0' );

 	   currentEntry = currentEntry->nextEntry;
        }
        return( bytesUsed );
}

void 
Attributes::remove( const CString& key )
{
	if ( headEntry != NULL ) {		  // Check for empty list
	   AttributeEntry *currentEntry = headEntry;

	   if ( key == currentEntry->key ) {	  // Remove first entry
	      headEntry = currentEntry->nextEntry;
	      if ( tailEntry == currentEntry ) {  // We"re removing only entry
		 tailEntry = NULL;
	      }
	      delete currentEntry;

	   } else {
	      AttributeEntry *followingEntry = currentEntry->nextEntry;
	      while ( followingEntry != NULL ) {

	         if ( key == followingEntry->key ) {
		    currentEntry->nextEntry = followingEntry->nextEntry;
		    if ( tailEntry == followingEntry ) {
		       tailEntry = currentEntry;
		    }
	 	    delete followingEntry;
		    break;
	         } 
	      currentEntry = followingEntry;
	      followingEntry = currentEntry->nextEntry;
	      }				
	   }			
	}		
}

const CString& 
Attributes::operator[]( const CString& key ) const
{
	return( fetch(key) );
}

Attributes& 
Attributes::operator=( const Attributes& attributes )
{
	/*
	 * First, release space used by entries currently in the list
	 */
	AttributeEntry *currentEntry = headEntry;
	AttributeEntry *followingEntry;

	while ( currentEntry != NULL ) {
	   followingEntry = currentEntry->nextEntry;
	   delete currentEntry;
	   currentEntry = followingEntry;
	}

	/* 
	 * Second, copy the entries from the passed attribitues list into
	 * this Attributes list.
	 */

	AttributeEntry *origEntry = attributes.headEntry; 

	if ( origEntry == NULL ) {		// an empty list

	   headEntry = NULL;
	   tailEntry = NULL;

	} else {				// a non-empty list

	   AttributeEntry *newEntry = new AttributeEntry;	

	   headEntry = newEntry;			
	   newEntry->key = origEntry->key;
	   newEntry->value = origEntry->value;

	   while ( origEntry->nextEntry != NULL ) {
	      origEntry = origEntry->nextEntry;
	      newEntry->nextEntry = new AttributeEntry;

	      newEntry = newEntry->nextEntry;
	      newEntry->key = origEntry->key;
	      newEntry->value = origEntry->value;
	   }

	   tailEntry = newEntry;

	}

	return *this;
}

Boolean_
Attributes::operator==( const Attributes& attributes ) const
{
	AttributeEntry *entryA = headEntry;
	AttributeEntry *entryB = attributes.headEntry;

	while( entryA != NULL ) {		// A list still has entries

	   if (  entryB != NULL ) {		// B list still has entries
		/*
	 	 *   Test to see if the keys and values of the 2 entries
		 *   are equivalent. 
		 */	
	    	if ( ( entryA->key != entryB->key ) ||
	             ( entryA->value != entryB->value ) ) {
	    		return( FALSE_ );	// Entries differ
		}
	    } else {
		return( FALSE_ );		// B list ran out before A
	    }
	    
	    entryA = entryA->nextEntry;
	    entryB = entryB->nextEntry;
	}

	if ( entryB != NULL ) {			// A list ran out before B
	    return( FALSE_ );
	} else {
	    return( TRUE_ );		// A Match!
	}
}

Boolean_
Attributes::operator!=( const Attributes& attributes ) const
{
	return( CnvToBoolean_( !( *this == attributes ) ) );
}

void 
Attributes::printOn( ostream& os ) const
{
	os << "<Attributes begin>\n";
	AttributeEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   os << "Attr Key:   " << currentEntry->key << NL;
	   os << "     Value: " << currentEntry->value << NL;

	   currentEntry = currentEntry->nextEntry;
	}
	os << "<Attributes end>\n";
}

