/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Robert Olson (olson@cs.uiuc.edu)
 * Contributing Author: Ruth A. Aydt (aydt@.cs.uiuc.edu)
 * Contributing Author: Bill Whitehouse (whitehou@.cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Pipes/RCS/InputFileStreamPipe.C,v 1.20 1994/02/25 04:34:11 aydt Exp $
 */

#ifdef __GNUG__
#include <std.h>
#else
#include <fcntl.h>
#include <stdlib.h>
#include <sysent.h>
#include <string.h>
#include <unistd.h>
#endif
#include <sys/stat.h>

#include "InputFileStreamPipe.h"

#include "AsciiPipeReader.h"
#include "BinaryPipeReader.h"
#include "ConversionPipeReader.h"
#include "DataCharacteristics.h"
#include "SystemErrors.h"

InputFileStreamPipe::InputFileStreamPipe( const char *name, 
					  int bytesInReadPool ) 
{	
	_setClassName( MY_CLASS );
	_setObjectName( name );

	fileDescriptor = -1;
	myFileType = INVALID_TYPE;
	fileCharacteristics = NULL;

	endOfFileReached = TRUE_;
	nextByteInFile = 0;

	peekBuffer = NULL;
	peekBufferSize = 0;
	peekBufferOffset = 0;
	peekLength = 0;

	theReadPool = NULL;
	readPoolSize = 0;
	readPoolOffset = 0;
	bytesReadIntoPool = 0;

	/*
	 * File must exist, must not be a directory, and must be accessible
	 * for reading.
	 */
	struct stat statbuf;

	if ( stat((char *)name, &statbuf) != 0 ) {	
	    error( "Can't open file %s: %s", name, errorString() );
	} else {
	    if ( (statbuf.st_mode & S_IFDIR) ) {	// A directory
	        error( "Selected Input File %s is a directory.", name );
	    } else {					// Not a directory
		if ( ( fileDescriptor = open( name, O_RDONLY ) ) == -1 ) {
	   	    error( "Can't open file %s: %s", name, errorString() );
	        }
	    }
	}

   	/*
         * If we were able to open the file,
         *   - Scan the beginning of the file determine the type of the file.
         *   - If it isn't an SDDF file, reset pointer to the head of the file.
         *   - If this is a BINARY file, create DataCharacteristics instance to 
         *     hold the characteristics.
	 *   - If we are able to initialize the characteristics information, 
         *     then determine if the charactistics of the input file match
	 *     those of the native machine.
	 *   - If we are unable to initialize the characteristics information,
	 *     then print an error message, close file, reset fileDescriptor 
	 *     to -1 and set type back to INVALID;
         */
        if ( fileDescriptor != -1 ) {
            char buf[5];

            int bytesRead = read( fileDescriptor, buf, 5 );
            if ( bytesRead == 5 ) {
                if ( strncmp( buf, "SDDFA", 5 ) == 0 ) {
                    myFileType = ASCII_SDDF;
                } else if ( strncmp( buf, "SDDFB", 5 ) == 0 ) {
                    myFileType = BINARY_SDDF;
                } 
            } 

	    if ( myFileType == INVALID_TYPE ) {
		myFileType = NOT_SDDF;
	        lseek( fileDescriptor, 0, SEEK_SET );
	    } else if ( myFileType == BINARY_SDDF ) {
                fileCharacteristics = new DataCharacteristics();

		if ( fileCharacteristics->initializeFromFile( 
						this, fileDescriptor ) ) {
						
		    if ( fileCharacteristics->notNative() ) {
			myFileType = CONVERT_BINARY_SDDF;
		    }
		} else {			// initialization failed
		    error( "Unable to set data characteristics for %s",
			    name );
		    delete fileCharacteristics;
		    close( fileDescriptor );
		    fileDescriptor = -1;
		    myFileType = INVALID_TYPE;
		}
            }
        }

	/*
         * If we get this far and fileDescriptor is not -1, then we really
         * are ready to process the file.  Create the read pool and initialize
         * it.
         */

	if ( fileDescriptor != -1 ) {
	    nextByteInFile = lseek( fileDescriptor, 0, SEEK_CUR );
            readPoolSize = bytesInReadPool;
            theReadPool = new char[ readPoolSize ];
            bytesReadIntoPool = read( fileDescriptor, theReadPool, 
						      readPoolSize );
	    if ( bytesReadIntoPool > 0 ) {
	        endOfFileReached = FALSE_;
	    }
	}

}

InputFileStreamPipe::~InputFileStreamPipe()
{

	if ( peekBufferSize > 0 ) {
	    delete[] peekBuffer;
	}

	if ( theReadPool != NULL ) {
	    delete[] theReadPool;
	}

	if ( fileCharacteristics != NULL ) {
	    delete fileCharacteristics;
	}

	if ( fileDescriptor != -1 ) {
	    close( fileDescriptor );
	}
}

int
InputFileStreamPipe::_readPool( char *theBuffer, int amountToRead )
{
	/*
	 * Shouldn't get here with theReadPool == NULL.  But, we don't
	 * check for this error because it adds a lot of overhead. 
	 * The user should call ::successfulOpen before trying to access 
	 * the data.
	 */

	int bytesRead = 0;	    
	int poolBytesUnread = bytesReadIntoPool - readPoolOffset;

	while( amountToRead > 0 ) {
	    if ( amountToRead <= poolBytesUnread ) {
		// We can read all we need from the pool
	    	memcpy( &theBuffer[bytesRead], &theReadPool[readPoolOffset], 
						     amountToRead );
		readPoolOffset += amountToRead;
		bytesRead += amountToRead;
		amountToRead = 0;
	    } else {
		// Only part of what we want is in the pool - fetch that part
		memcpy( &theBuffer[bytesRead], &theReadPool[readPoolOffset], 
						     poolBytesUnread );
		bytesRead += poolBytesUnread;
		amountToRead -= poolBytesUnread;

		// We"ve now consumed all the data in the pool, if the
		// last file read got a full buffer, then read again.  If not,
		// set amountToRead = 0 to force exit from while() statement.
		if ( bytesReadIntoPool == readPoolSize ) {
		    bytesReadIntoPool = read( fileDescriptor, theReadPool, 
							    readPoolSize );
		    readPoolOffset = 0;
		    poolBytesUnread = bytesReadIntoPool;
		} else {
		    amountToRead = 0;
		}
	    }
	}

	return bytesRead;
}

int 		/* virtual */
InputFileStreamPipe::bytesFree() const
{
	return 0;
}

int 		/* virtual */
InputFileStreamPipe::bytesReady() const
{
	return 10000000;
}

PipeReader *	/* virtual */
InputFileStreamPipe::createPipeReader() const
{
	PipeReader *reader = NULL;

	if ( fileDescriptor == -1 ) {
	    error( "Input file open failed; no Reader can be created." );
	} else if ( myFileType == ASCII_SDDF ) {
            reader = new AsciiPipeReader( (StreamPipe *)this );
        } else if ( myFileType == BINARY_SDDF ) {
            reader = new BinaryPipeReader( (StreamPipe *)this );
        } else if ( myFileType == CONVERT_BINARY_SDDF ) {
            reader = new ConversionPipeReader( (StreamPipe *)this );
        } else if ( myFileType == NOT_SDDF ) {
            error( "Reader cannot be created for non-SDDF file." );
        } else {					// Shouldn"t happen!
	    error( "File had unknown type - no Reader created." );
	}

	if ( reader != NULL ) {
	    if ( ! reader->successfulConnection() ) {
		// We really don"t expect to get here, but just in case!
		delete reader;
		reader = NULL;
	    }
	}

	return( reader );
}

int 		/* virtual */
InputFileStreamPipe::get( char *data, int length )
{

	if ( length <= 0 ) {
	    return length;
	}

	/* 
	 * We expect to advance in the file by length bytes, if something
	 * goes wrong, we adjust later.
	 */
	nextByteInFile += length;

	/*
	 * Here, all we need is in the peek buffer
	 */
	if ( length <= peekLength ) {		
	    memcpy( data, &peekBuffer[peekBufferOffset], length );
	    peekBufferOffset += length;
	    peekLength -= length;
	    return length;
	}

	/*
	 * We know we must read at least some data from the file.
	 * If there is anything in the peek buffer read that first,
	 * then read directly from file to get the rest of the data.
	 */
	int  lengthToRead = length;

	if ( peekLength > 0  ) {
	    memcpy( data, &peekBuffer[peekBufferOffset], peekLength );
	    lengthToRead -= peekLength;
	    data += peekLength;

	    peekLength = 0;
	}

	lengthToRead -= _readPool( data, lengthToRead );

	if ( lengthToRead > 0 )  {		// Didn"t find all we wanted!
	    endOfFileReached = TRUE_;
	    nextByteInFile -= lengthToRead;	// Adjust byte offset
	}

	return ( length - lengthToRead );
}

off_t		/* virtual */
InputFileStreamPipe::getFileSize( )
{
	off_t currentOffset = lseek( fileDescriptor, 0, SEEK_CUR );
	if ( currentOffset == -1 ) {
	    abort( "lseek to find current offset failed: %s", 
							errorString() );
	}

	off_t endOffset = lseek( fileDescriptor, 0, SEEK_END );
	if ( endOffset == -1 ) {
	    abort( "lseek to find end offset failed: %s", errorString() );
	}

	if ( lseek( fileDescriptor, currentOffset, SEEK_SET )  == -1 ) {
	    abort( "lseek to reposition at %d failed: %s", currentOffset,
							errorString() );
	}

	return endOffset;
}

int 		/* virtual */
InputFileStreamPipe::getSkippingPeek( char *data, int length )
{
	nextByteInFile += peekLength;	// Advance past "peeked" bytes
	peekLength = 0;			// Flag that peek buffer "empty now"

	if ( length <= 0 ) {
	    return length;
	}

	nextByteInFile += length;	// Expect to get it all
	int  lengthToRead = length;

	lengthToRead -= _readPool( data, lengthToRead );

	if ( lengthToRead > 0 )  {		// Didn"t find all we wanted!
	    endOfFileReached = TRUE_;
	    nextByteInFile -= lengthToRead;	// Adjust byte offset
	}

	return ( length - lengthToRead );
}


Boolean_	/* virtual */
InputFileStreamPipe::isEmpty() const
{
	/* Note:  It is possible that we are 'looking' at an empty pipe, but
	 * that endOfFileReached is not set.  It only gets set if we try to 
	 * read past the data available - not if we read only what is there.
	 * We need a "getc/ungetc" to make this really work.  The true test
	 * is to peek and see if you get the number expected. */

	return endOfFileReached;
}

Boolean_	/* virtual */
InputFileStreamPipe::isFull()  const
{
	return TRUE_;
}

int 		/* virtual */
InputFileStreamPipe::peek( char *data, int length )
{
	if ( length <= 0 ) {
	    return length;
	}

	// Notice that peeking doesn"t advance our current byte offset.

	if ( peekLength > 0 ) {
    	    warning( "Ack! Only one peek at a time, please." );
	    return 0;
	} else {
	    if ( peekBufferSize < length ) {
		if ( peekBufferSize > 0 ) {
		    delete[] peekBuffer;
		}
	        peekBufferSize = length;
	        peekBuffer = new char[peekBufferSize];
	    }
	    peekBufferOffset = 0;
	    peekLength = _readPool( peekBuffer, length );

	    if ( peekLength < length ) {
	    	endOfFileReached = TRUE_;
	    }
	    memcpy( data, peekBuffer, peekLength );
	    return peekLength;
	}
}

int 		/* virtual */
InputFileStreamPipe::put( const char *, int )
{
	abort( "Cannot put() to an InputFileStreamPipe" );
	return 0;		// should not get here
}

void		/* virtual */
InputFileStreamPipe::setByteOffset( off_t newByteOffset )
{
	/*
	 * To reset our position in the file we do several things...
	 * - Make sure endOfFileReached is accurate.
	 * - Reset our peek buffer to be empty.
	 * - Reset our read-ahead pool to have no valid data.
	 * - Reset the nextByteInFile
	 * - Seek to new position.  If this fails, abort.
	 */

	if ( newByteOffset < nextByteInFile ) {
	    endOfFileReached = FALSE_;
	}
	peekLength = 0;
	readPoolOffset = readPoolSize;
	bytesReadIntoPool = readPoolSize;
	nextByteInFile = newByteOffset;
	if ( ( lseek( fileDescriptor, nextByteInFile, SEEK_SET ) ) == -1 ) {
	    abort( "lseek to offset %d failed: %s", nextByteInFile, 
						    errorString() );
	}
}

void 		/* virtual */
InputFileStreamPipe::printOn( ostream & os ) const
{
	os << form( "InputFileStreamPipe %s: \n",
		    getObjectName() );
}

/*
 * Initialize the static data structures
 */
const char *const InputFileStreamPipe::MY_CLASS = "InputFileStreamPipe";

