/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */

#include "Assert.h"
#include "Defines.h"
#include "RecordDictionary.h"
#include "StructureDescriptor.h"

RecordDictionary::RecordDictionary()
{
	_setClassName( MY_CLASS );
	headEntry = NULL;
	tailEntry = NULL;
}

RecordDictionary::~RecordDictionary()
{
	RecordDictEntry *currentEntry = headEntry;
	RecordDictEntry *followingEntry;

	while ( currentEntry != NULL ) {
		followingEntry = currentEntry->nextEntry;
		delete currentEntry;
		currentEntry = followingEntry;
	}
}

Boolean_
RecordDictionary::contains( int dossierTag ) const
{
	RecordDictEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( dossierTag == currentEntry->dossier.getTag() ) {
	      return( TRUE_ );
	   } else {
	      currentEntry = currentEntry->nextEntry;
	   }
	}
	return FALSE_;
}

int
RecordDictionary::entryCount() const
{
	int count = 0;
	RecordDictEntry *currentEntry = headEntry;
	RecordDictEntry *followingEntry;

	while ( currentEntry != NULL ) {
		count++;
		followingEntry = currentEntry->nextEntry;
		currentEntry = followingEntry;
	}
	return count;
}

RecordDossier& 
RecordDictionary::fetch( int dossierTag ) const
{
	RecordDictEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( dossierTag == currentEntry->dossier.getTag() ) {
	      return( currentEntry->dossier );
	   } else {
	      currentEntry = currentEntry->nextEntry;
	   }
	}
	return noDossier;			// Not in list
}

Array *
RecordDictionary::getArrayP( int dossierTag, const CString& fieldName ) const
{
	RecordDictEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( dossierTag == currentEntry->dossier.getTag() ) {
	      return( currentEntry->dossier.getArrayP( fieldName ) );
	   } else {
	      currentEntry = currentEntry->nextEntry;
	   }
	}
	return NULL;                    	// Not in list
}

Array *
RecordDictionary::getArrayP( int dossierTag, int fieldID ) const
{
	RecordDictEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( dossierTag == currentEntry->dossier.getTag() ) {
	      return( currentEntry->dossier.getArrayP( fieldID ) );
	   } else{
	      currentEntry = currentEntry->nextEntry;
	   }
	}
	return NULL;                    	// Not in list
}

Value &
RecordDictionary::getValue( int dossierTag, const CString& fieldName ) const
{
	RecordDictEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( dossierTag == currentEntry->dossier.getTag() ) {
	      return( currentEntry->dossier.getValue( fieldName ) );
	   } else {
	      currentEntry = currentEntry->nextEntry;
	   }
	}
	return( Value::NOVALUE );                    // Not in list
}

Value &
RecordDictionary::getValue( int dossierTag, int fieldID ) const
{
	RecordDictEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( dossierTag == currentEntry->dossier.getTag() ) {
	      return( currentEntry->dossier.getValue( fieldID ) );
	   } else {
	      currentEntry = currentEntry->nextEntry;
	   }
	}
	return( Value::NOVALUE );                    // Not in list
}

Value *
RecordDictionary::getValueP( int dossierTag, const CString& fieldName ) const
{
	RecordDictEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( dossierTag == currentEntry->dossier.getTag() ) {
	      return( currentEntry->dossier.getValueP( fieldName ) );
	   } else {
	      currentEntry = currentEntry->nextEntry;
	   }
	}
	return NULL; 	                     // Not in list
}

Value *
RecordDictionary::getValueP( int dossierTag, int fieldID ) const
{
	RecordDictEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( dossierTag == currentEntry->dossier.getTag() ) {
	      return( currentEntry->dossier.getValueP( fieldID ) );
	   } else {
	      currentEntry = currentEntry->nextEntry;
	   }
	}
	return NULL;	                    // Not in list
}

Boolean_
RecordDictionary::insert( int dossierTag, 
			  const StructureDescriptor& structDescr )
{
	if ( contains( dossierTag) ) {	      // Verify it isn"t already there
	   return FAILURE_;
	}

	RecordDictEntry *newEntry = new RecordDictEntry( dossierTag, 
							 structDescr );

	if ( headEntry == NULL ) {
	   headEntry = newEntry;
	   tailEntry = newEntry;
	} else {
	   tailEntry->nextEntry = newEntry;
	   tailEntry = newEntry;
	}
	return SUCCESS_;
}

Boolean_
RecordDictionary::setValue( int dossierTag, const CString& fieldName, 
  			    const Value& fieldValue )
{
        RecordDictEntry *currentEntry = headEntry;

        while ( currentEntry != NULL ) {
           if ( dossierTag == currentEntry->dossier.getTag() ) {
              return( currentEntry->dossier.setValue( fieldName, fieldValue ) );
           } else {
              currentEntry = currentEntry->nextEntry;
	   }
        }
  	warning( "\n\tsetValue() failed - no record found with tag %d",
								dossierTag );
        return FAILURE_;                    // Not in list
}

Boolean_
RecordDictionary::setValue( int dossierTag, int fieldID, 
			    const Value& fieldValue )
{
        RecordDictEntry *currentEntry = headEntry;

        while ( currentEntry != NULL ) {
           if ( dossierTag == currentEntry->dossier.getTag() ) {
              return( currentEntry->dossier.setValue( fieldID, fieldValue ) );
           } else {
              currentEntry = currentEntry->nextEntry;
	   }
        }
  	warning( "\n\tsetValue() failed - no record found with tag %d",
								dossierTag );
        return FAILURE_;                    // Not in list
}

void 			/* virtual */
RecordDictionary::printOn( ostream& os ) const
{
	os << "<RecordDictionary begin> \n";
	RecordDictEntry *currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   os << currentEntry->dossier;
	   currentEntry = currentEntry->nextEntry;
	}
	os << "<RecordDictionary end> \n";
}

/*
 * Initialize the static data 
 */
const char *const RecordDictionary::MY_CLASS = "RecordDictionary";

/*
 * Definition of the static data if not done in a central place
 */
#ifdef CONSTRUCT_NONLOCAL_STATICS
	RecordDossier RecordDictionary::noDossier( CString::NOMATCH );
#endif
