/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

#include "Assert.h"
#include "SystemDepend.h"
#include "SDDFparam.h"
#include "TraceParam.h"
#include "Trace.h"

/*
 *  Internal.c:
 *	This file contains the internal, machine independent trace
 *	library routines.  These routines are invoked by the user
 *	accessible trace library to buffer and record data.
 */

/*	The following variables are shared among the routines of
 *	this file (Internal.c), Interface.c, and the system dependent
 *	code.
 */

TR_BUFFER	traceBuffer;		/* local trace buffer structure	    */

unsigned int	defaultBufferSize =	/* default trace buffer size	    */
			DEFAULTBUFFERSIZE;

unsigned int	defaultBufferType =	/* default trace buffer type	    */
			DEFAULTBUFFERTYPE;


					/* scatter table of chains	    */
					/* containing all event types	    */
TR_EVENT	*eventHashTable[EVENTHASHTABLESIZE];

unsigned int	defaultTraceLevel =	/* default event trace level	    */
			DEFAULTTRACELEVEL;

unsigned int	defaultEventFlags =	/* default event flags		    */
			DEFAULTEVENTFLAGS;

CLOCK		defaultLowWater =	/* default values for adaptive	    */
			LOWWATERMARK;
CLOCK		defaultHighWater =	/* tracing control parameters	    */
			HIGHWATERMARK;

int		(*defaultTraceFunction)() =
			(int (*)()) 0;	/* default event trace function	    */

					/* default trace record function    */
TR_RECORD	*(*defaultRecordFunction)() =
			externalEventRecord;

unsigned long	defaultCountFreq =	/* default count write frequency    */
			DEFAULTCOUNTFREQ;

USER_RECORD	*userRecordDescriptors =	/* list for user-supplied   */
			(USER_RECORD *) 0;	/* record descriptors	    */

TR_CLOCK	epoch;			/* time at the beginning of	    */
					/* instrumentation		    */

CLOCK		clockStopped =		/* when clock stopped in	    */
			NOSUCHTIME;	/* toggleClock()		    */

CLOCK		noSuchClock =		/* constant clock value		    */
			NOSUCHTIME;

CLOCK		zeroClock =		/* constant clock value		    */
			ZEROTIME;

CLOCK		clockPrecision =	/* clock granularity		    */
			CLOCKPRECISION;

int		traceLibraryMode =	/* library operational mode	    */
			MODE_SYNC;

int		libraryInitLevel =	/* library initialization level	    */
			UNINITIALIZED;

/*
 *	initLibrary:
 *	   Initialize the trace library up to the desiredLevel, if
 *	   the current initialization level is less.
 */

initLibrary( desiredLevel )

int	desiredLevel;
{
	/* Basic initialization section					    */

	if ( desiredLevel < INIT_BASIC )
		return;

	if ( libraryInitLevel < INIT_BASIC )
		basicLibraryInit();


	/* Full initialization section					    */

	if ( desiredLevel < INIT_FULL )
		return;

	if ( libraryInitLevel < INIT_FULL )
		fullLibraryInit();


	return;
}


/*
 *	basicLibraryInit:
 *	   Perform initialization of the trace library data structures.
 *	   Called only by initLibrary to bring the trace library
 *	   initialization level up to at least INIT_BASIC.
 */

basicLibraryInit()
{
	int	i;

	TRgetClock( &epoch );		/* record the time that tracing	    */
					/* began --- needed for later	    */
					/* computation of elapsed time	    */

	TRinit();			/* system-dependent initialize	    */

					/* initialize event hash table	    */
	for ( i = 0; i < EVENTHASHTABLESIZE; i++ )
		eventHashTable[ i ] = (TR_EVENT *) 0;

					/* allocate a buffer of the	    */
					/* default size			    */
	traceBuffer.bufferStart = TRgetBuffer( defaultBufferSize );

	if ( traceBuffer.bufferStart == (char *) 0 )
		TRfailure(
		    "initBuffers: unable to allocate trace buffer" );

	traceBuffer.bufferCurrent = traceBuffer.bufferStart;

	traceBuffer.bufferEnd = traceBuffer.bufferStart
				+ defaultBufferSize;

	/* The default instrumentation mode is complete data		    */
	/* recording (i.e., each time a buffer fills it is dumped	    */
	/* to secondary storage for later analysis			    */

	traceBuffer.bufferType = defaultBufferType;

	traceBuffer.bufferFlags = BUFFERFLAG_EMPTY;

	traceBuffer.bufferQueHead = (PUT_REQUEST *) 0;

	traceBuffer.bufferQueTail = (PUT_REQUEST *) 0;


	/* The libraryInitLevel must be increased here because the	    */
	/* calls to initEvent immediately following will themselves	    */
	/* call initLibrary( INIT_BASIC ).  Changing libraryInitLevel	    */
	/* here makes initLibrary reentrant.				    */

	libraryInitLevel = INIT_BASIC;


	/* Initialize all trace library internal events.  There are	    */
	/* several event types corresponding to events within the	    */
	/* trace library itself and are not accessible to the user.	    */
	/* See the initEvent code for special details of internal event	    */
	/* initialization.						    */

	initEvent( EVENTID_AGGREGATE );

	initEvent( EVENTID_BADCLOCK );

	initEvent( EVENTID_DUMPTIME );

#ifdef META_TRACING
	initEvent( EVENTID_CLOCKTIME );

	initEvent( EVENTID_TRACETIME );

	initEvent( EVENTID_COUNTTIME );

	initEvent( EVENTID_INTVLTIME );

	initEvent( EVENTID_DUMMY );
#endif /* META_TRACING */


	writeSDDFfileHeader();		/* write SDDF file header	    */

	writeRecordDescriptors();	/* write trace record descriptors   */
					/* to the trace buffer		    */
#ifdef DEBUG
	fprintf( debugFile, "basicLibraryInit:\tdone\n" );
	fflush( debugFile );
#endif /* DEBUG */

	return;
}



/*
 *	fullLibraryInit:
 *	   Perform initialization of the trace library internal events.
 *	   Called only by initLibrary to bring the trace library
 *	   initialization level up to INIT_FULL.
 */

fullLibraryInit()
{
	/* Ordinarily the meta-performance internal events are suppressed   */

#ifdef META_TRACING
	/* Record how much time is required for obtaining a clock value	    */

	startTimeEvent( EVENTID_CLOCKTIME );
	(void) getClock();
	endTimeEvent( EVENTID_CLOCKTIME );

	/* Record how much time is required for recording each class	    */
	/* of events:  full traces (no data), counts, and intervals.	    */

	startTimeEvent( EVENTID_TRACETIME );
	traceEvent( EVENTID_DUMMY, (char *) 0, 0 );
	endTimeEvent( EVENTID_TRACETIME );

	startTimeEvent( EVENTID_COUNTTIME );
	countEvent( EVENTID_DUMMY );
	endTimeEvent( EVENTID_COUNTTIME );

	startTimeEvent( EVENTID_INTVLTIME );
	startTimeEvent( EVENTID_DUMMY );
	endTimeEvent( EVENTID_DUMMY );
	endTimeEvent( EVENTID_INTVLTIME );
#endif /* META_TRACING */

	libraryInitLevel = INIT_FULL;

	return;
}



/*
 *	newTraceFile:
 *	   Called by setTraceFileName to set the name of the local
 *	   trace file before initialization.  Returns FAILURE if called
 *	   after trace library initialization, because the trace file
 *	   has already been opened then.
 */

newTraceFile( traceFileName )

char	*traceFileName;
{
	if ( libraryInitLevel > UNINITIALIZED )
		return FAILURE;

	/* At this point, the trace library has not been initialized.	    */
	/* DO NOT insert any debugging statements here, since they	    */
	/* can only be used after library initialization.		    */

	/* Trace file code is entirely within the domain of system-	    */
	/* dependent code.  Simply handoff the trace file name to the	    */
	/* appropriate pre-initialization system-dependent routine.	    */

	TRsetTraceFile( traceFileName );

	return SUCCESS;
}


#ifdef DEBUG

/*
 *	newDebugFile:
 *	   Called by setDebugFileName to set the name of the local
 *	   debug file before initialization.  Returns FAILURE if called
 *	   after trace library initialization, because the debug file
 *	   has already been opened then.
 */

newDebugFile( debugFileName )

char	*debugFileName;
{
	if ( libraryInitLevel > UNINITIALIZED )
		return FAILURE;

	/* At this point, the trace library has not been initialized.	    */
	/* DO NOT insert any debugging statements here, since they	    */
	/* can only be used after library initialization.		    */

	/* Debug file code is entirely within the domain of system-	    */
	/* dependent code.  Simply handoff the debug file name to the	    */
	/* appropriate pre-initialization system-dependent routine.	    */

	TRsetDebugFile( debugFileName );

	return SUCCESS;
}

#endif /* DEBUG */


/*
 *	newSocketAddress:
 *	   Called by setTraceSocketAddress to set the address of the
 *	   trace output socket before initialization.  Returns FAILURE
 *	   if called after trace library initialization, because the
 * 	   socket or trace output file has already been opened then.
 */

newSocketAddress( socketAddress, dgramPort, streamPort )

unsigned long	socketAddress;
int		dgramPort;
int		streamPort;
{
	if ( libraryInitLevel > UNINITIALIZED )
		return FAILURE;

	/* At this point, the trace library has not been initialized.	    */
	/* DO NOT insert any debugging statements here, since they	    */
	/* can only be used after library initialization.		    */

	/* Socket output is entirely within the domain of system-	    */
	/* dependent code.  Simply handoff the socket address to the	    */
	/* appropriate pre-initialization system-dependent routine.	    */

	TRsetSocketAddress( socketAddress, dgramPort, streamPort );

	return SUCCESS;
}



/*
 *	newUserRecord:
 *	   Called by setRecordDescriptor to set the pointer to a function
 *	   which writes out user-supplied record descriptors at library
 *	   initialization time.  Immediately invokes the pointed-to
 *	   function if after basic library initialization.  Returns
 *	   FAILURE if called after full library initialization, otherwise
 *	   returns SUCCESS.
 */

newUserRecord( recordFunction )

int	(*recordFunction)();
{
	USER_RECORD	*newRecordDescriptor;

	if ( libraryInitLevel == INIT_FULL )
		return FAILURE;

	if ( libraryInitLevel > UNINITIALIZED ) {

		(* recordFunction )();

		return SUCCESS;
	}

	/* At this point, the trace library has not been		    */
	/* initialized.  DO NOT insert any debugging statements		    */
	/* here, since they can only be used after library		    */
	/* initialization.						    */

	/* Allocate a list element for this user record descriptor	    */
	/* function and add it to the list.				    */

	newRecordDescriptor = (USER_RECORD *)
					TRgetBuffer( sizeof(USER_RECORD) );

	if ( newRecordDescriptor == (USER_RECORD *) 0 )
	    TRfailure(
		"newUserRecord: unable to allocate user record descriptor" );

	newRecordDescriptor->recordDescriptorFunction = recordFunction;
	newRecordDescriptor->nextUserRecord = userRecordDescriptors;

	userRecordDescriptors = newRecordDescriptor;

	return SUCCESS;
}


/*
 *	adaptiveTrace:
 *	   This routine adaptively adjusts the tracing level based on
 *	   the observed event rate.  If the rate is too high, event
 *	   tracing is disabled and counting is substituted.
 */

adaptiveTrace( eventPointer, currentTime )

TR_EVENT	*eventPointer;
CLOCK		currentTime;
{
	CLOCK		timeDelta;

	Assert( eventPointer != (TR_EVENT *) 0 );

	/* Verify that occurrences of the event type are presently	    */
	/* being recorded (as opposed to being ignored).		    */

	if ( eventPointer->eventHard == LEVEL_IGNORE )
		return LEVEL_IGNORE;

	/* If this is not the first occurrence of this event type,	    */
	/* the effective current (i.e soft) trace level may change.	    */

	if ( clockCompare( eventPointer->eventLast, noSuchClock ) != 0 ) {
					/* compute the time since this	    */
		timeDelta =		/* event last occurred		    */
			clockSubtract( currentTime, eventPointer->eventLast );

		/* If the event rate is low enough that the delta T	    */
		/* since the last event is greater than the high water	    */
		/* mark, we can raise the trace level, but not over the	    */
		/* hard trace level limit for this event type.		    */

		if (( eventPointer->eventSoft < eventPointer->eventHard ) &&
		    ( clockCompare( timeDelta, eventPointer->eventHighWater )
		    > 0 ))
			eventPointer->eventSoft++;

		/* If the event rate is high enough that the delta T	    */
		/* since the last event is less than the low water mark,    */
		/* we must reduce the trace level, but not to the point	    */
		/* that the event is completely and permanently ignored.    */

		if (( eventPointer->eventSoft > (LEVEL_IGNORE + 1) ) &&
		    ( clockCompare( timeDelta, eventPointer->eventLowWater )
		    < 0 ))
			eventPointer->eventSoft--;
	}

	return eventPointer->eventSoft;
}


/*
 *	eventHashFunction:
 *	   Active event descriptors are stored in short chains (linked
 *	   lists), the head pointers for which are kept in a hash table.
 *	   This function accepts an event type as an argument and returns
 *	   an index into this hash table.  The return value must therefore
 *	   be between 0 and EVENTHASHTABLESIZE - 1, inclusive.  Note that
 *	   this function operates regardless of whether or not eventType
 *	   is a previously recognized (therefore initialized) event type,
 *	   the return value indicates only where in the event hash table
 *	   the given eventType would appear, if initialized.
 */

eventHashFunction( eventType )

int	eventType;
{
	int	hashIndex;

					/* simplest hash function	    */
	hashIndex = eventType % EVENTHASHTABLESIZE;

	while ( hashIndex < 0 )
		hashIndex += EVENTHASHTABLESIZE;

	return hashIndex;
}


/*
 *	initEvent:
 *	   Initialize the event type specified by eventType.  This
 *	   routine is called when the application program has not specified
 *	   event parameters.  By default, event tracing is enabled, and
 *	   the widest possible range of event rates are supported with
 *	   adaptive tracing.  Return value is the pointer to the event
 *	   descriptor for this event type.
 */

TR_EVENT *
initEvent( eventType )

int	eventType;
{
	TR_EVENT	*eventPointer;
	TR_EVENT	**eventChainHead;

	TRACELIB_INIT( INIT_BASIC );	/* initialize trace library	    */

	eventPointer = (TR_EVENT *) TRgetBuffer( sizeof(TR_EVENT) );

	if ( eventPointer == (TR_EVENT *) 0 )
		TRfailure(
		    "initEvent: unable to allocate event descriptor" );

	eventPointer->eventID = eventType;
	eventPointer->eventCount = 0;
	eventPointer->eventCountFreq = defaultCountFreq;
	eventPointer->eventLowWater = defaultLowWater;
	eventPointer->eventHighWater = defaultHighWater;
	eventPointer->eventIntervalStack = (TM_LIST *) 0;
	eventPointer->eventLast = noSuchClock;

	/* Internal events are initialized in special ways.		     */

	switch( eventType ) {

	case EVENTID_AGGREGATE:
	    eventPointer->eventFlags = EVENTFLAG_INTERNAL;
	    eventPointer->eventHard = defaultTraceLevel;
	    eventPointer->eventSoft = defaultTraceLevel;
	    eventPointer->eventTraceFunction = defaultTraceFunction;
	    eventPointer->eventRecordFunction = nullRecordFunction;
	    break;

	case EVENTID_BADCLOCK:
	    /* Set the hard and soft trace levels for EVENTID_BADCLOCK	    */
	    /* to LEVEL_COUNT if clock precision exceptions are to be	    */
	    /* included in the trace output, or just use setEventLevel.	    */

	    eventPointer->eventFlags = EVENTFLAG_INTERNAL;
	    eventPointer->eventHard = LEVEL_IGNORE;
	    eventPointer->eventSoft = LEVEL_COUNT;
	    eventPointer->eventTraceFunction = internalEventTrace;
	    eventPointer->eventRecordFunction = internalEventRecord;
	    break;

	case EVENTID_DUMPTIME:
	case EVENTID_CLOCKTIME:
	case EVENTID_TRACETIME:
	case EVENTID_COUNTTIME:
	case EVENTID_INTVLTIME:
	    eventPointer->eventFlags = EVENTFLAG_INTERNAL;
	    eventPointer->eventHard = LEVEL_INTERVAL;
	    eventPointer->eventSoft = LEVEL_INTERVAL;
	    eventPointer->eventTraceFunction = internalEventTrace;
	    eventPointer->eventRecordFunction = internalEventRecord;
	    break;

	case EVENTID_DUMMY:
	    eventPointer->eventFlags = EVENTFLAG_INTERNAL;
	    eventPointer->eventHard = LEVEL_MAXIMUM;
	    eventPointer->eventSoft = LEVEL_MAXIMUM;
	    eventPointer->eventTraceFunction = internalEventTrace;
	    eventPointer->eventRecordFunction = nullRecordFunction;
	    break;

	/* Remainder of initialization for all user (non-internal) events:  */

	default:
	    eventPointer->eventFlags = defaultEventFlags;
	    eventPointer->eventHard = defaultTraceLevel;
	    eventPointer->eventSoft = defaultTraceLevel;
	    eventPointer->eventTraceFunction = defaultTraceFunction;
	    eventPointer->eventRecordFunction = defaultRecordFunction;
	    break;
	}

	/* Now insert the new event type descriptor into its		    */
	/* corresponding hash chain.  The following code implements	    */
	/* unordered chains, but this could easily be changed to make	    */
	/* them ordered.  There's not much of an advantage in so doing,	    */
	/* the chains will be very short if the hashing function was	    */
	/* wisely chosen.						    */

	eventChainHead = & eventHashTable[ eventHashFunction( eventType ) ];

	eventPointer->eventLink = *eventChainHead;

	*eventChainHead = eventPointer;

#ifdef DEBUG
	fprintf( debugFile, "initEvent:\tInitialized event id %d\n",
		eventPointer->eventID );
	fflush( debugFile );
#endif /* DEBUG */

	return eventPointer;
}


/*
 *	eventSearch:
 *	   This function permits other parts of the trace library to
 *	   perform traversals of the entire data structure of initialized
 *	   event types.  To start a search, call eventSearch with a null
 *	   pointer.  A pointer to an event descriptor is returned.  To
 *	   get successive event descriptors, pass the last descriptor
 *	   returned as the argument.  A null pointer is returned when
 *	   the last event descriptor is passed to it.  Nested searches
 *	   may be performed without collision, this function maintains
 *	   no static data about the progress of a search.
 */

TR_EVENT *
eventSearch( eventPointer )

TR_EVENT	*eventPointer;
{
	int	hashIndex;

	if ( eventPointer == (TR_EVENT *) 0 ) {
					/* Beginning new search, start at  */
					/* beginning of event hash table.  */
		hashIndex = -1;

	} else if ( eventPointer->eventLink == (TR_EVENT *) 0 ) {
					/* End of this chain, start search */
					/* with next entry in hash table.  */
		hashIndex = eventHashFunction( eventPointer->eventID );

	} else {
					/* Return next one in this chain.  */
		return eventPointer->eventLink;
	}

	/* Return event descriptor at head of next (nonempty) chain in	   */
	/* event hash table, commencing with the value of hashIndex + 1.   */

	while ( ++hashIndex < EVENTHASHTABLESIZE )
		if ( eventHashTable[ hashIndex ] != (TR_EVENT *) 0 )
			return eventHashTable[ hashIndex ];

	/* The entire data structure has now been searched.		   */

	return (TR_EVENT *) 0;
}


/*
 *	cleanupLibrary:
 *	   This function is called by endTracing at clean-up time to reset
 *	   the trace buffer and event descriptors data structures.
 */

cleanupLibrary()
{
	TR_EVENT	*eventPointer;
	TR_EVENT	**eventChainHead;
	int		hashIndex;

	/* DO NOT insert any debugging code before the libraryInitLevel	    */
	/* test, since such code is dependent on library initialization	    */

	if ( libraryInitLevel == UNINITIALIZED )
		return;

#ifdef DEBUG
	fprintf( debugFile, "cleanupLibrary\n" );
	fflush( debugFile );
#endif /* DEBUG */

	TRcleanup();			/* system-dependent clean-up	    */

					/* trace buffer is empty now,	    */
					/* storage may be freed		    */

	if ( traceBuffer.bufferStart != (char *) 0 ) {
		TRfreeBuffer( traceBuffer.bufferStart );
		traceBuffer.bufferStart = (char *) 0;
	}

					/* empty the event hash table	    */

	for ( hashIndex = 0, eventChainHead = & eventHashTable[ 0 ];
	      hashIndex < EVENTHASHTABLESIZE;
	      hashIndex++, eventChainHead++ ) {
					/* free each chain in table	    */
	    while ( *eventChainHead != (TR_EVENT *) 0 ) {
		eventPointer = (*eventChainHead)->eventLink;
		TRfreeBuffer( (char *) *eventChainHead );
		*eventChainHead = eventPointer;
	    }
	}


					/* Restore default parameters:	    */

	defaultBufferSize =		/* default trace buffer size	    */
		DEFAULTBUFFERSIZE;

	defaultBufferType =		/* default trace buffer type	    */
		DEFAULTBUFFERTYPE;


	defaultTraceLevel =		/* default event trace level	    */
		DEFAULTTRACELEVEL;

	defaultEventFlags =		/* default event flags		    */
		DEFAULTEVENTFLAGS;

	defaultTraceFunction =		/* default event trace function	    */
		(int (*)()) 0;

	defaultRecordFunction =		/* default trace record function    */
		externalEventRecord;

	defaultCountFreq =		/* default count write frequency    */
		DEFAULTCOUNTFREQ;


	clockStopped =			/* when clock stopped in	    */
		noSuchClock;		/* toggleClock()		    */

	traceLibraryMode = MODE_SYNC;	/* library operational mode	    */

	libraryInitLevel = UNINITIALIZED;
}


/*
 *	locateEvent:
 *	   Search the list of initialized event types.  If one matching
 *	   eventType is found, return a pointer to its event descriptor.
 *	   Otherwise, return a null pointer.
 */

TR_EVENT *
locateEvent( eventType )

int	eventType;
{
	TR_EVENT	*eventPointer;

	eventPointer = eventHashTable[ eventHashFunction( eventType ) ];

	while ( eventPointer != (TR_EVENT *) 0 ) {
		if ( eventPointer->eventID == eventType )
			break;
		eventPointer = eventPointer->eventLink;
	}

	return eventPointer;
}


/*
 *	dumpBuffer:
 *	   Dump the contents of the trace buffer to the trace output file.
 *	   This function should not be called directly by any interface
 *	   routine, but only by those portions of the system-dependent
 *	   code which handle the necessary synchronization (if any).
 */

dumpBuffer()
{
	PUT_REQUEST	*request;

#ifdef DEBUG
	fprintf( debugFile, "dumpBuffer\n" );
	fflush( debugFile );
#endif /* DEBUG */

	/* If the buffer is circular and it has wrapped around, then	    */
	/* we must first dump the portion not yet overwritten.		    */

	if (( traceBuffer.bufferType == BUFFER_CIRCULAR )
	   && (( traceBuffer.bufferFlags & BUFFERFLAG_WRAP ) != 0 ))
		TRflush( traceBuffer.bufferCurrent,
			 traceBuffer.bufferEnd );

	TRflush( traceBuffer.bufferStart, traceBuffer.bufferCurrent );

	/* reset the pointers and flags to indicate an empty buffer	    */

	traceBuffer.bufferCurrent = traceBuffer.bufferStart;
	traceBuffer.bufferFlags = BUFFERFLAG_EMPTY;

	/* If there's a put request at the head of the queue which is	    */
	/* larger than the empty buffer, flush it out separately.	    */
	/* The put request queue is updated in putBytes().		    */

	if ( traceBuffer.bufferQueHead != (PUT_REQUEST *) 0 ) {
	    request = traceBuffer.bufferQueHead;
	    if ( request->reqSize >=
		    ( traceBuffer.bufferEnd - traceBuffer.bufferStart )) {
		TRflush( request->reqCurrent,
			 ( request->reqCurrent + request->reqSize ));
	    }
	}

	return SUCCESS;
}


/*
 *	putBytes:
 *	   Accepts a request to copy a sequence of bytes into the trace
 *	   buffer.  This function maintains the queue of pending put
 *	   requests, satisfying them in the order they are received.
 *	   The frequency of memory allocation and data copying is minimized
 *	   by doing this only when the request queue has more than one
 *	   element or when writing of the current request (the one at
 *	   the head of the put queue) must be interrupted for buffer
 *	   dumping.  Further, no put request is reallocated or recopied
 *	   once done the first time.
 */

putBytes( dataAddress, byteCount )

char		*dataAddress;
unsigned int	byteCount;
{
	int		spaceFree;
	PUT_REQUEST	*request;
	char		*copyRequest;

	if (( byteCount == 0 ) || ( dataAddress == (char *) 0 ))
		return SUCCESS;

#ifdef DEBUG
	fprintf( debugFile, "putBytes:\tbyteCount = %u\n", byteCount );
	fflush( debugFile );
#endif /* DEBUG */

	/* Allocate a put request descriptor for this request		    */

	request = (PUT_REQUEST *) TRgetBuffer( sizeof(PUT_REQUEST) );

	if ( request == (PUT_REQUEST *) 0 )
		TRfailure(
			  "putBytes: unable to allocate put request header" );

	request->reqSize = byteCount;
	request->reqNext = (PUT_REQUEST *) 0;

	/* Link this request onto the tail of the trace buffer's request    */
	/* queue.  If the queue is currently non-empty, then this request   */
	/* must first be copied into an allocated block of memory.  This    */
	/* invocation of putBytes will then return, since the enqueued	    */
	/* request will be serviced by a higher-level call to putBytes.	    */

	if ( traceBuffer.bufferQueTail != (PUT_REQUEST *) 0 ) {
		traceBuffer.bufferQueTail->reqNext = request;
		traceBuffer.bufferQueTail = request;
		copyRequest = TRgetBuffer( byteCount );

		if ( copyRequest == (char *) 0 )
			TRfailure(
				  "putBytes: unable to copy put request" );

		TRbcopy( dataAddress, copyRequest, byteCount );
		request->reqStart = copyRequest;
		request->reqCurrent = copyRequest;
#ifdef DEBUG
		fprintf( debugFile, "putBytes:\tput request enqueued\n" );
		fflush( debugFile );
#endif /* DEBUG */
		return SUCCESS;
	}

	/* This request will be (at least for the moment) the only one	    */
	/* on the trace buffer's request queue.				    */

	traceBuffer.bufferQueHead = request;
	traceBuffer.bufferQueTail = request;
	request->reqStart = (char *) 0;		/* Not allocated buffer	    */
	request->reqCurrent = dataAddress;

	/* Service each request on the queue until the queue is empty.	    */

	while ( traceBuffer.bufferQueHead != (PUT_REQUEST *) 0 ) {

		/* Reuse some local variables for coding convenience.	    */

		request = traceBuffer.bufferQueHead;
		dataAddress = request->reqCurrent;
		byteCount = request->reqSize;

		spaceFree = traceBuffer.bufferEnd - traceBuffer.bufferCurrent;

		while (( byteCount > 0 ) && ( byteCount >= spaceFree )) {

			/* The amount of data can exceed the buffer's	    */
			/* capacity.  When this happens, we must either	    */
			/* wrap around in the buffer or dump the buffer's   */
			/* contents to free additional space for the	    */
			/* remaining data.  If the buffer is to be dumped,  */
			/* the request descriptor at the head of the	    */
			/* queue (the current request) must be updated.	    */
			/* Also, if the current request is not yet in an    */
			/* allocated buffer, that must now be done.	    */

			switch( traceBuffer.bufferType ) {

			case BUFFER_CIRCULAR:
			    TRbcopy( dataAddress, traceBuffer.bufferCurrent,
				     spaceFree );

			    traceBuffer.bufferCurrent += spaceFree;
			    dataAddress += spaceFree;
			    byteCount -= spaceFree;

			    traceBuffer.bufferFlags |= BUFFERFLAG_WRAP;
			    break;

			default:
			case BUFFER_DUMP:
			    if ( request->reqStart == (char *) 0 ) {
				    copyRequest = TRgetBuffer( byteCount );

				    if ( copyRequest == (char *) 0 )
				     TRfailure(
				      "putBytes: unable to copy put request" );

				    TRbcopy( dataAddress, copyRequest,
					     byteCount );
				    request->reqStart = copyRequest;
				    dataAddress = copyRequest;
			    }
			    request->reqCurrent = dataAddress;
			    request->reqSize = byteCount;
#ifdef DEBUG
			    fprintf( debugFile,
					"putBytes:\tcalling TRdump\n" );
			    fflush( debugFile );
#endif /* DEBUG */
			    TRdump();		/* dump the trace buffer    */
			    traceBuffer.bufferFlags = BUFFERFLAG_EMPTY;

			    /*
			     * If this request was larger than the empty
			     * buffer size, then the dumpBuffer routine
			     * flushed the request out immediately after
			     * dumping the buffer itself.
			     */

			    if ( byteCount >=
				( traceBuffer.bufferEnd -
				  traceBuffer.bufferStart ))

				byteCount = 0;

			    break;
			}

			/* Update to reflect the dump or overwrite	    */

			traceBuffer.bufferCurrent = traceBuffer.bufferStart;

			spaceFree = traceBuffer.bufferEnd
					- traceBuffer.bufferCurrent;
		}

		/* Finally, place the remaining data (less than a	    */
		/* complete buffer) into the buffer.			    */

		if ( byteCount > 0 ) {
			TRbcopy( dataAddress, traceBuffer.bufferCurrent,
				 (int) byteCount );

			traceBuffer.bufferCurrent += byteCount;
		}

		/* The current request has now been completely satisfied.   */
		/* Free up any allocated storage and remove the put	    */
		/* request descriptor from the head of the queue.	    */

		if ( request->reqStart != (char *) 0 )
			TRfreeBuffer( request->reqStart );

		traceBuffer.bufferQueHead = request->reqNext;

		TRfreeBuffer( (char *) request );
	}

	/* The put request queue has been emptied, there are no more	    */
	/* requests to service.						    */

	traceBuffer.bufferQueTail = (PUT_REQUEST *) 0;

	return SUCCESS;
}


/*
 *	writeTrace:
 *	   Writes a trace event and its data to the trace buffer.
 */

writeTrace( eventPointer, traceData, byteCount )

TR_EVENT	*eventPointer;
char		*traceData;
unsigned int	byteCount;
{
	int		status;
	TR_RECORD	*recordPointer;

	Assert( eventPointer != (TR_EVENT *) 0 );

	/* Construct a trace event record with time and other data	    */

#ifdef DEBUG
	fprintf( debugFile, "writeTrace:\tevent id = %d, byteCount=%u\n",
		 eventPointer->eventID, byteCount );

	fflush( debugFile );
#endif /* DEBUG */

	recordPointer = (*eventPointer->eventRecordFunction)
			( RECORD_TRACE, eventPointer, noSuchClock,
			  traceData, byteCount );

	/* Write the record to the trace buffer				    */

	status = putBytes( recordPointer->recordContents,
				(unsigned) recordPointer->recordLength );

	return status;
}


/*
 *	writeCount:
 *	   Writes a count event to the trace buffer.
 */

writeCount( eventPointer )

TR_EVENT	*eventPointer;
{
	int		status;
	TR_RECORD	*recordPointer;

	Assert( eventPointer != (TR_EVENT *) 0 );

	/* Construct a count event record with time and other data	    */

#ifdef DEBUG
	fprintf( debugFile, "writeCount:\tevent id=%d\n",
		 eventPointer->eventID );
	fflush( debugFile );
#endif /* DEBUG */

	recordPointer = (*eventPointer->eventRecordFunction)
			( RECORD_COUNT, eventPointer, noSuchClock,
			  (char *) 0, 0 );

	/* Write the record to the trace buffer				    */

	status = putBytes( recordPointer->recordContents,
				(unsigned) recordPointer->recordLength );

	return status;
}


/*
 *	writeInterval:
 *	   Writes an interval event to the trace buffer.
 */

writeInterval( eventPointer )

TR_EVENT	*eventPointer;
{
	int		status;
	TR_RECORD	*recordPointer;

	Assert( eventPointer != (TR_EVENT *) 0 );

	/* Construct an interval event record with time and other data	    */

#ifdef DEBUG
	fprintf( debugFile, "writeInterval:\tevent id=%d\n",
		 eventPointer->eventID );
	fflush( debugFile );
#endif /* DEBUG */

	recordPointer = (*eventPointer->eventRecordFunction)
			( RECORD_INTERVAL, eventPointer, noSuchClock,
			  (char *) 0, 0 );

	/* Write the record to the trace buffer				    */

	status = putBytes( recordPointer->recordContents,
				(unsigned) recordPointer->recordLength );

	return status;
}


/*
 *	getClock:
 *	   Returns the elapsed time since the instrumentation code
 *	   was initialized.
 */

CLOCK
getClock()
{
	TR_CLOCK	currentTime;
	CLOCK		timeDelta;

	TRgetClock( &currentTime );

	timeDelta = TRclockDifference ( currentTime, epoch );

	return timeDelta;
}


/*
 *	sddfRecord:
 *	   Generate a SDDF binary format record for one of the
 *	   full trace, event count, or event interval classes of
 *	   events.
 */

TR_RECORD *
sddfRecord( recordFamily, recordType, eventPointer, timeStamp,
	     dataPointer, dataLength )

int		recordFamily;
int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD		traceRecord;
	static char			*recordBuffer = (char *) 0;
	static int			bufferLength = 0;
	struct recordDataTrace		*recordHeaderTrace;
	struct recordDataCount		*recordHeaderCount;
	struct recordDataInterval	*recordHeaderInterval;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) ||
	       ( recordType == RECORD_COUNT ) ||
	       ( recordType == RECORD_INTERVAL )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) || \
	       ( recordType == RECORD_COUNT ) || \
	       ( recordType == RECORD_INTERVAL ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

	Assert( eventPointer != (TR_EVENT *) 0 );

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.				    */

	switch ( recordType ) {

	case RECORD_TRACE:
		traceRecord.recordLength = sizeof *recordHeaderTrace;
		if (( dataLength > 0 ) && ( dataPointer != (char *) 0 ))
			traceRecord.recordLength += dataLength;
		break;

	case RECORD_COUNT:
		traceRecord.recordLength = sizeof *recordHeaderCount;
		break;

	case RECORD_INTERVAL:
		traceRecord.recordLength = sizeof *recordHeaderInterval;
		break;
	}

	/* If there is a previously-allocated buffer and its size will	    */
	/* hold this record, re-use the buffer.  Otherwise, deallocate	    */
	/* the buffer (if allocated) and allocate a bigger one.		    */

	if ( bufferLength < traceRecord.recordLength ) {

		if ( recordBuffer != (char *) 0 )
			TRfreeBuffer( recordBuffer );

		recordBuffer = TRgetBuffer( traceRecord.recordLength );

		if ( recordBuffer == (char *) 0 )
			TRfailure(
			     "cannot allocate storage for trace record" );

		bufferLength = traceRecord.recordLength;
	}

	traceRecord.recordContents = recordBuffer;

	/* Load the trace record fields into the allocated buffer	    */

	switch ( recordType ) {

	case RECORD_TRACE:
		recordHeaderTrace = (struct recordDataTrace *)
							recordBuffer;

		/* The actual record length will be less than that	    */
		/* allocated because of structure padding.  Correct the	    */
		/* record length in a portable manner.			    */

		traceRecord.recordLength =
					& recordHeaderTrace->userData[ 0 ]
					- recordBuffer;
		if (( dataLength > 0 ) && ( dataPointer != (char *) 0 ))
			traceRecord.recordLength += dataLength;

		recordHeaderTrace->packetLength =
				traceRecord.recordLength;
		recordHeaderTrace->packetType = PKT_DATA;
		recordHeaderTrace->packetTag = recordFamily | recordType;
		recordHeaderTrace->clockDimension = sizeof(CLOCK)/sizeof(int);
		recordHeaderTrace->timeStamp = timeStamp;
		recordHeaderTrace->seconds = clockToSeconds( timeStamp );
		recordHeaderTrace->eventID = eventPointer->eventID;
		recordHeaderTrace->nodeNumber = TRgetNode();
		if (( dataLength > 0 ) && ( dataPointer != (char *) 0 )) {
			recordHeaderTrace->userDataLength = dataLength;
			TRbcopy( dataPointer,
				 & recordHeaderTrace->userData[ 0 ],
				 dataLength );
		} else
			recordHeaderTrace->userDataLength = 0;
		break;

	case RECORD_COUNT:
		recordHeaderCount = (struct recordDataCount *)
							recordBuffer;
		recordHeaderCount->packetLength =
				traceRecord.recordLength;
		recordHeaderCount->packetType = PKT_DATA;
		recordHeaderCount->packetTag = recordFamily | recordType;
		recordHeaderCount->clockDimension = sizeof(CLOCK)/sizeof(int);
		recordHeaderCount->timeStamp = timeStamp;
		recordHeaderCount->seconds = clockToSeconds( timeStamp );
		recordHeaderCount->eventID = eventPointer->eventID;
		recordHeaderCount->nodeNumber = TRgetNode();
		recordHeaderCount->eventCount = eventPointer->eventCount;
		break;

	case RECORD_INTERVAL:
		Assert( eventPointer->eventIntervalStack != (TM_LIST *) 0 );

		recordHeaderInterval = (struct recordDataInterval *)
							recordBuffer;
		recordHeaderInterval->packetLength =
				traceRecord.recordLength;
		recordHeaderInterval->packetType = PKT_DATA;
		recordHeaderInterval->packetTag = recordFamily | recordType;
		recordHeaderInterval->clockDimension =
						sizeof(CLOCK)/sizeof(int);
		recordHeaderInterval->timeStamp = timeStamp;
		recordHeaderInterval->seconds = clockToSeconds( timeStamp );
		recordHeaderInterval->eventID = eventPointer->eventID;
		recordHeaderInterval->nodeNumber = TRgetNode();
		recordHeaderInterval->intervalDimension =
						sizeof(CLOCK)/sizeof(int);
		recordHeaderInterval->eventInterval = clockSubtract( timeStamp,
				eventPointer->eventIntervalStack->timeValue );
		recordHeaderInterval->eventSeconds =
			clockToSeconds( recordHeaderInterval->eventInterval );
		break;
	}

	return & traceRecord;
}


/*
 *	sddfWriteInteger:
 *	   Copy the integer value specified by the argument integerValue
 *	   into the SDDF record descriptor buffer pointed to by the argument
 *	   recordPointer, updating the pointer accordingly.  This function
 *	   is called only by sddfDescriptor.
 */

sddfWriteInteger( recordPointer, integerValue )

char	**recordPointer;
int	integerValue;
{
	TRbcopy( (char *) &integerValue, *recordPointer, sizeof(int) );
	*recordPointer += sizeof(int);
}


/*
 *	sddfWriteString:
 *	   Copy the character string pointed to by the argument stringPointer
 *	   into the SDDF record descriptor buffer pointed to by the argument
 *	   recordPointer, updating the pointer accordingly.  This function
 *	   is called only by sddfDescriptor.
 */

sddfWriteString( recordPointer, stringPointer )

char	**recordPointer;
char	*stringPointer;
{
	int	stringLength;		/* including final null character    */

	stringLength = 1 + strlen( stringPointer );
	TRbcopy( stringPointer, *recordPointer, stringLength );
	*recordPointer += stringLength;
}


/*
 *	sddfDescriptor:
 *	   Generate a SDDF binary format record descriptor for one of
 *	   the full trace, event count, or event interval classes of
 *	   events in a given event family.
 *
 *	NOTE:  recordBuffer is defined large enough to hold the largest
 *	SDDF record descriptor packet which will be encountered by this
 *	implementation.  It should eventually be recoded so that this
 *	is determined dynamically, reallocating and copying the packet
 *	so far if necessary.
 */

TR_RECORD *
sddfDescriptor( recordFamily, recordType, recordName, recordDescription )

int	recordFamily;
int	recordType;
char	*recordName;
char	*recordDescription;
{
	static TR_RECORD	traceRecord;
	static char		recordBuffer[ 1024 ];
	char			*recordPointer;
	char			*stringPointer;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) ||
	       ( recordType == RECORD_COUNT ) ||
	       ( recordType == RECORD_INTERVAL )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) || \
	       ( recordType == RECORD_COUNT ) || \
	       ( recordType == RECORD_INTERVAL ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

	/* Allow space at the beginning of the record for the packet	    */
	/* length, which will be computed after the packet is complete.	    */

	recordPointer = recordBuffer;
	sddfWriteInteger( &recordPointer, 0 );

	/* Write the record type descriptor.				    */

	sddfWriteInteger( &recordPointer, PKT_DESCRIPTOR );

	/* Write the record tag descriptor.				    */

	sddfWriteInteger( &recordPointer, ( recordFamily | recordType ));

	/* Write the record name.					    */

	sddfWriteString( &recordPointer, recordName );

	/* Write the record attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the record attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "description" );

	sddfWriteString( &recordPointer, recordDescription );

	/* Write the record field count.				    */

	switch( recordType ) {

	case RECORD_TRACE:
	case RECORD_COUNT:
		sddfWriteInteger( &recordPointer, 5 );
		break;

	case RECORD_INTERVAL:
		sddfWriteInteger( &recordPointer, 6 );
		break;

	}


	/* FIELD 1:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Timestamp" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Time" );

	sddfWriteString( &recordPointer, "Timestamp" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 1 );


	/* FIELD 2:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Seconds" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Seconds" );

	sddfWriteString( &recordPointer, "Floating Point Timestamp" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, DOUBLE );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 3:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Event Identifier" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "ID" );

	sddfWriteString( &recordPointer, "Event ID" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 4:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Processor Number" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Node" );

	sddfWriteString( &recordPointer, "Processor number" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 5:							    */

	/* Write the field name.					    */

	switch ( recordType ) {

	case RECORD_TRACE:
		stringPointer = "User Defined Data";
		break;

	case RECORD_COUNT:
		stringPointer = "Event Count";
		break;

	case RECORD_INTERVAL:
		stringPointer = "Interval Seconds";
		break;
	}

	sddfWriteString( &recordPointer, stringPointer );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	switch ( recordType ) {

	case RECORD_TRACE:
		stringPointer = "Data";
		break;

	case RECORD_COUNT:
		stringPointer = "Count";
		break;

	case RECORD_INTERVAL:
		stringPointer = "Interval Seconds";
		break;
	}

	sddfWriteString( &recordPointer, stringPointer );


	switch ( recordType ) {

	case RECORD_TRACE:
		stringPointer = "User data";
		break;

	case RECORD_COUNT:
		stringPointer = "Event count";
		break;

	case RECORD_INTERVAL:
		stringPointer = "Floating point interval duration";
		break;
	}

	sddfWriteString( &recordPointer, stringPointer );

	/* Write the field type.					    */

	switch ( recordType ) {

	case RECORD_TRACE:
		sddfWriteInteger( &recordPointer, CHARACTER );
		break;

	case RECORD_COUNT:
		sddfWriteInteger( &recordPointer, INTEGER );
		break;

	case RECORD_INTERVAL:
		sddfWriteInteger( &recordPointer, DOUBLE );
		break;
	}

	/* Write the field dimension.					    */

	switch ( recordType ) {

	case RECORD_TRACE:
		sddfWriteInteger( &recordPointer, 1 );
		break;

	case RECORD_COUNT:
		sddfWriteInteger( &recordPointer, 0 );
		break;

	case RECORD_INTERVAL:
		sddfWriteInteger( &recordPointer, 0 );
		break;
	}


	/* FIELD 6:							    */

	if ( recordType == RECORD_INTERVAL ) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Time Interval" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Interval" );

		sddfWriteString( &recordPointer, "Time interval" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 1 );

	}


	/* The entire record descriptor packet has been written.	    */

	traceRecord.recordLength = recordPointer - recordBuffer;
	traceRecord.recordContents = recordBuffer;

	/* Finally, copy the completed packet length at the beginning	    */
	/* of the record itself.					    */

	recordPointer = recordBuffer;
	sddfWriteInteger( &recordPointer, traceRecord.recordLength );

	return & traceRecord;
}


/*
 *	externalEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce Pablo external event type trace records.
 */

TR_RECORD *
externalEventRecord( recordType, eventPointer, timeStamp,
		     dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	return sddfRecord( FAMILY_EXTERNAL, recordType, eventPointer,
			    timeStamp, dataPointer, dataLength );
}


/*
 *	nullRecordFunction:
 *	   This function always returns null trace records.  Any
 *	   event type whose recording function is set to this will
 *	   not generate any data in the trace file whatsoever.  But
 *	   the event type will still exercise other parts of the
 *	   trace library software.  For example, the internal event
 *	   EVENTID_AGGREGATE, which controls the aggregate event
 *	   rate for all user events, employs this function.
 */

/*ARGSUSED*/

TR_RECORD *
nullRecordFunction( recordType, eventPointer, timeStamp,
		     dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD	traceRecord;

	traceRecord.recordLength = 0;
	traceRecord.recordContents = (char *) 0;

	return &traceRecord;
}


/*
 *	internalEventRecord:
 *	   This function generates trace records for internal events.
 */

TR_RECORD *
internalEventRecord( recordType, eventPointer, timeStamp,
		     dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	int	recordFamily;

	Assert( eventPointer != (TR_EVENT *) 0 );

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( eventPointer->eventID == EVENTID_AGGREGATE ) ||
	       ( eventPointer->eventID == EVENTID_BADCLOCK ) ||
	       ( eventPointer->eventID == EVENTID_DUMPTIME ) ||
	       ( eventPointer->eventID == EVENTID_CLOCKTIME ) ||
	       ( eventPointer->eventID == EVENTID_TRACETIME ) ||
	       ( eventPointer->eventID == EVENTID_COUNTTIME ) ||
	       ( eventPointer->eventID == EVENTID_INTVLTIME ) ||
	       ( eventPointer->eventID == EVENTID_DUMMY )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( eventPointer->eventID == EVENTID_AGGREGATE ) || \
	       ( eventPointer->eventID == EVENTID_BADCLOCK ) || \
	       ( eventPointer->eventID == EVENTID_DUMPTIME ) || \
	       ( eventPointer->eventID == EVENTID_CLOCKTIME ) || \
	       ( eventPointer->eventID == EVENTID_TRACETIME ) || \
	       ( eventPointer->eventID == EVENTID_COUNTTIME ) || \
	       ( eventPointer->eventID == EVENTID_INTVLTIME ) || \
	       ( eventPointer->eventID == EVENTID_DUMMY ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

	switch ( eventPointer->eventID ) {
	case EVENTID_AGGREGATE:
		recordFamily = FAMILY_AGGREGATE;
		break;
	case EVENTID_BADCLOCK:
		recordFamily = FAMILY_BADCLOCK;
		break;
	case EVENTID_DUMPTIME:
		if ( libraryInitLevel < INIT_FULL )
			return nullRecordFunction(
					recordType, eventPointer, timeStamp,
					dataPointer, dataLength );
		recordFamily = FAMILY_DUMPTIME;
		break;
	case EVENTID_CLOCKTIME:
		recordFamily = FAMILY_CLOCKTIME;
		break;
	case EVENTID_TRACETIME:
		recordFamily = FAMILY_TRACETIME;
		break;
	case EVENTID_COUNTTIME:
		recordFamily = FAMILY_COUNTTIME;
		break;
	case EVENTID_INTVLTIME:
		recordFamily = FAMILY_INTVLTIME;
		break;
	case EVENTID_DUMMY:
		recordFamily = FAMILY_DUMMY;
		break;
	}

	return sddfRecord( recordFamily, recordType, eventPointer,
			    timeStamp, dataPointer, dataLength );
}


/*
 *	internalEventTrace:
 *	   This function is invoked to determine trace levels for
 *	   internal events.  Ordinarily, trace levels of internal
 *	   events which use this function are invariant.  But it
 *	   is possible to cause selected internal events to be
 *	   ignored, and that is checked for here.  More commonly,
 *	   the EVENTID_AGGREGATE internal event type would use the
 *	   adaptiveTrace function in place of this function, to
 *	   control the aggregate user event rate.
 */

/*ARGSUSED*/

internalEventTrace( eventPointer, currentTime )

TR_EVENT	*eventPointer;
CLOCK		currentTime;
{
	Assert( eventPointer != (TR_EVENT *) 0 );

#ifdef DEBUG
	fprintf( debugFile,
		"internalEventTrace:\ttracing on internal event %d\n",
		eventPointer->eventID );
	fflush( debugFile );
#endif /* DEBUG */

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( eventPointer->eventID == EVENTID_AGGREGATE ) ||
	       ( eventPointer->eventID == EVENTID_BADCLOCK ) ||
	       ( eventPointer->eventID == EVENTID_DUMPTIME ) ||
	       ( eventPointer->eventID == EVENTID_CLOCKTIME ) ||
	       ( eventPointer->eventID == EVENTID_TRACETIME ) ||
	       ( eventPointer->eventID == EVENTID_COUNTTIME ) ||
	       ( eventPointer->eventID == EVENTID_INTVLTIME ) ||
	       ( eventPointer->eventID == EVENTID_DUMMY )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( eventPointer->eventID == EVENTID_AGGREGATE ) || \
	       ( eventPointer->eventID == EVENTID_BADCLOCK ) || \
	       ( eventPointer->eventID == EVENTID_DUMPTIME ) || \
	       ( eventPointer->eventID == EVENTID_CLOCKTIME ) || \
	       ( eventPointer->eventID == EVENTID_TRACETIME ) || \
	       ( eventPointer->eventID == EVENTID_COUNTTIME ) || \
	       ( eventPointer->eventID == EVENTID_INTVLTIME ) || \
	       ( eventPointer->eventID == EVENTID_DUMMY ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

	/* Verify that occurrences of the event type are presently	    */
	/* being recorded (as opposed to being ignored).		    */

	if ( eventPointer->eventHard == LEVEL_IGNORE )
		return LEVEL_IGNORE;

	return eventPointer->eventSoft;		/* Trace level constant	    */
}


/*
 *	writeSDDFfileHeader:
 *	   This function generates the SDDF binary format output
 *	   file header.  While the content of this header is very
 *	   system-dependent, its contents are bound at compile time.
 *
 *	   At the moment, very few options provided for in the SDDF
 *	   format are actually supported in Pablo.  While this function
 *	   ought to migrate down into system-dependent code, it is here
 *	   until more SDDF options are supported.  Even then, we might
 *	   come up with a portable way of determining things like
 *	   integer and floating point formats, distinguishing ASCII
 *	   from EBCDIC, like is done with the byte ordering.  At the
 *	   moment it's not worth the effort.
 */

writeSDDFfileHeader()
{
	char	buf[ 256 ], *bp, byteOrdering;
	long	endian = 0x7f;

	bp = (char *) &endian;

	if ( *bp == 0x7f )
		byteOrdering = Sddf_LITTLE_ENDIAN;
	else
		byteOrdering = Sddf_BIG_ENDIAN;

	bp = & buf[ 0 ];

	*bp++ = 'S';
	*bp++ = 'D';
	*bp++ = 'D';
	*bp++ = 'F';
	*bp++ = 'B';
	*bp++ = 0;			/* will contain header length	    */
	*bp++ = byteOrdering;
	*bp++ = Sddf_TWOS_COMPLEMENT;
	*bp++ = Sddf_IEEE_S;
	*bp++ = Sddf_IEEE_D;
	*bp++ = Sddf_IEEE_E;
	*bp++ = Sddf_ASCII;
	*bp++ = sizeof(char);
	*bp++ = sizeof(short);
	*bp++ = sizeof(int);
	*bp++ = sizeof(long);
	*bp++ = sizeof(float);
	*bp++ = sizeof(double);
	*bp++ = sizeof(double);		/* who actually has long double?    */

	buf[ 5 ] = bp - buf;

	putBytes( buf, (unsigned)( bp - buf ));

	return;
}


/*
 *	writeRecordDescriptors:
 *	   This function generates the record descriptors for the
 *	   beginning of the SDDF binary format trace output file.
 *	   Within any one of the classes of events defined by the
 *	   trace library (full traces, counts, and intervals) the
 *	   fields which make up that record are invariant.  However,
 *	   each individual [event family, event class] combination
 *	   must have its own unique record name and description.
 */

writeRecordDescriptors()
{
	USER_RECORD	*recordDescriptorPointer;
	TR_RECORD	*recordPointer;

	/* Invoke the functions to generate user-supplied record	    */
	/* descriptors, if any.						    */

	while ( userRecordDescriptors != (USER_RECORD *) 0 ) {

		recordDescriptorPointer = userRecordDescriptors;

		userRecordDescriptors =
				recordDescriptorPointer->nextUserRecord;

		(* recordDescriptorPointer->recordDescriptorFunction )();

		TRfreeBuffer( (char *) recordDescriptorPointer );
	}


	/* No dummy internal event descriptors ordinarily.  This family	    */
	/* of internal events has its record function set to be the	    */
	/* nullRecordFunction in initEvent, hence it never generates	    */
	/* any trace records at all.					    */

#ifdef DUMMY_DESCRIPTORS
	/* Write the dummy internal event family descriptors		    */

	recordPointer = sddfDescriptor(	FAMILY_DUMMY, RECORD_TRACE,
		"Dummy Trace", "Trace dummy event" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = sddfDescriptor(	FAMILY_DUMMY, RECORD_COUNT,
		"Dummy Count", "Count dummy event" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = sddfDescriptor(	FAMILY_DUMMY, RECORD_INTERVAL,
		"Dummy Interval", "Interval dummy event" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );
#endif /* DUMMY_DESCRIPTORS */

#ifdef META_TRACING
	/* Write the clock cost interval internal event descriptor	    */

	recordPointer = sddfDescriptor(	FAMILY_CLOCKTIME, RECORD_INTERVAL,
		"Clock Cost",
		"Clock cost event" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	/* Write the trace cost interval internal event descriptor	    */

	recordPointer = sddfDescriptor(	FAMILY_TRACETIME, RECORD_INTERVAL,
		"Trace Cost",
		"Trace cost event" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	/* Write the count cost interval internal event descriptor	    */

	recordPointer = sddfDescriptor(	FAMILY_COUNTTIME, RECORD_INTERVAL,
		"Count Cost",
		"Count cost event" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	/* Write the interval cost interval internal event descriptor	    */

	recordPointer = sddfDescriptor(	FAMILY_INTVLTIME, RECORD_INTERVAL,
		"Interval Cost",
		"Interval cost event" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );
#endif /* META_TRACING */

	/* Write the dump time interval internal event descriptor	    */

	recordPointer = sddfDescriptor(	FAMILY_DUMPTIME, RECORD_INTERVAL,
		"Dump Cost",
		"Dump cost event" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	/* Write the clock exception count internal event descriptor	    */

	recordPointer = sddfDescriptor(	FAMILY_BADCLOCK, RECORD_COUNT,
		"Clock Exception",
		"Clock exception event" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );


	/* Write the external event family descriptors			    */

	recordPointer = sddfDescriptor( FAMILY_EXTERNAL, RECORD_TRACE,
		"Generic Trace", "Generic trace event" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = sddfDescriptor( FAMILY_EXTERNAL, RECORD_COUNT,
		"Generic Count", "Generic count event" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = sddfDescriptor( FAMILY_EXTERNAL, RECORD_INTERVAL,
		"Generic Interval", "Generic interval event" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	return;
}


/*
 *	clockAdd:
 *	   This function adds the timestamp values given by its two arguments
 *	   and returns the sum.  Both parameters and the return value are of
 *	   type CLOCK.
 */

CLOCK
clockAdd( clock1, clock2 )

CLOCK	clock1, clock2;
{
	CLOCK	clockSum;
	long	carryBit = 0;

	clockSum.clkLow = clock1.clkLow + clock2.clkLow;

	if (( clockSum.clkLow < clock1.clkLow ) ||
	    ( clockSum.clkLow < clock2.clkLow ))
		carryBit = 1;

	clockSum.clkHigh = clock1.clkHigh + clock2.clkHigh + carryBit;

	return clockSum;
}


/*
 *	clockSubtract:
 *	   This function subtracts the timestamp values given by its two
 *	   arguments and returns the difference.  Both parameters and the
 *	   return value are of type CLOCK.
 */

CLOCK
clockSubtract( clock1, clock2 )

CLOCK	clock1, clock2;
{
	CLOCK	clockDiff;

	clockDiff.clkHigh = clock1.clkHigh - clock2.clkHigh;

	if ( clock1.clkLow >= clock2.clkLow ) {

		clockDiff.clkLow = clock1.clkLow - clock2.clkLow;

	} else {

		clockDiff.clkLow = 1 + ~( clock2.clkLow - clock1.clkLow );
		clockDiff.clkHigh--;

	}

	return clockDiff;
}


/*
 *	clockCompare:
 *	   This function compares the timestamp values given by its two
 *	   arguments and returns the integer -1, 0, or 1 as its first
 *	   argument is respectively less than, equal to, or greater than
 *	   its second argument.
 */

int
clockCompare( clock1, clock2 )

CLOCK	clock1, clock2;
{
	if ( clock1.clkHigh < clock2.clkHigh )
		return -1;

	if ( clock1.clkHigh > clock2.clkHigh )
		return 1;

	if ( clock1.clkHigh >= 0 ) {
		if ( clock1.clkLow < clock2.clkLow )
			return -1;
		if ( clock1.clkLow > clock2.clkLow )
			return 1;
	} else {
		if ( clock1.clkLow < clock2.clkLow )
			return 1;
		if ( clock1.clkLow > clock2.clkLow )
			return -1;
	}

	return 0;
}


/*
 *	clockToSeconds:
 *	   This function converts a CLOCK value to double-precision floating
 *	   seconds, returning the result.
 */

double
clockToSeconds( clockValue )

CLOCK	clockValue;
{
	return	( 4294967296.0 * clockValue.clkHigh + clockValue.clkLow )
		/ CLOCKRATE;
}


/*
 *	newBufferSize:
 *	   Set the trace buffer size to the number of bytes specified by
 *	   its non-negative argument.  Must be done before basic library
 *	   initialization.
 */

newBufferSize( bufferSize )

int	bufferSize;
{
	if (( bufferSize < 0 ) || ( libraryInitLevel > UNINITIALIZED ))
		return FAILURE;

	/*
	 * If unbuffered mode is selected, this automatically inhibits
	 * synchronous operation.  This may be overriden by a determined
	 * user.  Also, substitute a buffer size of 1 byte rather than
	 * not allocating any buffer at all.  Nothing will ever be
	 * written in such a buffer, but it makes coding simpler.
	 */

	if ( bufferSize == 0 ) {
		traceLibraryMode = MODE_ASYNC;
		bufferSize = 1;
	}

	Assert( bufferSize > 0 );

	defaultBufferSize = (unsigned) bufferSize;

	return SUCCESS;
}
