/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 *  Trace.h:
 *	This file contains all the top-level data structure type
 *	definitions for the Pablo instrumentation interface library.
 */


/*
 *	Portable Clock type:
 *	   Clock values in the portable code are all differences between
 *	   native clock values (type TR_CLOCK).  These are represented
 *	   as signed 64-bit values and measure nonnegative time intervals
 *	   in whatever units the native clock does.  The elements of the
 *	   structure contain the low- and high-order 32 bits of the clock
 *	   difference value.
 */

typedef struct {

	unsigned long	clkLow;		/* low-order 32 bits		    */

	long		clkHigh;	/* high-order 32 bits		    */

} CLOCK;


/*
 *	Trace Record descriptor:
 *	   The functions which assemble event trace records return
 *	   structures of this type, which simply bind together a pointer
 *	   to the contents of a trace record itself along with an
 *	   integer giving the number of bytes in the record.  Typically
 *	   each such function would declare one of these structures as
 *	   static and then return a pointer to it.  If the trace record
 *	   contents have constant size, the function may also declare
 *	   a static buffer for holding those contents, in which case the
 *	   pointer in the trace record descriptor returned will always
 *	   be the same.  Variable-size trace record contents will require
 *	   that the function allocate storage for holding the trace
 *	   record itself, or declare a static buffer large enough to
 *	   contain the largest possible record.
 */

typedef struct traceRecord {

	int	recordLength;		/* number of bytes		    */

	char	*recordContents;	/* trace record contents	    */

} TR_RECORD;


/*
 *	Pablo Record Data packet structures:
 *	   The Pablo binary file format defines several types of packets
 *	   which may appear in trace output files.  One of these is the
 *	   Record Data packet.  Instances of these packets which are
 *	   produced by this trace library all have one of the following
 *	   structures, depending on which types of records they are.
 */


/* RECORD_TRACE type Record Data packet:				    */

struct recordDataTrace {

	int	packetLength;		/* bytes in packet		    */

	int	packetType;		/* == PKT_DATA			    */

	int	packetTag;		/* event family | RECORD_TRACE	    */

	int	clockDimension;		/* number of ints in a CLOCK	    */

	CLOCK	timeStamp;		/* time record generated	    */

	double	seconds;		/* floating-point timestamp	    */

	int	eventID;		/* ID of corresponding event	    */

	int	nodeNumber;		/* occurred on which node	    */

	int	userDataLength;		/* bytes of user-supplied data	    */

	/*
	 *	The remainder of this packet is occupied by the user-
	 *	supplied data itself, zero or more bytes.  This is
	 *	variable in length and cannot be determined before
	 *	run time.
	 */

	char	userData[ 1 ];		/* user-supplied data		    */

};


/* RECORD_COUNT type Record Data packet:				    */

struct recordDataCount {

	int	packetLength;		/* bytes in packet		    */

	int	packetType;		/* == PKT_DATA			    */

	int	packetTag;		/* event family | RECORD_COUNT	    */

	int	clockDimension;		/* number of ints in a CLOCK	    */

	CLOCK	timeStamp;		/* time record generated	    */

	double	seconds;		/* floating-point timestamp	    */

	int	eventID;		/* ID of corresponding event	    */

	int	nodeNumber;		/* occurred on which node	    */

	long	eventCount;		/* count of event occurrences	    */
};


/* RECORD_INTERVAL type Record Data packet:				    */

struct recordDataInterval {

	int	packetLength;		/* bytes in packet		    */

	int	packetType;		/* == PKT_DATA			    */

	int	packetTag;		/* event family|RECORD_INTERVAL	    */

	int	clockDimension;		/* number of ints in a CLOCK	    */

	CLOCK	timeStamp;		/* time record generated	    */

	double	seconds;		/* floating-point timestamp	    */

	int	eventID;		/* ID of corresponding event	    */

	int	nodeNumber;		/* occurred on which node	    */

	double	eventSeconds;		/* floating-point duration	    */

	int	intervalDimension;	/* number of ints in a CLOCK	    */

	CLOCK	eventInterval;		/* duration of interval event	    */
};



/*
 *	Trace Buffer Put Request descriptor:
 *	   Requests to deposit data into the trace buffer must be served
 *	   chronologically if the results are to be meaningful.  It is
 *	   possible for a request to be only partially satisfied when the
 *	   buffer fills, requiring the buffer to be dumped to an output
 *	   file.  The act of dumping the trace buffer may generate internal
 *	   events, with their own trace output data.  However, the pending
 *	   data must be written out before any data generated by the buffer
 *	   dump may be written to the buffer.  The trace buffer maintains
 *	   a queue (FIFO list) of put requests not yet fully satisfied.
 */

typedef struct putRequest {

	char		*reqStart;	/* start of allocated block	    */

	char		*reqCurrent;	/* next byte to be output	    */

	unsigned int	reqSize;	/* number of bytes remaining	    */

	struct putRequest
			*reqNext;	/* next request in queue	    */

} PUT_REQUEST;


/*
 *	Trace Buffer descriptor:
 *	   The local trace buffer holds trace records until dumped.
 *	   The trace buffer descriptor contains fields which describe
 *	   the actual buffer being used.  The user program may modify
 *	   defaults for some of these fields.
 */

typedef struct traceBuffer {

  	char		*bufferStart;	/* pointer to buffer beginning	    */

  	char		*bufferEnd;	/* pointer to buffer end	    */

  	char		*bufferCurrent;	/* pointer to buffer position	    */

  	unsigned int	bufferType;	/* action at end of buffer	    */

  	unsigned int 	bufferFlags;	/* current state of buffer	    */

	PUT_REQUEST	*bufferQueHead;	/* pointer to head of request queue */

	PUT_REQUEST	*bufferQueTail;	/* pointer to tail of request queue */

} TR_BUFFER;


/*
 *	Time list:
 *	   Definition of a structure which may be used for a linked list
 *	   of event times.  This is used within the event descriptor
 *	   data structure below for stacking interval events.
 */

typedef struct timeList {

	struct timeList	*timeNext;	/* next time in list		    */

	CLOCK		timeValue;	/* the event time datum		    */

} TM_LIST;


/*
 *	Event descriptor:
 *	   Each type of event is described by an instance of the following
 *	   structure.  Notice that several fields can be independently
 *	   adjusted by user interface functions.
 */

typedef struct traceEvent {

	int		eventID;	/* event type identifier	    */

	unsigned int	eventFlags;	/* event type attributes	    */

	unsigned long	eventCount;	/* event count for this type	    */

	unsigned long	eventCountFreq;	/* how frequently count traces	    */
					/* will be written for this	    */
					/* event type, every Nth one	    */

  	unsigned int	eventHard;	/* optionally set by user to	    */
					/* indicate the maximum trace	    */
					/* level			    */

  	unsigned int	eventSoft;	/* controlled by software, but	    */
					/* always <= the hard limit	    */

	CLOCK		eventLowWater;	/* lower threshhold for event	    */
					/* throttling (i.e., conversion */
					/* to counts)			    */

	CLOCK		eventHighWater;	/* upper threshold for event	    */
					/* throttling			    */

	TM_LIST				/* timestamp for stopwatch	    */
			*eventIntervalStack;

	CLOCK		eventLast;	/* timestamp for the last 	    */
					/* of this event		    */

					/* user trace function		    */
  	int		(*eventTraceFunction)();

					/* trace record function	    */
	TR_RECORD	*(*eventRecordFunction)();

	struct traceEvent
			*eventLink;	/* linked list pointer		    */

} TR_EVENT;


/*
 *	User Record Descriptor Function list:
 *	   The user may specify, by calling setRecordDescriptor prior
 *	   to trace library initialization, one or more functions
 *	   which produce record descriptors for user-defined trace
 *	   records.  Pointers to these functions are kept in a linked
 *	   list until library initialization time.
 */

typedef struct userRecord {

	int				/* record descriptor function	    */
			(*recordDescriptorFunction)();

	struct userRecord		/* linked list pointer		    */
			*nextUserRecord;

} USER_RECORD;



/* Type declarations for all system-dependent interface functions:	    */

#ifndef TRsetTraceFile
extern void	TRsetTraceFile();
#endif /* TRsetTraceFile */

#ifndef TRsetDebugFile
#ifdef DEBUG
extern void	TRsetDebugFile();
#endif /* DEBUG */
#endif /* TRsetDebugFile */

#ifndef TRsetSocketAddress
extern void	TRsetSocketAddress();
#endif /* TRsetSocketAddress */

#ifndef TRinit
extern void	TRinit();
#endif /* TRinit */

#ifndef TRcleanup
extern void	TRcleanup();
#endif /* TRcleanup */

#ifndef TRgetClock
extern void	TRgetClock();
#endif /* TRgetClock */

#ifndef TRclockDifference
extern CLOCK	TRclockDifference();
#endif /* TRclockDifference */

#ifndef TRincrementClock
extern void	TRincrementClock();
#endif /* TRincrementClock */

#ifndef TRgetBuffer
extern char  *	TRgetBuffer();
#endif /* TRgetBuffer */

#ifndef TRfreeBuffer
extern void	TRfreeBuffer();
#endif /* TRfreeBuffer */

#ifndef TRfailure
extern void	TRfailure();
#endif /* TRfailure */

#ifndef TRflush
extern void	TRflush();
#endif /* TRflush */

#ifndef TRdump
extern void	TRdump();
#endif /* TRdump */

#ifndef TRbcopy
extern void	TRbcopy();
#endif /* TRbcopy */

#ifndef TRlock
extern TR_LOCK	TRlock();
#endif /* TRlock */

#ifndef TRunlock
extern void	TRunlock();
#endif /* TRunlock */

#ifndef TRgetNode
extern int	TRgetNode();
#endif /* TRgetNode */


/* Type declarations for some portable library internal functions:	    */

extern TR_EVENT		*initEvent();
extern TR_EVENT		*eventSearch();
extern TR_EVENT		*locateEvent();
extern CLOCK		getClock();
extern CLOCK		clockAdd();
extern CLOCK		clockSubtract();
extern int		adaptiveTrace();
extern int		internalEventTrace();
extern TR_RECORD	*externalEventRecord();
extern TR_RECORD	*internalEventRecord();
extern TR_RECORD	*nullRecordFunction();
extern double		clockToSeconds();


/*
 *	These variables and definitions are shared among the system
 *	independent and dependent portions of the instrumentation library.
 */

extern TR_BUFFER	traceBuffer;	/* local trace buffer structure	    */

					/* default trace buffer size	    */
extern unsigned int	defaultBufferSize;

					/* default trace buffer type	    */
extern unsigned int	defaultBufferType;


					/* scatter table of chains	    */
					/* containing all event types	    */
extern TR_EVENT		*eventHashTable[];

					/* default event trace level	    */
extern unsigned int	defaultTraceLevel;

					/* default event flags		    */
extern unsigned int	defaultEventFlags;

					/* default values for adaptive	    */
					/* tracing control parameters	    */
extern CLOCK		defaultLowWater;
extern CLOCK		defaultHighWater;

					/* default event trace function	    */
extern int		(*defaultTraceFunction)();

					/* default trace record function    */
extern TR_RECORD	*(*defaultRecordFunction)();

					/* default count write frequency    */
extern unsigned long	defaultCountFreq;


extern TR_CLOCK		epoch;		/* time at the beginning of	    */
					/* instrumentation		    */

extern CLOCK		clockStopped;	/* when clock stopped in	    */
					/* toggleclock()		    */

extern CLOCK		noSuchClock;	/* constant NOSUCHTIME value	    */

extern CLOCK		zeroClock;	/* constant ZEROTIME value	    */

extern CLOCK		clockPrecision;	/* clock granularity		    */

extern int		traceLibraryMode;	/* operational mode	    */

extern int		libraryInitLevel;	/* initialization level	    */


/* Macro for optimization of library initialization			    */

#define TRACELIB_INIT(n)  if (libraryInitLevel < n) initLibrary(n)
