/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 * CMMDtrace.c:
 *	This file contains interface functions for tracing the specific
 *	CMMD message-passing routines found on TMC CM-5 systems.  It is
 *	intended to be used with the Pablo instrumentation library and
 *	the extensions to that library found in the file MesgTrace.c.
 *	Interface functions for Fortran precede their C counterparts.
 */

#include "SystemDepend.h"
#undef MESGTRACE
#include "CMMDtrace.h"


/*
 * CMMDtraceInit:
 *	This function is the CMMD tracing interface function for
 *	initialization of the Pablo instrumentation library message-
 *	passing extensions.  Inputs are arrays of the event IDs defined
 *	above.  This function must be invoked before calling any of the
 *	CMMD-specific message-passing trace interface functions below.
 */

cmmdtraceinit_()
{
	CMMDtraceInit();
}

CMMDtraceInit()
{
	static int	beginIDs[ 5 ] = {
			sendSyncBeginID, receiveSyncBeginID,
			sendAsyncBeginID, receiveAsyncBeginID,
			mesgBeginID
	};

	static int	endIDs[ 5 ] = {
			sendSyncEndID, receiveSyncEndID,
			sendAsyncEndID, receiveAsyncEndID,
			mesgEndID
	};

	initMesgTrace( beginIDs, endIDs );

#ifdef DEBUG
	fprintf( debugFile, "CMMDtraceInit done\n" );
	fflush( debugFile );
#endif /* DEBUG */
}


/*
 * traceSENDSYNC:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_send_block.
 */

int 
tracesendsync_( destinationNode, messageTag, dataPointer, messageSize )

int	*destinationNode;
int	*messageTag;
void	*dataPointer;
int	*messageSize;
{
	return traceSENDSYNC( *destinationNode, *messageTag, dataPointer,
			      *messageSize );
}

int
traceSENDSYNC( destinationNode, messageTag, dataPointer, messageSize )

int	destinationNode;
int	messageTag;
void	*dataPointer;
int	messageSize;
{
	int	status;
	struct {
		int	tag;
		int	length;
		int	node;
		int	pid;
	} sendSyncArgs;

#ifdef DEBUG
	fprintf( debugFile, "traceSENDSYNC\n" );
	fflush( debugFile );
#endif /* DEBUG */

	sendSyncArgs.tag = messageTag;
	sendSyncArgs.length = messageSize;
	sendSyncArgs.node = destinationNode;
	sendSyncArgs.pid = 0;

	traceEvent( sendSyncBeginID, (char *) &sendSyncArgs,
		    sizeof sendSyncArgs );

	status = CMMD_send_block( destinationNode, messageTag, dataPointer,
				  messageSize );

	traceEvent( sendSyncEndID, (char *) 0, 0 );

	return status;
}


/*
 * traceRECVSYNC:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_receive_block.
 */

int
tracerecvsync_( sourceNode, messageTag, dataPointer, messageSize )

int	*sourceNode;
int	*messageTag;
void	*dataPointer;
int	*messageSize;
{
	return traceRECVSYNC( *sourceNode, *messageTag, dataPointer,
			      *messageSize );
}

int
traceRECVSYNC( sourceNode, messageTag, dataPointer, messageSize )

int	sourceNode;
int	messageTag;
void	*dataPointer;
int	messageSize;
{
	int	status;
	struct {
		int	tag;
		int	length;
	} recvSyncArgs;

#ifdef DEBUG
	fprintf( debugFile, "traceRECVSYNC\n" );
	fflush( debugFile );
#endif /* DEBUG */

	recvSyncArgs.tag = messageTag;
	recvSyncArgs.length = messageSize;

	traceEvent( receiveSyncBeginID, (char *) &recvSyncArgs,
		    sizeof recvSyncArgs );

	status = CMMD_receive_block( sourceNode, messageTag, dataPointer,
				     messageSize );

	if ( sourceNode == CMMD_ANY_NODE )
		sourceNode = CMMD_msg_sender();

	traceEvent( receiveSyncEndID, (char *) &sourceNode,
		    sizeof sourceNode );

	return status;
}


/*
 * traceSENDASYNC:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_send_async.
 */

CMMD_mcb
tracesendasync_( destinationNode, messageTag, dataPointer, messageSize,
		 handler, handler_arg )

int	*destinationNode;
int	*messageTag;
void	*dataPointer;
int	*messageSize;
void	(*handler)();
void	*handler_arg;
{
	return traceSENDASYNC( *destinationNode, *messageTag, dataPointer,
			       *messageSize, handler, handler_arg );
}

CMMD_mcb
traceSENDASYNC( destinationNode, messageTag, dataPointer, messageSize,
		handler, handler_arg )

int	destinationNode;
int	messageTag;
void	*dataPointer;
int	messageSize;
void	(*handler)();
void	*handler_arg;
{
	CMMD_mcb	mcb;
	struct {
		int	tag;
		int	length;
		int	node;
		int	pid;
	} sendAsyncArgs;

#ifdef DEBUG
	fprintf( debugFile, "traceSENDASYNC\n" );
	fflush( debugFile );
#endif /* DEBUG */

	sendAsyncArgs.tag = messageTag;
	sendAsyncArgs.length = messageSize;
	sendAsyncArgs.node = destinationNode;
	sendAsyncArgs.pid = 0;

	traceEvent( sendAsyncBeginID, (char *) &sendAsyncArgs,
		    sizeof sendAsyncArgs );

	mcb = CMMD_send_async( destinationNode, messageTag, dataPointer,
			       messageSize, handler, handler_arg );

	traceEvent( sendAsyncEndID, (char *) 0, 0 );

	return mcb;
}


/*
 * traceRECVASYNC:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_receive_async.
 */

CMMD_mcb
tracerecvasync_( sourceNode, messageTag, dataPointer, messageSize,
		 handler, handler_arg )

int	*sourceNode;
int	*messageTag;
void	*dataPointer;
int	*messageSize;
void	(*handler)();
void	*handler_arg;
{
	return traceRECVASYNC( *sourceNode, *messageTag, dataPointer,
			       *messageSize, handler, handler_arg );
}

CMMD_mcb
traceRECVASYNC( sourceNode, messageTag, dataPointer, messageSize,
		handler, handler_arg )

int	sourceNode;
int	messageTag;
void	*dataPointer;
int	messageSize;
void	(*handler)();
void	*handler_arg;
{
	CMMD_mcb	mcb;
	struct {
		int	tag;
		int	length;
	} recvAsyncArgs;

#ifdef DEBUG
	fprintf( debugFile, "traceRECVASYNC\n" );
	fflush( debugFile );
#endif /* DEBUG */

	recvAsyncArgs.tag = messageTag;
	recvAsyncArgs.length = messageSize;

	traceEvent( receiveAsyncBeginID, (char *) &recvAsyncArgs,
		    sizeof recvAsyncArgs );

	mcb = CMMD_receive_async( sourceNode, messageTag, dataPointer,
				  messageSize, handler, handler_arg );

	traceEvent( receiveAsyncEndID, (char *) 0, 0 );

	return mcb;
}


/*
 * traceREDUCEINT:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_reduce_int.
 */

int
tracereduceint_( value, combiner )

int		*value;
CMMD_combiner_t	*combiner;
{
	return traceREDUCEINT( *value, *combiner );
}

int
traceREDUCEINT( value, combiner )

int		value;
CMMD_combiner_t	combiner;
{
 	int	status;
 	int	reduceType;

#ifdef DEBUG
	fprintf( debugFile, "traceREDUCEINT\n" );
	fflush( debugFile );
#endif /* DEBUG */

	switch( combiner ) {
	case CMMD_combiner_add:
		reduceType = PABLO_CMMD_REDUCE_ADD;
		break;
	case CMMD_combiner_max:
		reduceType = PABLO_CMMD_REDUCE_MAX;
		break;
	case CMMD_combiner_min:
		reduceType = PABLO_CMMD_REDUCE_MIN;
		break;
	case CMMD_combiner_ior:
		reduceType = PABLO_CMMD_REDUCE_IOR;
		break;
	case CMMD_combiner_xor:
		reduceType = PABLO_CMMD_REDUCE_XOR;
		break;
	case CMMD_combiner_and:
		reduceType = PABLO_CMMD_REDUCE_AND;
		break;
	default:
		reduceType = -1;
		break;
	}

	traceEvent( mesgBeginID, (char *) &reduceType, sizeof reduceType );

	status = CMMD_reduce_int( value, combiner );

	traceEvent( mesgEndID, (char *) &reduceType, sizeof reduceType );

	return status;
}


/*
 * traceREDUCEUINT:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_reduce_uint.
 */

unsigned
tracereduceuint_( value, combiner )

unsigned	*value;
CMMD_combiner_t	*combiner;
{
	return traceREDUCEUINT( *value, *combiner );
}

unsigned
traceREDUCEUINT( value, combiner )

unsigned	value;
CMMD_combiner_t	combiner;
{
 	unsigned	status;
 	int		reduceType;

#ifdef DEBUG
	fprintf( debugFile, "traceREDUCEUINT\n" );
	fflush( debugFile );
#endif /* DEBUG */

	switch( combiner ) {
	case CMMD_combiner_uadd:
		reduceType = PABLO_CMMD_REDUCE_ADD;
		break;
	case CMMD_combiner_umax:
		reduceType = PABLO_CMMD_REDUCE_MAX;
		break;
	case CMMD_combiner_umin:
		reduceType = PABLO_CMMD_REDUCE_MIN;
		break;
	case CMMD_combiner_ior:
		reduceType = PABLO_CMMD_REDUCE_IOR;
		break;
	case CMMD_combiner_xor:
		reduceType = PABLO_CMMD_REDUCE_XOR;
		break;
	case CMMD_combiner_and:
		reduceType = PABLO_CMMD_REDUCE_AND;
		break;
	default:
		reduceType = -1;
		break;
	}

	traceEvent( mesgBeginID, (char *) &reduceType, sizeof reduceType );

	status = CMMD_reduce_uint( value, combiner );

	traceEvent( mesgEndID, (char *) &reduceType, sizeof reduceType );

	return status;
}


/*
 * traceREDUCEFLOAT:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_reduce_float.
 */

double
tracereducefloat_( value, combiner )

float		*value;
CMMD_combiner_t	*combiner;
{
	return traceREDUCEFLOAT( *value, *combiner );
}

double
traceREDUCEFLOAT( value, combiner )

float		value;
CMMD_combiner_t	combiner;
{
 	double	status;
 	int	reduceType;

#ifdef DEBUG
	fprintf( debugFile, "traceREDUCEFLOAT\n" );
	fflush( debugFile );
#endif /* DEBUG */

	switch( combiner ) {
	case CMMD_combiner_fadd:
	case CMMD_combiner_add:
		reduceType = PABLO_CMMD_REDUCE_ADD;
		break;
	case CMMD_combiner_fmax:
	case CMMD_combiner_max:
		reduceType = PABLO_CMMD_REDUCE_MAX;
		break;
	case CMMD_combiner_fmin:
	case CMMD_combiner_min:
		reduceType = PABLO_CMMD_REDUCE_MIN;
		break;
	default:
		reduceType = -1;
		break;
	}

	traceEvent( mesgBeginID, (char *) &reduceType, sizeof reduceType );

	status = CMMD_reduce_float( value, combiner );

	traceEvent( mesgEndID, (char *) &reduceType, sizeof reduceType );

	return status;
}


/*
 * traceREDUCEDOUBLE:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_reduce_double.
 */

double
tracereducedouble_( value, combiner )

double		*value;
CMMD_combiner_t	*combiner;
{
	return traceREDUCEDOUBLE( *value, *combiner );
}

double
traceREDUCEDOUBLE( value, combiner )

double		value;
CMMD_combiner_t	combiner;
{
 	double	status;
 	int	reduceType;

#ifdef DEBUG
	fprintf( debugFile, "traceREDUCEDOUBLE\n" );
	fflush( debugFile );
#endif /* DEBUG */

	switch( combiner ) {
	case CMMD_combiner_dadd:
	case CMMD_combiner_add:
		reduceType = PABLO_CMMD_REDUCE_ADD;
		break;
	case CMMD_combiner_dmax:
	case CMMD_combiner_max:
		reduceType = PABLO_CMMD_REDUCE_MAX;
		break;
	case CMMD_combiner_dmin:
	case CMMD_combiner_min:
		reduceType = PABLO_CMMD_REDUCE_MIN;
		break;
	default:
		reduceType = -1;
		break;
	}

	traceEvent( mesgBeginID, (char *) &reduceType, sizeof reduceType );

	status = CMMD_reduce_double( value, combiner );

	traceEvent( mesgEndID, (char *) &reduceType, sizeof reduceType );

	return status;
}


/*
 * traceSCANINT:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_scan_int.
 */

int
tracescanint_( value, combiner, direction, smode, sbit, inclusion )

int			*value;
CMMD_combiner_t		*combiner;
CMMD_scan_direction_t	*direction;
CMMD_segment_mode_t	*smode;
int			*sbit;
CMMD_scan_inclusion_t	*inclusion;
{
	return traceSCANINT( *value, *combiner, *direction, *smode, *sbit,
			     *inclusion );
}

int
traceSCANINT( value, combiner, direction, smode, sbit, inclusion )

int			value;
CMMD_combiner_t		combiner;
CMMD_scan_direction_t	direction;
CMMD_segment_mode_t	smode;
int			sbit;
CMMD_scan_inclusion_t	inclusion;
{
 	int	status;
 	int	scanType;

#ifdef DEBUG
	fprintf( debugFile, "traceSCANINT\n" );
	fflush( debugFile );
#endif /* DEBUG */

	switch( combiner ) {
	case CMMD_combiner_add:
		scanType = PABLO_CMMD_SCAN_ADD;
		break;
	case CMMD_combiner_max:
		scanType = PABLO_CMMD_SCAN_MAX;
		break;
	case CMMD_combiner_min:
		scanType = PABLO_CMMD_SCAN_MIN;
		break;
	case CMMD_combiner_ior:
		scanType = PABLO_CMMD_SCAN_IOR;
		break;
	case CMMD_combiner_xor:
		scanType = PABLO_CMMD_SCAN_XOR;
		break;
	case CMMD_combiner_and:
		scanType = PABLO_CMMD_SCAN_AND;
		break;
	default:
		scanType = -1;
		break;
	}

	traceEvent( mesgBeginID, (char *) &scanType, sizeof scanType );

	status = CMMD_scan_int( value, combiner, direction, smode, sbit,
				inclusion );

	traceEvent( mesgEndID, (char *) &scanType, sizeof scanType );

	return status;
}


/*
 * traceSCANUINT:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_scan_uint.
 */

unsigned
tracescanuint_( value, combiner, direction, smode, sbit, inclusion )

unsigned		*value;
CMMD_combiner_t		*combiner;
CMMD_scan_direction_t	*direction;
CMMD_segment_mode_t	*smode;
int			*sbit;
CMMD_scan_inclusion_t	*inclusion;
{
	return traceSCANUINT( *value, *combiner, *direction, *smode, *sbit,
			      *inclusion );
}

unsigned
traceSCANUINT( value, combiner, direction, smode, sbit, inclusion )

unsigned		value;
CMMD_combiner_t		combiner;
CMMD_scan_direction_t	direction;
CMMD_segment_mode_t	smode;
int			sbit;
CMMD_scan_inclusion_t	inclusion;
{
 	unsigned	status;
 	int		scanType;

#ifdef DEBUG
	fprintf( debugFile, "traceSCANUINT\n" );
	fflush( debugFile );
#endif /* DEBUG */

	switch( combiner ) {
	case CMMD_combiner_uadd:
		scanType = PABLO_CMMD_SCAN_ADD;
		break;
	case CMMD_combiner_umax:
		scanType = PABLO_CMMD_SCAN_MAX;
		break;
	case CMMD_combiner_umin:
		scanType = PABLO_CMMD_SCAN_MIN;
		break;
	case CMMD_combiner_ior:
		scanType = PABLO_CMMD_SCAN_IOR;
		break;
	case CMMD_combiner_xor:
		scanType = PABLO_CMMD_SCAN_XOR;
		break;
	case CMMD_combiner_and:
		scanType = PABLO_CMMD_SCAN_AND;
		break;
	default:
		scanType = -1;
		break;
	}

	traceEvent( mesgBeginID, (char *) &scanType, sizeof scanType );

	status = CMMD_scan_uint( value, combiner, direction, smode, sbit,
				 inclusion );

	traceEvent( mesgEndID, (char *) &scanType, sizeof scanType );

	return status;
}


/*
 * traceSCANFLOAT:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_scan_float.
 */

double
tracescanfloat_( value, combiner, direction, smode, sbit, inclusion )

float			*value;
CMMD_combiner_t		*combiner;
CMMD_scan_direction_t	*direction;
CMMD_segment_mode_t	*smode;
int			*sbit;
CMMD_scan_inclusion_t	*inclusion;
{
	return traceSCANFLOAT( *value, *combiner, *direction, *smode, *sbit,
			       *inclusion );
}

double
traceSCANFLOAT( value, combiner, direction, smode, sbit, inclusion )

float			value;
CMMD_combiner_t		combiner;
CMMD_scan_direction_t	direction;
CMMD_segment_mode_t	smode;
int			sbit;
CMMD_scan_inclusion_t	inclusion;
{
 	double	status;
 	int	scanType;

#ifdef DEBUG
	fprintf( debugFile, "traceSCANFLOAT\n" );
	fflush( debugFile );
#endif /* DEBUG */

	switch( combiner ) {
	case CMMD_combiner_fadd:
	case CMMD_combiner_add:
		scanType = PABLO_CMMD_SCAN_ADD;
		break;
	case CMMD_combiner_fmax:
	case CMMD_combiner_max:
		scanType = PABLO_CMMD_SCAN_MAX;
		break;
	case CMMD_combiner_fmin:
	case CMMD_combiner_min:
		scanType = PABLO_CMMD_SCAN_MIN;
		break;
	default:
		scanType = -1;
		break;
	}

	traceEvent( mesgBeginID, (char *) &scanType, sizeof scanType );

	status = CMMD_scan_float( value, combiner, direction, smode, sbit,
				  inclusion );

	traceEvent( mesgEndID, (char *) &scanType, sizeof scanType );

	return status;
}


/*
 * traceSCANDOUBLE:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_scan_double.
 */

double
tracescandouble_( value, combiner, direction, smode, sbit, inclusion )

double			*value;
CMMD_combiner_t		*combiner;
CMMD_scan_direction_t	*direction;
CMMD_segment_mode_t	*smode;
int			*sbit;
CMMD_scan_inclusion_t	*inclusion;
{
	return traceSCANDOUBLE( *value, *combiner, *direction, *smode, *sbit,
				*inclusion );
}

double
traceSCANDOUBLE( value, combiner, direction, smode, sbit, inclusion )

double			value;
CMMD_combiner_t		combiner;
CMMD_scan_direction_t	direction;
CMMD_segment_mode_t	smode;
int			sbit;
CMMD_scan_inclusion_t	inclusion;
{
 	double	status;
 	int	scanType;

#ifdef DEBUG
	fprintf( debugFile, "traceSCANDOUBLE\n" );
	fflush( debugFile );
#endif /* DEBUG */

	switch( combiner ) {
	case CMMD_combiner_dadd:
	case CMMD_combiner_add:
		scanType = PABLO_CMMD_SCAN_ADD;
		break;
	case CMMD_combiner_dmax:
	case CMMD_combiner_max:
		scanType = PABLO_CMMD_SCAN_MAX;
		break;
	case CMMD_combiner_dmin:
	case CMMD_combiner_min:
		scanType = PABLO_CMMD_SCAN_MIN;
		break;
	default:
		scanType = -1;
		break;
	}

	traceEvent( mesgBeginID, (char *) &scanType, sizeof scanType );

	status = CMMD_scan_double( value, combiner, direction, smode, sbit,
				   inclusion );

	traceEvent( mesgEndID, (char *) &scanType, sizeof scanType );

	return status;
}


/*
 * traceCONCATNODES:
 *	Thinking Machines CM-5 interface function for tracing calls to
 *	CMMD_concat_with_nodes.
 */

void
traceconcatnodes_( element, buffer, elem_length )

void	*element;
void	*buffer;
int	*elem_length;
{
	traceCONCATNODES( element, buffer, *elem_length );
}

void
traceCONCATNODES( element, buffer, elem_length )

void	*element;
void	*buffer;
int	elem_length;
{
 	int	concatType = PABLO_CMMD_CONCAT;

#ifdef DEBUG
	fprintf( debugFile, "traceCONCATNODES\n" );
	fflush( debugFile );
#endif /* DEBUG */

	traceEvent( mesgBeginID, (char *) &concatType, sizeof concatType );

	CMMD_concat_with_nodes( element, buffer, elem_length );

	traceEvent( mesgEndID, (char *) &concatType, sizeof concatType );

	return;
}
