/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 * MesgTrace.c
 *	This file contains extensions to the Pablo instrumentation library
 *	which facilitate tracing of message passing functions.  This is done
 *	by defining new families of trace records, message send and receive.
 *	The message send and receive events occur only as full traces (i.e.
 *	neither count nor interval events) with their relevant arguments
 *	(e.g. message type) passed as additional user data.  The message
 *	send trace records are like the "ordinary" user event trace record
 *	defined by the instrumentation library except that in place of
 *	user data the message type, node, and pid arguments are substituted.
 *	The message receive trace records replace the user data field with
 *	the message type select argument.  There are separate types of
 *	message send and receive trace records for both the blocking
 *	and asynchronous message passing functions.  In addition, each
 *	of these trace events has a corresponding "end of event" family,
 *	with a trace record indicating the duration of the message-
 *	passing interface routine call.
 */

#include "Assert.h"
#include "SystemDepend.h"
#include "SDDFparam.h"
#include "TraceParam.h"
#include "Trace.h"
#include "MesgTrace.h"

/*
 * It is explicitly assumed that if the message send/receive end
 * events are used at all, then they always occur immediately following
 * their corresponding message send/receive begin events, with no other
 * intervening message begin event of the same kind.  Message begin and
 * end events may not be stacked.
 */

CLOCK	lastSyncSendBegin = NOSUCHTIME;
CLOCK	lastSyncRecvBegin = NOSUCHTIME;
CLOCK	lastAsyncSendBegin = NOSUCHTIME;
CLOCK	lastAsyncRecvBegin = NOSUCHTIME;
CLOCK	lastMesgOpBegin = NOSUCHTIME;


/*
 * SUBROUTINE INITMESGTRACE (C function initMesgTrace):
 *	This function initializes data structures specific to
 *	the message send/receive tracing extensions of the Pablo
 *	instrumentation library.  The arguments to initMesgTrace
 *	are two vectors of event IDs, those which mark the beginning of
 *	(entry to) a message passing interface routine, and those which
 *	mark the end of (exit from) the same routines.  Each of these
 *	two vectors contains exactly 5 elements, corresponding IN ORDER
 *	to synchronous (i.e. blocking) send message passing events,
 *	synchronous receive message events, asynchronous (nonblocking)
 *	send message events, asynchronous receive message events, and
 *	miscellaneous message-passing operation events.
 *	No event ID may appear more than once in these vectors.
 *
 *	Sample Fortran invocation:
 *	    INTEGER BEGINIDS(5),ENDIDS(5)
 *	    INTEGER SYNCSENDBEGINID,SYNCSENDENDID
 *	    INTEGER SYNCRECVBEGINID,SYNCRECVENDID
 *	    INTEGER ASYNCSENDBEGINID,ASYNCSENDENDID
 *	    INTEGER ASYNCRECVBEGINID,ASYNCRECVENDID
 *	    INTEGER MESGOPBEGINID,MESGOPENDID
 *	Following initialization of these event IDs:
 *	    DATA BEGINIDS/SYNCSENDBEGINID,SYNCRECVBEGINID,
 *	   +              ASYNCSENDBEGINID,ASYNCRECVBEGINID,
 *	   +              MESGOPBEGINID/
 *	    DATA ENDIDS/SYNCSENDENDID,SYNCRECVENDID,
 *	   +            ASYNCSENDENDID,ASYNCRECVENDID,
 *	   +            MESGOPENDID/
 *	    CALL INITMESGTRACE(BEGINIDS,ENDIDS);
 *
 *	Sample C invocation:
 *	    int syncSendBeginID, syncSendEndID;
 *	    int syncRecvBeginID, syncRecvEndID;
 *	    int asyncSendBeginID, asyncSendEndID;
 *	    int asyncRecvBeginID, asyncRecvEndID;
 *	    int mesgOpBeginID, mesgOpEndID;
 *	Following initialization of these event IDs:
 *	    int beginIDs[5] = {	syncSendBeginID,
 *				syncRecvBeginID,
 *				asyncSendBeginID,
 *				asyncRecvBeginID,
 *				mesgOpBeginID };
 *	    int endIDs[5] = {	syncSendEndID,
 *				syncRecvEndID,
 *				asyncSendEndID,
 *				asyncRecvEndID,
 *				mesgOpEndID };
 *	    initMesgTrace( beginIDs, endIDs );
 */

initmesgtrace_( beginIDs, endIDs )

int	*beginIDs, *endIDs;
{
	return initMesgTrace( beginIDs, endIDs );
}


initMesgTrace(  beginIDs, endIDs )

int	*beginIDs, *endIDs;
{
	extern TR_RECORD	*syncSendBeginEventRecord();
	extern TR_RECORD	*syncRecvBeginEventRecord();
	extern TR_RECORD	*asyncSendBeginEventRecord();
	extern TR_RECORD	*asyncRecvBeginEventRecord();
	extern TR_RECORD	*mesgOpBeginEventRecord();
	extern TR_RECORD	*syncSendEndEventRecord();
	extern TR_RECORD	*syncRecvEndEventRecord();
	extern TR_RECORD	*asyncSendEndEventRecord();
	extern TR_RECORD	*asyncRecvEndEventRecord();
	extern TR_RECORD	*mesgOpEndEventRecord();

	if (( beginIDs == (int *) 0 ) || ( endIDs == (int *) 0 ))
		return FAILURE;

	/* Perform message-tracing extension pre-initialization, if not	    */
	/* already done.						    */

	preInitMesgTrace();

	/* Configure the trace record-generating function for both the	    */
	/* blocking and nonblocking message send and receive begin events.  */

	setEventRecordFunction( beginIDs[ 0 ], syncSendBeginEventRecord );
	setEventRecordFunction( beginIDs[ 1 ], syncRecvBeginEventRecord );
	setEventRecordFunction( beginIDs[ 2 ], asyncSendBeginEventRecord );
	setEventRecordFunction( beginIDs[ 3 ], asyncRecvBeginEventRecord );
	setEventRecordFunction( beginIDs[ 4 ], mesgOpBeginEventRecord );

	/* Configure the trace record-generating function for both the	    */
	/* blocking and nonblocking message send and receive end events.    */

	setEventRecordFunction( endIDs[ 0 ], syncSendEndEventRecord );
	setEventRecordFunction( endIDs[ 1 ], syncRecvEndEventRecord );
	setEventRecordFunction( endIDs[ 2 ], asyncSendEndEventRecord );
	setEventRecordFunction( endIDs[ 3 ], asyncRecvEndEventRecord );
	setEventRecordFunction( endIDs[ 4 ], mesgOpEndEventRecord );

#ifdef DEBUG
	fprintf( debugFile, "initMesgTrace done\n" );
	fflush( debugFile );
#endif /* DEBUG */

	return SUCCESS;
}


/*
 *	preInitMesgTrace:
 *	   This function calls the trace library interface function
 *	   setRecordDescriptor, which records the address of the
 *	   procedure that generates the record descriptors for the
 *	   message passing trace event families.  It is automatically
 *	   invoked by initMesgTrace.  However, the user may elect to
 *	   explicitly call preInitMesgTrace before initMesgTrace does.
 *	   Since initMesgTrace calls the setEventRecordFunction interface
 *	   function of the trace library, that will cause the trace
 *	   library to perform its basic initialization.  One of the
 *	   effects of trace library basic initialization is to output
 *	   record descriptors for all event families defined up to
 *	   that point, by invoking all of the functions passed as
 *	   arguments to setRecordDescriptor, and then to output the
 *	   predefined, built-in record descriptors for internal event
 *	   families.  If no user-defined record descriptors beyond
 *	   those needed by this message-passing trace extension are
 *	   to be used, then preInitMesgTrace need not be invoked.
 *	   However, if there are other such record descriptors (such
 *	   as an application also using the procedure tracing extension)
 *	   and if the user desires all these record descriptors to be
 *	   output before the trace library's internal event family
 *	   record descriptors, then all the required setRecordDescriptor
 *	   interface calls must be done before the trace library basic
 *	   initialization is performed.  preInitMesgTrace may be used
 *	   for this very purpose.
 */

preinitmesgtrace_()
{
	preInitMesgTrace();
}

preInitMesgTrace()
{
	extern int	writeMesgRecordDescriptors();
	static int	preInitDone = FALSE;

	if ( preInitDone == TRUE )
		return;

	/* Give the instrumentation library a pointer to the function	    */
	/* in which we output the specialized record descriptors for	    */
	/* message sends and receives.					    */

	setRecordDescriptor( writeMesgRecordDescriptors );

	preInitDone = TRUE;

	return;
}


/*
 *	writeMesgRecordDescriptors:
 *	   This function generates the record descriptors for the
 *	   message send/receive event families.  It will be invoked
 *	   by the instrumentation library initialization routines.
 *	   Patterned after instrumentation library internal function
 *	   writeRecordDescriptors.
 */

writeMesgRecordDescriptors()
{
	TR_RECORD	*recordPointer, *mesgSDDFdescriptor();

#ifdef DEBUG
	fprintf( debugFile, "writeMesgRecordDescriptors\n" );
	fflush( debugFile );
#endif /* DEBUG */

	recordPointer = mesgSDDFdescriptor( FAMILY_SYNCSEND_BEGIN,
					    RECORD_TRACE,
					    "Blocking Send Begin",
					    "Blocking Message Send Begin" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = mesgSDDFdescriptor( FAMILY_SYNCRECV_BEGIN,
					    RECORD_TRACE,
					    "Blocking Receive Begin",
					    "Blocking Message Receive Begin" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = mesgSDDFdescriptor( FAMILY_ASYNCSEND_BEGIN,
					    RECORD_TRACE,
					    "Nonblocking Send Begin",
					    "Nonblocking Message Send Begin" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = mesgSDDFdescriptor( FAMILY_ASYNCRECV_BEGIN,
					    RECORD_TRACE,
					    "Nonblocking Receive Begin",
					    "Nonblocking Message Receive Begin" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = mesgSDDFdescriptor( FAMILY_MESGOP_BEGIN,
					    RECORD_TRACE,
					    "Message Operation Begin",
					    "Miscellaneous Message Operation Begin" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );


	recordPointer = mesgSDDFdescriptor( FAMILY_SYNCSEND_END,
					    RECORD_TRACE,
					    "Blocking Send End",
					    "Blocking Message Send End" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = mesgSDDFdescriptor( FAMILY_SYNCRECV_END,
					    RECORD_TRACE,
					    "Blocking Receive End",
					    "Blocking Message Receive End" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = mesgSDDFdescriptor( FAMILY_ASYNCSEND_END,
					    RECORD_TRACE,
					    "Nonblocking Send End",
					    "Nonblocking Message Send End" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = mesgSDDFdescriptor( FAMILY_ASYNCRECV_END,
					    RECORD_TRACE,
					    "Nonblocking Receive End",
					    "Nonblocking Message Receive End" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = mesgSDDFdescriptor( FAMILY_MESGOP_END,
					    RECORD_TRACE,
					    "Message Operation End",
					    "Miscellaneous Message Operation End" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );


	return;
}


/*
 *	mesgSDDFdescriptor:
 *	   Generate a SDDF binary format record descriptor for the
 *	   full trace class of events in any of the message send
 *	   or message receive families of events.  Patterned after the
 *	   instrumentation library internal function sddfDescriptor.
 */

TR_RECORD *
mesgSDDFdescriptor( recordFamily, recordType, recordName, recordDescription )

int	recordFamily;
int	recordType;
char	*recordName;
char	*recordDescription;
{
	static TR_RECORD	traceRecord;
	static char		recordBuffer[ 1024 ];
	char			*recordPointer;
	char			*stringPointer;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordFamily == FAMILY_SYNCSEND_BEGIN ) ||
	       ( recordFamily == FAMILY_SYNCRECV_BEGIN ) ||
	       ( recordFamily == FAMILY_ASYNCSEND_BEGIN ) ||
	       ( recordFamily == FAMILY_ASYNCRECV_BEGIN ) ||
	       ( recordFamily == FAMILY_MESGOP_BEGIN ) ||
	       ( recordFamily == FAMILY_SYNCSEND_END ) ||
	       ( recordFamily == FAMILY_SYNCRECV_END ) ||
	       ( recordFamily == FAMILY_ASYNCSEND_END ) ||
	       ( recordFamily == FAMILY_ASYNCRECV_END ) ||
	       ( recordFamily == FAMILY_MESGOP_END )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordFamily == FAMILY_SYNCSEND_BEGIN ) || \
	       ( recordFamily == FAMILY_SYNCRECV_BEGIN ) || \
	       ( recordFamily == FAMILY_ASYNCSEND_BEGIN ) || \
	       ( recordFamily == FAMILY_ASYNCRECV_BEGIN ) || \
	       ( recordFamily == FAMILY_MESGOP_BEGIN ) || \
	       ( recordFamily == FAMILY_SYNCSEND_END ) || \
	       ( recordFamily == FAMILY_SYNCRECV_END ) || \
	       ( recordFamily == FAMILY_ASYNCSEND_END ) || \
	       ( recordFamily == FAMILY_ASYNCRECV_END ) || \
	       ( recordFamily == FAMILY_MESGOP_END ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

	Assert( recordType == RECORD_TRACE );

	/* Allow space at the beginning of the record for the packet	    */
	/* length, which will be computed after the packet is complete.	    */

	recordPointer = recordBuffer;
	sddfWriteInteger( &recordPointer, 0 );

	/* Write the record type descriptor.				    */

	sddfWriteInteger( &recordPointer, PKT_DESCRIPTOR );

	/* Write the record tag descriptor.				    */

	sddfWriteInteger( &recordPointer, ( recordFamily | recordType ));

	/* Write the record name.					    */

	sddfWriteString( &recordPointer, recordName );

	/* Write the record attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the record attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "description" );

	sddfWriteString( &recordPointer, recordDescription );

	/* Write the record field count.				    */

	switch ( recordFamily ) {

	case FAMILY_MESGOP_BEGIN:
		sddfWriteInteger( &recordPointer, 5 );
		break;

	case FAMILY_SYNCRECV_BEGIN:
	case FAMILY_ASYNCRECV_BEGIN:
	case FAMILY_SYNCSEND_END:
	case FAMILY_ASYNCSEND_END:
	case FAMILY_ASYNCRECV_END:
		sddfWriteInteger( &recordPointer, 6 );
		break;

	case FAMILY_SYNCRECV_END:
	case FAMILY_MESGOP_END:
		sddfWriteInteger( &recordPointer, 7 );
		break;

	case FAMILY_SYNCSEND_BEGIN:
	case FAMILY_ASYNCSEND_BEGIN:
		sddfWriteInteger( &recordPointer, 8 );
		break;
	}


	/* FIELD 1:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Timestamp" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Time" );

	sddfWriteString( &recordPointer, "Timestamp" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 1 );


	/* FIELD 2:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Seconds" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Seconds" );

	sddfWriteString( &recordPointer, "Floating Point Timestamp" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, DOUBLE );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 3:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Event Identifier" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "ID" );

	sddfWriteString( &recordPointer, "Event ID" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 4:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Processor Number" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Node" );

	sddfWriteString( &recordPointer, "Processor number" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 5:							    */

	if (( recordFamily == FAMILY_SYNCSEND_END ) ||
	    ( recordFamily == FAMILY_SYNCRECV_END ) ||
	    ( recordFamily == FAMILY_ASYNCSEND_END ) ||
	    ( recordFamily == FAMILY_ASYNCRECV_END ) ||
	    ( recordFamily == FAMILY_MESGOP_END )) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Message Seconds" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Message Seconds" );

		sddfWriteString( &recordPointer, "Floating Point Message Duration" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, DOUBLE );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );

	}


	/* FIELD 6 (5 for the BEGIN record families):			    */

	if (( recordFamily == FAMILY_SYNCSEND_BEGIN ) ||
	    ( recordFamily == FAMILY_SYNCRECV_BEGIN ) ||
	    ( recordFamily == FAMILY_ASYNCSEND_BEGIN ) ||
	    ( recordFamily == FAMILY_ASYNCRECV_BEGIN )) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Message Type" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Type" );

		sddfWriteString( &recordPointer, "Message Type" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );

	} else if ( recordFamily == FAMILY_MESGOP_BEGIN ) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Operation Type" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Operation" );

		sddfWriteString( &recordPointer, "Operation Type" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );

	} else {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Message Duration" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Duration" );

		sddfWriteString( &recordPointer, "Message Duration" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 1 );
	}


	/* FIELD 7 (6 for the BEGIN record families):			    */

	if (( recordFamily == FAMILY_SYNCSEND_BEGIN ) ||
	    ( recordFamily == FAMILY_SYNCRECV_BEGIN ) ||
	    ( recordFamily == FAMILY_ASYNCSEND_BEGIN ) ||
	    ( recordFamily == FAMILY_ASYNCRECV_BEGIN )) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Message Length" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Length" );

		sddfWriteString( &recordPointer, "Message Length" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );

	} else if ( recordFamily == FAMILY_SYNCRECV_END ) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Sending processor number" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Source node" );

		sddfWriteString( &recordPointer, "Sending processor number" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );

	} else if ( recordFamily == FAMILY_MESGOP_END ) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Operation Type" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Operation" );

		sddfWriteString( &recordPointer, "Operation Type" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );
	}


	/* FIELD 7:							    */

	if (( recordFamily == FAMILY_SYNCSEND_BEGIN ) ||
	    ( recordFamily == FAMILY_ASYNCSEND_BEGIN )) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Destination Node" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "DestNode" );

		sddfWriteString( &recordPointer, "Destination Node" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );
	}


	/* FIELD 8:							    */

	if (( recordFamily == FAMILY_SYNCSEND_BEGIN ) ||
	    ( recordFamily == FAMILY_ASYNCSEND_BEGIN )) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Destination PID" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "PID" );

		sddfWriteString( &recordPointer, "Destination PID" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );
	}


	/* The entire record descriptor packet has been written.	    */

	traceRecord.recordLength = recordPointer - recordBuffer;
	traceRecord.recordContents = recordBuffer;

	/* Finally, copy the completed packet length at the beginning	    */
	/* of the record itself.					    */

	recordPointer = recordBuffer;
	sddfWriteInteger( &recordPointer, traceRecord.recordLength );

	return & traceRecord;
}


/*
 *	syncSendBeginEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce blocking message send event family trace records.
 *	   Patterned after the instrumentation library internal functions
 *	   externalEventRecord and sddfRecord.
 */

TR_RECORD *
syncSendBeginEventRecord( recordType, eventPointer, timeStamp,
			  dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct syncSendBeginRecordData	syncSendBeginRecordHeader;

	/* This structure defines how the send arguments are passed in	    */
	/* as user data, pointed to by dataPointer.			    */

	struct syncSendArgs {
		long	type;
		long	length;
		long	node;
		long	pid;
	}					*syncSendArguments;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) &&
	       ( eventPointer != (TR_EVENT *) 0 ) &&
	       ( dataLength == sizeof(struct syncSendArgs) ) &&
	       ( dataPointer != (char *) 0 )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) && \
	       ( eventPointer != (TR_EVENT *) 0 ) && \
	       ( dataLength == sizeof(struct syncSendArgs) ) && \
	       ( dataPointer != (char *) 0 ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

#ifdef DEBUG
	fprintf( debugFile, "syncSendBeginEventRecord\n" );
	fflush( debugFile );
#endif /* DEBUG */

	syncSendArguments = (struct syncSendArgs *) dataPointer;

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	lastSyncSendBegin = timeStamp;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.  This is constant.		    */

	traceRecord.recordLength = sizeof syncSendBeginRecordHeader;

	traceRecord.recordContents = (char *) &syncSendBeginRecordHeader;

	/* Load the trace record fields into the allocated buffer	    */

	syncSendBeginRecordHeader.packetLength = traceRecord.recordLength;
	syncSendBeginRecordHeader.packetType = PKT_DATA;
	syncSendBeginRecordHeader.packetTag = FAMILY_SYNCSEND_BEGIN |
						recordType;
	syncSendBeginRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	syncSendBeginRecordHeader.timeStamp = timeStamp;
	syncSendBeginRecordHeader.seconds = clockToSeconds( timeStamp );
	syncSendBeginRecordHeader.eventID = eventPointer->eventID;
	syncSendBeginRecordHeader.nodeNumber = TRgetNode();
	syncSendBeginRecordHeader.messageType = syncSendArguments->type;
	syncSendBeginRecordHeader.messageLength = syncSendArguments->length;
	syncSendBeginRecordHeader.destNode = syncSendArguments->node;
	syncSendBeginRecordHeader.messagePid = syncSendArguments->pid;

	return & traceRecord;
}


/*
 *	syncRecvBeginEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce blocking message receive event family trace records.
 *	   Patterned after the instrumentation library internal functions
 *	   externalEventRecord and sddfRecord.
 */

TR_RECORD *
syncRecvBeginEventRecord( recordType, eventPointer, timeStamp,
		     dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct syncRecvBeginRecordData	syncRecvBeginRecordHeader;

	/* This structure defines how the receive arguments are passed in   */
	/* as user data, pointed to by dataPointer.			    */

	struct syncRecvArgs {
		long	type;
		long	length;
	}					*syncRecvArguments;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) &&
	       ( eventPointer != (TR_EVENT *) 0 ) &&
	       ( dataLength == sizeof(struct syncRecvArgs) ) &&
	       ( dataPointer != (char *) 0 )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) && \
	       ( eventPointer != (TR_EVENT *) 0 ) && \
	       ( dataLength == sizeof(struct syncRecvArgs) ) && \
	       ( dataPointer != (char *) 0 ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

#ifdef DEBUG
	fprintf( debugFile, "syncRecvBeginEventRecord\n" );
	fflush( debugFile );
#endif /* DEBUG */

	syncRecvArguments = (struct syncRecvArgs *) dataPointer;

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	lastSyncRecvBegin = timeStamp;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.  This is constant.		    */

	traceRecord.recordLength = sizeof syncRecvBeginRecordHeader;

	traceRecord.recordContents = (char *) &syncRecvBeginRecordHeader;

	/* Load the trace record fields into the allocated buffer	    */

	syncRecvBeginRecordHeader.packetLength = traceRecord.recordLength;
	syncRecvBeginRecordHeader.packetType = PKT_DATA;
	syncRecvBeginRecordHeader.packetTag = FAMILY_SYNCRECV_BEGIN |
						recordType;
	syncRecvBeginRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	syncRecvBeginRecordHeader.timeStamp = timeStamp;
	syncRecvBeginRecordHeader.seconds = clockToSeconds( timeStamp );
	syncRecvBeginRecordHeader.eventID = eventPointer->eventID;
	syncRecvBeginRecordHeader.nodeNumber = TRgetNode();
	syncRecvBeginRecordHeader.messageType = syncRecvArguments->type;
	syncRecvBeginRecordHeader.messageLength = syncRecvArguments->length;

	return & traceRecord;
}


/*
 *	asyncSendBeginEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce nonblocking message send event family trace records.
 *	   Patterned after the instrumentation library internal functions
 *	   externalEventRecord and sddfRecord.
 */

TR_RECORD *
asyncSendBeginEventRecord( recordType, eventPointer, timeStamp,
		      dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct asyncSendBeginRecordData	asyncSendBeginRecordHeader;

	/* This structure defines how the send arguments are passed in	    */
	/* as user data, pointed to by dataPointer.			    */

	struct asyncSendArgs {
		long	type;
		long	length;
		long	node;
		long	pid;
	}					*asyncSendArguments;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) &&
	       ( eventPointer != (TR_EVENT *) 0 ) &&
	       ( dataLength == sizeof(struct asyncSendArgs) ) &&
	       ( dataPointer != (char *) 0 )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) && \
	       ( eventPointer != (TR_EVENT *) 0 ) && \
	       ( dataLength == sizeof(struct asyncSendArgs) ) && \
	       ( dataPointer != (char *) 0 ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

#ifdef DEBUG
	fprintf( debugFile, "asyncSendBeginEventRecord\n" );
	fflush( debugFile );
#endif /* DEBUG */

	asyncSendArguments = (struct asyncSendArgs *) dataPointer;

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	lastAsyncSendBegin = timeStamp;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.  This is constant.		    */

	traceRecord.recordLength = sizeof asyncSendBeginRecordHeader;

	traceRecord.recordContents = (char *) &asyncSendBeginRecordHeader;

	/* Load the trace record fields into the allocated buffer	    */

	asyncSendBeginRecordHeader.packetLength = traceRecord.recordLength;
	asyncSendBeginRecordHeader.packetType = PKT_DATA;
	asyncSendBeginRecordHeader.packetTag = FAMILY_ASYNCSEND_BEGIN |
						recordType;
	asyncSendBeginRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	asyncSendBeginRecordHeader.timeStamp = timeStamp;
	asyncSendBeginRecordHeader.seconds = clockToSeconds( timeStamp );
	asyncSendBeginRecordHeader.eventID = eventPointer->eventID;
	asyncSendBeginRecordHeader.nodeNumber = TRgetNode();
	asyncSendBeginRecordHeader.messageType = asyncSendArguments->type;
	asyncSendBeginRecordHeader.messageLength = asyncSendArguments->length;
	asyncSendBeginRecordHeader.destNode = asyncSendArguments->node;
	asyncSendBeginRecordHeader.messagePid = asyncSendArguments->pid;

	return & traceRecord;
}


/*
 *	asyncRecvBeginEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce nonblocking message receive event family trace records.
 *	   Patterned after the instrumentation library internal functions
 *	   externalEventRecord and sddfRecord.
 */

TR_RECORD *
asyncRecvBeginEventRecord( recordType, eventPointer, timeStamp,
		      dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct asyncRecvBeginRecordData	asyncRecvBeginRecordHeader;

	/* This structure defines how the receive arguments are passed in   */
	/* as user data, pointed to by dataPointer.			    */

	struct asyncRecvArgs {
		long	type;
		long	length;
	}					*asyncRecvArguments;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) &&
	       ( eventPointer != (TR_EVENT *) 0 ) &&
	       ( dataLength == sizeof(struct asyncRecvArgs) ) &&
	       ( dataPointer != (char *) 0 )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) && \
	       ( eventPointer != (TR_EVENT *) 0 ) && \
	       ( dataLength == sizeof(struct asyncRecvArgs) ) && \
	       ( dataPointer != (char *) 0 ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

#ifdef DEBUG
	fprintf( debugFile, "asyncRecvBeginEventRecord\n" );
	fflush( debugFile );
#endif /* DEBUG */

	asyncRecvArguments = (struct asyncRecvArgs *) dataPointer;

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	lastAsyncRecvBegin = timeStamp;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.  This is constant.		    */

	traceRecord.recordLength = sizeof asyncRecvBeginRecordHeader;

	traceRecord.recordContents = (char *) &asyncRecvBeginRecordHeader;

	/* Load the trace record fields into the allocated buffer	    */

	asyncRecvBeginRecordHeader.packetLength = traceRecord.recordLength;
	asyncRecvBeginRecordHeader.packetType = PKT_DATA;
	asyncRecvBeginRecordHeader.packetTag = FAMILY_ASYNCRECV_BEGIN |
						recordType;
	asyncRecvBeginRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	asyncRecvBeginRecordHeader.timeStamp = timeStamp;
	asyncRecvBeginRecordHeader.seconds = clockToSeconds( timeStamp );
	asyncRecvBeginRecordHeader.eventID = eventPointer->eventID;
	asyncRecvBeginRecordHeader.nodeNumber = TRgetNode();
	asyncRecvBeginRecordHeader.messageType = asyncRecvArguments->type;
	asyncRecvBeginRecordHeader.messageLength = asyncRecvArguments->length;

	return & traceRecord;
}


/*
 *	mesgOpBeginEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce miscellanous message event family trace records.
 *	   Patterned after the instrumentation library internal functions
 *	   externalEventRecord and sddfRecord.
 */

TR_RECORD *
mesgOpBeginEventRecord( recordType, eventPointer, timeStamp,
			dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct mesgOpBeginRecordData	mesgOpBeginRecordHeader;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) &&
	       ( eventPointer != (TR_EVENT *) 0 ) &&
	       ( dataLength == sizeof(int) ) &&
	       ( dataPointer != (char *) 0 )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) && \
	       ( eventPointer != (TR_EVENT *) 0 ) && \
	       ( dataLength == sizeof(int) ) && \
	       ( dataPointer != (char *) 0 ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

#ifdef DEBUG
	fprintf( debugFile, "mesgOpBeginEventRecord\n" );
	fflush( debugFile );
#endif /* DEBUG */

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	lastMesgOpBegin = timeStamp;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.  This is constant.		    */

	traceRecord.recordLength = sizeof mesgOpBeginRecordHeader;

	traceRecord.recordContents = (char *) &mesgOpBeginRecordHeader;

	/* Load the trace record fields into the allocated buffer	    */

	mesgOpBeginRecordHeader.packetLength = traceRecord.recordLength;
	mesgOpBeginRecordHeader.packetType = PKT_DATA;
	mesgOpBeginRecordHeader.packetTag = FAMILY_MESGOP_BEGIN | recordType;
	mesgOpBeginRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	mesgOpBeginRecordHeader.timeStamp = timeStamp;
	mesgOpBeginRecordHeader.seconds = clockToSeconds( timeStamp );
	mesgOpBeginRecordHeader.eventID = eventPointer->eventID;
	mesgOpBeginRecordHeader.nodeNumber = TRgetNode();
	mesgOpBeginRecordHeader.operationType = * (int *) dataPointer;

	return & traceRecord;
}



/*
 *	syncSendEndEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce blocking message send event family trace records.
 *	   Patterned after the instrumentation library internal functions
 *	   externalEventRecord and sddfRecord.
 */

TR_RECORD *
syncSendEndEventRecord( recordType, eventPointer, timeStamp,
		        dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct syncSendEndRecordData	syncSendEndRecordHeader;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) &&
	       ( eventPointer != (TR_EVENT *) 0 ) &&
	       ( dataLength == 0 )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) && \
	       ( eventPointer != (TR_EVENT *) 0 ) && \
	       ( dataLength == 0 ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

#ifdef DEBUG
	fprintf( debugFile, "syncSendEndEventRecord\n" );
	fflush( debugFile );
#endif /* DEBUG */

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.  This is constant.		    */

	traceRecord.recordLength = sizeof syncSendEndRecordHeader;

	traceRecord.recordContents = (char *) &syncSendEndRecordHeader;

	/* Load the trace record fields into the allocated buffer	    */

	syncSendEndRecordHeader.packetLength = traceRecord.recordLength;
	syncSendEndRecordHeader.packetType = PKT_DATA;
	syncSendEndRecordHeader.packetTag = FAMILY_SYNCSEND_END |
						recordType;
	syncSendEndRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	syncSendEndRecordHeader.timeStamp = timeStamp;
	syncSendEndRecordHeader.seconds = clockToSeconds( timeStamp );
	syncSendEndRecordHeader.eventID = eventPointer->eventID;
	syncSendEndRecordHeader.nodeNumber = TRgetNode();
	syncSendEndRecordHeader.durationDim = sizeof(CLOCK)/sizeof(int);
	syncSendEndRecordHeader.mesgDuration =
		( clockCompare( lastSyncSendBegin, noSuchClock ) == 0 )
		? noSuchClock : clockSubtract( timeStamp, lastSyncSendBegin );
	syncSendEndRecordHeader.mesgSeconds =
		clockToSeconds( syncSendEndRecordHeader.mesgDuration );

	lastSyncSendBegin = noSuchClock;

	return & traceRecord;
}


/*
 *	syncRecvEndEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce blocking message receive event family trace records.
 *	   Patterned after the instrumentation library internal functions
 *	   externalEventRecord and sddfRecord.
 */

TR_RECORD *
syncRecvEndEventRecord( recordType, eventPointer, timeStamp,
		        dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct syncRecvEndRecordData	syncRecvEndRecordHeader;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) &&
	       ( eventPointer != (TR_EVENT *) 0 ) &&
	       ( dataLength == sizeof(int) )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) && \
	       ( eventPointer != (TR_EVENT *) 0 ) && \
	       ( dataLength == sizeof(int) ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

#ifdef DEBUG
	fprintf( debugFile, "syncRecvEndEventRecord\n" );
	fflush( debugFile );
#endif /* DEBUG */

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.  This is constant.		    */

	traceRecord.recordLength = sizeof syncRecvEndRecordHeader;

	traceRecord.recordContents = (char *) &syncRecvEndRecordHeader;

	/* Load the trace record fields into the allocated buffer	    */

	syncRecvEndRecordHeader.packetLength = traceRecord.recordLength;
	syncRecvEndRecordHeader.packetType = PKT_DATA;
	syncRecvEndRecordHeader.packetTag = FAMILY_SYNCRECV_END |
						recordType;
	syncRecvEndRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	syncRecvEndRecordHeader.timeStamp = timeStamp;
	syncRecvEndRecordHeader.seconds = clockToSeconds( timeStamp );
	syncRecvEndRecordHeader.eventID = eventPointer->eventID;
	syncRecvEndRecordHeader.nodeNumber = TRgetNode();
	syncRecvEndRecordHeader.durationDim = sizeof(CLOCK)/sizeof(int);
	syncRecvEndRecordHeader.mesgDuration =
		( clockCompare( lastSyncRecvBegin, noSuchClock ) == 0 )
		? noSuchClock : clockSubtract( timeStamp, lastSyncRecvBegin );
	syncRecvEndRecordHeader.mesgSeconds =
		clockToSeconds( syncRecvEndRecordHeader.mesgDuration );
	syncRecvEndRecordHeader.sourceNode = * (int *) dataPointer;

	lastSyncRecvBegin = noSuchClock;

	return & traceRecord;
}


/*
 *	asyncSendEndEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce nonblocking message send event family trace records.
 *	   Patterned after the instrumentation library internal functions
 *	   externalEventRecord and sddfRecord.
 */

TR_RECORD *
asyncSendEndEventRecord( recordType, eventPointer, timeStamp,
		         dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct asyncSendEndRecordData	asyncSendEndRecordHeader;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) &&
	       ( eventPointer != (TR_EVENT *) 0 ) &&
	       ( dataLength == 0 )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) && \
	       ( eventPointer != (TR_EVENT *) 0 ) && \
	       ( dataLength == 0 ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

#ifdef DEBUG
	fprintf( debugFile, "asyncSendEndEventRecord\n" );
	fflush( debugFile );
#endif /* DEBUG */

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.  This is constant.		    */

	traceRecord.recordLength = sizeof asyncSendEndRecordHeader;

	traceRecord.recordContents = (char *) &asyncSendEndRecordHeader;

	/* Load the trace record fields into the allocated buffer	    */

	asyncSendEndRecordHeader.packetLength = traceRecord.recordLength;
	asyncSendEndRecordHeader.packetType = PKT_DATA;
	asyncSendEndRecordHeader.packetTag = FAMILY_ASYNCSEND_END |
						recordType;
	asyncSendEndRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	asyncSendEndRecordHeader.timeStamp = timeStamp;
	asyncSendEndRecordHeader.seconds = clockToSeconds( timeStamp );
	asyncSendEndRecordHeader.eventID = eventPointer->eventID;
	asyncSendEndRecordHeader.nodeNumber = TRgetNode();
	asyncSendEndRecordHeader.durationDim = sizeof(CLOCK)/sizeof(int);
	asyncSendEndRecordHeader.mesgDuration =
		( clockCompare( lastAsyncSendBegin, noSuchClock ) == 0 )
		? noSuchClock : clockSubtract( timeStamp, lastAsyncSendBegin );
	asyncSendEndRecordHeader.mesgSeconds =
		clockToSeconds( asyncSendEndRecordHeader.mesgDuration );

	lastAsyncSendBegin = noSuchClock;

	return & traceRecord;
}


/*
 *	asyncRecvEndEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce nonblocking message receive event family trace records.
 *	   Patterned after the instrumentation library internal functions
 *	   externalEventRecord and sddfRecord.
 */

TR_RECORD *
asyncRecvEndEventRecord( recordType, eventPointer, timeStamp,
		         dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct asyncRecvEndRecordData	asyncRecvEndRecordHeader;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) &&
	       ( eventPointer != (TR_EVENT *) 0 ) &&
	       ( dataLength == 0 )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) && \
	       ( eventPointer != (TR_EVENT *) 0 ) && \
	       ( dataLength == 0 ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

#ifdef DEBUG
	fprintf( debugFile, "asyncRecvEndEventRecord\n" );
	fflush( debugFile );
#endif /* DEBUG */

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.  This is constant.		    */

	traceRecord.recordLength = sizeof asyncRecvEndRecordHeader;

	traceRecord.recordContents = (char *) &asyncRecvEndRecordHeader;

	/* Load the trace record fields into the allocated buffer	    */

	asyncRecvEndRecordHeader.packetLength = traceRecord.recordLength;
	asyncRecvEndRecordHeader.packetType = PKT_DATA;
	asyncRecvEndRecordHeader.packetTag = FAMILY_ASYNCRECV_END |
						recordType;
	asyncRecvEndRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	asyncRecvEndRecordHeader.timeStamp = timeStamp;
	asyncRecvEndRecordHeader.seconds = clockToSeconds( timeStamp );
	asyncRecvEndRecordHeader.eventID = eventPointer->eventID;
	asyncRecvEndRecordHeader.nodeNumber = TRgetNode();
	asyncRecvEndRecordHeader.durationDim = sizeof(CLOCK)/sizeof(int);
	asyncRecvEndRecordHeader.mesgDuration =
		( clockCompare( lastAsyncRecvBegin, noSuchClock ) == 0 )
		? noSuchClock : clockSubtract( timeStamp, lastAsyncRecvBegin );
	asyncRecvEndRecordHeader.mesgSeconds =
		clockToSeconds( asyncRecvEndRecordHeader.mesgDuration );

	lastAsyncRecvBegin = noSuchClock;

	return & traceRecord;
}


/*
 *	mesgOpEndEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce miscellanous message event family trace records.
 *	   Patterned after the instrumentation library internal functions
 *	   externalEventRecord and sddfRecord.
 */

TR_RECORD *
mesgOpEndEventRecord( recordType, eventPointer, timeStamp,
		      dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD			traceRecord;
	static struct mesgOpEndRecordData	mesgOpEndRecordHeader;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) &&
	       ( eventPointer != (TR_EVENT *) 0 ) &&
	       ( dataLength == sizeof(int) ) &&
	       ( dataPointer != (char *) 0 )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) && \
	       ( eventPointer != (TR_EVENT *) 0 ) && \
	       ( dataLength == sizeof(int) ) && \
	       ( dataPointer != (char *) 0 ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

#ifdef DEBUG
	fprintf( debugFile, "mesgOpEndEventRecord\n" );
	fflush( debugFile );
#endif /* DEBUG */

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.  This is constant.		    */

	traceRecord.recordLength = sizeof mesgOpEndRecordHeader;

	traceRecord.recordContents = (char *) &mesgOpEndRecordHeader;

	/* Load the trace record fields into the allocated buffer	    */

	mesgOpEndRecordHeader.packetLength = traceRecord.recordLength;
	mesgOpEndRecordHeader.packetType = PKT_DATA;
	mesgOpEndRecordHeader.packetTag = FAMILY_MESGOP_END | recordType;
	mesgOpEndRecordHeader.clockDimension = sizeof(CLOCK)/sizeof(int);
	mesgOpEndRecordHeader.timeStamp = timeStamp;
	mesgOpEndRecordHeader.seconds = clockToSeconds( timeStamp );
	mesgOpEndRecordHeader.eventID = eventPointer->eventID;
	mesgOpEndRecordHeader.nodeNumber = TRgetNode();
	mesgOpEndRecordHeader.durationDim = sizeof(CLOCK)/sizeof(int);
	mesgOpEndRecordHeader.mesgDuration =
		( clockCompare( lastMesgOpBegin, noSuchClock ) == 0 )
		? noSuchClock : clockSubtract( timeStamp, lastMesgOpBegin );
	mesgOpEndRecordHeader.mesgSeconds =
		clockToSeconds( mesgOpEndRecordHeader.mesgDuration );
	mesgOpEndRecordHeader.operationType = * (int *) dataPointer;

	lastMesgOpBegin = noSuchClock;

	return & traceRecord;
}
