/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 * LoopTrace.c
 *	This file contains extensions to the Pablo instrumentation library
 *	which facilitate loop entry and exit tracing.  This is done by
 *	defining new families of trace records, loop entry and exit.  The
 *	loop entry trace record is like the "ordinary" user event trace
 *	record defined by the instrumentation library except that there is no
 *	field for user data.  The loop exit trace record replaces the
 *	user data field with a loop duration field.  The function which
 *	generates the loop exit records does so by a simple mapping from
 *	loop exit event IDs to the corresponding loop entry event IDs.
 */

#include "Assert.h"
#include "SystemDepend.h"
#include "SDDFparam.h"
#include "TraceParam.h"
#include "Trace.h"

/* Define the masks for the loop entry/exit event families,		    */
/* similarly to FAMILY_EXTERNAL and internal event families in		    */
/* the include file TraceParam.h.					    */

#define FAMILY_LOOPENTRY	0160		/* loop entry		    */
#define FAMILY_LOOPEXIT 	0170		/* loop exit		    */


/* Number of loops being traced by this code				    */
static int	numberLoops = 0;

/* Each loop being traced has associated with it a distinct pair of	    */
/* entry and exit event IDs.  This code maintains a vector of such	    */
/* matchings which permits the ready identification of an event ID as	    */
/* being either an entry event or an exit event and for which loop.	    */

typedef struct loopEventMatch {

	int			entryID;	/* loop entry event ID	    */

	int			exitID;		/* loop exit event ID	    */

} LOOP_EVENTS;

static LOOP_EVENTS	*loopEvents =		/* array of event ID pairs  */
			(LOOP_EVENTS *) 0;

/*
 * For each loop being traced this code maintains a stack of loop entry
 * times.  Each loop entry event causes the corresponding loop's stack to
 * be pushed, each loop exit event causes the corresponding loop's stack
 * to be popped, and from the difference in time between entry and exit
 * the loop duration may be calculated in a very straightforward
 * subtraction.  The above loop entry-exit event ID matching is used to
 * map events to their corresponding loops.
 */

typedef struct loopEntryTime {

	CLOCK			entryTime;	/* when loop entered	    */

	struct loopEntryTime	*nextTime;	/* stack pointer down	    */

} LOOP_ENTRY;

static LOOP_ENTRY	**loopEntryStack =	/* array of pointers to	    */
			(LOOP_ENTRY **) 0;	/* stack top elements	    */

extern CLOCK	popLoopEntry();


/* Define data structure types for loop entry and exit trace records,	    */
/* similarly to record data structures in Trace.h			    */

/* FAMILY_LOOPENTRY family Record Data packets:				    */

struct loopEntryTraceRecordData {

	int	packetLength;		/* bytes in packet		    */

	int	packetType;		/* == PKT_DATA			    */

	int	packetTag;		/* FAMILY_LOOPENTRY | RECORD_TRACE  */

	int	clockDimension;		/* number of ints in a CLOCK	    */

	CLOCK	timeStamp;		/* time record generated	    */

	double	seconds;		/* floating-point timestamp	    */

	int	eventID;		/* ID of corresponding event	    */

	int	nodeNumber;		/* occurred on which node	    */

	int	loopIndex;		/* index into loops table	    */

	long	sourceByte;		/* source code byte offset in file  */

	int	sourceLine;		/* source code line number in file  */
};

struct loopEntryCountRecordData {

	int	packetLength;		/* bytes in packet		    */

	int	packetType;		/* == PKT_DATA			    */

	int	packetTag;		/* FAMILY_LOOPENTRY | RECORD_COUNT  */

	int	clockDimension;		/* number of ints in a CLOCK	    */

	CLOCK	timeStamp;		/* time record generated	    */

	double	seconds;		/* floating-point timestamp	    */

	int	eventID;		/* ID of corresponding event	    */

	int	nodeNumber;		/* occurred on which node	    */

	int	loopIndex;		/* index into loops table	    */

	long	sourceByte;		/* source code byte offset in file  */

	int	sourceLine;		/* source code line number in file  */

	long	eventCount;		/* count of event occurrences	    */
};


/* FAMILY_LOOPEXIT family Record Data packets:				    */

struct loopExitTraceRecordData {

	int	packetLength;		/* bytes in packet		    */

	int	packetType;		/* == PKT_DATA			    */

	int	packetTag;		/* FAMILY_LOOPEXIT | RECORD_TRACE   */

	int	clockDimension;		/* number of ints in a CLOCK	    */

	CLOCK	timeStamp;		/* time record generated	    */

	double	seconds;		/* floating-point timestamp	    */

	int	eventID;		/* ID of corresponding event	    */

	int	nodeNumber;		/* occurred on which node	    */

	int	loopIndex;		/* index into loops table	    */

	long	sourceByte;		/* source code byte offset in file  */

	int	sourceLine;		/* source code line number in file  */

	int	durationDim;		/* number of ints in a CLOCK	    */

	CLOCK	loopDuration;		/* total duration of loop	    */

	double	loopSeconds;		/* floating-point loop duration	    */
};

struct loopExitCountRecordData {

	int	packetLength;		/* bytes in packet		    */

	int	packetType;		/* == PKT_DATA			    */

	int	packetTag;		/* FAMILY_LOOPEXIT | RECORD_COUNT   */

	int	clockDimension;		/* number of ints in a CLOCK	    */

	CLOCK	timeStamp;		/* time record generated	    */

	double	seconds;		/* floating-point timestamp	    */

	int	eventID;		/* ID of corresponding event	    */

	int	nodeNumber;		/* occurred on which node	    */

	int	loopIndex;		/* index into loops table	    */

	long	sourceByte;		/* source code byte offset in file  */

	int	sourceLine;		/* source code line number in file  */

	int	durationDim;		/* number of ints in a CLOCK	    */

	CLOCK	loopDuration;		/* total duration of loop	    */

	double	loopSeconds;		/* floating-point loop duration	    */

	long	eventCount;		/* count of event occurrences	    */
};


/* Define data structures used to contain source code location data for	    */
/* Pablo instrumenting parser-generated code.				    */

static long	loopByteOffset = -1;	/* source code byte offset	    */

static int	loopLineNumber = -1;	/* source code line number	    */



/*
 * SUBROUTINE INITLOOPTRACE (C function initLoopTrace):
 *	This function initializes data structures specific to
 *	the loop entry/exit tracing extensions of the Pablo
 *	instrumentation library.  The argument numLoops specifies
 *	how many loops are to be traced.  The arguments loopEntryID
 *	and loopExitID, respectively, are vectors specifying the event IDs
 *	to be used as entry and exit events for each of the loops.
 *	No event ID may be used as both an entry and an exit event.
 *	Further, there must be a one-to-one mapping between entry and
 *	exit event IDs.  The order of appearance of the event IDs in
 *	the two lists dictates which ones are paired as well as what
 *	loop index is associated with each pair.
 *
 *	Sample Fortran invocation (using 27 as number of loops):
 *		INTEGER ENT(27),EXT(27)
 *		DATA ENT/list of 27 loop entry event IDs/
 *		DATA EXT/list of 27 loop exit event IDs/
 *		CALL INITLOOPTRACE(27,ENT,EXT)
 *	Sample C invocation:
 *		int entries[27] = { list of 27 entry IDs };
 *		int exits[27] = { list of 27 exit IDs };
 *		initLoopTrace( 27, entries, exits );
 */

initlooptrace_( numLoops, loopEntryID, loopExitID )

int	*numLoops, *loopEntryID, *loopExitID;
{
	return initLoopTrace( *numLoops, loopEntryID, loopExitID );
}


initLoopTrace( numLoops, loopEntryID, loopExitID )

int	numLoops, *loopEntryID, *loopExitID;
{
	extern TR_RECORD	*loopEventRecord();
	int			loopIndex;

	if (( numLoops <= 0 ) ||
	    ( loopEntryID == (int *) 0 ) ||
	    ( loopExitID == (int *) 0 ))
		return FAILURE;

	/* Allocate space to store a copy of the loop entry-exit	    */
	/* event ID matchings and also the loop entry stacks.	    */

	loopEvents = (LOOP_EVENTS *) TRgetBuffer(
					    numLoops * sizeof(LOOP_EVENTS) );

	if ( loopEvents == (LOOP_EVENTS *) 0 )
		TRfailure( "cannot allocate loop events matching" );

	loopEntryStack = (LOOP_ENTRY **) TRgetBuffer(
					    numLoops * sizeof(LOOP_ENTRY *) );

	if ( loopEntryStack == (LOOP_ENTRY **) 0 )
		TRfailure( "cannot allocate loop entry stack pointers" );

	/* Perform loop tracing extension pre-initialization, if	    */
	/* not already done.						    */

	preInitLoopTrace();

	/* Initialize the loop events matching from the arguments	    */
	/* passed.  Initialize the loop entry stack pointers.	    */
	/* Configure the trace record-generating function for these events. */

	for ( loopIndex = 0; loopIndex < numLoops; loopIndex++ ) {

		loopEvents[ loopIndex ].entryID = loopEntryID[ loopIndex ];
		loopEvents[ loopIndex ].exitID = loopExitID[ loopIndex ];

		loopEntryStack[ loopIndex ] = (LOOP_ENTRY *) 0;

		setEventRecordFunction( loopEntryID[ loopIndex ],
				        loopEventRecord );
		setEventRecordFunction( loopExitID[ loopIndex ],
				        loopEventRecord );
	}

	numberLoops = numLoops;

#ifdef DEBUG
	fprintf( debugFile, "initLoopTrace done\n" );
	fflush( debugFile );
#endif /* DEBUG */

	return SUCCESS;
}


/*
 *	preInitLoopTrace:
 *	   This function calls the trace library interface function
 *	   setRecordDescriptor, which records the address of the
 *	   loop that generates the record descriptors for the
 *	   loop trace event families.  It is automatically
 *	   invoked by initLoopTrace.  However, the user may elect to
 *	   explicitly call preInitLoopTrace before initLoopTrace does.
 *	   Since initLoopTrace calls the setEventRecordFunction interface
 *	   function of the trace library, that will cause the trace
 *	   library to perform its basic initialization.  One of the
 *	   effects of trace library basic initialization is to output
 *	   record descriptors for all event families defined up to
 *	   that point, by invoking all of the functions passed as
 *	   arguments to setRecordDescriptor, and then to output the
 *	   predefined, built-in record descriptors for internal event
 *	   families.  If no user-defined record descriptors beyond
 *	   those needed by this loop tracing extension are
 *	   to be used, then preInitLoopTrace need not be invoked.
 *	   However, if there are other such record descriptors (such
 *	   as an application also using the procedure tracing extension)
 *	   and if the user desires all these record descriptors to be
 *	   output before the trace library's internal event family
 *	   record descriptors, then all the required setRecordDescriptor
 *	   interface calls must be done before the trace library basic
 *	   initialization is performed.  preInitLoopTrace may be used
 *	   for this very purpose.
 */

preinitlooptrace_()
{
	preInitLoopTrace();
}

preInitLoopTrace()
{
	extern int	writeLoopRecordDescriptors();
	static int	preInitDone = FALSE;

	if ( preInitDone == TRUE )
		return;

	/* Give the instrumentation library a pointer to the function	    */
	/* in which we output the specialized record descriptors for	    */
	/* loop entry/exit.					    */

	setRecordDescriptor( writeLoopRecordDescriptors );

	preInitDone = TRUE;
}


/*
 *	writeLoopRecordDescriptors:
 *	   This function generates the record descriptors for the
 *	   loop entry/exit event families.  It will be invoked
 *	   by the instrumentation library initialization routines.
 *	   Patterned after instrumentation library internal function
 *	   writeRecordDescriptors.
 */

writeLoopRecordDescriptors()
{
	TR_RECORD	*recordPointer, *loopSDDFdescriptor();

#ifdef DEBUG
	fprintf( debugFile, "writeLoopRecordDescriptors\n" );
	fflush( debugFile );
#endif /* DEBUG */

	recordPointer = loopSDDFdescriptor( FAMILY_LOOPENTRY, RECORD_TRACE,
		"Loop Entry Trace", "Loop Entry Trace Record" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = loopSDDFdescriptor( FAMILY_LOOPEXIT, RECORD_TRACE,
		"Loop Exit Trace", "Loop Exit Trace Record" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = loopSDDFdescriptor( FAMILY_LOOPENTRY, RECORD_COUNT,
		"Loop Entry Count", "Loop Entry Count Record" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	recordPointer = loopSDDFdescriptor( FAMILY_LOOPEXIT, RECORD_COUNT,
		"Loop Exit Count", "Loop Exit Count Record" );
	putBytes( recordPointer->recordContents,
		  (unsigned) recordPointer->recordLength );

	return;
}


/*
 *	loopSDDFdescriptor:
 *	   Generate a SDDF binary format record descriptor for the
 *	   full trace class of events in either the loop entry
 *	   or loop exit family of events.  Patterned after the
 *	   instrumentation library internal function sddfDescriptor.
 */

TR_RECORD *
loopSDDFdescriptor( recordFamily, recordType, recordName, recordDescription )

int	recordFamily;
int	recordType;
char	*recordName;
char	*recordDescription;
{
	static TR_RECORD	traceRecord;
	static char		recordBuffer[ 1024 ];
	char			*recordPointer;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordFamily == FAMILY_LOOPENTRY ) ||
	       ( recordFamily == FAMILY_LOOPEXIT )
	      );

	Assert(
	       ( recordType == RECORD_TRACE ) ||
	       ( recordType == RECORD_COUNT )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordFamily == FAMILY_LOOPENTRY ) || \
	       ( recordFamily == FAMILY_LOOPEXIT ) \
	      );

	Assert( \
	       ( recordType == RECORD_TRACE ) || \
	       ( recordType == RECORD_COUNT ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

	/* Allow space at the beginning of the record for the packet	    */
	/* length, which will be computed after the packet is complete.	    */

	recordPointer = recordBuffer;
	sddfWriteInteger( &recordPointer, 0 );

	/* Write the record type descriptor.				    */

	sddfWriteInteger( &recordPointer, PKT_DESCRIPTOR );

	/* Write the record tag descriptor.				    */

	sddfWriteInteger( &recordPointer, ( recordFamily | recordType ));

	/* Write the record name.					    */

	sddfWriteString( &recordPointer, recordName );

	/* Write the record attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the record attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "description" );

	sddfWriteString( &recordPointer, recordDescription );

	/* Write the record field count.				    */

	switch (( recordFamily | recordType )) {

	case FAMILY_LOOPENTRY | RECORD_TRACE:
		sddfWriteInteger( &recordPointer, 7 );
		break;

	case FAMILY_LOOPENTRY | RECORD_COUNT:
		sddfWriteInteger( &recordPointer, 8 );
		break;

	case FAMILY_LOOPEXIT | RECORD_TRACE:
		sddfWriteInteger( &recordPointer, 9 );
		break;

	case FAMILY_LOOPEXIT | RECORD_COUNT:
		sddfWriteInteger( &recordPointer, 10 );
		break;
	}


	/* FIELD 1:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Timestamp" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Time" );

	sddfWriteString( &recordPointer, "Timestamp" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 1 );


	/* FIELD 2:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Seconds" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Seconds" );

	sddfWriteString( &recordPointer, "Floating Point Timestamp" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, DOUBLE );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 3:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Event Identifier" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "ID" );

	sddfWriteString( &recordPointer, "Event ID" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 4:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Processor Number" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Node" );

	sddfWriteString( &recordPointer, "Processor number" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 5:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Loop Index" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Loop" );

	sddfWriteString( &recordPointer, "Loop Index" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 6:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Source Byte" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Byte" );

	sddfWriteString( &recordPointer, "Source Byte Offset" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 7:							    */

	/* Write the field name.					    */

	sddfWriteString( &recordPointer, "Source Line" );

	/* Write the field attribute count.				    */

	sddfWriteInteger( &recordPointer, 1 );

	/* Write the field attribute string pair(s).			    */

	sddfWriteString( &recordPointer, "Line" );

	sddfWriteString( &recordPointer, "Source Line Number" );

	/* Write the field type.					    */

	sddfWriteInteger( &recordPointer, INTEGER );

	/* Write the field dimension.					    */

	sddfWriteInteger( &recordPointer, 0 );


	/* FIELD 8:							    */

	if ( recordFamily == FAMILY_LOOPEXIT ) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Loop Duration" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Duration" );

		sddfWriteString( &recordPointer, "Loop Duration" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 1 );

	}


	/* FIELD 9:							    */

	if ( recordFamily == FAMILY_LOOPEXIT ) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Loop Seconds" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Loop Seconds" );

		sddfWriteString( &recordPointer, "Floating Point Loop Duration" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, DOUBLE );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );

	}


	/* FIELD 10 (8 for FAMILY_LOOPENTRY):				    */

	if ( recordType == RECORD_COUNT ) {

		/* Write the field name.				    */

		sddfWriteString( &recordPointer, "Event Count" );

		/* Write the field attribute count.			    */

		sddfWriteInteger( &recordPointer, 1 );

		/* Write the field attribute string pair(s).		    */

		sddfWriteString( &recordPointer, "Count" );

		sddfWriteString( &recordPointer, "Event Count" );

		/* Write the field type.				    */

		sddfWriteInteger( &recordPointer, INTEGER );

		/* Write the field dimension.				    */

		sddfWriteInteger( &recordPointer, 0 );

	}


	/* The entire record descriptor packet has been written.	    */

	traceRecord.recordLength = recordPointer - recordBuffer;
	traceRecord.recordContents = recordBuffer;

	/* Finally, copy the completed packet length at the beginning	    */
	/* of the record itself.					    */

	recordPointer = recordBuffer;
	sddfWriteInteger( &recordPointer, traceRecord.recordLength );

	return & traceRecord;
}


/*
 *	loopEventRecord:
 *	   This function generates trace records for events which are
 *	   to produce loop entry or exit event family trace records.
 *	   Patterned after the instrumentation library internal functions
 *	   externalEventRecord and sddfRecord.
 */

TR_RECORD *
loopEventRecord( recordType, eventPointer, timeStamp,
		dataPointer, dataLength )

int		recordType;
TR_EVENT	*eventPointer;
CLOCK		timeStamp;
char		*dataPointer;
unsigned int	dataLength;
{
	static TR_RECORD		traceRecord;
	static char			*recordBuffer = (char *) 0;
	static int			bufferLength = 0;
	struct loopEntryTraceRecordData	*entryTraceRecordHeader;
	struct loopEntryCountRecordData	*entryCountRecordHeader;
	struct loopExitTraceRecordData	*exitTraceRecordHeader;
	struct loopExitCountRecordData	*exitCountRecordHeader;
	int				loopIndex;
	int				recordFamily;

#ifdef ASSERT_NO_BACKSLASH
	Assert(
	       ( recordType == RECORD_TRACE ) ||
	       ( recordType == RECORD_COUNT )
	      );

	Assert(
	       ( eventPointer != (TR_EVENT *) 0 ) &&
	       ( dataLength == 0 )
	      );
#else /* ASSERT_NO_BACKSLASH */
	Assert( \
	       ( recordType == RECORD_TRACE ) || \ 
	       ( recordType == RECORD_COUNT ) \
	      );

	Assert( \
	       ( eventPointer != (TR_EVENT *) 0 ) && \
	       ( dataLength == 0 ) \
	      );
#endif /* ASSERT_NO_BACKSLASH */

#ifdef DEBUG
	fprintf( debugFile, "loopEventRecord\n" );
	fflush( debugFile );
#endif /* DEBUG */

	/* Determine whether this is a loop entry or a loop	    */
	/* exit family event by lookup in the loop event ID matching.  */

	loopIndex = findLoopEvent( eventPointer->eventID );

	if ( loopIndex < 0 )
		return nullRecordFunction( recordType, eventPointer,
					timeStamp, dataPointer, dataLength );

	if ( loopEvents[ loopIndex ].entryID == eventPointer->eventID )
		recordFamily = FAMILY_LOOPENTRY;
	else
		recordFamily = FAMILY_LOOPEXIT;

	/* The time stamp stored in the event descriptor will be used	    */
	/* unless one is specified in the timeStamp parameter.		    */

	if ( clockCompare( timeStamp, noSuchClock ) == 0 )
		timeStamp = eventPointer->eventLast;

	/* Determine how many bytes of storage will be needed for the	    */
	/* contents of the trace record.				    */

	switch (( recordFamily | recordType )) {

	case FAMILY_LOOPENTRY | RECORD_TRACE:
		traceRecord.recordLength = sizeof *entryTraceRecordHeader;
		break;

	case FAMILY_LOOPENTRY | RECORD_COUNT:
		traceRecord.recordLength = sizeof *entryCountRecordHeader;
		break;

	case FAMILY_LOOPEXIT | RECORD_TRACE:
		traceRecord.recordLength = sizeof *exitTraceRecordHeader;
		break;

	case FAMILY_LOOPEXIT | RECORD_COUNT:
		traceRecord.recordLength = sizeof *exitCountRecordHeader;
		break;
	}

	/* If there is a previously-allocated buffer and its size will	    */
	/* hold this record, re-use the buffer.  Otherwise, deallocate	    */
	/* the buffer (if allocated) and allocate a bigger one.		    */

	if ( bufferLength < traceRecord.recordLength ) {

		if ( recordBuffer != (char *) 0 )
			TRfreeBuffer( recordBuffer );

		recordBuffer = TRgetBuffer( traceRecord.recordLength );

		if ( recordBuffer == (char *) 0 )
			TRfailure(
			     "cannot allocate storage for trace record" );

		bufferLength = traceRecord.recordLength;
	}

	traceRecord.recordContents = recordBuffer;

	/* Load the trace record fields into the allocated buffer	    */

	switch (( recordFamily | recordType )) {

	case FAMILY_LOOPENTRY | RECORD_TRACE:
		entryTraceRecordHeader = (struct loopEntryTraceRecordData *)
							recordBuffer;
		entryTraceRecordHeader->packetLength =
				traceRecord.recordLength;
		entryTraceRecordHeader->packetType = PKT_DATA;
		entryTraceRecordHeader->packetTag = recordFamily | recordType;
		entryTraceRecordHeader->clockDimension = sizeof(CLOCK)/sizeof(int);
		entryTraceRecordHeader->timeStamp = timeStamp;
		entryTraceRecordHeader->seconds = clockToSeconds( timeStamp );
		entryTraceRecordHeader->eventID = eventPointer->eventID;
		entryTraceRecordHeader->nodeNumber = TRgetNode();
		entryTraceRecordHeader->loopIndex = loopIndex;
		entryTraceRecordHeader->sourceByte = loopByteOffset;
		entryTraceRecordHeader->sourceLine = loopLineNumber;

		pushLoopEntry( loopIndex, timeStamp );
		break;

	case FAMILY_LOOPENTRY | RECORD_COUNT:
		entryCountRecordHeader = (struct loopEntryCountRecordData *)
							recordBuffer;
		entryCountRecordHeader->packetLength =
				traceRecord.recordLength;
		entryCountRecordHeader->packetType = PKT_DATA;
		entryCountRecordHeader->packetTag = recordFamily | recordType;
		entryCountRecordHeader->clockDimension = sizeof(CLOCK)/sizeof(int);
		entryCountRecordHeader->timeStamp = timeStamp;
		entryCountRecordHeader->seconds = clockToSeconds( timeStamp );
		entryCountRecordHeader->eventID = eventPointer->eventID;
		entryCountRecordHeader->nodeNumber = TRgetNode();
		entryCountRecordHeader->loopIndex = loopIndex;
		entryCountRecordHeader->sourceByte = loopByteOffset;
		entryCountRecordHeader->sourceLine = loopLineNumber;
		entryCountRecordHeader->eventCount = eventPointer->eventCount;

		pushLoopEntry( loopIndex, timeStamp );
		break;

	case FAMILY_LOOPEXIT | RECORD_TRACE:
		exitTraceRecordHeader = (struct loopExitTraceRecordData *)
							recordBuffer;
		exitTraceRecordHeader->packetLength =
				traceRecord.recordLength;
		exitTraceRecordHeader->packetType = PKT_DATA;
		exitTraceRecordHeader->packetTag = recordFamily | recordType;
		exitTraceRecordHeader->clockDimension = sizeof(CLOCK)/sizeof(int);
		exitTraceRecordHeader->timeStamp = timeStamp;
		exitTraceRecordHeader->seconds = clockToSeconds( timeStamp );
		exitTraceRecordHeader->eventID = eventPointer->eventID;
		exitTraceRecordHeader->nodeNumber = TRgetNode();
		exitTraceRecordHeader->loopIndex = loopIndex;
		exitTraceRecordHeader->sourceByte = loopByteOffset;
		exitTraceRecordHeader->sourceLine = loopLineNumber;
		exitTraceRecordHeader->durationDim = sizeof(CLOCK)/sizeof(int);
		exitTraceRecordHeader->loopDuration =
				popLoopEntry( loopIndex, timeStamp );
		exitTraceRecordHeader->loopSeconds =
			clockToSeconds( exitTraceRecordHeader->loopDuration );
		break;

	case FAMILY_LOOPEXIT | RECORD_COUNT:
		exitCountRecordHeader = (struct loopExitCountRecordData *)
							recordBuffer;
		exitCountRecordHeader->packetLength =
				traceRecord.recordLength;
		exitCountRecordHeader->packetType = PKT_DATA;
		exitCountRecordHeader->packetTag = recordFamily | recordType;
		exitCountRecordHeader->clockDimension = sizeof(CLOCK)/sizeof(int);
		exitCountRecordHeader->timeStamp = timeStamp;
		exitCountRecordHeader->seconds = clockToSeconds( timeStamp );
		exitCountRecordHeader->eventID = eventPointer->eventID;
		exitCountRecordHeader->nodeNumber = TRgetNode();
		exitCountRecordHeader->loopIndex = loopIndex;
		exitCountRecordHeader->sourceByte = loopByteOffset;
		exitCountRecordHeader->sourceLine = loopLineNumber;
		exitCountRecordHeader->eventCount = eventPointer->eventCount;
		exitCountRecordHeader->durationDim = sizeof(CLOCK)/sizeof(int);
		exitCountRecordHeader->loopDuration =
				popLoopEntry( loopIndex, timeStamp );
		exitCountRecordHeader->loopSeconds =
			clockToSeconds( exitCountRecordHeader->loopDuration );
		break;
	}

	return & traceRecord;
}


/*
 *	findLoopEvent:
 *	   Search the loop entry/exit event ID matching data
 *	   structure for an event ID (either entry or exit) which is
 *	   the same as the argument eventID.  If found, return the
 *	   index from that table, which will be between 0 and
 *	   numberLoops - 1, inclusive.  If not found, return -1;
 */

findLoopEvent( eventID )

int	eventID;
{
	int	loopIndex;

	for ( loopIndex = 0; loopIndex < numberLoops; loopIndex++ ) {
		if ( loopEvents[ loopIndex ].entryID == eventID ||
		     loopEvents[ loopIndex ].exitID == eventID )
			return loopIndex;
	}

	return -1;
}


/*
 *	pushLoopEntry:
 *	   Push a loop entry time onto the corresponding stacks.
 */

pushLoopEntry( loopIndex, entryTime )

int	loopIndex;
CLOCK	entryTime;
{
	LOOP_ENTRY	*timeElement;

	Assert(( loopIndex >= 0 ) && ( loopIndex < numberLoops ));

	/* Allocate a new loop entry element and push it onto the	    */
	/* corresponding loop's stack.					    */

	timeElement = (LOOP_ENTRY *) TRgetBuffer( sizeof(LOOP_ENTRY) );
	if ( timeElement == (LOOP_ENTRY *) 0 )
		TRfailure( "unable to allocate loop entry element" );

	timeElement->entryTime = entryTime;
	timeElement->nextTime = loopEntryStack[ loopIndex ];
	loopEntryStack[ loopIndex ] = timeElement;

	return;
}


/*
 *	popLoopEntry:
 *	   Pop a loop entry time off the corresponding stacks,
 *	   compute and return the loop's total duration.
 */

CLOCK
popLoopEntry( loopIndex, exitTime )

int	loopIndex;
CLOCK	exitTime;
{
	LOOP_ENTRY	*timeElement;
	CLOCK		duration;

	Assert(( loopIndex >= 0 ) && ( loopIndex < numberLoops ));

	timeElement = loopEntryStack[ loopIndex ];

	/* If corresponding loop's stack is empty, this is an error.	    */

	if ( timeElement == (LOOP_ENTRY *) 0 )
		return noSuchClock;

	/* Now pop the top entry time off of the corresponding stack.	    */

	loopEntryStack[ loopIndex ] = timeElement->nextTime;

	/* Compute the difference between this loop's entry and		    */
	/* exit times; this is its total duration.			    */

	duration = clockSubtract( exitTime, timeElement->entryTime );

	/* Finally, deallocate this structure and return.		    */

	TRfreeBuffer( (char *) timeElement );

	return duration;
}


/*
 *	PabloTraceLoop:
 *	   Interface function called by code instrumented with parser
 *	   to generate loop trace record with location data.
 */

PabloTraceLoop( eventID, byteOffset, lineNumber )

int		eventID;
unsigned long	byteOffset;
unsigned int	lineNumber;
{
	int	returnValue;

	loopByteOffset = (long) byteOffset;
	loopLineNumber = (int) lineNumber;

	returnValue = traceEvent( eventID, (char *) 0, 0 );

	loopByteOffset = -1;
	loopLineNumber = -1;

	return returnValue;
}


/*
 *	PabloCountLoop:
 *	   Interface function called by code instrumented with parser
 *	   to generate loop count record with location data.
 */

PabloCountLoop( eventID, byteOffset, lineNumber )

int		eventID;
unsigned long	byteOffset;
unsigned int	lineNumber;
{
	int	returnValue;

	loopByteOffset = (long) byteOffset;
	loopLineNumber = (int) lineNumber;

	returnValue = countEvent( eventID );

	loopByteOffset = -1;
	loopLineNumber = -1;

	return returnValue;
}
