/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/polar/RCS/Polar.c,v 1.11 1994/02/25 04:47:12 aydt Exp $
 */

#include <math.h>
#include <stdio.h>
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include "PolarP.h"

#define Min(a, b)	((a) < (b) ? (a) : (b))
#define XtStrlen(s)	((s) ? strlen(s) : 0)
#define Offset(field)	XtOffset(PolarWidget, field)
#define Norm(w, v)	(((float)v-(float)w->polar.min_value) /	\
			 ((float)w->polar.max_value -			\
			 (float)w->polar.min_value))

double	cos(), sin();

/***********************/
/* priviate procedures */
/***********************/
static void	ClassInitialize(), Initialize();
static void	Realize(), Destroy(), Resize(), Redisplay();
static Boolean	SetValues();

static void	DrawPolarFace(), DrawPolarAxes();
static void	DrawPolarValues();
static void	PolarSelect(), PolarNotify();

/*********************/
/* public procedures */
/*********************/
extern void	PolarSetValues(), PolarGetNearestPoint();

/*************************/
/* polar translations   */
/*************************/
static char polar_translations[] = "	\
  <Btn1Down>:	PolarSelect()\n	\
  <Btn1Up>:	PolarNotify()	\
";

/********************/
/* polar actions   */
/********************/
static XtActionsRec	polar_actions[] = {
  {"PolarSelect",	(XtActionProc) PolarSelect	},
  {"PolarNotify",	(XtActionProc) PolarNotify	},
};

/********************/
/* polar resources */
/********************/
static XtResource resources[] = {
  /* core resources */
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     Offset(core.width), XtRString, "100"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     Offset(core.height), XtRString, "100"},
  {XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
     Offset(core.background_pixel), XtRString, "white"},
  /* polar resources */
  {XtNcircleColor, XtCCircleColor, XtRPixel, sizeof(Pixel),
     Offset(polar.circle_color), XtRString, "black"},
  {XtNpointColor, XtCPointColor, XtRPixel, sizeof(Pixel),
     Offset(polar.point_color), XtRString, "black"},
  {XtNaxisColor, XtCAxisColor, XtRPixel, sizeof(Pixel),
     Offset(polar.axis_color), XtRString, "black"},
  {XtNbackingStore, XtCBackingStore, XtRBackingStore, sizeof(int),
     Offset(polar.backing_store), XtRString, "default"},
  {XtNnumaxes, XtCNumaxes, XtRInt, sizeof(int), 
     Offset(polar.numaxes), XtRString, "8" },
  {XtNnumCircles, XtCNumCircles, XtRInt, sizeof(int),
     Offset(polar.numcircles), XtRString, "4" },
  {XtNpointCount, XtCPointCount, XtRInt, sizeof(int),
     Offset(polar.ptcnt), XtRString, "0" },
  {XtNmagnitude, XtCMagnitude, XtRInt, sizeof(int),
     Offset(polar.magnitude), XtRString, "100"},
  {XtNmarkerSize, XtCMarkerSize, XtRInt, sizeof(int),
     Offset(polar.markersize), XtRString, "2" },
  {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     Offset(polar.select), XtRCallback, (caddr_t) NULL},
};
#undef offset

/***********************/
/* polar class record */
/***********************/
PolarClassRec polarClassRec = {
  { /* core fields */
    /* superclass		*/	&widgetClassRec,
    /* class_name		*/	"Polar",
    /* widget_size		*/	sizeof(PolarRec),
    /* class_initialize		*/	ClassInitialize,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	polar_actions,
    /* num_actions		*/	XtNumber(polar_actions),
    /* resources		*/	resources,
    /* resource_count		*/	XtNumber(resources),
    /* xrm_class		*/	NULL,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	TRUE,
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	polar_translations,
    /* query_geometry           */	XtInheritQueryGeometry,
    /* display_accelerator      */	XtInheritDisplayAccelerator,
    /* extension                */	NULL
    }
};

WidgetClass polarWidgetClass =(WidgetClass) &polarClassRec;

/************************************************************************/
/*                          PRIVATE PROCEDURES                          */
/************************************************************************/

/************************************************************************/
/* ClassInitialize()							*/
/************************************************************************/
static void ClassInitialize()
{
  XtAddConverter(XtRString, XtRBackingStore, XmuCvtStringToBackingStore,
		 NULL, 0);
}

/************************************************************************/
/* Initialize()								*/
/************************************************************************/
static void Initialize(request, new)
Widget request, new;
{
  int		i;
  PolarWidget	w =(PolarWidget)new;
  XtGCMask	GCmask;
  XGCValues	GCvalues;

#ifdef DEBUG
printf("Initialize\n");
#endif
  for (i=0; i < MAXPOINTS; i++) {
    w->polar.Xvalues[i] = 0;
    w->polar.Yvalues[i] = 0;
  }

  if(w->core.width == 0)
    w->core.width = MIN_SIZE;
  if(w->core.height == 0)
    w->core.height = MIN_SIZE;

  GCmask = GCForeground | GCBackground;
  GCvalues.background = w->core.background_pixel;
  GCvalues.foreground = w->polar.circle_color;
  w->polar.circleGC = XtGetGC((Widget)w, GCmask, &GCvalues);
  
  GCmask = GCForeground;
  GCvalues.foreground = w->polar.point_color;
  w->polar.pointGC = XtGetGC((Widget)w, GCmask, &GCvalues);

  GCmask = GCForeground;
  GCvalues.foreground = w->polar.axis_color;
  w->polar.axisGC = XtGetGC((Widget)w, GCmask, &GCvalues);
  
  GCmask = GCForeground;
  GCvalues.foreground = w->core.background_pixel;
  w->polar.eraseGC = XtGetGC((Widget)w, GCmask, &GCvalues);
  
} /* Initialize */

/************************************************************************/
/* SetValues()								*/
/************************************************************************/
static Boolean SetValues(gcurrent, grequest, gnew)
Widget	gcurrent, grequest, gnew;
{
  int			i;
  register PolarWidget	current =(PolarWidget) gcurrent;
  register PolarWidget	new =(PolarWidget) gnew;
  Boolean		redraw = FALSE;
  XtGCMask		GCmask;
  XGCValues		GCvalues;
  Boolean               realized = XtIsRealized(gcurrent);

#ifdef DEBUG
printf("SetValues\n");
#endif
  /* change face circle color */
  if((new->polar.circle_color != current->polar.circle_color)
      ||(new->core.background_pixel != current->core.background_pixel)) {
    GCmask = GCForeground | GCBackground;
    GCvalues.foreground = new->polar.circle_color;
    GCvalues.background = new->core.background_pixel;
    XtDestroyGC(current->polar.circleGC);
    new->polar.circleGC = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }
  
  /* change point color */
  if(new->polar.point_color != current->polar.point_color) {
    GCmask = GCForeground;
    GCvalues.foreground = new->polar.point_color;
    XtDestroyGC(current->polar.pointGC);
    new->polar.pointGC = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }
  
  /* change axis color */
  if(new->polar.axis_color != current->polar.axis_color) {
    GCmask = GCForeground;
    GCvalues.foreground = new->polar.axis_color;
    XtDestroyGC(current->polar.axisGC);
    new->polar.axisGC = XtGetGC(current, GCmask, &GCvalues);
    redraw = TRUE;
  }
  
  /* change erase color */
  if(new->core.background_pixel != current->core.background_pixel) {
    GCmask = GCForeground;
    GCvalues.foreground = new->core.background_pixel;
    XtDestroyGC(current->polar.eraseGC);
    new->polar.eraseGC = XtGetGC((Widget)current, GCmask, &GCvalues);
    redraw = TRUE;
  }
  
  /* change number of axes */
  if(new->polar.numaxes != current->polar.numaxes) {
    if(new->polar.numaxes > MAXAXES)
      new->polar.numaxes = MAXAXES;
  }

 /* change number of circles */
  if(new->polar.numcircles != current->polar.numcircles) {
    if(new->polar.numcircles > MAXCIRCLES)
      new->polar.numcircles = MAXCIRCLES;
  }

  if (redraw && realized) {
    Redisplay(new);
  }

  return(FALSE);
  
} /* SetValues */

/************************************************************************/
/* Realize()								*/
/************************************************************************/
static void Realize(gw, GCmask, attrs)
Widget			gw;
XtGCMask		*GCmask;
XSetWindowAttributes	*attrs;
{
  PolarWidget	w =(PolarWidget) gw;

#ifdef DEBUG
printf("Realize\n");
#endif
  *GCmask |= CWBitGravity;
  attrs->bit_gravity = ForgetGravity;
  switch(w->polar.backing_store) {
  case Always:
  case NotUseful:
  case WhenMapped:
    *GCmask |=CWBackingStore;
    attrs->backing_store = w->polar.backing_store;
    break;
  }
  XtCreateWindow(gw, InputOutput,(Visual *)CopyFromParent,
		 *GCmask, attrs);
  Resize(gw);

} /* Realize */

/************************************************************************/
/* Destroy()								*/
/************************************************************************/
static void Destroy(gw)
Widget gw;
{
  PolarWidget w =(PolarWidget) gw;

#ifdef DEBUG
printf("Destroy\n");
#endif
  XtDestroyGC(w->polar.circleGC);
  XtDestroyGC(w->polar.pointGC);
  XtDestroyGC(w->polar.axisGC);
  XtDestroyGC(w->polar.eraseGC);

} /* Destroy */

/************************************************************************/
/* Resize()								*/
/************************************************************************/
static void Resize(gw) 
Widget	gw;
{
  PolarWidget	w =(PolarWidget) gw;
  int		height;

#ifdef DEBUG
printf("Resize\n");
#endif
  if(XtIsRealized(gw)) {
    height = w->core.height;
    w->polar.radius= ((int) Min(w->core.width, height) - (int)(PADDING)) / 2;
    w->polar.centerX = w->core.width / 2;
    w->polar.centerY = w->polar.polar_offsetY + height / 2;
  }

} /* Resize */

/************************************************************************/
/* Redisplay()								*/
/************************************************************************/
static void Redisplay(gw, event, region)
Widget	gw;
XEvent	*event;				/* unused */
Region	region;				/* unused */
{
  char		buf[512];
  PolarWidget	w =(PolarWidget) gw;
  
#ifdef DEBUG
printf("Redisplay\n");
#endif

  if (XtIsRealized((Widget) w)) {
     DrawPolarFace(w);
     DrawPolarAxes(w);
     DrawPolarValues(w);
  }

} /* Redisplay */

/************************************************************************/
/* DrawPolarFace()							*/
/************************************************************************/
static void DrawPolarFace(w)
PolarWidget	w;
{
  register int	i;
  double	fraction;
  int		size;
	
#ifdef DEBUG
printf("DrawPolarFace\n");
#endif

  XFillArc(XtDisplay(w), XtWindow(w), w->polar.eraseGC,
           w->polar.centerX - w->polar.radius,
           w->polar.centerY - w->polar.radius,
           2*w->polar.radius, 2*w->polar.radius,
	   0, 360*64);

  XDrawArc(XtDisplay(w), XtWindow(w), w->polar.circleGC,
	   w->polar.centerX - w->polar.radius,
	   w->polar.centerY - w->polar.radius,
	   2*w->polar.radius, 2*w->polar.radius,
	   0, 360*64);

  for (i = 0; i < w->polar.numcircles; i++) {
     fraction = (double)(i + 1) / (double)(w->polar.numcircles);
     size = 2 * fraction * (w->polar.radius);

     XDrawArc(XtDisplay(w), XtWindow(w), w->polar.circleGC,
           w->polar.centerX - size / 2,
           w->polar.centerY - size / 2,
           size, size, 0, 360*64);
  }

} /* DrawPolarFace */

/************************************************************************/
/* DrawPolarAxes()							*/
/************************************************************************/
static void DrawPolarAxes(w)
PolarWidget	w;
{
  register int	i;
  int		perimx, perimy;
  double	fraction;
  double	angle, cosangle, sinangle;
	
#ifdef DEBUG
printf("DrawPolarAxes\n");
#endif
  for(i=0; i<w->polar.numaxes; i++) {
    fraction = ((double) i) / ((double) w->polar.numaxes);
    angle = NINETY + TWOPI - (TWOPI * fraction);
    cosangle = cos(angle);
    sinangle = sin(angle);
    perimx = w->polar.centerX +
      (int) ((double) w->polar.radius) * cosangle;
    perimy = w->polar.centerY -
      (int) ((double) w->polar.radius) * sinangle;
    XDrawLine(XtDisplay(w), XtWindow(w), w->polar.axisGC,
	      w->polar.centerX, w->polar.centerY, perimx, perimy);
  }

} /* DrawPolarAxes */

/************************************************************************/
/* DrawPolarValues()							*/
/************************************************************************/
static void DrawPolarValues(w)
PolarWidget	w;
{
  register int	i;
  int	   x1, y1;
  double   mag, x, y;
  double	angle, cosangle, sinangle;

#ifdef DEBUG
printf("DrawPolarValues\n");
#endif

  for(i=0; i<w->polar.ptcnt; i++) {
    x = (double) w->polar.Xvalues[i];
    y = (double) w->polar.Yvalues[i];

    mag = hypot(y, x);
    angle = atan2(y, x);

    cosangle = cos(angle);
    sinangle = sin(angle);

    x = w->polar.centerX +
      (int)((double)(w->polar.radius)) *
	    cosangle * (mag / (double)(w->polar.magnitude));
    y = w->polar.centerY -
      (int)((double)(w->polar.radius)) *
	    sinangle * (mag / (double)(w->polar.magnitude));

    x1 = x - w->polar.markersize/2;
    y1 = y - w->polar.markersize/2;

/*
    XDrawPoint(XtDisplay(w), XtWindow(w), w->polar.pointGC,
		   x1, y1);

    XFillArc(XtDisplay(w), XtWindow(w), w->polar.pointGC,
           x1 - 1, y1 - 1,
           2, 2, 0, 360*64);
*/

    XFillArc(XtDisplay(w), XtWindow(w), w->polar.pointGC,
           x1, y1, w->polar.markersize, w->polar.markersize,
           0, 360*64);
  }    

} /* DrawPolarValues */

/************************************************************************/
/* Euclid() computes the distance between a pair of points		*/
/************************************************************************/
static int Euclid(x0, y0, x1, y1)
int x0, y0;
double x1, y1;
{
        int xPoint = (int)x1;
	int yPoint = (int)y1;

        int Idist;
	double	Dist;
#ifdef DEBUG
printf("Euclid\n");
#endif
	Dist = (x0 - xPoint) * (x0 - xPoint) + (y0 - yPoint) * (y0 - yPoint);
	Dist = sqrt (Dist);
	Idist = (int) Dist;
	return (Dist);

} /* Euclid */

/************************************************************************/
/* PolarSelect() processes button down event.				*/
/************************************************************************/
static void PolarSelect(w, event)
PolarWidget		w;
XButtonPressedEvent	*event;
{
        w->polar.location.x = event->x;
	w->polar.location.y = event->y;
} /* PolarSelect */

/************************************************************************/
/* PolarNotify() processes button up event.				*/
/************************************************************************/
static void PolarNotify(w, event)
PolarWidget		w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
printf("PolarNotify\n");
#endif

  XtCallCallbacks(w, XtNselect, (caddr_t) w->polar.value_select);

} /* PolarNotify */

/************************************************************************/
/*                          PUBLIC PROCEDURES                           */
/************************************************************************/

/************************************************************************/
/* PolarSetValues()							*/
/************************************************************************/
extern void PolarSetValues(w, ptcnt, Xvalues, Yvalues)
PolarWidget	w;
int		ptcnt;
int		*Xvalues;
int		*Yvalues;
{
  int	i;
  double mag;

#ifdef DEBUG
printf("PolarSetValues\n");
#endif

  if ((ptcnt < 0) || (ptcnt > MAXPOINTS)) {
     fprintf(stderr, "PolarSetValues:  Too many values (%d)\n",ptcnt);
     w->polar.ptcnt = MAXPOINTS;
  } else
     w->polar.ptcnt = ptcnt;

  for (i=0; i < ptcnt; i++) {
    w->polar.dataXValues[i] = Xvalues[i];
    w->polar.dataYValues[i] = Yvalues[i];

    mag = Xvalues[i] * Xvalues[i] + Yvalues[i] * Yvalues[i];
    mag = sqrt(mag);

    if (fabs(mag) > w->polar.magnitude) {
      printf("PolarSetValues: tried to set a value (%d,%d) ", Xvalues[i],
		Yvalues[i]);
      printf("greater than the maximal (%d)\n", w->polar.magnitude);

      if (Xvalues[i] < 0)
         w->polar.Xvalues[i] = -Xvalues[i] * (w->polar.magnitude / mag);
      else
	 w->polar.Xvalues[i] = Xvalues[i] * (w->polar.magnitude / mag);

      if (Yvalues[i] < 0)
         w->polar.Yvalues[i] = -Yvalues[i] * (w->polar.magnitude / mag);
      else
	 w->polar.Yvalues[i] = Yvalues[i] * (w->polar.magnitude / mag);
    }
    else {
      w->polar.Xvalues[i] = Xvalues[i];
      w->polar.Yvalues[i] = Yvalues[i];
    }
  }

  Redisplay(w);
    
} /* PolarSetValues */


/************************************************************************/
/* PolarGetNearestPoint()
/************************************************************************/
extern void PolarGetNearestPoint(w, xNearest, yNearest, index)
PolarWidget w;
int         *xNearest, *yNearest;
int         *index;
{
  int i;
  double   mag, x, y;
  double	angle, cosangle, sinangle;
  double currentDistance;
  double closestDistance = (double)w->core.width * w->core.height;

  if ( w->polar.ptcnt == 0 ) {
      *index = -1;
      return;
  }

  for (i=0; i<w->polar.ptcnt; i++) {
    x = w->polar.Xvalues[i];
    y = w->polar.Yvalues[i];
    mag = hypot(y, x);
    angle = atan2(y, x);

    cosangle = cos(angle);
    sinangle = sin(angle);

    x = w->polar.centerX +
      (int)((double)(w->polar.radius)) *
	    cosangle * (mag / (double)(w->polar.magnitude));
    y = w->polar.centerY -
      (int)((double)(w->polar.radius)) *
	    sinangle * (mag / (double)(w->polar.magnitude));

      currentDistance = Euclid( w->polar.location.x, w->polar.location.y,
			        x, y );
      if (currentDistance < (double)closestDistance ) {
	 closestDistance = currentDistance;
	 *xNearest = w->polar.dataXValues[i];
	 *yNearest = w->polar.dataYValues[i];
	 *index = i;
      }
  }
}



