/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */


#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include "ParallelCoordsP.h"

#define REALLOC_INTERVAL    10

#define Offset(field)       XtOffset( ParallelCoordsWidget, field )


/* Private procedures */
static void      ClassInitialize(), Initialize();
static void      Realize(), Resize(), Redisplay(), Destroy();
static Boolean   SetValues();

static void      DrawAxes(), DrawData();
static void      ParallelCoordsSelect(), ParallelCoordsNotify();

/* Public procedures */
extern void      ParallelCoordsSetValues();

/* Utility procedures */


/* ParallelCoords translations */
static char parallelCoords_translations[] = " \
  <Btn1Down>:    ParallelCoordsSelect()\n     \
  <Btn1Up>:      ParallelCoordsNotify()       \
";

/* ParallelCoords actions */
static XtActionsRec parallelCoords_actions[] = {
  {"ParallelCoordsSelect",    (XtActionProc) ParallelCoordsSelect },
  {"ParallelCoordsNotify",    (XtActionProc) ParallelCoordsNotify },
};

/* ParallelCoords resources */
static XtResource resources[] = {
   /* CORE */
   {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
      Offset(core.width), XtRString, "100"},
   {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
      Offset(core.height), XtRString, "200"},

   /* ParallelCoords resources */
   {XtNminValue,  XtCMinValue, XtRInt, sizeof(int),
      Offset(parallelCoords.minValue), XtRString, "0"},
   {XtNmaxValue, XtCMaxValue, XtRInt, sizeof(int),
      Offset(parallelCoords.maxValue), XtRString, "100000"},
   {XtNpadding, XtCPadding, XtRInt, sizeof(int),
      Offset(parallelCoords.padding), XtRString, "4" },
   {XtNbaseColor, XtCBaseColor, XtRPixel, sizeof(Pixel),
      Offset(parallelCoords.baseColor), XtRString, "black" },
   {XtNaxisColor, XtCAxisColor, XtRPixel, sizeof(Pixel),
      Offset(parallelCoords.axisColor), XtRString, "black" },
   {XtNlineColor, XtCLineColor, XtRPixel, sizeof(Pixel),
      Offset(parallelCoords.lineColor), XtRString, "blue" },
   {XtNnumAxes, XtCNumAxes, XtRInt, sizeof(int),
      Offset(parallelCoords.numAxes), XtRString, "2"},
   {XtNorientation, XtCOrientation, XtROrientation, sizeof(XtOrientation),
      Offset(parallelCoords.orientation), XtRString, "horizontal" },
   {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
      Offset(parallelCoords.select), XtRCallback, (caddr_t) NULL },
};
#undef Offset

/* ParallelCoords class record */
ParallelCoordsClassRec parallelCoordsClassRec = {
  (WidgetClass) &widgetClassRec,        /* superclass                   */
  "ParallelCoords",                     /* class_name                   */
  sizeof(ParallelCoordsRec),            /* size                         */
  ClassInitialize,                      /* class_initialize             */
  NULL,                                 /* class_part_initialize        */
  FALSE,                                /* class_init'ed                */
  Initialize,                           /* initialize                   */
  NULL,                                 /* initialize_hook              */
  Realize,                              /* realize                      */
  parallelCoords_actions,               /* actions                      */
  XtNumber(parallelCoords_actions),     /* num_actions                  */
  resources,                            /* resources                    */
  XtNumber(resources),                  /* resource_count               */
  NULLQUARK,                            /* xrm_class                    */
  TRUE,                                 /* compress_motion              */
  TRUE,                                 /* compress_exposure            */
  TRUE,                                 /* compress_enterleave          */
  FALSE,                                /* visible_interest             */
  Destroy,                              /* destroy                      */
  Resize,                               /* resize                       */
  Redisplay,                            /* expose                       */
  SetValues,                            /* set_values                   */
  NULL,                                 /* set_values_hook              */
  XtInheritSetValuesAlmost,             /* set_values_almost            */
  NULL,                                 /* get_values_hook              */
  NULL,                                 /* accept_focus                 */
  XtVersion,                            /* version                      */
  NULL,                                 /* callback_private             */
  parallelCoords_translations,          /* translationManager table     */
  XtInheritQueryGeometry,               /* query_geometry               */
  XtInheritDisplayAccelerator           /* display_accelerator          */
};

WidgetClass parallelCoordsWidgetClass = (WidgetClass)&parallelCoordsClassRec;

static XrmQuark  XtQEhorizontal;
static XrmQuark  XtQEvertical;


/* --------------------------------------------------------------------- */
/*                           PRIVATE PROCEDURES                          */
/* --------------------------------------------------------------------- */

/* DESTROY() */
static void Destroy(bww)
Widget bww;
{
     ParallelCoordsWidget w = (ParallelCoordsWidget) bww;

} /* DESTROY */


/* CVTSTRINGTOORIENTATION() */
static void CvtStringToOrientation(args, num_args, fromVal, toVal)
XrmValuePtr    *args;         /* unused */
Cardinal       *num_args;     /* unused */
XrmValuePtr    fromVal;
XrmValuePtr    toVal;
{
     static XtOrientation orient;
     XrmQuark             q;
     char                 lowerCaseName[80];

     /* convert incoming string name to lower case */
     q = XrmStringToQuark(lowerCaseName);

     if (q == XtQEhorizontal) {
        orient = XtorientHorizontal;
	(*toVal).size = sizeof(XtOrientation);
	(*toVal).addr = (caddr_t) &orient;
	return;
     } 
     if (q == XtQEvertical) {
        orient = XtorientVertical;
	(*toVal).size = sizeof(XtOrientation);
	(*toVal).addr = (caddr_t) &orient;
	return;
     }
} 
/* CVTSTRINGTOORIENTATION */


/* CLASSINITIALIZE */
static void ClassInitialize() 
{
     XtQEhorizontal = XrmStringToQuark(XtEhorizontal);
     XtQEvertical = XrmStringToQuark(XtEvertical);
     XtAddConverter( XtRString, XtROrientation, CvtStringToOrientation,
		     NULL, (Cardinal)0 );
} /* CLASSINITIALIZE */


/* INITIALIZE */
static void Initialize( request, new )
Widget request;
Widget new;
{
     ParallelCoordsWidget    w = (ParallelCoordsWidget) new;
     XGCValues               gcValues;
     XtGCMask                gcMask;

     /* initialize the base GC */
     gcValues.foreground = w->parallelCoords.baseColor;
     gcMask = GCForeground;
     w->parallelCoords.baseGC = XtGetGC( new, gcMask, &gcValues );

     /* initialize the axis GC */
     gcValues.foreground = w->parallelCoords.axisColor;
     gcMask = GCForeground;
     w->parallelCoords.axisGC = XtGetGC( new, gcMask, &gcValues );

     /* initialize the line GC */
     gcValues.foreground = w->parallelCoords.lineColor;
     gcMask = GCForeground;
     w->parallelCoords.lineGC = XtGetGC( new, gcMask, &gcValues );

     /* initialize variables */
     w->parallelCoords.dataValues = NULL;
     w->parallelCoords.dataValueCnt = 0;

} /* INITIALIZE */


/* SETVALUES */

static Boolean SetValues( wcurrent, wrequest, wnew )
Widget wcurrent;
Widget wrequest;
Widget wnew;
{
      ParallelCoordsWidget    current = (ParallelCoordsWidget) wcurrent;
      ParallelCoordsWidget    new = (ParallelCoordsWidget) wnew;
      XGCValues    gcValues;
      XtGCMask     gcMask;
      Boolean      redraw = FALSE;
      Boolean      realized = XtIsRealized( current );

      /* change of  color */

      /* change of size */
      if ( (new->core.height != current->core.height ) ||
           (new->core.width != current->core.width ) ) {
      }
      
      return( redraw );
} /* SETVALUES */


/* REALIZE */
static void Realize(gw, GCmask, attrs)
Widget                   gw;
XtGCMask                 *GCmask;
XSetWindowAttributes     *attrs;
{
      ParallelCoordsWidget w = (ParallelCoordsWidget)gw;
      XtGCMask      pixmapGCmask;
      XGCValues     pixmapGCvalues;

      *GCmask |= CWBitGravity;
      attrs->bit_gravity = ForgetGravity;
      switch(w->parallelCoords.backing_store) {
         case Always:
         case NotUseful:
         case WhenMapped:
             *GCmask |=CWBackingStore;
              attrs->backing_store = w->parallelCoords.backing_store;
             break;
      }
      /* Create the widget window */
      XtCreateWindow(w, InputOutput,(Visual *)CopyFromParent,
		     *GCmask, attrs); 
      /* Create the pixmap GC */
      pixmapGCmask = GCForeground;
      pixmapGCvalues.foreground = w->core.background_pixel;
      w->parallelCoords.pixmapGC = XtGetGC( gw, pixmapGCmask, &pixmapGCvalues);
      
      /* Create the drawing pixmap */
      w->parallelCoords.drawPixmap = XCreatePixmap( XtDisplay(w),XtWindow(w),
					      w->core.width, w->core.height,
				              DefaultDepth( XtDisplay(w),
					      DefaultScreen(XtDisplay(w)) ) );
      Resize(gw);
} /* Realize */


/* RESIZE */
static void Resize(pcw)
Widget pcw;
{
      ParallelCoordsWidget    w = (ParallelCoordsWidget) pcw;
      Display *display = XtDisplay(pcw);
      int screenNumber = DefaultScreen( XtDisplay(pcw) );

      if ( XtIsRealized(pcw) ) {
         /* Size the drawing pixmap */
	 XFreePixmap( display, w->parallelCoords.drawPixmap );
	 w->parallelCoords.drawPixmap = XCreatePixmap( display, XtWindow(pcw),
					        w->core.width, w->core.height,
					        DefaultDepth( display,
							      screenNumber ) );

        Redisplay(pcw);
     }
} /* RESIZE */


/* REDISPLAY */
static void Redisplay( bww, event, region )
Widget bww;
XEvent *event;
Region region;
{
      ParallelCoordsWidget    w = (ParallelCoordsWidget) bww;
      
      XFillRectangle( XtDisplay(w), w->parallelCoords.drawPixmap,
		      w->parallelCoords.pixmapGC,
		      0,0,  w->core.width, w->core.height );

      if ( XtIsRealized(bww) ) {
         DrawAxes(w);
	 DrawData(w);
      }
      XCopyArea( XtDisplay(w), w->parallelCoords.drawPixmap, XtWindow(w),
		 w->parallelCoords.pixmapGC,
		 0,0,  w->core.width, w->core.height, 0,0 );
} /* REDISPLAY */


/* DrawAxes */
static void DrawAxes( w )
ParallelCoordsWidget w;
{
      int i;
      int currentX, currentY;
      if ( w->parallelCoords.orientation == XtorientHorizontal ) {
	     /* draw base */
	     currentY = (w->core.height - w->parallelCoords.padding);
	     XDrawLine( XtDisplay(w), w->parallelCoords.drawPixmap,
		        w->parallelCoords.baseGC,
                        w->parallelCoords.padding, currentY,
		       (w->core.width - w->parallelCoords.padding), currentY );

	     /* draw axes */
	     for (i=0; i< w->parallelCoords.numAxes; i++) {
	         int currentX = w->parallelCoords.padding +
		             (( w->core.width - 2*w->parallelCoords.padding) /
			        (w->parallelCoords.numAxes-1)) * i;
	         XDrawLine(XtDisplay(w), w->parallelCoords.drawPixmap,
			   w->parallelCoords.axisGC,
			   currentX, w->core.height- w->parallelCoords.padding,
			   currentX, w->parallelCoords.padding );
	     }
	       
       } else if ( w->parallelCoords.orientation == XtorientVertical ) {
             /* not yet implemented */
       } 
} /* DrawAxes */


/* DrawData */
static void DrawData( w )
ParallelCoordsWidget w;
{
       int i, j;
       int dim = w->parallelCoords.numAxes;
       int dataRange = w->parallelCoords.maxValue - w->parallelCoords.minValue;
       int axisHeight = (w->core.height - 2 * w->parallelCoords.padding);
       int baseWidth = (w->core.width - 2 * w->parallelCoords.padding);

       float dataPctage;
       XPoint *linePoints = (XPoint *)XtMalloc( dim * sizeof(XPoint) );
       
       for (i=0; i < w->parallelCoords.dataValueCnt; i++) {
	   for (j=0; j < dim; j++ ) {
	       dataPctage = (float) (w->parallelCoords.dataValues[i*dim+j] -
		                        (float)w->parallelCoords.minValue) /
		            (float)dataRange;        
	       linePoints[j].x = w->parallelCoords.padding +
    		                     ((float)baseWidth/(float)(dim-1)) * j;  
	       linePoints[j].y = w->core.height - w->parallelCoords.padding - 
	                                  (dataPctage * (float)axisHeight);
	   }
	   XDrawLines( XtDisplay(w), w->parallelCoords.drawPixmap,
		       w->parallelCoords.lineGC,
                       linePoints, dim, CoordModeOrigin );
       }
       XtFree( linePoints );
}


/* ParallelCoordsSelect() processes buttonDown event */
static void ParallelCoordsSelect(w, event) 
ParallelCoordsWidget    w;
XButtonPressedEvent    *event;
{
} /* ParallelCoordsSelect */


/* ParallelCoordsNotify() processes buttonUp event */
static void ParallelCoordsNotify(w, event)
ParallelCoordsWidget    w;
XButtonPressedEvent    *event;
{
     XtCallCallbacks( w, XtNselect, NULL );
}  /* ParallelCoordsNotify */


     
/* PUBLIC PROCEDURES ---------------------------------------------------- */

/* ParallelCoordsSetValues() sets the current point set for the PC plot */
extern void ParallelCoordsSetValues( w, values, valuecnt )
ParallelCoordsWidget w;
int                  *values;
int                  valuecnt;
{
     int i;
     int numAxes = w->parallelCoords.numAxes;

     /* allocate space for data */
     if ( w->parallelCoords.dataValues == NULL ) {
        w->parallelCoords.dataValues = (int *)XtMalloc(
					    valuecnt * numAxes * sizeof(int));
     } else if ( valuecnt > w->parallelCoords.dataValueCnt ) {
        w->parallelCoords.dataValues = (int *)XtRealloc( 
					     w->parallelCoords.dataValues,
				            valuecnt * numAxes * sizeof(int) );
     }
     /* store data */
     for (i=0; i < valuecnt * numAxes; i++) {
         w->parallelCoords.dataValues[i] = values[i];
     }
     w->parallelCoords.dataValueCnt = valuecnt;

     /* draw */
     Redisplay(w);
} /* ParallelCoordsSetValues */



/* ParallelCoordsSetValue() adds a single point to the PC dataset */
extern void ParallelCoordsSetValue( w, value )
ParallelCoordsWidget w;
int                  *value;
{
     int i;
     int numAxes = w->parallelCoords.numAxes;
     int numPoints = w->parallelCoords.dataValueCnt;

     /* allocate space for data */
     if ( w->parallelCoords.dataValues == NULL ) {
        w->parallelCoords.dataValues = (int *)XtMalloc( numAxes * sizeof(int));
     } else {
        w->parallelCoords.dataValues = (int *)XtRealloc( 
				       w->parallelCoords.dataValues,
			               (numPoints+1) * numAxes * sizeof(int) );
     }
     /* store data */
     for (i=0; i < numAxes; i++) {
         w->parallelCoords.dataValues[numPoints*numAxes + i] = value[i];
     }
     w->parallelCoords.dataValueCnt++; 

     /* draw */
     Redisplay(w);
} /* ParallelCoordsSetValue */


