/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Allen D. Malony (malony@uicsrd.csrd.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/gantt/RCS/Gantt.c,v 1.9 1994/02/25 04:45:59 aydt Exp $
 */

#include <stdio.h>
#include <math.h>
#include <X11/IntrinsicP.h>
#include <X11/Xresource.h>
#include <X11/StringDefs.h>
#include "GanttP.h"

#define Offset(field) XtOffset(GanttWidget, field)

/**********************/
/* private procedures */
/**********************/
static void	ClassInitialize(), Initialize();
static void	Destroy(), DestroyColorGCs(), CreateColorGCs();
static void	Resize(), Redisplay();
static void	ClearGantt(), DrawTicks(), DrawLines(), DrawGantt();
static void	DrawBorder(), EraseLine(), DrawLine(), EraseTicks();
static void	GanttSelect(), GanttNotify();
static Boolean	SetValues();
static int	Euclid();


/*********************/
/* public procedures */
/*********************/
extern void	GanttAddLine(), GanttDeleteLine();
extern void	GanttMapLine(), GanttUnMapLine();
extern void	GanttShowGantt(), GanttClearGantt();
extern void	GanttSetColorMap();

/**********************/
/* gantt translations */
/**********************/
static char gantt_translations[] = "	\
  <Btn1Down>:	GanttSelect()\n		\
  <Btn1Up>:	GanttNotify()		\
";

/*****************/
/* gantt actions */
/*****************/
static XtActionsRec	gantt_actions[] = {
  {"GanttSelect",	(XtActionProc) GanttSelect	},
  {"GanttNotify",	(XtActionProc) GanttNotify	},
};

/*******************/
/* gantt resources */
/*******************/
static XtResource resources[] = {
  /******************/
  /* core resources */
  /******************/
  {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
     Offset(core.width), XtRString, "200"},
  {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
     Offset(core.height), XtRString, "200"},
  /*******************/
  /* gantt resources */
  /*******************/
  {XtNxmin, XtCXmin, XtRInt, sizeof(int),
     Offset(gantt.xmin), XtRString, "1"},
  {XtNxmax, XtCXmax, XtRInt, sizeof(int),
     Offset(gantt.xmax), XtRString, "100"},
  {XtNymin, XtCYmin, XtRInt, sizeof(int),
     Offset(gantt.ymin), XtRString, "1"},
  {XtNymax, XtCYmax, XtRInt, sizeof(int),
     Offset(gantt.ymax), XtRString, "100"},
  {XtNxticks, XtCXticks, XtRInt, sizeof(int),
     Offset(gantt.xticks), XtRString, "5"},
  {XtNyticks, XtCYticks, XtRInt, sizeof(int),
     Offset(gantt.yticks), XtRString, "5"},
  {XtNtickColor, XtCTickColor, XtRPixel, sizeof(Pixel),
     Offset(gantt.tickcolor), XtRString, "black"},
  {XtNcolors, XtCColors, XtRPointer, sizeof(Pixel *),
     Offset(gantt.colors), XtRString, NULL},
  {XtNnumColors, XtCNumColors, XtRInt, sizeof(int),
     Offset(gantt.numcolors), XtRString, "0"},
  {XtNorientation, XtCOrientation, XtROrientation, sizeof(XtOrientation),
     Offset(gantt.orientation), XtRString, "vertical"},
  {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
     Offset(gantt.select), XtRCallback, (caddr_t) NULL},
};
#undef Offset

/**********************/
/* gantt class record */
/**********************/
GanttClassRec ganttClassRec = {
  (WidgetClass) &widgetClassRec,	/* superclass		  */	
  "Gantt",				/* class_name		  */
  sizeof(GanttRec),			/* size			  */
  ClassInitialize,			/* class_initialize	  */
  NULL,					/* class_part_initialize  */
  FALSE,				/* class_inited		  */
  Initialize,				/* initialize		  */
  NULL,					/* initialize_hook	  */
  XtInheritRealize,			/* realize		  */
  gantt_actions,			/* actions		  */
  XtNumber(gantt_actions),		/* num_actions		  */
  resources,				/* resources		  */
  XtNumber(resources),			/* resource_count	  */
  NULLQUARK,				/* xrm_class		  */
  TRUE,					/* compress_motion	  */
  TRUE,					/* compress_exposure	  */
  TRUE,					/* compress_enterleave    */
  FALSE,				/* visible_interest	  */
  Destroy,				/* destroy		  */
  Resize,				/* resize		  */
  Redisplay,				/* expose		  */
  SetValues,				/* set_values		  */
  NULL,					/* set_values_hook	  */
  XtInheritSetValuesAlmost,		/* set_values_almost	  */
  NULL,					/* get_values_hook	  */
  NULL,					/* accept_focus		  */
  XtVersion,				/* version		  */
  NULL,					/* callback_private	  */
  gantt_translations,			/* tm_table		  */
  XtInheritQueryGeometry,		/* query_geometry         */
  XtInheritDisplayAccelerator,		/* display_accelerator    */
};

WidgetClass ganttWidgetClass = (WidgetClass)&ganttClassRec;

static XrmQuark  XtQEhorizontal;
static XrmQuark  XtQEvertical;

/************************************************************************/
/*                          PRIVATE PROCEDURES                          */
/************************************************************************/

/************************************************************************/
/* CvtStringToOrientation()						*/
/************************************************************************/
static void CvtStringToOrientation(args, num_args, fromVal, toVal)
XrmValuePtr	*args;		/* unused */
Cardinal	*num_args;	/* unused */
XrmValuePtr	fromVal;
XrmValuePtr	toVal;
{
  static XtOrientation	orient;
  XrmQuark		q;
  char			lowerName[1000];

  strcpy( lowerName, fromVal->addr );
  q = XrmStringToQuark(lowerName);
  if (q == XtQEhorizontal) {
    orient = XtorientHorizontal;
    (*toVal).size = sizeof(XtOrientation);
    (*toVal).addr = (caddr_t) &orient;
    return;
  }
  if (q == XtQEvertical) {
    orient = XtorientVertical;
    (*toVal).size = sizeof(XtOrientation);
    (*toVal).addr = (caddr_t) &orient;
    return;
  }

} /* CvtStringToOrientation */

/************************************************************************/
/* ClassInitialize()							*/
/************************************************************************/
static void ClassInitialize()
{
#ifdef DEBUG
printf("ClassInitialize\n");
#endif
  XtQEhorizontal = XrmStringToQuark(XtEhorizontal);
  XtQEvertical   = XrmStringToQuark(XtEvertical);
  XtAddConverter(XtRString, XtROrientation, CvtStringToOrientation,
		 NULL, (Cardinal) 0);

} /* ClassInitialize */

/************************************************************************/
/* Initialize()								*/
/************************************************************************/
static void Initialize(request, new)
Widget	request;
Widget	new;
{
  int		i;
  GanttWidget	w = (GanttWidget) new;
  Pixel		foreground;
  XGCValues	gcValues;
  XtGCMask	gcMask;

#ifdef DEBUG
printf("Initialize\n");
#endif
  /* determine foreground pixel */
  if (XWhitePixelOfScreen(XtScreen(w)) == w->core.background_pixel)
    foreground = XBlackPixelOfScreen(XtScreen(w));
  else
    foreground = XWhitePixelOfScreen(XtScreen(w));

  /* initialize tick GC */
  gcValues.foreground = w->gantt.tickcolor;
  gcMask = GCForeground;
  w->gantt.tickGC = XtGetGC(new, gcMask, &gcValues);

  /* initialize erase GC */
  gcValues.foreground = w->core.background_pixel;
  gcMask = GCForeground;
  w->gantt.eraseGC = XtGetGC(new, gcMask, &gcValues);

  /* initialize line colors */
  gcValues.foreground = foreground;
  gcMask = GCForeground;
  for (i=0; i<MAXCOLORS; i++) {
    w->gantt.colorGCs[i] = XtGetGC(new, gcMask, &gcValues);
  }
  w->gantt.numcolors = 0;

  if (w->core.width <= 0)
    w->gantt.width = DEFAULT_WIDTH;
  else
    w->gantt.width = w->core.width;
  if (w->core.height <= 0)
    w->gantt.height = DEFAULT_HEIGHT;
  else
    w->gantt.height = w->core.height;

  for (i=0; i<MAXLINES; i++) {
    w->gantt.c[i] = -1;
    w->gantt.p[i] = 0;
  }

} /* Initialize */

/************************************************************************/
/* SetValues()								*/
/************************************************************************/
static Boolean SetValues(gcurrent, grequest, gnew)
Widget	gcurrent;
Widget	grequest;
Widget	gnew;
{
  GanttWidget	current = (GanttWidget) gcurrent;
  GanttWidget	new = (GanttWidget) gnew;
  XGCValues	gcValues;
  XtGCMask	gcMask;
  Boolean	redraw = FALSE;
  Boolean       realized = XtIsRealized(gcurrent);

#ifdef DEBUG
printf("SetValues\n");
#endif
  /* change of erase color */
  if (new->core.background_pixel != current->core.background_pixel) {
    XtDestroyGC(current->gantt.eraseGC);
    gcValues.foreground = new->core.background_pixel;
    gcMask = GCForeground;
    new->gantt.eraseGC = XtGetGC(gcurrent, gcMask, &gcValues);
  }

  /* change of tick color */
  if (new->gantt.tickcolor != current->gantt.tickcolor) {
    XtDestroyGC(current->gantt.tickGC);
    gcValues.foreground = new->gantt.tickcolor;
    gcMask = GCForeground;
    new->gantt.tickGC = XtGetGC(gcurrent, gcMask, &gcValues);
  }

  /* change of height or width */
  if ((new->core.height != current->core.height)
      || (new->core.width != current->core.width)) {
    new->gantt.height = new->core.height;
    new->gantt.width = new->core.width;
    redraw = TRUE;
  }

  /* change number of tick marks */
  if ((new->gantt.xticks != current->gantt.xticks)
      || (new->gantt.yticks != current->gantt.yticks)) {
    EraseTicks(current);
    DrawTicks(new, new->gantt.tickGC);
  }

  /* change of minimum and maximum values */
  if ((new->gantt.xmin != current->gantt.xmin)
      || (new->gantt.xmax != current->gantt.xmax)
      || (new->gantt.ymin != current->gantt.ymin)
      || (new->gantt.ymax != current->gantt.ymax)) {
    redraw = TRUE;
  }

  return(redraw);

} /* SetValues */

/************************************************************************/
/* Destroy()								*/
/************************************************************************/
static void Destroy(bgw)
Widget bgw;
{
  GanttWidget w = (GanttWidget) bgw;

#ifdef DEBUG
printf("Destroy\n");
#endif
  XtDestroyGC(w->gantt.eraseGC);
  XtDestroyGC(w->gantt.tickGC);
  DestroyColorGCs(w);
  XtRemoveAllCallbacks(bgw, XtNselect);

} /* destroy */

/************************************************************************/
/* CreateColorGCs()							*/
/************************************************************************/
static void CreateColorGCs(w)
GanttWidget	w;
{
  int		i;
  XGCValues	gcValues;
  XtGCMask	gcMask;
    
#ifdef DEBUG
printf("CreateColorGCs\n");
#endif
  gcMask = GCForeground;
  for (i=0; ((i<w->gantt.numcolors) && (i<MAXCOLORS)); i++) {
    gcValues.foreground = w->gantt.colors[i];
    w->gantt.colorGCs[i] = XtGetGC( (Widget)w, gcMask, &gcValues);
  }
  for (i=w->gantt.numcolors; i<MAXCOLORS; i++) {
    if (XWhitePixelOfScreen(XtScreen(w)) == w->core.background_pixel)
      gcValues.foreground = XBlackPixelOfScreen(XtScreen(w));
    else
      gcValues.foreground = XWhitePixelOfScreen(XtScreen(w));
    w->gantt.colorGCs[i] = XtGetGC( (Widget)w, gcMask, &gcValues);
  }

} /* CreateColorGCs */

/************************************************************************/
/* DestroyColorGCs()							*/
/************************************************************************/
static void DestroyColorGCs(w)
GanttWidget	w;
{
  register	i;
  
#ifdef DEBUG
printf("DestroyColorGCs\n");
#endif
  for (i=0; i<MAXCOLORS; i++)
      XtDestroyGC(w->gantt.colorGCs[i]);

} /* DestroyColorGCs */

/************************************************************************/
/* Resize()								*/
/************************************************************************/
static void Resize(gw)
Widget	gw;
{
  GanttWidget	w = (GanttWidget) gw;

#ifdef DEBUG
printf("Resize\n");
#endif
  w->gantt.height = w->core.height;
  w->gantt.width = w->core.width;
  Redisplay(gw);

} /* Resize */

/************************************************************************/
/* Redisplay()								*/
/************************************************************************/
static void Redisplay(gw, event, region)
Widget	gw;
XEvent	*event;
Region	region;
{
  GanttWidget	w = (GanttWidget) gw;

#ifdef DEBUG
printf("Redisplay\n");
#endif
  if (XtIsRealized((Widget) w)) {
     ClearGantt(w);
     DrawGantt(w);
  }

} /* Redisplay */

/************************************************************************/
/* ClearGantt() clears the Gantt chart.					*/
/************************************************************************/
static void ClearGantt(w)
GanttWidget	w;
{
#ifdef DEBUG
printf("ClearGantt\n");
#endif
  XClearArea(XtDisplay(w), XtWindow(w), 0, 0,
	     w->gantt.width, w->gantt.height, FALSE);

} /* ClearGantt */

/************************************************************************/
/* DrawGantt() draw the entire Gantt chart.				*/
/************************************************************************/
static void DrawGantt(w)
GanttWidget	w;
{
  DrawTicks(w, w->gantt.tickGC);
  DrawLines(w);

} /* DrawGantt */

/************************************************************************/
/* DrawBorder(w) draws a border around the Gantt chart.			*/
/************************************************************************/
static void DrawBorder(w)
GanttWidget	w;
{
  int	width = w->gantt.width, height = w->gantt.height;
  GC	gc = w->gantt.tickGC;

  XDrawRectangle(XtDisplay(w), XtWindow(w), gc,
		 0, 0, width, height);

} /* DrawBorder */

/************************************************************************/
/* DrawTicks() draws tick marks on the Gantt chart.			*/
/************************************************************************/
static void DrawTicks(w, gc)
GanttWidget	w;
GC		gc;
{
  int	i;
  int	xticks = w->gantt.xticks, yticks = w->gantt.yticks;
  int	height = w->gantt.height, width = w->gantt.width;
  float	delta;

#ifdef DEBUG
printf("DrawTicks\n");
#endif
  if (xticks > 0) {
    if (w->gantt.orientation == XtorientVertical) {
      delta = (float) height / (float) xticks;
      for (i=1; i<xticks; i++) {
	XDrawLine(XtDisplay(w), XtWindow(w), gc,
		0, (int) ((float) i * delta),
		TICKLENGTH, (int) ((float) i * delta));
      }
    }
    else {
      delta = (float) width / (float) xticks;
      for (i=1; i<xticks; i++) {
	XDrawLine(XtDisplay(w), XtWindow(w), gc,
		  (int) ((float) i * delta), height,
		  (int) ((float) i * delta), height - TICKLENGTH);
      }
    }
  }
  if (yticks > 0) {
    if (w->gantt.orientation == XtorientVertical) {
      delta = (float) width / (float) yticks;
      for (i=1; i<yticks; i++) {
	XDrawLine(XtDisplay(w), XtWindow(w), gc,
		  (int) ((float) i * delta), height,
		  (int) ((float) i * delta), height - TICKLENGTH);
      }
    }
    else {
      delta = (float) height / (float) yticks;
      for (i=1; i<yticks; i++) {
	XDrawLine(XtDisplay(w), XtWindow(w), gc,
		  0, (int) ((float) i * delta),
		  TICKLENGTH, (int) ((float) i * delta));
      }
    }
  }

} /* DrawTicks */

/************************************************************************/
/* EraseTicks() erases tick marks on the Gantt chart.			*/
/************************************************************************/
static void EraseTicks(w)
GanttWidget	w;
{
  int	i;
  float	delta;

#ifdef DEBUG
printf("EraseTicks\n");
#endif
  DrawTicks(w, w->gantt.eraseGC);

} /* EraseTicks */

/************************************************************************/
/* DrawLine() draws the line lid in the Gantt chart using color GC gc.	*/
/************************************************************************/
static void DrawLine(w, lid, gc)
GanttWidget	w;
int		lid;
GC		gc;
{
  int	i, b, p = w->gantt.p[lid];
  float	xmin = (float) w->gantt.xmin, xmax = (float) w->gantt.xmax;
  float	ymin = (float) w->gantt.ymin, ymax = (float) w->gantt.ymax;
  float	xdiff = xmax - xmin, ydiff = ymax - ymin;
  float	xprev, xcurr, yprev, ycurr;

#ifdef DEBUG
printf("DrawLine\n");
#endif

  if (p != 0) {
/*
    for (b=0; b<p; b++) {
      if ((xmin <= w->gantt.lx[lid][b]) &&
	  (xmax >= w->gantt.lx[lid][b]) &&
	  (ymin <= w->gantt.ly[lid][b]) &&
	  (ymax >= w->gantt.ly[lid][b]))
	break;
    }
*/
    b = 0;
    if (b >= p)
      return;
    if (w->gantt.orientation == XtorientVertical) {
      yprev = (((float) w->gantt.lx[lid][b] - xmin) / xdiff);
      xprev = (((float) w->gantt.ly[lid][b] - ymin) / ydiff);
    }
    else {
      xprev = (((float) w->gantt.lx[lid][b] - xmin) / xdiff);
      yprev = ((ymax - (float) w->gantt.ly[lid][0] - ymin) / ydiff);
    }
    xprev = (float) w->gantt.width * xprev;
    yprev = (float) w->gantt.height * yprev;
    for (i=b+1; i<p; i++) {
      if (w->gantt.orientation == XtorientVertical) {
	ycurr = (((float) w->gantt.lx[lid][i] - xmin) / xdiff);
	xcurr = (((float) w->gantt.ly[lid][i] - ymin) / ydiff);
      }
      else {
	xcurr = (((float) w->gantt.lx[lid][i] - xmin) / xdiff);
	ycurr = ((ymax - (float) w->gantt.ly[lid][i] - ymin) / ydiff);
      }
      xcurr = (float) w->gantt.width * xcurr;
      ycurr = (float) w->gantt.height * ycurr;
      XDrawLine(XtDisplay(w), XtWindow(w), gc,
		(int) (xprev + 0.5), (int) (yprev + 0.5),
		(int) (xcurr + 0.5), (int) (ycurr + 0.5));
      xprev = xcurr;
      yprev = ycurr;
/*
      if ((xmax <= w->gantt.lx[lid][i]) && (ymax <= w->gantt.ly[lid][i]))
	break;
*/
    }
  }

} /* DrawLine */

/************************************************************************/
/* EraseLine() erases line lid from the Gantt chart.			*/
/************************************************************************/
static void EraseLine(w, lid)
GanttWidget	w;
int		lid;
{

#ifdef DEBUG
printf("EraseLine\n");
#endif
  if (XtIsRealized((Widget) w)) {
    DrawLine(w, lid, w->gantt.eraseGC);
  }

} /* EraseLine */

/************************************************************************/
/* DrawLines() draws the lines currently defined for the Gantt chart	*/
/* in their associated color.						*/
/************************************************************************/
static void DrawLines(w)
GanttWidget	w;
{
  int	i;

#ifdef DEBUG
printf("DrawLines\n");
#endif
  if (XtIsRealized((Widget) w)) {
    for (i=0; i<MAXLINES; i++) {
      DrawLine(w, i, w->gantt.colorGCs[w->gantt.c[i]]);
    }
  }

} /* DrawLines */

/************************************************************************/
/* GanttNotify() processes button up event.				*/
/************************************************************************/
static void GanttNotify(w, event)
GanttWidget		w;
XButtonReleasedEvent	*event;
{
#ifdef DEBUG
printf("GanttNotify\n");
#endif

  XtCallCallbacks( (Widget)w, XtNselect, &(w->gantt.location));

} /* GanttNotify */

/************************************************************************/
/* GanttSelect() processes button down event.				*/
/************************************************************************/
static void GanttSelect(w, event)
GanttWidget		w;
XButtonPressedEvent	*event;
{
  int	i, b, p;
  float	xmin = (float) w->gantt.xmin, xmax = (float) w->gantt.xmax;
  float	ymin = (float) w->gantt.ymin, ymax = (float) w->gantt.ymax;
  float	xdiff = xmax - xmin, ydiff = ymax - ymin;
  float	xprev, xcurr, yprev, ycurr;
  int	xloc = event -> x, yloc = event -> y;

  int	MinD, Line, LineLoc, tmp, lid;

#ifdef DEBUG
printf("FindDist\n");
#endif

 for (lid = 0; lid < MAXLINES; lid++) {

  p = w->gantt.p[lid];
  MinD = xmax + ymax;

  if (p != 0) {
    b = 0;
    if (b >= p)
      return;
    if (w->gantt.orientation == XtorientVertical) {
      yprev = (((float) w->gantt.lx[lid][b] - xmin) / xdiff);
      xprev = (((float) w->gantt.ly[lid][b] - ymin) / ydiff);
    }
    else {
      xprev = (((float) w->gantt.lx[lid][b] - xmin) / xdiff);
      yprev = ((ymax - (float) w->gantt.ly[lid][0] - ymin) / ydiff);
    }
    xprev = (float) w->gantt.width * xprev;
    yprev = (float) w->gantt.height * yprev;
    for (i=b+1; i<p; i++) {
      if (w->gantt.orientation == XtorientVertical) {
	ycurr = (((float) w->gantt.lx[lid][i] - xmin) / xdiff);
	xcurr = (((float) w->gantt.ly[lid][i] - ymin) / ydiff);
      }
      else {
	xcurr = (((float) w->gantt.lx[lid][i] - xmin) / xdiff);
	ycurr = ((ymax - (float) w->gantt.ly[lid][i] - ymin) / ydiff);
      }
      xcurr = (float) w->gantt.width * xcurr;
      ycurr = (float) w->gantt.height * ycurr;
      tmp = Euclid(xloc, yloc, (int) (xcurr + 0.5), (int) (ycurr + 0.5));

      if (tmp < MinD) { 
	MinD = tmp;
	LineLoc = i;
 	Line = lid;
      }

      xprev = xcurr;
      yprev = ycurr;
    }
  }
 }

/* 17 Jan 1994: PCR:  Fixed the 'found no line' test 			*/
/* If no line was found, MinD should be xmax + ymax, not xmax * ymax 	*/
  if (MinD == xmax + ymax) {
      w -> gantt.location.x = -1;
      w -> gantt.location.y = -1;
      w -> gantt.location.lid = -1;
  } else {
      w -> gantt.location.x = w->gantt.lx[Line][LineLoc];
      w -> gantt.location.y = w->gantt.ly[Line][LineLoc];
      w -> gantt.location.lid = Line;
  }

} /* GanttSelect */

/************************************************************************/
/*	Euclid() finds the Euclidean distance between a pair of points.	*/ 
/************************************************************************/
static int Euclid(x1, y1, x2, y2)
int		x1, y1, x2, y2;
{
	int	Idist;
	double	Dist, dist;

#ifdef DEBUG
	printf("Euclid x1: %d y1: %d x2: %d y2: %d\n", x1, y1, x2, y2);
#endif

	dist = (x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2);
	Dist = sqrt (dist);
	Idist = (int) Dist;

#ifdef DEBUG
	printf("Euclid dist: %f Idist: %d\n", dist, Idist);
#endif

	return (Idist);
}

/************************************************************************/
/*                          PUBLIC PROCEDURES                           */
/************************************************************************/

/************************************************************************/
/* GanttSetColorMap() establishes the color map.  numcolors is the	*/
/* number of colors in the map and colors is a pointer to an array of	*/
/* pixels.  GanttShowGantt() must be called before the new colors are	*/
/* activated.								*/
/************************************************************************/
extern void GanttSetColorMap(w, numcolors, colors)
GanttWidget	w;
int		numcolors;
Pixel		*colors;
{
#ifdef DEBUG
printf("GanttSetColorMap\n");
#endif
  DestroyColorGCs(w);
  w->gantt.numcolors = numcolors;
  w->gantt.colors = colors;
  CreateColorGCs(w);

  if (XtIsRealized((Widget) w))
     DrawLines(w);

} /* GanttSetColorMap */

/************************************************************************/
/* GanttAddLine() loads a new line into the Gantt widget.  The argument	*/
/* lid gives the id of the line, n is the number of points in the line,	*/
/* cid is the color choice, and x and y are pointers to arrays of x and	*/
/* y coordinate values, respectively.  GanttShowGantt() must be called	*/
/* before the new line is displayed.  By default, new lines are mapped.	*/
/************************************************************************/
extern void GanttAddLine(w, lid, n, color, x, y)
GanttWidget	w;
int		lid;			/* line id			*/
int		n;			/* number of pints in line	*/
int		color;			/* color choice			*/
int		*x, *y;			/* x and y values		*/
{
  int	i;

#ifdef DEBUG
printf("GanttAddLine\n");
#endif
  if (lid > MAXLINES) {
    printf("GanttAddLine: line id %d greater than max lines %d\n",
	   lid, MAXLINES);
    return;
  }
  if (n > MAXPOINTS) {
    printf("GanttAddLine: line id %d has too many points (%d)\n",
	   lid, n);
    return;
  }
  w->gantt.lx[lid] = x;
  w->gantt.ly[lid] = y;
  w->gantt.p[lid] = n;
  if (color > MAXCOLORS)
    color = 0;
  w->gantt.c[lid] = color;

} /* GanttAddLine */

/************************************************************************/
/* GanttDeleteLine() deletes the line lid.  GanttShowGantt() must be	*/
/* called before the line is removed from the display.			*/
/************************************************************************/
extern void GanttDeleteLine(w, lid)
GanttWidget	w;
int		lid;
{
#ifdef DEBUG
printf("GanttDeleteLine\n");
#endif
  w->gantt.p[lid] = 0;

} /* GanttDeleteLine */

/************************************************************************/
/* GanttMapLine() maps (i.e. displays) the line lid in the gantt chart.	*/
/************************************************************************/
extern void GanttMapLine(w, lid)
GanttWidget	w;
int		lid;
{
#ifdef DEBUG
printf("GanttMapLine\n");
#endif
  if (XtIsRealized((Widget) w))
     DrawLine(w, lid, w->gantt.colorGCs[w->gantt.c[lid]]);

} /* GanttMapLine */

/************************************************************************/
/* GanttUnMapLine() unmaps (i.e. erases) the line lid in the gantt	*/
/* chart.								*/
/************************************************************************/
extern void GanttUnMapLine(w, lid)
GanttWidget	w;
int		lid;
{
#ifdef DEBUG
printf("GanttUnMapLine\n");
#endif

  if (XtIsRealized((Widget) w))
     EraseLine(w, lid);

} /* GanttUnMapLine */

/************************************************************************/
/* GanttShowGantt() draws the gantt chart.				*/
/************************************************************************/
extern void GanttShowGantt(w)
GanttWidget	w;
{
#ifdef DEBUG
printf("GanttShowGantt\n");
#endif

  if (XtIsRealized((Widget) w))
     DrawGantt(w);

} /* GanttShowGantt */

/************************************************************************/
/* GanttClearGantt() clears the gantt chart.				*/
/************************************************************************/
extern void GanttClearGantt(w)
GanttWidget	w;
{
#ifdef DEBUG
printf("GanttClearGantt\n");
#endif

  if (XtIsRealized((Widget) w)) 
     ClearGantt(w);

} /* GanttClearGantt */
