/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */


#include <stdio.h>
#include <limits.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include "BoxAndWhiskerP.h"

#define REALLOC_INTERVAL    10

#define Offset(field)       XtOffset( BoxAndWhiskerWidget, field )


/* Private procedures */
static void      ClassInitialize(), Initialize();
static void      Realize(), Resize(), Redisplay(), Destroy();
static Boolean   SetValues();

static void      Exchange();
static void      DrawBox(), DrawWhiskers();
static void      BoxAndWhiskerSelect(), BoxAndWhiskerNotify();
static int       BoxAndWhiskerSelectElement(), BoxAndWhiskerGetPctile();

/* Public procedures */
extern void      BoxAndWhiskerSetValue(), BoxAndWhiskerSetValues();

/* Utility procedures */


/* Box&Whisker translations */
static char boxAndWhisker_translations[] = " \
  <Btn1Down>:    BoxAndWhiskerSelect()\n     \
  <Btn1Up>:      BoxAndWhiskerNotify()       \
";

/* Box&Whisker actions */
static XtActionsRec boxAndWhisker_actions[] = {
  {"BoxAndWhiskerSelect",    (XtActionProc) BoxAndWhiskerSelect },
  {"BoxAndWhiskerNotify",    (XtActionProc) BoxAndWhiskerNotify },
};

/* Box&Whisker resources */
static XtResource resources[] = {
   /* CORE */
   {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
      Offset(core.width), XtRString, "100"},
   {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
      Offset(core.height), XtRString, "200"},

   /* Box&Whisker resources */
   {XtNminValue, XtCMinValue, XtRInt, sizeof(int),
      Offset(boxAndWhisker.minValue), XtRString, "0"},
   {XtNmaxValue, XtCMaxValue, XtRInt, sizeof(int),
      Offset(boxAndWhisker.maxValue), XtRString, "100000"},
   {XtNminValues, XtCMinValues, XtRInt, sizeof(int),
      Offset(boxAndWhisker.minValues), XtRString, NULL },
   {XtNmaxValues, XtCMaxValues, XtRInt, sizeof(int),
      Offset(boxAndWhisker.maxValues), XtRString, NULL},

   {XtNsmallestValue, XtCSmallestValue, XtRInt, sizeof(int),
      Offset(boxAndWhisker.smallestValue), XtRString, "0" },
   {XtNlargestValue, XtCLargestValue, XtRInt, sizeof(int),
      Offset(boxAndWhisker.largestValue), XtRString, "0" },
   {XtNmedianValue, XtCMedianValue, XtRInt, sizeof(int),
      Offset(boxAndWhisker.median), XtRString, "0" },
   {XtNQ1Value, XtCQ1Value, XtRInt, sizeof(int),
      Offset(boxAndWhisker.Q1value), XtRString, "0" },
   {XtNQ3Value, XtCQ3Value, XtRInt, sizeof(int),
      Offset(boxAndWhisker.Q3value), XtRString, "0" },

   {XtNsmallestIndex, XtCSmallestIndex, XtRInt, sizeof(int),
      Offset(boxAndWhisker.smallestIndex), XtRString, "0" },
   {XtNlargestIndex, XtCLargestIndex, XtRInt, sizeof(int),
      Offset(boxAndWhisker.largestIndex), XtRString, "0" },
   {XtNmedianIndex, XtCMedianIndex, XtRInt, sizeof(int),
      Offset(boxAndWhisker.medianIndex), XtRString, "0" },
   {XtNQ1Index, XtCQ1Index, XtRInt, sizeof(int),
      Offset(boxAndWhisker.Q1index), XtRString, "0" },
   {XtNQ3Index, XtCQ3Index, XtRInt, sizeof(int),
      Offset(boxAndWhisker.Q3index), XtRString, "0" },

   {XtNdata, XtCData, XtRPointer, sizeof(int **),
      Offset(boxAndWhisker.data), XtRString, NULL},
   {XtNdataValues, XtCDataValues, XtRPointer, sizeof(int *),
      Offset(boxAndWhisker.dataValues), XtRString, NULL},
   {XtNdataCnt, XtCDataCnt, XtRInt, sizeof(int),
      Offset(boxAndWhisker.dataValueCnt), XtRString, "0" },
   {XtNpadding, XtCPadding, XtRInt, sizeof(int),
      Offset(boxAndWhisker.padding), XtRString, "4" },
   {XtNboxColor, XtCBoxColor, XtRPixel, sizeof(Pixel),
      Offset(boxAndWhisker.boxColor), XtRString, "black" },
   {XtNwhiskerColor, XtCWhiskerColor, XtRPixel, sizeof(Pixel),
      Offset(boxAndWhisker.whiskerColor), XtRString, "black" },
   {XtNbaseColors, XtCBaseColors, XtRPointer, sizeof(Pixel *),
      Offset(boxAndWhisker.baseColors), XtRString, NULL },
   {XtNorientation, XtCOrientation, XtROrientation, sizeof(XtOrientation),
      Offset(boxAndWhisker.orientation), XtRString, "horizontal" },
   {XtNselect, XtCCallback, XtRCallback, sizeof(caddr_t),
      Offset(boxAndWhisker.select), XtRCallback, (caddr_t) NULL },
};
#undef Offset

/* Box&Whisker class record */
BoxAndWhiskerClassRec boxAndWhiskerClassRec = {
  (WidgetClass) &widgetClassRec,        /* superclass                   */
  "BoxAndWhisker",                      /* class_name                   */
  sizeof(BoxAndWhiskerRec),             /* size                         */
  ClassInitialize,                      /* class_initialize             */
  NULL,                                 /* class_part_initialize        */
  FALSE,                                /* class_init'ed                */
  Initialize,                           /* initialize                   */
  NULL,                                 /* initialize_hook              */
  Realize,                              /* realize                      */
  boxAndWhisker_actions,                /* actions                      */
  XtNumber(boxAndWhisker_actions),      /* num_actions                  */
  resources,                            /* resources                    */
  XtNumber(resources),                  /* resource_count               */
  NULLQUARK,                            /* xrm_class                    */
  TRUE,                                 /* compress_motion              */
  TRUE,                                 /* compress_exposure            */
  TRUE,                                 /* compress_enterleave          */
  FALSE,                                /* visible_interest             */
  Destroy,                              /* destroy                      */
  Resize,                               /* resize                       */
  Redisplay,                            /* expose                       */
  SetValues,                            /* set_values                   */
  NULL,                                 /* set_values_hook              */
  XtInheritSetValuesAlmost,             /* set_values_almost            */
  NULL,                                 /* get_values_hook              */
  NULL,                                 /* accept_focus                 */
  XtVersion,                            /* version                      */
  NULL,                                 /* callback_private             */
  boxAndWhisker_translations,           /* translationManager table     */
  XtInheritQueryGeometry,               /* query_geometry               */
  XtInheritDisplayAccelerator           /* display_accelerator          */
};

WidgetClass boxAndWhiskerWidgetClass = (WidgetClass)&boxAndWhiskerClassRec;

static XrmQuark  XtQEhorizontal;
static XrmQuark  XtQEvertical;


/* --------------------------------------------------------------------- */
/*                           PRIVATE PROCEDURES                          */
/* --------------------------------------------------------------------- */

/* DESTROY() */
static void Destroy(bww)
Widget bww;
{
     BoxAndWhiskerWidget w = (BoxAndWhiskerWidget) bww;

} /* DESTROY */


/* CVTSTRINGTOORIENTATION() */

static void CvtStringToOrientation(args, num_args, fromVal, toVal)
XrmValuePtr    *args;         /* unused */
Cardinal       *num_args;     /* unused */
XrmValuePtr    fromVal;
XrmValuePtr    toVal;
{
     static XtOrientation orient;
     XrmQuark             q;
     char                 lowerCaseName[80];

/* / convert incoming string name to lower case / */
     strcpy( lowerCaseName, fromVal->addr );
/* LowerCase( (char *) fromVal->addr, lowerCaseName ); */
     q = XrmStringToQuark(lowerCaseName);

     if (q == XtQEhorizontal) {
        orient = XtorientHorizontal;
	(*toVal).size = sizeof(XtOrientation);
	(*toVal).addr = (caddr_t) &orient;
	return;
     } 
     if (q == XtQEvertical) {
        orient = XtorientVertical;
	(*toVal).size = sizeof(XtOrientation);
	(*toVal).addr = (caddr_t) &orient;
	return;
     }
}
/* CVTSTRINGTOORIENTATION */


/* CLASSINITIALIZE */
static void ClassInitialize() 
{
     XtQEhorizontal = XrmStringToQuark(XtEhorizontal);
     XtQEvertical = XrmStringToQuark(XtEvertical);
     XtAddConverter( XtRString, XtROrientation, CvtStringToOrientation,
		     NULL, (Cardinal)0 );
} /* CLASSINITIALIZE */


/* INITIALIZE */
static void Initialize( request, new )
Widget request;
Widget new;
{
     BoxAndWhiskerWidget     w = (BoxAndWhiskerWidget) new;
     XGCValues               gcValues;
     XtGCMask                gcMask;

     /* initialize base color set */
     w->boxAndWhisker.baseColors = (Pixel *) XtMalloc( 3 * sizeof(Pixel) );
     w->boxAndWhisker.baseColors[0] = w->boxAndWhisker.boxColor;
     w->boxAndWhisker.baseColors[1] = w->boxAndWhisker.whiskerColor;
     w->boxAndWhisker.baseColors[2] = -1;

     /* initialize box GC */
     gcValues.foreground = w->boxAndWhisker.boxColor;
     gcMask = GCForeground;
     w->boxAndWhisker.boxGC = XtGetGC( new, gcMask, &gcValues );

     /* initialize whisker GC */
     gcValues.foreground = w->boxAndWhisker.whiskerColor;
     gcMask = GCForeground;
     w->boxAndWhisker.whiskerGC = XtGetGC( new, gcMask, &gcValues );

     /* initialize the pixmap GC */
     gcValues.foreground = w->core.background_pixel;
     gcMask = GCForeground;
     w->boxAndWhisker.pixmapGC = XtGetGC( new, gcMask, &gcValues );

     /* initialize variables */
     w->boxAndWhisker.minValues = (int *)XtMalloc( sizeof(int) );
     w->boxAndWhisker.maxValues = (int *)XtMalloc( sizeof(int) );
     w->boxAndWhisker.minValues[0] = w->boxAndWhisker.minValue;
     w->boxAndWhisker.maxValues[0] = w->boxAndWhisker.maxValue;

     w->boxAndWhisker.data = NULL;
     w->boxAndWhisker.dataValues = NULL;
     w->boxAndWhisker.dataValueCnt = 0;

} /* INITIALIZE */


/* SETVALUES */

static Boolean SetValues( wcurrent, wrequest, wnew )
Widget wcurrent;
Widget wrequest;
Widget wnew;
{
      BoxAndWhiskerWidget    current = (BoxAndWhiskerWidget) wcurrent;
      BoxAndWhiskerWidget    request = (BoxAndWhiskerWidget) wrequest;
      BoxAndWhiskerWidget    new = (BoxAndWhiskerWidget) wnew;
      XGCValues    gcValues;
      XtGCMask     gcMask;
      Boolean      redraw = FALSE, newdata = FALSE;
      Boolean      realized = XtIsRealized( wcurrent );


      int **ourData, *ourMinValue, *ourMaxValue;
      static Pixel *ourColors = NULL;

      /* generic color set */
      if ( ourColors == NULL ) {
         ourColors = (Pixel *)XtMalloc( 3 * sizeof(Pixel) );
	 ourColors[2] = -1;
	 new->boxAndWhisker.baseColors = ourColors;
      }

      /* change of box color */
      if ( new->boxAndWhisker.boxColor != current->boxAndWhisker.boxColor ) {
         XtDestroyGC( current->boxAndWhisker.boxGC );
	 gcValues.foreground = new->boxAndWhisker.boxColor;
	 gcMask = GCForeground;
	 new->boxAndWhisker.boxGC = XtGetGC( current, gcMask, &gcValues );
	 new->boxAndWhisker.baseColors[0] = new->boxAndWhisker.boxColor;
	 redraw = TRUE;
      }

      /* change of whisker color */
      if ( new->boxAndWhisker.whiskerColor != 
	                              current->boxAndWhisker.whiskerColor ) {
         XtDestroyGC( current->boxAndWhisker.whiskerGC );
	 gcValues.foreground = new->boxAndWhisker.whiskerColor;
	 gcMask = GCForeground;
	 new->boxAndWhisker.whiskerGC = XtGetGC( current, gcMask, &gcValues );
	 new->boxAndWhisker.baseColors[1] = new->boxAndWhisker.whiskerColor;
	 redraw = TRUE;
      }

      /* change of size */
      if ( (new->core.height != current->core.height ) ||
           (new->core.width != current->core.width ) ) {
      }

      /* min/max change */
      if ( request->boxAndWhisker.minValues != NULL ) {
         ourMinValue = (int *)XtMalloc( 2 * sizeof(int) );
	 ourMaxValue = (int *)XtMalloc( 2 * sizeof(int) );
	 ourMinValue[0] = new->boxAndWhisker.minValues[0];
	 ourMinValue[1] = NULL;
	 ourMaxValue[0] = new->boxAndWhisker.maxValues[0];
	 ourMaxValue[1] = NULL;
	 new->boxAndWhisker.minValues = ourMinValue;
	 new->boxAndWhisker.maxValues = ourMaxValue;
	 new->boxAndWhisker.minValue = new->boxAndWhisker.minValues[0];
	 new->boxAndWhisker.maxValue = new->boxAndWhisker.maxValues[0];
	 redraw = TRUE;
      }

      /* new data */
      if ( ( current == NULL ) && ( new->boxAndWhisker.data != NULL ) ) {
         newdata = TRUE;
      } else if ( new->boxAndWhisker.data != current->boxAndWhisker.data ) {
         newdata = TRUE;
      }
      if ( ( newdata ) && ( new->boxAndWhisker.dataValueCnt > 0 ) ) {
	 new->boxAndWhisker.dataValues = (int *)XtMalloc( 
			     new->boxAndWhisker.dataValueCnt * sizeof(int) );
	 (void)memcpy( new->boxAndWhisker.dataValues,
		       new->boxAndWhisker.data[0], 
		       new->boxAndWhisker.dataValueCnt * sizeof(int) );
	 new->boxAndWhisker.dataIndices =(int *)XtMalloc(
			     new->boxAndWhisker.dataValueCnt * sizeof(int) );
	 ourData = (int **)XtMalloc( 2 * sizeof(int *) );
	 new->boxAndWhisker.data = ourData;
	 new->boxAndWhisker.data[0] = new->boxAndWhisker.dataValues;
	 new->boxAndWhisker.data[1] = NULL;
	 redraw = TRUE;
      } 

      if ( redraw && realized ) {
          Redisplay(new);
      }

      return( redraw );
} /* SETVALUES */


/* REALIZE */
static void Realize(gw, GCmask, attrs)
Widget                  gw;
XtGCMask                *GCmask;
XSetWindowAttributes    *attrs;
{
  BoxAndWhiskerWidget w = (BoxAndWhiskerWidget) gw;
  
  *GCmask |= CWBitGravity;
  attrs->bit_gravity = ForgetGravity;
  switch(w->boxAndWhisker.backing_store) {
  case Always:
  case NotUseful:
  case WhenMapped:
    *GCmask |=CWBackingStore;
    attrs->backing_store = w->boxAndWhisker.backing_store;
    break;
  }
  /* Create the widget window */
  XtCreateWindow(gw, InputOutput,(Visual *)CopyFromParent,
		 *GCmask, attrs);
  Resize(gw);
}

      
/* RESIZE */
static void Resize(bww)
Widget bww;
{
      BoxAndWhiskerWidget    w = (BoxAndWhiskerWidget) bww;

      if ( XtIsRealized(bww) ) {
         if ( w->boxAndWhisker.dataValues != NULL ) {
            BoxAndWhiskerSetValues( w, w->boxAndWhisker.dataValues,
				    w->boxAndWhisker.dataValueCnt );
	  }
      }
} /* RESIZE */

      
/* REDISPLAY */
static void Redisplay( bww, event, region )
Widget bww;
XEvent *event;
Region region;
{
      BoxAndWhiskerWidget    w = (BoxAndWhiskerWidget) bww;
      
      if ( XtIsRealized(bww) ) {
         DrawBox(w);
	 DrawWhiskers(w);
      }
} /* REDISPLAY */



/* Exchange */
static void Exchange( a, b )
int *a, *b;
{
      int temp;
      temp = *a;
      *a = *b;
      *b = temp;
}


/* DrawBox */
static void DrawBox( w )
BoxAndWhiskerWidget w;
{
      int boxTop, boxBottom, boxMinExtent, boxMaxExtent, medianExtent;
      int boxValueRange = w->boxAndWhisker.maxValue-w->boxAndWhisker.minValue;
      int usableWidth = w->core.width - 2 * w->boxAndWhisker.padding;

      if ((!XtIsRealized( (Widget)w )) || (w->boxAndWhisker.dataValueCnt==0))
         return;

      if ( w->boxAndWhisker.orientation == XtorientHorizontal ) {
	     boxTop = w->boxAndWhisker.padding;
	     boxBottom = w->core.height - w->boxAndWhisker.padding;
	     boxMinExtent =
                ((float)(w->boxAndWhisker.Q1value - w->boxAndWhisker.minValue)/
		(float)boxValueRange) * usableWidth + w->boxAndWhisker.padding;
	     boxMaxExtent =
                ((float)(w->boxAndWhisker.Q3value - w->boxAndWhisker.minValue)/
		(float)boxValueRange) * usableWidth + w->boxAndWhisker.padding;
	     medianExtent = 
                ((float)(w->boxAndWhisker.median - w->boxAndWhisker.minValue) /
		(float)boxValueRange) * usableWidth + w->boxAndWhisker.padding;
	     XDrawLine( XtDisplay(w), XtWindow(w), w->boxAndWhisker.boxGC,
                        medianExtent,boxTop,   medianExtent,boxBottom );
	       
       } else if ( w->boxAndWhisker.orientation == XtorientVertical ) {
	     boxTop =
	        ((float)(w->boxAndWhisker.Q1value - w->boxAndWhisker.minValue)/
	       (float)boxValueRange ) * usableWidth + w->boxAndWhisker.padding;
	     boxBottom = 
	        ((float)(w->boxAndWhisker.Q3value - w->boxAndWhisker.minValue)/
	       (float)boxValueRange ) * usableWidth + w->boxAndWhisker.padding;
	     boxMinExtent = w->boxAndWhisker.padding;
	     boxMaxExtent = w->core.width - w->boxAndWhisker.padding;
	     medianExtent =
	        ((float)(w->boxAndWhisker.median - w->boxAndWhisker.minValue) /
	       (float)boxValueRange ) * usableWidth + w->boxAndWhisker.padding;
             XDrawLine( XtDisplay(w), XtWindow(w), w->boxAndWhisker.boxGC,
                        boxMinExtent,medianExtent, boxMaxExtent,medianExtent );
       } 
       /* lines drawn in order [top, left, bottom, right] */
       XDrawLine( XtDisplay(w), XtWindow(w), w->boxAndWhisker.boxGC,
      	          boxMaxExtent,boxTop,   boxMinExtent,boxTop );
       XDrawLine( XtDisplay(w), XtWindow(w), w->boxAndWhisker.boxGC,
	          boxMinExtent,boxTop,   boxMinExtent,boxBottom );
       XDrawLine( XtDisplay(w), XtWindow(w), w->boxAndWhisker.boxGC,
		  boxMinExtent,boxBottom,   boxMaxExtent,boxBottom );
       XDrawLine( XtDisplay(w), XtWindow(w), w->boxAndWhisker.boxGC,
		  boxMaxExtent,boxBottom,   boxMaxExtent,boxTop );
} /* DrawBox */


/* DrawWhiskers */
static void DrawWhiskers(w)
BoxAndWhiskerWidget w;
{
     int boxMinExtent, boxMaxExtent;
     int whiskerMinExtent, whiskerMaxExtent;
     int boxValueRange = w->boxAndWhisker.maxValue-w->boxAndWhisker.minValue;
     int usableWidth = w->core.width - 2 * w->boxAndWhisker.padding;

     if ( (!(XtIsRealized( (Widget)w))) || (w->boxAndWhisker.dataValueCnt==0)) 
        return;

     if ( w->boxAndWhisker.orientation == XtorientHorizontal ) {
         boxMinExtent = 
	      ((float)(w->boxAndWhisker.Q1value - w->boxAndWhisker.minValue) /
	       (float)boxValueRange) * usableWidth + w->boxAndWhisker.padding;
	 boxMaxExtent =
	      ((float)(w->boxAndWhisker.Q3value - w->boxAndWhisker.minValue) /
	       (float)boxValueRange) * usableWidth + w->boxAndWhisker.padding;
	 whiskerMinExtent =
          ((float)(w->boxAndWhisker.smallestValue - w->boxAndWhisker.minValue)/
	       (float)boxValueRange ) * usableWidth + w->boxAndWhisker.padding;
	 whiskerMaxExtent = 
          ((float)(w->boxAndWhisker.largestValue - w->boxAndWhisker.minValue)/
	       (float)boxValueRange ) * usableWidth + w->boxAndWhisker.padding;
         /* trailing whisker */
         XDrawLine( XtDisplay(w), XtWindow(w), w->boxAndWhisker.whiskerGC,
	            boxMinExtent, w->core.height/2,
	            whiskerMinExtent, w->core.height/2 );
         /* leading whisker */
         XDrawLine( XtDisplay(w), XtWindow(w), w->boxAndWhisker.whiskerGC,
		    boxMaxExtent, w->core.height/2, 
	            whiskerMaxExtent, w->core.height/2 );

     } else if ( w->boxAndWhisker.orientation == XtorientVertical ) {
         boxMinExtent =
             ((float)(w->boxAndWhisker.Q1value - w->boxAndWhisker.minValue) / 
	       (float)boxValueRange) * usableWidth + w->boxAndWhisker.padding;
	 boxMaxExtent =
             ((float)(w->boxAndWhisker.Q3value - w->boxAndWhisker.minValue) /
	       (float)boxValueRange) * usableWidth + w->boxAndWhisker.padding;
	 whiskerMinExtent = 
         ((float)(w->boxAndWhisker.smallestValue - w->boxAndWhisker.minValue) /
	       (float)boxValueRange ) * usableWidth + w->boxAndWhisker.padding;
	 whiskerMaxExtent = 
         ((float)(w->boxAndWhisker.largestValue - w->boxAndWhisker.minValue) /
	       (float)boxValueRange ) * usableWidth + w->boxAndWhisker.padding;
	 /* trailing whisker */
	 XDrawLine( XtDisplay(w), XtWindow(w), w->boxAndWhisker.whiskerGC,
		    w->core.width/2, boxMinExtent, 
	            w->core.width/2, whiskerMinExtent );
	 /* leading whisker */
	 XDrawLine( XtDisplay(w), XtWindow(w), w->boxAndWhisker.whiskerGC,
		    w->core.width/2, boxMaxExtent,
		    w->core.width/2, whiskerMaxExtent );
     }
}
     

/* BoxAndWhiskerSelect() processes buttonDown event */
static void BoxAndWhiskerSelect(w, event)
BoxAndWhiskerWidget    w;
XButtonPressedEvent    *event;
{
} /* BoxAndWhiskerSelect */


/* BoxAndWhiskerNotify() processes buttonUp event */
static void BoxAndWhiskerNotify(w, event)
BoxAndWhiskerWidget    w;
XButtonPressedEvent    *event;
{
     XtCallCallbacks( w, XtNselect, NULL );
}  /* BoxAndWhiskerNotify */



/* BoxAndWhiskerGetMinAndMax() simultaneously finds the min and max of
    the sample data */
static void BoxAndWhiskerGetMinAndMax( w )
BoxAndWhiskerWidget    w;
{
     int i;
     int valuecnt = w->boxAndWhisker.dataValueCnt;

     w->boxAndWhisker.smallestValue = w->boxAndWhisker.dataValues[0];
     w->boxAndWhisker.largestValue = w->boxAndWhisker.dataValues[0];
     w->boxAndWhisker.smallestIndex = 0;
     w->boxAndWhisker.largestIndex = 0;

     for (i=1; i<valuecnt; i++) {
        if (w->boxAndWhisker.dataValues[i] < w->boxAndWhisker.smallestValue ) {
           w->boxAndWhisker.smallestValue = w->boxAndWhisker.dataValues[i];
	   w->boxAndWhisker.smallestIndex = w->boxAndWhisker.dataIndices[i];
        } else if (w->boxAndWhisker.dataValues[i] >
		                              w->boxAndWhisker.largestValue ) {
           w->boxAndWhisker.largestValue = w->boxAndWhisker.dataValues[i];
	   w->boxAndWhisker.largestIndex = w->boxAndWhisker.dataIndices[i];
	}
     }
}


/* BoxAndWhiskerSelectElement() finds and returns the ith element in the
          ordering of a sample given an unordered array */
static int BoxAndWhiskerSelectElement( w, initialIndex, finalIndex,
				       initialPartitionIndex )
BoxAndWhiskerWidget w;
int                 initialIndex, finalIndex;
int                 initialPartitionIndex;
{
     int initialValue;
     int foundPartitionIndex, randomPartitionIndex;
     int foundPartitionDistance;
     int i,j;
     double ourRandomNumber;

     if ( initialIndex == finalIndex ) {
        w->boxAndWhisker.currentIndex =
	                        w->boxAndWhisker.dataIndices[initialIndex];
        return( w->boxAndWhisker.dataValues[initialIndex] );
     } 

     /* this should be a random number -- but it really doesn't matter */
     ourRandomNumber = (double)rand() / (double)INT_MAX;
     randomPartitionIndex = (int)(ourRandomNumber*(finalIndex-initialIndex+1))+
                                 initialIndex;
/*
     randomPartitionIndex = 2;
*/
     Exchange( &w->boxAndWhisker.dataValues[initialIndex],
	       &w->boxAndWhisker.dataValues[randomPartitionIndex] );
     Exchange( &w->boxAndWhisker.dataIndices[initialIndex],
	       &w->boxAndWhisker.dataIndices[randomPartitionIndex] );

     initialValue = w->boxAndWhisker.dataValues[initialIndex];
     i = initialIndex - 1;
     j = finalIndex + 1;
     while(1) {    
         i++;
         while( w->boxAndWhisker.dataValues[i] < initialValue ) {
              i++;
	 }
         j--;
	 while( w->boxAndWhisker.dataValues[j] > initialValue ) {
              j--;
	 }
	 if ( i < j ) {
            Exchange( &w->boxAndWhisker.dataValues[i], 
		      &w->boxAndWhisker.dataValues[j] );
	    Exchange( &w->boxAndWhisker.dataIndices[i], 
		      &w->boxAndWhisker.dataIndices[j] );
	 } else {
            foundPartitionIndex = j;
	    break;
	 }
     }
     foundPartitionDistance = foundPartitionIndex - initialIndex + 1;

     if ( initialPartitionIndex <= foundPartitionDistance ) {
        return( BoxAndWhiskerSelectElement( w,
					    initialIndex, foundPartitionIndex,
				            initialPartitionIndex ));
     } else {
        return( BoxAndWhiskerSelectElement( w,
					    foundPartitionIndex+1, finalIndex,
			      initialPartitionIndex-foundPartitionDistance ));
     }
} /* BoxAndWhiskerSelectElement */


     
/* BoxAndWhiskerGetPctile() finds and returns the element at the given
          percentile within the data */
static int BoxAndWhiskerGetPctile( w, percentile )
BoxAndWhiskerWidget w;
int                 percentile;
{
     int dataIndex = (float)w->boxAndWhisker.dataValueCnt * 
                     ((float)percentile/100.);
     int dataElement;

     return( BoxAndWhiskerSelectElement( w,
		      	                 0, w->boxAndWhisker.dataValueCnt-1, 
				         dataIndex ) );  
} /* BoxAndWhiskerGetPctile */


     
/* PUBLIC PROCEDURES ---------------------------------------------------- */

/* BoxAndWhiskerSetValues() sets the current point set for the B&W plot */
extern void BoxAndWhiskerSetValues( w, values, valuecnt )
BoxAndWhiskerWidget w;
int                 *values;
int                 valuecnt;
{
     int i;
     int currentMin, currentMax;

     if ( w->boxAndWhisker.dataValues == NULL ) {
        w->boxAndWhisker.dataValues = (int *)XtMalloc( valuecnt * sizeof(int));
	w->boxAndWhisker.dataIndices =(int *)XtMalloc(valuecnt * sizeof(int) );
	w->boxAndWhisker.data = (int **)XtMalloc( 2 * sizeof(int *) );
	w->boxAndWhisker.data[1] = NULL;
     } else if ( valuecnt != w->boxAndWhisker.dataValueCnt ) {
        w->boxAndWhisker.dataValues = (int *)XtRealloc( 
			   w->boxAndWhisker.dataValues, valuecnt*sizeof(int) );
	w->boxAndWhisker.dataIndices = (int *)XtRealloc(
		          w->boxAndWhisker.dataIndices, valuecnt*sizeof(int) );
     }
     w->boxAndWhisker.dataValueCnt = valuecnt;
     w->boxAndWhisker.data[0] = w->boxAndWhisker.dataValues;

     for (i=0; i<valuecnt; i++) {
         w->boxAndWhisker.dataValues[i] = values[i];
	 w->boxAndWhisker.dataIndices[i] = i;
     }

     /* find min and max */ 
     BoxAndWhiskerGetMinAndMax( w );

     /* find median and the quartiles */
     w->boxAndWhisker.median = BoxAndWhiskerGetPctile( w,50 );
     w->boxAndWhisker.medianIndex = w->boxAndWhisker.currentIndex;
     w->boxAndWhisker.Q1value = BoxAndWhiskerGetPctile( w,25 );
     w->boxAndWhisker.Q1index = w->boxAndWhisker.currentIndex;
     w->boxAndWhisker.Q3value = BoxAndWhiskerGetPctile( w,75 );
     w->boxAndWhisker.Q3index = w->boxAndWhisker.currentIndex;

     /* draw */
     XClearWindow( XtDisplay(w), XtWindow(w) );
     DrawBox(w);
     DrawWhiskers(w);
} /* BoxAndWhiskerSetValues */


/* BoxAndWhiskerSetValue() adda one new datapoint to the current set */
extern void BoxAndWhiskerSetValue( w, value )
BoxAndWhiskerWidget w;
int                 value;
{
     int newPointIndex;
     int currentMin, currentMax;

     if ( w->boxAndWhisker.dataValues == NULL ) {
        w->boxAndWhisker.dataValues = (int *)XtMalloc( 
					   REALLOC_INTERVAL *sizeof(int));
	w->boxAndWhisker.data = (int **)XtMalloc( 2 * sizeof(int *) );
	w->boxAndWhisker.data[1] = NULL;
	w->boxAndWhisker.dataIndices = (int *)XtMalloc(
					   REALLOC_INTERVAL *sizeof(int));
	w->boxAndWhisker.dataValueCnt = 1;
	newPointIndex = 0;
     } else {
        if ( (w->boxAndWhisker.dataValueCnt % REALLOC_INTERVAL)== 0 ) {
            w->boxAndWhisker.dataValues = (int *)XtRealloc( 
	                                  (int *)w->boxAndWhisker.dataValues,
	     (w->boxAndWhisker.dataValueCnt+1+REALLOC_INTERVAL)*sizeof(int) );
	    w->boxAndWhisker.dataIndices = (int *)XtRealloc( 
	                                 (int *)w->boxAndWhisker.dataIndices,
	     (w->boxAndWhisker.dataValueCnt+1+REALLOC_INTERVAL)*sizeof(int) );
	}
        newPointIndex = w->boxAndWhisker.dataValueCnt;
        w->boxAndWhisker.dataValueCnt += 1;
     }
     w->boxAndWhisker.dataValues[newPointIndex] = value;
     w->boxAndWhisker.dataIndices[newPointIndex] = newPointIndex;
     w->boxAndWhisker.data[0] = w->boxAndWhisker.dataValues;

     /* find min and max */ 
     BoxAndWhiskerGetMinAndMax( w );

     /* find median and the quartiles */
     w->boxAndWhisker.median = BoxAndWhiskerGetPctile( w,50 );
     w->boxAndWhisker.medianIndex = w->boxAndWhisker.currentIndex;
     w->boxAndWhisker.Q1value = BoxAndWhiskerGetPctile( w,25 );
     w->boxAndWhisker.Q1index = w->boxAndWhisker.currentIndex;
     w->boxAndWhisker.Q3value = BoxAndWhiskerGetPctile( w,75 );
     w->boxAndWhisker.Q3index = w->boxAndWhisker.currentIndex;

     /* draw */
     XClearWindow( XtDisplay(w), XtWindow(w) );
     DrawBox(w);
     DrawWhiskers(w);
} /* BoxAndWhiskerSetValue */









