/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 * Author: Ruth A. Aydt (aydt@cs.uicu.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * BargraphFormWrapper.cc:  Class wrapper for the Bargraph performance widgets
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/Widgets/Wrappers/RCS/BargraphFormWrapper.C,v 1.21 1994/02/25 04:44:53 aydt Exp $
 */

#include "BargraphFormWrapper.h"

extern "C" {       
#include "Bargraph.h"
}

#include <Xm/Form.h>	// Form widget class name definition
#include "FUParams.h"
#include "ParamDisplay.h"
#include "XArgs.h"	// defines the X widget argument structure type

#define ADJ_BAR_WIDTH_ARY 0.125
#define ADJ_BAR_HEIGHT_ARY 0.125

#define max(a,b)        ( (a) > (b) ? (a):(b) )

BargraphFormWrapper::BargraphFormWrapper( WidgetBase *par, 
		                          BarGraphFU *bargraphFUPtr, 
					  const XArgs& args, 
					  const CString& name ) 
		   : PerfWidget( par )
{
	_setClassName( MY_CLASS );
   	_setType( "BargraphFormWrapper" );
   	barGraphFUthisPtr = bargraphFUPtr;

   	// Create the Motif Form widget
   	widget = XtCreateManagedWidget( name.getValue(), xmFormWidgetClass, 
				        par->getWidget(), args.getArgs(), 
				        args.getNArgs() );
   	Arg arg;		
   	XtSetArg( arg, XmNfractionBase, (XtArgVal)10000 );
   	XtSetValues( widget, &arg, 1 );

	// Set initial number of columns and rows
   	formCols = 1;  
   	formRows = 1;

   	// Create a single Bargraph widget to be the initial child of the Form
   	childList = new Widget[1];
   	childList[0] = XtCreateManagedWidget( "Bar:0,0", bargraphWidgetClass, 
					      widget, NULL, 0 );
	// Create Min/Max Label & other labels
   	labels.createMinMaxLabel( widget, "BargraphMinMaxLabel" );
	setBargraphLabel( L_LEFT_VERT, NULLCHARSTRING );
	setBargraphLabel( L_BOTTOM_HORIZ, NULLCHARSTRING );

	if ( barGraphFUthisPtr != NULL ) {
	    addCallback( childList[0], XtNselect, 
			 &Callback::callback1, this, NULL );
	} else {
	    addCallback( childList[0], XtNselect, 
			 &Callback::callback2, this, NULL );
	}
}                       

BargraphFormWrapper::~BargraphFormWrapper()
{
	XtDestroyWidget( widget );
	delete [] childList;
}

void 
BargraphFormWrapper::_adjustLayout( int nCols, int nRows )
{
       	// Unmanage all the child widgets before resize
       	XtUnmanageChildren( childList, (formRows * formCols) );

        // The plan here is to determine the current height of "any" bar
        // (we use the first bar as our canonical bar), then set the new
        // width and height (again for a single bar).

        Dimension barWidth, barHeight;
        Dimension oldSingleBarWidth, oldSingleBarHeight;
        XtVaGetValues( childList[0],
                       XmNwidth, &oldSingleBarWidth,
                       XmNheight, &oldSingleBarHeight,
                          NULL );
        barWidth = (oldSingleBarWidth*formCols) / nCols;
        barHeight = (oldSingleBarHeight*formRows) / nRows;

        _genChildLayout( nCols, nRows );

        setBargraphAttr(XmNwidth, barWidth );
        setBargraphAttr(XmNheight, barHeight );

        // Make visible all the new repositioned stuff 
	XtManageChildren( childList, (nRows * nCols) );
        XFlush( XtDisplay( widget ) );
}        

Widget 
BargraphFormWrapper::_createChild( char *newWidgetName, int iCol, int iRow,  
                         int barLayoutXOrg, int barLayoutYOrg,
                         int barWidthProportion, int barHeightProportion )
{
   	int i = 0;
   	Arg arglist[4];

   	// Attach the widget to the Form for RESIZING
   	XtArgVal leftPos = barLayoutXOrg + iCol * barWidthProportion;
   	XtArgVal rightPos = leftPos + barWidthProportion;
   	XtArgVal topPos =  barLayoutYOrg + iRow * barHeightProportion;
   	XtArgVal bottomPos = topPos +  barHeightProportion;
   	XtSetArg( arglist[i], XmNleftPosition, leftPos ); i++;
   	XtSetArg( arglist[i], XmNrightPosition, rightPos ); i++;
   	XtSetArg( arglist[i], XmNtopPosition, topPos ); i++;
   	XtSetArg( arglist[i], XmNbottomPosition, bottomPos ); i++;

   	Widget newBar = XtCreateWidget( newWidgetName, bargraphWidgetClass, 
					widget, arglist, i );
						
	if ( barGraphFUthisPtr != NULL ) {
   	    addCallback( newBar, XtNselect, &Callback::callback1, this, NULL );
	} else {
   	    addCallback( newBar, XtNselect, &Callback::callback2, this, NULL );
	}
   	return( newBar );
}

void 
BargraphFormWrapper::_findBarsFormPosition( int nColElements, int nRowElements, 
			    int *barLayoutXOrg, int *barLayoutYOrg,
		            int *barWidthProportion, int *barHeightProportion )
{

  	int iFractionBase, lLeft, widthP, lBottom, lTop, heightP;
  	getBargraphFormAttr( XmNfractionBase, (XtArgVal) &iFractionBase );

  	if ( labels.isALabel( L_LEFT_VERT ) ) {
      	    lLeft = labels.getLabelHeight( L_LEFT_VERT );
      	    widthP = (int)( ( iFractionBase - (2 * lLeft) ) / nColElements );
  	} else if ( ( nRowElements > 1 ) && ( formRows == 1 ) ) {
            // If resizing to an array, the labelling layout is fixed 
      	    lLeft = labels.getLabelHeight( L_LEFT_VERT );
            widthP = (int)( ( iFractionBase - (2 * lLeft) ) / nColElements );
   	} else {
            lLeft = 0;
      	    widthP = (int)( iFractionBase/nColElements );
	}
	*barLayoutXOrg = lLeft;
	*barWidthProportion = widthP;

  	if ( labels.isALabel( L_BOTTOM_HORIZ ) ) {
	    lBottom = labels.getLabelHeight( L_BOTTOM_HORIZ );
	    lTop = labels.getLabelHeight( L_TOP_HORIZ );
	    heightP = (int)( (iFractionBase - lBottom - lTop) / nRowElements );
  	} else if ( ( nRowElements > 1 ) && ( formRows == 1 ) ) {
	    lBottom = labels.getLabelHeight( L_BOTTOM_HORIZ );
	    lTop = labels.getLabelHeight( L_TOP_HORIZ );
	    heightP = (int)( (iFractionBase - lBottom - lTop) / nRowElements );
  	} else {
	    lTop = labels.getLabelHeight( L_TOP_HORIZ );
	    heightP = (int)( (iFractionBase - lTop) / nRowElements );
  	} 
  	*barLayoutYOrg = labels.getLabelHeight( L_TOP_HORIZ );
	*barHeightProportion = heightP;
}

void 
BargraphFormWrapper::_genChildLayout( int nNewCols, int nNewRows )
{
  	// Calculate the new size and create a new WidgetList to hold children
  	int newSize = nNewRows * nNewCols;
	WidgetList newChildList = new Widget[ newSize ];

	// Figure out where bars should be in new display
  	int barLayoutXOrg, barLayoutYOrg;
	int barWidthProportion, barHeightProportion;
  	_findBarsFormPosition( nNewCols, nNewRows, 
		               &barLayoutXOrg, &barLayoutYOrg, 
		               &barWidthProportion, &barHeightProportion ); 

	// Now, for every element in the new and/or old display...
	int maxRow = max( formRows, nNewRows );
	int maxCol = max( formCols, nNewCols );

  	int iRow, iCol;
  	int iEleNew, iEleOld;
  	char newWidgetName[MAX_WIDGET_NAME_LEN];

    	for ( iRow = 0; iRow < maxRow; iRow++ ) {
      	    for ( iCol = 0; iCol< maxCol; iCol++) { 

		// Find new and old positons ( Note: both are not always valid )
       		iEleNew = iRow * nNewCols + iCol;	
       		iEleOld = iRow * formCols + iCol;

        	if ( (iRow < nNewRows) && (iCol < nNewCols) ) {
		    //  This (iRow, iCol) is in New Display
           	    if ( (iRow < formRows) && (iCol < formCols)){
           	        // Also in Old Display - reuse the Widget.
               		_setABarFormPosition( iCol, iRow,
                                 barLayoutXOrg, barLayoutYOrg,
				 barWidthProportion, barHeightProportion );
			newChildList[ iEleNew ] = childList[ iEleOld ];
           	    } else {
	       		// Not in Old Display - Create new Bargraph Widget
               		sprintf( newWidgetName, "Bar:%d,%d", iRow, iCol ); 
			newChildList [ iEleNew ] = _createChild( newWidgetName, 
						     iCol, iRow, barLayoutXOrg,
						     barLayoutYOrg, 
						     barWidthProportion, 
						     barHeightProportion );
           	    }
        	} else {  
		    // This (iRow, iCol) is not in New Display
		    if ( (iRow < formRows) && (iCol < formCols) ) {
		        // This (iRow, iCol) in Old Display; Destroy the Widget
   	   	        XtDestroyWidget( childList[ iEleOld ] );
		    }
        	}
      	    }
    	}

  	// The new list is now complete. Clear out the old space & adjust ptrs
  	delete [] childList; 
	childList = newChildList;
  	formRows = nNewRows;
  	formCols = nNewCols;
}

void 
BargraphFormWrapper::_setABarFormPosition( int iCol, int iRow, 
			      int barLayoutXOrg, int barLayoutYOrg,
                              int barWidthProportion, int barHeightProportion )
{
     	Arg arglist[4];
     	XtSetArg( arglist[0], XmNleftPosition,
	          (XtArgVal)( barLayoutXOrg + iCol*barWidthProportion ) ); 
     	XtSetArg( arglist[1], XmNrightPosition,
		 (XtArgVal)( barLayoutXOrg + (iCol+1)*barWidthProportion ) );
     	XtSetArg( arglist[2], XmNtopPosition, 
		  (XtArgVal)( barLayoutYOrg + iRow*barHeightProportion ) );
     	XtSetArg( arglist[3], XmNbottomPosition, 
	          (XtArgVal)( barLayoutYOrg + (iRow+1)*barHeightProportion ) );

     	int iEle = iRow * formCols + iCol;
     	XtSetValues( childList[iEle], arglist, 4 );
}

void 				/* virtual */
BargraphFormWrapper::callback1( Widget callbackWidget, XtPointer /* ptr1 */,
				XtPointer /* ptr2 */ )
{
	/* 
	 * This callback gets the row and column of the selected bargraph
	 * from the widget's name and calls the fuCallback method of the 
	 * BargraphFU with those values.
	 */
    	int iRow, iCol;
    	sscanf( XtName(callbackWidget), "Bar:%d,%d", &iRow, &iCol );
    	if ( ( iRow >=0 ) && ( iCol >=0) ) {
       	    barGraphFUthisPtr->fuCallback( iRow, iCol ); 
    	}
}

void 				/* virtual */
BargraphFormWrapper::callback2( Widget callbackWidget, XtPointer /* ptr1 */,
				XtPointer ptr2 )
{
	/*
	 * This callback displays a dialog giving the bargraph value and
	 * % of maximum for the selected bargraph.  If there are multiple
	 * bargraphs in the display, then the row and column are also shown.
	 * It is registered if the object that created the BargarphFormWrapper
	 * instance is not a BargraphFU.
	 */

    	int iRow, iCol;
    	sscanf( XtName(callbackWidget), "Bar:%d,%d", &iRow, &iCol );

	if ( ( iRow < 0 ) || ( iCol < 0 ) ) {
  	    return;
	}

        FUParams params;

        if ( formRows != 1 ) {
            params.addDisplayParam( "Row", iRow );
	}
        if ( formCols != 1 ) {
            params.addDisplayParam( "Column", iCol );
	}

	int value = (int)ptr2;
        params.addDisplayParam( "Displayed Value", value );

	int maxValue;
   	XtVaGetValues( callbackWidget, XtNmaxValue, (XtArgVal)&maxValue, NULL );
	double percent =  ( (double)value / (double)maxValue ) * 100.0;
        params.addDisplayParam( "% of Maximum", percent );

        ParamDisplay *pd = new ParamDisplay( Pablo::TopLevel(), params, 
					     "Bargraph Callback" );
}

void
BargraphFormWrapper::getBargraphAttr( String name, XtArgVal value ) const
{
   	Arg arg;
   	XtSetArg( arg, name, value );
   	XtGetValues( childList[0], &arg, 1 );
}

void 
BargraphFormWrapper::getBargraphFormAttr( String name, XtArgVal value ) const
{
  	Arg arg;
  	XtSetArg( arg, name, value );
  	XtGetValues( widget, &arg, 1 );
}

void 
BargraphFormWrapper::getBargraphLabel( int whichLabel, 
				       const char **aLabel ) const
{
	labels.getLabel( whichLabel, aLabel );
} 

void
BargraphFormWrapper::setBargraphAttr( String name, XtArgVal value )
{
	Arg arg;
	XtSetArg( arg, name, value );

	for ( int i = 0; i < (formCols * formRows); i++ ) {
	    XtSetValues( childList[i], &arg, 1 );
	}
}

void 
BargraphFormWrapper::setBargraphFormAttr( String name, XtArgVal value )
{
  	Arg arg;
  	XtSetArg( arg, name, value );
  	XtSetValues( widget, &arg, 1 );
}

void 
BargraphFormWrapper::setBargraphLabel( double fuMin, double fuMax )
{
     labels.setLabel( fuMin, fuMax );
}

void 
BargraphFormWrapper::setBargraphLabel( int whichLabel, const char *stringLabel )
{
     	Boolean_ layoutChange; 
     	if ( whichLabel == L_LEFT_VERT ) {
            layoutChange = labels.setLabel( widget, "BargraphLabelVert",
					    whichLabel, stringLabel );
     	} else if ( whichLabel == L_BOTTOM_HORIZ ) {
            layoutChange = labels.setLabel( widget, "BargraphLabelHoriz",
					    whichLabel, stringLabel );
     	}

     	if ( layoutChange && ( formRows == 1 ) ) {
            int barLayoutXOrg, barLayoutYOrg;
            int barWidthProportion, barHeightProportion;
            _findBarsFormPosition( formCols, formRows,
     		                   &barLayoutXOrg, &barLayoutYOrg,
			           &barWidthProportion, &barHeightProportion );
            for ( int iCol=0; iCol < formCols; iCol++ ) {
                for ( int iRow=0; iRow < formRows; iRow++ ) {
                    _setABarFormPosition( iCol, iRow, barLayoutXOrg, 
					  barLayoutYOrg, barWidthProportion, 
					  barHeightProportion );
	        }
            }
      	}
}

void
BargraphFormWrapper::setDisplayValues( int value ) 
{
	Assert( (formRows == 1) && (formCols == 1) );

   	BargraphSetValue( (BargraphWidget)childList[0], value );
}

void
BargraphFormWrapper::setDisplayValues( int nCols, int *values ) 
{
	if ( ( nCols != formCols ) || ( formRows != 1 ) ) {
	    _adjustLayout( nCols, 1 );
	}

	int i;
   	for ( i = 0; i < nCols; i++ ) {
     	    BargraphSetValue( (BargraphWidget)childList[i], values[i] );
   	}
}

void
BargraphFormWrapper::setDisplayValues( int nCols, int nRows, int *values ) 
{
	if ( ( nCols != formCols ) || ( nRows != formRows ) ) {
	    _adjustLayout( nCols, nRows );
	}

	int nBars = nCols * nRows;
	int i;
      	for ( i = 0; i < nBars; i++ ) {
            BargraphSetValue( (BargraphWidget)childList[i], values[i] );
        }
}

void
BargraphFormWrapper::setSingleValue( int iCol, int iRow, int value )
{
	if ( ( iCol >= formCols ) || ( iRow >= formRows ) ) {
	    warning( "setSingleValue: Index (%d, %d) out of range: (%d, %d)",
		      iCol, iRow, formCols-1, formRows-1 );
	    return;
	}

       	int iEle = ( iRow * formCols ) + iCol;
        BargraphSetValue( (BargraphWidget)childList[iEle], value );
}
	
void 
BargraphFormWrapper::updateDisplayLayout( int nCols, int nRows )
{
    	if ( ( nCols != formCols ) || ( nRows != formRows ) ) {
	    _adjustLayout( nCols, nRows );
	}
}        

/*
 *      Initialize the static data.   Only executed once.
 */
const char *const BargraphFormWrapper::MY_CLASS = "BargraphFormWrapper";
