/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Author: Robert Olson (olson@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Wrapper/RCS/InputPort.C,v 1.18 1994/02/25 04:35:56 aydt Exp $
 */

#include "InputPort.h"

#include "Assert.h"
#include "InputPipeSocket.h"

InputPort::InputPort( const CString& pname ) 
{
	_setClassName( MY_CLASS );

	name = pname;
	allowedTraits = NULL;
	allowedCnt = 0;
	constraintsCnt = 0;
	valueStatusFlag = INVALID_;
	socketNumber = -1;
	inputSocket = NULL;
}

InputPort::~InputPort()
{
	if ( allowedCnt != 0 ) { 
	    int i;
	    for ( i = 0; i < allowedCnt; i++ ) {
	    	delete allowedTraits[i];
	    }
	    delete[] allowedTraits;
	}
}

Boolean_ 
InputPort::_acceptsTraits( const DataTraits& traits ) const
{
	/* 
	 * If we haven't yet strictly defined the allowable traits by
	 * setting currentTraits through addBinding(), then just make 
	 * sure that traits is part of the acceptable set of traits.  
	 * If we have strictly defined traits already, then require an 
	 * exact match.
	 */

	if ( constraintsCnt == 0 ) {
	    Assert( currentTraits.isUndefined() ); 
	    DataTraits dt;

	    int i;
	    for ( i = 0; i < allowedCnt; i++ ) {
		dt = *allowedTraits[i];
		if ( ( dt == traits ) || 
		     ( ( dt.getType() == traits.getType() )  &&
		       ( dt.getDimension() == -1 ) ) ) {
		    return TRUE_;
		}
	    }
	    return FALSE_;
	} else {
	    return ( currentTraits == traits );
	}
}

void 
InputPort::addTraits( const DataTraits& traits )
{
	DataTraits* *dtp = allowedTraits;

	allowedTraits = new DataTraits* [ allowedCnt + 1 ];

	if ( allowedCnt != 0 ) {
	    int i;
	    for ( i = 0; i < allowedCnt; i++ ) {
	        allowedTraits[i] = dtp[i];
	    }
	    delete[] dtp;
	}
	DataTraits *dt = new DataTraits(traits);
	allowedTraits[ allowedCnt ] = dt;
	allowedCnt++;
}

void 
InputPort::addTraits( MachineDataType type, int dimension )
{
	addTraits( DataTraits( type, dimension ) );
}

void
InputPort::clearTraits()
{
	currentTraits.clear();
	constraintsCnt = 0;
	valueStatusFlag = INVALID_;
}

void 
InputPort::concatValidTraits( CString& cstr ) const
{
        if ( constraintsCnt > 0 ) {
            cstr = cstr + form( "  Type = %s;  Dimension = %d;\n", 
				   currentTraits.getTypeName(),
                                   currentTraits.getDimension() );
	    cstr = cstr + " *Restricted by previous binding to port.\n";
        } else {
            int i;
            DataTraits *dt;
            for ( i = 0; i < allowedCnt; i++ ) {
                dt = allowedTraits[i];;
                cstr = cstr + form("  Type = %s;   ", dt->getTypeName() );
                if ( ( dt->getDimension() == -1 )  ) {
                    cstr = cstr + "Dimension = ANY;\n";
                } else {
                    cstr = cstr + form( "Dimension = %d;\n", 
                                         dt->getDimension() );
                }
            }
        }
}

Boolean_
InputPort::constrainTraits( const DataTraits& traits ) 
{
	if ( _acceptsTraits( traits ) ) {
	    currentTraits = traits;
	    constraintsCnt++;
	    return TRUE_;
	} else {
	    return FALSE_;
	}
}

const RecordDossier& 
InputPort::getActiveRecord() const
{
	if ( ( valueStatusFlag == INVALID_ ) || ( inputSocket == NULL ) ) {
	    return ( RecordDossier::NODOSSIER );
	} else {
	    return( inputSocket->getActiveRecord() );
	}
}

int 
InputPort::getActiveFieldID() const
{
	if ( valueStatusFlag == INVALID_ ) {
	    return ( -1 );
	} else {
	    return( fieldNum );
	}
}

const Value& 
InputPort::getValue()
{
	if ( valueStatusFlag == INVALID_ ) {
	    error( "No value avaliable" );
	    return Value::NOVALUE;
	} else {
	    valueStatusFlag = USED_;
	    return *valueP;
	}
}

const Value *
InputPort::getValueP()
{
	if ( valueStatusFlag == INVALID_ ) {
	    error( "No value avaliable" );
	    return &Value::NOVALUE;
	} else {
	    valueStatusFlag = USED_;
	    return valueP;
	}
}

void
InputPort::lockTraits()
{
	Assert( currentTraits.isDefined() );
	constraintsCnt++;
}

void
InputPort::loosenTraits()
{
	/*
	 * If this is the last constraint that is enforcing the currentTraits 
	 * on the port, then clear currentTraits.  
	 */
	if ( --constraintsCnt == 0 ) {
	    currentTraits.clear();
	    valueStatusFlag = INVALID_;	    
	}
	Assert( constraintsCnt >= 0 );
}

void
InputPort::resetTraits( const DataTraits& traits, int count )
{
	currentTraits = traits;
	constraintsCnt = count;
}

void
InputPort::setSocketBinding( int socketIdx, InputPipeSocket *socket )
{
	// If we"ve changed our source socket then invalidate value 
	if ( ( socketNumber != socketIdx ) && ( inputSocket != socket ) ) {
	    valueStatusFlag = INVALID_;
	}

	socketNumber = socketIdx;
	inputSocket = socket;
}

void 
InputPort::setValueP( Value *valuePtr, int fieldID )
{
	Assert( (fieldID == -1) || (valueStatusFlag != NEW_) );
	valueStatusFlag = NEW_;
	valueP = valuePtr;
	fieldNum = fieldID;
}

void
InputPort::verifyStatus( )
{
	/*
	 *  If status is invalid, that's fine with us.
	 */
	if ( valueStatusFlag == INVALID_ ) {	
	    return;
	} 

	/* 
	 * If value is bound to user input, no problem - it will be updated
	 * when module runs so we can toss the 'old' value.
	 */
	if ( inputSocket == NULL ) {
	    valueStatusFlag = INVALID_;
	    return;
	}

	/*
	 * Check the action table in the input socket to see if this
	 * binding is still active. If it isn't change status to INVALID_;
	 */
	if ( ! inputSocket->bindingToInputActive( this, fieldNum ) ) {
	    valueStatusFlag = INVALID_;
	}

}

void 
InputPort::printOn( ostream& os ) const 
{
	os << form( "InputPort %s: status=%d \n", name.getValue(), 
						  valueStatusFlag );
	if ( valueStatusFlag ) {
	    valueP->printOn( os );
	}
}

/*
 * Initialize the static data 
 */
const char *const InputPort::MY_CLASS = "InputPort";

