/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Contributing Author: Brian K. Totty (totty@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * Value.C -  methods for the Value class.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Primitives/RCS/Value.C,v 1.23 1994/03/15 19:43:36 aydt Exp $
 */

#include "Assert.h"
#include "DataCharacteristics.h"
#include "GetPut.h"
#include "Value.h"

/*
 * Constructors and Destructors
 */
Value::Value()
{
}

Value::Value( MachineDataType _type, int _dimension )
	    : traits( _type, _dimension )
{
	if ( _dimension != 0 ) {		// new value is an array
	    val.a = new Array( _type, _dimension );
	} 
}

Value::Value( const DataTraits& _traits )
	    : traits( _traits )
{
	if ( traits.isArray() ) {
	    val.a = new Array( traits.getType(), traits.getDimension() );
	} 
}

Value::Value( const Value& value )
{
	traits = value.traits;
	if ( traits.isArray() ) {
	    val.a = new Array( value.val.a );
	} else {
	    val = value.val;
	}
}

Value::Value( char dataValue )
	    : traits( CHARACTER )
{
	val.c = dataValue;
}

Value::Value( int dataValue )
	    : traits( INTEGER )
{
	val.i = dataValue;
}

Value::Value( float dataValue )
	    : traits( FLOAT )
{
	val.f = dataValue;
}

Value::Value( double dataValue )
	    : traits( DOUBLE )
{
	val.d = dataValue;
}

Value::Value( Array *dataValue )
	    : traits( dataValue->getType(), dataValue->getDimension() )
{
	val.a = new Array( dataValue );
}

Value::~Value( )
{
	if ( traits.isArray() ) {
	    delete val.a;
	}
}

/*	
 *	Methods 	
 */

int
Value::bitsToObj( const char* bufPtr )
{
	/* 
	 * I return from lots of places here in an attempt to save some time.
	 * Also include UNDEFINED in switch to keep compiler happy.
	 */

	MachineDataType myType = traits.getType();

	if ( myType == UNDEFINED ) {
	    MSG_OBJ.error( "bitsToObj() undefined type." );
	    return -1;
	}

	if ( traits.isArray() ) {
	    Assert( traits.getType() == val.a->getType() );
	    Assert( traits.getDimension() == val.a->getDimension() );
	    return( val.a->bitsToObj( bufPtr ) );
	} else {
	    switch ( myType ) {
	  	case CHARACTER:
	       	    return( functionGet( bufPtr, &val.c ) );
	    	    //break;
	  	case INTEGER:
	    	    return( functionGet( bufPtr, &val.i ) );
	    	    //break;
	  	case FLOAT:
	    	    return( functionGet( bufPtr, &val.f ) );
	    	    //break;
	  	case DOUBLE:
	    	    return( functionGet( bufPtr, &val.d ) );
	    	    //break;
	        case UNDEFINED:
	            default:
	    	    return -1;
	    	    //break;
	    }
	}	

}

int 
Value::bytesNeeded() const
{
	if ( traits.isArray() ) {
	    return val.a->bytesNeeded();
	} else {
	    return traits.getAtomSize();
	}
}

int
Value::cbitsToObj( const char* bufPtr, const DataCharacteristics *cnvInfo )
{
	/* 
	 * I return from lots of places here in an attempt to save some time.
	 * Also include UNDEFINED in switch to keep compiler happy.
	 */

	MachineDataType myType = traits.getType();

	if ( myType == UNDEFINED ) {
	    MSG_OBJ.error( "cbitsToObj() undefined type." );
	    return -1;
	}

	if ( traits.isArray() ) {
	    Assert( traits.getType() == val.a->getType() );
	    Assert( traits.getDimension() == val.a->getDimension() );
	    return( val.a->cbitsToObj( bufPtr, cnvInfo ) );
	} else {
	    switch ( myType ) {
	  	case CHARACTER:
	       	    return( cnvInfo->functionGet( bufPtr, &val.c ) );
	    	    //break;
	  	case INTEGER:
	    	    return( cnvInfo->functionGet( bufPtr, &val.i ) );
	    	    //break;
	  	case FLOAT:
	    	    return( cnvInfo->functionGet( bufPtr, &val.f ) );
	    	    //break;
	  	case DOUBLE:
	    	    return( cnvInfo->functionGet( bufPtr, &val.d ) );
	    	    //break;
	        case UNDEFINED:
	            default:
	    	    return -1;
	    	    //break;
	    }
	}	

}

int 
Value::cbytesNeeded( const DataCharacteristics *cnvInfo ) const
{
	if ( traits.isArray() ) {
	    return val.a->cbytesNeeded( cnvInfo );
	} else {
	    return cnvInfo->getAtomSize( traits.getType() );
	}
}

Value 
Value::getValueAs( MachineDataType newType ) const 
{
	_newValue.traits.setTraits( newType, 0 );

	switch (newType) {
	    case CHARACTER:
	        _newValue.val.c = this->operator char();
	        break;

	    case INTEGER:
	        _newValue.val.i = this->operator int();
	        break;

	    case FLOAT:
	        _newValue.val.f = this->operator float();
	        break;

	    case DOUBLE:
	        _newValue.val.d = this->operator double();
	        break;

	    case UNDEFINED:
	        break;
	}
	return _newValue;
}

void
Value::getValueAs( MachineDataType newType, Value& newValue ) const 
{
      newValue.traits.setTraits( newType, 0 );

      switch (newType) {
          case CHARACTER:
              newValue.val.c = this->operator char();
              break;

          case INTEGER:
              newValue.val.i = this->operator int();
              break;

          case FLOAT:
              newValue.val.f = this->operator float();
              break;

          case DOUBLE:
              newValue.val.d = this->operator double();
              break;

          case UNDEFINED:
              break;
      }
}

int
Value::objToBits( char *const bufPtr, int bufLen ) const
{
	/* Array - Array class method does the work */
	if ( traits.isArray() ) {
	    return ( val.a->objToBits( bufPtr, bufLen ) );
	} 

	/* Is buffer large enough to hold value? If not, return -1*/
	if (  traits.getAtomSize()  > bufLen ) {
	    return -1 ; 
	}

	/* Normal case - do appropriate casts. */
	int bytesUsed = 0;
	switch ( traits.getType() ) {
	    case CHARACTER:
	        bytesUsed = functionPut( bufPtr, &val.c );
	        break;
	    case INTEGER:
	        bytesUsed = functionPut( bufPtr, &val.i );
	        break;
	    case FLOAT:
	        bytesUsed = functionPut( bufPtr, &val.f );
	        break;
	    case DOUBLE:
	        bytesUsed = functionPut( bufPtr, &val.d );
	        break;
	    case UNDEFINED:
	        bytesUsed = 0;
	        break;
	}	
	return bytesUsed ;
}

int
Value::objToCbits( char *const bufPtr, int bufLen,
		   		 const DataCharacteristics *cnvInfo ) const
{
	/* Array - Array class method does the work */
	if ( traits.isArray() ) {
	    return ( val.a->objToCbits( bufPtr, bufLen, cnvInfo ) );
	} 

	/* Is buffer large enough to hold value? If not, return -1*/
	if (  cnvInfo->getAtomSize( traits.getType() )  > bufLen ) {
	    return -1 ; 
	}

	/* Normal case - do appropriate casts. */
	int bytesUsed = 0;
	switch ( traits.getType() ) {
	    case CHARACTER:
	        bytesUsed = cnvInfo->functionPut( bufPtr, &val.c );
	        break;
	    case INTEGER:
	        bytesUsed = cnvInfo->functionPut( bufPtr, &val.i );
	        break;
	    case FLOAT:
	        bytesUsed = cnvInfo->functionPut( bufPtr, &val.f );
	        break;
	    case DOUBLE:
	        bytesUsed = cnvInfo->functionPut( bufPtr, &val.d );
	        break;
	    case UNDEFINED:
	        bytesUsed = 0;
	        break;
	}	
	return bytesUsed ;
}


Boolean_ 
Value::setQuantity( const Value& newValue )
{
	/* 
	 * First, we insist that the dimension of the current and new
	 * Values are the same.  If not, return failure.
	 */
	DataTraits newTraits = newValue.getTraits();

	if ( traits.getDimension() != newTraits.getDimension() ) {
	    return FAILURE_;
	}

	/* 
	 * Next, we insist that the type of the newValue can be cast into
	 * the type of the old value without losing precision.  If they
	 * can"t, return failure.
	 */
	MachineDataType type = traits.getType();
	MachineDataType ntype = newTraits.getType();

	if ( castTypeTable[type][ntype] == UNDEFINED ) {
	    return FAILURE_;
	} 
	
	/* 
	 * If we've gotten this far, then things will succeed!
	 * Handle Scalars and Arrays separately.
	 */
	if ( traits.isScalar() ) {
	    /* 
	     * Our result type will be the same as the original type
	     */
	    switch ( type ) {
	        case CHARACTER:
	            val.c =  newValue.operator char();
	            break;
	        case INTEGER:
	            val.i = newValue.operator int();
	            break;
	        case FLOAT:
	            val.f = newValue.operator float();
	            break;
	        case DOUBLE:
		    val.d = newValue.operator double();
	            break;
	        case UNDEFINED:
	    	break;
	    }
	} else {
	    /*
	     * We will not change the type or dimension of the Array
	     * we are pointing to, however the dimension sizes may be
	     * different so we adjust those to match the new values.
	     */
	    Array *newValueArray = (Array *)newValue;	   

	    val.a->setDimSizes( newValueArray->getDimSizes() );  
	    val.a->setCellValues( newValueArray );
	}
	return SUCCESS_;

}

/*
 * Assignment operator overloading
 */

Value& 
Value::operator=( const Value& value )
{
	if ( traits.isScalar() && value.traits.isScalar() ) {	      // S->S
	    traits = value.traits;
	    val = value.val;
	} else if ( traits.isArray() && value.traits.isScalar() ) {   // A->S
	    delete val.a;
	    traits = value.traits;
	    val = value.val;
	} else if ( traits.isArray() && value.traits.isArray() ) {    // A->A
	    traits = value.traits;
	    *val.a = *value.val.a;
	} else {						      // S->A
	    traits = value.traits;
	    val.a = new Array( value.val.a );
	}
	    
	return *this;
}

Value& 
Value::operator=( char c )
{
	if ( traits.isArray() ) delete( val.a );

	traits.setTraits( CHARACTER, 0 );
	val.c = c;
	return *this;
}

Value& 
Value::operator=( int i )
{
	if ( traits.isArray() ) delete( val.a );

	traits.setTraits( INTEGER, 0 );
	val.i = i;
	return *this;
}

Value& 
Value::operator=( float f )
{
	if ( traits.isArray() ) delete( val.a );

	traits.setTraits( FLOAT, 0 );
	val.f = f;
	return *this;
}

Value& 
Value::operator=( double d )
{
	if ( traits.isArray() ) delete( val.a );

	traits.setTraits( DOUBLE, 0 );
	val.d = d;
	return *this;
}

Value& 
Value::operator=( const Array * array )
{
	if ( traits.isArray() ) {    			// A->A
	    traits.setTraits( array->getType(), array->getDimension() );
	    *val.a = *array;
	} else {					// S->A
	    traits.setTraits( array->getType(), array->getDimension() );
	    val.a = new Array( array );
	}

	return *this;
}

/*
 * Conversion operators
 */

Value::operator char() const
{
	if ( traits.isArray() ) return 0;

	switch ( traits.getType() ) {
	    case CHARACTER:
	        return val.c;
	        //break;
	    case INTEGER:
	        return (char) val.i;
	        //break;
	    case FLOAT:
	        return (char) val.f;
	        //break;
	    case DOUBLE:
	        return (char) val.d;
	        //break;
	    case UNDEFINED:
	        return (char) 0;
	        //break;
	}
	return (char) 0;		// to keep g++ happy; shouldn't get here
}

Value::operator int() const
{
	if ( traits.isArray() ) return 0;

	switch ( traits.getType() ) {
	    case CHARACTER:
	        return (int) val.c;
	        //break;
	    case INTEGER:
	        return val.i;
	        //break;
	    case FLOAT:
	        return (int) val.f;
	        //break;
	    case DOUBLE:
	        return (int) val.d;
	        //break;
	    case UNDEFINED:
	        return 0;
	        //break;
	}
	return 0;			// to keep g++ happy; shouldn't get here
}

Value::operator float() const
{
	if ( traits.isArray() ) return 0;

	switch ( traits.getType() ) {
	    case CHARACTER:
	        return (float) val.c;
	        //break;
	    case INTEGER:
	        return (float) val.i;
	        //break;
	    case FLOAT:
	        return val.f;
	        //break;
	    case DOUBLE:
	        return (float) val.d;
	        //break;
	    case UNDEFINED:
	        return 0;
	        //break;
	}
	return 0.0;			// to keep g++ happy; shouldn't get here
}

Value::operator double() const
{
	if ( traits.isArray() ) return 0;

	switch ( traits.getType() ) {
	    case CHARACTER:
	        return (double) val.c;
	        //break;
	    case INTEGER:
	        return (double) val.i;
	        //break;
	    case FLOAT:
	        return (double) val.f;
	        //break;
	    case DOUBLE:
	        return val.d;
	        //break;
	    case UNDEFINED:
	        return 0;
	        //break;
	}
	return 0.0;			// to keep g++ happy; shouldn't get here
}


Value::operator Array*() const
{
	if ( traits.isArray() ) {
	    return val.a;
	} else {
	    return 0;
	}

}

/*
 * Relational and Equality operators
 */

Boolean_
Value::operator< ( const Value& v )
{
	/*	
	 * Currently, this operation is not supported for arrays
	 */
	if ( traits.isArray() || v.traits.isArray() ) {
	    MSG_OBJ.error( "operator< unsupported for Arrays." );
	    return ( FALSE_ );
	}

	MachineDataType type = traits.getType();
	MachineDataType vtype = v.traits.getType();
	MachineDataType resultType = resultTypeTable[type][vtype];

	if ( resultType == UNDEFINED ) {
            /*
             * At least one of the input values was of type 
             * UNDEFINED which is currently unsupported.
             */
            MSG_OBJ.error( "erator< unsupported for UNDEFINED type" );
            return ( FALSE_ );

        } else if ( ( resultType == type ) && ( resultType == vtype ) ) {
	    /* 
	     * Simple case where both input Values have the same type.
	     */
	    switch ( resultType ) {
	        case CHARACTER:
	    	    return CnvToBoolean_( val.c < v.val.c ) ;
	    	    //break;

	        case INTEGER:
	    	    return CnvToBoolean_( val.i < v.val.i );
	    	    //break;

	        case FLOAT:
	    	    return CnvToBoolean_( val.f < v.val.f );
	    	    //break;

	        case DOUBLE:
	    	    return CnvToBoolean_( val.d < v.val.d );
	    	    //break;

	        case UNDEFINED:		// shouldn"t get here
	    	    return ( FALSE_ );		
	    	    //break;
	    }
	} else {
	    /* 
	     * We have at least one input type that is different than
	     * the result type.  Perform coercion into Values of desired
	     * type and then do the division operation on those.
	     */
	    return ( getValueAs(resultType) < v.getValueAs(resultType) ); 

	}
	return FALSE_;		// Shouldn"t get here!
}

Boolean_
Value::operator> ( const Value& v )
{
	/*	
	 * Currently, this operation is not supported for arrays
	 */
	if ( traits.isArray() || v.traits.isArray() ) {
	    MSG_OBJ.error( "operator> unsupported for Arrays." );
	    return ( FALSE_ );
	}

	MachineDataType type = traits.getType();
	MachineDataType vtype = v.traits.getType();
	MachineDataType resultType = resultTypeTable[type][vtype];

	if ( resultType == UNDEFINED ) {
	    /*
	     * At least one of the input values was of type 
	     * UNDEFINED which is currently unsupported.
	     */
            MSG_OBJ.error( "operator> unsupported for UNDEFINED type" );
	    return ( FALSE_ );

	} else if ( ( resultType == type ) && ( resultType == vtype ) ) {
	    /* 
	     * Simple case where both input Values have the same type.
	     */
	    switch ( resultType ) {
	        case CHARACTER:
	    	    return CnvToBoolean_( val.c > v.val.c ) ;
	    	    //break;

	        case INTEGER:
	    	    return CnvToBoolean_( val.i > v.val.i );
	    	    //break;

	        case FLOAT:
	    	    return CnvToBoolean_( val.f > v.val.f );
	    	    //break;

	        case DOUBLE:
	    	    return CnvToBoolean_( val.d > v.val.d );
	    	    //break;

	        case UNDEFINED:		// shouldn"t get here
	            return ( FALSE_ );
	    	    //break;
	    }
	} else {
	    /* 
	     * We have at least one input type that is different than
	     * the result type.  Perform coercion into Values of desired
	     * type and then do the division operation on those.
	     */
	    return ( getValueAs(resultType) > v.getValueAs(resultType) ); 

	}
	return ( FALSE_ );		// Shouldn"t get here!
}

Boolean_
Value::operator<= ( const Value& v )
{
	/*	
	 * Currently, this operation is not supported for arrays
	 */
	if ( traits.isArray() || v.traits.isArray() ) {
	    MSG_OBJ.error( "operator<= unsupported for Arrays." );
	    return ( FALSE_ );
	}

	MachineDataType type = traits.getType();
	MachineDataType vtype = v.traits.getType();
	MachineDataType resultType = resultTypeTable[type][vtype];

	if ( resultType == UNDEFINED ) {
	    /*
	     * At least one of the input values was of type 
	     * UNDEFINED which is currently unsupported.
	     */
            MSG_OBJ.error( "operator<= unsupported for UNDEFINED type" );
	    return ( FALSE_ );

	} else if ( ( resultType == type ) && ( resultType == vtype ) ) {
	    /* 
	     * Simple case where both input Values have the same type.
	     */
	    switch ( resultType ) {
	        case CHARACTER:
	    	    return CnvToBoolean_( val.c <= v.val.c ) ;
	    	    //break;

	        case INTEGER:
	    	    return CnvToBoolean_( val.i <= v.val.i );
	    	    //break;

	        case FLOAT:
	    	    return CnvToBoolean_( val.f <= v.val.f );
	    	    //break;

	        case DOUBLE:
	    	    return CnvToBoolean_( val.d <= v.val.d );
	    	    //break;

	        case UNDEFINED:		// shouldn"t get here
	            return ( FALSE_ );
	    	    //break;
	    }
	} else {
	    /* 
	     * We have at least one input type that is different than
	     * the result type.  Perform coercion into Values of desired
	     * type and then do the division operation on those.
	     */
	    return ( getValueAs(resultType) <= v.getValueAs(resultType) ); 

	}
	return FALSE_;		// Shouldn"t get here!
}

Boolean_
Value::operator>= ( const Value& v )
{
	/*	
	 * Currently, this operation is not supported for arrays
	 */
	if ( traits.isArray() || v.traits.isArray() ) {
	    MSG_OBJ.error( "operator>= unsupported for Arrays." );
	    return ( FALSE_ );
	}

	MachineDataType type = traits.getType();
	MachineDataType vtype = v.traits.getType();
	MachineDataType resultType = resultTypeTable[type][vtype];

	if ( resultType == UNDEFINED ) {
	    /*
	     * At least one of the input values was of type 
	     * UNDEFINED which is currently unsupported.
	     */
            MSG_OBJ.error( "operator>= unsupported for UNDEFINED type" );
	    return ( FALSE_ );

	} else if ( ( resultType == type ) && ( resultType == vtype ) ) {
	    /* 
	     * Simple case where both input Values have the same type.
	     */
	    switch ( resultType ) {
	        case CHARACTER:
	    	    return CnvToBoolean_( val.c >= v.val.c ) ;
	    	    //break;

	        case INTEGER:
	    	    return CnvToBoolean_( val.i >= v.val.i );
	    	    //break;

	        case FLOAT:
	    	    return CnvToBoolean_( val.f >= v.val.f );
	    	    //break;

	        case DOUBLE:
	    	    return CnvToBoolean_( val.d >= v.val.d );
	    	    //break;

	        case UNDEFINED:		// shouldn"t get here
	            return ( FALSE_ );
	    	    //break;
	    }
	} else {
	    /* 
	     * We have at least one input type that is different than
	     * the result type.  Perform coercion into Values of desired
	     * type and then do the division operation on those.
	     */
	    return ( getValueAs(resultType) >= v.getValueAs(resultType) ); 

	}
	return FALSE_;		// Shouldn"t get here!
}

Boolean_
Value::operator== ( const Value& v )
{
	/*	
	 * Currently, this operation is not supported for arrays
	 */
	if ( traits.isArray() || v.traits.isArray() ) {
	    MSG_OBJ.error( "operator== unsupported for Arrays." );
	    return ( FALSE_ );
	}

	MachineDataType type = traits.getType();
	MachineDataType vtype = v.traits.getType();
	MachineDataType resultType = resultTypeTable[type][vtype];

	if ( resultType == UNDEFINED ) {
	    /*
	     * At least one of the input values was of type 
	     * UNDEFINED which is currently unsupported.
	     */
            MSG_OBJ.error( "operator== unsupported for UNDEFINED type" );
	    return ( FALSE_ );

	} else if ( ( resultType == type ) && ( resultType == vtype ) ) {
	    /* 
	     * Simple case where both input Values have the same type.
	     */
	    switch ( resultType ) {
	        case CHARACTER:
	    	    return CnvToBoolean_( val.c == v.val.c ) ;
	    	    //break;

	        case INTEGER:
	    	    return CnvToBoolean_( val.i == v.val.i );
	    	    //break;

	        case FLOAT:
	    	    return CnvToBoolean_( val.f == v.val.f );
	    	    //break;

	        case DOUBLE:
	    	    return CnvToBoolean_( val.d == v.val.d );
	    	    //break;

	        case UNDEFINED:		// shouldn"t get here
	            return ( FALSE_ );
	    	    //break;
	    }
	} else {
	    /* 
	     * We have at least one input type that is different than
	     * the result type.  Perform coercion into Values of desired
	     * type and then do the division operation on those.
	     */
	    return ( getValueAs(resultType) == v.getValueAs(resultType) ); 

	}
	return FALSE_;		// Shouldn"t get here!
}

Boolean_
Value::operator!= ( const Value& v )
{
	/*	
	 * Currently, this operation is not supported for arrays
	 */
	if ( traits.isArray() || v.traits.isArray() ) {
	    MSG_OBJ.error( "operator!= unsupported for Arrays." );
	    return ( FALSE_ );
	}

	MachineDataType type = traits.getType();
	MachineDataType vtype = v.traits.getType();
	MachineDataType resultType = resultTypeTable[type][vtype];

	if ( resultType == UNDEFINED ) {
	    /*
	     * At least one of the input values was of type 
	     * UNDEFINED which is currently unsupported.
	     */
            MSG_OBJ.error( "operator!= unsupported for UNDEFINED type" );
	    return ( FALSE_ );

	} else if ( ( resultType == type ) && ( resultType == vtype ) ) {
	    /* 
	     * Simple case where both input Values have the same type.
	     */
	    switch ( resultType ) {
	        case CHARACTER:
	    	    return CnvToBoolean_( val.c != v.val.c ) ;
	    	    //break;

	        case INTEGER:
	    	    return CnvToBoolean_( val.i != v.val.i );
	    	    //break;

	        case FLOAT:
	    	    return CnvToBoolean_( val.f != v.val.f );
	    	    //break;

	        case DOUBLE:
	    	    return CnvToBoolean_( val.d != v.val.d );
	    	    //break;

	        case UNDEFINED:		// shouldn"t get here
	            return ( FALSE_ );
	    	    //break;
	    }
	} else {
	    /* 
	     * We have at least one input type that is different than
	     * the result type.  Perform coercion into Values of desired
	     * type and then do the division operation on those.
	     */
	    return ( getValueAs(resultType) != v.getValueAs(resultType) ); 

	}
	return FALSE_;		// Shouldn"t get here!
}

/*
 * Unary and Binary Arithmetic operators
 */

Value 
Value::operator- ()
{
	/*	
	 * Currently, this operation is not supported for arrays
	 * or UNDEFINED types
	 */
	if ( traits.isArray() )
	    return ( Value::NOVALUE );

	MachineDataType type = traits.getType();
	if ( type == UNDEFINED ) {
	    return ( Value::NOVALUE );
	}

	_newValue.traits.setTraits( type, 0 );

	switch ( type ) {
	    case CHARACTER:
	        _newValue.val.c = -val.c;
	        break;

	    case INTEGER:
	        _newValue.val.i = -val.i;
	        break;

	    case FLOAT:
	        _newValue.val.f = -val.f;
	        break;

	    case DOUBLE:
	        _newValue.val.d = -val.d;
	        break;

	    case UNDEFINED:
	        break;
	}
	return _newValue;
}

Value 
Value::operator+ ( const Value& v )
{
	/*	
	 * Currently, this operation is not supported for arrays
	 */
	if ( traits.isArray() || v.traits.isArray() )
	    return ( Value::NOVALUE );

	MachineDataType type = traits.getType();
	MachineDataType vtype = v.traits.getType();
	MachineDataType resultType = resultTypeTable[type][vtype];

	if ( resultType == UNDEFINED ) {
	    /*
	     * At least one of the input values was of type 
	     * UNDEFINED which is currently unsupported.
	     */
	    return ( Value::NOVALUE );

	} else if ( ( resultType == type ) && ( resultType == vtype ) ) {
	    /* 
	     * Simple case where both input Values have the same type.
	     */
	    _newValue.traits.setTraits( resultType, 0 );

	    switch ( resultType ) {
	        case CHARACTER:
	    	    _newValue.val.c = val.c + v.val.c;
	    	    break;

	        case INTEGER:
	    	    _newValue.val.i = val.i + v.val.i;
	    	    break;

	        case FLOAT:
	    	    _newValue.val.f = val.f + v.val.f;
	    	    break;

	        case DOUBLE:
	    	    _newValue.val.d = val.d + v.val.d;
	    	    break;

	        case UNDEFINED:
                    break;
	    }
	    return _newValue;

	} else {
	    /* 
	     * We have at least one input type that is different than
	     * the result type.  Perform coercion into Values of desired
	     * type and then do the division operation on those.
	     */
	    return ( getValueAs(resultType) + v.getValueAs(resultType) ); 

	}
}

Value 
Value::operator- ( const Value& v )
{
	/*	
	 * Currently, this operation is not supported for arrays
	 */
	if ( traits.isArray() || v.traits.isArray() )
	    return ( Value::NOVALUE );

	MachineDataType type = traits.getType();
	MachineDataType vtype = v.traits.getType();
	MachineDataType resultType = resultTypeTable[type][vtype];

	if ( resultType == UNDEFINED ) {
	    /*
	     * At least one of the input values was of type 
	     * UNDEFINED which is currently unsupported.
	     */
	    return ( Value::NOVALUE );

	} else if ( ( resultType == type ) && ( resultType == vtype ) ) {
	    /* 
	     * Simple case where both input Values have the same type.
	     */
	    _newValue.traits.setTraits( resultType, 0 );

	    switch ( resultType ) {
	        case CHARACTER:
	    	    _newValue.val.c = val.c - v.val.c;
	    	    break;

	        case INTEGER:
	    	    _newValue.val.i = val.i - v.val.i;
	    	    break;

	        case FLOAT:
	    	    _newValue.val.f = val.f - v.val.f;
	    	    break;

	        case DOUBLE:
	    	    _newValue.val.d = val.d - v.val.d;
	    	    break;

	        case UNDEFINED:
	    	    break;
	    }
	    return _newValue;

	} else {
	    /* 
	     * We have at least one input type that is different than
	     * the result type.  Perform coercion into Values of desired
	     * type and then do the division operation on those.
	     */
	    return ( getValueAs(resultType) - v.getValueAs(resultType) ); 

	}
}

Value 
Value::operator* ( const Value& v )
{
	/*	
	 * Currently, this operation is not supported for arrays
	 */
	if ( traits.isArray() || v.traits.isArray() )
	    return ( Value::NOVALUE );

	MachineDataType type = traits.getType();
	MachineDataType vtype = v.traits.getType();
	MachineDataType resultType = resultTypeTable[type][vtype];

	if ( resultType == UNDEFINED ) {
	    /*
	     * At least one of the input values was of type 
	     * UNDEFINED which is currently unsupported.
	     */
	    return ( Value::NOVALUE );

	} else if ( ( resultType == type ) && ( resultType == vtype ) ) {
	    /* 
	     * Simple case where both input Values have the same type.
	     */
	    _newValue.traits.setTraits( resultType, 0 );

	    switch ( resultType ) {
	        case CHARACTER:
	    	    _newValue.val.c = val.c * v.val.c;
	    	    break;

	        case INTEGER:
	    	    _newValue.val.i = val.i * v.val.i;
	    	    break;

	        case FLOAT:
	    	    _newValue.val.f = val.f * v.val.f;
	    	    break;

	        case DOUBLE:
	    	    _newValue.val.d = val.d * v.val.d;
	    	    break;

	        case UNDEFINED:
	    	    break;
	    }
	    return _newValue;

	} else {
	    /* 
	     * We have at least one input type that is different than
	     * the result type.  Perform coercion into Values of desired
	     * type and then do the division operation on those.
	     */
	    return ( getValueAs(resultType) * v.getValueAs(resultType) ); 

	}
}

Value 
Value::operator/ ( const Value& v )
{
	/*	
	 * Currently, this operation is not supported for arrays
	 */
	if ( traits.isArray() || v.traits.isArray() )
	    return ( Value::NOVALUE );

	MachineDataType type = traits.getType();
	MachineDataType vtype = v.traits.getType();
	MachineDataType resultType = resultTypeTable[type][vtype];

	if ( resultType == UNDEFINED ) {
	    /*
	     * At least one of the input values was of type 
	     * UNDEFINED which is currently unsupported.
	     */
	    return ( Value::NOVALUE );

	} else if ( ( resultType == type ) && ( resultType == vtype ) ) {
	    /* 
	     * Simple case where both input Values have the same type.
	     */
	    _newValue.traits.setTraits( resultType, 0 );

	    switch ( resultType ) {
	        case CHARACTER:
	    	    _newValue.val.c = val.c / v.val.c;
	    	    break;

	        case INTEGER:
	    	    _newValue.val.i = val.i / v.val.i;
	    	    break;

	        case FLOAT:
	    	    _newValue.val.f = val.f / v.val.f;
	    	    break;

	        case DOUBLE:
	    	    _newValue.val.d = val.d / v.val.d;
	    	    break;

	        case UNDEFINED:
	    	    break;
	    }
	    return _newValue;

	} else {
	    /* 
	     * We have at least one input type that is different than
	     * the result type.  Perform coercion into Values of desired
	     * type and then do the division operation on those.
	     */
	    return ( getValueAs(resultType) / v.getValueAs(resultType) ); 

	}
}

void 		/* virtual */
Value::printOn( ostream& os ) const
{
	os << "<Value start>\n";
	traits.printOn( os );

	if ( traits.isArray() ) {
	    os << NL << *val.a;
	} else {
	    switch ( traits.getType() ) {
	    	case CHARACTER:
	    	    os << " value=" << this->operator char() ;
	    	    break;
	    	case INTEGER:
	    	    os << " value=" << this->operator int() ;
	    	    break;
	        case FLOAT:
	    	    os << " value=" << this->operator float();
	    	    break;
	        case DOUBLE:
	    	    os << " value=" << this->operator double();
	    	    break;
	        case UNDEFINED:
	    	    os << "??";
	    	    break;
	    }
	}
	os << "\n<Value end>\n";
}

/*
 * Definition of the static data 
 */
Value Value::_newValue;
Value Value::NOVALUE;
