/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Author: Robert Olson (olson@cs.uiuc.edu)
 * Contributing Author: Brian Totty (totty@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * FIFOStreamPipe.C:  Methods of the class FIFOStreamPipe
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Pipes/RCS/FIFOStreamPipe.C,v 1.16 1994/03/17 16:06:27 aydt Exp $
 */

#include <memory.h>
#include <stdlib.h>
#include <string.h>

#include "FIFOStreamPipe.h"
#include "BinaryPipeReader.h"
#include "PacketHeader.h"

FIFOStreamPipe::FIFOStreamPipe( int size, const CString& name ) 
{
	_setClassName( MY_CLASS );
	_setObjectName( name.getValue() );

	peekLength = 0;

	qSize = size;		
	qBuffer = new char[ qSize ]; 
	qHead = 0;
	qTail = 0;
	qBytesInUse = 0;
}

FIFOStreamPipe::~FIFOStreamPipe()
{
	delete[] qBuffer;
}

PipeReader *	/* virtual */
FIFOStreamPipe::createPipeReader() const
{
	PipeReader *reader = new BinaryPipeReader( (StreamPipe*) this );

	if ( reader != NULL ) {
       	    if ( ! reader->successfulConnection() ) {
	        // We really don"t expect to get here, but just in case!
	       delete reader;
	       reader = NULL;
	    }
	}

	return ( reader );
}

int 
FIFOStreamPipe::put( const char *data, int length )
{
	/* 
	 * If we want to write more than we have space for, print warning &
	 * don't write anything.
	 */
	if ( length > ( qSize - qBytesInUse ) ) {
	    error( "Pipe has only %d bytes free - %d are needed. ", 
					      ( qSize - qBytesInUse ), length );
	    return 0;
	}

	/*
	 * Write to end of the buffer first.  It may all fit or it may not.
	 * If it doesn't, write remainder at beginning of buffer.
	 */
	int bytesToCopy = qSize - qHead;		
	if ( length < bytesToCopy ) {	
	    bytesToCopy = length;
	}

	if ( bytesToCopy > 0 ) {		
	    memcpy( &(qBuffer[qHead]), data, bytesToCopy );
	    data += bytesToCopy;
	    qHead += bytesToCopy;
	}

	bytesToCopy = length - bytesToCopy;	

	if ( bytesToCopy > 0 ) {	
	    memcpy( qBuffer, data, bytesToCopy ); 
	    qHead = bytesToCopy;
	}

	qBytesInUse += length;
	return ( length );
}

int 
FIFOStreamPipe::get( char *data, int length )
{
	peekLength = 0;		// Always reset peek buffer

	/* 
	 * If we want to fetch more than the pipe contains, print warning &
	 * don't fetch anything.
	 */
	if ( qBytesInUse < length ) {
	    error( "Pipe contains %d bytes - %d were requested. ", qBytesInUse,
					                           length );
	    return 0;
	}

	/*
	 * Get from the end of the buffer first.  That may be all we need or
	 * it may not be.  If it isn't, get the remainder from the beginning
	 * of the buffer.
	 */
	int bytesToCopy = qSize - qTail;
	if ( length < bytesToCopy ) {
	    bytesToCopy = length;
	}

	if ( bytesToCopy > 0 ) {
	    memcpy( data, &(qBuffer[qTail]), bytesToCopy );
	    qTail += bytesToCopy;
	    data += bytesToCopy;
	}

	bytesToCopy = length - bytesToCopy;

	if ( bytesToCopy > 0 ) {
	    memcpy( data, qBuffer, bytesToCopy );
	    qTail = bytesToCopy;
	}

	qBytesInUse -= length;
	return( length );
}

int 
FIFOStreamPipe::getSkippingPeek( char *data, int length )
{
	/* 
	 * If we have peeked into the queue, then we want to discard the
	 * bytes we've already looked at.  If there are fewer than peekLength
	 * bytes in the queue buffer, then print a warning and return 0 without
	 * doing any get.
	 *
	 * If there is no active peek, or if we can discard the peekLength
	 * bytes, then call ::get() to do the actual read from the buffer.
	 */
	if ( peekLength != 0 ) {
	    if ( qBytesInUse < peekLength ) {
	        error( "Pipe contains %d bytes - minimum of %d were expected.",
				                          qBytesInUse, length );
	        return 0;
	    }

	    qTail += peekLength;
	    if ( qTail > qSize ) {
		qTail -= qSize;
	    }
	    qBytesInUse -= peekLength;

	    peekLength = 0;	
	}

	return( get( data, length ) );
}

int 
FIFOStreamPipe::peek( char *data, int length )
{
        if ( length <= 0 ) {
	    return length;
	}

	if ( peekLength > 0 ) {
	    warning( "FIFOStreamPipe: Ack! Only one peek at a time, please." );
	    return 0;
	} else {
	    /* 
	     * If we want to see more than the pipe contains, don't get 
	     * anything and return 0.
	     */
	    if ( qBytesInUse < length ) {
	        return 0;
	    }

	    /*
	     * Get from the end of the buffer first.  That may be all we need or
	     * it may not be.  If it isn't, get the remainder from the beginning
	     * of the buffer.
	     */
	    int bytesToCopy = qSize - qTail;
	    if ( length < bytesToCopy ) {
	        bytesToCopy = length;
	    }
    
	    if ( bytesToCopy > 0 ) {
	        memcpy( data, &(qBuffer[qTail]), bytesToCopy );
	        data += bytesToCopy;
	    }
    
	    bytesToCopy = length - bytesToCopy;
    
	    if ( bytesToCopy > 0 ) {
	        memcpy( data, qBuffer, bytesToCopy );
	    }
    
	    peekLength = length;
	    return peekLength;
        }
}
    
int 
FIFOStreamPipe::bytesReady() const
{
	return( qBytesInUse );
}

int 
FIFOStreamPipe::bytesFree() const
{
	return( qSize - qBytesInUse );
}

Boolean_ 
FIFOStreamPipe::isEmpty() const
{
	return( CnvToBoolean_(qBytesInUse == 0) );
}

Boolean_ 
FIFOStreamPipe::isFull() const
{
	return( CnvToBoolean_(qBytesInUse == qSize ));
}


void 	
FIFOStreamPipe::printOn( ostream& os ) const
{
	os << form( "FIFOStreamPipe(%s) 0x%x\n", getObjectName(), this );
	os << form( "Head is %d;  Tail is %d; Bytes in Use: %d\n",
		      qHead, qTail, qBytesInUse );
	os << "\n\n";
}

/*
 * Initialize the static data structures
 */
const char *const FIFOStreamPipe::MY_CLASS = "FIFOStreamPipe";
