/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Contributing Author: Tom Birkett (birkett@cs.uiuc.edu)
 * Contributing Author: Bobby A.A. Nazief (nazief@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * AsciiPipeWriter.cc -  write packets to the ascii output pipe(s).
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Pipes/RCS/AsciiPipeWriter.C,v 1.23 1994/02/25 04:34:18 aydt Exp $
 */

#include <stdlib.h>
#include <string.h>

#include "AsciiPipeWriter.h"
#include "Attributes.h"
#include "AttributesIterator.h"
#include "DataTraits.h"
#include "FieldDescriptor.h"
#include "RecordDossier.h"
#include "StructureDescriptor.h"
#include "StructureDescriptorIterator.h"
#include "StreamPipe.h"
#include "SystemErrors.h"

#define ALLOC_SIZE 8192 	/* Size of increments to use when      */
				/* allocating space for packetBuffer   */

AsciiPipeWriter::AsciiPipeWriter( StreamPipe *thePipe ) 
		:PipeWriter( thePipe )
{
	_setClassName( MY_CLASS );
	currentHeader.type = PIPE_EMPTY;
	currentHeader.length = 0;

	bufferSize = ALLOC_SIZE;
	packetBuffer = (char *) malloc( bufferSize );
	if ( packetBuffer == NULL ) {
	    abort( "Can't malloc space for buffer size %d: %s",
		    bufferSize,  errorString() );
	}

	/*
	 * We only allow Ascii pipe writers to be connected to file pipes
	 * that are open for output.
	 */
	if ( thePipe->isFilePipe() && thePipe->isOutputPipe() ) {
	    _writeSddfHeader();
	    startOffset = thePipe->getByteOffset();
	} else {
	    error( "\n Can't connect to data stream %s", 
		    thePipe->getObjectName() );
	    deletePipe( thePipe );
	} 
}

AsciiPipeWriter::~AsciiPipeWriter()
{
	free( packetBuffer );
}

void 
AsciiPipeWriter::_checkBufferSize( int cnt )
{
	if ( (bufp+cnt) > (packetBuffer+bufferSize) ) {
	      _increaseBufferSize();
	      _checkBufferSize( cnt );
	}
}

void
AsciiPipeWriter::_increaseBufferSize( int newSize )
{
	int bufpOffset = bufp - packetBuffer;	// Remember offset into buffer

	if ( newSize != 0 ) {
	    bufferSize = newSize ;
	} else {
	    bufferSize = bufferSize + ALLOC_SIZE;
	}

	packetBuffer = (char *) realloc( packetBuffer, bufferSize );
	if ( packetBuffer == NULL ) {
            abort( "Can't realloc space for buffer size %d: %s",
                    bufferSize,  errorString() );
        } else {
	    bufp = packetBuffer + bufpOffset;
	}
}

void
AsciiPipeWriter::_putArray( Array *arrayPtr )
{
	char      buffer[64];
	int       buflen;
	int       dim = arrayPtr->getDimension();
	const int *dimSizes = arrayPtr->getDimSizes();
	int       cellCount = arrayPtr->getCellCount();

	/*	newline and tab - back up over space char 	*/
	bufp--;
	_checkBufferSize( 2 );
	_putC('\n'); _putC('\t');

	/* 	dimension size information			*/
	int i;
	for ( i = 0; i < dim; i++ ) {
		_checkBufferSize( 1 );
		_putC('[');
		sprintf( buffer, "%d", dimSizes[i] );
		buflen = strlen( buffer );
		_checkBufferSize( buflen ); 
		strncpy( bufp, buffer, buflen );
		bufp += buflen;
		_checkBufferSize( 1 );
		_putC(']');
	}		

	/*	array elements					*/
	_checkBufferSize( 4 );
	_putC(' '); _putC('{' ); _putC('\n'); _putC('\t');

	if ( cellCount != 0 ) {
	    if ( arrayPtr->getType() == CHARACTER ) {
		int strLength = dimSizes[dim-1];	// length of the strings
		int strCount  = cellCount / strLength;  // number of strings
		int strIdx    = 0;			// current string
		int strNum, charNum;

		for ( strNum = 0; strNum < strCount; strNum++ ) {
		    _checkBufferSize( 1 );
		    _putC('"');

		    for ( charNum =0; charNum < strLength; charNum++ ) {
			char c = 
			     (char) arrayPtr->getTheCellValue( strIdx+charNum );
			if ( c == '\0' ) {
			    break;
			} else {
			    _checkBufferSize( 1 );
			    _putC( c );
			}
		    }

		    strIdx += strLength;
		    if ( strNum < ( strCount - 1 ) ) {
			_checkBufferSize( 4 );
			_putC('"'); _putC(','); _putC('\n'); _putC('\t');
		    } else {
			_checkBufferSize( 1 );
			_putC('"');
		    }
		}
	    } else {
		Value value;

		for( i = 0; i < cellCount; i++ ) {
			value = arrayPtr->getTheCellValue( i );
			_putValue( &value );
			if ( i < ( cellCount - 1 ) ) {
				_checkBufferSize( 3 );
				_putC(','); _putC('\n'); _putC('\t');
			}
		}
	    }
	}
	_checkBufferSize( 3 );
	_putC('\n'); _putC('\t'); _putC('}');
}

void
AsciiPipeWriter::_putValue( Value *valuePtr )
{
	char  buffer[64];
	int   buflen;

	switch( valuePtr->getTraits().getType() ) {
	  case CHARACTER:
		_checkBufferSize( 1 );
		_putC( valuePtr->operator char() );
		break;
	  case INTEGER:
		sprintf( buffer, "%d", valuePtr->operator int() );
		buflen = strlen( buffer );
		_checkBufferSize( buflen ); 
		strncpy( bufp, buffer, buflen );
		bufp += buflen;
		break;
	  case FLOAT:
		sprintf( buffer, "%.7g", valuePtr->operator float() );
		buflen = strlen( buffer );
		_checkBufferSize( buflen ); 
		strncpy( bufp, buffer, buflen );
		bufp += buflen;
		break;
	  case DOUBLE:
		sprintf( buffer, "%.16g", valuePtr->operator double() );
		buflen = strlen( buffer );
		_checkBufferSize( buflen ); 
		strncpy( bufp, buffer, buflen );
		bufp += buflen;
		break;
	  case UNDEFINED:
	  default:
		break;
	}
}

void
AsciiPipeWriter::_writeCString( const CString& cstr )
{
	char c;
	int length = cstr.length();

	_checkBufferSize( length + 2 ); 	// String plus 2 "s
	_putC('"');

	for ( int i = 0; i < length; i++ ) {
		switch ( c = cstr[i] ) {
		  case '\b':
		  case '\f':
		  case '\r':
		  case '\t':
		  case '\v':
		  case '\\':
		  case '\'':
		  case '\"':
		  case '\0':
			/* need extra space for the \  - length of string *
			 * remaining, extra space for \ and trailing "    */ 
			_checkBufferSize( length - i + 1 + 1 ); 
			_putC('\\');
			_putC( c );
			break;
		  default:
			_putC( c );
			break;
		}
	}
	_putC('"');
}

void
AsciiPipeWriter::_writePacket()
{
	enum Flag{ notSet, isSet }; 
	Flag errorFlag = notSet;

	switch( currentHeader.type ) {
	  case PKT_ATTRIBUTE:
		if ( *packetBuffer != '/' ) {
		  error( "Attribute packet should start with '/'" );
		  errorFlag = isSet;
		}
		break;
	  case PKT_DESCRIPTOR:
		if ( *packetBuffer != '#' ) {
		  error( "Descriptor packet should start with '#'" );
		  errorFlag = isSet;
		}
		break;
	  case PKT_DATA:
		if ( *packetBuffer != '"' ) {
		  error( "Data packet should start with '\"'" );
		  errorFlag = isSet;
		}
		break;
	  case PKT_COMMAND:
		if ( *packetBuffer != '%' ) {
		  error( "Command packet should start with '%'" );
		  errorFlag = isSet;
		}
		break;
	  default:
		error( "Unknown packet type" );
		errorFlag = isSet;
		break;
	}
	if ( errorFlag ) {
		error( "Following packet will not be written to pipe: %s ",
			packetBuffer );
	} else {
		int p;
		for ( p = 0; p < activePipes; p++ ) {
		    int nWritten;
		    nWritten = pipe[p]->put( packetBuffer, 
							currentHeader.length );
		    if ( nWritten != currentHeader.length ) {
			error( "writePacket failed! pipe=%d len=%d written=%d",
			             p, currentHeader.length, nWritten );
		    }
		    nWritten = pipe[p]->put( "\n", 1 );
		    if ( nWritten != 1 ) {
			error( "writePacket failed! pipe=%d len=%d written=%d",
			             p, 1, nWritten);
		    }
		}
	}
}

void		/* virtual */
AsciiPipeWriter::_writeSddfHeader() const
{
	int nWritten = pipe[activePipes-1]->put( "SDDFA\n", 6 );
	if ( nWritten != 6 ) {
	    abort( "Unable to write SDDF file header to pipe %s",
		    pipe[activePipes-1]->getObjectName() );
	}
}

Boolean_        /* virtual */
AsciiPipeWriter::addPipe( StreamPipe *thePipe )
{
        /*
         * We only allow ascii pipe writers to be connected to file pipes.
         */
        Boolean_ rc;
        if ( thePipe->isFilePipe() ) {
            rc = PipeWriter::addPipe( thePipe );
        } else {
            error( "Can't connect to data stream %s (Not a File)",
                    thePipe->getObjectName() );
            rc =  FAILURE_;
        }
        return rc;
}

void		/* virtual */
AsciiPipeWriter::putAttributes()
{
	bufp = packetBuffer;
	_checkBufferSize( 3 );
	_putC('/'); _putC('*'); _putC('\n');

	AttributesIterator *iterator = new AttributesIterator( pipeAttributes );
	CString key, value;

	while ( ( key = iterator->next() ) != CString::NOMATCH ) {
		value = iterator->theValue();

		_checkBufferSize( 3 );
		_putC(' '); _putC('*'); _putC(' ');
		_writeCString( key );
		_checkBufferSize( 1 );
		_putC(' ');
		_writeCString( value );
		_checkBufferSize( 1 );
		_putC('\n');
	}
	delete iterator;

	_checkBufferSize( 7 );
	_putC(' '); _putC('*'); _putC('/');  _putC(' ');
	_putC(';'); _putC(';'); _putC('\n');

	currentHeader.length = bufp - packetBuffer;
	currentHeader.type = PKT_ATTRIBUTE;
	currentHeader.tag = 0;			// tag not used for Attr pkt

	_writePacket();
}

void		/* virtual */
AsciiPipeWriter::putAttributes( const Attributes& attributes )
{
	pipeAttributes = attributes;
	putAttributes();
}

void		/* virtual */
AsciiPipeWriter::putCommand( int pktTag )
{
	bufp = packetBuffer;
	sprintf( bufp, "%%%d: ;;\n", pktTag );
	while ( *bufp++ != '\n' ) ;

	currentHeader.length = bufp - packetBuffer;
	currentHeader.type = PKT_COMMAND;
	currentHeader.tag = pktTag;

	_writePacket();
}
	
void		/* virtual */
AsciiPipeWriter::putData( const RecordDossier& recordDossier )
{
	bufp = packetBuffer;
	_writeCString( recordDossier.getName() );
	_checkBufferSize( 3 );
	_putC(' '); _putC('{'); _putC(' ');

	Value *valuePtr;
	int fieldCount = recordDossier.entryCount();
	for ( int i = 0; i < fieldCount; i++ ) {
		valuePtr = recordDossier.getValueP( i );
		if ( valuePtr->getTraits().isArray() ) {
			_putArray( (Array *)(* valuePtr) );
		} else {
			_putValue( valuePtr );
		}
		if ( i < ( fieldCount - 1 ) ) {
			_checkBufferSize( 2 );
			_putC(','); _putC(' ');
		}
	}

	_checkBufferSize( 5 );
	_putC(' '); _putC('}'); _putC(';'); _putC(';'); _putC('\n');

	currentHeader.length = bufp - packetBuffer;
	currentHeader.type = PKT_DATA;
	currentHeader.tag = recordDossier.getTag();

	_writePacket();
}

void		/* virtual */
AsciiPipeWriter::putDescriptor( const StructureDescriptor& structDescr,
				 int pktTag )
{
	CString key, value;

	bufp = packetBuffer;

	/* 	Packet Tag					*/
	sprintf( bufp, "#%d:\n", pktTag );
	while ( *bufp++ != '\n' ) ;

	/* 	Structure Descriptor Attributes (if any) 	*/
	AttributesIterator *attrIter = 
			new AttributesIterator( structDescr.getAttributes() );

	while( ( key = attrIter->next() ) != CString::NOMATCH ) {
		value = attrIter->theValue();

		_checkBufferSize( 3 );
		_putC('/'); _putC('/'); _putC(' ');
		_writeCString( key );
		_checkBufferSize( 1 );
		_putC(' ');
		_writeCString( value );
		_checkBufferSize( 1 );
		_putC('\n');
	}
	delete attrIter;

	/*	Structure Descriptor Name 			*/
	_writeCString( structDescr.getName() );
	_checkBufferSize( 3 );
	_putC(' '); _putC('{'); _putC('\n');

	/*	Fields in the Structure Descriptor 		*/
	StructureDescriptorIterator *structIter = 
				new StructureDescriptorIterator( structDescr );
	
	MachineDataType	fieldType;
	CString		fieldName;
	int		fieldDim;
	FieldDescriptor field = structIter->first();

	while( field.notNull() ) {

		/*	Field Attributes ( if any )		*/
		attrIter = new AttributesIterator( field.getAttributes() );
		while( ( key = attrIter->next() ) != CString::NOMATCH ) {
			value = attrIter->theValue();
	
			_checkBufferSize( 4 );
			_putC('\t'); _putC('/'); _putC('/'); _putC(' ');
			_writeCString( key );
			_checkBufferSize( 1 );
			_putC(' ');
			_writeCString( value );
			_checkBufferSize( 1 );
			_putC('\n');
		}
		delete attrIter;

		/* 	Field type, name, and dimensions	*/
		_checkBufferSize(1);
		_putC('\t'); 
		switch ( fieldType = field.getType() ) {
	  	  case CHARACTER:
			_checkBufferSize( 4 );
			_putC('c'); _putC('h'); _putC('a'); _putC('r');
			break;
	  	  case INTEGER:
			_checkBufferSize( 3 );
			_putC('i'); _putC('n'); _putC('t');
			break;
	  	  case FLOAT:
			_checkBufferSize( 5 );
			_putC('f'); _putC('l'); _putC('o'); 
			_putC('a'); _putC('t');
			break;
	  	  case DOUBLE:
			_checkBufferSize( 6 );
			_putC('d'); _putC('o'); _putC('u');
			_putC('b'); _putC('l'); _putC('e');
			break;
	  	  case UNDEFINED:
		  default:
			_checkBufferSize( 9 );
			_putC('u'); _putC('n'); _putC('d'); 
			_putC('e'); _putC('f'); _putC('i'); 
			_putC('n'); _putC('e'); _putC('d'); 
			break;
		}
		_checkBufferSize(1);
		_putC('\t'); 

		fieldName = field.getName();
		_writeCString( fieldName );

		for ( fieldDim=field.getDimension(); fieldDim>0; fieldDim-- ) {
			_checkBufferSize( 2 );
			_putC('['); _putC(']');
		}
		_checkBufferSize( 2 );
		_putC(';'); _putC('\n');

		field = structIter->next();
	}
	delete structIter;

	_checkBufferSize( 4 );
	_putC('}'); _putC(';'); _putC(';'); _putC('\n');

	currentHeader.length = bufp - packetBuffer;
	currentHeader.type = PKT_DESCRIPTOR;
	currentHeader.tag = pktTag;

	_writePacket();
}

/*
 * Initialize the static data 
 */
const char *const AsciiPipeWriter::MY_CLASS = "AsciiPipeWriter";
