/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Robert Olson (olson@cs.uiuc.edu)
 * Contributing Author: Ruth Aydt (aydt@cs.uiuc.edu)
 * Contributing Author: Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * PabloMainInterface.cc: 	Implements the primary user interface
 *			for the Pablo data analysis environment
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Interface/RCS/PabloMainInterface.C,v 1.36 1994/04/11 17:09:15 aydt Exp $	
 *
 */

//#define DEBUG_MI

#include <pwd.h>
#include <dirent.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>

#ifndef __GNUG__
#include <sysent.h>
#else
#include <std.h>
extern "C" {
int gethostname( char *name, int namelen );
}
#endif

#ifdef __PARAGON__
extern "C" {
extern int  strcasecmp(const char*, const char*);
}
#endif

#include <Xm/CascadeB.h>
#include <Xm/LabelG.h>
#include <Xm/PushBG.h>
#include <Xm/SeparatoG.h>
#include <Xm/MainW.h>


#ifdef XPM
#include <X11/xpm.h>
#endif


#include "PabloMainInterface.h"
#include "InfrastructureManager.h"
#include "AddModuleDialog.h"
#include "DefaultsDialog.h"
#include "FileSelectionDialog.h"
#include "FunctionalUnit.h"
#include "GeneralDialogBox.h"
#include "GraphDisplay.h"
#include "GraphNode.h"
#include "ModuleId.h"
#include "ModuleIdPtrList.h"
#include "ObjectRegistry.h"
#include "Pablo.h"
#include "PabloHelpSystem.h"
#include "PabloPixmaps.h"
#include "RunDialog.h"
#include "StructureDictionary.h"
#include "SystemErrors.h"
#include "TagMappingDictionary.h"
#include "TranscriptWindowBuf.h"
#include "WidgetBase.h"
#include "Wrapper.h"


// Stuff for console logging. Bufs are defined in Pablo.cc
extern TranscriptWindowBuf PabloTranscriptBuf;
extern TranscriptWindowBuf PabloTranscriptErrorBuf;


static char *InterfaceStateNames[] = {
	"Idle",
	"Waiting for first module or exit",
	"Waiting for second module or exit",
	"Adding module connection",
	"Configuring ",
	"Waiting for node selection or exit",
	"Executing graph continuously",
	"Executing graph N times",
	"Waiting to run ",
	"Running module ",
	"Execution complete",
	NULL,
};

static char *CompleteConfigMsg = 
	"Complete configuration currently in progress before \n%s.";

/*  
 * TransitionTable - What State changes are valid. We go to Execution
 * Complete when Quiet and variable executionDone == TRUE_.
 *
 *          Quiet First Second Link Config Press Run Discrete Wait Single 
 * Quiet      Y     Y     N      N     Y      Y    Y     Y      Y     N   
 * First      Y     N     Y      N     N      N    N     N      N     N  
 * Second     Y     N     N      Y     N      N    N     N      N     N
 * Link       N     Y     N      N     N      N    N     N      N     N
 * Config     Y     N     N      N     N      Y    N     N      N     N
 * Press      Y     N     N      N     N      N    N     N      N     N 
 * Run        Y     Y     N      N     Y      Y    Y     Y      N     N 
 * Discrete   Y     N     N      N     N      Y    N     N      N     N
 * Wait       N     N     N      N     N      N    N     N      N     Y
 * Single     Y     N     N      N     N      N    N     N      Y     N
 */

static const Boolean_ StateTransition[10][10] =
{
	/* Quiet */
  	{ SUCCESS_, SUCCESS_, FAILURE_, FAILURE_, SUCCESS_, 
          SUCCESS_, SUCCESS_, SUCCESS_, SUCCESS_, FAILURE_, },

	/* First */
  	{ SUCCESS_, FAILURE_, SUCCESS_, FAILURE_, FAILURE_, 
          FAILURE_, FAILURE_, FAILURE_, FAILURE_, FAILURE_, },

	/* Second */
  	{ SUCCESS_, FAILURE_, FAILURE_, SUCCESS_, FAILURE_, 
          FAILURE_, FAILURE_, FAILURE_, FAILURE_, FAILURE_, },

	/* Link */
  	{ FAILURE_, SUCCESS_, FAILURE_, FAILURE_, FAILURE_, 
          FAILURE_, FAILURE_, FAILURE_, FAILURE_, FAILURE_, },

	/* Config */
  	{ SUCCESS_, FAILURE_, FAILURE_, FAILURE_, FAILURE_, 
          SUCCESS_, FAILURE_, FAILURE_, FAILURE_, FAILURE_, },

	/* Press */
  	{ SUCCESS_, FAILURE_, FAILURE_, FAILURE_, FAILURE_, 
          FAILURE_, FAILURE_, FAILURE_, FAILURE_, FAILURE_, },

	/* Run */
  	{ SUCCESS_, SUCCESS_, FAILURE_, FAILURE_, SUCCESS_, 
          SUCCESS_, SUCCESS_, SUCCESS_, FAILURE_, FAILURE_, },

	/* Discrete */
  	{ SUCCESS_, FAILURE_, FAILURE_, FAILURE_, FAILURE_, 
          SUCCESS_, FAILURE_, FAILURE_, FAILURE_, FAILURE_, },

	/* Wait   */
  	{ FAILURE_, FAILURE_, FAILURE_, FAILURE_, FAILURE_, 
          FAILURE_, FAILURE_, FAILURE_, FAILURE_, SUCCESS_, },

	/* Single */
  	{ SUCCESS_, FAILURE_, FAILURE_, FAILURE_, FAILURE_, 
          FAILURE_, FAILURE_, FAILURE_, SUCCESS_, FAILURE_, },
};

static XmString InterfaceStateStrings[nInterfaceStates];
static Widget fileWidget;
static Widget runWidget;
static Widget configureWidget;
static Widget moduleWidget;
static Widget helpOnModuleInstanceWidget;


/********************** G l o b a l   V a r i a b l e s **********************/

PabloData pabloData;


/********* C O N S T R U C T O R S   A N D   D E S T R U C T O R S *********/


PabloMainInterface::PabloMainInterface( int *argc, char **argv )
{
	_setClassName( MY_CLASS );

	consoleLogFP = NULL;
	moduleNowBeingConfigured = NULL;
	executionDone = FALSE_;
	state = Quiet;

	createInterface( "Pablo", argc, argv );

	processArguments( *argc, argv );

	/* Dialogs not yet created so set pointers/flags appropriately */
	FSD = NULL;			
	FSD_busy = FALSE_;	

	addModuleDialog = NULL;	
	runPanelDialog = NULL;
	defaultsDialog = NULL;
	_desensitizeCnt = 0;
	_desensitizeRunCnt = 0;
}

PabloMainInterface::~PabloMainInterface()
{
}

void 
PabloMainInterface::createInterface( char *interfaceClass, int *argc, 
				     char **argv )
{
	if ( singleInstance != NULL ) {
	   abort( "Only one instance of primary Pablo interface allowed" );
	} else {
	   singleInstance = this;
	}
	
	topLevel = XtVaAppInitialize( &appContext, interfaceClass,
                                      options, XtNumber(options),
                                      argc, argv, NULL, NULL );

	/*
	 * Work-around if above fails...
	 *
	 * topLevel = XtAppInitialize( &appContext, interfaceClass,
         *                          options, XtNumber(options),
         *                          argc, argv, NULL, NULL, 0 );
	 */

	baseTopLevel = new WidgetBase( NULL );	// for compatability with
	baseTopLevel->setWidget( topLevel );	// the remainder of the system

	loadResources();

	mainPabloInterface = XtVaCreateManagedWidget( "mainInterface",
			             xmMainWindowWidgetClass, topLevel, 
				     NULL );

	mainPabloMenu = XmCreateMenuBar( mainPabloInterface, "MenuBar", 
				     NULL, 0 );

	/****  Set up menus ****/
#include "PabloMainMenus.def"


	int i;
	for ( i = 0; pabloMenuData[i].menuTitle != NULL; i++ ) {
	   buildMenuItem( mainPabloMenu, pabloMenuData[i] );
	}

 	workWindow = XtVaCreateWidget ( "workWindow",
			      xmFormWidgetClass, mainPabloInterface, 
                              NULL );

	createPabloPixmaps();

	/*
	*
	*	Create the forms for status labels and values above 
	*	the graph window.  Use two individual row-column widgets
	*	so that the values can resize without the labels resizing.
	*	Attach values to the labels on the left and the pixmap on
	*	the right so that they don't change size every time one
	*       of them is updated --- too much flicker that way.
	*/
	Widget statusLabels = XtVaCreateWidget ( "statusLabels",
                              xmRowColumnWidgetClass, workWindow,
			      XmNleftAttachment, XmATTACH_FORM,
			      XmNtopAttachment, XmATTACH_FORM,
                              NULL );

	Widget statusValues = XtVaCreateWidget ( "statusValues",
                              xmRowColumnWidgetClass, workWindow,
			      XmNleftAttachment, XmATTACH_WIDGET,
                       	      XmNleftWidget, statusLabels,
			      XmNrightAttachment, XmATTACH_WIDGET,
                       	      XmNrightWidget, pixmapFrame,
			      XmNtopAttachment, XmATTACH_FORM,
                              NULL );

	XmString text;		// used for text
	Widget frame;		// used for frames
	Widget label;		// used for labels

	/*
	*
	*	Create the system state label and value.  
	*
	*/
	text = XmStringCreateLtoR( "System Status:", XmSTRING_DEFAULT_CHARSET );
	label = XtVaCreateManagedWidget( "stateLabel",
                              xmLabelWidgetClass, statusLabels,
			      XmNalignment, XmALIGNMENT_BEGINNING,
			      XmNresizable, FALSE,
			      XmNlabelString, text,
                              NULL );
	XmStringFree( text );

 	frame = XtVaCreateManagedWidget( "stateValueFrame",
                              xmFrameWidgetClass, statusValues,
                              NULL );

 	stateValue = XtVaCreateWidget( "stateValue",
                              xmLabelWidgetClass, frame,
			      XmNalignment, XmALIGNMENT_BEGINNING,
			      XmNresizable, FALSE,
                              NULL );
  	initializeStateStrings();
        _setState( Quiet );

	/*
	*
	*	Create the modification date label and value
	*
	*/
	text = XmStringCreateLtoR( "Last Modification Date: ",
			      XmSTRING_DEFAULT_CHARSET );

	label = XtVaCreateManagedWidget( "dateLabel",
                              xmLabelWidgetClass, statusLabels,
			      XmNalignment, XmALIGNMENT_BEGINNING,
			      XmNresizable, FALSE,
			      XmNlabelString, text,
                              NULL );
	XmStringFree( text );

 	frame = XtVaCreateManagedWidget( "dateValueFrame",
                              xmFrameWidgetClass, statusValues,
                              NULL );

	time_t now = time(0);
	char *date = ctime( &now );
	date[24] = '\0';
 	text =  XmStringCreateLtoR( date, XmSTRING_DEFAULT_CHARSET );
 	dateValue = XtVaCreateWidget( "dateValue",
                              xmLabelWidgetClass, frame,
			      XmNresizable, FALSE,
			      XmNalignment, XmALIGNMENT_BEGINNING,
        		      XmNlabelString, text, 
                              NULL );
        XmStringFree( text );

	/*
	*
	*	Create the system configuration label and value
	*
	*/
	text =  XmStringCreateLtoR( "Configuration Name: ",
					    XmSTRING_DEFAULT_CHARSET );
	label = XtVaCreateManagedWidget( "configLabel",
                              xmLabelWidgetClass, statusLabels,
			      XmNalignment, XmALIGNMENT_BEGINNING,
			      XmNresizable, FALSE,
			      XmNlabelString, text,
                              NULL );
	XmStringFree( text );

 	frame = XtVaCreateManagedWidget( "configValueFrame",
                              xmFrameWidgetClass, statusValues,
                              NULL );

 	text =  XmStringCreateLtoR( "NULL Configuration",
                                            XmSTRING_DEFAULT_CHARSET );
 	configValue = XtVaCreateWidget( "configValue",
                              xmLabelWidgetClass, frame,
			      XmNalignment, XmALIGNMENT_BEGINNING,
			      XmNresizable, FALSE,
			      XmNlabelString, text,
                              NULL );
	XmStringFree( text );

	/*
	 *
	 *	Create the scrollbox the will contain the Pablo
	 *	module graph
	 *
	*/

	scrollBox = XtVaCreateWidget( "scrollBox",
				xmScrolledWindowWidgetClass, workWindow, 
				XmNscrollingPolicy, XmAUTOMATIC,
				XmNscrollBarDisplayPolicy, XmSTATIC,
				XmNleftAttachment, XmATTACH_FORM,
                       		XmNrightAttachment, XmATTACH_FORM,
                       		XmNbottomAttachment, XmATTACH_FORM,
                       		XmNtopAttachment, XmATTACH_WIDGET,
                       		XmNtopWidget, statusValues,
				XmNtopOffset, 0,
				NULL );

  	baseGraphWindow = new WidgetBase( NULL );  // for compatability with
        baseGraphWindow->setWidget( scrollBox ); // remainder of system

        graphDisplay = new GraphDisplay( baseGraphWindow );

 	XtManageChild( mainPabloMenu );
	XtManageChild( workWindow );

	XtManageChild( stateValue );
	XtManageChild( dateValue );
	XtManageChild( configValue );

	XtManageChild( pixmapFrame );

	XtManageChild( statusLabels );
	XtManageChild( statusValues );

	XtManageChild( scrollBox );
	XtManageChild( graphDisplay->getWidget() );

        XtRealizeWidget( topLevel );
}	

void
PabloMainInterface::buildMenuItem( Widget w, menuLine menu )
{
	Widget		pulldownMenu;
	Widget		cascadeWidget;
	Widget		entryWidget;
	XmString	textString;

	int	i;

	pulldownMenu = XmCreatePulldownMenu( w, "pabloPulldown", NULL, 0 ); 

	textString = XmStringCreateSimple( menu.menuTitle );

	/*
	*
	*   Create the pulldown menu entry
	*
	*/

	cascadeWidget =
		XtVaCreateManagedWidget( menu.menuTitle, 
					 xmCascadeButtonWidgetClass, 
				         w,
					 XmNsubMenuId, pulldownMenu,
					 XmNlabelString, textString,
					 XmNmnemonic, menu.mnemonicCharacter,
					 NULL );

	Boolean_ thisIsHelpMenu = FALSE_;
 	if ( strcmp( menu.menuTitle, "Help" ) == 0 ) {
           XtVaSetValues( w, XmNmenuHelpWidget, cascadeWidget, NULL );
	   thisIsHelpMenu = TRUE_;
        }

 	if ( strcmp( menu.menuTitle, "File" ) == 0 ) {
	   fileWidget = cascadeWidget;
	} else if ( strcmp( menu.menuTitle, "Run" ) == 0 ) {
	   runWidget = cascadeWidget;
	} else if ( strcmp( menu.menuTitle, "Configure" ) == 0 ) {
	   configureWidget = cascadeWidget;
	} else if ( strcmp( menu.menuTitle, "Module" ) == 0 ) {
	   moduleWidget = cascadeWidget;
	}

	XmStringFree( textString );

	/*
        *
        *   Create the list of items in the pulldown menu
        *
        */

	for (i = 0; menu.menuEntry[i].menuLabel != NULL; i++) {
	   entryWidget =
		XtVaCreateManagedWidget( menu.menuEntry[i].menuLabel,
					 *(menu.menuEntry[i].widgetClass),
					 pulldownMenu,
					 NULL );
 	   if ( thisIsHelpMenu && 
	        ( strcmp( menu.menuEntry[i].menuLabel, "On Module Instance" ) 
		      == 0) ) {
		helpOnModuleInstanceWidget = entryWidget;
	   }

	   if ( menu.menuEntry[i].mnemonicCharacter ) {
	      XtVaSetValues( entryWidget,
			     XmNmnemonic, menu.menuEntry[i].mnemonicCharacter,
			     NULL ); 
	   }

	   if ( menu.menuEntry[i].acceleratorString ) {
	      textString = XmStringCreateSimple(
				menu.menuEntry[i].acceleratorText );

	      XtVaSetValues( entryWidget,
                           XmNaccelerator, menu.menuEntry[i].acceleratorString,
			   XmNacceleratorText, textString, 
                           NULL );

	      XmStringFree( textString );
	   }

  	   if ( menu.menuEntry[i].callbackFunction ) {
	      XtAddCallback( entryWidget, XmNactivateCallback,
			     menu.menuEntry[i].callbackFunction,
			     menu.menuEntry[i].callbackData );
	   }
	}
}	

void
PabloMainInterface::createPabloPixmaps()
{
	unsigned int depth = DefaultDepth( XtDisplay(workWindow),
				   DefaultScreen( XtDisplay(workWindow) ) );
	Pixmap pabloPixmap;
	int pixmapNotCreated = 1;

	/* 
	 * If the user has the XPM library and a color display, then
	 * use the color pixmap.  If there is no XPM, if we don't have
	 * a color screen, or if the xpm color pixmap fails, then use
	 * two-tone pixmap.
	 */

#ifdef XPM
        if ( depth > 1 ) {
	    Pixmap pabloShapeMask;
	    pixmapNotCreated = XpmCreatePixmapFromData( 
					   XtDisplay( workWindow ),
				           XtScreen( workWindow )->root,
				           pabloPixmap_xpm,
				           &pabloPixmap,
				           &pabloShapeMask,
				           NULL );
				       
	    if ( pixmapNotCreated ) {
		cerr << "Unable to load Pablo color pixmap\n";
	   	cerr << form( "  xpm error code = %d\n", pixmapNotCreated );
	    }
	}
#endif

	if ( pixmapNotCreated ) {
	    Colormap cmap;
	    XColor pabloColor;
	    XColor unused;
	    XtVaGetValues( workWindow, XmNcolormap, &cmap, NULL );

	    XAllocNamedColor( XtDisplay(workWindow), cmap,
			      "dark orange", &pabloColor, &unused );
	    Pixel darkOrange = pabloColor.pixel;

	    XAllocNamedColor( XtDisplay(workWindow), cmap,
                              "blue", &pabloColor, &unused );
	    Pixel blue = pabloColor.pixel;

            pabloPixmap = XCreatePixmapFromBitmapData( 
					   XtDisplay( workWindow ),
                      	                   XtScreen(workWindow)->root,
                                           pabloPixmap_xbm, 
				           pabloPixmap_xbm_width,
                               	           pabloPixmap_xbm_height, 
				           darkOrange,
                               	           blue,
					   depth );

	    if ( pabloPixmap == XmUNSPECIFIED_PIXMAP ) {
	       cerr << ( "Unable to load Pablo pixmap\n" );
	       pixmapNotCreated = 1;
	    } else {
	       pixmapNotCreated = 0;
	    }
	}	

	pixmapFrame = XtVaCreateWidget( "pixmapFrame",
                              		xmFrameWidgetClass, workWindow,
                              		XmNrightAttachment, XmATTACH_FORM,
                              		XmNtopAttachment, XmATTACH_FORM,
                              		NULL );

	if ( ! pixmapNotCreated ) {
	    pabloGraphic = XtVaCreateWidget( "pabloPixmap",
			  		xmLabelGadgetClass, pixmapFrame,
			  		XmNlabelType, XmPIXMAP,
			  		XmNlabelPixmap, pabloPixmap,
			  		NULL );
	    XtManageChild( pabloGraphic );

            XtVaSetValues( topLevel, XtNiconPixmap, pabloPixmap, NULL );
	}
}

void 
PabloMainInterface::loadResources()
{
	XtGetApplicationResources( topLevel,
				   (XtPointer) &pabloData,
				   resources, XtNumber(resources),
				   (ArgList) NULL, (Cardinal) 0);

#ifdef DEBUG_MI 

#define showTF( var ) var ? "TRUE" : "FALSE"

	cout << "Retrieved Pablo Application Resources:\n";

	cout << form( "runStepSize = %d\n", 
		       pabloData.runStepSize );
 	cout << form( "defaultConfigFileDir = %s\n", 
		       pabloData.defaultConfigFileDir );
 	cout << form( "defaultDataFileDir = %s\n", 
		       pabloData.defaultDataFileDir );
 	cout << form( "defaultHelpFileDir = %s\n", 
		       pabloData.defaultHelpFileDir );
	cout << form( "consoleLog = %s\n", 
		       pabloData.consoleLog );
	cout << form( "roundNodes = %s\n", 
		       showTF( pabloData.roundNodes ) );
	cout << form( "eventInterval = %d\n", 
		       pabloData.eventInterval );
	cout << form( "maxEventInterval = %d\n", 
		       pabloData.maxEventInterval );
	cout << form( "timeInterval = %d\n", 
		       pabloData.timeInterval );
	cout << form( "maxTInterval = %d\n", 
		       pabloData.maxTimeInterval );
        /*
	cout << form( "restartReuseInfile = %s\n",
		       showTF( pabloData.restartReuseInFile ) );
        cout << form( "restartReuseOutfile = %s\n",
		       showTF( pabloData.restartReuseOutFile ) );
        cout << form( "restartClearModules = %s\n",
		       showTF( pabloData.restartClearModules ) );
	*/
        cout << form( "displayBytesProcessed = %s\n",
		       showTF( pabloData.displayBytesProcessed ) );
        cout << form( "verifyDeleteGraph = %s\n",
		       showTF( pabloData.verifyDeleteGraph ) );
        cout << form( "animateGraph = %s\n",
		       showTF( pabloData.animateGraph ) );

#endif DEBUG_MI

	/* 
	 * Check various defaults 
	 */
	DIR *dirPtr;

	if ( pabloData.runStepSize <= 0 ) {
	    cerr << form( "Warning: invalid runStep size: %d\n",
			   pabloData.runStepSize );
	    cerr << "	Resetting to 1. \n";
	    pabloData.runStepSize = 1;
	}

	dirPtr = opendir( pabloData.defaultConfigFileDir );
	if ( dirPtr == NULL ) {
	    cerr << form( "Warning: cannot open defaultConfigFileDir %s:  %s\n",
			   pabloData.defaultConfigFileDir, errorString() );
	    cerr << "   Resetting to current directory. \n";
	    pabloData.defaultConfigFileDir = ".";
	} else {
	    closedir( dirPtr );
	}

	dirPtr = opendir( pabloData.defaultDataFileDir );
	if ( dirPtr == NULL ) {
	    cerr << form( "Warning: cannot open defaultDataFileDir %s:  %s\n",
			   pabloData.defaultDataFileDir, errorString() );
	    cerr << "   Resetting to current directory. \n";
	    pabloData.defaultDataFileDir = ".";
	} else {
	    closedir( dirPtr );
	}
			
	updateConsoleLog();

}

void 
PabloMainInterface::processArguments( int argc, char **argv )
{
	int i;

	for ( i = 1; i < argc; i++ ) {
	   if ( strcmp(argv[i], "-openDir") == 0 ) {
	      if ( ++i >= argc ) {
	         cerr << "-openDir needs an argument\n";
		 break;
	      }
	      dirToOpen = argv[i];

	    } else if ( strcmp(argv[i], "-open") == 0 ) {
		if ( ++i >= argc ) {
		   cerr << "-open needs an argument\n";
		   break;
		}
		fileToOpen = argv[i];
	    } else {
		cerr << "Invalid argument " << argv[i] << NL;
	    }
	}
}

void 
PabloMainInterface::autoLoadConfiguration()
{
	if ( fileToOpen != "" ) {
	    _readLayoutFromFile( fileToOpen );
            _updateConfigFilename( fileToOpen.getValue() );
	} else if ( dirToOpen != "" ) {
	    _loadConfigurationFromDir( dirToOpen );
	    _updateConfigFilename( dirToOpen.getValue() );
	}
}

void 
PabloMainInterface::initializeStateStrings()
{
	int i;

	for (i = 0; i < nInterfaceStates; i++) {
	   InterfaceStateStrings[i] =
                XmStringCreateLtoR( InterfaceStateNames[i],
                                    XmSTRING_DEFAULT_CHARSET );
	}
}

void 
PabloMainInterface::_updateConfigFilename( const char* fileName )
{
	/*
	 * If fileName exists and isn't a directory, update the Modification
	 * time from the modification time of the file.  We don't update
	 * the Modification time if it's a directory because that is
	 * taken care of elsewhere using the value from the config "Info" file.
	 */

	XmString	configMsg;
	char		configPath[100];
	
	char*		configText;
	char		tmpText[100];
	int		i;

	struct stat statbuf;
	if ( stat( (char *)fileName, &statbuf ) == 0 ) {

	    if ( !( statbuf.st_mode & S_IFDIR ) ) {
	        char *modifyDate = ctime( &statbuf.st_mtime );
		modifyDate[24] = '0';
		_updateModifyDate( modifyDate );
	    }
	       
	    // Three cases:
	    //	(a) no slashes, just use the string
	    //  (b) trailing slash
  	    //  (c) no trailing slash

	    if ( strrchr( fileName, '/' ) == NULL ) { 
	        strcpy( configPath, fileName );
	    } else if ( fileName[strlen(fileName) - 1] == '/' ) {
	        strcpy( tmpText, fileName );
	        tmpText[strlen(fileName) - 1] = '\0';
	        configText = strrchr( tmpText, '/' );
	        configText++;
	        i = 0;
	        while ( (configPath[i] = *configText++) != '\0' ) {
	           i++;
		}
	    } else {
                configText = strrchr( fileName, '/' );
                configText++;
                i = 0;
                while ( (configPath[i] = *configText++) != '\0' ) {
                    i++;
	        }
	    }

 	    configMsg = XmStringCreateLtoR( configPath, 
				            XmSTRING_DEFAULT_CHARSET );

	} else {
	    _updateModifyDate( NULL );		// uses current time

 	    configMsg = XmStringCreateLtoR( (char *)fileName,
                                             XmSTRING_DEFAULT_CHARSET );
	}

        XtVaSetValues( configValue, XmNlabelString, (XtArgVal)configMsg, NULL );
	XmStringFree( configMsg );
}

void
PabloMainInterface::_desensitizeMenuOptions()
{
	/*
	 * The run option is also desensitized when the run panel is
	 * active so we manage it "separately". 
	 */
	_desensitizeRunOption();

        XtVaSetValues( fileWidget, XmNsensitive, False, NULL );
        XtVaSetValues( configureWidget, XmNsensitive, False, NULL );
        XtVaSetValues( moduleWidget, XmNsensitive, False, NULL );
        XtVaSetValues( helpOnModuleInstanceWidget, XmNsensitive, False, NULL );
	_desensitizeCnt++;
}

void
PabloMainInterface::_sensitizeMenuOptions()
{
	/*
	 * The run option is also desensitized when the run panel is
	 * active so we manage it "separately". 
	 */
	_sensitizeRunOption();

	if ( --_desensitizeCnt == 0 ) {
            XtVaSetValues( fileWidget, XmNsensitive, True, NULL );
            XtVaSetValues( configureWidget, XmNsensitive, True, NULL );
            XtVaSetValues( moduleWidget, XmNsensitive, True, NULL );
            XtVaSetValues( helpOnModuleInstanceWidget, XmNsensitive, 
								True, NULL );
	}
}

void
PabloMainInterface::_desensitizeRunOption()
{
	XtVaSetValues( runWidget, XmNsensitive, False, NULL );
	_desensitizeRunCnt++;
}

void
PabloMainInterface::_sensitizeRunOption()
{
	if ( --_desensitizeRunCnt == 0 ) {
	    XtVaSetValues( runWidget, XmNsensitive, True, NULL );
	}
}

void
PabloMainInterface::_updateModifyDate( const char *modifyDate )
{
	const char *theDate;

	if ( modifyDate == NULL ) {
	    time_t now = time(0);
	    char *dateNow = ctime( &now );
	    dateNow[24] = '\0';
	    theDate = dateNow;
	} else {
	    theDate = modifyDate;
	}

	XmString dateMessage = XmStringCreateLtoR( (char *) theDate, 
						   XmSTRING_DEFAULT_CHARSET );
        XtVaSetValues( dateValue, XmNlabelString, (XtArgVal)dateMessage, NULL );
        XmStringFree( dateMessage );
}

Boolean_
PabloMainInterface::_setState( InterfaceState newState )
{
	static char messageBuffer[200];
#ifdef DEBUG_MI	
	   cout << form( "got _setState %d %s old state was %d %s\n",
			  newState, InterfaceStateNames[newState],
			  state, InterfaceStateNames[state] );
#endif DEBUG_MI

        if ( StateTransition[state][newState] == FAILURE_ ) {
            error( "Invalid selection:\n Can't change to %s from %s",
		     InterfaceStateNames[newState], 
		     InterfaceStateNames[state] );
	    return FAILURE_;
	}

	if ( ( newState == Configuring ) )  {
	    if ( moduleNowBeingConfigured ) {
	        sprintf( messageBuffer, "Configuring module %s", 
	      		   (const char *)moduleNowBeingConfigured->getName() );
	    } else {
	        sprintf( messageBuffer, "Configuring module <NO_MODULE>" );
	    }
	    XmString stateMessage = XmStringCreateLtoR( messageBuffer,
                                                    XmSTRING_DEFAULT_CHARSET );
            XtVaSetValues( stateValue, XmNlabelString, 
				               (XtArgVal) stateMessage, NULL );
            XmStringFree( stateMessage );

	} else if ( newState == WaitingForSingleStepContinue )  {
	    sprintf( messageBuffer, "Waiting to run %s", 
	      		       (const char *)moduleAboutToRun->getName() );
	    XmString stateMessage = XmStringCreateLtoR( messageBuffer,
                                                   XmSTRING_DEFAULT_CHARSET );
            XtVaSetValues( stateValue, XmNlabelString, 
				      (XtArgVal) stateMessage, NULL );
            XmStringFree( stateMessage );

	} else if ( newState == RunningSingleModule )  {
	    sprintf( messageBuffer, "Running module %s", 
	      		       (const char *)moduleAboutToRun->getName() );
	    XmString stateMessage = XmStringCreateLtoR( messageBuffer,
                                                   XmSTRING_DEFAULT_CHARSET );
            XtVaSetValues( stateValue, XmNlabelString, 
				      (XtArgVal) stateMessage, NULL );
            XmStringFree( stateMessage );

	} else if ( newState == Quiet ) {

	   if ( executionDone == FALSE_ ) {
	       XtVaSetValues( stateValue, XmNlabelString,
                          (XtArgVal) InterfaceStateStrings[newState], NULL );
	   } else {
	       XtVaSetValues( stateValue, XmNlabelString,
                          (XtArgVal) InterfaceStateStrings[DoneRunning], NULL );
	   }

	} else {

	   XtVaSetValues( stateValue, XmNlabelString,
                         (XtArgVal) InterfaceStateStrings[newState], NULL );

	}

	state = newState;

	XmUpdateDisplay( topLevel );

        XEvent event;
        while ( XtAppPending( appContext ) ) {
           XtAppNextEvent( appContext, &event );
           XtDispatchEvent( &event );
        }

	return SUCCESS_;
}
	
/*********************** A C T I O N   M E T H O D S ***********************/

void 
PabloMainInterface::run()
{
        Pablo::HelpSystem()->setRootHelpDir(pabloData.defaultHelpFileDir);

	autoLoadConfiguration();

	XtAppMainLoop( appContext );
}

/*************** Manage a single File Selection Dialog ***********************/

Boolean_
PabloMainInterface::reserveFSD()
{
	if ( FSD_busy ) {
	    warning( "File Selection already in progress. " );
	    return FAILURE_;
	} 

	FSD_busy = TRUE_;

	if ( FSD == NULL ) {		// first time, so create the dialog
	    FSD = new FileSelectionDialog();
	} 

	return SUCCESS_;
}

void
PabloMainInterface::releaseFSD()
{
	Assert( FSD_busy );

	FSD_busy = FALSE_;
}

/******** Helper routines not directly associated with menu items  ************/

void
PabloMainInterface::checkpoint()
{
 	warning( "Pablo checkpointing is not yet implemented" );
}

Boolean_
PabloMainInterface::isCameraOn()
{
        return( runPanelDialog->isCameraOn() );
}

void
PabloMainInterface::takeSnapshot( Widget subjectWidget, const char *fuClass,
				  const char *fuName )
{
        runPanelDialog->takeSnapshot( subjectWidget, fuClass, fuName );
}


void 
PabloMainInterface:: updateConsoleLog()
{
	if ( pabloData.consoleLog == NULL || *pabloData.consoleLog == 0 ) {
	   if ( consoleLogFP != NULL ) { // close existing log if it is open	
#ifdef DEBUG_MI	
	      cout << "Closing console log " << consoleLogFilename << NL;
#endif DEBUG_MI
	      fclose( consoleLogFP );
	      consoleLogFP = NULL;
	   }
	} else {
	   if (consoleLogFP != NULL) {
	      if ( consoleLogFilename != pabloData.consoleLog ) {
		 //
		 // Close the log if it is open to a different
		 // file than the new one
		 // 
#ifdef DEBUG_MI	
	         cout << "Closing console log ";
		 cout << consoleLogFilename << NL;
		 cout << "in order to open new log ";
		 cout << pabloData.consoleLog << NL;
#endif DEBUG_MI
		 fclose( consoleLogFP );
	         PabloTranscriptBuf.setLogFile( NULL );
	         PabloTranscriptErrorBuf.setLogFile( NULL );
	     } else {
		 //
		 // If the currently opened log is the same file, just
		 // return
		 //
#ifdef DEBUG_MI	
		 cout << "Do not reopen console logs of the same name\n";
#endif DEBUG_MI
		 return;
	     }
	   }

	   //
	   // If we are here, we are opening a new log, and the old
	   // log is now closed.
	   // 
	   consoleLogFilename = pabloData.consoleLog;
	   consoleLogFP = fopen( consoleLogFilename.getValue(), "w" );

	   if ( consoleLogFP == NULL ) {
	      error( "Could not open execution logfile %s: %s. \n \
		     Setting console logfile to /dev/tty.",
		     consoleLogFilename.getValue(), errorString() );
	      pabloData.consoleLog = "/dev/tty";
	      consoleLogFilename = pabloData.consoleLog;
	      consoleLogFP = fopen( consoleLogFilename.getValue(), "w" );
	   } 
#ifdef DEBUG_MI	
	   cout << "Opened console log " << consoleLogFilename << NL;
#endif DEBUG_MI
	}

	PabloTranscriptBuf.setLogFile( consoleLogFP );
	PabloTranscriptErrorBuf.setLogFile( consoleLogFP );
}

void
PabloMainInterface::addSignalWidget( Widget inputSignalWidget ) 
{
        signalWidgetList.addElement( inputSignalWidget );
}


/******************* ACTIONS FOR MENU ITEMS  *********************************/

//--------- File ----------

void
PabloMainInterface::deleteConfig( Boolean_ calledFromCallback )
{
	if ( graphDisplay->nodeCount() == 0 ) {
      	    error( "No graph modules to delete." );
	    return;
	}

	if ( moduleNowBeingConfigured != NULL ) {
	    error( "Cannot delete graph while configuration is in progress." );
	    return;
	}

	if ( calledFromCallback && pabloData.verifyDeleteGraph ) {
	    if ( ! Pablo::GeneralDialog()->runQuestion( 
					   "Really Delete Graph?" ) ) {
		return;
	    }
	}

	runStop();
	Pablo::IM()->reset();
	graphDisplay->reset();
	Pablo::TagMappingDict()->clearAllEntries();
	Pablo::StructureDict()->reset();
	if ( addModuleDialog != NULL ) {
	    addModuleDialog->resetUniqueNums();
	}
	_updateConfigFilename( "NULL Configuration" );
	executionDone = FALSE_;
	_setState( Quiet );
}

void 
PabloMainInterface::layoutLoadFromFile()
{
	if ( graphDisplay->nodeCount() != 0 ) {
	    error( "Can't load layout when modules already exist." );
	    return; 
	}

	if ( !reserveFSD() ) {
	    return;
	}

	//
	// Set the title.
	// Set our mask to match any files in the graph configuration
	// directory. Restrict to display only files.
	//

	FSD->setXmStringResource( XmNdialogTitle, "Load Layout from File" );
	CString dirMask = pabloData.defaultConfigFileDir;
	dirMask = dirMask + "/*";

	FSD->setXmStringResource( XmNdirMask, (char *)dirMask.getValue() );
	FSD->setResource( XmNfileTypeMask, (XtArgVal) XmFILE_REGULAR );

	Boolean_ done = FALSE_;

	while ( !done ) {
	    if ( FSD->run() ) {			// Okay button
	        const CString& file = FSD->getFileName();
		if ( _readLayoutFromFile( file ) ) {
		    done = TRUE_;
                    _updateConfigFilename( file.getValue() );
		} else {
		    if ( graphDisplay->nodeCount() != 0 ) {
			//
			// If part of the layout has already been displayed,
			// we want to clear it.  I can't get the display to 
			// update (as requested in deleteConfig() unless I 
			// exit the callback so I set done to TRUE_, even
			// though I really want to let the user reselect
			// from the FSD.	-Ruth 5/93
			// 
		        deleteConfig();
			done = TRUE_;		
		    }
		}
            } else {				// Cancel button
	        done = TRUE_;
	    }
	}

	releaseFSD();
}

void 
PabloMainInterface::configLoadFromDir()
{
	if ( graphDisplay->nodeCount() != 0 ) {
	    error( "Can't load configuration when modules already exist." );
	    return; 
	}

	if ( !reserveFSD() ) {
 	    return;
	}

	//
	// Set the title.
	// Set our mask to match any directories in the configuration
	// directory.  Restrict to display only directories.
	//

	FSD->setXmStringResource( XmNdialogTitle, 
				  "Load Configuration from Directory" );

	CString dirMask = pabloData.defaultConfigFileDir;
	dirMask = dirMask + "/*";

	FSD->setXmStringResource( XmNdirMask, (char *)dirMask.getValue() );
	FSD->setResource( XmNfileTypeMask, (XtArgVal) XmFILE_DIRECTORY );

	Boolean_ done = FALSE_;

	while ( !done ) {
	    if ( FSD->run() ) {			// Okay button
	        CString file = FSD->getFileName();
	        if ( _loadConfigurationFromDir( file ) ) {
		    done = TRUE_;
	            _updateConfigFilename( file.getValue() );
		} else {
		    if ( graphDisplay->nodeCount() != 0 ) {
   	                //
                        // If part of the layout has already been displayed,
                        // we want to clear it.  I can't get the display to 
                        // update (as requested in deleteConfig() unless I 
                        // exit the callback so I set done to TRUE_, even
                        // though I really want to let the user reselect
                        // from the FSD.        -Ruth 5/93
                        // 
		        deleteConfig();
			done = TRUE_;		
		    }
		}
	    } else {				// Cancel button
	        done = TRUE_;
	    }
	}

	releaseFSD();
}

void 
PabloMainInterface::layoutSaveToFile()
{
	if ( graphDisplay->nodeCount() == 0 ) {
	    error( "No modules to be saved." );
	    return;
	}

	if ( !reserveFSD() ) {
	    return;
	}

	//
	// Set the title.
	// Set our mask to match any files in the configuration
	// directory.  Restrict to display only files.
	//

	FSD->setXmStringResource( XmNdialogTitle, "Save Layout to File" );

	CString dirMask = pabloData.defaultConfigFileDir;
	dirMask = dirMask + "/*";

	FSD->setXmStringResource( XmNdirMask, (char *)dirMask.getValue() );
	FSD->setResource( XmNfileTypeMask, (XtArgVal) XmFILE_REGULAR );
	
	Boolean_ done = FALSE_;

	while ( !done ) {
	    if ( FSD->run() ) {			// Okay button
		const char *filename = FSD->getFileName().getValue();
	        FILE *fp = fopen( filename, "w" );

		if ( fp == NULL ) {
	    	   error( "Cannot open file - %s", filename );
		} else {
		   _updateConfigFilename( filename );
	           Pablo::IM()->writeLayoutToFP( fp );
		   graphDisplay->writeLayoutToFP( fp );
		   fclose( fp );
		   done = TRUE_;
		}
	    } else  {				// "Cancel" was pressed
		done = TRUE_;
	    }
	}

	releaseFSD();
}

void 
PabloMainInterface::configSaveToDir()
{
	if ( graphDisplay->nodeCount() == 0 ) {
	    error( "No modules to be saved." );
	    return;
	}

	if ( ! Pablo::IM()->isConfigured() ) {
	    error( "Graph must be configured before it can be saved." );
	    return;
	}

	if ( !reserveFSD() ) {
	    return;
	}

	//
	// Set the title.
	// Set our mask to match any directories in the configuration
	// directory.  Restrict to display only directory.
	//

	FSD->setXmStringResource( XmNdialogTitle, 
				  "Save Configuration to Directory" );

	CString dirMask = pabloData.defaultConfigFileDir;
	dirMask = dirMask + "/*";

	FSD->setXmStringResource( XmNdirMask, (char *)dirMask.getValue() );
	FSD->setResource( XmNfileTypeMask, (XtArgVal) XmFILE_DIRECTORY );

	Boolean_ done = FALSE_;

	while ( !done ) {
	    if ( FSD->run() ) {			// Okay button
	        CString file = FSD->getFileName();
	        if ( _saveConfigurationToDir( file ) ) {
                    _updateConfigFilename( file.getValue() );
		    done = TRUE_;
		}
	    } else {				// Cancel button
	        done = TRUE_;
	    }
	}
	releaseFSD();
}

void 
PabloMainInterface::restartExecution()
{
	if ( graphDisplay->nodeCount() == 0 ) {
	    error( "No modules to be run." );
	    return;
	}

	if ( ! Pablo::IM()->isConfigured() ) {
	    error( "Graph must be configured before it can be run." );
	    return;
	}

	runStop();
	Pablo::IM()->restart();
	executionDone = FALSE_;
	_setState( Quiet );

}

void 
PabloMainInterface::openTranscript()
{
	Pablo::Transcript()->openWindow();
}

void
PabloMainInterface::fileQuit()
{
	delete Pablo::IM();
	exit(0);
}

//--------- Run ----------

void 
PabloMainInterface::runPanel()
{
        /*
         * Come here when "RunPanel" is selected from the Run pull-down menu.
         * The steps are:
         * 1) If configuration in process, don't allow run panel.
	 * 2) Stop execution if in progress.  
	 * 3) If unable to set state to idle, don't allow run panel.
         * 4) Disable Run pull-down menu to avoid "2 masters" interaction 
         *    problems.
         * 5) Create the Run Panel Dialog if it doesn't already exist.
	 * 6) Raise the Run Panel Dialog.
	 */

	if ( moduleNowBeingConfigured != NULL ) {
	    error( CompleteConfigMsg, "selecting Run Panel" );
	    return;
	}

	runStop();

	if ( _setState( Quiet ) == FAILURE_ ) {
	    return;
	}

	_desensitizeRunOption();

	if ( runPanelDialog == NULL ) {		
	    runPanelDialog = new RunDialog( this );
	} 

	runPanelDialog->raiseDialog();
}

void
PabloMainInterface::runPanelDismiss()
{
	/*
	 * Come here when the Run Panel is dismissed.  
	 * We re-enable the Run pull-down menu (unless it's been
	 * disabled for some other reason).
	 */
	_sensitizeRunOption();
}

void 
PabloMainInterface::runGraphDiscrete( int stepCount )
{
	if ( executionDone ) {
	    return;
	}

	if ( moduleNowBeingConfigured != NULL ) {
	    error( CompleteConfigMsg, "executing graph" );
	    return;
	}

	if ( _setState( DiscreteExecution ) == FAILURE_ ) {
	    return;
	} else {
	    runStop();
            executionDone = CnvToBoolean_( !( Pablo::IM()->run(stepCount) ) );
	    _setState( Quiet );
	}
}

void 
PabloMainInterface::runRun()
{
	if ( executionDone ) {
 	    return;
	}

	if ( moduleNowBeingConfigured != NULL ) {
	    error( CompleteConfigMsg, "executing graph" );
	    return;
	}

	if ( _setState( Running ) == FALSE_ ) {
	    return;
	}

	XEvent event;
	running = TRUE_;
	int runStepSize = pabloData.runStepSize;

	while ( running && !executionDone ) {
	    executionDone = CnvToBoolean_( !( Pablo::IM()->run(runStepSize) ) );

	    while ( XtAppPending( appContext ) ) {
		XtAppNextEvent( appContext, &event);
		XtDispatchEvent( &event );
	    }

	    if ( state != Running ) {	 // something else may have happened
		if ( _setState( Running ) == FAILURE_ ) {
		    running = FALSE_;
		}
	    }
	}
	
	if ( executionDone ) {
 	    int i;
	    for ( i=0; i<signalWidgetList.count(); i++ ) {
                XtCallCallbacks( (Widget)signalWidgetList.getElement(i), 
				 XmNactivateCallback, (XtPointer)NULL );
	    }
	} 

	_setState( Quiet );
	running = FALSE_;
}

void 
PabloMainInterface::runStop()
{
	running = FALSE_;
}

Boolean_
PabloMainInterface::runModule()
{
       	/*
         * We get to this point in two ways:  
	 * 
         * 1) If the Run Panel execution mode is "Module by Module"
         * and the RUN button is pushed. ( state == Quiet )
	 * We disable the File, Configure, and Module items on the main
	 * menu so that our state can't get disrupted & then begin running 
	 * the graph once by calling IM()->run( 1, TRUE_).   After we return
	 * from that call, we reenable the File, Configure, and Module items.
	 *
         * 2) If we are in the middle of executing a graph (haven't yet
	 * returned from the call to IM()->run() above), and the Run Panel 
	 * RUN button is pushed.  ( state == WaitingForSingleStepContinue )  
	 * We change the state to RunningSingleModule which is recognized in 
	 * the waitForStepContinue() method & it returns to IM->run() to 
         * execute the next module.  
	 * 
         * We return TRUE_ when the entire graph has been executed.
         */

	if ( executionDone ) {
	    return TRUE_;
	}

	if ( moduleNowBeingConfigured != NULL ) {
	    error( CompleteConfigMsg, "executing modules" );
	    return TRUE_;
	}

	Boolean_ graphDone = FALSE_;

	if ( state == Quiet ) {
            XtVaSetValues( fileWidget, XmNsensitive, False, NULL );
            XtVaSetValues( configureWidget, XmNsensitive, False, NULL );
            XtVaSetValues( moduleWidget, XmNsensitive, False, NULL );

	    executionDone = CnvToBoolean_( !( Pablo::IM()->run( 1, TRUE_ ) ) );
            _setState( Quiet );

            XtVaSetValues( fileWidget, XmNsensitive, True, NULL );
            XtVaSetValues( configureWidget, XmNsensitive, True, NULL );
            XtVaSetValues( moduleWidget, XmNsensitive, True, NULL );

	    graphDone = TRUE_;

	} else if ( state == WaitingForSingleStepContinue ) {
	    _setState( RunningSingleModule );

	} else {
	    warning( "Unexpected state in runModule: %s\n", 
		     InterfaceStateNames[state] );

	}
	return graphDone;

}

void 
PabloMainInterface::waitForStepContinue( ModuleId *modId )
{
	if ( ( state != Quiet ) && ( state != RunningSingleModule ) ){
	    warning( "Unexpected state in waitForStepContinue: %s\n",
		     InterfaceStateNames[state] );
	}

	moduleAboutToRun = modId;
	_setState( WaitingForSingleStepContinue );

	XEvent event;
	while ( state == WaitingForSingleStepContinue ) {
	    XtAppNextEvent( appContext, &event );
	    XtDispatchEvent( &event );
	}

	if ( state != RunningSingleModule ) {
	    warning( "waitForStepContinue: EXPECTED Running; GOT %s\n", 
			   InterfaceStateNames[state] );
	}
}

//--------- Configure ----------

void
PabloMainInterface::graphConfigure()
{
	if ( graphDisplay->nodeCount() == 0 ) {
	    error( "No modules to configure. " );
	    return;
	}

	if ( moduleNowBeingConfigured != NULL ) {
	    error( "Configuration already in progress. ");
	    return;
	}

	if ( state != Quiet ) {
	    error( "Status must be Idle before beginning graph configuration.");
	    return;
	}

	Pablo::IM()->configure();
	_updateModifyDate( NULL );
}

void
PabloMainInterface::moduleConfigure( Boolean_ fuParamsOnly )
{
	if ( graphDisplay->nodeCount() == 0 ) {
	    error( "No modules to configure." );
	    return;
	}

	if ( moduleNowBeingConfigured != NULL ) {
	    error( "Configuration already in progress. ");
	    return;
	}

	if ( !Pablo::IM()->isConfigured() ) {
	    error( "Graph must be configured before individual modules" );
	    return;
	}

	if ( _setState( WaitingForPressOnNode ) == FALSE_ ) {
	    return;
	}

	_desensitizeMenuOptions();
	pressedNode = -1;

	XEvent event;
	while ( state != Quiet ) {
	    XtAppNextEvent( appContext, &event );
	    XtDispatchEvent( &event );
	}

	if ( pressedNode == -1 ) {
	    _sensitizeMenuOptions();
	    return;
	}

	ModuleId *module = Pablo::IM()->lookupModule( pressedNode );
	if ( module == NULL ) {
	    error( "configureModule: Got module == NULL" );
	    _sensitizeMenuOptions();
	    return;
	}

	// We allow user to get help on Module Instance at this point
        XtVaSetValues( helpOnModuleInstanceWidget, XmNsensitive, True, NULL );

        moduleNowBeingConfigured = module;
	Wrapper *wrapper = module->getWrapper();

	if ( wrapper ) {
	    if ( _setState( Configuring ) ) { 
		if ( fuParamsOnly ) {
		    wrapper->configureParams( );
		} else {
		    wrapper->configure( TRUE_ );
		}
                _updateModifyDate( NULL );
	        _setState( Quiet );
	    }
	}

	moduleNowBeingConfigured = NULL;
	_sensitizeMenuOptions();
}

void 
PabloMainInterface::defaultsConfigure()
{
	if ( defaultsDialog == NULL ) {
	   defaultsDialog = new DefaultsDialog( this );
	}

	defaultsDialog->raiseDialog();
}

void
PabloMainInterface::defaultsConfigureOK()
{
	updateConsoleLog();
}

void 
PabloMainInterface::beginConfigureOf( ModuleId *id )
{
	if ( moduleNowBeingConfigured != NULL ) {
	    error( "Configuration already in progress..." );
	    return;
	}
	moduleNowBeingConfigured = id;
	_setState( Configuring );
}

void 
PabloMainInterface::endConfigureOf( ModuleId *id )
{
	if ( moduleNowBeingConfigured != id ) {
	    warning( "endConfigureOf(): Unexpected Module mismatch." );
	}
	moduleNowBeingConfigured = NULL;
	_updateModifyDate( NULL );
	_setState( Quiet );
}

//--------- Module ----------

void
PabloMainInterface::moduleAdd()
{
	if ( addModuleDialog == NULL ){
	    addModuleDialog = new AddModuleDialog( this );
	}
	addModuleDialog->raiseDialog();
}

void 
PabloMainInterface::moduleAddRequest( const CString& fuClass, 
				      const CString& fuName )
{
	/* 
	 * This is called by the addModuleDialog when the "Add" button
	 * is selected to add a new module.
	 */

	if ( graphDisplay->containsNode( fuName ) ) {
	    error( "Module with name %s already exists.", fuName.getValue() );
	    return;
	}

	int mIdx;

	if ( fuClass == "FileInput" ) {
	    mIdx = Pablo::IM()->addFileInputModule( fuName, CString::NOMATCH );
	} else if ( fuClass == "FileOutput" ) {
	    mIdx = Pablo::IM()->addFileOutputModule( fuName, CString::NOMATCH );
	} else {
	    mIdx = Pablo::IM()->addModule( fuClass, fuName );
	}

	if ( mIdx == -1 ) {
    	    warning( "Could not create Functional Unit %s for module %s",
		   fuClass.getValue(), fuName.getValue() );
	} else {
	    int nIdx = graphDisplay->addNode( fuName ); 
	    Assert( nIdx == mIdx );
	}
}

void
PabloMainInterface::moduleConnect()
{
	if ( graphDisplay->nodeCount() < 2 ) {
	    error( "There are fewer than two modules - connection impossible.");
	    return;
	}

	if ( moduleNowBeingConfigured != NULL ) {
	    error( CompleteConfigMsg, "adding more connections" );
	    return;
	}

	_desensitizeMenuOptions();
	XEvent event;
	do {
	    connectFrom = connectTo = -1;
	    if ( _setState( WaitingForFirstInConnection ) == FAILURE_ ) {
	        _sensitizeMenuOptions();
		return;
	    }
	    
	    while ( (state != Quiet) && ( state != UpdateLink ) ) {
	        XtAppNextEvent( appContext, &event );
	        XtDispatchEvent( &event );
	    }

	    if ( state == UpdateLink ) {
		if ( Pablo::IM()->addConnection( connectFrom, connectTo ) ) {
		    graphDisplay->addEdge( connectFrom, connectTo );
	        }
	    }
	 } while ( state != Quiet );
	 _sensitizeMenuOptions();
}

void 
PabloMainInterface::moduleDelete()
{
	if ( graphDisplay->nodeCount() == 0 ) {
	    error( "No modules to delete." );
	    return;
	}

	if ( moduleNowBeingConfigured != NULL ) {
	    error( "Cannot delete module while configuration is in progress." );
	    return;
	}

	InterfaceState saveState = state;

	if ( _setState( WaitingForPressOnNode ) == FAILURE_ ) {
	    return;
	}

	_desensitizeMenuOptions();
	pressedNode = -1;

	XEvent	event;
	while ( state != Quiet ) {
	    XtAppNextEvent( appContext, &event );
	    XtDispatchEvent( &event );
	}

	_setState( saveState );
	_sensitizeMenuOptions();

	if ( pressedNode == -1 ) {
	    return;
	}

	ModuleId *module = Pablo::IM()->lookupModule( pressedNode );
	if ( module == NULL ) {
	    error( "moduleDelete: Got module == NULL" );
	    return;
	}

	/*
	 * If this is our last module, basically call deleteConfig()
	 * If it isn't...
	 *  The IM->deleteModule() will fail if this module has any
	 *   output pipes associated with it.  
	 *  A successful IM->deleteModule() removes the specified module
	 *   and any input pipes (connections) to the module.
	 *  The graphDisplay->deleteNode() removes the node bubble and
	 *   any lines associated with incoming edges.
	 *  If we've just deleted our last module, then reset the state
	 *   of the Infrastructure Manager, the TagMappingDictionary, and
	 *   the Global StructureDictionary so that are just like when we
	 *   started up.
	 */

	if ( graphDisplay->nodeCount() == 1 ) {
	    deleteConfig();
	} else {
	    if ( Pablo::IM()->deleteModule( pressedNode ) ) {
	    	graphDisplay->deleteNode( pressedNode );
	    }
	} 
}

//--------- Help ----------

void 
PabloMainInterface::helpOnTopic( char* topic )
{
  	Pablo::HelpSystem()->giveHelpOn( topic );
}

void
PabloMainInterface::moduleHelp()
{
	if ( graphDisplay->nodeCount() == 0 ) {
	    error( "A module must exist before this option can be invoked" );
	    return;
	}

	if ( ( state == WaitingForFirstInConnection ) || 
	     ( state == WaitingForSecondInConnection ) ||
	     ( state == WaitingForPressOnNode ) ) {
	    error( "Cannot select help on Module Instance while Module \
		    selection in progress." );
	    return;
	}

        InterfaceState saveState = state;

        if ( _setState( WaitingForPressOnNode ) == FAILURE_ ) {
            return;
        }

	_desensitizeMenuOptions();
        pressedNode = -1;

        XEvent  event;
        while ( state != Quiet ) {
            XtAppNextEvent( appContext, &event );
            XtDispatchEvent( &event );
        }

        _setState( saveState );
	_sensitizeMenuOptions();

	// Make sure Help On Module Instance is again sensitized 
	XtVaSetValues( helpOnModuleInstanceWidget, XmNsensitive, True, NULL );
	
	if ( pressedNode == -1 ) {
	    return;
	}

	ModuleId *module = Pablo::IM()->lookupModule( pressedNode );
	if ( module == NULL ) {
	    error( "moduleHelp: Got node == NULL" );
	    return;
	}

	CString fuClass = module->getFuClassName();
	Pablo::HelpSystem()->giveHelpOn( fuClass.getValue() );
}

//--------- Debug ---------

void 
PabloMainInterface::debugDumpState()
{
	cout << "----------- Infrastructure Manager Begin ------------------\n";
	cout << *Pablo::IM() << NL;
	cout << "------------ Infrastructure Manager End -------------------\n";
}

void 
PabloMainInterface::showCompiledOptions()
{
	extern CString getCompiledOptions();
	CString opts = getCompiledOptions();

	info( opts.getValue() );
}

/******************* STATIC CALLBACKS FOR MENU ITEMS   ************************/


//--------- File ----------

void 
PabloMainInterface::deleteConfigCallback ( Widget /* w */,
                                           XtPointer clientData,
                                           XtPointer /* callData */ )
{
        PabloMainInterface *obj = (PabloMainInterface *) clientData;
        obj->deleteConfig( TRUE_ );
}

void 
PabloMainInterface::layoutLoadCallback ( Widget /* w */,
			                 XtPointer clientData,
				         XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->layoutLoadFromFile();
}

void 
PabloMainInterface::configLoadCallback ( Widget /* w */,
				         XtPointer clientData,
				         XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->configLoadFromDir();
}

void 
PabloMainInterface::layoutSaveCallback ( Widget /* w */,
				         XtPointer clientData,
				         XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->layoutSaveToFile();
}

void 
PabloMainInterface::configSaveCallback ( Widget /* w */,
				          XtPointer clientData,
				          XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->configSaveToDir();
}

void 
PabloMainInterface::restartCallback ( Widget /* w */,
				      XtPointer clientData,
				      XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->restartExecution();
}

void 
PabloMainInterface::transcriptCallback ( Widget /* w */,
                                         XtPointer clientData,
                                         XtPointer /* callData */ )
{
        PabloMainInterface *obj = (PabloMainInterface *) clientData;
        obj->openTranscript();
}

void 
PabloMainInterface::quitCallback( Widget /* w */,
				  XtPointer clientData,
				  XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->fileQuit();
}


//--------- Run ----------

void 
PabloMainInterface::panelCallback( Widget /* w */,
				   XtPointer clientData,
				   XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->runPanel();
}

void 
PabloMainInterface::singleStepCallback( Widget /* w */,
				        XtPointer clientData,
				        XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->runGraphDiscrete( 1 );
}

void 
PabloMainInterface::runCallback( Widget /* w */,
			         XtPointer clientData,
				 XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->runRun();
}

void 
PabloMainInterface::stopCallback( Widget /* w */,
				  XtPointer clientData,
				  XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->runStop();
}

//--------- Configure ----------

void 
PabloMainInterface::graphConfigureCallback( Widget /* w */,
				            XtPointer clientData,
				            XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->graphConfigure();
}

void 
PabloMainInterface::moduleConfigureCallback( Widget /* w */,
				             XtPointer clientData,
				             XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->moduleConfigure( FALSE_ );		// bindings & fu parameters
}

void 
PabloMainInterface::parameterConfigureCallback( Widget /* w */,
				                XtPointer clientData,
				                XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->moduleConfigure( TRUE_ );		// fu parameters only
}

void 
PabloMainInterface::defaultsConfigureCallback( Widget /* w */,
                                               XtPointer clientData,
                                               XtPointer /* callData */ )
{
        PabloMainInterface *obj = (PabloMainInterface *) clientData;
        obj->defaultsConfigure();
}

//--------- Module --------

void 
PabloMainInterface::addModuleCallback( Widget /* w */,
				       XtPointer clientData,
				       XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->moduleAdd();
}

void 
PabloMainInterface::connectModuleCallback( Widget /* w */,
                                           XtPointer clientData,
                                           XtPointer /* callData */ )
{
        PabloMainInterface *obj = (PabloMainInterface *) clientData;
        obj->moduleConnect();
}

void 
PabloMainInterface::deleteModuleCallback( Widget /* w */,
                                          XtPointer clientData,
                                          XtPointer /* callData */ )
{
        PabloMainInterface *obj = (PabloMainInterface *) clientData;
        obj->moduleDelete();
}

//--------- Help --------

void 
PabloMainInterface::helpOnFileCallback ( Widget /* w */,
				         XtPointer clientData,
				         XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->helpOnTopic( "FileMenu" );
}

void
PabloMainInterface::helpOnRunCallback ( Widget /* w */,
			                XtPointer clientData,
				        XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->helpOnTopic( "RunMenu" );
}

void 
PabloMainInterface::helpOnConfigureCallback ( Widget /* w */,
				              XtPointer clientData,
				              XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->helpOnTopic( "ConfigureMenu" );
}

void 
PabloMainInterface::helpOnModuleCallback ( Widget /* w */,
				           XtPointer clientData,
				           XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->helpOnTopic( "ModuleMenu" );
}

void 
PabloMainInterface::helpOnInstanceCallback ( Widget /* w */,
				             XtPointer clientData,
				             XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->moduleHelp();
}

void 
PabloMainInterface::helpOnDebugCallback( Widget /* w */,
				         XtPointer clientData,
				         XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->helpOnTopic( "DebugMenu" );
}

void 
PabloMainInterface::helpOnHelpCallback( Widget /* w */,
				        XtPointer clientData,
				        XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->helpOnTopic( "HelpOnHelp" );
}

void 
PabloMainInterface::helpOnPabloCallback( Widget /* w */,
				         XtPointer clientData,
				         XtPointer /* callData */ )
{
	PabloMainInterface *obj = (PabloMainInterface *) clientData;
    	obj->helpOnTopic( "GeneralInfo" );
}

//--------- Debug -------

void
PabloMainInterface::debugDumpStateCallback ( Widget /* w */,
                                             XtPointer clientData,
                                             XtPointer /* callData */ )
{
        PabloMainInterface *obj = (PabloMainInterface *) clientData;
        obj->debugDumpState();
}

void
PabloMainInterface::debugCompiledOptionsCallback ( Widget /* w */,
                                                   XtPointer clientData,
                                                   XtPointer /* callData */ )
{
        PabloMainInterface *obj = (PabloMainInterface *) clientData;
        obj->showCompiledOptions();
}

/********** ROUTINES FOR HANDLING CLICK ON GRAPH NODE ***********************/

void
PabloMainInterface::nodeCallback( GraphNode *node, XEvent *event )
{
	/* 
	 * We get here via the GraphNode event handler
	 */

	if ( event->type != ButtonPress ) {
	    warning( "Not a button press in nodeCallback" );
	    return;
	}

	switch ( event->xbutton.button ) {
	   case 1:
	      _buttonOnePress( node, event );
	      break;

	   case 3:
	      _buttonThreePress( node, event );
	      break;
	}

}

void
PabloMainInterface::_buttonOnePress( GraphNode *node, XEvent * /* event */ )
{
	int nodeIndex = graphDisplay->getNodeIndex( node );

	if ( nodeIndex == -1 ) {
	    warning( "Invalid node selected - try again" );
	} else {
	    switch ( state ) {
	      case WaitingForFirstInConnection:
	       	  connectFrom = nodeIndex;
		  _setState( WaitingForSecondInConnection );
		  break;

	      case WaitingForSecondInConnection:
		  connectTo = nodeIndex;
		  _setState( UpdateLink );
		  break;

	      case WaitingForPressOnNode:
		  pressedNode = nodeIndex;
		  _setState( Quiet );
		  break;

	      default:
		  break;
	    }
	}
}

void
PabloMainInterface::_buttonThreePress( GraphNode * /* node */, 
				       XEvent * /* event */ )
{
	/*
	 * This is a way out of node selection
	 */

	switch ( state ) {
	    case WaitingForFirstInConnection:
	    case WaitingForSecondInConnection:
	    case WaitingForPressOnNode:
		_setState( Quiet );
		break;

	    default:
		break;
	}
        return;
}

/**** HELPER ROUTINES TO SAVE AND LOAD LAYOUTS AND CONFIGURATIONS ********/

Boolean_ 
PabloMainInterface::_loadConfigurationFromDir( const CString& dir )
{
	const char *dirfile = dir.getValue();
	struct stat statbuf;

	if ( stat( (char *)dirfile, &statbuf ) == 0 ) {
	    if ( !(statbuf.st_mode & S_IFDIR) ) {
	        error( "%s is not a directory", dirfile );
	        return FAILURE_;
	    }
	} else {
	    error( "Error opening configuration directory %s: %s", dirfile, 
							  errorString() );
	    return FAILURE_;
	}

	char filename[1024];
	sprintf( filename, "%s/Info", dirfile );

	FILE *fp = fopen( filename, "r" );
	if ( fp == NULL ) {
	    error( "Error opening %s: %s", filename, errorString() );
	    return FAILURE_;
	}

	char buf[1024];
	Boolean_ firstLine = TRUE_;

	while ( fgets( buf, sizeof(buf), fp ) != NULL ) {
	    cout << buf;
	    if ( firstLine ) {
		char *bufP = buf;
		if ( strncmp( bufP, "Pablo saved state written ", 26 ) == 0 ) {
		    bufP += 26;
		    bufP[24] = '\0';
		    _updateModifyDate( bufP );
		} else {
		    _updateModifyDate( NULL );	// no date found
		}
		firstLine = FALSE_;
	    }
	}
	fclose( fp );

	sprintf( filename, "%s/Layout", dirfile );
	RETURN_ON_FAILURE( _readLayoutFromFile( filename ) );

	sprintf( filename, "%s/StructureDictionary", dirfile );
	if ( ! Pablo::StructureDict()->loadFromFile( filename ) ) {
	    error( "Error loading Dictionary from file %s", filename );
	    return FAILURE_;
	}

	if ( ! Pablo::IM()->loadConfigurationFromDir( dir ) ) {
	    error( "Error loading configuration from dir %s", dirfile );
	    return FAILURE_;
	}

	return SUCCESS_;
}

Boolean_
PabloMainInterface::_parseLine( char *line, int lineno )
{

#define MATCH(x) ( strcasecmp( cmd, (x) ) == 0 )

	char cmd[50];
	int n = sscanf( line, "%s", cmd );

	if ( n != 1 ) {
	   return FAILURE_;
	}

	char	FUName[50];
	char	moduleName[50];
	char	filename[1024];

	int	moduleIdx;
	int	nodeIdx;

	int	from;
	int	to;

	int	x;
	int	y;

	if ( MATCH("F") ) {
	   n = sscanf( line, "%s %s %s", cmd, FUName, moduleName );

	   if ( n != 3 ) {
              return FAILURE_;
	   }

	   moduleIdx = Pablo::IM()->addModule( FUName, moduleName );
	   if ( moduleIdx == -1 ) {
	      error( "Could not add module %s", moduleName );
	      return FAILURE_;
	   }
	   nodeIdx = graphDisplay->addNode( moduleName );
	   Assert( nodeIdx == moduleIdx );
		
	} else if ( MATCH("l") || MATCH("loc") ) {
	   n = sscanf( line, "%s %s %d %d", cmd, moduleName, &x, &y );

	   if ( n != 4 ) {
              return FAILURE_;
	   }

	   graphDisplay->moveNode( moduleName, x, y );

	} else if ( MATCH("C") ) {
	   n = sscanf( line, "%s %d %d", cmd, &from, &to );

	   if ( n != 3 ) {
              return FAILURE_;
	   }

	   if ( Pablo::IM()->addConnection( from, to ) ) {
	       graphDisplay->addEdge( from, to );
	   } else {
	       error( "Couldn't add pipe between modules %d and %d; line %d\n",
							    from, to, lineno );
	   }

	} else if ( MATCH("iw") ) {
	    /* 
	     * A FileInputWrapper. 
	     * Fields:
	     *     0 == command
	     *     1 == moduleName
	     *     2 == filename
	     */	 

	   n = sscanf( line, "%s %s %s", cmd, moduleName, filename );

	   if ( n != 3 ) {
              return FAILURE_;
	    }

	   moduleIdx = Pablo::IM()->addFileInputModule( moduleName, filename );
	   nodeIdx = graphDisplay->addNode( moduleName );
	   Assert( nodeIdx == moduleIdx );

	} else if ( MATCH("ow") ) {
	    /* 
	     * A FileOutputWrapper. 
	     * Fields:
	     *     0 == command
	     *     1 == moduleName
	     *     2 == filename
	     */	 

	   n = sscanf( line, "%s %s %s", cmd, moduleName, filename );

	   if ( n != 3 ) {
              return FAILURE_;
	   }

	   moduleIdx = Pablo::IM()->addFileOutputModule( moduleName, filename );
	   nodeIdx = graphDisplay->addNode( moduleName );
	   Assert( nodeIdx == moduleIdx );

	} else {
           return FAILURE_;
	}

	return SUCCESS_;
}

Boolean_ 
PabloMainInterface::_readLayoutFromFile( const CString& file )
{
	FILE *fp;

	if ( ( fp = fopen( file.getValue(), "r" ) ) == NULL ) {
	    error( "\nUnable to open layout file %s. %s", 
		     (const char *)file, errorString() );
	    return FALSE_;
	}

	graphDisplay->unmanage();

	char buf[1024];
	int lineno = 0;
	Boolean_ goodLayout = TRUE_;

	while ( goodLayout && fgets( buf, sizeof(buf), fp ) != NULL ) {
	    lineno++;
	    goodLayout = _parseLine( buf, lineno );
	}

	if ( !goodLayout ) {
	    warning( "Error reading line %d of Layout file %s", 
					lineno, (const char *) file );
	}

	fclose( fp );
	graphDisplay->manage();

	return goodLayout;
}

Boolean_ 
PabloMainInterface::_saveConfigurationToDir( const CString& dir ) const
{
	const char *dirfile = dir.getValue();

	struct stat statbuf;
	if ( stat( (char *)dirfile, &statbuf ) == 0 ) {	    // It exists
	    if ( !( statbuf.st_mode & S_IFDIR ) ) { // It isn"t a directory
	        error( "%s is not a directory", dirfile );
	        return FAILURE_;
	    }
	} else {
	    if ( errno != ENOENT ) {		    // It exists but stat failed
	        error( "Stat %s failed: %s", dirfile, errorString() );
	        return FAILURE_;
	    } else {				    // Does not exist 
		if ( mkdir((char *)dirfile, 0755) < 0 ) {      // mkdir
		   error( "Mkdir %s failed: %s", dirfile, errorString() );
		   return FAILURE_;
		}
	    }
	}
	
	// At this point the directory exists  
	// Save the date, user, and host information in dir/Info
	char filename[1024];

	sprintf( filename, "%s/Info", dirfile );
	FILE *fp = fopen( filename, "w" );
	if ( fp == NULL ) {
	    error( "Error opening %s: %s", filename, errorString() );
	    return FAILURE_;
	}

	time_t now = time(0);
	char *date = ctime( &now );
	struct passwd *pw = getpwuid( getuid() );
	char *user = (pw == NULL) ? "Unknown" : pw->pw_name;
	char host[128];
	gethostname( host, sizeof(host) );

	char *s = strchr( date, '\n' );
	if (s) {
	    *s = 0;
	}

	fprintf( fp, "Pablo saved state written %s\nby %s on %s\n",
		     date, user, host );
	fclose( fp );
	
	// Save the module and graph layouts in dir/Layout
	sprintf( filename, "%s/Layout", dirfile );
	fp = fopen( filename, "w" );

	if ( fp == NULL ) {
	    error( "Error opening %s: %s", filename, errorString() );
	    return FAILURE_;
	}

	if ( ! Pablo::IM()->writeLayoutToFP( fp ) ) {
	    warning( "Error writing module layout to file %s", filename );
	    return FAILURE_;
	}

	if ( ! graphDisplay->writeLayoutToFP( fp ) ) {
	    warning( "Error writing graph layout to file %s", filename );
	    return FAILURE_;
	}
	fclose( fp );

	// Write the global Structure Dictionary in dir/StructureDictionary

	sprintf( filename, "%s/StructureDictionary", dirfile );
	if ( ! Pablo::StructureDict()->saveToFile( filename ) ) {
	    warning( "Error writing dictionary to file %s", filename );
	    return FAILURE_;
	}

	//
	// Write the TagMappingDict. 
	// NOTE --- THIS FILE IS NOT RELOADED. MORE FOR VERIFICATION PURPOSES
	//
	sprintf( filename, "%s/TagMappingDictionary", dirfile );
	Pablo::TagMappingDict()->saveToFile( filename );
	
	// Now the fun part. Save the rest of the configuration
	// information by calling the InfrastructureManager method.
	
	return ( Pablo::IM()->saveConfigurationToDir( dirfile ) );
}

/*
 * Initialize the static data
 */
const char *const PabloMainInterface::MY_CLASS = "PabloMainInterface";
PabloMainInterface *PabloMainInterface::singleInstance = NULL;

