/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Author: Robert Olson (olson@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * Wrapper.h: Base wrapper class implementing interface between a 
 *	      functional unit, an input file, or an output file and 
 *	      the Pablo infrastructure.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Includes/RCS/Wrapper.h,v 1.20 1994/03/15 16:27:42 aydt Exp $
 *	
 */

#ifndef Wrapper_h
#define Wrapper_h

#include "Obj.h"
#include "PacketHeader.h"

#include "Attributes.h"
#include "BinaryPipeWriter.h"
#include "ModuleId.h"
#include "RecordDossier.h"
#include "StructureDescriptor.h"

class FunctionalUnit;
class RecordDictionary;		
class StreamPipe;

enum WrapperClass {
	GenericWrapperClass,
	FileInputWrapperClass,
	FileOutputWrapperClass,
	FUWrapperClass,
};

class Wrapper : public Obj {
/*
 * Class implementing interface between a FU, an input file, or an output file,
 * and the Pablo infrastructure.  Hides details of 'internal object' from
 * the infrastructure.  A Wrapper and it's 'internals' are often referred
 * to as a module.
 */
private:
	char *_objectName;		        // ClassName + extra info. 
					        // Returned by getObjectName(). 
					        // Set by _setObjectName().

	CString moduleName;			// Name of associated module

protected:
	int 		      moduleIdentifier; // ID for TagMappingDict
	
	BinaryPipeWriter      *binaryOutput;	// OutputPipe(s)
	RecordDictionary      *outputDictionary;
	Boolean_ 	      needsConfiguration;

//	***** Protected Methods Available to all Derived Classes *****

	/* Method _setObjectName:		Sets _objectName to combination
	*					of class name and argument.   */
	void _setObjectName( const char *name );

public:
//	***** Constructors and Destructors *****

        /* Method Wrapper: 			The Wrapper constructor.      */
	Wrapper();

        /* Method ~Wrapper: 			The Wrapper destructor.       */
	~Wrapper();

//	***** Virtual Functions *****

	/* Method addInputPipe: 		Add an input pipe.  Some 
	*					Wrapper subclasses will not
	*					accept input pipes. Arguments
	*					are pipe and source */ 
	virtual Boolean_ addInputPipe( StreamPipe* ,  ModuleId* ) {
		subclassResponsibility( "Wrapper::addInputPipe ");
		return FALSE_;
		} ;

	/* Method addOutputPipe: 		Add an output pipe.	      */
	virtual Boolean_ addOutputPipe( StreamPipe *pipe );

	/* Method configure:			Perform operations necessary
	*					to configure this module.    
	*					The argument, singleModule, 
	*					is TRUE_ if this configure() 
	*					is not part of system-wide 
	*					configuraion.     */
	virtual void configure( Boolean_ ) {
		subclassResponsibility( "Wrapper::configure" ); 
		} ;

	/* Method configureParams:		Redisplay and possibly allow
	*					changes to parameters of this
	*				        module (no display of binding 
	*					information). If undefined for
	*					subclass, call configure()    */
	virtual void configureParams( ) {
		configure( TRUE_ );
		} ;

	/* Method deleteOutputPipe:		Delete the specified pipe.    */	virtual void deleteOutputPipe( StreamPipe *pipe );

	/* Method flushOutputDictionary:	Write output dictionary to 
	*					the output pipe(s), updating
	*					it first if necessary.        */
	virtual void flushOutputDictionary();

	/* Method getFunctionalUnit:		Get FU associated with this
	*					wrapper. NULL if none exists  */
	virtual FunctionalUnit * getFunctionalUnit() const;

	/* Method getObjectName:		Overlaods Obj::getObjectName()
	*					to return _objectName.        */
	virtual const char * getObjectName() const;

	/* Method getWrapperClass:		Returns the class of the 
	*					Wrapper instance for which it
	*					is called.                    */
 	virtual WrapperClass getWrapperClass() const {
		return GenericWrapperClass;
                } ;

	/* Method init:				Initialize Wrapper state.     
	*					Output pipes are NOT reset,
	*					all else is.                  */
	virtual void init();

	/* Method loadConfigurationFromDir:	Load saved config info - args
	*					are dir and moduleIndex       */
	virtual Boolean_ loadConfigurationFromDir( const CString&, int ) {
		subclassResponsibility( "Wrapper::loadConfigurationFromDir" );
		return FALSE_;
		} ;

	/* Method ready:			Returns TRUE_ if module is
	*					ready to run.                 */
	virtual Boolean_ ready() { 
		subclassResponsibility( "Wrapper::ready" );
		return FALSE_;
		} ;

	/* Method restart:		        Clears any internal data state
	*					associated with the module while
	*					maintaining binding and 
	*					configuration parameters.     */
	virtual void restart() {
		subclassResponsibility( "Wrapper::restart" );
		} ;

	/* Method run:				Run the module 'once'. errorFlag
	*					is set if an error is seen
	*					when running.                 */
	virtual void run( Boolean_& errorFlag) {
		subclassResponsibility( "Wrapper::run" );
		errorFlag = TRUE_;
		} ;

	/* Method saveConfigurationToDir:	Save the configuration info   */
	virtual Boolean_ saveConfigurationToDir( const CString& dir,
						 int moduleIndex ) const;

	/* Method writeLayoutToFP:	        Method to save layout info. 
	*					Arg is fp	              */
	virtual void writeLayoutToFP( FILE * ) const {
		subclassResponsibility("Wrapper::writeLayoutToFP"); 
		} ;

//	***** Non-Virtual Functions *****

	/* Method getIdentifier:		Get the module identifier.    */
	int getIdentifier() const; 

	/* Method getName:			Get the name of the module 
	*					associated with this wrapper. */
	const CString& getName() const;

	/* Method isConfigured:			Returns TRUE_ if this module
	*					has been configured.          */
	Boolean_ isConfigured() const;

	/* Method setIdentifier:		Set the module Identifier     */
	void setIdentifier( int identifier ); 

	/* Method setName:			Save the name of the module
	*					associated with this wrapper. */
	void setName( const CString& name );

	/* Method writeAttributes:		Write Attributes to output    */
	void writeAttributes( Attributes& attr ) const;

	/* Method writeCommand:			Write Command to output       */
	void writeCommand( int tag ) const;

	/* Method writeData:			Write Data to output          */
	void writeData( RecordDossier& dossier ) const;

	/* Method writeDescriptor:		Write Descriptor to output    */
	void writeDescriptor( const StructureDescriptor& descr, int tag ) const;

//	***** Functions Defined for all Classes *****

	/* Method printOn: Stream output function */
	virtual void printOn(ostream& os = cout) const;
};

//	***** Inline Function Definitions *****

inline int
Wrapper::getIdentifier() const { 
	return moduleIdentifier; 
}

inline const CString& 
Wrapper::getName() const 
{ 	
	return moduleName;
}

inline Boolean_
Wrapper::isConfigured() const
{
	return ( needsConfiguration ? FALSE_ : TRUE_ );
}

inline void
Wrapper::setIdentifier( int identifier ) 
{ 
	moduleIdentifier = identifier; 
}

inline void
Wrapper::setName( const CString & name )  
{ 	
	moduleName = name;
}

inline void
Wrapper::writeAttributes( Attributes &attr ) const
{
	binaryOutput->putAttributes( attr );
}

inline void
Wrapper::writeCommand( int tag ) const
{
	binaryOutput->putCommand( tag );
}

inline void
Wrapper::writeData( RecordDossier &dossier ) const
{
	binaryOutput->putData( dossier );
}

inline void
Wrapper::writeDescriptor( const StructureDescriptor &descr, int tag ) const
{
	binaryOutput->putDescriptor( descr, tag );
}

//	***** General Operators *****

inline ostream& operator<<( ostream& os, Wrapper& obj )
{
	obj.printOn( os );
	return os;
}

#endif
