/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Philip C. Roth (proth@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * UtilFU.h: A generic utilization functional unit
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Includes/RCS/UtilFU.h,v 1.11 1994/03/15 16:26:29 aydt Exp $
 */

#ifndef	UtilFU_h
#define	UtilFU_h

#include <X11/Intrinsic.h>
#include "CStringObjList.h"
#include "DisplayFunctionalUnit.h"
#include "Value.h"
#include "EventNode.h"
#include "UtilDisplay.h"


 // define some constants for specifying array sizes
 // g++ allows this to be done with 'const int' but CC does not.

#define MAXINFOWIDGETS	15

class UtilFU;
class InputPort;
class NumberLine;
class UtilDisplayManager;


 // In order to support Motif callbacks, which cannot be dynamic objects,
 // static member functions must be used as stubs which call member objects.
 // This struct allows a UtilFU object to be passed through the static
 // functions to the actual members which will operate on it.
 // This is mostly used in popping up/down Motif dialogs. 

struct UtilFUData {
	Widget	dialog;		// dialog widget which needs to be popped down
	UtilFU *ufu;		// reference to associated object
};



 // In order to compute sliding window averages for the utilizations,
 // this struct keeps a running total of several statistics for each
 // PE under consideration.

struct procData {
	double	duration;	// total time PE was blocked in window
	float	util;		// utilization for the given PE

	procData( void );	// constructor for initialization
};



 // Declaration of struct containing UtilFU class data whose default
 // values are taken from X resources
 // Note that floats are used for windowSize and windowIncr ( instead
 // of doubles ) since the X resource manager doesn't support doubles
 // as a resource type.

typedef struct {
	int	numProcessors;		// number of PEs to track
	float 	windowSize;		// size of sliding window
	float	windowIncr;		// sliding window increment
	Boolean	initialStateIsBusy;	// flag indicating initial state as busy
	char*	horiLabel;		// horizontal display label
	char*	vertLabel;		// vertical display label
} DefaultUtilFUData, *DefaultUtilFUDataPtr;



class UtilFU : public DisplayFunctionalUnit {
/*
 * UtilFU is Pablo functional unit for displaying the
 * utilization of a set of processing elements
 */

private:
	static const char *const MY_CLASS;	// the class name for UtilFU

	static const int max_name_len;		// len of longest display name

	static Boolean_ startedRunning;	// flags whether weve started running

	Boolean_ isConfigured;		// flags whether configure() is done



	// X/Motif data and constants

	Widget infoWidgets[MAXINFOWIDGETS];	// information-producing widgets
	enum { disp_list, 			// indices for certain
	       disp_type, 			// widgets in the infoWidgets
	       disp_name,			// array
	       disp_mnam,
	       hori_labl,
	       vert_labl, 
	       indv_slct,
	       aggr_slct, } ;

	Widget	dialog,			// dialog hierarchies for popups
		typeDialog,
		modDialog,
		delDialog;

	static DefaultUtilFUData defaultData;	// defaults taken from X
						// resource files


	// UtilDisplay data

	int numDisplays;			// count of existing displays
	UtilDisplay *displays;			// list of existing displays
	UtilDisplayManager *dispManager;	// manager for displays
	



	// Pablo input port data

	InputPort	*brTime;		// timestamp of blocking receive
	InputPort 	*brProcNum;		// number of blocked PE 
	int	brTimeDim,			// dimension of incoming data
		brProcNumDim;
	Value	brTimeValue,			// input data values
		brProcNumValue;

	// data dealing with event sliding windows

	int	numProcessors;		// max number of PEs to handle
	NumberLine *numberline;		// object to allow choosing subset
					//   of available PEs
	double		leadEdgeTS;	// timestamp of leading edge of window
	EventNode	*leadInterval;	// ptr to interval containing lead
					//   edge of window

	double		windowSize;	// size of sliding data windows
	double		windowIncrement;// amount window should slide, per slide

	EventNode	*evts;		// list of events in the sliding window
	EventNode	*lastEvt;	// last event in sliding window
	procData	*pwinds;		// array of utilization data
	static	CStringObjList	initialStatusList;	// for configuration of
							// initial state of
							// PEs in window
	BusyStatus	initialState;	// initial state of PEs in window


	/* Method _addDisplay:		Actual callback for adding
	 *				new UtilDisplays with the
	 *				given name			*/
	void _addDisplay( char *name, UtilsDisplayedType dispType );


	/* Method _advanceWindow:	Advances the sliding
	 *				time window used to compute
	 *				utilizations			*/
	void _advanceWindow( void );


	/* Method _deleteDisplay:	Actual callback for deleting 
	 *				existing displays		*/
	//
	void _deleteDisplay( UtilFUData *data );


	/* Method _getCurrentDisplay:	Retrieves the currently
	 *				selected UtilDisplay		*/
	UtilDisplay * _getCurrentDisplay( void );


	/* Method _handleAggregateToggle: Actual callback for handling the
	 *				changes caused by activating the
	 *				Aggregate ToggleButton in the
	 *				"New Display" dialog		*/ 
	void _handleAggregateToggle( void );


	/* Method _handleBrowse:	Actual callback for handling 
	 *				browsing through the UtilDisplay
	 *				type list			*/
	void _handleBrowse( XmListCallbackStruct *cb );


	/* Method _handleIndividualToggle: Actual callback for handling the
	 *				changes caused by activating the
	 *				Individual ToggleButton in the
	 *				"New Display" dialog		*/
	void _handleIndividualToggle( void );


	/* Method _handleNewDisplay:	Actual callback which controls
	 *				the creation of new UtilDisplay
	 *				objects				*/
	void _handleNewDisplay( UtilFUData *data );


	/* Method _handleSpecifyRest:	Actual callback which controls
	 *				the completion of the specification
	 *				of UtilDisplay parameters	*/
	void _handleSpecifyRest( UtilFUData *data );


	/* Method _handleModifyDisplay:	Actual callback for finishing
	 *				the modification of existing
	 *				UtilDisplays			*/
	void _handleModifyDisplay( UtilFUData *data );


	/* Method _handleDeleteDisplay:	Actual callback for deleting 
	 *				existing UtilDisplays		*/
	void _handleDeleteDisplay( UtilFUData *data );


	/* Method _infoDialog:		Creates an information
	 *				dialog that displays the
	 *				string str			*/ 
	void _infoDialog( UtilFUData *data, char *str );


	/* Method _makeDelDialog:	Creates the Delete Dialog,
	 *				used for deleting existing
	 *				UtilDisplays			*/
	Widget _makeDelDialog( UtilFUData *data );


	/* Method _makeMainDialog:	Creates the Main UtilFU
	 *				dialog				*/
	Widget _makeMainDialog( void );


	/* Method _makeModDialog:	Creates the Modify Dialog,
	 *				used for modifying existing
	 *				UtilDisplays			*/
	Widget _makeModDialog( UtilFUData *data );


	/* Method _makeTypeDialog:	Creates the New Dialog,
	 *				used for specifying the
	 *				type of a new UtilDisplay	*/
	Widget _makeTypeDialog( UtilFUData *data );


	/* Method _modifyDisplay:	Actual callback for modifying
	 *				existing UtilDisplays		*/
	void _modifyDisplay( UtilFUData *data );


	/* Method _popdownCleanup:	Pops down a dialog that is
	 *				currently displayed, and 
	 *				cleans up any storage allocations */
	void _popdownCleanup( UtilFUData *data );


	/* Method _updateDisplayList:	Forces update of the visual
	 *				aspects of the existing
	 *				performance widgets to reflect
	 *				changes to the current display	*/
	void _updateDisplayList( void );


public:

	/* Method UtilFU:		The class constructor		*/
	UtilFU( void );			


	/* Method ~UtilFU:		The class destructor		*/
	virtual ~UtilFU( void );


	/* Method configure:		Configures the input pipe and
	 *				run-time behavior of the UtilFU	*/
	virtual void configure( void );


	/* Method copy:			Produces a copy of this FU	*/
	virtual FunctionalUnit * copy( void );


	/* Method init:			Initializes all UtilFU
	 *				data which could not be initialized
	 *				in the class constructor	*/
	virtual void init( void );


	/* Method loadConfigFromFile:	Scans the given stream
	 *				to load and create a
	 *				previously saved UtilFU and
	 *				its UtilDisplays		*/
	virtual Boolean_ loadConfigFromFile( const CString& );


	/* Method ready:		Indicates UtilFU"s readiness
	 *				to execute			*/
	virtual Boolean_ ready();

	/* Method run:			Processes a new trace record,
	 *				computing new utilizations,
	 *				and broadcasting the values to 
	 *				the UtilDisplays		*/
	virtual void run( Boolean_& errorFlag );


	/* Method saveConfigToFile:	Controls saving a
	 *				representation of the configured
	 *				UtilFU and all its displays to the
	 *				given stream			*/
	virtual Boolean_ saveConfigToFile( const CString& ) const;


	/* Method StartedRunning:	Indicates whether any UtilFU has
	*				ever started execution.		*/
	Boolean_ StartedRunning( void ) const;


	/* Method cancelCB:		XmNactivate callback for cancel 
	 *				buttons everywhere		*/
	static void cancelCB( Widget, UtilFUData*, XtPointer );


	/* Method deleteCB:		XmNactivate callback for 
	 *				controlling UtilDisplay deletion */
	static void deleteCB( Widget, UtilFU*, XtPointer );


	/* Method deleteDisplayCB:	XmNactivate callback for finishing 
	 *				deletion of a UtilDisplay	*/
	static void deleteDisplayCB( Widget, UtilFUData*, XtPointer );


	/* Method handleAggregateToggleCB: XmNvalueChanged callback for
	 *				handling the toggle button
	 *				controlling aggregate UtilDisplay
	 *				types				*/
	static void handleAggregateToggleCB( Widget, UtilFUData*, XtPointer );


	/* Method handleIndividualToggleCB: XmNvalueChanged callback for
	 *				handling the toggle button
	 *				controllling individual UtilDisplay
	 *				types				*/
	static void handleIndividualToggleCB( Widget, UtilFUData*, XtPointer );


	/* Method helpCB:		XmNactivate callback for handling
	 *				requests for help		*/
	static void helpCB( Widget, char*, XtPointer );


	/* Method newCB:		XmNactivate callback for handling
	 *				creation of new UtilDisplays	*/
	static void newCB( Widget, UtilFU*, XtPointer );


	/* Method modifyCB:		XmNactivate callback for handling
	 *				modifications to existing
	 *				UtilDisplays			*/
	static void modifyCB( Widget, UtilFU*, XtPointer );


	/* Method modifyDisplayCB:	XmNactivate callback for finishing
	 *				modification of existing
	 *				UtilDisplays			*/
	static void modifyDisplayCB( Widget, UtilFUData*, XtPointer );


	/* Method specifyRestCB:	XmNactivate callback for finishing
	 *				creation/modification of
	 *				UtilDisplays			*/
	static void specifyRestCB( Widget, UtilFUData*, XtPointer );


	/* Method typeBrowseCB:		XmNbrowseSelection callback for
	 *				browsing through list of types of
	 *				UtilDisplays			*/
	static void typeBrowseCB( Widget, UtilFU*, XmListCallbackStruct* );
};


/*
 * We follow Pablo convention and place inline functions outside the
 * class declaration.
 */
inline
Boolean_
UtilFU::StartedRunning( void ) const
{
	return startedRunning;
}


#endif	UtilFU_h
