/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Robert Olson (olson@cs.uiuc.edu)
 * Contributing Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Contributing Author: Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 */
/*
 * PabloMainInterface.h: Implements the high-level interface to the Pablo
 *			performance analysis environment
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Includes/RCS/PabloMainInterface.h,v 1.11 1994/03/23 20:37:22 aydt Exp $
 */

#ifndef PabloMainInterface_h
#define PabloMainInterface_h

#include <stdio.h>
#include <X11/Intrinsic.h>

#include "CString.h"
#include "Obj.h"
#include "VoidPtrList.h"

class AddModuleDialog;
class DefaultsDialog;
class FileSelectionDialog;
class GraphDisplay;
class ModuleId;
class RunDialog;
class WidgetBase;

enum InterfaceState { Quiet,
		      WaitingForFirstInConnection,
		      WaitingForSecondInConnection,
		      UpdateLink,
		      Configuring,
		      WaitingForPressOnNode,
		      Running,
		      DiscreteExecution,
		      WaitingForSingleStepContinue,
		      RunningSingleModule,
		      DoneRunning,
		      LastAndUnusedInterfaceState,
		      };

#define nInterfaceStates LastAndUnusedInterfaceState

struct menuItem {
        char          *menuLabel;
        WidgetClass   *widgetClass;
        char          mnemonicCharacter;
        char          *acceleratorString;
        char          *acceleratorText;
        void          (*callbackFunction) ( Widget, XtPointer, XtPointer );
        XtPointer     callbackData;
};

struct menuLine {
        char         *menuTitle;
        char         mnemonicCharacter;
        menuItem     *menuEntry;
};

class GraphNode;
class GraphEdge;

class PabloMainInterface : public Obj {
/*
 * 	PabloMainInterface is the primary user interface for the Pablo
 *	performance analysis environment.  All user interactions are
 *	initiated through this code or code invoked by this code.
 */
private:
	static const char *const MY_CLASS;
	static PabloMainInterface* singleInstance;

	Boolean_       executionDone;	/* Are we done running the graph?     */
        InterfaceState state;	        /* The current state                  */
	int	       _desensitizeCnt; /* Times menus desensitized.          */
	int	       _desensitizeRunCnt; /* Times run menu desensitized.    */

        /* Method _buttonOnePress:      Invoked when a user positions the mouse
	*				on a graph node and clicks the left
	*				button. Called via nodeCallback().    */
        void    _buttonOnePress( GraphNode *node, XEvent *event );

        /* Method _buttonThreePress:   Invoked when a user positions the mouse
	*				on a graph node and clicks the right
	*				button. Called via nodeCallback().    */
        void    _buttonThreePress( GraphNode *node, XEvent *event );

	/* Method _desensitizeMenuOptions:  Turns off sensitivity for the
	*				File, Run, Configure, and Module 
	*				pulldown menus and the Help on Module
	*				Instance.                             */
	void _desensitizeMenuOptions();

	/* Method _desensitizeRunOption:  Turns off sensitivity for the
	*				Run pulldown menu.                    */
	void _desensitizeRunOption();

	/* Method _loadConfigurationFromDir: Loads a graph configuration
	*				from the specified directory	      */
        Boolean_ _loadConfigurationFromDir( const CString& dir );

        /* Method _parseLine:           Parses a command line from the
        *                               initialization of Pablo               */
        Boolean_ _parseLine( char *line, int lineNo );

	/* Method _readLayoutFromFile:	Reads a graph layout from the
	*				specified file			      */
	Boolean_ _readLayoutFromFile( const CString& file );

	/* Method _saveConfigurationToDir: Saves a graph configuration in
	*				the specified directory		      */
	Boolean_ _saveConfigurationToDir( const CString& dir ) const;

	/* Method _sensitizeMenuOptions:  Turns on sensitivity for the
	*				File, Run, Configure, and Module 
	*				pulldown menus and the Help on Module
	*				Instance.                             */
	void _sensitizeMenuOptions();

	/* Method _sensitizeRunOption:  Turns on sensitivity for the
	*				Run pulldown menu.                    */
	void _sensitizeRunOption();

	/* Method _setState:		Updates the graph canvas to show
	*				the current state. Returns FALSE_		*			        if the transition from current->new
	*				state cannot be made.                 */
        Boolean_  _setState( InterfaceState newState );


protected:

	CString	fileToOpen;	/* load layout from this file (if specified)  */
	CString	dirToOpen;	/* load configuration from this directory (if */
				/* specified)				      */


	XtAppContext	appContext;

	WidgetBase *	baseTopLevel;
	WidgetBase *	baseGraphWindow;

	Widget	mainPabloInterface;
	Widget	mainPabloMenu;

	Widget	topLevel;
	Widget	workWindow;

	Widget	stateValue;
	Widget	dateValue;
	Widget	configValue;

	Widget	scrollBox;

	Widget	pabloGraphic;
	Widget	pixmapFrame;

	VoidPtrList     signalWidgetList;        /* used to send callbacks */
	GraphDisplay	*graphDisplay;

        ModuleId *moduleNowBeingConfigured;
	ModuleId *moduleAboutToRun;

        int     connectFrom;
        int     connectTo;
        int     pressedNode;

        FileSelectionDialog *FSD;
        Boolean_        FSD_busy;

	AddModuleDialog *addModuleDialog;
        RunDialog       *runPanelDialog;
        DefaultsDialog  *defaultsDialog;

        Boolean_        running;

        CString         consoleLogFilename;
        FILE            *consoleLogFP;

	/* Method addModuleCallback:	static callback for adding
	*				analysis modules		*/
        static void addModuleCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method debugCompiledOptionsCallback: static callback for
	*				compiled options		*/
        static void debugCompiledOptionsCallback ( Widget w,
                                        XtPointer clData, XtPointer caData );

	/* Method debugDumpObjectCallback: static callback for
					dumping objects			*/
        static void debugDumpObjectCallback ( Widget w,
                                        XtPointer clData, XtPointer caData );

	/* Method debugDumpStateCallback:	static callback for
	*				dumping the infrastructure
	*				manager state			*/
        static void debugDumpStateCallback ( Widget w,
                                        XtPointer clData, XtPointer caData );

	/* Method configLoadCallback:	static callback for loading a
	*				graph configuration		*/
        static void configLoadCallback ( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method configSaveCallback:	static callback for saving a
	*				graph configuration		*/
        static void configSaveCallback ( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method connectModuleCallback:	static callback for
	*				connecting two or more modules	*/
        static void connectModuleCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method defaultsConfigureCallback:	static callback for
	*				editing the Pablo defaults	*/
        static void defaultsConfigureCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method deleteConfigCallback:	static callback for
	*				deleting the current analysis
	*				graph from the canvas		*/
        static void deleteConfigCallback ( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method deleteModuleCallback:	static callback for deleting a
	*				module from the canvas		*/
        static void deleteModuleCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method graphConfigureCallback:	static callback for 
	*				configuring the current graph	*/
        static void graphConfigureCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method helpOnConfigureCallback:	static callback for
	*				help on the configuration menu	*/
        static void helpOnConfigureCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method helpOnDebugCallback:	static callback for help on the
	*				debug menu			*/
        static void helpOnDebugCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method helpOnFileCallback:	static callback for help on the
	*				file menu			*/
        static void helpOnFileCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method helpOnHelpCallback:	static callback for help on the
	*				help menu			*/
        static void helpOnHelpCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method helpOnInstanceCallback:	static callback for help
	*				on a particular module instance	*/
        static void helpOnInstanceCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method helpOnModuleCallback:		static callback for help
	*				on the module menu		*/
        static void helpOnModuleCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method helpOnPabloCallback:	static callback for a brief
	*				description of Pablo		*/
        static void helpOnPabloCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method helpOnRunCallback:	static callback for help on the
	*				run menu			*/
        static void helpOnRunCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method layoutLoadCallback:	static callback for loading an
	*				analysis graph layout		*/
	static void layoutLoadCallback ( Widget w,
					 XtPointer clData, XtPointer caData );

	/* Method layoutSaveCallback:	static callback for saving an
	*				analysis graph layout		*/
	static void layoutSaveCallback ( Widget w,
					 XtPointer clData, XtPointer caData );

	/* Method moduleConfigureCallback:	static callback for 
	*				configuring a specific module	*/
        static void moduleConfigureCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method panelCallback:	static callback for creating an
	*				instance of the run panel dialog */
        static void panelCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method parameterConfigureCallback:	static callback for 
	*				configuring FU parameters of a specific 
	*				module				*/
        static void parameterConfigureCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method quitCallback:		static callback for exiting Pablo */
	static void quitCallback( Widget w,
					 XtPointer clData, XtPointer caData );

	/* Method restartCallback:	Static callback for restarting the
	*				execution of the analysis graph       */
	static void restartCallback( Widget w, XtPointer clData, 
					       XtPointer caData );

	/* Method runCallback:		static callback for executing the
	*				analysis graph			*/
        static void runCallback( Widget w,
                                        XtPointer clData, XtPointer caData );

	/* Method singleStepCallback:	static callback for single stepping
	*				the execution of the analysi graph */
        static void singleStepCallback( Widget w,
                                         XtPointer clData, XtPointer caData );

	/* Method stopCallback:		static callback for stopping the
	*				execution of the analysis graph	*/
        static void stopCallback( Widget w,
                                        XtPointer clData, XtPointer caData );

	/* Method transcriptCallback:	static callback for creating an
	*				instance of the transcript dialog */
        static void transcriptCallback ( Widget w,
                                         XtPointer clData, XtPointer caData );


        /* Method autoLoadConfiguration: loads any graph configuration
        *                               that has been specified on the
        *                               command line                    */
        void	autoLoadConfiguration();

        /* Method buildMenuItem:        constructs the high-level
         *                              menu items for the interface    */
        void	buildMenuItem( Widget w, menuLine menu ) ;


        /* Method configLoadFromDir:    loads a graph configuration from
        *                               a directory                     */
        void    configLoadFromDir();

        /* Method configSaveToDir:      saves a graph configures to a
        *                               directory                       */
        void    configSaveToDir();

	/* Method createInterface:	constructs the primary user
	*				interface (menus and canvas)	*/
	void	createInterface( char *interfaceClass, int *argc, char **argv );

	/* Method createPabloPixmaps:	creates the pixmaps used for
	*				icons in the primary interface	*/
	void	createPabloPixmaps();

        /* Method debugDumpState:       dumps the current state of the
        *                               Pablo infrastructure manager    */
        void    debugDumpState();

        /* Method defaultsConfigure:    invokes the Pablo defaults
        *                               configuration dialog            */
        void    defaultsConfigure();

        /* Method deleteConfig:         Deletes the current analysis graph from
	*				the canvas.  If calledFromCallback and 
	*				resource says to ask, user is asked to 
	*				verify before action is taken.        */
        void deleteConfig( Boolean_ calledFromCallback = FALSE_ );

        /* Method fileQuit:             exits Pablo                     */
        void    fileQuit();

        /* Method graphConfigure:       allows the user to configure all
        *                               modules in the current analysis
        *                               graph                           */
        void    graphConfigure();

        /* Method helpOnTopic:          invokes the help system to provide
        *                               help on the specified topic     */
        void    helpOnTopic( char *topic );

	/* Method initializeStateStrings: initializes the suite of potential
	*				graph execution states for later
	*				display on the graph canvas	*/
        void 	initializeStateStrings();

        /* Method layoutLoadFromFile:   loads a graph layout from a file */
        void    layoutLoadFromFile();

        /* Method layoutSaveToFile:     saves a graph layout to a file  */
        void    layoutSaveToFile();

	/* Method loadResources:	loads the Pablo X resources from
	*				the X database			*/
	void	loadResources();

        /* Method moduleAdd:            invokes the add module popup
        *                               dialog to add a module to the
        *                               current analysis graph          */
        void    moduleAdd();

        /* Method moduleConfigure:      Allows the user to configure an
        *                               individual analysis module on the
        *                               current analysis graph.          */
        void moduleConfigure( Boolean_ fuParamsOnly );

        /* Method moduleConnect:        connects two modules in the
        *                               current analysis graph          */
        void    moduleConnect();

        /* Method moduleDelete:         deletes a module from the current
        *                               analysis graph                  */
        void    moduleDelete();

        /* Method moduleHelp:           invokes the help system to provide
        *                               interactive help on the selected
        *                               analysis module                 */
        void    moduleHelp();

        /* Method openTranscript:       pops up the transcript of Pablo
        *                               messages                        */
        void	openTranscript();

	/* Method processArguments:	processes any command line
	*				arguments			*/
	void	processArguments( int argc, char **argv );

        /* Method releaseFSD:           releases a file selection dialog,
        *                               ensuring that only one is active
        *                               at any time                     */
        void    releaseFSD();

        /* Method reserveFSD:           reserves a file selection dialog,
        *                               ensuring that only one is active
        *                               at any time                     */
        Boolean_ reserveFSD();

	/* Method restartExecution:	restart the execution of the current
	*				execution graph clearing any data
	*				states but maintaining display positions
	*				and configuration parameters.         */
	void restartExecution();

        /* Method runPanel:     invoked to popup an instance of the run
        *                       panel dialog, allowing user control of
        *                       analysis module graph execution         */
        void    runPanel();

        /* Method showCompiledOptions:  displays the options with which
        *                               Pablo was configured            */
        void    showCompiledOptions();

        /* Method _updateConfigFilename:   Updates the graph canvas to show
        *                                  the current configuration name.
	*				   If "configName" is a file, it's 
	*				   modification date is also used.
	*				   If it's neither a file nor a 
	*				   directory, current time is used.   */
        void _updateConfigFilename( const char* configName );

	/* Method _updateModifyDate:	updates the last modification date in
	*			        the control area with the value of
	*				modifyDate or the current time
	*			        if modifyDate == NULL 	              */
	void _updateModifyDate( const char *modifyDate );

	/* Method updateConsoleLog:	updates the popup transcript of 
	*				Pablo messages			*/
	void	updateConsoleLog();

public:
        /* Method PabloMainInterface: The constructor			*/
	PabloMainInterface( int *argc, char **argv );

        /* Method ~PabloMainInterface: The destructor			*/
	~PabloMainInterface();

        /* Method addSignalWidget:      Add a callback widget to the list */
	void  addSignalWidget( Widget inputSignalWidget );

        /* Method beginConfigureOf:     called by the InfrastructureManager
        *                               at the beginning of a module's
        *                               configuration                    */
        void beginConfigureOf(ModuleId *id);

        /* Method checkpoint:           checkpoints the graph state     */
        void    checkpoint();

	/* Method defaultsConfigureOK:	Called by the OK callback of the
	*				Defaults Dialog to sync with the
	*				Run Panel Dialog.                     */
	void defaultsConfigureOK();

        /* Method endConfigureOf:       called by the InfrastructureManager
        *                               at the end of a module's
        *                               configuration                   */
        void endConfigureOf( ModuleId *id );

        /* Method getAppContext:        returns the application context */
        XtAppContext * getAppContext();

	/* Method getGraphDisplay:	returns a pointer to the
	*				display containing the analysis
	*				graph				*/
	GraphDisplay * getGraphDisplay();

        /* Method getTopLevel:          returns the highest level widget */
        WidgetBase * getTopLevel();

	/* Method isCameraOn:           returns TRUE_ if the global snapshot
					camera is on */
        Boolean_ isCameraOn();

	/* Method takeSnapshot:         take a "picture" of the given widget */
        void takeSnapshot( Widget subjectWidget, const char *fuClass,
			   const char *fuName );

	/* Method moduleAddRequest: 	invoked by an instance of the
	*				AddModuleDialog when a functional
	*				unit is selected to be added	*/
	void moduleAddRequest( const CString &fuClass, const CString &fuName );

        /* Method nodeCallback:         Invoked in response to a user
        *                               mouse click on a graph node. Called
	*				by GraphNode event handler.           */
        void nodeCallback( GraphNode *node, XEvent *event );

        /* Method run:                  runs the Pablo analysis
        *                               environment                     */
        void    run();

	/* Method runModule:		executes a single Pablo graph
	*				module. Returns TRUE_ when 
	*				last module in graph has been
	*				run.                            */
        Boolean_ runModule();

	/* Method runPanelDismiss:	Called by the dismiss callback of the
	* 				Run Panel Dialog to reactivate the
	*				Run Menu and synchronize with 
	*			        Defaults Dialog                       */
	void runPanelDismiss();

	/* Method runRun:		executes the complete graph	*/
        void    runRun();
	
	/* Method runGraphDiscrete:	runs the complete graph in
	*				discrete mode			*/
        void    runGraphDiscrete( int stepCount );

	/* Method runStop:		halts graph execution		*/
        void    runStop();

	/* Method waitForStepContinue:	waits until single step graph
	*				execution is reenabled		*/
	void	waitForStepContinue( ModuleId *modId );

};

inline WidgetBase *
PabloMainInterface::getTopLevel() {
	return baseTopLevel;
}

inline XtAppContext *
PabloMainInterface::getAppContext() {
        return &appContext;
}

inline GraphDisplay *
PabloMainInterface::getGraphDisplay() {
	return graphDisplay;
}

#endif
