/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Author: Robert Olson (olson@csluiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * InputPort.h: Keeps information about a Functional Unit input port.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Includes/RCS/InputPort.h,v 1.10 1994/03/15 17:32:20 aydt Exp $
 */

#ifndef InputPort_h
#define InputPort_h

#include "CString.h"
#include "DataTraits.h"
#include "RecordDossier.h"
#include "Value.h"

class InputPipeSocket;

enum PortStatus_ { INVALID_ = 0, NEW_ = 1, USED_ = 2, IGNORED_ = 3 };	

class InputPort : public Obj {
/*
* 	An InputPort is the interface between the FU inputs and the Wrapper.
*	The FU will create the InputPort, name it, set the types, and notify 
*	the Wrapper that it has another InputPort.
*
*	During the Configuration of the FU, various (record,field) mappings
*	from a particular input pipe are made to this FU.  The traits are
*	'set in stone' with the first binding to the InputPort.
*
*	During execution of the configured graph, the InputPipeSocket
*       within the FUWrapper updates the valueStatusFlag to INVALID_ 
*       when a Discard or Copy Through value is seen on the bound input pipe.  
*       When a Field Extract is seen, valueP is updated & the valueStatusFlag 
*       is set to NEW_ via ::setValueP().
*
*	When the FU runs, it calls getValue or getValueP and uses the 
*       Value pointed to by valueP; the valueStatusFlag is set to USED_ by
*       the getValue or getValueP call.
*  
*       If an FU runs and does not call getValue or getValueP for all the
*       input ports, the ports that were not used still have the NEW_
*       status when the FU exits. The FUWrapper::run() method calls
*	InputPort::markValueSeen on each inputPort -- it sets the status
*	of any port still marked as NEW_ to IGNORED_. This indicates that 
*	the values weren't consumed, but that it is safe to overwrite them.  
*       They are similiar to "USED_"... only the FU would really care to 
*	treat them differently. (Maybe we should force the FU to call 
*	markValueSeen() on unused ports and take it out of the FUWrapper...)
*
*	Usually the input ports are bound to fields from input records on
*	input sockets, however sometimes they may be bound to user-input
*	constants.  If that is the case, socketNumber will be equal to the
*	number of sockets, inputSocket will be NULL and fieldNum will be -1;
*/	
private:
	static const char *const MY_CLASS;      // The Class name	      

	CString 	      name;		// Port Name; Set in ctor     
	DataTraits* 	      *allowedTraits;	// Traits allowed by FU       
	int		      allowedCnt;	// Number of allowed traits

	int		      socketNumber;	// Number of bound socket
	InputPipeSocket       *inputSocket;	// Link to socket bound to port
	DataTraits 	      currentTraits;	// Traits fixed by constraints
	int		      constraintsCnt;	// Constraints on currentTraits

	const Value 	      *valueP;		// Link to value in RecordDict
	int		      fieldNum;		// Field whose Value is in port
	PortStatus_ 	      valueStatusFlag;	// Status of value 

	/* Method _acceptsTraits:		Returns TRUE_ if the traits are
	*					valid. Else returns FALSE_.   */
	Boolean_ _acceptsTraits( const DataTraits& traits ) const;


public:
	/* Method InputPort: 			The constructor 	      */
	InputPort( const CString& pname );

	/* Method ~InputPort: 			The destructor 	 	      */
	~InputPort();

	/* Method addTraits: 			Add traits to the list of 
	*					valid traits                  */
	void addTraits( const DataTraits& traits );

	/* Method addTraits: 			Add traits to the list of 
	*					valid traits                  */
	void addTraits( MachineDataType type, int dimension = 0 );

	/* Method clearTraits:			Clear current traits and reset
	*					constraintsCnt to 0.          */
	void clearTraits();

	/* Method clearValueP:			Set status flag to INVALID_   */
	void clearValueP();

	/* Method concatValidTraits:		Append to cstr a list of
	*					traits that are valid for this
	*					port. 	                      */
	void concatValidTraits( CString& cstr ) const; 

	/* Method constrainTraits:		Indicates new binding to port.
	*					Sets currentTraits if not yet
	*					determined. Returns FAILURE_
	*					if currentTraits already set
	*					and traits do not match, or if
	*					traits is not in allowed set. */
	Boolean_ constrainTraits( const DataTraits& traits );

	/* Method getActiveRecord:		Returns the entire RecordDossier
	*					from which the port value is
	*					being set. If the port status is
	*					INVALID_ or port is bound to a
	*					user-input constant, returns
	*					RecordDossier::noMatch.       */
	const RecordDossier& getActiveRecord() const;

	/* Method getActiveFieldID:		Returns Field Number 
	*					corresponding to port value.
	*				  	If port status is INVALID_,
	*					or port is bound to a user-input
	*					constant, -1 is returned.     */
	int getActiveFieldID() const;
	      
	/* Method getConstraintsCnt:		Returns the number of times
	*					current Traits is constrained */
	int getConstraintsCnt() const;

	/* Method getTraits:			Return the current traits for 
	*					the port.		      */
	const DataTraits& getTraits() const;

	/* Method getName:			Returns the name 	      */
	const CString& getName() const;

	/* Method getSocketNumber:		Returns the number of the socket
	*					bound to this input port.     */
	int getSocketNumber() const;

	/* Method getValue: 			Return the Value and set 
	*					status flag to USED_          */
	const Value& getValue();

	/* Method getValueP:			Returns pointer to the Value
	*					in the inputSocket and sets
	*					status flag to USED_          */
	const Value * getValueP();

	/* Method lockTraits:			Lock the currentTraits.       */
	void lockTraits();

	/* Method loosenTraits:			Indicates binding to port 
	*                                       deleted.  Decrements 
	*					constraintsCnt, and if zero 
	*					then clears currentTraits.    */
	void loosenTraits();

	/* Method markValueSeen:		If the status is NEW_, change
	*					it to IGNORED_; Called by
	*					FUWrapper.                    */
	void markValueSeen();

	/* Method resetTraits:			Reset current traits to the
	*					specified value and also
	*					reset constraints count.     */
	void resetTraits( const DataTraits& traits, int count );

	/* Method setSocketBinding:		Set the socket binding info
	*					for the port.                 */
	void setSocketBinding( int sockeIdx, InputPipeSocket *socket );

	/* Method setValueP: 			Set the value ptr and change
	*					value status flag to NEW_     */
	void setValueP( Value *valuePtr, int fieldID );

	/* Method valueAvailable: 	        Returns TRUE_ if there is a 
	*				        Value available               */
	Boolean_ valueAvailable() const;

	/* Method valueStatus:			Returns status of value       */
	PortStatus_ valueStatus() const;

	/* Method verifyStatus:			Verify that a status of valid
	*					still holds.  Called by IC
	*					after reconfiguration.        */
	void verifyStatus();

	/* Method printOn: 			Stream output function        */
	void printOn( ostream& os ) const;
};

inline void
InputPort::clearValueP()
{
	valueStatusFlag = INVALID_;
}

inline int
InputPort::getConstraintsCnt() const
{
	return constraintsCnt;
}

inline const DataTraits&
InputPort::getTraits() const
{
	return currentTraits;
}

inline const CString& 
InputPort::getName() const        
{ 
	return name; 
}

inline int
InputPort::getSocketNumber() const
{
	return socketNumber;
}

inline void
InputPort::markValueSeen()
{
	if ( valueStatusFlag == NEW_ ) {
		valueStatusFlag = IGNORED_;
	}
}
	
inline Boolean_
InputPort::valueAvailable() const
{
	return ( CnvToBoolean_(valueStatusFlag != INVALID_) );
}

inline PortStatus_
InputPort::valueStatus() const
{
	return valueStatusFlag;
}

inline ostream& operator<<( ostream& os, InputPort& p )
{
	p.printOn( os );
	return os;
}

#endif InputPort_h
