/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * UnaryMathFU.cc - A polymorophic functional unit for unary math
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/UnaryMathFU.C,v 1.17 1994/03/15 16:42:09 aydt Exp $
 */

#include "UnaryMathFU.h"
#include "AbsoluteValue.h"
#include "Negation.h"
#include "SquareRoot.h"
#include "UnaryCount.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "SystemErrors.h"

UnaryMathFU::UnaryMathFU()
{
	_setClassName( MY_CLASS );

	operand = NULL;
	result = NULL;
	myOperator = NULL;
	myOperatorNumber = -1;
	isReady = FALSE_;
}

UnaryMathFU::~UnaryMathFU()
{
	delete operand;
	delete result;
	delete myOperator;
}

void 
UnaryMathFU::_setOperator( const int opId )
{
	if ( myOperator != NULL ) {
  	    delete myOperator;
	}
  	myOperatorNumber = opId;

  	switch ( opId ) {
     	    case ABSOLUTEVALUE:
                myOperator = new AbsoluteValue;
                break;

     	    case NEGATION:
                myOperator = new Negation;
                break;

     	    case SQUAREROOT:
                myOperator = new SquareRoot;
                break;

	    case INCREMENT:
	        myOperator = new UnaryCount;
	        break;

     	    default:
                warning( 
		    "\nOperator %d does not correspond to any known operation",
	            opId );
                myOperatorNumber = -1;
                break;
  	}

  	if ( myOperatorNumber < 0 ) {
      	    isReady = FALSE_;
  	} else {
            isReady = TRUE_;
  	}
}

void                            /* virtual */
UnaryMathFU::configureOperation()
{
  	int opId = -1;
  	FUParams params;
  	params.addRadioButtonsParam( "Unary Operation",
			    BaseFUParamEntry::Integer, 0, UnaryOpNameList );

	ParamConfig pc( Pablo::TopLevel(), params, getName() );
  	BaseFUParamEntry& opEntry = params.getEntry( "Unary Operation" );

        if ( myOperatorNumber != -1 ) {
            opEntry.setValue( myOperatorNumber );
        }
        pc.run();

  	if ( opEntry.valueIsValid() ) {
     	    opId = opEntry.getValue().getInteger();
  	} else {
     	    warning( "Operation selection was not valid." );
	    opId = 0;
  	}

  	_setOperator( opId );
}

FunctionalUnit *		/* virtual */
UnaryMathFU::copy()
{
	UnaryMathFU *copy = new UnaryMathFU();
	return copy;
}

void 				/* virtual */
UnaryMathFU::init()
{
	/*
	 * Set up operation name list if it hasn't already been done.
	 */
	if ( UnaryOpNameList.isEmpty() ) {
            UnaryOpNameList.addElement( "Absolute Value" );
            UnaryOpNameList.addElement( "Negation" );
            UnaryOpNameList.addElement( "Square Root" );
            UnaryOpNameList.addElement( "Increment" );
	}

        /*
         * Continue by clearing up any 'leftovers' that will be around if
         * init() isn't being called for the first time.  The goal is
         * to start with a 'virgin' FU.
         */
	if ( operand != NULL ) {
 	    delete operand;
	}
	if ( result != NULL ) {
 	    delete result;
	}
        if ( myOperator != NULL ) {
            delete myOperator;
            myOperator = NULL;
            myOperatorNumber = -1;
            isReady = FALSE_;
        }

	operand = new InputPort( "Operand" );
	operand->addTraits( INTEGER, -1 );
	operand->addTraits( FLOAT, -1 );
	operand->addTraits( DOUBLE, -1 );
	_addInputPort( operand );

	result = new OutputPort( "Result" );
	_addOutputPort( result );
}

Boolean_ 			/* virtual */
UnaryMathFU::inputTraitsValid()
{
	/*
	 * Because many of the unary mathematical operations return
	 * double precision values, we simply force all data types
	 * to be double precision
	 */
	DataTraits dataT = operand->getTraits();

	DataTraits resultTraits( DOUBLE, dataT.getDimension() );
	result->setTraits( resultTraits );

	return SUCCESS_;
}

Boolean_                        /* virtual */
UnaryMathFU::loadConfigFromFile( const CString& fileName )
{
  	char opName[256];
  	int op;

  	FILE *fp = fopen( fileName.getValue(), "r" );
  	if ( fp == NULL ) {
    	    warning( "\nUnable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
  	} else {
	    fscanf( fp, "%*[^\n]\n" );  // Comment line

    	    fscanf( fp, "%[^\n]\n", opName );
            if ( ( op = UnaryOpNameList.lookup( opName ) ) < 0 ) {
                warning( "\n%s is not a valid operation name", opName );
            } else {
                _setOperator( op );	     // sets isReady flag if successful
            }
    	    fclose( fp );
  	}

  	if ( isReady == FALSE_ ) { 	    // one last chance to configure
            configureOperation();
	    configure();
  	}

  	return( isReady );
}

Boolean_ 			/* virtual */
UnaryMathFU::ready()
{
	return isReady;
}

void 				/* virtual */
UnaryMathFU::run(  Boolean_& /* errorFlag */ )
{
	Assert( operand->valueAvailable() );

        myOperator->getUnaryResult( *( operand->getValueP() ) , resultV );
	result->setValue( resultV );
	_writeOutput();
}

Boolean_ 			/* virtual */
UnaryMathFU::saveConfigToFile( const CString& fileName ) const
{
  	Boolean_ retCode = FAILURE_;

  	FILE *fp = fopen( fileName.getValue(), "w" );

  	if ( fp == NULL ) {
     	    error( "\nUnable to open %s: %s", fileName.getValue(), 
					      errorString() );
  	} else {
	    fprintf( fp, "# Operation Name\n" );
    	    fprintf( fp, "%s\n", 
		(const char *)UnaryOpNameList.getElement( myOperatorNumber ) );

    	   retCode = SUCCESS_;
    	   fclose( fp );
  	}

  	return retCode;
}
    
/*
 * 	Initialize the static data.   Only executed once.
 */
const char *const UnaryMathFU::MY_CLASS = "UnaryMathFU";
