/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 * Contributing Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * SynthesizeCoordinates.cc - A functional unit for populating a 
 * 2-dimensional array with coordinates of (a, b, ... z) values.
 * Suitable as input to the Scatter3D and XYGraph functional units.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/SynthesizeCoordinates.C,v 1.10 1994/03/15 16:42:08 aydt Exp $
 */

#include "SynthesizeCoordinates.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "InputPort.h"
#include "OutputPort.h"
#include "SystemErrors.h"

SynthesizeCoordinates::SynthesizeCoordinates()
{
	_setClassName( MY_CLASS );

	for ( int i = 0; i < MAXCOORDAXES; i++ ) {
	   coordValue[i] = NULL;
	}
	array = NULL;
}

SynthesizeCoordinates::~SynthesizeCoordinates()
{
	for ( int i = 0; i < indexCount; i++ ) {
	    delete coordValue[i];
	}
	delete array;
}

void
SynthesizeCoordinates::_increaseSize()
{
	/* 
	 * Increase the size of the array by one element and
	 * copy the cells from the old array to the new.  
	 * numCells always reflects the number of cells in the
	 * outputValue array.
	 */
	Array *oldArrayP = (Array *) outputValue;
	Array *newArrayP = new Array( DOUBLE, 2 );

	int newDimSize[2];
	newDimSize[0] = numCells + 1;
	newDimSize[1] = indexCount;
	newArrayP->setDimSizes( newDimSize );

	int i, j;
	for ( i = 0; i < numCells; i++ ) {
	   for (j = 0; j < indexCount; j++) {
	      newArrayP->setCellValue( oldArrayP->getCellValue(i, j), i, j );
	   }
	}

	outputValue = newArrayP;
	delete newArrayP;

	numCells++;
}

void
SynthesizeCoordinates::_reduceArraySize()
{
	/*
	 * numCells is the size of our active array
	 * maxCells is the size of our new array
	 * arrayIndex-1 is the latest entry in the active array.
	 *
	 * We downsize our array, saving the latest "maxCells-1" entries -
	 * the oldest entry ends up in position 0 in the new array.
	 */
	 
	Assert( numCells > maxCells );

	Array *oldArrayP = (Array *) outputValue;
	Array *newArrayP = new Array( DOUBLE, 2 );

	int dimSize[2];
	dimSize[0] = maxCells;
	dimSize[1] = indexCount + 1;
	newArrayP->setDimSizes( dimSize );

	int newIndex = maxCells - 2;
	int oldIndex = arrayIndex - 1;
	while ( newIndex >= 0 ) {
	    if ( oldIndex < 0 ) {
		oldIndex = numCells - 1;
	    }
	    
	    for (int i = 0; i < indexCount; i++) {
	       newArrayP->setCellValue( oldArrayP->getCellValue( oldIndex, i ),
				        newIndex, i );
	    }

	    oldIndex--;
	    newIndex--;
	}
	outputValue = newArrayP;
	delete newArrayP;

	numCells = maxCells;
	arrayIndex = numCells - 1;	// fill last cell next time run

}

void
SynthesizeCoordinates::_setPorts( int portCount )
{
	Assert( indexCount == -1 );

	char tmpName[50];
	for (int i = 0; i < portCount; i++) {
	   sprintf(tmpName, "Coordinate-%d", i); 

	   coordValue[i] = new InputPort( tmpName );

	   coordValue[i]->addTraits( INTEGER, 0 );
	   coordValue[i]->addTraits( FLOAT, 0 );
	   coordValue[i]->addTraits( DOUBLE, 0 );

	   _addInputPort( coordValue[i] );
	}


	indexCount = portCount;
}

void
SynthesizeCoordinates::_shuffleEntries()
{
	/* 
	 * numCells is the size of our active array.
	 * arrayIndex is the oldest entry in the active array.
	 *
	 * Shuffle the entries in the array so that the oldest entry
	 * is in position 0 and the most recent entry is in position 
	 * numCells-1.  This allows the array to grow and maintain 
	 * the most history possible before overwriting.
	 */
	if ( arrayIndex == numCells ) { 	// no need to shuffle
	    return;
	}

	Array *oldArrayP = (Array *) outputValue;
	Array *newArrayP = new Array( oldArrayP );

	int newIndex = 0;
	int oldIndex = arrayIndex;

	while ( newIndex < numCells ) {
	    if ( oldIndex == numCells ) {
		oldIndex = 0;
	    }

	    for (int i = 0; i < indexCount; i++) {
	       newArrayP->setCellValue( oldArrayP->getCellValue( oldIndex, i ),
				        newIndex, i );
	    }

	    oldIndex++;
	    newIndex++;
	}
	outputValue = newArrayP;
	delete newArrayP;

	arrayIndex = numCells;		// fill last pair next time run
}

void
SynthesizeCoordinates::configureOperation()
{
	/*
	 * We only want to allow user to select if this is the
	 * initial configuration - else just display the choice!
	 */
	FUParams coordParams;
	char question[128];

	if ( indexCount == -1 ) {
	    sprintf( question, "Number of coordinate axes (1-%d)", 
								MAXCOORDAXES );
 	    coordParams.addTextParam( question, BaseFUParamEntry::Integer, 2 );
	} else {
	    char tmpString[50];
	    sprintf( tmpString, "%d", indexCount ); 
	    coordParams.addDisplayParam( "Number of coordinate axes", 
							       tmpString );
	}

	ParamConfig pc( Pablo::TopLevel(), coordParams, getName() );
	pc.run();

	if ( indexCount == -1 ) {
 	    BaseFUParamEntry& coordEntry = coordParams.getEntry( question );

	    int tmpCount;

	    if ( coordEntry.valueIsValid() ) {
		tmpCount = coordEntry.getValue().getInteger();
		if ( tmpCount > MAXCOORDAXES ) {
		    error( "Maximum number of axes supported is %d", 
								MAXCOORDAXES );
		    tmpCount = MAXCOORDAXES;
		}
	    } else {
	       tmpCount = 2;
	    } 

	    _setPorts( tmpCount );
	}
}

void                             /* virtual */
SynthesizeCoordinates::configure()
{
  	FUParams params;
	params.addRadioButtonsParam( "History Option", 
				     BaseFUParamEntry::Integer, 0,
				     HistoryOption );
  	params.addTextParam( 
           "Maximum Number of Data Points (Used only if Finite History)",
	    BaseFUParamEntry::Integer, maxCells );

  	ParamConfig pc( Pablo::TopLevel(), params, getName() );
 	BaseFUParamEntry& optionEntry = params.getEntry( "History Option" );
  	BaseFUParamEntry& maxEntry = params.getEntry( 
 	 "Maximum Number of Data Points (Used only if Finite History)" );

	if ( histOption != -1 ) {
	    optionEntry.setValue( histOption );
	}

  	pc.run();

        if ( optionEntry.valueIsValid() ) {
            histOption = optionEntry.getValue().getInteger();
        } else {
            warning( "Invalid history option - defaulting to Infinite") ;
	    histOption = INFINITE;
        }

	if ( histOption == FINITE ) {
  	    if ( maxEntry.valueIsValid() ) {
    	        maxCells = maxEntry.getValue().getInteger();

    	        if ( maxCells <= 0 ) {
      	             warning( 
			"Maximum Number must be positive. Defaulting to 1." );
		     maxCells = 1;
    	      	}
  	    } else {
    	    warning( "Invalid Maximum Number of Coordinates. Defaulting to 1" );
		maxCells = 1;
	    }

	    /* 
	     * If we've gone to a smaller array, resize & save most recent
	     * history entries.
	     * If we're going to a larger array, shuffle entries so most
	     * recent is in last array position.
	     */
	    if ( maxCells < numCells ) {
	        _reduceArraySize();
	    } else if ( maxCells > numCells ) {
		_shuffleEntries();
	    }
	} else {
	    _shuffleEntries();		// Preserve as much history as possible
	    maxCells = 1;		// Never used if infinite!
	}

	isConfigured = TRUE_;
}

FunctionalUnit *		/* virtual */
SynthesizeCoordinates::copy()
{
	SynthesizeCoordinates *copy = new SynthesizeCoordinates();
	return copy;
}

void 				/* virtual */
SynthesizeCoordinates::init()
{
	/*
	 * The HistoryOption list is a static so only set it the first time
	 * an instance of this class is created.
	 */
	if ( HistoryOption.isEmpty() ) {
            HistoryOption.addElement( "Infinite" );
            HistoryOption.addElement( "Finite" );
	}

	/*
	 * Continue by clearing up any leftovers that will be around if 
	 * init() isn't being called for the first time.  The goal is to
	 * start with a 'virgin' FU.
	 */

	for (int i = 0; i < MAXCOORDAXES; i++) {
	   if ( coordValue[i] != NULL ) {
	      delete coordValue[i];
	   }
	}
	if ( array  != NULL ) {
	   delete array;
	}

	/*
	 * Here's the real code to initialize the FU; In this FU we create our
	 * InputPorts in either configureOperation() or loadConfiguration()
	 */
	array = new OutputPort( "Set of Coordinates" );
	_addOutputPort( array );

	outputValue = new Array( DOUBLE, 2 );

	isConfigured = FALSE_;
	indexCount = -1;
	numCells = 0;			// Coordinates in array
	arrayIndex = 0;			// Next index to insert at
	maxCells = 1;			// Max triples if FINITE
	histOption = -1;		// Not yet selected
}

Boolean_			/* virtual */
SynthesizeCoordinates::inputTraitsValid()
{
        /*
         * SynthesizeCoordinates will always output an array with
	 * Traits=( DOUBLE, 2 ), and our input ports have types
	 * that are restrictive enough to guarantee any bindings
	 * are valid without further verification.
         */
        DataTraits arrayTraits( DOUBLE, 2 );
        array->setTraits( arrayTraits );

        return ( TRUE_ );
}

Boolean_
SynthesizeCoordinates::loadConfigFromFile( const CString& fileName )
{
  	FILE *fp = fopen( fileName.getValue(), "r" );

	if ( fp == NULL ) {
    	    warning( "\nUnable to open %s: %s\n", fileName.getValue(),
	    					errorString() );
	
	} else { 
	    char   opName[256];
	    int	   portCount;

	    fscanf( fp, "%*[^\n]\n" );      // Comment line

    	    if ( fscanf( fp, "%d %d %s\n", &maxCells, &portCount, 
						opName ) == 3 ) {
		if ( ( histOption = HistoryOption.lookup( opName ) ) < 0 ) {
                    warning( 
			"Invalid history option (%s) - defaulting to Infinite",
		         opName );
		    histOption = INFINITE;
                } else {
		    isConfigured = TRUE_;
		    _setPorts( portCount );
		}
    	    }
    	    fclose( fp );
  	}

	if (  indexCount == -1 ) {
	   configureOperation();
	}
	if ( isConfigured == FALSE_ ) {		// force manual configuration
	   configure();
	}

  	return( isConfigured );
}

Boolean_ 			/* virtual */
SynthesizeCoordinates::ready()
{
	return( isConfigured );
}

void 				/* virtual */
SynthesizeCoordinates::run(  Boolean_& /* errorFlag */ )
{
	/* 
	 * If we keep infinite history, then we always increase the
	 * size of the array & insert at the end of the array.
	 * If we keep finite history, then we increase the size of the
	 * array until the max size has been reached.  After that,
	 * we start wrapping around.
	 */
	if ( histOption == INFINITE ) {
	    _increaseSize();
	} else {
	    if ( numCells < maxCells ) {
		_increaseSize();
	    } else {
	        if ( arrayIndex >= maxCells ) {
		   arrayIndex = 0;
		}
	    }
	}

	Array* outputArrayP = (Array *)outputValue;

	for (int i = 0; i < indexCount; i++) {
	   Assert( coordValue[i]->valueAvailable() );
  	   outputArrayP->setCellValue( coordValue[i]->getValue(), 
				       arrayIndex, i );
	}

	arrayIndex++;

	array->setValue( outputValue );
	_writeOutput();
}


Boolean_                        /* virtual */
SynthesizeCoordinates::saveConfigToFile( const CString& fileName ) const
{
  	Boolean_ retCode;

  	FILE *fp = fopen( fileName.getValue(), "w" );

  	if ( fp == NULL ) {
    	    error( "\nUnable to open %s: %s\n", fileName.getValue(),
	  				errorString() );
    	    retCode = FAILURE_;
  	} else {
	    fprintf( fp, "# Max Entries, Coord Count , History Option\n" );
    	    fprintf( fp, "%d %d %s\n", maxCells, indexCount,
		(const char *) HistoryOption.getElement( histOption ) );
 	    	
    	    fclose( fp );
    	    retCode = SUCCESS_;
  	}

  	return retCode;
}

/*
 * 	Initialize the static data.   Only executed once.
 */
const char *const SynthesizeCoordinates::MY_CLASS = "SynthesizeCoordinates";
CStringObjList SynthesizeCoordinates::HistoryOption;
