/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * Scatter3DFU.cc - A polymorophic functional unit for a dynamic 3D
 *		    scatter plot
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/Scatter3DFU.C,v 1.24 1994/03/15 16:42:01 aydt Exp $
 */

#include "Scatter3DFU.h"
#include "Scatter3DWrapper.h"

#include "FUParams.h"
#include "ParamConfig.h"
#include "InputPort.h"
#include "SystemErrors.h"

#define NormX(x) ( ( ((double) x) - minX ) / (maxX - minX) )
#define NormY(y) ( ( ((double) y) - minY ) / (maxY - minY) )
#define NormZ(z) ( ( ((double) z) - minZ ) / (maxZ - minZ) )
#define Round(x) ( x >= 0.0 ? (int)(x + .5) : (int)(x - .5) )


Scatter3DFU::Scatter3DFU()
{
	_setClassName( MY_CLASS );
	input = NULL;
	dialog = NULL;
	scatter3d = NULL;
}

Scatter3DFU::~Scatter3DFU()
{
	delete input;
	delete scatter3d;
	delete dialog;
}

void 				/* virtual */
Scatter3DFU::configure()
{

#ifndef XtNautoScale
#define XtNautoScale	"autoScale"
#endif

#ifndef XtNminXValue
#define XtNminXValue	"minXValue"
#define XtNmaxXValue	"maxXValue"
#define XtNminYValue	"minYValue"
#define XtNmaxYValue	"maxYValue"
#define XtNminZValue	"minZValue"
#define XtNmaxZValue	"maxZValue"
#define XtNmarkerSize	"markerSize"
#endif

	double 	fuMinX, fuMaxX, fuMinY, fuMaxY, fuMinZ, fuMaxZ;
	int	fuAutoScale, fuMarkerSize;
	char    *labelX, *labelY, *labelZ;

	if ( ! isConfigured ) {
		int wMinX, wMaxX, wMinY, wMaxY, wMinZ, wMaxZ;
			
		fuAutoScale = 1;		// Auto Scale = TRUE
       		scatter3d->getScatterAttr( XtNmarkerSize,(XtArgVal)&fuMarkerSize);

		scatter3d->getScatterAttr( XtNminXValue, (XtArgVal) &wMinX );
		scatter3d->getScatterAttr( XtNmaxXValue, (XtArgVal) &wMaxX );
		scatter3d->getScatterAttr( XtNminYValue, (XtArgVal) &wMinY );
		scatter3d->getScatterAttr( XtNmaxYValue, (XtArgVal) &wMaxY );
 		scatter3d->getScatterAttr( XtNminZValue, (XtArgVal) &wMinZ );
                scatter3d->getScatterAttr( XtNmaxZValue, (XtArgVal) &wMaxZ );

		/* defaults for min and max initially match widget values */
		fuMinX = wMinX;		
		fuMaxX = wMaxX;
		fuMinY = wMinY;
		fuMaxY = wMaxY;
		fuMinZ = wMinZ;
                fuMaxZ = wMaxZ;

		/* the offset & range values never change as the widget min */
		/* and max never change.  These are used for scaling data.  */
		xOffset = wMinX;	
		yOffset = wMinY;
		zOffset = wMinZ;
		xRange  = wMaxX - wMinX;
		yRange  = wMaxY - wMinY;
		zRange  = wMaxZ - wMinZ;

		isConfigured = TRUE_;
	} else {
		fuMarkerSize = markerSize;
		fuAutoScale = autoScale;
		fuMinX = minX;
		fuMaxX = maxX;
		fuMinY = minY;
		fuMaxY = maxY;
 		fuMinZ = minZ;
                fuMaxZ = maxZ;
	}

	scatter3d->getScatterLabel( L_X, &labelX );
	scatter3d->getScatterLabel( L_Y, &labelY );
	scatter3d->getScatterLabel( L_Z, &labelZ );

	FUParams 	params;
	CStringObjList	itemList;

	itemList.addElement("False");
	itemList.addElement("True");
 	params.addRadioButtonsParam( "Automatically scale",
				     BaseFUParamEntry::Integer, 1, itemList );

        params.addTextParam( "Data marker size",
				     BaseFUParamEntry::Integer, fuMarkerSize );

        params.addTextParam( "Minimum X", BaseFUParamEntry::Real, fuMinX );
        params.addTextParam( "Maximum X", BaseFUParamEntry::Real, fuMaxX );
        params.addTextParam( "Minimum Y", BaseFUParamEntry::Real, fuMinY );
        params.addTextParam( "Maximum Y", BaseFUParamEntry::Real, fuMaxY );
 	params.addTextParam( "Minimum Z", BaseFUParamEntry::Real, fuMinZ );
        params.addTextParam( "Maximum Z", BaseFUParamEntry::Real, fuMaxZ );
	params.addTextParam( "X Label", BaseFUParamEntry::Str, labelX );
	params.addTextParam( "Y Label", BaseFUParamEntry::Str, labelY );
	params.addTextParam( "Z Label", BaseFUParamEntry::Str, labelZ );

	ParamConfig pc( Pablo::TopLevel(), params, getName() );
	pc.run();

 	BaseFUParamEntry& scaleEntry = params.getEntry( "Automatically scale");
        BaseFUParamEntry& markerEntry = params.getEntry( "Data marker size");
 	BaseFUParamEntry& xMinEntry = params.getEntry( "Minimum X");
 	BaseFUParamEntry& xMaxEntry = params.getEntry( "Maximum X");
 	BaseFUParamEntry& yMinEntry = params.getEntry( "Minimum Y");
 	BaseFUParamEntry& yMaxEntry = params.getEntry( "Maximum Y");
	BaseFUParamEntry& zMinEntry = params.getEntry( "Minimum Z");
        BaseFUParamEntry& zMaxEntry = params.getEntry( "Maximum Z");
	BaseFUParamEntry& xLabelEntry = params.getEntry( "X Label" );
	BaseFUParamEntry& yLabelEntry = params.getEntry( "Y Label" );
	BaseFUParamEntry& zLabelEntry = params.getEntry( "Z Label" );

	scaleEntry.setValue( fuAutoScale );

	if (markerEntry.valueIsValid()) {
		fuMarkerSize = markerEntry.getValue().getInteger();
	} else {
		warning( "Scatter3D data marker size was not valid.\n") ;
	}
	if (scaleEntry.valueIsValid()) {
		fuAutoScale= scaleEntry.getValue().getInteger();
	} else {
		warning( "Scatter3D autoscale selection was not valid.\n") ;
	}
	if (xMinEntry.valueIsValid()) {
		fuMinX = xMinEntry.getValue().getReal();
	} else {
		warning( "Scatter3D X Minimum Value was not valid.\n") ;
	}
	if (xMaxEntry.valueIsValid()) {
		fuMaxX = xMaxEntry.getValue().getReal();
	} else {
		warning( "Scatter3D X Maximum Value was not valid.\n") ;
	}
	if (yMinEntry.valueIsValid()) {
		fuMinY = yMinEntry.getValue().getReal();
	} else {
		warning( "Scatter3D Y Minimum Value was not valid.\n") ;
	}
	if (yMaxEntry.valueIsValid()) {
		fuMaxY = yMaxEntry.getValue().getReal();
	} else {
		warning( "Scatter3D Y Maximum Value was not valid.\n") ;
	}
 	if (zMinEntry.valueIsValid()) {
                fuMinZ = zMinEntry.getValue().getReal();
        } else {
                warning( "Scatter3D Z Minimum Value was not valid.\n") ;
        }
        if (zMaxEntry.valueIsValid()) {
             fuMaxZ = zMaxEntry.getValue().getReal();
        } else {
                warning( "Scatter3D Z Maximum Value was not valid.\n") ;
        }

	autoScale = fuAutoScale;
	markerSize = fuMarkerSize;
	minX = fuMinX;
	maxX = fuMaxX;
	minY = fuMinY;
	maxY = fuMaxY;
 	minZ = fuMinZ;
        maxZ = fuMaxZ;

	scatter3d->setScatterLabel( L_X, xLabelEntry.getValue().getString() );
	scatter3d->setScatterLabel( L_Y, yLabelEntry.getValue().getString() );
	scatter3d->setScatterLabel( L_Z, zLabelEntry.getValue().getString() );

	if (autoScale == 0)
	   scatter3d->setScatterAttr( XtNautoScale, FALSE );
	else
	   scatter3d->setScatterAttr( XtNautoScale, TRUE );

	scatter3d->setScatterAttr( XtNmarkerSize, markerSize );
        scatter3d->setPerfWidgetColors();
}

FunctionalUnit *		/* virtual */
Scatter3DFU::copy()
{
	Scatter3DFU *copy = new Scatter3DFU();
	return copy;
}


void 			/* virtual */
Scatter3DFU::init()
{
	if ( input != NULL ) {
	    delete input;
	}
	if ( scatter3d != NULL ) {
	    delete scatter3d;
	}
	if ( dialog != NULL ) {
	    delete dialog;
	}

	isConfigured = FALSE_;

	input =  new InputPort( "Input" );
	input->addTraits( INTEGER, 2 );
	input->addTraits( FLOAT, 2 );
	input->addTraits( DOUBLE, 2 );
	_addInputPort( input );

	dialog = _getTitledFormDialog( "FUDialog", getName() );
	scatter3d = new Scatter3DWrapper( dialog, NullArgs, "Contents" );

	dialog->manage();
	XtVaSetValues( dialog->getWidget(), XmNdefaultPosition, False, NULL );
}

Boolean_		/* virtual */
Scatter3DFU::loadConfigFromFile( const CString&  fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
        } else {
	    int x, y, width, height;
	    int labelSize;
	    char buf[256];

	    fscanf( fp, "%*[^\n]\n" );
            if ( fscanf( fp, "%lf %lf %lf %lf %lf %lf %d %d %d %d %d %d\n",
                           &minX, &maxX, &minY, &maxY, &minZ, &maxZ, 
			   &autoScale, &markerSize,
			   &x, &y, &width, &height ) != 12 ) {
                warning( "Unable to read configuration information from %s\n",
                          fileName.getValue() );
            } else {
	        dialog->unmanage();

		// ----- Comment Line; X Label Size and String
		labelSize = 0;
                fscanf( fp, "%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		if ( labelSize == 0 ) {
		    strcpy( buf, "\0" );
		}
                scatter3d->setScatterLabel( L_X, buf );

		// ----- Comment Line; Y Label Size and String
		labelSize = 0;
                fscanf( fp, "\n%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		if ( labelSize == 0 ) {
		    strcpy( buf, "\0" );
		}
                scatter3d->setScatterLabel( L_Y, buf );

		// ----- Comment Line; Z Label Size and String
		labelSize = 0;
                fscanf( fp, "\n%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		if ( labelSize == 0 ) {
		    strcpy( buf, "\0" );
		}
                scatter3d->setScatterLabel( L_Z, buf );

		/*
                 * We recalculate the Offset and Range based on the widget
		 * min and max values.  Then we use the parameters we 
		 * loaded from the file to get other resources.
                 * Finally, initialize the static color table.
                 */

     		int wMinX, wMaxX, wMinY, wMaxY, wMinZ, wMaxZ;
		scatter3d->getScatterAttr( XtNminXValue, (XtArgVal) &wMinX );
		scatter3d->getScatterAttr( XtNmaxXValue, (XtArgVal) &wMaxX );
		scatter3d->getScatterAttr( XtNminYValue, (XtArgVal) &wMinY );
		scatter3d->getScatterAttr( XtNmaxYValue, (XtArgVal) &wMaxY );
 		scatter3d->getScatterAttr( XtNminZValue, (XtArgVal) &wMinZ );
                scatter3d->getScatterAttr( XtNmaxZValue, (XtArgVal) &wMaxZ );

		xOffset = wMinX;	
		yOffset = wMinY;
		zOffset = wMinZ;
		xRange  = wMaxX - wMinX;
		yRange  = wMaxY - wMinY;
		zRange  = wMaxZ - wMinZ;

		if (autoScale == 0) {
	   		scatter3d->setScatterAttr( XtNautoScale, FALSE );
		} else {
	   		scatter3d->setScatterAttr( XtNautoScale, TRUE );
		} 
		scatter3d->setScatterAttr( XtNmarkerSize, markerSize );
        	scatter3d->setPerfWidgetColors();

		dialog->manage();
		scatter3d->setPerfWidgetPosition( x, y, width, height );

                isConfigured = TRUE_;
	    }
            fclose( fp );
        }

	if ( ! isConfigured ) {
	    configure();
	}

        return isConfigured;
}

Boolean_ 		/* virtual */
Scatter3DFU::ready()
{
	return isConfigured;
}

void 			/* virtual */
Scatter3DFU::run(  Boolean_& errorFlag )
{
	Assert( input->valueAvailable() );

	const Value *valueP = input->getValueP();

	if ( valueP->getTraits().isScalar() ) {
		warning( "Scatter3D needs an array as input" );
		errorFlag = TRUE_;
		return;
	} else {
	    if ( valueP->getTraits().getDimension() != 2 ) {
		warning( "Scatter3D needs a 2-d array as input" );
		errorFlag = TRUE_;
		return;
	    }
	}

	const Array *a = valueP->operator Array*();
	const int *dimSizes = a->getDimSizes();

	if ( dimSizes[1] != 3) {
	   warning(
	     "Scatter3D only accepts points specified with (X,Y,Z) triples." );
		errorFlag = TRUE_;
	   return;
	}

	int nTriples = dimSizes[0];
	int *xptr = new int [ nTriples ];
	int *yptr = new int [ nTriples ];
	int *zptr = new int [ nTriples ];

	for ( int p = 0; p < nTriples; p++ ) {
	   double xin = a->getCellValue(p,0);
	   double yin = a->getCellValue(p,1);
	   double zin = a->getCellValue(p,2);
	   xptr[p] = Round(( NormX(xin) * xRange )) + xOffset;
	   yptr[p] = Round(( NormY(yin) * yRange )) + yOffset;
 	   zptr[p] = Round(( NormZ(zin) * zRange )) + zOffset;

	   /******* DEBUGGING ***********
	    cout << form( "Scatter3D::RUN Triple number=%d; Xin=%f; Xout=%d\n", 
			  p, xin, xptr[p] );
	   cout << form( "                                 Yin=%f; Yout=%d\n",
			  yin, yptr[p] );
	   cout << form( "                                 Zin=%f; Zout=%d\n",
                          zin, zptr[p] );
	   ******* DEBUGGING ***********/

	}

	scatter3d->setPerfWidgetValues( xptr, yptr, zptr, nTriples, 0 );

	delete[] xptr;
	delete[] yptr;
	delete[] zptr;
}

Boolean_		/* virtual */
Scatter3DFU::saveConfigToFile( const CString& fileName ) const
{
        Boolean_ result;
        FILE *fp = fopen( fileName.getValue(), "w" );

        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            result = FAILURE_;
        } else {
      	    int     x, y, width, height;
	    char *label;
 	    scatter3d->getPerfWidgetPosition( &x, &y, &width, &height );

            fprintf( fp, 
	     "# MinX MaxX MinY MaxY MinZ MaxZ AutoScale MarkSz X Y Wdth Hght\n"
	     );
            fprintf( fp, "%lf %lf %lf %lf %lf %lf %d %d %d %d %d %d\n",
                           minX, maxX, minY, maxY, minZ, maxZ, 
			   autoScale, markerSize,
			   x, y, width, height );

            scatter3d->getScatterLabel( L_X, &label );
            fprintf( fp, "# X Label\n" );
            fprintf( fp, "%d %s\n", strlen( label), label );

            scatter3d->getScatterLabel( L_Y, &label );
            fprintf( fp, "# Y Label\n" );
            fprintf( fp, "%d %s\n", strlen( label ), label );

            scatter3d->getScatterLabel( L_Z, &label );
            fprintf( fp, "# Z Label\n" );
            fprintf( fp, "%d %s\n", strlen( label ), label );

            fclose( fp );
            result = SUCCESS_;
        }
        return result;
}
	

// Global class definition
const char *const Scatter3DFU::MY_CLASS = "Scatter3DFU";
