/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Authors: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *                       Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * ContourFU.cc - A polymorophic functional unit for a dynamic contour display
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/ContourFU.C,v 1.26 1994/03/15 16:41:33 aydt Exp $
 */

#include <stdlib.h>

#include "ContourFU.h"
#include "ContourFormWrapper.h"

#include "FUParams.h"
#include "InputPort.h"
#include "ParamConfig.h"
#include "ParamDisplay.h"
#include "SystemErrors.h"


ContourFU::ContourFU()
{
	_setClassName( MY_CLASS );

	inputPort = NULL;
	dialog = NULL;
	contourForm = NULL;
}

ContourFU::~ContourFU()
{
	delete inputPort;
	delete contourForm;
	delete dialog;
}

void 
ContourFU::configure()
{
#ifndef XtNrows
#define XtNrows       "rows"
#define XtNcolumns    "columns"
#endif

#ifndef XtNcontourCnt
#define XtNcontourCnt "contourCnt"
#endif

	/* 
	 * The initial number of rows and columns are selectable by the user, 
	 * and both the Contour widget and the ContourFU agree on what they are.
	 * If an Array with larger row and/or column count is seen, the
	 * display will resize to accomodate the new dimensions.
	 */

	int fuRows, fuCols, fuCnt, fuUpdate;
        const char *labelBottom, *labelLeft;

	if ( ! isConfigured ) {
	    contourForm->getContourAttr( XtNrows, (XtArgVal) &fuRows );
	    contourForm->getContourAttr( XtNcolumns, (XtArgVal) &fuCols );
	    contourForm->getContourAttr( XtNcontourCnt,(XtArgVal) &fuCnt );
	    fuUpdate = 1;

	    isConfigured = TRUE_;
	} else {
	    fuRows = numRows;
	    fuCols = numCols;
	    fuCnt  = contourCount;
	    fuUpdate = updateRate;
	}

	// --- CONFIGURATION PARAMETERS --------------------------------------
        contourForm->getContourLabel( L_BOTTOM_HORIZ, &labelBottom );
        contourForm->getContourLabel( L_LEFT_VERT, &labelLeft );

	FUParams params;
	params.addTextParam( "Initial Number of Rows", 
				BaseFUParamEntry::Integer, fuRows );
	params.addTextParam( "Initial Number of Columns", 
				BaseFUParamEntry::Integer, fuCols );
	params.addTextParam( "Number of Contours", 
				BaseFUParamEntry::Integer, fuCnt );
	params.addTextParam( "Iterations between Updates",
				BaseFUParamEntry::Integer, fuUpdate );
        params.addTextParam( "Vertical axis label",
			     BaseFUParamEntry::Str, labelLeft);
        params.addTextParam( "Horizontal axis label",
			     BaseFUParamEntry::Str, labelBottom );
	CString configPanelName = "Contour FU Config: " + getName();
        ParamConfig pc( Pablo::TopLevel(), params, configPanelName );
	pc.run();

	BaseFUParamEntry& rowEntry = 
				params.getEntry( "Initial Number of Rows" );
	BaseFUParamEntry& colEntry = 
				params.getEntry( "Initial Number of Columns" );
        BaseFUParamEntry& cntEntry = 
				params.getEntry( "Number of Contours" );
        BaseFUParamEntry& updateEntry = 
				params.getEntry( "Iterations between Updates" );
        BaseFUParamEntry &vertLabelEntry =
		                params.getEntry( "Vertical axis label" );
        BaseFUParamEntry &horizLabelEntry = 
		                params.getEntry( "Horizontal axis label" );

        // === PARAMETER ERROR-CHECKING =======================================
        if ( rowEntry.valueIsValid() ) {
	    fuRows = rowEntry.getValue().getInteger();
	} else {
	    warning( "Row entry was invalid.\n" );
	}
	if ( colEntry.valueIsValid() ) {
	    fuCols = colEntry.getValue().getInteger();
	} else {
	    warning( "Column entry was invalid.\n" );
	}
	if ( cntEntry.valueIsValid() ) {
	    fuCnt = cntEntry.getValue().getInteger();
	} else {
	    warning( "Number of Contours value was invalid.\n" );
	}
	if ( updateEntry.valueIsValid() ) {
	    fuUpdate = updateEntry.getValue().getInteger();
	} else {
	    warning( "Iterations between Updates was invalid.\n" );
	}
	if ( fuRows > 1) {
	    numRows = fuRows;
	} else {
	    warning( "Row entry must be greater than  1. Resetting to 2." );
	    numRows = 2;
	}
	if ( fuCols > 1) {
	    numCols = fuCols;
	} else {
            warning( "Column entry must be greater than 1. Resetting to 2." );
	    numCols = 2;
	}
	if ( fuUpdate > 0) {
	    updateRate = fuUpdate;
	} else {
            warning( "Iteration count must be greater than 0. Resetting to 1.");
	    updateRate = 1;
	}

        // --- FU AND WIDGET CONFIGURATION DATA SETUPS -----------------------
	contourCount = fuCnt;

	contourForm->setContourLabel( L_LEFT_VERT, 
	 		      vertLabelEntry.getValue().getString() );
        contourForm->setContourLabel( L_BOTTOM_HORIZ, 
			      horizLabelEntry.getValue().getString() );
        contourForm->setContourAttr( XtNrows, numRows );
	contourForm->setContourAttr( XtNcolumns, numCols );
	contourForm->setContourAttr( XtNcontourCnt, contourCount );
        contourForm->setPerfWidgetColors();
}

FunctionalUnit *
ContourFU::copy()
{
	ContourFU *copy = new ContourFU();
	return copy;
}

void 
ContourFU::fuCallback( int iRow, int iCol )
{
	if ( inputValue.isUndefined() ) {
	    return;
	}

	FUParams params;
	Value callbackDataValue;

    	if ( ( iRow >=0 ) && ( iCol >=0 ) ) {
	    Array *arrayInputValue = (Array *)inputValue;
            const int *dimSizes = arrayInputValue->getDimSizes();

            if ( inputValue.getTraits().getDimension() == 2 ) {
                if ( ( iRow < dimSizes[0] ) && ( iCol < dimSizes[1] ) ) {
                    arrayInputValue->getCellValue( callbackDataValue,
						  iRow, iCol );  
                    params.addDisplayParam( "Current Contour Cell Value",
						  callbackDataValue );
	        } else {
                    params.addDisplayParam( "Current Contour Cell Value",
						  noDataMSG );
	        }

                params.addDisplayParam( "Contour Row", iRow );
                params.addDisplayParam( "Contour Column", iCol );

                CString callbackTitle = "Callback: " + getName();
                ParamDisplay *pd = new ParamDisplay( Pablo::TopLevel(), params,
						     callbackTitle );
            } 
	}
}

void 
ContourFU::init()
{
	if ( inputPort != NULL ) {
	    delete inputPort;
	}
	if ( contourForm != NULL ) {
	    delete contourForm;
	}
	if ( dialog != NULL ) {
	    delete dialog;
	}
	
	inputPort = new InputPort( "Input" );
	inputPort->addTraits( INTEGER, 2 );
	inputPort->addTraits( FLOAT, 2 );
	inputPort->addTraits( DOUBLE, 2 );
	_addInputPort( inputPort );

	dialog = _getTitledFormDialog( "FUDialog", getName() );
	contourForm = new ContourFormWrapper( dialog, this, NullArgs,
							    "Contents" );
	dialog->manage();
	XtVaSetValues( dialog->getWidget(), XmNdefaultPosition, False, NULL );

	isConfigured = FALSE_;
}

Boolean_ 			/* virtual */
ContourFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "\nUnable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
        } else {
	    int x, y, width, height;
	    int labelSize;
	    char buf[LABEL_MAX_LEN];

            fscanf( fp, "%*[^\n]\n" );
            if ( fscanf( fp, "%d %d %d %d %d %d %d %d\n", 
			   &numRows, &numCols, &contourCount, &updateRate,
			   &x, &y, &width, &height ) != 8 ) {
		warning( "Unable to read configuration information from %s\n",
			  fileName.getValue() );
	    } else {
		dialog->unmanage();
		labelSize = 0;
		fscanf(fp, "%*[^\n]\n" );
		fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		if ( labelSize == 0 ) {
		    strcpy( buf, "\0" );
	    	}
	        contourForm->setContourLabel( L_BOTTOM_HORIZ, buf );

		labelSize = 0;
		fscanf(fp, "\n%*[^\n]\n" );
		fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		if ( labelSize == 0 ) {
		    strcpy( buf, "\0" );
	    	}
	        contourForm->setContourLabel( L_LEFT_VERT, buf );

                /*
 	 	 * We configure based on parameters we loaded from the file,
	 	 * and the static color table.
	 	 */
        	contourForm->setContourAttr( XtNrows, numRows );
        	contourForm->setContourAttr( XtNcolumns, numCols );
        	contourForm->setContourAttr( XtNcontourCnt, contourCount );
        	contourForm->setPerfWidgetColors(); 

		dialog->manage();
                contourForm->setPerfWidgetPosition( x, y, width, height );

		isConfigured = TRUE_;
            } 
	    fclose( fp );
        }

	if ( ! isConfigured ) {
	    configure();
	}
        return isConfigured; 
}

Boolean_ 
ContourFU::ready()
{
	return isConfigured;
}

void 			/* virtual */
ContourFU::run(  Boolean_& errorFlag )
{
	Assert( inputPort->valueAvailable() );

	/*
	 * Do we really run this time?  If yes, reset counter, read in
	 * data, increase display rows and/or columns if new data out of 
	 * range, fill in array of floats with data values for each 
	 * (row, col) of display, and update the display.  Note that we
	 * base the size of our array on numRows and numCols which may
	 * be greater than the size of our inputValue as given by nx and ny.
	 */
	if ( ++updateCount >= updateRate ) {
 	    updateCount = 0;

	    inputValue = inputPort->getValue();	// inputValue used in fuCallback
	    Array *a = (Array *) inputValue;
	    const int *dimSizes = a->getDimSizes();
	    int nx = dimSizes[0];
	    int ny = dimSizes[1];

	    if ( nx > numRows ) {
	        numRows = nx;
	        contourForm->setContourAttr( XtNrows, numRows );
	    }
	    if ( ny > numCols ) {
	        numCols = ny;
	        contourForm->setContourAttr( XtNcolumns, numCols );
	    }

	    float *vals = (float *)malloc( numRows * numCols * sizeof(float) );
	    if ( vals == NULL ) {
	    	warning( "Cannot allocate memory for values." );
		errorFlag = TRUE_;
	    } else {
	        float *vp = vals;
	        int    x,y;

	        for ( x = 0; x < numRows; x++ ) {
	      	    for ( y = 0; y < numCols; y++ ) {
			if ( ( x < nx ) && ( y < ny ) ) {
	         	    *vp++ = (float)a->getCellValue( x, y );
	      		} else {
			    *vp++ = 0.0;
			}
	   	    }
		}

	   	contourForm->setDisplayValues( vals );
	   	free( vals );
	    }
	} else {
	    /*
	     * We never use the value if we don't update the display, but
	     * we do get a point to it so that the status of the input port
	     * will be reset from NEW to USED.
	     */
	    const Value *vp = inputPort->getValueP();	
	}
}

Boolean_		/* virtual */
ContourFU::saveConfigToFile( const CString& fileName ) const
{
	Boolean_ result;
        FILE *fp = fopen( fileName.getValue(), "w" );

        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
	    result = FAILURE_;
        } else {
	    int         x, y, width, height; 
	    const char *label;

	    contourForm->getPerfWidgetPosition( &x, &y, &width, &height );

	    fprintf( fp, "# Rows, Cols, NumContour, Iter, X, Y, Wdth, Hght\n" );
            fprintf( fp, "%d %d %d %d %d %d %d %d\n", 
	      		  numRows, numCols, contourCount, updateRate, 
			  x, y, width, height );

            contourForm->getContourLabel( L_BOTTOM_HORIZ, &label );
            fprintf( fp, "# Horizontal Label\n" );
            fprintf( fp, "%d %s\n", strlen( label ), label );

            contourForm->getContourLabel( L_LEFT_VERT, &label );
            fprintf( fp, "# Vertical Label\n" );
            fprintf( fp, "%d %s\n", strlen( label ), label );

   	    fclose( fp );
	    result = SUCCESS_;
        }
        return result;
}

/*
 * 	Initialize the static data.   Only executed once.
 */
const char *const ContourFU::MY_CLASS = "ContourFU";

