/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Authors: Ruth A. Aydt  (aydt@cs.uiuc.edu)
 *                       Keith A. Shields (shields@cs.uiuc.edu)
 *                       Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * ChartFU.cc - A polymorophic functional unit for a dynamic strip chart.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/ChartFU.C,v 1.20 1994/03/15 16:41:31 aydt Exp $
 */

#include "ChartFU.h"
#include "ChartFormWrapper.h"

#include "InputPort.h"
#include "SystemErrors.h"

#define Round(x)	( (int)(x + .5) )

ChartFU::ChartFU()
{
	_setClassName( MY_CLASS );
	inputPort = NULL;
	dialog = NULL;
	chartForm = NULL;
}

ChartFU::~ChartFU()
{
	delete inputPort;
	delete chartForm;
	delete dialog;
}

void                  /* virtual */
ChartFU::configure()
{
        /* 
	 * Check and store the input port dimension for vector/array support.
	 * Our inputPort Traits should be set by now and do not change!
	 */
        dimension = inputPort->getTraits().getDimension();
}
        
FunctionalUnit *		/* virtual */
ChartFU::copy()
{
	ChartFU *copy = new ChartFU();
	return copy;
}

void 				/* virtual */
ChartFU::init()
{
	/*
	 * Start by claring up any 'leftovers' that will be around if
	 * init() isn't being called for the first time.  The goal is to
	 * start with a 'virgin' FU.
	 */
	if ( inputPort != NULL ) {
	    delete inputPort;
	}
	if ( chartForm != NULL ) {
	    delete chartForm;
	}
	if ( dialog != NULL ) {
	    delete dialog;
	}

	/* 
	 * Here's the real code to initialize the FU
	 */
	inputValue = Value::NOVALUE;

	inputPort = new InputPort( "Input" );
	inputPort->addTraits( INTEGER, 0 );
	inputPort->addTraits( INTEGER, 1 );
	inputPort->addTraits( INTEGER, 2 );
	inputPort->addTraits( FLOAT, 0 );
	inputPort->addTraits( FLOAT, 1 );
	inputPort->addTraits( FLOAT, 2 );
	inputPort->addTraits( DOUBLE, 0 );
        inputPort->addTraits( DOUBLE, 1 );
        inputPort->addTraits( DOUBLE, 2 );
	_addInputPort( inputPort );

	dialog = _getTitledFormDialog( "FUDialog", getName() );
	chartForm = new ChartFormWrapper( dialog, NullArgs, "Contents" );

	dialog->manage();
	XtVaSetValues( dialog->getWidget(), XmNdefaultPosition, False, NULL );
}
 
Boolean_ 		/* virtual */
ChartFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "\nUnable to open %s:\n%s", fileName.getValue(),
                                                errorString() );
	    configure();
        } else {
	    int    x, y, width, height;

	    fscanf( fp, "%*[^\n]\n" );		// Comment line
	    if ( fscanf( fp, "%d %d %d %d %d\n", 
	     		&x, &y, &width, &height, &dimension ) == 5 ) {
	     
                chartForm->setPerfWidgetPosition( x, y, width, height );
	    } else {
		warning( "\nUnable to read configuration information from %s", 
			  fileName.getValue() );
      	    }
	    fclose( fp );
        }
        return SUCCESS_;
}

Boolean_
ChartFU::ready()
{
	return TRUE_;
}

void 				/* virtual */
ChartFU::run( Boolean_& /* errorFlag */ )
{
	Assert( inputPort->valueAvailable() );
	inputValue = inputPort->getValue();

        switch ( dimension ) {
	    case SCALAR:
	       {
               int intValue = Round( (double)inputValue );
               chartForm->setChildValues( intValue );
	       }
	       break;

            case VECTOR:
            case ARRAY_2D:
	       {
               Array *aInputData = (Array *) inputValue;

               // ----- Get the number of elements in the vector or array.
               const int *aDimSizes = aInputData->getDimSizes();
               int nRows, nCols;

               if ( dimension == VECTOR ) {
                   nRows = 1;
                   nCols = aDimSizes[0];
               } else if ( dimension == ARRAY_2D ) {
                   nRows = aDimSizes[0];
                   nCols = aDimSizes[1];
	       }

               // ----- Reconfigure the Chart display if new size data
               if ( chartForm->sizeChanged( nCols, nRows ) == TRUE_) {
                   chartForm->resize( nCols, nRows );
                   chartForm->setNumberChildren( nCols, nRows );
               }
           
	       // ----- Fill the vector/array structure with input data
	       int nValues = nRows * nCols;
               int *vals = new int [ nValues ];

               for ( int i = 0; i < nValues; i++ ) { 
                   vals[ i ] =  Round( (double)aInputData->getTheCellValue(i) );
               }

	       if ( dimension == VECTOR ) {
                  chartForm->setChildValues( nCols, vals );      
               } else {          
                  chartForm->setChildValues( dimension, aDimSizes, vals );
	       }
               delete[] vals;
	       }
               break;
	}
}

Boolean_		/* virtual */
ChartFU::saveConfigToFile( const CString& fileName ) const
{
	Boolean_ result;

        FILE *fp = fopen( fileName.getValue(), "w" );
        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                              errorString() );
	    result = FAILURE_;
        } else {
	    int	 x, y, width, height; 

	    chartForm->getPerfWidgetPosition( &x, &y, &width, &height);
	    fprintf( fp, "# X, Y, Width, Height, Dimension\n" );
	    fprintf( fp, "%d %d %d %d %d\n", x, y, width, height, dimension );

	    fclose( fp );
	    result = SUCCESS_;
        }

        return result;
}
/*
 *      Initialize the static data.   Only executed once.
 */
const char *const ChartFU::MY_CLASS = "ChartFU";

	
