/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Daniel A. Reed (reed@cs.uiuc.edu)
 * Contributing Authors: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *                       Bradley W. Schwartz (schwartz@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * BubbleFU.cc - A polymorophic functional unit for a dynamic bubble display
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/BubbleFU.C,v 1.28 1994/03/15 16:41:30 aydt Exp $
 */

#include "BubbleFU.h"
#include "BubbleFormWrapper.h"

#include "BooleanArray.h"
#include "FUParams.h"
#include "ParamConfig.h"
#include "ParamDisplay.h"
#include "InputPort.h"
#include "SystemErrors.h"

// Macros
#define Normalize(x) (  ( ((double) x) - minValue ) / (maxValue - minValue)  )
#define Round(x)	( (int)(x + .5) )

BubbleFU::BubbleFU()
{
	_setClassName( MY_CLASS );

	input = NULL;	
	index0 = NULL;
	index1 = NULL;
	dialog = NULL;
	bubbleForm = NULL;

	cellUsed = NULL;
	dataArrayP = NULL;
}

BubbleFU::~BubbleFU()
{
	delete input;
	delete index0;
	delete index1;
	delete dialog;
	delete bubbleForm;
	delete cellUsed;
	delete dataArrayP;
}

void
BubbleFU::_resizeCellUsed( int newDim0, int newDim1 )
{
	const int *dimSize = cellUsed->getDimSizes();
	int newDimSize[2];

	if ( newDim0 > dimSize[0] ) {
	    newDimSize[0] = newDim0;
	} else {
	    newDimSize[0] = dimSize[0];
	}

	if ( newDim1 > dimSize[1] ) {
	    newDimSize[1] = newDim1;
	} else {
	    newDimSize[1] = dimSize[1];
	}

	BooleanArray *newCellUsed = new BooleanArray( 2 );
	newCellUsed->setDimSizes( newDimSize );

	int x, y;
	int cell = 0;
	for ( x = 0; x < newDimSize[0]; x++ ) {
	   for ( y = 0; y < newDimSize[1]; y++ ) {
               if ( ( x < dimSize[0] ) && ( y < dimSize[1] ) ) {
                   newCellUsed->setTheCellValue( cell,
						 cellUsed->getCellValue(x, y) );
               } 
	   cell++;
           }
        }
	delete cellUsed;
	cellUsed = newCellUsed;
}

void
BubbleFU::_resizeDataArrayP( int newDim0, int newDim1 )
{
	const int *dimSize = dataArrayP->getDimSizes();
	int newDimSize[2];

	if ( newDim0 > dimSize[0] ) {
	    newDimSize[0] = newDim0;
	} else {
	    newDimSize[0] = dimSize[0];
	}

	if ( newDim1 > dimSize[1] ) {
	    newDimSize[1] = newDim1;
	} else {
	    newDimSize[1] = dimSize[1];
	}

	Array *newArrayP = new Array( DOUBLE, 2 );
        newArrayP->setDimSizes( newDimSize );

	int x, y;
	int cell = 0;
	for ( x = 0; x < newDimSize[0]; x++ ) {
	   for ( y = 0; y < newDimSize[1]; y++ ) {
               if ( ( x < dimSize[0] ) && ( y < dimSize[1] ) ) {
	          newArrayP->setTheCellValue( cell,
					      dataArrayP->getCellValue(x, y) );
               } 
	       cell++;
           }
        }
	delete dataArrayP;
	dataArrayP = newArrayP;
}

void
BubbleFU::_setPortOption( int theOption )
{
	Assert( portOption == -1 );	   // Shouldn"t be here if already set!

	if (  ( theOption != VECTOR_ARRAY ) &&
	      ( theOption != VECTOR_ELEMENT ) &&
	      ( theOption != ARRAY_ELEMENT ) ) {
	    warning( "Invalid Port Option.\nDefaulting to Array or Vector." );
	    theOption = VECTOR_ARRAY;
	}

	portOption = theOption;

	switch ( portOption ) {
	  case VECTOR_ARRAY: 
	      input = new InputPort( "Input" );
	      input->addTraits( INTEGER, 1 );
	      input->addTraits( INTEGER, 2 );
	      input->addTraits( FLOAT, 1 );
	      input->addTraits( FLOAT, 2 );
	      input->addTraits( DOUBLE, 1 );
	      input->addTraits( DOUBLE, 2 );
	      _addInputPort( input );
	      break;

	  case VECTOR_ELEMENT:
	      index0 = new InputPort( "Index" );
	      index0->addTraits( INTEGER, 0 );
	      _addInputPort( index0 );

              input = new InputPort( "Vector Element" );
              input->addTraits( INTEGER, 0 );
              input->addTraits( FLOAT, 0 );
              input->addTraits( DOUBLE, 0 );
              _addInputPort( input );
	      break;

          case ARRAY_ELEMENT:
              index0 = new InputPort( "Index 0" );
              index0->addTraits( INTEGER, 0 );
              _addInputPort( index0 );

              index1 = new InputPort( "Index 1" );
              index1->addTraits( INTEGER, 0 );
              _addInputPort( index1 );

              input = new InputPort( "Array Element" );
              input->addTraits( INTEGER, 0 );
              input->addTraits( FLOAT, 0 );
              input->addTraits( DOUBLE, 0 );
              _addInputPort( input );
              break;
	
	  default:
	      abort( "Invalid port option - You shouldn't be here!");
	      break;
	}
}
void 
BubbleFU::configure()
{
#ifndef XtNrows
#define XtNrows "rows"
#define XtNcolumns "columns"
#endif

#ifndef XtNminValue
#define	XtNminValue "minValue"
#define	XtNmaxValue "maxValue"
#endif
	/* 
	 * The initial number of rows and columns are selectable by the user, 
	 * and both the Bubble widget and the BubbleFU agree on what they are.
	 * If an Array with larger row and/or column count is seen, the
	 * display will resize to accomodate the new dimensions.
	 *
	 * If the input is a vector, then we always force number of rows to 1.
	 *
	 * The expected minimum and maximum data values are selected by
	 * the user, and these are used to scale the actual value seen by the
	 * BubbleFU into the range of integer values accepted by the Bubble 
	 * widget.  We never change the widget's minimum and maximum values.
	 */

	int    fuRows, fuCols;
	double fuMin,  fuMax;
	const char *labelBottom, *labelLeft;

	if ( ! isConfigured ) {
	    int wMin, wMax;

	    bubbleForm->getBubbleAttr( XtNrows, (XtArgVal) &fuRows );
	    bubbleForm->getBubbleAttr( XtNcolumns, (XtArgVal) &fuCols );
	    bubbleForm->getBubbleAttr( XtNminValue, (XtArgVal) &wMin );
            bubbleForm->getBubbleAttr( XtNmaxValue, (XtArgVal) &wMax );
               
	    fuMin = wMin;
	    fuMax = wMax;
	    valueOffset = wMin;
	    valueRange = wMax - wMin;

	    isConfigured = TRUE_;
	} else {
	    fuRows = numRows;
	    fuCols = numCols;
	    fuMin = minValue;
	    fuMax = maxValue;
	}

	Boolean_ isVector = FALSE_;

	if ( portOption == VECTOR_ELEMENT ) {
	    isVector = TRUE_;
	} else if ( portOption == VECTOR_ARRAY ) {
	    if ( input->getTraits().getDimension() == 1 ) {
	        isVector = TRUE_;
	    }
	}
	if ( isVector ) {
	    fuRows = 1;
	}

	// --- CONFIGURATION PARAMETERS --------------------------------------
        bubbleForm->getBubbleLabel( L_BOTTOM_HORIZ, &labelBottom );
        bubbleForm->getBubbleLabel( L_LEFT_VERT, &labelLeft );

	FUParams params;
	if ( !isVector ) {
	    params.addTextParam( "Initial Number of Rows", 
				BaseFUParamEntry::Integer, fuRows );
	}
	params.addTextParam( "Initial Number of Columns", 
				BaseFUParamEntry::Integer, fuCols );
 	params.addTextParam( "Minimum value", BaseFUParamEntry::Real, fuMin );
        params.addTextParam( "Maximum value", BaseFUParamEntry::Real, fuMax );
        params.addTextParam( "Vertical axis label",
			     BaseFUParamEntry::Str, labelLeft);
        params.addTextParam( "Horizontal axis label",
			     BaseFUParamEntry::Str, labelBottom );
	CString configPanelName = "Bubble FU Config: " + getName();
        ParamConfig pc( Pablo::TopLevel(), params, configPanelName );
	pc.run();

	BaseFUParamEntry& rowEntry = 
				params.getEntry( "Initial Number of Rows" );
	BaseFUParamEntry& colEntry = 
				params.getEntry( "Initial Number of Columns" );
 	BaseFUParamEntry& minEntry = params.getEntry( "Minimum value" );
        BaseFUParamEntry& maxEntry = params.getEntry( "Maximum value" );

        BaseFUParamEntry &vertLabelEntry =
		              params.getEntry( "Vertical axis label" );
        BaseFUParamEntry &horizLabelEntry = 
		              params.getEntry( "Horizontal axis label" );

	// === PARAMETER ERROR-CHECKING =======================================
	if ( !isVector ) {
	    if ( rowEntry.valueIsValid() ) {
	        fuRows = rowEntry.getValue().getInteger();
	    } else {
	        warning( "Bubble functional unit row entry was invalid.\n" );
	    }
	}
	if ( colEntry.valueIsValid() ) {
	    fuCols = colEntry.getValue().getInteger();
	} else {
	    warning( "Bubble functional unit column entry was invalid.\n" );
	}
	if ( minEntry.valueIsValid() ) {
            fuMin = minEntry.getValue().getReal();
        } else {
            warning( "Bubble functional unit minimum value was invalid.\n" );
	}
        if ( maxEntry.valueIsValid() ) {
            fuMax = maxEntry.getValue().getReal();
        } else {
            warning( "Bubble functional unit maximum value was not valid.\n" );
	}

	if ( fuMin >= fuMax ) {
	    warning( "Minimum >= Maximum.  Resetting Maximum to Minimum+1.\n");
	    fuMax = fuMin + 1;
	}

	// --- FU AND WIDGET CONFIGURATION DATA SETUPS -----------------------
	minValue = fuMin;
	maxValue = fuMax;
	numRows = fuRows;
	numCols = fuCols;

	bubbleForm->setBubbleLabel( fuMin, fuMax );
        bubbleForm->setBubbleLabel( L_BOTTOM_HORIZ, 
			      horizLabelEntry.getValue().getString() );
	bubbleForm->setBubbleLabel( L_LEFT_VERT, 
	 		      vertLabelEntry.getValue().getString() );
        bubbleForm->setBubbleAttr( XtNrows, fuRows );
	bubbleForm->setBubbleAttr( XtNcolumns, fuCols );

        bubbleForm->setPerfWidgetColors();

	/* 
	 * Set up the array we use to track which cells have been
	 * updated.
	 */
	int dimSize[2];
	dimSize[0] = numRows;
	dimSize[1] = numCols;

	if ( portOption == VECTOR_ARRAY ) {
	    cellUsed->setDimSizes( dimSize );
	} else {
	    dataArrayP->setDimSizes( dimSize );
	}
}

void
BubbleFU::configureOperation()
{
	/*
	 * We only want to allow user to select if this is the
	 * initial configuration - else just display the choice!
	 */
	FUParams portParams;
	Boolean_ readOnly = FALSE_;

	if ( portOption == -1 ) {
 	    portParams.addRadioButtonsParam( "Data Input",
					     BaseFUParamEntry::Integer, 0,
				             PortOptionList );
	} else {
	    portParams.addDisplayParam( "Configured Data Input Type", 
			PortOptionList.getElement( portOption ) );
	    readOnly = TRUE_;
	}

	ParamConfig pc( Pablo::TopLevel(), portParams, getName(), readOnly );
	pc.run();

	if ( portOption == -1 ) {
 	    BaseFUParamEntry& portEntry = portParams.getEntry( "Data Input" );

	    int theOption = -1;

	    if ( portEntry.valueIsValid() ) {
		theOption = portEntry.getValue().getInteger();
	    } 

	    _setPortOption( theOption );
	}
}

FunctionalUnit *
BubbleFU::copy()
{
	BubbleFU *copy = new BubbleFU();
	return copy;
}

void 
BubbleFU::fuCallback( int iRow, int iCol )
{
        //if (!input->valueAvailable()) return;
	if ( inputValue.isUndefined() ) {
	    return;
	}

        FUParams      params;
        Value         callbackDataValue;

        if ( ( iRow >= 0 ) && ( iCol >= 0 ) ) {

	  switch( portOption ) {
	     case VECTOR_ARRAY:
		{
                Array *arrayInputValue = (Array *)inputValue;
	        const int *dimSizes = (int *)arrayInputValue->getDimSizes();
                int runTimeDimension = inputValue.getTraits().getDimension();

                if ( runTimeDimension == 2 ) {
                   if ( (iRow < dimSizes[0]) && (iCol < dimSizes[1]) ) {
                      arrayInputValue->getCellValue( callbackDataValue,
						     iRow, iCol );
                      params.addDisplayParam( "Current Bubble Cell Value",
                         				   callbackDataValue );
	           } else {
                      params.addDisplayParam( "Current Bubble Cell Value",
                         				   noDataMSG );
	           }
                   params.addDisplayParam("Bubble Row", iRow );
                   params.addDisplayParam("Bubble Column", iCol );

                } else if ( runTimeDimension == 1 ) {
                   if ( iCol < dimSizes[0] ) {
                      arrayInputValue->getCellValue( callbackDataValue, iCol );
                      params.addDisplayParam( "Current Bubble Cell Value",
                         				   callbackDataValue );
                   } else {
                      params.addDisplayParam( "Current Bubble Cell Value",
                         				   noDataMSG );
                   }
                   params.addDisplayParam("Bubble Column", iCol );

                }

		}
	        break;

	     case VECTOR_ELEMENT:
                dataArrayP->getCellValue( callbackDataValue, 0, iCol );
                params.addDisplayParam( "Current Bubble Cell Value",
                         				   callbackDataValue );
                params.addDisplayParam( "Bubble Column", iCol );
	        break;

	     case ARRAY_ELEMENT:
	        dataArrayP->getCellValue( callbackDataValue, iRow, iCol );
                params.addDisplayParam( "Current Bubble Cell Value",
                         				   callbackDataValue );
                params.addDisplayParam("Bubble Row", iRow );
                params.addDisplayParam("Bubble Column", iCol );
	        break;
	   }

	   CString callbackTitle = "Callback: " + getName();
	   ParamDisplay *pd = new ParamDisplay( Pablo::TopLevel(), params,
					        callbackTitle );
     	}
}

void 
BubbleFU::init()
{
	/*
	 * Start by clearing up any leftovers that will be around if 
	 * init() isn't being called for the first time.  The goal is to
	 * start with a 'virgin' FU.
	 */

	if ( input != NULL ) {
	    delete input;
	}
	if ( index0 != NULL ) {
	   delete index0;
	}
        if ( index1 != NULL ) {
           delete index1;
        }
	if ( bubbleForm != NULL ) {
	    delete bubbleForm;
	}
	if ( dialog != NULL ) {
	    delete dialog;
	}
        if ( cellUsed != NULL ) {
            delete cellUsed;
        }
        if ( dataArrayP != NULL ) {
            delete dataArrayP;
        }

	/*
	 * Here's the real code to initialize the FU; In this FU we create our
	 * Input Ports in either configureOperation() or loadConfiguration()
	 */
	portOption = -1;
	isConfigured = FALSE_;
	dataValue = Value::NOVALUE;
	inputValue = Value::NOVALUE;

        cellUsed = new BooleanArray( 2 );
	dataArrayP = new Array( DOUBLE, 2 );

	if ( PortOptionList.isEmpty() ) {
	    PortOptionList.addElement( "Array or Vector" );
	    PortOptionList.addElement( "Vector Index and Element" );
	    PortOptionList.addElement( "Array Indices and Element" );
	}

	dialog = _getTitledFormDialog( "FUDialog", getName() );
	bubbleForm = new BubbleFormWrapper( dialog, this, NullArgs, 
							  "Contents" );
	dialog->manage();
	XtVaSetValues( dialog->getWidget(), XmNdefaultPosition, False, NULL );
}


Boolean_		/* virtual */
BubbleFU::loadConfigFromFile( const CString& fileName )
{
        FILE *fp = fopen( fileName.getValue(), "r" );

        if ( fp == NULL ) {
            warning( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
        } else {
	    int	x, y, width, height;
	    int theOption;
	    int labelSize;
	    char buf[LABEL_MAX_LEN];

	    // ----- Comment line and numeric parameters -----
            fscanf( fp, "%*[^\n]\n" );
   
            if ( fscanf( fp, "%lf %lf %d %d %d %d %d %d %d\n",
                           &minValue, &maxValue, &numRows, &numCols,
			   &x, &y, &width, &height, &theOption ) != 9 ) {
		warning( "Unable to read configuration information from %s\n",
			  fileName.getValue() );
	    } else {
		dialog->unmanage();
	        _setPortOption( theOption );

		// ----- Comment Line; Horizontal Label Size and String
		labelSize = 0;
                fscanf( fp, "%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		if ( labelSize == 0 ) {
		    strcpy( buf, "\0" );
		}
                bubbleForm->setBubbleLabel( L_BOTTOM_HORIZ, buf );

		// ----- Comment Line; Vertical Label Size and String
		labelSize = 0;
                fscanf( fp, "\n%*[^\n]\n" );
                fscanf( fp, "%d%*c%[^\n]", &labelSize, buf );
		if ( labelSize == 0 ) {
		    strcpy( buf, "\0" );
		}
                bubbleForm->setBubbleLabel( L_LEFT_VERT, buf );

                /*
                 * We recalculate the valueOffset and valueRange based
		 * on the widget's min and max values.  Then we use the
		 * parameters we loaded from the file to set other resources.
                 * Initialize the static color table and set up arrays
		 * used to track which cells have been updated.
                 */

     		int wMin, wMax;
                bubbleForm->getBubbleAttr( XtNminValue, (XtArgVal) &wMin );
                bubbleForm->getBubbleAttr( XtNmaxValue, (XtArgVal) &wMax );
                valueOffset = wMin;
                valueRange = wMax - wMin;

		bubbleForm->setBubbleAttr( XtNrows, numRows );
		bubbleForm->setBubbleAttr( XtNcolumns, numCols );
                bubbleForm->setBubbleLabel( minValue, maxValue );
        	bubbleForm->setPerfWidgetColors();

		dialog->manage();
		bubbleForm->setPerfWidgetPosition( x, y, width, height );

		int dimSize[2];
		dimSize[0] = numRows;
		dimSize[1] = numCols;

		if ( portOption == VECTOR_ARRAY ) {
		    cellUsed->setDimSizes( dimSize );
		} else {
	            dataArrayP->setDimSizes( dimSize );
		}

                isConfigured = TRUE_;
            } 
            fclose( fp );
        }

	if ( portOption == -1 ) {
	    configureOperation();
	}

	if ( ! isConfigured ) {
	    configure();
	}
	return isConfigured;
}

Boolean_ 
BubbleFU::ready()
{
	return isConfigured;
}

void 
BubbleFU::run(  Boolean_& /* errorFlag */ ) {
	/*
	 * Note:
	 * The sparcs are REALLY slow at multiplication, so instead
	 * of using the (x,y) coordinates into our arrays, we keep
	 * the absolute "cell" index, and use that instead.  You
	 * don't see the multiply here - it's in the Array::setCellValue()
	 * and getCellValue() code.  This also shows up in the resize
	 * Array code.	-- RAA 2/9/93
	 */

	int	nx, ny, x, y, val;
	int	ix, iy;
	int 	cell;
       	double	dvalScaled;

	Boolean_ resize = FALSE_;

	Assert( input->valueAvailable() );
	inputValue = input->getValue();

	switch ( portOption ) {

	    case VECTOR_ARRAY:
	      {
	      int dimension = inputValue.getTraits().getDimension();
	      Assert( (dimension == ARRAY_2D) || (dimension == VECTOR) );
	
	      Array *a = (Array *)inputValue;
	      const int *dimSizes = a->getDimSizes();

	      if ( dimension == ARRAY_2D ) {
	          if ( ( nx = dimSizes[0] ) > numRows ) {	
		      numRows = nx;
		      bubbleForm->setBubbleAttr( XtNrows, numRows );
		      resize = TRUE_;
	    	  }

	    	  if ( ( ny = dimSizes[1] ) > numCols ) {
	              numCols = ny;
	              bubbleForm->setBubbleAttr( XtNcolumns, numCols );
	              resize = TRUE_;
	    	  }

	    	  if ( resize ) {				
	              _resizeCellUsed( nx, ny );
	    	  }

		  cell = 0;
	    	  for ( x = 0; x < nx; x++ ) {
	              for ( y = 0; y < ny; y++ ) {
	                  a->getTheCellValue( dataValue, cell );

	                  if ( (double)dataValue != minValue ) {  
			      /* flag updated cells */
		              cellUsed->setTheCellValue( cell, TRUE_ );
	                  }

		          /*
		           * Only worry about this value if it has been 'active'
		           * at some time during the run.
		           */

	                  if ( cellUsed->getTheCellValue( cell ) ) {  
                              dvalScaled = warnings.fuDataRangeWarningCheck(
					               this, (double)dataValue,
				                       minValue, maxValue );

                              val = Round( ( Normalize( dvalScaled ) *
							         valueRange ) );
                              val += valueOffset;    
	      	              bubbleForm->setDisplayValue( x, y, val );
	                  }

			  cell++;
	              }			  	  // for y
	          } 			          // for x
	      
	      } else {			  	  // VECTOR
	          Assert( numRows == 1 );

	          if ( ( ny = dimSizes[0] ) > numCols ) {	
	              numCols = ny;
	              bubbleForm->setBubbleAttr( XtNcolumns, numCols );
	              _resizeCellUsed( 1, ny );
	          }

	          for ( y = 0; y < ny; y++ ) {
	              a->getCellValue( dataValue, y );
	              if ( (double)dataValue != minValue ) {	
			  /* flag updated cells */
		          cellUsed->setTheCellValue( y, TRUE_ );
	              }

		      /*
		       * Only worry about this value if it has been 'active' at
		       * some time during the run.
		       */

	             if ( cellUsed->getTheCellValue( y ) ) {  
                         dvalScaled = warnings.fuDataRangeWarningCheck( this, 
					(double)dataValue, minValue, maxValue );

                         val = Round( ( Normalize(dvalScaled) * valueRange ) );
                         val += valueOffset; 
	      	         bubbleForm->setDisplayValue( 0, y, val );
	             }
	         }				// for y
	      } 				// end of if

	      }
	      break;				// case VECTOR_ARRAY

	    case VECTOR_ELEMENT:
              Assert( index0->valueAvailable() );
              iy = (int) index0->getValue();

              if ( iy >= numCols ) {
                  numCols = iy + 1;
                  bubbleForm->setBubbleAttr( XtNcolumns, numCols );
                  _resizeDataArrayP( 1, numCols );
              }
	      dataArrayP->setTheCellValue( iy, inputValue );

              dvalScaled = warnings.fuDataRangeWarningCheck( this,
                                (double)inputValue, minValue, maxValue );
              val = Round( ( Normalize( dvalScaled ) * valueRange ) );
              val += valueOffset;               // Widget minimum may not be 0

              bubbleForm->setDisplayValue( 0, iy, val );
              break;

	    case ARRAY_ELEMENT:
	      Assert( index0->valueAvailable() );
	      Assert( index1->valueAvailable() );
	      ix = (int) index0->getValue();
              iy = (int) index1->getValue();

              if ( ix >= numRows ) {
                  numRows = ix + 1;
                  bubbleForm->setBubbleAttr( XtNrows, numRows );
                  resize = TRUE_;
              }

              if ( iy >= numCols ) {
                  numCols = iy + 1;
                  bubbleForm->setBubbleAttr( XtNcolumns, numCols );
                  resize = TRUE_;
              }

              if ( resize ) {
                  _resizeDataArrayP( numRows, numCols );
              }

              dataArrayP->setCellValue( inputValue, ix, iy );

              dvalScaled = warnings.fuDataRangeWarningCheck( this,
              			(double)inputValue, minValue, maxValue );
              val = Round( ( Normalize( dvalScaled ) * valueRange ) );
              val += valueOffset;       	// Widget minimum may not be 0

              bubbleForm->setDisplayValue( ix, iy, val );
	      break;
	}
}

Boolean_		/* virtual */
BubbleFU::saveConfigToFile( const CString& fileName ) const
{
        Boolean_ result;
        FILE *fp = fopen( fileName.getValue(), "w" );

        if ( fp == NULL ) {
            error( "Unable to open %s: %s\n", fileName.getValue(),
                                                  errorString() );
            result = FAILURE_;
	} else {
            int         x, y, width, height;
	    const char *label;

	    bubbleForm->getPerfWidgetPosition( &x, &y, &width, &height );

            fprintf( fp, "# Min Max NumRows NumCols X Y Wdth Hght PortOpt\n" );
            fprintf( fp, "%lf %lf %d %d %d %d %d %d %d\n", 
			 minValue, maxValue, numRows, numCols,
			 x, y, width, height, portOption );

            bubbleForm->getBubbleLabel( L_BOTTOM_HORIZ, &label );
            fprintf( fp, "# Horizontal Label\n" );
	    fprintf( fp, "%d %s\n", strlen( label ), label );

            bubbleForm->getBubbleLabel( L_LEFT_VERT, &label );
            fprintf( fp, "# Vertical Label\n" );
	    fprintf( fp, "%d %s\n", strlen( label ), label );

            fclose( fp );
            result = SUCCESS_;
        }
        return result;
}
	
// Global class definition
const char *const BubbleFU::MY_CLASS = "BubbleFU";
CStringObjList BubbleFU::PortOptionList;
