/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Authors: Philip C. Roth (proth@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*************************************************************
 * BarGraphUtilDisplay.C
 *
 * Implementation of a UtilDisplay showing utilization on an 
 * array of bar graphs.
 *
 * $Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/FunctionalUnits/RCS/BarGraphUtilDisplay.C,v 1.10 1994/02/25 04:33:10 aydt Exp $
 *
 *************************************************************/
#include <stream.h>
#include <strstream.h>
#include <X11/Intrinsic.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/RowColumn.h>
#include <Xm/ScrolledW.h>
#include "LabelUtilities.h"
#include "BarGraphUtilDisplay.h"
#include "Selection.h"
#include "UtilDisplayManager.h"


extern "C" {
#include "Bargraph.h"
}


/*
 * #defines for array sizes
 * g++ allows 'const int' for this, but CC doesn't
 */
#define MAX_LABEL_SIZE	16


BarGraphUtilDisplay::BarGraphUtilDisplay( char *name, UtilFU *ufu, Widget par )
	: UtilDisplay( name, ufu, par )
{
	XtManageChild( getDialog() );
}


BarGraphUtilDisplay::~BarGraphUtilDisplay( void )
{
	XtDestroyWidget( getDialog() );
}




/*
 * _handleResize()
 * 
 * Resize the performance widgets so they fill their dialog, unless
 * resizing them would make them too small.
 */
/* virtual */
void
BarGraphUtilDisplay::_handleResize( Widget w )
{
	Widget cwind, wwind;		/* clip and work windows for w */
	Dimension cwidth, cheight;	/* dimensions of clip window */
	Dimension natWidth, natHeight;	/* natural size of nonscrolled dialog */
	Dimension labelWidth, labelHeight,/* RowColumn Dimension attributes */
		marginWidth, marginHeight, spacing;
	int numSetups;
	int i;


	/*
	 * Check that there is a minimum size 
	 */
	getNaturalDimensions( natWidth, natHeight );
	if( natWidth == 0 || natHeight == 0 ) {
		return;				// no, so do nothing
	}


	XtUnmanageChild( getWorkWindow() );


	/* 
	 * get new size of clip window 
	 */
	XtVaGetValues( w,
			XmNworkWindow,	&wwind,
			XmNclipWindow,	&cwind,
			NULL );
	XtVaGetValues( cwind,
			XtNwidth, 	&cwidth,
			XtNheight, 	&cheight,
			NULL );


	/*
	 * check whether the clip window is larger or smaller than the
	 * natural size of the workWindow
	 */
	Dimension width = ( cwidth > natWidth ? cwidth : natWidth );
	Dimension height = ( cheight > natHeight ? cheight : natHeight );

	XtVaSetValues( getWorkWindow(),
			XtNwidth,	width,
			XtNheight,	height,
			NULL );


	/*
	 * find out the amount of width actually available to the performance
	 * widgets ( i.e., not taken up by labels, margins, etc.
	 */
	UtilsDisplayedType disp = getDisplayedType();
	if( disp == IndividualsShown ) {
		if( !(numSetups = getNumIndividuals()) ) {
			return;	
		}
	}
	else {
		numSetups = numAggregates;
	}

	/*
	 * compute new width of performance widgets
	 */
	XtVaGetValues( getVertLabelWidget(),
			XtNwidth,	&labelWidth,
			NULL );
	width -= labelWidth;
	XtVaGetValues( getRowCol(),
			XmNmarginWidth,	&marginWidth,
			XmNmarginHeight, &marginHeight,
			XmNspacing,	&spacing,
			NULL );
	width -= ( 2 * marginWidth + numSetups * spacing );
	width -= 20;			// to ensure the ScrollBars go away

	/*
	 * compute new height of underlying RowColumn widget
	 * ( This handles setting height of performance widgets )
	 */
	XtVaGetValues( getHoriLabelWidget(),
			XtNheight,	&labelHeight,
			NULL );
	height -= ( 2*marginHeight + labelHeight + getPerfLabelDim() );
	height -= 20;			// to ensure the ScrollBars go away


	/* 
	 * Compute proposed width of performance widgets.
	 * Note we checked for numSetups == 0 above.
	 */
	Dimension proposedWidth = width / numSetups;

	if( disp == IndividualsShown ) {
		for( i = 0; i < numSetups; i++ ) {
			XtVaSetValues( individualUtils[i],
				XtNwidth,	proposedWidth,
				NULL );
		}
	}
	else {
		for( i = 0; i < numSetups; i++ ) {
			XtVaSetValues( aggregateUtils[i],
				XtNwidth,	proposedWidth,
				NULL );
		}
	}
	XtManageChild( getWorkWindow() );
}



/*
 * changeValue()
 *
 * update the utilization value for PE proc to utilization util
 */
/* virtual */
void
BarGraphUtilDisplay::changeValue( int proc, float util )
{
	switch( getDisplayedType() ) {
		case AggregateShown:
			BargraphSetValue( (BargraphWidget)aggregateUtils[proc],
						(int)(util*100) );
			break;

		case IndividualsShown:
			{
				int i = _indexIntoMap( proc );

				if( i != notDisplayed ) {
				BargraphSetValue( 
					(BargraphWidget)individualUtils[i], 
						(int)(util*100) );
				}
			}
			break;
	}
}





/*
 * createPerfSetup()
 *
 * Creates a widget subtree for displaying utilization on a bargraph
 * performance widget
 */
Widget
BarGraphUtilDisplay::createPerfSetup( const char *labl )
{
	Dimension height;


	Widget form = XtVaCreateWidget( "form",
			xmFormWidgetClass, getRowCol(),
			XmNtopAttachment,	XmATTACH_FORM,
			XmNbottomAttachment,	XmATTACH_FORM,
			NULL );

	/* create the performance widget's PE label */
	XmString xstr = XmStringCreateLtoR( (char*)labl, 
					XmSTRING_DEFAULT_CHARSET );

	Widget label = XtVaCreateManagedWidget( "bglabel",
			xmLabelWidgetClass, form,
			XmNlabelString,		xstr,
			NULL );
	XtVaGetValues( label,
			XtNheight,	&height,
			NULL );
	setPerfLabelDim( height );

	Widget retval = XtVaCreateManagedWidget( "bargraph",
			bargraphWidgetClass, form,
			XmNbottomWidget,	label,
			NULL );
	XtAddCallback( retval, XtNselect, (XtCallbackProc)selectCB, this );
	BargraphSetValue( (BargraphWidget)retval, 0 );

	XtManageChild( form );
	return retval;
}



/*
 * getType()
 *
 * Returns the character string indicating the type of this UtilDisplay
 */
/* virtual */
const char * const
BarGraphUtilDisplay::getType( void ) const
{
	return "Bar Graph";
}




/*
 * updatePerfWidgets()
 *
 * This method is called whenever the performance widget selection is
 * changed.
 * All deallocation of prior widgets is assumed to be complete before the
 * start of this method.
 */
/* virtual */
void
BarGraphUtilDisplay::updatePerfWidgets( void )
{
	int	perfWCount = 0;			// # of dialog perf widgets
	Dimension width = 0;			// Dimensions of perf. widgets
	Dimension natWidth, natHeight;		// natural dimensions of work
						// window
	Position x, y;				// screen loc of this display

	/* save screen location of this display */
	/* no use saving dimensions, they probably should be changed */
	XtVaGetValues( getDialog(),
		XmNx,	&x,
		XmNy,	&y,
		NULL );

	/* pop down the dialog */
	XtUnmanageChild( getDialog() );	

	XtVaSetValues( getRowCol(),
		XmNorientation,		XmHORIZONTAL,
		NULL );

	if( getDisplayedType() == AggregateShown ) {
		/* create performance widgets for average utilization */
		perfWCount += numAggregates;

		/* create performance widget for aggregate utilizations */
		aggregateUtils[Average] = createPerfSetup( averageLabelString );
		aggregateUtils[Minima] = createPerfSetup( minimumLabelString );
		aggregateUtils[Maxima] = createPerfSetup( maximumLabelString );

		XtVaGetValues( aggregateUtils[Average],
			XtNwidth, 	&width,
			NULL );

	}

	else {
		/* create performance widgets for individual utilization */

		/* determine how many individuals to show */
		int icount = 0;
		Selection *sel = getSelection();
		_makeMap( sel, icount );
		setNumIndividuals( icount );
		perfWCount += icount;

		if( icount ) {
			int i;

			/* allocate enough space for individuals */
			individualUtils = (Widget*)XtMalloc( icount *
						sizeof( BargraphWidget ) );

			/* create widget subtrees for each individual */
			for( i = 0; i < icount; i++ ) {
				char buf[ MAX_LABEL_SIZE ];	// for labels


				/* create label string for this PE */
				ostrstream temp( buf, MAX_LABEL_SIZE );
				temp << map[i] << '\0';

				individualUtils[i] = createPerfSetup( buf );
			}

			XtVaGetValues( individualUtils[0],
				XtNwidth, 	&width,
				NULL );
		}
	}

	/* Find natural Dimensions for the new work window */
	XtVaGetValues( getWorkWindow(),		
		XtNwidth,	&natWidth,
		XtNheight,	&natHeight,
		NULL );
	setNaturalDimensions( natWidth, natHeight );

	/* Resize the dialog based on the natural size of the work window */
	initialResizeDialog();

	/* pop up the dialog */
	XtManageChild( getDialog() );	

	/* reset screen position */
	XtVaSetValues( getDialog(),
		XmNx,	x,
		XmNy,	y,
		NULL );
}


