/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * StructureDescriptor.cc - methods of the class used to describe a structure
 *			    made up of one or more fields of data.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Descriptors/RCS/StructureDescriptor.C,v 1.15 1994/02/25 04:31:51 aydt Exp $
 */

#include <string.h>

#include "StructureDescriptor.h"

#include "DataCharacteristics.h"
#include "Defines.h"
#include "GetPut.h"

static char tmpBuffer[4096];	// used to get/put if conversion required

StructureDescriptor::StructureDescriptor()
{
	_setClassName( MY_CLASS );
	headEntry = NULL;
	tailEntry = NULL;
}

StructureDescriptor::StructureDescriptor( const CString& structureName )
{
	_setClassName( MY_CLASS );
	setName( structureName );

	headEntry = NULL;
	tailEntry = NULL;
}

StructureDescriptor::StructureDescriptor( const CString& structureName, 
			          const Attributes& structureAttributes)
{
	_setClassName( MY_CLASS );
	setName( structureName );
	attributes = structureAttributes;
	headEntry = NULL;
	tailEntry = NULL;
}

StructureDescriptor::StructureDescriptor( const StructureDescriptor 
							& structureDescr )
{
	_setClassName( MY_CLASS );
	setName( structureDescr.name );

	attributes = structureDescr.attributes;

	StructureEntry *origEntry = structureDescr.headEntry;

	if ( origEntry == NULL ) {		// no fields
	    headEntry = NULL;
	    tailEntry = NULL;
	} else {
	    StructureEntry *newEntry = new StructureEntry;

	    headEntry = newEntry;
	    newEntry->field = origEntry->field;

	    while ( origEntry->nextEntry != NULL ) {
	       origEntry = origEntry->nextEntry;
	       newEntry->nextEntry = new StructureEntry;

	       newEntry = newEntry->nextEntry;
	       newEntry->field = origEntry->field;
	    }

	    tailEntry = newEntry;
	}
}

StructureDescriptor::~StructureDescriptor()
{
	StructureEntry *currentEntry = headEntry;
	StructureEntry *followingEntry;

	while ( currentEntry != NULL ) {
	   followingEntry = currentEntry->nextEntry;
	   delete currentEntry;
	   currentEntry = followingEntry;
	}
}

/*
 *  Parse binary data into Structure Descriptor
 *      <record name><attributes><field count><field descriptor>*
 *      record name ::= string
 *      field count ::= int
 */
int
StructureDescriptor::bitsToObj( const char *bufPtr)
{
	/*
	 * First, release space used by entries currently in the list 
	 */
	StructureEntry *currentEntry = headEntry;
	StructureEntry *followingEntry;

	while( currentEntry != NULL ) {
		followingEntry = currentEntry->nextEntry;
		delete currentEntry;
		currentEntry = followingEntry;
	}
	headEntry = tailEntry = NULL;

	/*
	 * Next, populate with data from the bitstream
	 */
	FieldDescriptor	fieldDescriptor;
	int		fieldCount;
        int     	currentOffset;

        name = (char *) bufPtr;
        currentOffset = name.length() + 1;

        currentOffset += attributes.bitsToObj( bufPtr+currentOffset );
        currentOffset += functionGet( bufPtr+currentOffset, &fieldCount );

        while ( fieldCount > 0 ) {
           currentOffset += fieldDescriptor.bitsToObj( bufPtr+currentOffset );
           insert( fieldDescriptor );
           fieldCount--;
        }

        return currentOffset;
}

int
StructureDescriptor::bytesNeeded( ) const
{
        int bytesRequired = name.length() + 1;
        bytesRequired += attributes.bytesNeeded( ) ;
        bytesRequired += sizeof( int );			// field Count

	StructureEntry* currentEntry = headEntry;
	while ( currentEntry != NULL ) {
		bytesRequired += currentEntry->field.bytesNeeded();
		currentEntry = currentEntry->nextEntry;
	}

        return bytesRequired;
}

int
StructureDescriptor::cbitsToObj( const char *bufPtr,
				 const DataCharacteristics *cnvInfo )
{
	/*
	 * First, release space used by entries currently in the list 
	 */
	StructureEntry *currentEntry = headEntry;
	StructureEntry *followingEntry;

	while( currentEntry != NULL ) {
		followingEntry = currentEntry->nextEntry;
		delete currentEntry;
		currentEntry = followingEntry;
	}
	headEntry = tailEntry = NULL;

	/*
	 * Next, populate with data from the bitstream
	 */
	FieldDescriptor	fieldDescriptor;
	int fieldCount;
        int currentOffset = 0;
	char *tmpPtr = tmpBuffer;
	do {
	    currentOffset += cnvInfo->functionGet( bufPtr+currentOffset, 
						   tmpPtr );
	} while ( *tmpPtr++ != '\0' );

        name = tmpBuffer;

        currentOffset += attributes.cbitsToObj( bufPtr+currentOffset, cnvInfo );
        currentOffset += cnvInfo->functionGet( bufPtr+currentOffset, 
					       &fieldCount );

        while ( fieldCount > 0 ) {
           currentOffset += fieldDescriptor.cbitsToObj( bufPtr+currentOffset,
							cnvInfo);
           insert( fieldDescriptor );
           fieldCount--;
        }

        return currentOffset;
}

int
StructureDescriptor::cbytesNeeded( const DataCharacteristics *cnvInfo ) const
{
        int bytesRequired = ( name.length() + 1 ) *
					cnvInfo->getAtomSize( CHARACTER );
        bytesRequired += attributes.cbytesNeeded( cnvInfo ) ;
        bytesRequired += cnvInfo->intSize();			// field Count

	StructureEntry* currentEntry = headEntry;
	while ( currentEntry != NULL ) {
		bytesRequired += currentEntry->field.cbytesNeeded( cnvInfo );
		currentEntry = currentEntry->nextEntry;
	}

        return bytesRequired;
}


Boolean_
StructureDescriptor::contains( const CString& fieldName ) const
{
	StructureEntry* currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( fieldName == currentEntry->field.getName() ) {
		return TRUE_;
	   } else {
		currentEntry = currentEntry->nextEntry;
	   }
	}

	return FALSE_;
}

int
StructureDescriptor::entryCount() const
{
	int count = 0;
	StructureEntry* currentEntry = headEntry;

	while ( currentEntry != NULL ) {
		count++;
		currentEntry = currentEntry->nextEntry;
	}

	return count;
}

const FieldDescriptor&
StructureDescriptor::fetch( const CString& fieldName ) const
{
	StructureEntry* currentEntry = headEntry;

	while ( currentEntry != NULL ) {
	   if ( fieldName == currentEntry->field.getName() ) {
		return currentEntry->field;
	   } else {
		currentEntry = currentEntry->nextEntry;
	   } 
	}

	return noField;
}

Boolean_ 
StructureDescriptor::insert( const FieldDescriptor& fieldDescr )
{
	if ( contains( fieldDescr.getName() ) ) {	// A duplicate
	   return FAILURE_;
	}

	StructureEntry* newEntry = new StructureEntry;
	newEntry->field = fieldDescr;

	if ( headEntry == NULL ) {
	   headEntry = newEntry;
	   tailEntry = newEntry;
	} else {
	   tailEntry->nextEntry = newEntry;
	   tailEntry = newEntry;
	}
	return SUCCESS_;
}

Boolean_
StructureDescriptor::isMatchFor( const StructureDescriptor& structureDescr,
				 Boolean_ matchOnName,
				 Boolean_ matchOnAttributes,
				 Boolean_ matchOnFields ) const
{
	if ( matchOnName ) {
	    if ( name != structureDescr.name ) {
	    	return FALSE_;
	    }
	}

	if ( matchOnAttributes ) {
	    if ( attributes != structureDescr.attributes ) {
	    	return FALSE_;
	    }
	}

	if ( matchOnFields ) {
	    StructureEntry* entryA = headEntry;
	    StructureEntry* entryB = structureDescr.headEntry;

	    while ( entryA != NULL ) {
		if ( ( entryB == NULL ) || ( entryA->field != entryB->field ) ){
		    return FALSE_;
		} 

		entryA = entryA->nextEntry;
		entryB = entryB->nextEntry;
	    }
	    if ( entryB != NULL ) {
		return FALSE_;
	    }
	}
	return TRUE_;
}


/*
 * Parse StructureDescriptor into binary data
 */
int
StructureDescriptor::objToBits( char *const bufPtr, int bufLen ) const
{
	if ( bytesNeeded() > bufLen ) {
		return -1;
	}

        strcpy( (char *)bufPtr, name.getValue() );
        int bytesUsed = name.length() + 1;

        bytesUsed += attributes.objToBits( bufPtr+bytesUsed, bufLen-bytesUsed );

        int fieldCount = entryCount();
        bytesUsed += functionPut( bufPtr+bytesUsed, &fieldCount );

	StructureEntry* currentEntry = headEntry;
	while ( currentEntry != NULL ) {
	    	bytesUsed += currentEntry->field.objToBits( bufPtr+bytesUsed, 
							    bufLen-bytesUsed );
		currentEntry = currentEntry->nextEntry;
	}

        return bytesUsed;
}

int
StructureDescriptor::objToCbits( char *const bufPtr, int bufLen,
				 const DataCharacteristics *cnvInfo ) const
{
	if ( cbytesNeeded( cnvInfo ) > bufLen ) {
		return -1;
	}

	int bytesUsed = 0;
	strcpy( tmpBuffer, (const char *)name );
	char *tmpPtr = tmpBuffer;

	do {
	    bytesUsed += cnvInfo->functionPut( bufPtr+bytesUsed, tmpPtr );
	} while ( *tmpPtr++ != '\0' );

        bytesUsed += attributes.objToCbits( bufPtr+bytesUsed, 
					    bufLen-bytesUsed, cnvInfo );

        int fieldCount = entryCount();
        bytesUsed += cnvInfo->functionPut( bufPtr+bytesUsed, &fieldCount );

	StructureEntry* currentEntry = headEntry;
	while ( currentEntry != NULL ) {
	    bytesUsed += currentEntry->field.objToCbits( bufPtr+bytesUsed, 
					            bufLen-bytesUsed, cnvInfo );
	    currentEntry = currentEntry->nextEntry;
	}

        return bytesUsed;
}

void 
StructureDescriptor::remove( const CString& fieldName )
{
	if ( headEntry != NULL ) {		  // Check for empty list
	   StructureEntry* currentEntry = headEntry;

	   if ( fieldName == currentEntry->field.getName() ) {  // Rm 1st entry
	      headEntry = currentEntry->nextEntry;
	      if ( tailEntry == currentEntry ) {  // We"re removing only entry
		 tailEntry = NULL;
		 Assert( headEntry == NULL );
	      }
	      delete currentEntry;

	   } else {

	      StructureEntry* followingEntry = currentEntry->nextEntry;
	      while ( followingEntry != NULL ) {

	         if ( fieldName == followingEntry->field.getName() ) {
		    currentEntry->nextEntry = followingEntry->nextEntry;
		    if ( tailEntry == followingEntry ) {
		       tailEntry = currentEntry;
		    }
		    delete followingEntry;
		    break;
	         } 
	      currentEntry = followingEntry;
	      followingEntry = currentEntry->nextEntry;
	      }
	   }
	}
}

StructureDescriptor& 
StructureDescriptor::operator=( const StructureDescriptor& structureDescr)
{
	/* 
	 * First, copy the variables which are part of the BaseDescriptor
	 * class.
	 */
	setName( structureDescr.name );
	attributes = structureDescr.attributes;

	/*
	 * Next, release space used by entries currently in the list
	 */
	StructureEntry *currentEntry = headEntry;
	StructureEntry *followingEntry;

	while ( currentEntry != NULL ) {
	   followingEntry = currentEntry->nextEntry;
	   delete currentEntry;
	   currentEntry = followingEntry;
	}

	/* 
	 * Finally, copy the entries from the passed Structure Descriptor into
	 * this StructureDescriptor.
	 */
	StructureEntry *origEntry = structureDescr.headEntry;

	if ( origEntry == NULL ) {		// no fields
	    headEntry = NULL;
	    tailEntry = NULL;
	} else {
	    StructureEntry *newEntry = new StructureEntry;

	    headEntry = newEntry;
	    newEntry->field = origEntry->field;

	    while ( origEntry->nextEntry != NULL ) {
	       origEntry = origEntry->nextEntry;
	       newEntry->nextEntry = new StructureEntry;

	       newEntry = newEntry->nextEntry;
	       newEntry->field = origEntry->field;
	    }

	    tailEntry = newEntry;
	}
	return *this;
}

void 		/* virtual */
StructureDescriptor::printOn( ostream& os ) const
{
	os << "<StructureDescriptor begin> \n";
	os << "name: ";
	name.printOn( os );
	os << NL;
	attributes.printOn( os );

	StructureEntry* currentEntry = headEntry;
	while ( currentEntry != NULL ) {
	        os << currentEntry->field;
		currentEntry = currentEntry->nextEntry;
	}
	os << "<StructureDescriptor end>\n";
}

/*
 * 	Initialize the static data.  
 */
const char *const StructureDescriptor::MY_CLASS = "StructureDescriptor";

/*
 * Definition of the static data if not done in a central place
 */
#ifdef CONSTRUCT_NONLOCAL_STATICS
	FieldDescriptor StructureDescriptor::noField( CString::NOMATCH );
#endif
