/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Tom Birkett (birkett@cs.uiuc.edu) 
 * Author: Tara Madhyastha (tara@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */

#include "SparcAudioHardware.h"
#include <malloc.h>

extern "C"
{
int poll(pollfd*,unsigned long,int);
int audio__setplayhdr(int, Audio_hdr*, int);
int audio__setpause(int, int);
int audio__setval(int, unsigned*, int);
int audio__setgain(int, double*, int);
int audio__flush(int, int);
unsigned int cvt(int);
int audio_cmp_hdr(Audio_hdr*, Audio_hdr*);
int audio_write_filehdr(int, Audio_hdr*, char*, unsigned);
int audio_read_filehdr(int, Audio_hdr*, char*, unsigned);
int audio_drain(int, int);
int audio_enc_to_str(Audio_hdr*, char*);
unsigned audio_secs_to_bytes(Audio_hdr*, double);
};


SparcAudioHardware::SparcAudioHardware()
{
  struct stat st;
  Volume = 0;
  Record_volume = 0;
  Audio_fd = -1;
  Play_port = AUDIO_SPEAKER;
  InstrumentNumber = 0;

  strcpy(Audio_dev, "/dev/audio");
  int  err = stat(Audio_dev, &st);
  if (err < 0) {
    fprintf(stderr, "SparcAudioHardware::SparcAudioHardware(): Audio_dev = %s,  %s\n", Audio_dev, sys_errlist[errno]);
    return;
  }
  if (!S_ISCHR(st.st_mode)) {
    fprintf(stderr, "SparcAudioHardware::SparcAudioHardware(): %s is not an audio device\n", Audio_dev);
    return;
  }
  /* Try it quickly, first */
  Audio_fd = open(Audio_dev, O_WRONLY | O_NDELAY);
  if ((Audio_fd < 0) && (errno == EBUSY)) {
    fprintf(stderr, "SparcAudioHardware::SparcAudioHardware(): %s is busy\n", 
	    Audio_dev);
    return;
  }
  if (Audio_fd < 0) {
    fprintf(stderr, "SparcAudioHardware::SparcAudioHardware(): Audio_dev = %s, %s",Audio_dev, sys_errlist[errno]);
    return;
  }
  /* Get the device output encoding configuration */
  if (audio_get_play_config(Audio_fd, &Dev_hdr) != AUDIO_SUCCESS) {
    fprintf(stderr, "SparcAudioHardware::SparcAudioHardware(): %s is not an audio device\n", Audio_dev);
    return;
  }
  (void) reset();
}

SparcAudioHardware::~SparcAudioHardware()
{
  /* flush output queues before exiting */
  if (Audio_fd >= 0) {
    (void) audio_drain(Audio_fd, True_);
    (void) close(Audio_fd);                 /* close output */
  }
}


Bool_
SparcAudioHardware::setVolume(unsigned volumeRequested)
{
  double vol;
  int err;

  if (volumeRequested != Volume) {
    if ( volumeRequested < 0 || volumeRequested > MAX_GAIN ) {
      cerr << form("Error: SparcAudioHardware::setVolume: volumeRequested = %d out of range; must be between 0 and %d\n", volumeRequested, MAX_GAIN);
      return (False_);
    }
    vol = (double) volumeRequested / (double) MAX_GAIN;
    err = audio_set_play_gain(Audio_fd, &vol);
    if (err != AUDIO_SUCCESS) {
      cerr << form("Error: SparcAudioHardware::setVolume: could not set output volume for %s\n", Audio_dev);
      return(False_);
    }
    Volume = volumeRequested;
  }
  return(True_);
}


Bool_
SparcAudioHardware::setRecordVolume(unsigned volumeRequested)
{
  double vol;
  int err;
  if (volumeRequested != Record_volume) {
    if ( volumeRequested < 0 || volumeRequested > MAX_GAIN ) {
      cerr << form("Error: SparcAudioHardware::setRecordVolume: volumeRequested = %d out of range; must be between 0 and %d\n", volumeRequested, MAX_GAIN);
      return (False_);
    }
    vol = (double) volumeRequested / (double) MAX_GAIN;
    err = audio_set_record_gain(Audio_fd, &vol);
    if (err != AUDIO_SUCCESS) {
      cerr << form("Error: SparcAudioHardware::setRecordVolume: could not set record volume for %s\n", Audio_dev);
      return(False_);
    }
    Record_volume = volumeRequested;
  }
  return(True_);
}

Bool_
SparcAudioHardware::playNote(char note,
			     double duration)

{
  int     numSamples;
  Bool_ okToWrite = False_;
  Samples  *samples;
  char *samplePtr, *buf;
  int     i, request;
  char NoteFileName[256];

  if (InstrumentNumber == 0) {  /* sine wave generation */
    Oscil   myOscil(Dev_hdr.sample_rate);
    samples = myOscil.bsamples(note,  duration);
    numSamples = samples->size();

    buf = ( char *) malloc(sizeof(char) * numSamples);
    samplePtr = buf;
    for (i=0; i < numSamples; i++) {
      samplePtr[i] = (unsigned char) cvt((((int) 
				       (samples->get(i) * 127)) + 127) * 16);
    }
    while (numSamples > 0) {
      request = AUDBUFSIZE > numSamples ? numSamples : AUDBUFSIZE;
      while (request  > 0) {
	playSamples(samplePtr, (int) request);
	samplePtr+=request;
	numSamples -= request;
	request = AUDBUFSIZE > numSamples ? numSamples : AUDBUFSIZE;
      }
    }
    delete samples;
    free(buf);
  } else {  
    sprintf(NoteFileName, "%s/%d/NOTES.%d", SAMPLE_LIBRARY, InstrumentNumber, (int) note);
    playSamples(NoteFileName, duration);
  }
  return(True_);
}

Bool_
SparcAudioHardware::playNumber(int num, int voice)
{
  int i;

  if (num >= 0 && num < 20) {          /* zero - 19 */
    playSamples(form("%s/voice/%d/%d.au", SAMPLE_LIBRARY, voice, num));
    return( True_ );
  }
  if (num < 100) { /* 20 - 99 */
    playSamples(form("%s/voice/%d/%d0.au", SAMPLE_LIBRARY, 
		     voice, ( num/10 ) ));
    if ((i = (num % 10)) != 0) {
      playSamples(form("%s/voice/%d/%d.au", SAMPLE_LIBRARY, 
		       voice, i));
    }
    return(True_);
  }
  
  if (num < 1000) {  /* 100 - 999 */

    playSamples(form("%s/voice/%d/%d.au",  SAMPLE_LIBRARY, 
		     voice, (num / 100)));
    playSamples(form("%s/voice/%d/hundred.au",  SAMPLE_LIBRARY, 
		     voice));
    if ((i = (num % 100)) != 0) {
      playNumber(i, voice);
    }
    return(True_);
  }

  if (num < 100000) { /* 1000 -- 99999 */
    playNumber((int) (num / 1000), voice);
    playSamples(form("%s/voice/%d/thousand.au",  SAMPLE_LIBRARY, 
		     voice));
    if ((i = (num % 1000)) != 0) {
      playNumber(i, voice);
    }
    return(True_);
  }
  /* else, 100,000 */
  playNumber(100, voice);
  playSamples(form("%s/voice/%d/thousand.au",  SAMPLE_LIBRARY, voice));
  return(True_);
}

Bool_
SparcAudioHardware::playNumber(double num, int voice)
{
  char buf[128];
  char *fraction;

  if (num > MAX_SPOKEN_NUMBER) {
    cerr << form("SparcAudioHardware::playNumber: cannot play numbers greater than %d, was given %2.3f.\n", MAX_SPOKEN_NUMBER, num);
    return(False_);
  } else {
    playNumber((int) num, voice);
    sprintf(buf, "%g", num);
    if ((fraction = strchr(buf, '.')) != NULL) {
      playFraction(fraction+1, voice);
    }
    return(True_);
  }
}

Bool_
SparcAudioHardware::playFraction(char *fraction, int voice)
{
  
  char tmp[256];
  int zeros=0;
  
  playSamples(form("%s/voice/%d/point.au", SAMPLE_LIBRARY, voice));
  
  for( ; *fraction != '\0'; *fraction++) {
    playSamples(form("%s/voice/%d/%c.au", SAMPLE_LIBRARY, 
		     voice, *fraction));
  }
  
  return(True_);
}


Bool_
SparcAudioHardware::reconfigDevice()
{
  int err;
  char msg[AUDIO_MAX_ENCODE_INFO];
  
  Dev_hdr = File_hdr;
  err = audio_set_play_config(Audio_fd, &Dev_hdr);
  
  switch (err) {
  case AUDIO_SUCCESS:
    return (True_);
    
  case AUDIO_ERR_NOEFFECT:
    // Couldn"t change the device.
    // Check to see if we"re nearly compatible.
    // audio_cmp_hdr() returns >0 if only sample rate difference.

    if (audio_cmp_hdr(&Dev_hdr, &File_hdr) > 0) {
      double  ratio;
      ratio = (double) abs((int)
			   (Dev_hdr.sample_rate - File_hdr.sample_rate)) /
			     (double) File_hdr.sample_rate;
      if (ratio <= SAMPLE_RATE_THRESHOLD) {
	cerr << form("SparcAudioHardware::reconfigDevice(): WARNING:  sampled at %d, playing at %d\n", 
		     File_hdr.sample_rate, Dev_hdr.sample_rate);
	return (True_);
      }
      cerr << form("SparcAudioHardware::reconfigDevice(): sample rate %d not available\n", File_hdr.sample_rate);
      return (False_);
    }
    (void) audio_enc_to_str(&File_hdr, msg);
    cerr << form("SparcAudioHardware::reconfigDevice(): encoding not available: %s\n", msg);
    return (False_);
  default:
    cerr << "SparcAudioHardware::reconfigDevice(): i/o error (set config)\n";
    return(False_);
    /*NOTREACHED*/
  }
}


Bool_
SparcAudioHardware::playSamples(char *filename, double duration)
{
  int ifd, ns, request;
  unsigned char *buf;
  int numsamples, err, cnt;
  
  if ((ifd = open(filename, O_RDONLY)) == -1) {
    cerr  << form("SparcAudioHardware::playSamples: file = %s, %s\n", filename,
		  sys_errlist[errno]);
    return(False_);
  }
  if ((buf = (unsigned char *) malloc(AUDBUFSIZE)) == NULL) {
    cerr << form("SparcAudioHardware::playSamples: can't allocate memory to buffer: %s\n", sys_errlist[errno]);
    close(ifd);
    return(False_);
  }
  err = audio_read_filehdr(ifd, &File_hdr, (char *)NULL, 0);
  
  if (err != AUDIO_SUCCESS) {
    cerr << form("SparcAudioHardware::playSamples:  %s is not a valid audio file\n", filename);
    (void) close(ifd);
    return(False_);
  } 
  /* Check the device configuration */
  if (audio_cmp_hdr(&Dev_hdr, &File_hdr) != 0) {
    // The device does not match the input file.
    // Wait for any old output to drain, then attempt
    // to reconfigure the audio device to match the
    // input data.
    if (audio_drain(Audio_fd, False_) != AUDIO_SUCCESS) {
     cerr << "Error: SparcAudioHardware::playSamples: AUDIO DRAIN error: %s\n";

      (void) close(ifd);
      return(False_);
    }
    if (!reconfigDevice()) {
      (void) close(ifd);
      return(False_);
    }
  }
  // here we copy the data
  if (duration < 0) { // just play the whole file
    while ((cnt = read(ifd, (char *) buf, AUDBUFSIZE)) >= 0) {
      err = write(Audio_fd, (char *) buf, cnt);
      if (err != cnt) {
	cerr << form("SparcAudioHardware::playSamples: output error %s\n",
		     sys_errlist[errno]);
      }
      if (cnt == 0)
	break;
    }
    if (cnt < 0) {
      cerr << form("SparcAudioHardware::playSamples: error reading %s: %s\n",
		   filename, sys_errlist[errno]);
    }
  } else { // play part of the file
    numsamples = (int) (File_hdr.sample_rate * duration);
    while (numsamples > 0) {
      request = AUDBUFSIZE > numsamples ? numsamples : AUDBUFSIZE;
      while ((ns = read(ifd, (char *) buf, request))  > 0) {
	err = write(Audio_fd, (char *) buf, ns);
	if (err != ns) {
	  cerr << form("SparcAudioHardware::playSamples: output error %s\n",
		       sys_errlist[errno]);
	}
	numsamples -= ns;
	request = AUDBUFSIZE > numsamples ? numsamples : AUDBUFSIZE;
      }
      if (lseek(ifd, 0, SEEK_SET) == -1) {
	cerr << form("SparcAudioHardware::playSamples: lseek error: %s\n",
		     sys_errlist[errno]);
      }
      // skip over the audio file header
      audio_read_filehdr(ifd, &File_hdr, (char *)NULL, 0);
    }
  }
  free(buf);
  (void) close(ifd);
  (void) audio_drain(Audio_fd, True_); // hopefully this will wait till done
  return(True_);
}


Bool_
SparcAudioHardware::playSamples( char *abuf, int buflen, int samplerate, 
				double duration)
{
  int ns, err, request;
  
  // Can"t play a buffer that is greater than the audio buffer size (yet)
  if (buflen > AUDBUFSIZE) {
    cerr << form("Error: SparcAudioHardware::playSamples: buffer cannot be greater than %d\n", AUDBUFSIZE);
    return(False_);
  }

  if (duration < 0) { // just play the whole buffer
    ns = buflen;
    err = write(Audio_fd, (char *) abuf, ns);
    if (err != ns) {
      cerr << form("SparcAudioHardware::playSamples: output error %s\n",
		   sys_errlist[errno]);
    }
    if (ns < 0) {
      cerr << form("SparcAudioHardware::playSamples: error reading buffer: %s\n",
		    sys_errlist[errno]);
    }
  } else { // play part of the buffer
    ns = (int) (samplerate * duration);
    while (ns > 0) {
      request = buflen > ns ? ns : buflen;
      err = write(Audio_fd, (char *) abuf, request);
      if (err != request) {
	cerr << form("SparcAudioHardware::playSamples: output error %s\n",
		     sys_errlist[errno]);
	}
      ns -= request;
    }
  }
  return(True_);
}

Bool_
SparcAudioHardware::stopPlayingSamples()
{
  /* flush output queues before exiting */
  if (Audio_fd >= 0) {
    (void) audio_flush_play(Audio_fd);
  }
  return(True_);
}

Bool_
SparcAudioHardware::setOutputPort(unsigned port)
{
  int err;
  
  if (Play_port != port) { // only change if play port has changed
    
    switch(port) {
    case AUDIO_SPEAKER:
    case AUDIO_HEADPHONE: 
      err = audio_set_play_port(Audio_fd, &port);
      break;
    default:
      cerr << form("Error: SparcAudioHardware::setOutputPort: invalid port %d\n",
		   (int) port);
      return(False_);
    }
    if (err != AUDIO_SUCCESS) {
      cerr << form("Error: SparcAudioHardware::setOutputPort: unable to set output port: %s\n", sys_errlist[errno]);
      return(False_);
    } else {
      Play_port = port;
      return(True_);
    }
  } 
  return(True_);
}
	

Bool_
SparcAudioHardware::recordSamples(char *filename, double duration)
{
  char *Info = NULL;
  unsigned Ilen =0;
  int ofd;
  unsigned char *buf;
  int err, cnt;
  struct pollfd pfd;
  unsigned Limit = AUDIO_UNKNOWN_SIZE;   /* recording limit */
  unsigned Size = 0;          /* size of output file */


  if ((buf = (unsigned char *) malloc(AUDBUFSIZE)) == NULL) {
    cerr << form("SparcAudioHardware::recordSamples: can't allocate memory to buffer: %s\n", sys_errlist[errno]);
    return(False_);
  }
  
/*  if((duration == HUGE_VAL) || (duration < 0.)) { */
  if(duration < 0.) {
    cerr  << form("SparcAudioHardware::recordSamples: invalid value for duration\n");
    return(False_);
  }

  if ((ofd = open(filename, O_WRONLY)) == -1) {
    cerr  << form("SparcAudioHardware::recordSamples: file = %s, %s\n", filename,
		  sys_errlist[errno]);
    return(False_);
  }


  /* write the header */
  if (audio_write_filehdr(ofd, &Dev_hdr, Info, Ilen) != AUDIO_SUCCESS) {
    cerr<< "SparcAudioHardware::recordSamples: error writing header\n";    
    (void) close(ofd);
    return(False_);
  }
  
  /* Set the device up for non-blocking reads */
  (void) fcntl(Audio_fd, F_SETFL, fcntl(Audio_fd, F_GETFL, 0) | O_NDELAY);
  
  /*
   * At this point, we're (finally) ready to copy the data.
   * Init a poll() structure, to use when there's nothing to read.
   */
  if (duration > 0) {
    Limit = (int) (Dev_hdr.sample_rate * duration);
  }
  pfd.fd = Audio_fd;
  pfd.events = POLLIN;
  while ((Limit == AUDIO_UNKNOWN_SIZE) || (Limit != 0)) {
    /* Fill the buffer or read to the time limit */
    cnt = read(Audio_fd, (char *)buf,
	       ((Limit != AUDIO_UNKNOWN_SIZE) && (Limit < sizeof (buf)) ?
		(int)Limit : sizeof (buf)));

    if (cnt == 0)           /* normally, eof can't happen */
      break;

    /* If error, probably have to wait for input */
    if (cnt < 0) {
      switch (errno) {
      case EAGAIN:
      case EWOULDBLOCK:
	(void) poll(&pfd, 1L, -1);
	break;
      default:
	cerr << form("Error: SparcAudioHardware::recordSamples: error reading: %s\n", sys_errlist[errno]);
	return(False_);
      }
      continue;
    }
    err = write(ofd, (char *)buf, cnt);
    if (err != cnt) {
	cerr << form("Error: SparcAudioHardware::recordSamples: error writing: %s\n", sys_errlist[errno]);
	break;
      }
    Size += cnt;
    if (Limit != AUDIO_UNKNOWN_SIZE) {
      Limit -= cnt;
    }
  }
  (void) close(ofd);                      /* close input file */

  
  /* Check for error during record */
  if (audio_get_record_error(Audio_fd, (unsigned *)&err) != AUDIO_SUCCESS) {
    cerr << "Error: SparcAudioHardware::recordSamples: error reading device status\n";
  } else {
    if (err) {
      cerr << "WARNING: SparcAudioHardware::recordSamples: data overflow occurred\n";
    }
  }
  close(ofd);
  return(True_);
}



Bool_
SparcAudioHardware::recordSamples(char *buf, int numsamples)
{
  int err, cnt;
  struct pollfd pfd;
  unsigned Limit = numsamples;   /* recording limit */
  
  if( numsamples < 0 ) {
    cerr  << form("SparcAudioHardware::recordSamples: invalid value for number of samples: %d\n", numsamples);
    return(False_);
  }
  
  /* Set the device up for non-blocking reads */
  (void) fcntl(Audio_fd, F_SETFL, fcntl(Audio_fd, F_GETFL, 0) | O_NDELAY);
  
  /*
   * At this point, we're (finally) ready to copy the data.
   * Init a poll() structure, to use when there's nothing to read.
   */

  pfd.fd = Audio_fd;
  pfd.events = POLLIN;

  /* Fill the buffer */
  cnt = read(Audio_fd, (char *)buf, Limit);

  /* If error, probably have to wait for input */
  if (cnt < 0) {
    switch (errno) {
    case EAGAIN:
    case EWOULDBLOCK:
      (void) poll(&pfd, 1L, -1);
      break;
    default:
      cerr << form("Error: SparcAudioHardware::recordSamples: error reading: %s\n", sys_errlist[errno]);
      return(False_);
    }
  }
  
  /* Check for error during record */
  if (audio_get_record_error(Audio_fd, (unsigned *)&err) != AUDIO_SUCCESS) {
    cerr << "Error: SparcAudioHardware::recordSamples: error reading device status\n";
  } else {
    if (err) {
      cerr << "WARNING: SparcAudioHardware::recordSamples: data overflow occurred\n";
    }
  }
  return(True_);
}

  
	
Bool_
SparcAudioHardware::pausePlay()
{
  if (audio_pause_play(Audio_fd) != AUDIO_SUCCESS) {
    cerr << "Error: SparcAudioHardware::pausePlay(): unable to pause play\n";
    return(False_);
  } else {
    return(True_);
  }
}

Bool_
SparcAudioHardware::resumePlay()
{
  if (audio_resume_play(Audio_fd) != AUDIO_SUCCESS) {
    cerr << "Error: SparcAudioHardware::resumePlay(): unable to resume play\n";
    return(False_);
  } else {
    return(True_);
  }
}

Bool_
SparcAudioHardware::reset()
{
  int errflag = 0;
  
  if (Audio_fd < 0) {
    cerr << form("Error: SparcAudioHardware::reset(): audio device not open\n");
    return(False_);    
  }

  if (stopPlayingSamples() == False_) {
    cerr << form("Error: SparcAudioHardware::reset(): Unable to stop play\n");
    return(False_);
  }

  /* initialize volume to default */
  if (setVolume(75) == False_) {
    cerr << form("Error: SparcAudioHardware:: Unable to initialize volume to 75\n");
    errflag++;
  }

  /* set recording volume to default */
  if (setRecordVolume(50) == False_) {
    cerr << form("Error: SparcAudioHardware::reset():  Unable to initialize record volume to 50\n");
    errflag++;
  }

  /* reset instrument number */
  InstrumentNumber = 0;

  /* initialize output port to default */
  if (setOutputPort(AUDIO_SPEAKER) == False_) {
    cerr << form("SparcAudioHardware::reset(): Unable to initialize output port to %d\n", Play_port);
    errflag++;
  }
  /* at the moment, I think the input port is just microphone in */

  if (errflag) {
    cerr << form("Error: SparcAudioHardware::reset(): unable to reset all defaults\n");
    return(False_);
  } else {
    return(True_);
  }
}

Bool_
SparcAudioHardware::selectInstrument(int number)
{
  InstrumentNumber = number;
  return(True_);
}


