/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Project Manager and Principal Investigator:
 *      Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */

#include "TCPSocket.h"
#include "SoundServer.h"
#include "MessageList.h"

#include <signal.h>
#include <stream.h>
#include <time.h>
#include <sys/wait.h>
#include <sys/socket.h>

#ifndef __GNUG__
#include <sysent.h>

#else
#include <std.h>

extern "C" {
int wait3( int*, int options, struct rusage* );
int mknod( const char*, int, int );
};

#ifndef SIG_TYP
#define SIG_TYP void*
#endif

#endif

#include <fcntl.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>



#define  MAXDEVICES  10

enum BOOLEAN_ {False = 0, True = 1};

DEVICETYPE _mySoundDevice;

struct OpenConnection {
  char *Description;
  char *PortName;
  char *HostName;
  char *NamedPipe;
  int pid;
  TCPSocket sock;
  ValList DefaultsTable;
  ValList SystemParameters;
  MessageList Messages;
};

PtrList ConnectionTable;
TCPSocket sock;
TCPSocket *csock;

int NumberLocalDevices;
SoundDeviceEntry *LocalDeviceTable[MAXDEVICES];

int NumberOpenConnections = 0;

BOOLEAN_ ConnectionExists = False;


void establish_connection(char *hostname, char *portname)
{
  cerr << form("naad: establishing connection to %s at %s\n", portname, hostname);
  int hostlen, portlen;
  int  pid, sfd;
  OpenConnection *oc, *cc;

  
  hostlen = strlen(hostname) + 1;
  portlen = strlen(portname) + 1;
  
  ConnectionTable.addElement((void *) (oc = new OpenConnection));

  NumberOpenConnections++;

  if (oc->sock.connectToService(hostname, portname) < 0) {
    cerr << form("Unable to initiate connection to %s at %s : %s\n", 
		 hostname, portname, sys_errlist[errno]);
    csock->send("");      
    cc = (OpenConnection *) ConnectionTable.deleteElement(--NumberOpenConnections);
    delete cc;
    return;
  }
  sfd = oc->sock.getFd();
  char foo;
  // get the tables; they are only sent once
  oc->DefaultsTable.binReceive(sfd);
//  cout << "got defaults\n";
  if (write(sfd, &foo, 1) < 0) {
    perror("write");
  }
  oc->SystemParameters.binReceive(sfd);
//  cout << "got system parameters\n";
  if (write(sfd, &foo, 1) < 0) {
    perror("write");
  }
  oc->Messages.binReceive(sfd);
//  cout << "got messages\n";
  if (write(sfd, &foo, 1) < 0) {
    perror("write");
  }
  
  oc->HostName =  new char[hostlen];
  strcpy(oc->HostName, hostname);
  
  oc->PortName =  new char[portlen];
  strcpy(oc->PortName, portname);
  
  oc->NamedPipe = new char[strlen(NETROOT) + portlen + hostlen];
  sprintf(oc->NamedPipe, "%s/%s@%s", NETROOT, portname, hostname);
  
  if (mknod(oc->NamedPipe, 0010777, 0) < 0) {
    cerr << form("Error: mknod: filename = %s: %s\n", 
		 oc->NamedPipe, sys_errlist[errno]);
  } else {
    if (mknod(form("%s.Ack", oc->NamedPipe), 0010777, 0) < 0) {
      cerr << form("Error: mknod: filename = %s.Ack: %s\n", 
		   oc->NamedPipe, sys_errlist[errno]);
    }
    csock->send(oc->NamedPipe);      
    csock->send(oc->DefaultsTable);
    csock->send(oc->SystemParameters);
    csock->send(oc->Messages);
  }
  
  pid = fork();
  
  if (pid == 0) { /* child */
    char buf, ack, numberInBatch;
    int i;

    char ackfile[256];
    sprintf(ackfile, form("%s.Ack", oc->NamedPipe));

    int pfd = open(oc->NamedPipe, O_RDONLY);

    int ackfd = open(ackfile, O_WRONLY);

// when the child is spawned off, we put an acknowledgement on the ack pipe.
// if no connection is opened in syncrhonous mode, this outstanding ack 
// is never read. Otherwise, it is read (meaning the last synchronous batch
// of commands has been executed) and replaced. 

    if (write(ackfd, &ack, 1) < 0) {
      cerr << form("localsoundserver: error in write %s\n", sys_errlist[errno]);    
    } else {
//      cerr << "localsoundserver: wrote initial ack\n";
    }

// the main loop.

    while(read(pfd, &numberInBatch, 1) != 0) {
      if (read(pfd, &ack, 1) != 1) {
	cerr << "Error: naad: expected acknowledgement status, received \
nothing\n";
      } else {
//	cerr << "naad: Acknowledgement status is " << (int) ack << NL;
      }
      write(sfd, &numberInBatch, 1);
      write(sfd, &ack, 1);
      
      for(i=0; i < numberInBatch; i++) {
	if (read(pfd, &buf, 1) != 1) {
	  cerr << "Error: naad: expected command, received nothing\n";
	}
	Message& m = (oc->Messages).getArg((int) buf);
	m.binReceive(pfd);
	write(sfd, &buf, 1);
	m.binSend(sfd);
      }
      if ((int) ack > 0) { // block till command is completed
//	cout << "naad: waiting for ack\n";
	if (read(sfd, &ack, 1) < 0) {
	  cerr << form("localsoundserver: error in read %s\n", sys_errlist[errno]);
	}
//	cout << "naad : read ack from sound server\n";

	if (write(ackfd, &ack, 1) < 0) {
	  cerr << form("localsoundserver: error in write %s\n", sys_errlist[errno]);
	}
//	cout << "naad: returned ack \n";
      }
    }


    close(pfd);
    close(ackfd);
    unlink(ackfile);
    unlink(oc->NamedPipe);
    close(sfd);
    
    exit(0);
  } else { /* parent */
    oc->pid = pid;
  }
}

void child_death()
{
  int i;
  int statusp, options;
  long tloc;
  int pid;

  OpenConnection *oc, *cc;

  time(&tloc);

  options = WNOHANG|WUNTRACED;
  if ((pid = wait3(&statusp, options, NULL)) != 0) {
    if (WIFSTOPPED(statusp)) {
      cerr << "Warning: child has been stopped due to signal " << (int) WSTOPSIG(statusp) << " at " << ctime(&tloc);
      cerr << "\tTreating the connection as still open.\n";
      return;
    }
    if (WIFSIGNALED(statusp)) {
      cerr << "naad: child terminated due to signal " << (int) WTERMSIG(statusp) << " at " << ctime(&tloc);
    }

    if (WIFEXITED(statusp)) {
      cerr << "naad: child exited with status " << (int) WEXITSTATUS(statusp) << " at " << ctime(&tloc);
    }

    for(i=0; i < ConnectionTable.numberArgs(); i++) {
      oc = (OpenConnection *) ConnectionTable.getElement(i);
      if (pid == oc->pid) {
	cc = (OpenConnection *) ConnectionTable.deleteElement(i);
	if (access(cc->NamedPipe, W_OK) == 0) {
	  unlink(cc->NamedPipe);
	}
	if (access(form("%s.Ack", oc->NamedPipe), R_OK) == 0) {
	  unlink(form("%s.Ack", oc->NamedPipe));
	}
	delete cc;
	NumberOpenConnections--;
	break;
      }
    }
  }
}

void initiate_connection()
{
  
  char *hostname, *portname;
  struct hostent *hentry;
  int i;
  
  OpenConnection *oc;

  
  hostname = csock->recString();
  portname = csock->recString();
  
  if ((hentry = gethostbyname(hostname)) == NULL) {
    cerr << form("Error: gethostbyname: %s\n", sys_errlist[errno]);
  } else {
    delete hostname;
    hostname = new char[strlen(hentry->h_name) + 1];
    strcpy(hostname, hentry->h_name);
  }
  
  ConnectionExists = False;
  
  NumberOpenConnections = ConnectionTable.numberArgs();
  
  for(i=0; i < NumberOpenConnections; i++) {
    oc = (OpenConnection *) ConnectionTable.getElement(i);
    if ((strcmp(hentry->h_name, oc->HostName) == 0)  &&
	(strcmp(portname, oc->PortName) == 0)) {
	  if (access(oc->NamedPipe, W_OK) < 0) {
	    cerr << form("Error: This Should Not Happen: Pipe does not exist when it should: %s\n", oc->NamedPipe);
	    cerr << "Assuming the connection is closed.\n";
	    oc = (OpenConnection *) ConnectionTable.deleteElement(i);
	    delete oc;
	    break;
	  } 
	  csock->send(oc->NamedPipe);
	  csock->send(oc->DefaultsTable);
	  csock->send(oc->SystemParameters);
	  csock->send(oc->Messages);
	  ConnectionExists = True;
	  break;
	}
  }
  if (ConnectionExists == False) { 
    establish_connection(hostname, portname);
  }

}


BOOLEAN_ initlocaldevices()
{
  FILE *fp;
  char s[256];

  if ((fp = fopen(LOCAL_DEVICE_FILE, "r")) == NULL) {
    cerr << form("Error: initlocaldevices(): unable to open %s : %s\n", 
		 LOCAL_DEVICE_FILE, sys_errlist[errno]);
    return(False);
  }
  while (fgets(s, 256, fp) != NULL) {
    if (s[0] != '#') { // AFTERTHOUGHT!!! make a leading # be a comment
                       // character for the line. Would you believe this
                       // was done to save a sentence of explanation in the
                       // damn User Guide???
      LocalDeviceTable[NumberLocalDevices] = new SoundDeviceEntry;
      sscanf(s, "\"%[^\"]\" %s", 
	     LocalDeviceTable[NumberLocalDevices]->Description,
	     LocalDeviceTable[NumberLocalDevices]->PortName);
      NumberLocalDevices++;

      if (NumberLocalDevices == MAXDEVICES) {
	cerr << form("WARNING: unable to handle more than %d devices; ignoring the rest\n", MAXDEVICES);
	break;
      }
    }
  }
  fclose(fp);
  return(True);
}



main(int argc, char *argv[])
{
  
  char command;
  int fd, i;
  filebuf fb;
  long tloc;

  umask(0); 

  if (argc > 1) { // cerr to log file
    if (fb.open(argv[1], output) == 0) {
      cerr << form("Error opening log file %s: %s\n", argv[1],
		   sys_errlist[errno]);
    } else {
      cerr = *new ostream(&fb);
    }
  }

  time(&tloc);
  cerr << "naad: starting at " << ctime(&tloc);

  if (initlocaldevices() == False) {
    cerr << "Assuming I don't have any local hardware\n";
  }

  if (sock.bindToNamedPort(MY_PORT) < 0) {
    cerr << ("Unable to bind to %s, exiting...\n", MY_PORT);
    exit(1);
  }
  
  /* set up signals */
  signal(SIGCHLD, (SIG_TYP) &child_death);

  int keepLooping = True;
  while( keepLooping ) {
    sock.listen(5);

    csock = sock.accept();
    
    time(&tloc);
    cerr << "naad: accepting connection at " << ctime(&tloc);
    fd = csock->getFd();
    while (csock->receive(command) > 0) {
      switch((int) command) {
      case INITIATE_CONNECTION:
	cerr << "naad: initate connection request\n";
	initiate_connection();
	break;
      case QUERY_NUMBER_DEVICES:
	cerr << "naad: query number devices request\n";
	csock->send((char) NumberLocalDevices);
	break;
      case QUERY_ALL_DEVICES:
	cerr << "naad: query all devices request\n";
	for(i=0; i < NumberLocalDevices; i++) {
          csock->send(LocalDeviceTable[i]->Description);
          csock->send(LocalDeviceTable[i]->PortName);
        }
        break;
      }
    }
    close(fd);
  }
}
