/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *                                           TM
 * The Pablo Performance Analysis Environment   software is *not* in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1987-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * Author: Ruth A. Aydt (aydt@cs.uiuc.edu)
 *
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR86-57696,
 * NSF CCR87-06653 and NSF CDA87-22836 (Tapestry), NASA ICLASS Contract
 * No. NAG-1-613, DARPA Contract No. DABT63-91-K-0004, by a grant
 * from the Digital Equipment Corporation External Research Program,
 * and by a collaborative research agreement with the Intel Supercomputer
 * Systems Division.
 *
 */
/*
 * FieldDescriptor.cc - methods of the class used to describe a field
 *			of data.
 *
 *	$Header: /mnt/Pablo-guitar/Stable.2-94/Visual/Src/System/Descriptors/RCS/FieldDescriptor.C,v 1.14 1994/03/15 19:46:20 aydt Exp $
 */
#include <string.h>

#include "FieldDescriptor.h"

#include "DataCharacteristics.h"
#include "Defines.h"
#include "GetPut.h"

static char tmpBuffer[4096];		// used to get/put if conversion needed

FieldDescriptor::FieldDescriptor()
{
	_setClassName( MY_CLASS );
}

FieldDescriptor::FieldDescriptor( const CString& _name )
{
	_setClassName( MY_CLASS );
	setName( _name );
}

FieldDescriptor::FieldDescriptor( const CString& _name, 
				  const Attributes& _attributes,
			   	  const MachineDataType _type,
				  const int _dimension ) 
				: traits( _type, _dimension)
{
	_setClassName( MY_CLASS );
	setName( _name );
	attributes = _attributes;
}

FieldDescriptor::FieldDescriptor( const FieldDescriptor& field )
{
	_setClassName( MY_CLASS );
	setName( field.name );
	attributes = field.attributes;
	traits = field.traits;
}

/*
 *  Parse binary data into Field Descriptor 
 *      <field name><attributes><field type><field dimension>
 *      record name     ::= string
 *      field type      ::= enum of MachineDataType
 *      field dimension ::= int
 */
int
FieldDescriptor::bitsToObj( const char *bufPtr )
{
	int	currentOffset;
	int	my_type; 
	int 	my_dim;

	setName( (char *) bufPtr );
	currentOffset = name.length() + 1;

	currentOffset += attributes.bitsToObj( bufPtr+currentOffset );

	currentOffset += functionGet( bufPtr+currentOffset, &my_type );
	currentOffset += functionGet( bufPtr+currentOffset, &my_dim );
	traits.setTraits ( (MachineDataType)my_type, my_dim );

	return( currentOffset );
}

int
FieldDescriptor::bytesNeeded( ) const
{
	int bytesRequired = name.length() + 1;
	bytesRequired += attributes.bytesNeeded();

	bytesRequired += 2*sizeof( int ) ; 

	return( bytesRequired );
}

int
FieldDescriptor::cbitsToObj( const char *bufPtr, 
			     const DataCharacteristics *cnvInfo )
{
	int currentOffset = 0;
	char *tmpPtr = tmpBuffer;

	do {
	    currentOffset += cnvInfo->functionGet( bufPtr+currentOffset,
						   tmpPtr ) ;
	} while ( *tmpPtr++ != '\0' );

	setName( tmpBuffer );

	currentOffset += attributes.cbitsToObj( bufPtr+currentOffset, cnvInfo );

	int	my_type; 
	int 	my_dim;
	currentOffset += cnvInfo->functionGet( bufPtr+currentOffset, &my_type );
	currentOffset += cnvInfo->functionGet( bufPtr+currentOffset, &my_dim );

	traits.setTraits ( (MachineDataType)my_type, my_dim );

	return( currentOffset );
}

int
FieldDescriptor::cbytesNeeded( const DataCharacteristics *cnvInfo ) const
{
	int bytesRequired = ( name.length() + 1 ) * 
					      cnvInfo->getAtomSize( CHARACTER );
	bytesRequired += attributes.cbytesNeeded( cnvInfo );

	bytesRequired += 2 * cnvInfo->intSize();

	return( bytesRequired );
}

int
FieldDescriptor::objToBits( char *const bufPtr, int bufLen ) const
{
	if ( bytesNeeded() > bufLen ) {
		return( -1 );
	}

	int my_type = traits.getType();
	int my_dim  = traits.getDimension();

        strcpy( (char *)bufPtr, (const char *)name );
        int bytesUsed = name.length() + 1;
	bytesUsed += attributes.objToBits( bufPtr+bytesUsed, bufLen-bytesUsed );
        bytesUsed += functionPut( bufPtr+bytesUsed, &my_type );
        bytesUsed += functionPut( bufPtr+bytesUsed, &my_dim );
        
        return( bytesUsed );
}

int
FieldDescriptor::objToCbits( char *const bufPtr, int bufLen,
			     const DataCharacteristics *cnvInfo ) const
{
	if ( cbytesNeeded( cnvInfo ) > bufLen ) {
		return( -1 );
	}

	int my_type = traits.getType();
	int my_dim  = traits.getDimension();

	int bytesUsed = 0;

	strcpy( tmpBuffer, (const char *)name );
	char *tmpPtr = tmpBuffer;

	do {
	    bytesUsed += cnvInfo->functionPut( bufPtr+bytesUsed, tmpPtr );
	} while ( *tmpPtr++ != '\0' );

	bytesUsed += attributes.objToCbits( bufPtr+bytesUsed, 
					    bufLen-bytesUsed, cnvInfo );
        bytesUsed += cnvInfo->functionPut( bufPtr+bytesUsed, &my_type );
        bytesUsed += cnvInfo->functionPut( bufPtr+bytesUsed, &my_dim );
        
        return( bytesUsed );
}
int 
FieldDescriptor::getDimension() const
{
	return( traits.getDimension() );
}

const DataTraits&
FieldDescriptor::getTraits() const
{
	return( traits );
}

MachineDataType 
FieldDescriptor::getType() const
{
	return( traits.getType() );
}

void 
FieldDescriptor::setDimension( const int _dimension )
{
	traits.setDimension( _dimension );
}

void
FieldDescriptor::setTraits( const DataTraits& _traits )
{
	traits = _traits;
}

void 
FieldDescriptor::setType( const MachineDataType _type )
{
	traits.setType( _type );
}

FieldDescriptor& 
FieldDescriptor::operator=( const FieldDescriptor& field )
{
	setName( field.name );
	attributes = field.attributes;
	traits = field.traits;

	return *this;
}

Boolean_
FieldDescriptor::operator==( const FieldDescriptor& field ) const
{
	if ( ( traits != field.traits )  ||
	     ( name != field.name )  ||
	     ( attributes != field.attributes ) ) {
		return( FALSE_ );
	} else {
		return( TRUE_ );
	}
}

Boolean_
FieldDescriptor::operator!=( const FieldDescriptor& field ) const
{
	if ( ( traits != field.traits )  ||
	     ( name != field.name )  ||
	     ( attributes != field.attributes ) ) {
		return( TRUE_ );
	} else {
		return( FALSE_ );
	}
}

void 		/* virtual */
FieldDescriptor::printOn( ostream& os ) const
{
	os << "<FieldDescriptor begin>\n";
	os << "name: ";
	name.printOn( os ); 
	os << NL;
	traits.printOn( os );
	attributes.printOn( os );
	os << "<FieldDescriptor end>\n";
}

/*
 * Initialize the static data 
 */
const char *const FieldDescriptor::MY_CLASS = "FieldDescriptor";
