/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 *  SystemDepend.c:
 *	This file contains all the system and machine dependent portions
 *	of the portable Pablo software performance instrumentation library.
 *	This particular version was developed for the:
 *
 *		Portable UNIX BSD 4.1 single-processor version,
 *		tested on Sun SPARCstation 2 (SunOS 4.1) and
 *		Digital DECstation 3100 & 5000 (Ultrix 4.1 & 4.2).
 *
 *	All the functions that appear below are REQUIRED by the portable
 *	Pablo trace library.  Their names, return types, and formal parameter
 *	types cannot be changed without concurrent changes to the portable
 *	portions of the trace library.
 *
 *	Most parts of the system dependent library are straightforward,
 *	and the necessary changes for porting to other systems should be
 *	obvious.
 */

#include "Assert.h"
#include "SystemDepend.h"
#include "TraceParam.h"
#include "Trace.h"

char	*traceFileName = (char *) NULL;	/* local trace file name	    */
char	defaultTraceFileName[ 128 ];	/* default trace file name	    */
FILE	*traceFile = (FILE *) NULL;	/* local trace file pointer	    */

unsigned long
	serverAddress = INADDR_ANY;	/* inet address for socket output   */
int	dgramPort = -1;			/* port for datagram output	    */
int	streamPort = -1;		/* port for stream output	    */
int	dgramSocket = -1;		/* socket for trace record output   */
int	streamSocket = -1;		/* socket for record descriptors    */

#ifdef DEBUG

char	*debugFileName = (char *) NULL;	/* debug output file name	    */
char	defaultDebugFileName[ 128 ];	/* default debug file name	    */
FILE	*debugFile = (FILE *) NULL;	/* debug output file pointer	    */

#endif /* DEBUG */


#ifndef TRsetTraceFile

/*
 *	TRsetTraceFile:
 *	   This function is called by newTraceFile before library
 *	   initialization ONLY (i.e. before TRinit called).  It should
 *	   record the name of the trace file requested by the user
 *	   program, and then open that file at initialization time.
 */

void
TRsetTraceFile( fileName )

char	*fileName;
{
	/* Has this function been called previously?  If so, discard	    */
	/* previously passed file name.					    */

	if (( traceFileName != (char *) NULL )
	   && ( traceFileName != defaultTraceFileName ))
		free( traceFileName );

	/* Allocate storage for copy of file name and make copy.	    */

	traceFileName = malloc( 1 + strlen( fileName ) );

	if ( traceFileName == (char *) NULL )
		TRfailure(
			"cannot allocate storage for trace file name" );

	strcpy( traceFileName, fileName );

	/* Un-do socket output mode, if previously set.			    */

	serverAddress = INADDR_ANY;
	dgramPort = streamPort = -1;
}

#endif /* TRsetTraceFile */


#ifndef TRsetDebugFile

#ifdef DEBUG

/*
 *	TRsetDebugFile:
 *	   This function is called by newDebugFile before library
 *	   initialization ONLY (i.e. before TRinit called).  It should
 *	   record the name of the debugging file requested by the user
 *	   program, and then open that file at initialization time.
 */

void
TRsetDebugFile( fileName )

char	*fileName;
{
	/* Has this function been called previously?  If so, discard	    */
	/* previously passed file name.					    */

	if (( debugFileName != (char *) NULL )
	   && ( debugFileName != defaultDebugFileName ))
		free( debugFileName );

	/* Allocate storage for copy of file name and make copy.	    */

	debugFileName = malloc( 1 + strlen( fileName ) );

	if ( debugFileName == (char *) NULL )
		TRfailure(
			"cannot allocate storage for debug file name" );

	strcpy( debugFileName, fileName );
}

#endif /* DEBUG */

#endif /* TRsetDebugFile */


#ifndef TRsetSocketAddress

/*
 *	TRsetSocketAddress:
 *	   This function is called by newSocketAddress before library
 *	   initialization ONLY (i.e. before TRinit called).  It merely
 *	   records the socket address and port(s) specified by the user
 *	   program, performing the socket set-up at initialization time.
 */

void
TRsetSocketAddress( socketAddress, udpPort, tcpPort )

unsigned long	socketAddress;
int		udpPort;
int		tcpPort;
{
	serverAddress = socketAddress;
	dgramPort = udpPort;
	streamPort = tcpPort;

	/* Un-do file output mode, if previously set.			    */

	if (( traceFileName != (char *) NULL )
	   && ( traceFileName != defaultTraceFileName )) {

		free( traceFileName );
		traceFileName = (char *) NULL;
	}
}

#endif /* TRsetSocketAddress */


#ifndef TRinit

/*
 *	TRinit:
 *	   This function performs the system-dependent portion of trace
 *	   library initialization.  It is invoked by the portable trace
 *	   library code, exactly once, when the first trace library
 *	   interface function is executed.  This implementation does
 *	   trace output initialization.
 */

void
TRinit()
{
	struct sockaddr_in	server;

	if ( serverAddress != INADDR_ANY ) {

	    /* initialize output socket					    */

	    dgramSocket = socket( AF_INET, SOCK_DGRAM, 0 );

	    if ( dgramSocket < 0 )
		TRfailure( "cannot create datagram socket for output" );

	    server.sin_family = AF_INET;
	    server.sin_port = htons( dgramPort );
	    server.sin_addr.s_addr = serverAddress;

	    /* "connect" the datagram socket to simplify coding		    */

	    if ( connect( dgramSocket, (struct sockaddr *) &server,
			  sizeof server ) < 0 )
		TRfailure( "cannot connect datagram socket" );

	    if ( streamPort > 0 ) {

		/* initialize stream socket				    */

		streamSocket = socket( AF_INET, SOCK_STREAM, 0 );

		if ( streamSocket < 0 )
		    TRfailure( "cannot create stream socket for output" );

		server.sin_family = AF_INET;
		server.sin_port = htons( streamPort );
		server.sin_addr.s_addr = serverAddress;

		if ( connect( streamSocket, (struct sockaddr *) &server,
			      sizeof server ) < 0 )
		    TRfailure( "cannot connect stream socket" );
	    }

	} else {

	    /* initialize local trace output file			    */

	    if ( traceFileName == (char *) NULL ) {
		/* Use default trace file name if not changed by user	    */

		strcpy( defaultTraceFileName, "Pablo.bin" );

		traceFileName = defaultTraceFileName;
	    }

	    traceFile = fopen( traceFileName, "w" );

	    if ( traceFile == (FILE *) NULL ) {
		fprintf( stderr,
		    "Pablo:\tWarning: cannot open trace file %s\n",
			traceFileName );

		traceFile = stdout;		/* last resort		    */

		if ( traceFileName != defaultTraceFileName )
			free( traceFileName );

		strcpy( defaultTraceFileName, "standard output" );

		traceFileName = defaultTraceFileName;
	    }
	}

#ifdef DEBUG
	/* initialize local debugging output file			    */

	if ( debugFileName == (char *) NULL ) {
		/* Use default debug file name if not changed by user	    */

		strcpy( defaultDebugFileName, "Pablo.dbug" );

		debugFileName = defaultDebugFileName;
	}

	debugFile = fopen( debugFileName, "w" );

	if ( debugFile == (FILE *) NULL ) {
		fprintf( stderr,
		    "Pablo:\tWarning: cannot open debug file %s\n",
			debugFileName );

		debugFile = stderr;		/* last resort		    */

		if ( debugFileName != defaultDebugFileName )
			free( debugFileName );

		strcpy( defaultDebugFileName, "standard error" );

		debugFileName = defaultDebugFileName;
	}
#endif	/* DEBUG */

	return;
}

#endif /* TRinit */


#ifndef TRcleanup

/*
 *	TRcleanup:
 *	   This function performs the system-dependent portion of trace
 *	   library termination.  It is invoked by the portable library
 *	   function endTracing, BEFORE the portable library data structures
 *	   are themselves cleaned up.  This implementation does the final
 *	   flushing of the trace buffer, and closing of the trace
 *	   (and possibly debugging) file(s).
 */

void
TRcleanup()
{
	toggleClock();			/* stop the tracing clock	    */

	dumpBuffer();			/* dump the trace records	    */

	toggleClock();			/* restart the clock		    */

	if ( dgramSocket >= 0 ) {

	    close( dgramSocket );	/* close output socket		    */

	    dgramSocket = -1;
	    dgramPort = -1;

	    serverAddress = INADDR_ANY;

	    if ( streamSocket >= 0 ) {

		close( streamSocket );

		streamSocket = -1;
		streamPort = -1;
	    }

	} else {

	    fflush( traceFile );

	    if ( traceFile != stdout )
		fclose( traceFile );		/* close trace file	    */

	    traceFile = (FILE *) NULL;

	    if ( traceFileName != defaultTraceFileName )
		free( traceFileName );

	    traceFileName = (char *) NULL;
	}

#ifdef DEBUG
	fflush( debugFile );

	if ( debugFile != stderr )
		fclose( debugFile );		/* close debug file	    */

	debugFile = (FILE *) NULL;

	if ( debugFileName != defaultDebugFileName )
		free( debugFileName );

	debugFileName = (char *) NULL;
#endif /* DEBUG */
}

#endif /* TRcleanup */


#ifndef TRgetClock

/*
 *	TRgetClock:
 *	   Retrieves the value of the system clock.  When using a clock
 *	   with coarse granularity, many trace events may appear simultaneous
 *	   even though other code was executed between calls to the 
 *	   instrumentation library.  There really isn't much one can do to
 *	   fix such a problem (other than harrassing vendors to provide
 *	   decent clocks at the user level).
 *
 *	   Likewise, two successive calls to some Unix clocks may yield
 *	   values that seem to differ even though the time between the
 *	   calls is less than the clock resolution.  In fact, the system
 *	   is simply incrementing the previous clock value to give the
 *	   illusion that the clock is better than it is.
 */

void
TRgetClock( currentTime )

TR_CLOCK	*currentTime;
{
	/* Note that on most UNIX machines, the application clock has	    */
	/* very poor resolution, typically hundreds of milliseconds.	    */

	ftime( currentTime );
}

#endif /* TRgetClock */


#ifndef TRclockDifference

/*
 *	TRclockDifference:
 *	   Computes the difference between two clock values.
 */

CLOCK
TRclockDifference( endTime, startTime )

TR_CLOCK	endTime;
TR_CLOCK	startTime;
{
	CLOCK	result;
	long	deltaT;

	deltaT = endTime.time - startTime.time;
	deltaT *= 1000;
	deltaT += (long) endTime.millitm - (long) startTime.millitm;

	result.clkLow = (unsigned long) deltaT;
	result.clkHigh = ( deltaT < 0 ) ? -1 : 0;

	return result;
}

#endif /* TRclockDifference */


#ifndef TRincrementClock

/*
 *	TRincrementClock:
 *	   Increments a clock value by a specified amount.  Note that
 *	   this does *not* change the actual time, merely the value of
 *	   the first argument.  This routine is perhaps better called
 *	   "timeAddition."
 */

void
TRincrementClock( clockValue, clockIncrement )

TR_CLOCK	*clockValue;
CLOCK		clockIncrement;
{
	time_t	seconds;
	short	millisecs;
	int	signFlag = 0;

	if ( clockIncrement.clkHigh < 0 ) {
		clockIncrement.clkHigh = ~ clockIncrement.clkHigh;
		clockIncrement.clkLow = 1 + ( ~ clockIncrement.clkLow );
		signFlag = -1;
	}

	seconds = clockValue->time;
	millisecs = (short) clockValue->millitm;

	/* This could be done more simply by leaving the sign on	    */
	/* clockIncrement, but we've been bitten before by C compilers	    */
	/* implementing buggy % operators when one (or both) operand(s)	    */
	/* is (are) negative.  Better to code for worst-case.		    */

	if ( signFlag < 0 ) {
		millisecs -= ( clockIncrement.clkLow % 1000 );
		while ( millisecs < 0 ) {
			millisecs += 1000;
			seconds--;
		}
		seconds -= ( clockIncrement.clkLow / 1000 );
	} else {
		millisecs += ( clockIncrement.clkLow % 1000 );
		while ( millisecs >= 1000 ) {
			millisecs -= 1000;
			seconds++;
		}
		seconds += ( clockIncrement.clkLow / 1000 );
	}

	clockValue->time = seconds;
	clockValue->millitm = millisecs;

	return;
}

#endif /* TRincrementClock */


#ifndef TRgetBuffer

/*
 *	TRgetBuffer:
 *	   Allocates a buffer of the specified size in bytes. 
 *	   Depending on the frequency of buffer allocation and
 *	   deallocation and the efficiency of your system's memory
 *	   allocator, you may want to maintain your own free memory
 *	   pool and allocate from it.
 */

char *
TRgetBuffer( bufferSize )

unsigned int	bufferSize;
{
	return malloc( bufferSize );
}

#endif /* TRgetBuffer */


#ifndef TRfreeBuffer

/*
 *	TRfreeBuffer:
 *	   Deallocates a buffer.  See the note above concerning
 *	   memory management.
 */

void
TRfreeBuffer( bufferPointer )

char	*bufferPointer;
{
	if ( bufferPointer != (char *) 0 )
		free( bufferPointer );
}

#endif /* TRfreeBuffer */


#ifndef TRfailure

/*
 *	TRfailure:
 *	   This routine is invoked on an error elsewhere in the
 *	   code.  At this point, we've concluded that the error
 *	   is not recoverable, so we print an error message and die.
 */

void
TRfailure( errorMessage )

char	*errorMessage;
{
	(void) fprintf( stderr, "Pablo:\tFatal error:\t%s\n",
			errorMessage );

	exit( 1 );
}

#endif /* TRfailure */


#ifndef TRflush

/*
 *	TRflush:
 *	   This routine is invoked when a portion of the trace buffer
 *	   has filled and needs to be emptied in preparation for the
 *	   arrival of additional trace records.
 *
 *	   Depending on the software environment, one may want to
 *	   analyze the data in some way, dump it secondary storage,
 *	   or send it elsewhere via some network.  This simple
 *	   interface merely writes the data to a file or socket.
 */

void
TRflush( bufferStart, bufferEnd )

char	*bufferStart;
char	*bufferEnd;
{
	if ( dgramSocket >= 0 ) {		/* socket output	    */

	    if ( libraryInitLevel == INIT_FULL )

		(void) write( dgramSocket, bufferStart,
			      bufferEnd - bufferStart );

	    else if ( streamSocket >= 0 )

		(void) write( streamSocket, bufferStart,
			      bufferEnd - bufferStart );

	    /* else discard SDDF header or record descriptor		    */

	} else					/* file output		    */

	    (void) fwrite( bufferStart, sizeof(char),
			   bufferEnd - bufferStart, traceFile );
}

#endif /* TRflush */


#ifndef TRdump

/*
 *	TRdump:
 *	   This routine is invoked when the local trace buffer has
 *	   filled.  At this point, a good implementation should force
 *	   not only this buffer to be dumped but also force other
 *	   processors to dump their buffers as well.  By forcing all
 *	   processors to collectively synchronize and dump their
 *	   instrumentation buffers, we minimize the perturbation across
 *	   the application's threads of control.
 *
 *	   For this single-processor UNIX version, there are no
 *	   other nodes, therefore no synchronization issues to be
 *	   concerned with.
 */

void
TRdump()
{
	toggleClock();			/* stop the tracing clock	    */

	dumpBuffer();			/* dump the trace records	    */

	toggleClock();			/* restart the clock		    */
}

#endif /* TRdump */


#ifndef TRbcopy

/*
 *	TRbcopy:
 *	   This routine copies a group of bytes from one location to
 *	   another.  As the name suggests, if your system has a bcopy
 *	   routine, this is the place to use it.
 */

void
TRbcopy( bufferFrom, bufferTo, bufferLength )

char	*bufferFrom;
char	*bufferTo;
int	bufferLength;
{
	bcopy( bufferFrom, bufferTo, bufferLength );
}

#endif /* TRbcopy */


#ifndef TRlock

/*
 *	TRlock:
 *	   This routine provides rudimentary locking for updates to
 *	   instrumentation data structures.  Although the need for this
 *	   may not seem obvious when only one application thread is
 *	   executing on each processor, it is important when the possibility
 *	   exists to asynchronously dump trace buffers.  In this case,
 *	   a trace buffer fill on another processor can interrupt this thread
 *	   to dump its buffer as well.  It is important that buffer pointers
 *	   and other trace library data structures not be modified during
 *	   this process.
 *
 *	   In this single-processor version, this is a no-op.
 */

TR_LOCK
TRlock()
{
	return (TR_LOCK) 0;
}

#endif /* TRlock */


#ifndef TRunlock

/*
 *	TRunlock:
 *	   This is the companion routine to TRlock.
 */

/*ARGSUSED*/
void
TRunlock( restoreValue )

TR_LOCK	restoreValue;
{
	return;
}

#endif /* TRunlock */


#ifndef TRgetNode

/*
 *	TRgetNode:
 *	   Return the local node identifier.
 *	   In this single-processor version, this is trivial.
 */

TRgetNode()
{
	return 0;
}

#endif /* TRgetNode */
