/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

#include "Assert.h"
#include "SystemDepend.h"
#include "TraceParam.h"
#include "Trace.h"

/*
 *  Interface.c:
 *	This file contains the user accessible Pablo trace library
 *	routines.  These routines invoke other routines internal to
 *	the trace library to buffer and record data.  Interface
 *	functions for Fortran appear immediately before their
 *	C counterparts.
 */


/*
 *	traceEvent:
 *	   An event of type eventType has occurred.  Note its occurrence
 *	   and write a trace message to the local trace buffer, if
 *	   complete tracing was enabled for this event type.
 */

traceevent_( eventType, tracePointer, byteCount )

int	*eventType;
char	*tracePointer;
int	*byteCount;
{
	/* The tracePointer argument is NOT a character string pointer,	    */
	/* but rather a general address.  For all we know it could be	    */
	/* an array of integers.  Unfortunately, some Fortran compilers	    */
	/* pass character strings differently, in which case this is	    */
	/* almost guaranteed to fail.  Caveat caller.			    */

	return traceEvent( *eventType, tracePointer, (unsigned) *byteCount );
}


traceEvent( eventType, tracePointer, byteCount )

int		eventType;
char		*tracePointer;
unsigned int	byteCount;
{
	TR_EVENT	*eventPointer, *aggregateEvent;
	TR_LOCK		criticalSection;
	CLOCK		currentTime, lastTime;
	int		traceLevel;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile, "traceEvent:\teventType=%d, byteCount=%u\n",
			eventType, byteCount );
	fflush( debugFile );
#endif /* DEBUG */

	eventPointer =			/* locate the event entry for	    */
	    locateEvent( eventType );	/* this event type		    */

	if ( eventPointer == (TR_EVENT *) 0 )
	    eventPointer =		/* this event type not		    */
		initEvent( eventType );	/* previously initialized	    */

	Assert( eventPointer != (TR_EVENT *) 0 );

	if (( eventPointer->eventFlags & EVENTFLAG_INTERNAL ) == 0 )

		TRACELIB_INIT( INIT_FULL );

	currentTime = getClock();

	if ( eventPointer->eventTraceFunction == (int (*)()) 0 )
		traceLevel = eventPointer->eventSoft;
	else
		traceLevel =		/* apply user's trace function	    */
		    (*eventPointer->eventTraceFunction)
			( eventPointer, currentTime );

	if ( traceLevel > LEVEL_IGNORE ) {
					/* record event occurrence	    */
		lastTime = eventPointer->eventLast;
		eventPointer->eventLast = currentTime;
		eventPointer->eventCount++;

		/* If this is a user event, generate an internal event	    */
		/* of type EVENTID_AGGREGATE.  Then examine the current	    */
		/* (soft) trace level for that internal event type.	    */
		/* If that level is lower than the user event's current	    */
		/* trace level, then this instance of the user event	    */
		/* should be controlled by the aggregate event rate	    */
		/* limits rather than the event-specific function.	    */
		/* This means that even though this individual user event   */
		/* type's trace level may be high enough to warrant	    */
		/* tracing, this may be inhibited when the aggregate event  */
		/* trace level (which is controlled by the aggregate rate   */
		/* of user events) has dropped lower.  Put another way,	    */
		/* at any given time, each user event's trace level is	    */
		/* externally bounded above by the trace level of the	    */
		/* internal event EVENTID_AGGREGATE.			    */

		if (( eventPointer->eventFlags & EVENTFLAG_INTERNAL )
		    == 0 ) {
			countEvent( EVENTID_AGGREGATE );

			aggregateEvent = locateEvent( EVENTID_AGGREGATE );

			if ( aggregateEvent->eventSoft < traceLevel )
				traceLevel = aggregateEvent->eventSoft;
		}

					/* generate full trace record	    */
		if ( traceLevel >= LEVEL_FULL )
		    writeTrace( eventPointer, tracePointer, byteCount );
		else if (( traceLevel >= LEVEL_COUNT )
		     && (( eventPointer->eventCount %
			   eventPointer->eventCountFreq ) == 0 ))
		    writeCount( eventPointer );

		/* If this is an event other than type EVENTID_BADCLOCK	    */
		/* for which a trace record has	just been written AND	    */
		/* if the time interval since the last occurrence of	    */
		/* this event type is less than the defined clock	    */
		/* granularity, note this by generating an internal	    */
		/* event of type EVENTID_BADCLOCK.			    */

		if (( eventType != EVENTID_BADCLOCK ) &&
		    ( traceLevel >= LEVEL_COUNT ) &&
		    ( clockCompare( lastTime, noSuchClock ) != 0 ) &&
		    ( clockCompare( clockSubtract( currentTime, lastTime ),
				    clockPrecision ) < 0 )
		   )
			countEvent ( EVENTID_BADCLOCK );
	}

	TRunlock( criticalSection );	/* exit the critical section	    */

	return SUCCESS;
}


/*
 *	countEvent:
 *	   An event of type eventType has occurred.  Note its occurrence
 *	   and write a trace record to the local trace buffer, if
 *	   count tracing was enabled for this event type.
 */


countevent_( eventType )

int	*eventType;
{
	return countEvent( *eventType );
}


countEvent( eventType )

int	eventType;
{
	TR_EVENT	*eventPointer, *aggregateEvent;
	TR_LOCK		criticalSection;
	CLOCK		currentTime, lastTime;
	int		traceLevel;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile, "countEvent:\teventType=%d\n", eventType );
	fflush( debugFile );
#endif /* DEBUG */

	eventPointer =			/* locate the event entry for	    */
	    locateEvent( eventType );	/* this event type		    */

	if ( eventPointer == (TR_EVENT *) 0 )
	    eventPointer =		/* this event type not		    */
		initEvent( eventType );	/* previously initialized	    */

	Assert( eventPointer != (TR_EVENT *) 0 );

	if (( eventPointer->eventFlags & EVENTFLAG_INTERNAL ) == 0 )

		TRACELIB_INIT( INIT_FULL );

	currentTime = getClock();

	if ( eventPointer->eventTraceFunction == (int (*)()) 0 )
		traceLevel = eventPointer->eventSoft;
	else
		traceLevel =		/* apply user's trace function	    */
		    (*eventPointer->eventTraceFunction)
			( eventPointer, currentTime );

	if ( traceLevel > LEVEL_IGNORE ) {
					/* record event occurrence	    */
		lastTime = eventPointer->eventLast;
		eventPointer->eventLast = currentTime;
		eventPointer->eventCount++;

		/* If this is a user event, generate an internal event	    */
		/* of type EVENTID_AGGREGATE.  Then examine the current	    */
		/* (soft) trace level for that internal event type.	    */
		/* If that level is lower than the user event's current	    */
		/* trace level, then this instance of the user event	    */
		/* should be controlled by the aggregate event rate	    */
		/* limits rather than the event-specific function.	    */
		/* This means that even though this individual user event   */
		/* type's trace level may be high enough to warrant	    */
		/* tracing, this may be inhibited when the aggregate event  */
		/* trace level (which is controlled by the aggregate rate   */
		/* of user events) has dropped lower.  Put another way,	    */
		/* at any given time, each user event's trace level is	    */
		/* externally bounded above by the trace level of the	    */
		/* internal event EVENTID_AGGREGATE.			    */

		if (( eventPointer->eventFlags & EVENTFLAG_INTERNAL )
		    == 0 ) {
			countEvent( EVENTID_AGGREGATE );

			aggregateEvent = locateEvent( EVENTID_AGGREGATE );

			if ( aggregateEvent->eventSoft < traceLevel )
				traceLevel = aggregateEvent->eventSoft;
		}

					/* generate count trace record	    */
		if (( traceLevel >= LEVEL_COUNT ) &&
		   (( eventPointer->eventCount %
		      eventPointer->eventCountFreq ) == 0 ))
		    writeCount( eventPointer );

		/* If this is an event other than type EVENTID_BADCLOCK	    */
		/* for which a trace record has	just been written AND	    */
		/* if the time interval since the last occurrence of	    */
		/* this event type is less than the defined clock	    */
		/* granularity, note this by generating an internal	    */
		/* event of type EVENTID_BADCLOCK.			    */

		if (( eventType != EVENTID_BADCLOCK ) &&
		    ( traceLevel >= LEVEL_COUNT ) &&
		    ( clockCompare( lastTime, noSuchClock ) != 0 ) &&
		    ( clockCompare( clockSubtract( currentTime, lastTime ),
				    clockPrecision ) < 0 )
		   )
			countEvent ( EVENTID_BADCLOCK );
	}

	TRunlock( criticalSection );	/* exit the critical section	    */

	return SUCCESS;
}


/*
 *	putAllCounts:
 *	   Record the current counts of all active event types in the
 *	   local trace buffer.  Event types which have been defined (i.e.
 *	   initialized) but which have not actually occurred do not
 *	   generate trace records.  Count records are not generated
 *	   for internal events, since some of those don't even define
 *	   count type trace records.
 */

putallcounts_()
{
	return putAllCounts();
}


putAllCounts()
{
	TR_EVENT	*eventPointer;
	TR_LOCK		criticalSection;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile, "putAllCounts\n" );
	fflush( debugFile );
#endif /* DEBUG */

	eventPointer = eventSearch( (TR_EVENT *) 0 );

	while ( eventPointer != (TR_EVENT *) 0 ) {
					/* write the current count to	    */
					/* the trace buffer with the	    */
					/* time of the last occurrence	    */
		if (( clockCompare( eventPointer->eventLast,
				    noSuchClock ) != 0 ) &&
		    (( eventPointer->eventFlags & EVENTFLAG_INTERNAL ) == 0 ))

			writeCount( eventPointer );

		eventPointer = eventSearch( eventPointer );
	}

	TRunlock( criticalSection );	/* unlock the critical section	    */

	return SUCCESS;
}


/*
 *	startTimeEvent:
 *	   Start an interval timer for an event of type eventType.  Calls
 *	   to this routine must be matched with calls to endTimeEvent,
 *	   although intervening calls to traceEvent and countEvent are
 *	   expressly permitted.  Nested calls to startTimeEvent and
 *	   endTimeEvent pairs are also permitted.
 */


starttimeevent_( eventType )

int	*eventType;
{
	return startTimeEvent( *eventType );
}


startTimeEvent( eventType )

int	eventType;
{
	TR_EVENT	*eventPointer;
	TR_LOCK		criticalSection;
	TM_LIST		*startingTime;
	int		status;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile, "startTimeEvent:\teventType=%d\n",
		eventType );
	fflush( debugFile );
#endif /* DEBUG */

	eventPointer =			/* locate the event entry for	    */
	    locateEvent( eventType );	/* this event type		    */

	if ( eventPointer == (TR_EVENT *) 0 )
	    eventPointer =		/* this event type not		    */
		initEvent( eventType );	/* previously initialized	    */

	Assert( eventPointer != (TR_EVENT *) 0 );

	if (( eventPointer->eventFlags & EVENTFLAG_INTERNAL ) == 0 )

		TRACELIB_INIT( INIT_FULL );

	/* Allocate a new time list element and push it onto the top of	    */
	/* the stack of interval starting times for this event type.	    */

	startingTime = (TM_LIST *) TRgetBuffer( sizeof(TM_LIST) );
	startingTime->timeNext = eventPointer->eventIntervalStack;
	startingTime->timeValue = getClock();

	eventPointer->eventIntervalStack = startingTime;

	TRunlock( criticalSection );	/* exit the critical section	    */

	return SUCCESS;
}


/*
 *	endTimeEvent:
 *	   Complete an interval timer for an event of type eventType.
 *	   Calls to this routine must be matched with previous calls
 *	   to startTimeEvent.
 */

endtimeevent_( eventType )

int	*eventType;
{
	return endTimeEvent( *eventType );
}


endTimeEvent( eventType )

int	eventType;
{
	TR_EVENT	*eventPointer, *aggregateEvent;
	TR_LOCK		criticalSection;
	TM_LIST		*timeListPointer;
	CLOCK		currentTime, lastTime;
	int		traceLevel;
	int		status;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile, "endTimeEvent:\teventType=%d\n", eventType );
	fflush( debugFile );
#endif /* DEBUG */

	currentTime = getClock();

	eventPointer =			/* locate the event entry for	    */
	    locateEvent( eventType );	/* this event type		    */

	if ( eventPointer == (TR_EVENT *) 0 )
	    eventPointer =		/* this event type not		    */
		initEvent( eventType );	/* previously initialized	    */

	Assert( eventPointer != (TR_EVENT *) 0 );

	if (( eventPointer->eventFlags & EVENTFLAG_INTERNAL ) == 0 )

		TRACELIB_INIT( INIT_FULL );

	if ( eventPointer->eventIntervalStack != (TM_LIST *) 0 ) {

		if ( eventPointer->eventTraceFunction == (int (*)()) 0 )
			traceLevel = eventPointer->eventSoft;
		else
			traceLevel =	/* apply user's trace function	    */
			    (*eventPointer->eventTraceFunction)
				( eventPointer, currentTime );

		/* Note that if the user trace function returns		    */
		/* LEVEL_IGNORE, it will be just as if the interval	    */
		/* timer had been cleared out with resetInterval.	    */

		if ( traceLevel > LEVEL_IGNORE ) {
					/* record event occurrence	    */
			lastTime = eventPointer->eventLast;
			eventPointer->eventLast = currentTime;
			eventPointer->eventCount++;

			/* If this is a user event, generate an		    */
			/* internal event of type EVENTID_AGGREGATE.	    */
			/* Then examine the current (soft) trace level	    */
			/* for that internal event type.  If that level	    */
			/* is lower than the user event's current	    */
			/* trace level, then this instance of the user	    */
			/* event should be controlled by the aggregate	    */
			/* event rate limits rather than the event-	    */
			/* specific function.				    */
			/* This means that even though this individual	    */
			/* user event type's trace level may be high	    */
			/* enough to warrant tracing, this may be inhibited */
			/* when the aggregate event trace level (which is   */
			/* controlled by the aggregate rate of user events) */
			/* has dropped lower.  Put another way, at any	    */
			/* given time, each user event's trace level is	    */
			/* externally bounded above by the trace level of   */
			/* the internal event EVENTID_AGGREGATE.	    */

			if (( eventPointer->eventFlags
			    & EVENTFLAG_INTERNAL ) == 0 ) {
				countEvent( EVENTID_AGGREGATE );

				aggregateEvent =
					locateEvent( EVENTID_AGGREGATE );

				if ( aggregateEvent->eventSoft
				   < traceLevel )
					traceLevel =
						aggregateEvent->eventSoft;
			}

					/* generate trace record	    */
			if ( traceLevel >= LEVEL_INTERVAL )
				writeInterval( eventPointer );
			else if (( traceLevel >= LEVEL_COUNT )
			     && (( eventPointer->eventCount %
				   eventPointer->eventCountFreq ) == 0 ))
				writeCount( eventPointer );

			/* If this is an event other than type		    */
			/* EVENTID_BADCLOCK for which a trace record	    */
			/* has just been written AND if the time	    */
			/* interval since the last occurrence of this	    */
			/* event type is less than the defined clock	    */
			/* granularity, note this by generating an	    */
			/* internal event of type EVENTID_BADCLOCK.	    */

			if (( eventType != EVENTID_BADCLOCK ) &&
			    ( traceLevel >= LEVEL_COUNT ) &&
			    ( clockCompare( lastTime, noSuchClock ) != 0 ) &&
			    ( clockCompare( clockSubtract( currentTime,
							   lastTime ),
					    clockPrecision ) < 0 )
			   )
				countEvent ( EVENTID_BADCLOCK );
		}

					/* erase outstanding interval	    */
		timeListPointer = eventPointer->eventIntervalStack->timeNext;
		TRfreeBuffer( (char *) eventPointer->eventIntervalStack );
		eventPointer->eventIntervalStack = timeListPointer;

		status = SUCCESS;

	} else {

		status = FAILURE;	/* no matching startTimeEvent	    */

	}

	TRunlock( criticalSection );

	return status;
}


/*
 *	configureEvent:
 *	   Set the user-configurable parameters for an event type.
 *	   These include the event identifier, the trace level (counting,
 *	   tracing, etc), the low and high watermarks for adaptive
 *	   conversion of tracing to counting, an optional user-
 *	   provided function to be applied to each occurrence of this
 *	   event before recording, another (possibly user-provided)
 *	   function which assembles the trace records for this event
 *	   type, and the count writing frequency.
 */

configureevent_( eventType, traceLevel, lowWater, highWater,
	         traceFunction, recordFunction, countFrequency )

int		*eventType;
int		*traceLevel;
CLOCK		*lowWater;
CLOCK		*highWater;
int		(*traceFunction)();
TR_RECORD	*(*recordFunction)();
int		*countFrequency;
{
	return configureEvent( *eventType, (unsigned) *traceLevel,
			       *lowWater, *highWater,
			       traceFunction, recordFunction,
			       (unsigned long) *countFrequency );
}


configureEvent( eventType, traceLevel, lowWater, highWater,
	        traceFunction, recordFunction, countFrequency )

int		eventType;
unsigned int	traceLevel;
CLOCK		lowWater;
CLOCK		highWater;
int		(*traceFunction)();
TR_RECORD	*(*recordFunction)();
unsigned long	countFrequency;
{
	TR_EVENT	*eventPointer;
	TR_LOCK		criticalSection;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile,
		"configureEvent:\teventType=%d\n", eventType );
	fflush( debugFile );
#endif /* DEBUG */

	eventPointer =			/* locate the event entry for	    */
	    locateEvent( eventType );	/* this event type		    */

	if ( eventPointer == (TR_EVENT *) 0 )
	    eventPointer =		/* this event type not		    */
		initEvent( eventType );	/* previously initialized	    */

	Assert( eventPointer != (TR_EVENT *) 0 );

	if (( LEVEL_IGNORE <= traceLevel )
	   && ( traceLevel <= LEVEL_MAXIMUM )) {
					/* set new trace level limit	    */
		eventPointer->eventHard = traceLevel;
		eventPointer->eventSoft = traceLevel;
	}

	/* The threshold for conversion of traces to counts determines	    */
	/* when tracing can be adaptively disabled in favor of counts,	    */
	/* to reduce instrumentation overhead.				    */

	if (( clockCompare( zeroClock, lowWater ) <= 0 ) &&
	    ( clockCompare( lowWater, highWater ) < 0 )) {
		eventPointer->eventLowWater = lowWater;
		eventPointer->eventHighWater = highWater;
	}

	eventPointer->eventTraceFunction = traceFunction;

	if ( recordFunction != (TR_RECORD *(*)()) 0 ) {
		eventPointer->eventRecordFunction = recordFunction;
	}

	if ( countFrequency != 0 ) {
		eventPointer->eventCountFreq = countFrequency;
	}

	TRunlock( criticalSection );	/* exit the critical section	    */

	return SUCCESS;
}


/*
 *	resetCount:
 *	   Reset the count of the number of occurrences of the specified
 *	   event to zero.
 */

resetcount_( eventType )

int	*eventType;
{
	return resetCount( *eventType );
}


resetCount( eventType )

int	eventType;
{
	TR_EVENT	*eventPointer;
	TR_LOCK		criticalSection;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile, "resetCount:\teventType=%d\n", eventType );
	fflush( debugFile );
#endif /* DEBUG */

	eventPointer =			/* locate the event entry for	    */
	    locateEvent( eventType );	/* this event type		    */

	if ( eventPointer == (TR_EVENT *) 0 )
	    eventPointer =		/* this event type not		    */
		initEvent( eventType );	/* previously initialized	    */

	Assert( eventPointer != (TR_EVENT *) 0 );

	eventPointer->eventCount = 0;

	TRunlock( criticalSection );	/* exit the critical section	    */

	return SUCCESS;
}


/*
 *	resetInterval:
 *	   Reset the interval timer associated with eventType, canceling
 *	   out a previous call to startTimeEvent before the corresponding
 *	   call to endTimeEvent.
 */

resetinterval_( eventType )

int	*eventType;
{
	return resetInterval( *eventType );
}


resetInterval( eventType )

int	eventType;
{
	TR_EVENT	*eventPointer;
	TR_LOCK		criticalSection;
	TM_LIST		*timeListPointer;
	int		status;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile, "resetInterval:\teventType=%d\n", eventType );
	fflush( debugFile );
#endif /* DEBUG */

	eventPointer =			/* locate the event entry for	    */
	    locateEvent( eventType );	/* this event type		    */

	if ( eventPointer == (TR_EVENT *) 0 )
	    eventPointer =		/* this event type not		    */
		initEvent( eventType );	/* previously initialized	    */

	Assert( eventPointer != (TR_EVENT *) 0 );

	if ( eventPointer->eventIntervalStack != (TM_LIST *) 0 ) {
					/* cancel interval timer	    */
		timeListPointer = eventPointer->eventIntervalStack->timeNext;
		TRfreeBuffer( (char *) eventPointer->eventIntervalStack );
		eventPointer->eventIntervalStack = timeListPointer;
		status = SUCCESS;
	} else {
		status = FAILURE;	/* interval timer not running	    */
	}

	TRunlock( criticalSection );	/* exit the critical section	    */

	return status;
}


/*
 *	toggleClock:
 *	   Toggle the state of the clock from running to stopped or stopped
 *	   to running.  This function is used during the dumping of buffers
 *	   to calculate buffer dumping overhead.  Every other call to
 *	   this function (the ones which restart the clock) generates
 *	   an internal event of type EVENTID_DUMPTIME.
 */

toggleclock_()
{
	return toggleClock();
}


toggleClock()
{
	TR_EVENT	*eventPointer;
	TR_LOCK		criticalSection;
	TM_LIST		*dumpTime;
	CLOCK		currentTime, clockAdjust;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

	currentTime = getClock();
	
	if ( clockCompare( clockStopped, noSuchClock ) == 0 ) {
#ifdef DEBUG
		fprintf( debugFile, "toggleClock:\tclock stopped\n" );
		fflush( debugFile );
#endif /* DEBUG */
					/* record the current time	    */
		clockStopped = currentTime;

	} else {
#ifdef DEBUG
		fprintf( debugFile, "toggleClock:\tclock started\n" );
		fflush( debugFile );
#endif /* DEBUG */
					/* restart the clock and	    */
					/* account for time stopped	    */
		clockAdjust = clockSubtract( currentTime, clockStopped );
		TRincrementClock( &epoch, clockAdjust );

		eventPointer =		/* locate the event entry	    */
			locateEvent( EVENTID_DUMPTIME );

		Assert( eventPointer != (TR_EVENT *) 0 );

		/* Simulate startTimeEvent for EVENTID_DUMPTIME		    */

		dumpTime = (TM_LIST *) TRgetBuffer( sizeof(TM_LIST) );
		dumpTime->timeNext = eventPointer->eventIntervalStack;
		dumpTime->timeValue = clockSubtract( clockStopped,
						     clockAdjust );

		eventPointer->eventIntervalStack = dumpTime;

		clockStopped = noSuchClock;

		/* Generate the internal event interval trace record	    */

		endTimeEvent( EVENTID_DUMPTIME );
	}

	TRunlock( criticalSection );	/* exit the critical section	    */

	return SUCCESS;
}


/*
 *	endTracing:
 *	   This function performs trace library clean-up at the
 *	   conclusion of tracing.
 */

endtracing_()
{
	return endTracing();
}


endTracing()
{
	/* So long as nothing - not even debugging output - is done	    */
	/* before the cleanupLibrary call, initLibrary() will not need	    */
	/* to be invoked here.  It would be ridiculous if we had to put	    */
	/* an initLibrary call immediately before cleanupLibrary!	    */
	/* In other words, we cannot generate even debugging output	    */
	/* in this function since the calling program may invoke the	    */
	/* endTracing function before the initLibrary function has been	    */
	/* invoked.							    */

	cleanupLibrary();		/* library clean-up		    */
}


/*
 *	setEventLevel:
 *	   Set the maximum (hard limit) trace level for an event type.
 *	   Trace level is unchanged and FAILURE returned if the argument
 *	   value is out of the range LEVEL_IGNORE to LEVEL_MAXIMUM.
 */

seteventlevel_( eventType, traceLevel )

int	*eventType;
int	*traceLevel;
{
	return setEventLevel( *eventType, *traceLevel );
}


setEventLevel( eventType, traceLevel )

int	eventType;
int	traceLevel;
{
	TR_EVENT	*eventPointer;
	TR_LOCK		criticalSection;
	int		status;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile,
		"setEventLevel:\teventType=%d, traceLevel=%d\n",
		eventType, traceLevel );
	fflush( debugFile );
#endif /* DEBUG */

	eventPointer =			/* locate the event entry for	    */
	    locateEvent( eventType );	/* this event type		    */

	if ( eventPointer == (TR_EVENT *) 0 )
	    eventPointer =		/* this event type not		    */
		initEvent( eventType );	/* previously initialized	    */

	Assert( eventPointer != (TR_EVENT *) 0 );

	if (( LEVEL_IGNORE <= traceLevel )
	   && ( traceLevel <= LEVEL_MAXIMUM )) {
					/* change the trace level	    */
		eventPointer->eventHard = (unsigned) traceLevel;
		eventPointer->eventSoft = (unsigned) traceLevel;
		status = SUCCESS;
	} else {
		status = FAILURE;
	}

	TRunlock( criticalSection );	/* exit the critical section	    */

	return status;
}


/*
 *	setDefaultTraceLevel:
 *	   Set the current default trace level, which is used to initialize
 *	   a new event (initEvent).  The default is unchanged if the argument
 *	   value is outside the range LEVEL_IGNORE to LEVEL_MAXIMUM, in which
 *	   case FAILURE will be returned.
 */

setdefaulttracelevel_( traceLevel )

int	*traceLevel;
{
	return setDefaultTraceLevel( *traceLevel );
}


setDefaultTraceLevel( traceLevel )

int	traceLevel;
{
	int		status;

	if (( LEVEL_IGNORE <= traceLevel )
	   && ( traceLevel <= LEVEL_MAXIMUM )) {
					/* set the default trace level	    */
		defaultTraceLevel = traceLevel;
		status = SUCCESS;
	} else {
		status = FAILURE;
	}

	return status;
}


/*
 *	setEventRateLimits:
 *	   Set the high and low incremental event rate margins which
 *	   control adaptive tracing for the specified event type.
 *	   No change of parameters will occur and FAILURE will be
 *	   returned if the arguments violate the requirement
 *	   0 <= lowWater < highWater.
 */

seteventratelimits_( eventType, lowWater, highWater )

int	*eventType;
CLOCK	*lowWater;
CLOCK	*highWater;
{
	return setEventRateLimits( *eventType, *lowWater, *highWater );
}


setEventRateLimits( eventType, lowWater, highWater )

int		eventType;
CLOCK		lowWater;
CLOCK		highWater;
{
	TR_EVENT	*eventPointer;
	TR_LOCK		criticalSection;
	int		status;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile,
		"setEventRateLimits:\teventType=%d\n", eventType );
	fflush( debugFile );
#endif /* DEBUG */

	eventPointer =			/* locate the event entry for	    */
	    locateEvent( eventType );	/* this event type		    */

	if ( eventPointer == (TR_EVENT *) 0 )
	    eventPointer =		/* this event type not		    */
		initEvent( eventType );	/* previously initialized	    */

	Assert( eventPointer != (TR_EVENT *) 0 );

	if (( clockCompare( zeroClock, lowWater ) <= 0 ) &&
	    ( clockCompare( lowWater, highWater ) < 0 )) {
					/* change the rate limits	    */
		eventPointer->eventLowWater = lowWater;
		eventPointer->eventHighWater = highWater;
		status = SUCCESS;
	} else {
		status = FAILURE;
	}

	TRunlock( criticalSection );	/* exit the critical section	    */

	return status;
}


/*
 *	setDefaultRateLimits:
 *	   Set the current default high and low incremental event rate
 *	   limits used to control adaptive tracing.  These are used to
 *	   set a new event (initEvent).  Neither default is changed and
 *	   FAILURE is returned if the rate limits violate the condition
 *	   0 <= lowWater < highWater.
 */

setdefaultratelimits_( lowWater, highWater )

CLOCK	*lowWater;
CLOCK	*highWater;
{
	return setDefaultRateLimits( *lowWater, *highWater );
}


setDefaultRateLimits( lowWater, highWater )

CLOCK	lowWater;
CLOCK	highWater;
{
	int	status;

	if (( clockCompare( zeroClock, lowWater ) <= 0 ) &&
	    ( clockCompare( lowWater, highWater ) < 0 )) {
					/* change the defaults		    */
		defaultLowWater = lowWater;
		defaultHighWater = highWater;
		status = SUCCESS;
	} else {
		status = FAILURE;
	}

	return status;
}


/*
 *	setEventTraceFunction:
 *	   Set the function to be invoked for dynamically determining
 *	   the current trace level for the specified event type.  If a
 *	   null pointer is passed, no function is invoked and the trace
 *	   level remains unchanged.  When events of this type occur in the
 *	   future, this function will be invoked with the same usage as
 *	   adaptiveTrace.  It should return the event type's current
 *	   trace level, used to determine whether and to what extent the
 *	   instance of the event will be traced.
 */

seteventtracefunction_( eventType, traceFunction )

int	*eventType;
int	(*traceFunction)();
{
	return setEventTraceFunction( *eventType, traceFunction );
}


setEventTraceFunction( eventType, traceFunction )

int		eventType;
int		(*traceFunction)();
{
	TR_EVENT	*eventPointer;
	TR_LOCK		criticalSection;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile,
	    "setEventTraceFunction:\teventType=%d, traceFunction=%#lx\n",
		eventType, traceFunction );
	fflush( debugFile );
#endif /* DEBUG */

	eventPointer =			/* locate the event entry for	    */
	    locateEvent( eventType );	/* this event type		    */

	if ( eventPointer == (TR_EVENT *) 0 )
	    eventPointer =		/* this event type not		    */
		initEvent( eventType );	/* previously initialized	    */

	Assert( eventPointer != (TR_EVENT *) 0 );

					/* change the trace function	    */
	eventPointer->eventTraceFunction = traceFunction;

	TRunlock( criticalSection );	/* exit the critical section	    */

	return SUCCESS;
}


/*
 *	setDefaultTraceFunction:
 *	   Set the current default trace function, which is used to set
 *	   a new event (initEvent).  If a null pointer is passed, the
 *	   default remains unchanged and FAILURE is returned.
 */

setdefaulttracefunction_( traceFunction )

int	(*traceFunction)();
{
	return setDefaultTraceFunction( traceFunction );
}


setDefaultTraceFunction( traceFunction )

int	(*traceFunction)();
{
	int	status;

	if ( traceFunction != (int (*)()) 0 ) {
					/* change the default function	    */
		defaultTraceFunction = traceFunction;
		status = SUCCESS;
	} else {
		status = FAILURE;
	}

	return status;
}


/*
 *	setTraceFileName:
 *	   If called prior to trace library initialization, this
 *	   function sets the name of the local trace file to be that
 *	   pointed to by its argument.  If called after library
 *	   initialization, this function has no effect and returns FAILURE.
 */

settracefilename_( traceFileName, stringLength )

char	*traceFileName;
long	stringLength;
{
	/* This Fortran interface is how both the DEC Fortran for Ultrix    */
	/* RISC systems and the Sun Fortran compilers pass character	    */
	/* strings.  This is definitely >>NOT<< how the VAX Fortran	    */
	/* compiler does things!					    */

	return setTraceFileName( traceFileName );
}


setTraceFileName( traceFileName )

char	*traceFileName;
{
	/* Simply call the library internal function to do this		    */

	return newTraceFile( traceFileName );
}


#ifdef DEBUG

/*
 *	setDebugFileName:
 *	   If called prior to trace library initialization, this
 *	   function sets the name of the local debug file to be that
 *	   pointed to by its argument.  If called after library
 *	   initialization, this function has no effect and returns FAILURE.
 */

setdebugfilename_( debugFileName, stringLength )

char	*debugFileName;
long	stringLength;
{
	/* This Fortran interface is how both the DEC Fortran for Ultrix    */
	/* RISC systems and the Sun Fortran compilers pass character	    */
	/* strings.  This is definitely >>NOT<< how the VAX Fortran	    */
	/* compiler does things!					    */

	return setDebugFileName( debugFileName );
}


setDebugFileName( debugFileName )

char	*debugFileName;
{
	/* Simply call the library internal function to do this		    */

	return newDebugFile( debugFileName );
}

#endif /* DEBUG */


/*
 *	setTraceSocketAddress:
 *	   If called prior to trace library initialization, this
 *	   function sets the address of the trace output socket
 *	   to that of its argument.  If called after library
 *	   initialization, this function has no effect and returns
 *	   FAILURE.  The other two arguments are port numbers for
 *	   the datagram socket for output of trace data records
 *	   (mandatory), and for the stream socket for output of
 *	   the SDDF file header and record descriptors (0 if they
 *	   should not be output by this node).
 */

settracesocketaddress_( socketAddress, dgramPort, streamPort )

unsigned long	*socketAddress;
int		*dgramPort;
int		*streamPort;
{
	return setTraceSocketAddress( *socketAddress,
				      *dgramPort, *streamPort );
}


setTraceSocketAddress( socketAddress, dgramPort, streamPort )

unsigned long	socketAddress;
int		dgramPort;
int		streamPort;
{
	return newSocketAddress( socketAddress, dgramPort, streamPort );
}


/*
 *	setRecordDescriptor:
 *	   If called prior to trace library initialization, this
 *	   function sets a pointer to a function which should generate
 *	   record descriptors for all user-defined trace records.
 *	   Upon initializing the trace library, this function will be
 *	   invoked, immediately after the trace library has generated
 *	   its hard-coded record descriptors.  If called after library
 *	   initialization, this function immediately invokes the argument
 *	   function.
 */

setrecorddescriptor_( recordFunction )

int	(*recordFunction)();
{
	return setRecordDescriptor( recordFunction );
}


setRecordDescriptor( recordFunction )

int	(*recordFunction)();
{
	/* Simply call the library internal function to do this		    */

	return newUserRecord( recordFunction );
}


/*
 *	setEventRecordFunction:
 *	   Set the function to be invoked for assembling trace records
 *	   for the specified event type.  If a null pointer is passed,
 *	   the parameter remains unchanged and FAILURE is returned.
 *	   When events of this type occur in the future, this function
 *	   will be invoked to construct an event trace record for this
 *	   event type.  It should return a pointer to the structure
 *	   holding the length of the trace record and a pointer to
 *	   the contents of the trace record itself (see Trace.h for
 *	   the type declaration).
 */

seteventrecordfunction_( eventType, recordFunction )

int		*eventType;
TR_RECORD	*(*recordFunction)();
{
	return setEventRecordFunction( *eventType, recordFunction );
}


setEventRecordFunction( eventType, recordFunction )

int		eventType;
TR_RECORD	*(*recordFunction)();
{
	TR_EVENT	*eventPointer;
	TR_LOCK		criticalSection;
	int		status;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile,
	    "setEventRecordFunction:\teventType=%d, recordFunction=%#lx\n",
		eventType, recordFunction );
	fflush( debugFile );
#endif /* DEBUG */

	eventPointer =			/* locate the event entry for	    */
	    locateEvent( eventType );	/* this event type		    */

	if ( eventPointer == (TR_EVENT *) 0 )
	    eventPointer =		/* this event type not		    */
		initEvent( eventType );	/* previously initialized	    */

	Assert( eventPointer != (TR_EVENT *) 0 );

	if ( recordFunction != (TR_RECORD *(*)()) 0 ) {
					/* change the record function	    */
		eventPointer->eventRecordFunction = recordFunction;
		status = SUCCESS;
	} else {
		status = FAILURE;
	}

	TRunlock( criticalSection );	/* exit the critical section	    */

	return status;
}


/*
 *	setDefaultRecordFunction:
 *	   Set the current default trace record function, which is used
 *	   to set a new event (initEvent).  If a null pointer is passed,
 *	   the default remains unchanged and FAILURE is returned.
 */

setdefaultrecordfunction_( recordFunction )

TR_RECORD	*(*recordFunction)();
{
	return setDefaultRecordFunction( recordFunction );
}


setDefaultRecordFunction( recordFunction )

TR_RECORD	*(*recordFunction)();
{
	int	status;

	if ( recordFunction != (TR_RECORD *(*)()) 0 ) {
					/* change the default function	    */
		defaultRecordFunction = recordFunction;
		status = SUCCESS;
	} else {
		status = FAILURE;
	}

	return status;
}


/*
 *	setEventCountFrequency:
 *	   Set the frequency with which event counts for eventType
 *	   are written out to the trace buffer.  The count write
 *	   frequency is unchanged and FAILURE returned if the argument
 *	   value is not a positive integer.
 */

seteventcountfrequency_( eventType, countFrequency )

int	*eventType;
int	*countFrequency;
{
	return setEventCountFrequency( *eventType, (long) *countFrequency );
}


setEventCountFrequency( eventType, countFrequency )

int	eventType;
long	countFrequency;
{
	TR_EVENT	*eventPointer;
	TR_LOCK		criticalSection;
	int		status;

	criticalSection = TRlock();	/* enter the critical section	    */

	TRACELIB_INIT( INIT_BASIC );	/* initialize library if needed	    */

#ifdef DEBUG
	fprintf( debugFile,
		"setEventCountFrequency:\teventType=%d, countFrequency=%ld\n",
		eventType, countFrequency );
	fflush( debugFile );
#endif /* DEBUG */

	eventPointer =			/* locate the event entry for	    */
	    locateEvent( eventType );	/* this event type		    */

	if ( eventPointer == (TR_EVENT *) 0 )
	    eventPointer =		/* this event type not		    */
		initEvent( eventType );	/* previously initialized	    */

	Assert( eventPointer != (TR_EVENT *) 0 );

	if ( countFrequency > 0 ) {
					/* change the count frequency	    */
		eventPointer->eventCountFreq = countFrequency;
		status = SUCCESS;
	} else {
		status = FAILURE;
	}

	TRunlock( criticalSection );	/* exit the critical section	    */

	return status;
}


/*
 *	setDefaultCountFrequency:
 *	   Set the current default count write frequency, which is used
 *	   to initialize a new event (initEvent).  The default is unchanged
 *	   if the argument value is not a positive integer, in which case
 *	   FAILURE will be returned.
 */

setdefaultcountfrequency_( countFrequency )

int	*countFrequency;
{
	return setDefaultCountFrequency( (long) *countFrequency );
}


setDefaultCountFrequency( countFrequency )

long	countFrequency;
{
	int	status;

	if ( countFrequency > 0 ) {
					/* set the default frequency	    */
		defaultCountFreq = countFrequency;
		status = SUCCESS;
	} else {
		status = FAILURE;
	}

	return status;
}



/*
 *	setTraceLibraryMode:
 *	   Set the trace library operational mode to be either synchronous
 *	   or asynchronous.  The default is synchronous in all trace library
 *	   versions except the sequential portable UNIX implementation, in
 *	   which case the mode is really irrelevant.  This ought to be done
 *	   before trace library initialization, but could be done after,
 *	   although with unpredictable results.
 */

settracelibrarymode_( mode )

int	*mode;
{
	return setTraceLibraryMode( *mode );
}


setTraceLibraryMode( mode )

int	mode;
{
	if (( mode == MODE_SYNC ) || ( mode == MODE_ASYNC )) {

		traceLibraryMode = mode;
		return SUCCESS;
	}

	return FAILURE;
}



/*
 *	setTraceBufferSize:
 *	   Set the size of the trace buffer to the number of bytes
 *	   specified by the non-negative argument.  This may only
 *	   be done before trace library initialization time.
 */

settracebuffersize_ ( bufferSize )

int	*bufferSize;
{
	return setTraceBufferSize( *bufferSize );
}


setTraceBufferSize( bufferSize )

int	bufferSize;
{
	return newBufferSize( bufferSize );
}
