%{
/*
 * This file is part of the Pablo Performance Analysis Environment
 *
 *          (R)
 * The Pablo    Performance Analysis Environment software is NOT in
 * the public domain.  However, it is freely available without fee for
 * education, research, and non-profit purposes.  By obtaining copies
 * of this and other files that comprise the Pablo Performance Analysis
 * Environment, you, the Licensee, agree to abide by the following
 * conditions and understandings with respect to the copyrighted software:
 * 
 * 1.  The software is copyrighted in the name of the Board of Trustees
 *     of the University of Illinois (UI), and ownership of the software
 *     remains with the UI. 
 *
 * 2.  Permission to use, copy, and modify this software and its documentation
 *     for education, research, and non-profit purposes is hereby granted
 *     to Licensee, provided that the copyright notice, the original author's
 *     names and unit identification, and this permission notice appear on
 *     all such copies, and that no charge be made for such copies.  Any
 *     entity desiring permission to incorporate this software into commercial
 *     products should contact:
 *
 *          Professor Daniel A. Reed                 reed@cs.uiuc.edu
 *          University of Illinois
 *          Department of Computer Science
 *          2413 Digital Computer Laboratory
 *          1304 West Springfield Avenue
 *          Urbana, Illinois  61801
 *          USA
 *
 * 3.  Licensee may not use the name, logo, or any other symbol of the UI
 *     nor the names of any of its employees nor any adaptation thereof in
 *     advertizing or publicity pertaining to the software without specific
 *     prior written approval of the UI.
 *
 * 4.  THE UI MAKES NO REPRESENTATIONS ABOUT THE SUITABILITY OF THE
 *     SOFTWARE FOR ANY PURPOSE.  IT IS PROVIDED "AS IS" WITHOUT EXPRESS
 *     OR IMPLIED WARRANTY.
 *
 * 5.  The UI shall not be liable for any damages suffered by Licensee from
 *     the use of this software.
 *
 * 6.  The software was developed under agreements between the UI and the
 *     Federal Government which entitle the Government to certain rights.
 *
 **************************************************************************
 *
 * Developed by: The TAPESTRY Parallel Computing Laboratory
 *		 University of Illinois at Urbana-Champaign
 *		 Department of Computer Science
 *		 1304 W. Springfield Avenue
 *		 Urbana, IL	61801
 *
 * Copyright (c) 1991-1994
 * The University of Illinois Board of Trustees.
 *	All Rights Reserved.
 *
 * PABLO is a registered trademark of
 * The Board of Trustees of the University of Illinois
 * registered in the U.S. Patent and Trademark Office.
 *
 * Author:  Roger J. Noe (noe@cs.uiuc.edu)
 * Contributing Author:  Daniel A. Reed (reed@cs.uiuc.edu)
 * Project Manager and Principal Investigator:
 *	Daniel A. Reed (reed@cs.uiuc.edu)
 *
 * Funded by: National Science Foundation grants NSF CCR87-06653 and
 * NSF CDA87-22836 (Tapestry), DARPA Contract No. DABT63-91-K-0004,
 * by a grant from the Digital Equipment Corporation External Research
 * Program, and by a collaborative research agreement with the Intel
 * Supercomputer Systems Division.
 *
 */

/*
 * Grammar.y:
 *	This is the yacc(1) input file for the Pablo instrumenting C
 *	parser.  The yacc program produces the function yyparse() from
 *	the C language grammar and actions contained within this file.
 */

#include <stdio.h>

#include "Parser.h"

#ifdef DEBUG
# define YYDEBUG	1
#endif /* DEBUG */

%}

%union {
	NODE   *node;
	TOKEN  *token;
}

%token <token>	IDENTIFIER TYPEDEF_NAME

%token <token>	INTEGER_CONSTANT CHARACTER_CONSTANT FLOATING_CONSTANT
%token <token>	STRING_LITERAL

%token <token>	SM CM COL LP RP LB RB LC RC ASTERISK AMPERSAND

%token <token>	PTR_OP MEMBER_OP
%token <token>	NOT_OP COMPL_OP INC_OP DEC_OP PLUS_OP MINUS_OP SIZEOF
%token <token>	DIV_OP MOD_OP
%token <token>	LSHF_OP RSHF_OP
%token <token>	LT_OP LE_OP GT_OP GE_OP
%token <token>	EQ_OP NE_OP
%token <token>	XOR_OP
%token <token>	BITOR_OP
%token <token>	AND_OP
%token <token>	OR_OP
%token <token>	COND_OP
%token <token>	ASSIGN_OP
%token <token>	ADD_ASSIGN SUB_ASSIGN MUL_ASSIGN DIV_ASSIGN MOD_ASSIGN
%token <token>	AND_ASSIGN XOR_ASSIGN OR_ASSIGN LSHF_ASSIGN RSHF_ASSIGN

%token <token>	TYPEDEF EXTERN STATIC AUTO REGISTER
%token <token>	CHAR SHORT INT LONG SIGNED UNSIGNED FLOAT DOUBLE VOID
%token <token>	CONST VOLATILE

%token <token>	STRUCT UNION ENUM

%token <token>	ELLIPSIS

%token <token>	SWITCH CASE DEFAULT IF ELSE WHILE DO FOR CONTINUE BREAK
%token <token>	GOTO RETURN


%type <token>	identifier  unary_operator  assignment_operator

%type <node>	primary_expr  postfix_expr  argument_expr_list  unary_expr
%type <node>	cast_expr  multiplicative_expr  additive_expr  shift_expr
%type <node>	relational_expr  equality_expr  and_expr  exclusive_or_expr
%type <node>	inclusive_or_expr  logical_and_expr  logical_or_expr
%type <node>	conditional_expr  assignment_expr  expr  constant_expr
%type <node>	declaration  declaration_specifiers  init_declarator_list
%type <node>	init_declarator  storage_class_specifier  type_specifier
%type <node>	type_qualifier  struct_or_union_specifier  struct_or_union
%type <node>	struct_declaration_list  struct_declaration
%type <node>	specifier_qualifier_list  struct_declarator_list
%type <node>	struct_declarator  enum_specifier  enumerator_list  enumerator
%type <node>	declarator  direct_declarator  pointer  type_qualifier_list
%type <node>	identifier_list  parameter_type_list  parameter_list
%type <node>	parameter_declaration  type_name  abstract_declarator
%type <node>	direct_abstract_declarator  initializer  initializer_list
%type <node>	statement  labeled_statement  compound_statement
%type <node>	expression_statement  selection_statement  iteration_statement
%type <node>	jump_statement  function_definition  function_body

%start		translation_unit

%%

primary_expr
	: identifier
		{
		  $$ = makePrimaryExpr( $1, (NODE *) NULL, (TOKEN *) NULL );
		}
	| INTEGER_CONSTANT
		{
		  $$ = makePrimaryExpr( $1, (NODE *) NULL, (TOKEN *) NULL );
		}
	| CHARACTER_CONSTANT
		{
		  $$ = makePrimaryExpr( $1, (NODE *) NULL, (TOKEN *) NULL );
		}
	| FLOATING_CONSTANT
		{
		  $$ = makePrimaryExpr( $1, (NODE *) NULL, (TOKEN *) NULL );
		}
	| STRING_LITERAL
		{
		  $$ = makePrimaryExpr( $1, (NODE *) NULL, (TOKEN *) NULL );
		}
	| LP expr RP
		{
		  $$ = makePrimaryExpr( $1, $2, $3 );
		}
	;

postfix_expr
	: primary_expr
		{
		  $$ = $1;
		}
	| postfix_expr LB expr RB
		{
		  $$ = makePostfixExpr( $1, $2, $3, $4 );
		}
	| postfix_expr LP RP
		{
		  $$ = makePostfixExpr( $1, $2, (NODE *) NULL, $3 );

		  if ( $1->nodeToken == IDENTIFIER )
			recordFunctionCall( $$ );
		}
	| postfix_expr LP argument_expr_list RP
		{
		  $$ = makePostfixExpr( $1, $2, $3, $4 );

		  if ( $1->nodeToken == IDENTIFIER )
			recordFunctionCall( $$ );
		}
	| postfix_expr MEMBER_OP identifier
		{
		  $$ = makePostfixExpr( $1, $2, (NODE *) NULL, $3 );
		}
	| postfix_expr PTR_OP identifier
		{
		  $$ = makePostfixExpr( $1, $2, (NODE *) NULL, $3 );
		}
	| postfix_expr INC_OP
		{
		  $$ = makePostfixExpr( $1, $2,
					(NODE *) NULL, (TOKEN *) NULL );
		}
	| postfix_expr DEC_OP
		{
		  $$ = makePostfixExpr( $1, $2,
					(NODE *) NULL, (TOKEN *) NULL );
		}
	;

argument_expr_list
	: assignment_expr
		{
		  $$ = $1;
		}
	| argument_expr_list CM assignment_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

unary_expr
	: postfix_expr
		{
		  $$ = $1;
		}
	| INC_OP unary_expr
		{
		  setStartToken( $2, $1 );
		  freeToken( $1 );
		  $$ = $2;
		}
	| DEC_OP unary_expr
		{
		  setStartToken( $2, $1 );
		  freeToken( $1 );
		  $$ = $2;
		}
	| unary_operator cast_expr
		{
		  setStartToken( $2, $1 );
		  freeToken( $1 );
		  $$ = $2;
		}
	| SIZEOF unary_expr
		{
		  setStartToken( $2, $1 );
		  freeToken( $1 );
		  $$ = $2;
		}
	| SIZEOF LP type_name RP
		{
		  setStartToken( $3, $1 );
		  setEndToken( $3, $4 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeToken( $4 );
		  $$ = $3;
		}
	;

unary_operator
	: AMPERSAND
		{
		  $$ = $1;
		}
	| ASTERISK
		{
		  $$ = $1;
		}
	| PLUS_OP
		{
		  $$ = $1;
		}
	| MINUS_OP
		{
		  $$ = $1;
		}
	| COMPL_OP
		{
		  $$ = $1;
		}
	| NOT_OP
		{
		  $$ = $1;
		}
	;

cast_expr
	: unary_expr
		{
		  $$ = $1;
		}
	| LP type_name RP cast_expr
		{
		  setStartToken( $4, $1 );
		  freeToken( $1 );
		  freeNodes( $2 );
		  freeToken( $3 );
		  $$ = $4;
		}
	;

multiplicative_expr
	: cast_expr
		{
		  $$ = $1;
		}
	| multiplicative_expr ASTERISK cast_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	| multiplicative_expr DIV_OP cast_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	| multiplicative_expr MOD_OP cast_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

additive_expr
	: multiplicative_expr
		{
		  $$ = $1;
		}
	| additive_expr PLUS_OP multiplicative_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	| additive_expr MINUS_OP multiplicative_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

shift_expr
	: additive_expr
		{
		  $$ = $1;
		}
	| shift_expr LSHF_OP additive_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	| shift_expr RSHF_OP additive_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

relational_expr
	: shift_expr
		{
		  $$ = $1;
		}
	| relational_expr LT_OP shift_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	| relational_expr GT_OP shift_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	| relational_expr LE_OP shift_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	| relational_expr GE_OP shift_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

equality_expr
	: relational_expr
		{
		  $$ = $1;
		}
	| equality_expr EQ_OP relational_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	| equality_expr NE_OP relational_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

and_expr
	: equality_expr
		{
		  $$ = $1;
		}
	| and_expr AMPERSAND equality_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

exclusive_or_expr
	: and_expr
		{
		  $$ = $1;
		}
	| exclusive_or_expr XOR_OP and_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

inclusive_or_expr
	: exclusive_or_expr
		{
		  $$ = $1;
		}
	| inclusive_or_expr BITOR_OP exclusive_or_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

logical_and_expr
	: inclusive_or_expr
		{
		  $$ = $1;
		}
	| logical_and_expr AND_OP inclusive_or_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

logical_or_expr
	: logical_and_expr
		{
		  $$ = $1;
		}
	| logical_or_expr OR_OP logical_and_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

conditional_expr
	: logical_or_expr
		{
		  $$ = $1;
		}
	| logical_or_expr COND_OP expr COL conditional_expr
		{
		  setStartNode( $5, $1 );
		  freeNodes( $1 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  freeToken( $4 );
		  $$ = $5;
		}
	;

assignment_expr
	: conditional_expr
		{
		  $$ = $1;
		}
	| unary_expr assignment_operator assignment_expr
		{
		  setStartNode( $3, $1 );
		  freeNodes( $1 );
		  freeToken( $2 );
		  $$ = $3;
		}
	;

assignment_operator
	: ASSIGN_OP
		{
		  $$ = $1;
		}
	| MUL_ASSIGN
		{
		  $$ = $1;
		}
	| DIV_ASSIGN
		{
		  $$ = $1;
		}
	| MOD_ASSIGN
		{
		  $$ = $1;
		}
	| ADD_ASSIGN
		{
		  $$ = $1;
		}
	| SUB_ASSIGN
		{
		  $$ = $1;
		}
	| LSHF_ASSIGN
		{
		  $$ = $1;
		}
	| RSHF_ASSIGN
		{
		  $$ = $1;
		}
	| AND_ASSIGN
		{
		  $$ = $1;
		}
	| XOR_ASSIGN
		{
		  $$ = $1;
		}
	| OR_ASSIGN
		{
		  $$ = $1;
		}
	;

expr
	: assignment_expr
		{
		  $$ = $1;
		}
	| expr CM assignment_expr
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

constant_expr
	: conditional_expr
		{
		  $$ = $1;
		}
	;

declaration
	: declaration_specifiers SM
		{
		  $$ = makeDeclaration( $1, (NODE *) NULL, $2 );
		}
	| declaration_specifiers init_declarator_list SM
		{
		  $$ = makeDeclaration( $1, $2, $3 );
		}
	;

declaration_specifiers
	: storage_class_specifier
		{
		  $$ = $1;
		}
	| storage_class_specifier declaration_specifiers
		{
		  $1->nodeNext = $2;
		  setEndNode( $1, $2 );
		  $$ = $1;
		}
	| type_specifier
		{
		  $$ = $1;
		}
	| type_specifier declaration_specifiers
		{
		  $1->nodeNext = $2;
		  setEndNode( $1, $2 );
		  $$ = $1;
		}
	| type_qualifier
		{
		  $$ = $1;
		}
	| type_qualifier declaration_specifiers
		{
		  $1->nodeNext = $2;
		  setEndNode( $1, $2 );
		  $$ = $1;
		}
	;

init_declarator_list
	: init_declarator
		{
		  $1->nodeList = (NODE *) NULL;
		  $1->nodeEndList = $1;
		  $$ = $1;
		}
	| init_declarator_list CM init_declarator
		{
		  freeToken( $2 );
		  $1->nodeEndList->nodeList = $3;
		  $1->nodeEndList = $3;
		  setEndNode( $1, $3 );
		  $$ = $1;
		}
	;

init_declarator
	: declarator
		{
		  $$ = makeInitDeclarator( $1, (TOKEN *) NULL, (NODE *) NULL );
		}
	| declarator ASSIGN_OP initializer
		{
		  $$ = makeInitDeclarator( $1, $2, $3 );
		}
	;

storage_class_specifier
	: TYPEDEF
		{
		  $$ = makeStorClassSpec( $1 );
		}
	| EXTERN
		{
		  $$ = makeStorClassSpec( $1 );
		}
	| STATIC
		{
		  $$ = makeStorClassSpec( $1 );
		}
	| AUTO
		{
		  $$ = makeStorClassSpec( $1 );
		}
	| REGISTER
		{
		  $$ = makeStorClassSpec( $1 );
		}
	;

type_specifier
	: CHAR
		{
		  $$ = makeTypeSpecifier( $1 );
		}
	| SHORT
		{
		  $$ = makeTypeSpecifier( $1 );
		}
	| INT
		{
		  $$ = makeTypeSpecifier( $1 );
		}
	| LONG
		{
		  $$ = makeTypeSpecifier( $1 );
		}
	| SIGNED
		{
		  $$ = makeTypeSpecifier( $1 );
		}
	| UNSIGNED
		{
		  $$ = makeTypeSpecifier( $1 );
		}
	| FLOAT
		{
		  $$ = makeTypeSpecifier( $1 );
		}
	| DOUBLE
		{
		  $$ = makeTypeSpecifier( $1 );
		}
	| VOID
		{
		  $$ = makeTypeSpecifier( $1 );
		}
	| struct_or_union_specifier
		{
		  $$ = $1;
		}
	| enum_specifier
		{
		  $$ = $1;
		}
	| TYPEDEF_NAME
		{
		  $$ = makeTypeSpecifier( $1 );
		}
	;

type_qualifier
	: CONST
		{
		  $$ = makeTypeQualifier( $1 );
		}
	| VOLATILE
		{
		  $$ = makeTypeQualifier( $1 );
		}
	;

struct_or_union_specifier
	: struct_or_union identifier LC struct_declaration_list RC
		{
		  $4->nodeEndList = (NODE *) NULL;
		  $1->nodeIdentifier = $2;
		  $1->nodeDeclList = $4;
		  setEndToken( $1, $5 );
		  freeToken( $3 );
		  freeToken( $5 );
		  $$ = $1;
		}
	| struct_or_union LC struct_declaration_list RC
		{
		  $3->nodeEndList = (NODE *) NULL;
		  $1->nodeIdentifier = (TOKEN *) NULL;
		  $1->nodeDeclList = $3;
		  setEndToken( $1, $4 );
		  freeToken( $2 );
		  freeToken( $4 );
		  $$ = $1;
		}
	| struct_or_union identifier
		{
		  $1->nodeIdentifier = $2;
		  $1->nodeDeclList = (NODE *) NULL;
		  setEndToken( $1, $2 );
		  $$ = $1;
		}
	;

struct_or_union
	: STRUCT
		{
		  $$ = makeStructOrUnion( $1 );
		}
	| UNION
		{
		  $$ = makeStructOrUnion( $1 );
		}
	;

struct_declaration_list
	: struct_declaration
		{
		  $1->nodeList = (NODE *) NULL;
		  $1->nodeEndList = $1;
		  $$ = $1;
		}
	| struct_declaration_list struct_declaration
		{
		  $1->nodeEndList->nodeList = $2;
		  $1->nodeEndList = $2;
		  setEndNode( $1, $2 );
		  $$ = $1;
		}
	;

struct_declaration
	: specifier_qualifier_list struct_declarator_list SM
		{
		  $$ = makeStructDeclaration( $1, $2, $3 );
		}
	;

specifier_qualifier_list
	: type_specifier
		{
		  $$ = $1;
		}
	| type_specifier specifier_qualifier_list
		{
		  $1->nodeNext = $2;
		  setEndNode( $1, $2 );
		  $$ = $1;
		}
	| type_qualifier
		{
		  $$ = $1;
		}
	| type_qualifier specifier_qualifier_list
		{
		  $1->nodeNext = $2;
		  setEndNode( $1, $2 );
		  $$ = $1;
		}
	;

struct_declarator_list
	: struct_declarator
		{
		  $1->nodeList = (NODE *) NULL;
		  $1->nodeEndList = $1;
		  $$ = $1;
		}
	| struct_declarator_list CM struct_declarator
		{
		  freeToken( $2 );
		  $1->nodeEndList->nodeList = $3;
		  $1->nodeEndList = $3;
		  setEndNode( $1, $3 );
		  $$ = $1;
		}
	;

struct_declarator
	: declarator
		{
		  $$ = makeStructDeclarator( $1, (TOKEN *) NULL,
						(NODE *) NULL );
		}
	| COL constant_expr
		{
		  $$ = makeStructDeclarator( (NODE *) NULL, $1, $2 );
		}
	| declarator COL constant_expr
		{
		  $$ = makeStructDeclarator( $1, $2, $3 );
		}
	;

enum_specifier
	: ENUM LC enumerator_list RC
		{
		  $$ = makeEnumSpecifier( $1, (TOKEN *) NULL, $2, $3, $4 );
		}
	| ENUM identifier LC enumerator_list RC
		{
		  $$ = makeEnumSpecifier( $1, $2, $3, $4, $5 );
		}
	| ENUM identifier
		{
		  $$ = makeEnumSpecifier( $1, $2, (TOKEN *) NULL,
					  (NODE *) NULL, (TOKEN *) NULL );
		}
	;

enumerator_list
	: enumerator
		{
		  $1->nodeList = (NODE *) NULL;
		  $1->nodeEndList = $1;
		  $$ = $1;
		}
	| enumerator_list CM enumerator
		{
		  freeToken( $2 );
		  $1->nodeEndList->nodeList = $3;
		  $1->nodeEndList = $3;
		  setEndNode( $1, $3 );
		  $$ = $1;
		}
	;

enumerator
	: identifier
		{
		  $$ = makeEnumerator( $1, (TOKEN *) NULL, (NODE *) NULL );
		}
	| identifier ASSIGN_OP constant_expr
		{
		  $$ = makeEnumerator( $1, $2, $3 );
		}
	;

declarator
	: direct_declarator
		{
		  $$ = $1;
		}
	| pointer direct_declarator
		{
		  $1->nodeEndList->nodeNext = $2;
		  $1->nodeEndList = (NODE *) NULL;
		  setEndNode( $1, $2 );
		  $$ = $1;
		}
	;

direct_declarator
	: identifier
		{
		  $$ = makeDirectDeclarator( $1, (TOKEN *) NULL,
			    (TOKEN *) NULL, (NODE *) NULL, (NODE *) NULL );
		}
	| LP declarator RP
		{
		  $$ = makeDirectDeclarator( (TOKEN *) NULL, $1, $3, $2,
					     (NODE *) NULL );
		}
	| direct_declarator LB RB
		{
		  $$ = makeDirectDeclarator( $2, $2, $3, $1, (NODE *) NULL );
		}
	| direct_declarator LB constant_expr RB
		{
		  freeNodes( $3 );
		  $$ = makeDirectDeclarator( $2, $2, $4, $1, (NODE *) NULL );
		}
	| direct_declarator LP RP
		{
		  $$ = makeDirectDeclarator( $2, $2, $3, $1, (NODE *) NULL );
		}
	| direct_declarator LP parameter_type_list RP
		{
		  $$ = makeDirectDeclarator( $2, $2, $4, $1, $3 );
		}
	| direct_declarator LP identifier_list RP
		{
		  $$ = makeDirectDeclarator( $2, $2, $4, $1, $3 );
		}
	;

pointer
	: ASTERISK
		{
		  $$ = makePointer( $1, (NODE *) NULL, (NODE *) NULL );
		}
	| ASTERISK type_qualifier_list
		{
		  $$ = makePointer( $1, $2, (NODE *) NULL );
		}
	| ASTERISK pointer
		{
		  $$ = makePointer( $1, $2, (NODE *) NULL );
		}
	| ASTERISK type_qualifier_list pointer
		{
		  $$ = makePointer( $1, $2, $3 );
		}
	;

type_qualifier_list
	: type_qualifier
		{
		  $1->nodeNext = (NODE *) NULL;
		  $1->nodeEndList = $1;
		  $$ = $1;
		}
	| type_qualifier_list type_qualifier
		{
		  $1->nodeEndList->nodeNext = $2;
		  $1->nodeEndList = $2;
		  setEndNode( $1, $2 );
		  $$ = $1;
		}
	;

identifier_list
	: identifier
		{
		  $$ = makeIdentifierList( (NODE *) NULL, (TOKEN *) NULL, $1 );
		}
	| identifier_list CM identifier
		{
		  $$ = makeIdentifierList( $1, $2, $3 );
		}
	;

parameter_type_list
	: parameter_list
		{
		  $$ = makeParameterTypeList( $1,
					    (TOKEN *) NULL, (TOKEN *) NULL );
		}
	| parameter_list CM ELLIPSIS
		{
		  $$ = makeParameterTypeList( $1, $2, $3 );
		}
	;

parameter_list
	: parameter_declaration
		{
		  $1->nodeList = (NODE *) NULL;
		  $1->nodeEndList = $1;
		  $$ = $1;
		}
	| parameter_list CM parameter_declaration
		{
		  freeToken( $2 );
		  $1->nodeEndList->nodeList = $3;
		  $1->nodeEndList = $3;
		  setEndNode( $1, $3 );
		  $$ = $1;
		}
	;

parameter_declaration
	: declaration_specifiers
		{
		  $$ = makeParamDeclaration( $1, (NODE *) NULL );
		}
	| declaration_specifiers declarator
		{
		  $$ = makeParamDeclaration( $1, $2 );
		}
	| declaration_specifiers abstract_declarator
		{
		  $$ = makeParamDeclaration( $1, $2 );
		}
	;

type_name
	: specifier_qualifier_list
		{
		  $$ = $1;
		}
	| specifier_qualifier_list abstract_declarator
		{
		  setEndNode( $1, $2 );
		  freeNodes( $2 );
		  $$ = $1;
		}
	;

abstract_declarator
	: pointer
		{
		  $1->nodeEndList = (NODE *) NULL;
		  $$ = $1;
		}
	| direct_abstract_declarator
		{
		  $$ = $1;
		}
	| pointer direct_abstract_declarator
		{
		  $1->nodeEndList->nodeNext = $2;
		  $1->nodeEndList = (NODE *) NULL;
		  setEndNode( $1, $2 );
		  $$ = $1;
		}
	;

direct_abstract_declarator
	: LP abstract_declarator RP
		{
		  $$ = makeDirAbstrDeclarator( (TOKEN *) NULL, $1, $3, $2,
						(NODE *) NULL );
		}
	| LB RB
		{
		  $$ = makeDirAbstrDeclarator( $1, $1, $2, (NODE *) NULL,
						(NODE *) NULL );
		}
	| LB constant_expr RB
		{
		  freeNodes( $2 );
		  $$ = makeDirAbstrDeclarator( $1, $1, $3, (NODE *) NULL,
						(NODE *) NULL );
		}
	| direct_abstract_declarator LB RB
		{
		  $$ = makeDirAbstrDeclarator( $2, $2, $3, $1, (NODE *) NULL );
		}
	| direct_abstract_declarator LB constant_expr RB
		{
		  freeNodes( $3 );
		  $$ = makeDirAbstrDeclarator( $2, $2, $4, $1, (NODE *) NULL );
		}
	| LP RP
		{
		  $$ = makeDirAbstrDeclarator( $1, $1, $2, (NODE *) NULL,
						(NODE *) NULL );
		}
	| LP parameter_type_list RP
		{
		  $$ = makeDirAbstrDeclarator( $1, $1, $3, (NODE *) NULL, $2 );
		}
	| direct_abstract_declarator LP RP
		{
		  $$ = makeDirAbstrDeclarator( $2, $2, $3, $1, (NODE *) NULL );
		}
	| direct_abstract_declarator LP parameter_type_list RP
		{
		  $$ = makeDirAbstrDeclarator( $2, $2, $4, $1, $3 );
		}
	;

initializer
	: assignment_expr
		{
		  $$ = $1;
		}
	| LC initializer_list RC
		{
		  setStartToken( $2, $1 );
		  setEndToken( $2, $3 );
		  freeToken( $1 );
		  freeToken( $3 );
		  $$ = $2;
		}
	| LC initializer_list CM RC
		{
		  setStartToken( $2, $1 );
		  setEndToken( $2, $4 );
		  freeToken( $1 );
		  freeToken( $3 );
		  freeToken( $4 );
		  $$ = $2;
		}
	;

initializer_list
	: initializer
		{
		  $$ = $1;
		}
	| initializer_list CM initializer
		{
		  setEndNode( $1, $3 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  $$ = $1;
		}
	;

statement
	: labeled_statement
		{
		  $$ = $1;
		}
	| compound_statement
		{
		  $$ = $1;
		}
	| expression_statement
		{
		  $$ = $1;
		}
	| selection_statement
		{
		  $$ = $1;
		}
	| iteration_statement
		{
		  recordLoop( $1 );
		  $$ = $1;
		}
	| jump_statement
		{
		  $$ = $1;
		}
	;

labeled_statement
	: identifier COL statement
		{
		  setStartToken( $3, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  $$ = $3;
		}
	| CASE constant_expr COL statement
		{
		  setStartToken( $4, $1 );
		  freeToken( $1 );
		  freeNodes( $2 );
		  freeToken( $3 );
		  $$ = $4;
		}
	| DEFAULT COL statement
		{
		  setStartToken( $3, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  $$ = $3;
		}
	;

compound_statement
	: LC RC
		{
		  $$ = makeCompoundStatement( $1, $2 );
		}
	| LC
		{
		  contextBegin();
		}
	  compound_statement_body RC
		{
		  contextEnd();
		  $$ = makeCompoundStatement( $1, $4 );
		}

compound_statement_body
	: statement_list
	| declaration_list
	| declaration_list statement_list
	;

declaration_list
	: declaration
		{
		  /* Note that declarations appear in the list pointed
		     to by currentContext->cntxtNodeHead in reverse order.  */

		  $1->nodeList = currentContext->cntxtNodeHead;
		  currentContext->cntxtNodeHead = $1;
		}
	| declaration_list declaration
		{
		  $2->nodeList = currentContext->cntxtNodeHead;
		  currentContext->cntxtNodeHead = $2;
		}
	;

statement_list
	: statement
		{
		  freeNodes( $1 );
		}
	| statement_list statement
		{
		  freeNodes( $2 );
		}
	;

expression_statement
	: SM
		{
		  $$ = makeExpressionStatement( (NODE *) NULL, $1 );
		}
	| expr SM
		{
		  $$ = makeExpressionStatement( $1, $2 );
		}
	;

selection_statement
	: IF LP expr RP statement
		{
		  setStartToken( $5, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  freeToken( $4 );
		  $$ = $5;
		}
	| IF LP expr RP statement ELSE statement
		{
		  setStartToken( $7, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  freeToken( $4 );
		  freeNodes( $5 );
		  freeToken( $6 );
		  $$ = $7;
		}
	| SWITCH LP expr RP statement
		{
		  setStartToken( $5, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  freeToken( $4 );
		  $$ = $5;
		}
	;

iteration_statement
	: WHILE LP expr RP statement
		{
		  setStartToken( $5, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  freeToken( $4 );
		  $$ = $5;
		}
	| DO statement WHILE LP expr RP SM
		{
		  setStartToken( $2, $1 );
		  setEndToken( $2, $7 );
		  freeToken( $1 );
		  freeToken( $3 );
		  freeToken( $4 );
		  freeNodes( $5 );
		  freeToken( $6 );
		  freeToken( $7 );
		  $$ = $2;
		}
	| FOR LP SM SM RP statement
		{
		  setStartToken( $6, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeToken( $3 );
		  freeToken( $4 );
		  freeToken( $5 );
		  $$ = $6;
		}
	| FOR LP SM SM expr RP statement
		{
		  setStartToken( $7, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeToken( $3 );
		  freeToken( $4 );
		  freeNodes( $5 );
		  freeToken( $6 );
		  $$ = $7;
		}
	| FOR LP SM expr SM RP statement
		{
		  setStartToken( $7, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeToken( $3 );
		  freeNodes( $4 );
		  freeToken( $5 );
		  freeToken( $6 );
		  $$ = $7;
		}
	| FOR LP SM expr SM expr RP statement
		{
		  setStartToken( $8, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeToken( $3 );
		  freeNodes( $4 );
		  freeToken( $5 );
		  freeNodes( $6 );
		  freeToken( $7 );
		  $$ = $8;
		}
	| FOR LP expr SM SM RP statement
		{
		  setStartToken( $7, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  freeToken( $4 );
		  freeToken( $5 );
		  freeToken( $6 );
		  $$ = $7;
		}
	| FOR LP expr SM SM expr RP statement
		{
		  setStartToken( $8, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  freeToken( $4 );
		  freeToken( $5 );
		  freeNodes( $6 );
		  freeToken( $7 );
		  $$ = $8;
		}
	| FOR LP expr SM expr SM RP statement
		{
		  setStartToken( $8, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  freeToken( $4 );
		  freeNodes( $5 );
		  freeToken( $6 );
		  freeToken( $7 );
		  $$ = $8;
		}
	| FOR LP expr SM expr SM expr RP statement
		{
		  setStartToken( $9, $1 );
		  freeToken( $1 );
		  freeToken( $2 );
		  freeNodes( $3 );
		  freeToken( $4 );
		  freeNodes( $5 );
		  freeToken( $6 );
		  freeNodes( $7 );
		  freeToken( $8 );
		  $$ = $9;
		}
	;

jump_statement
	: GOTO identifier SM
		{
		  freeToken( $2 );
		  $$ = makeJumpStatement( $1, $3 );
		}
	| CONTINUE SM
		{
		  $$ = makeJumpStatement( $1, $2 );
		}
	| BREAK SM
		{
		  $$ = makeJumpStatement( $1, $2 );
		}
	| RETURN SM
		{
		  $$ = makeJumpStatement( $1, $2 );
		}
	| RETURN expr SM
		{
		  freeNodes( $2 );
		  $$ = makeJumpStatement( $1, $3 );
		}
	;

translation_unit
	: external_declaration
	| translation_unit external_declaration
	;

external_declaration
	: function_definition
		{
		  recordFunctionDefinition( $1 );
		}
	| declaration
		{
		  /* Note that declarations appear in the list pointed
		     to by currentContext->cntxtNodeHead in reverse order.  */

		  $1->nodeList = currentContext->cntxtNodeHead;
		  currentContext->cntxtNodeHead = $1;
		}
	;

function_definition
	: declarator
		{
		  makeFunctionDefinition( (NODE *) NULL, $1 );
		}
	  function_body
		{
		  $$ = $3;
		}
	| declaration_specifiers declarator
		{
		  makeFunctionDefinition( $1, $2 );
		}
	  function_body
		{
		  $$ = $4;
		}
	;

function_body
	: compound_statement
		{
		  $$ = $1;
		}
	| declaration_list compound_statement
		{
		  $$ = $2;
		}
	;

identifier
	: IDENTIFIER
		{
		  $$ = $1;
		}
	;
%%
