/* Xab to SDDF converter created 12/21/93 by Wilson Swee (wilsons@cs.cmu.edu)
This program converts Xab tracefiles to SDDF format for use with the Pablo
system. There are 2 displays which you can view using the configuration
files provided in this distribution - Contour and Matrix displays. 
*/
#include <stream.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>

#include "InitializeStatic.C"

#include "AsciiPipeReader.h"
#include "AsciiPipeWriter.h"
#include "BinaryPipeWriter.h"
#include "InputFileStreamPipe.h"
#include "OutputFileStreamPipe.h"

#include "Attributes.h"
#include "PacketHeader.h"
#include "RecordDictionary.h"
#include "StructureDescriptor.h"

void processRecordFormats( PipeReader *InPipe, PipeWriter *OutPipe, int diag );
void processTraceFile( FILE *inFile, PipeWriter *OutPipe, int diag );

#define MAXXABTYPE	2		/* valid XAB types are 1 to 7 */
#define MAXXABVAL	64

#define PVM_EXIT        0
#define PVM_MCAST       1
#define PVM_MYTID       2
#define PVM_SEND        3
#define PVM_RENTRY      4
#define PVM_SPAWN       5
#define PVM_RDONE       6

RecordDictionary	*RecordDict;	/* our global record dictionary */

main()
{
	RecordDict = new RecordDictionary();	// Our global dictionary

	enum Status { INVALID, VALID };
	Status 	fileStatus;			// Status of file open        
	char    BUF[512];			// Buffer for keyboard input 

	/*******************************************************************
	 *    Get the name of the Ascii SDDF file containing the
	 *    descriptions of the SDDF record formats we will be 
	 *    generating.  Open the file and attach a Reader to it.
	 *******************************************************************/

	InputFileStreamPipe  	*FormatFile;	     // Pipe connected to the
						     // input format file.
	PipeReader		*FormatPipeReader;   // Reads SDDF packets 

	do {
		cout << "Ascii SDDF file containing record formats to output? ";
		cin >> BUF;

		/* 
		 * Create an input file stream pipe instance connected to
		 * the named file on disk.  If file open is successful, 
		 * connect an Ascii SDDF reader to the pipe.
		 */

		FormatFile = new InputFileStreamPipe( BUF );
		if ( FormatFile->successfulOpen() ) {
			fileStatus = VALID;
		        FormatPipeReader = new AsciiPipeReader( FormatFile );
		} else {
			cout << "ERROR: unable to open specified file.\n";
			fileStatus = INVALID;
			delete FormatFile;
		}
	} while ( fileStatus == INVALID );

	/*******************************************************************
	 *    Get the name of the XAB trace file. Open the file.
	 *******************************************************************/

	FILE *traceFile;		// XAB trace file to be converted
	char  traceFileName[256];	// Name of trace file

  	do {
      		cout << "XAB trace file or input? ";
	  	cin >> BUF;
		traceFile = fopen( BUF, "r" );
	  	if ( traceFile == NULL ) {
			cout << "Error opening XAB trace file for input.\n" ;
		} else {
			strcpy( traceFileName, BUF );
		}
	} while ( traceFile == NULL );


	/*******************************************************************
	 *    Ask if the output file should be Ascii or Binary format.
	 * *****************************************************************/

	enum FileType	{ UNDEFINED, ASCII, BINARY };
	FileType	fileType = UNDEFINED;

	while ( fileType == UNDEFINED ) {
	     	cout << "Do you want Ascii or Binary SDDF format (A or B) ? ";
		cin >> BUF;
		if ( BUF[0] == 'a' || BUF[0] == 'A' ) {
			fileType = ASCII;
		} else if ( BUF[0] == 'b' || BUF[0] == 'B' ) {
			fileType = BINARY;
		} else {
			cout << "ERROR: invalid format type.\n";
		}
	}

	/******************************************************************
	 *    Get the name of the output file. Open or Create it and 
	 *    attach a Writer to it.
	 ******************************************************************/

	OutputFileStreamPipe	*OutFile;	// Pipe connected to output file
	PipeWriter   		*PipeWriter;	// Writes SDDF packets to pipe 

	do {
		cout << "Please enter name of the output SDDF file: ";
		cin >> BUF;

		/* 
		 * Create an output file stream pipe instance connected to
		 * the named file on disk.  If file open is successful, 
		 * connect the appropriate (Ascii or Binary)  SDDF writer 
		 * to the pipe.
		 */
		OutFile = new OutputFileStreamPipe( BUF );
		if ( OutFile->successfulOpen() ) {
			fileStatus = VALID;
			if ( fileType == ASCII ) {
		        	PipeWriter = new AsciiPipeWriter( OutFile );
			} else {
				PipeWriter = new BinaryPipeWriter( OutFile );
			}
		} else {
			cout << "ERROR: unable to open specified file.\n";
			fileStatus = INVALID;
			delete OutFile;
		}
	} while ( fileStatus == INVALID );

	/******************************************************************
	 *    Ask if verbose diagnostic messages are desired.
	 ******************************************************************/

        int diagOutput;

        cout << "Do you want detailed diagnostics sent to stderr (Y or N) ? ";
        cin >> BUF;
        if ( BUF[0] == 'y' || BUF[0] == 'Y' ) {
            	diagOutput = 1;
        } else {
            	diagOutput = 0;
        }

	/* Main Stuff */

	/******************************************************************
	 *    Ask if verbose diagnostic messages are desired.
	 ******************************************************************/

	time_t now = time( 0 );
	char *date = ctime( &now );
	char *nl = index( date, '\n');
	if ( nl ) *nl = 0;

	Attributes	attributes;
	attributes.insert( "Trace from XAB data file", traceFileName );
	attributes.insert( "File converted", date );
	PipeWriter->putAttributes( attributes );

	cout << "\nStarting to process format file.\n";
	processRecordFormats( FormatPipeReader, PipeWriter, diagOutput );
	delete FormatPipeReader;		// Finished with Format File
	
	cout << "Starting to process XAB trace file.\n";
	processTraceFile( traceFile, PipeWriter, diagOutput );

	delete PipeWriter;
	delete FormatFile;
	delete OutFile; 			// Closes the file

}

void
processRecordFormats( PipeReader *InPipe, PipeWriter *OutPipe, int diag )
{
	Attributes		*Ap;
	StructureDescriptor	*SDp;
	PacketHeader PH = InPipe->getPacketHeader();

        while( PH.type != PIPE_EMPTY ) {

            switch( PH.type ) {
              case PKT_ATTRIBUTE:
            	  Ap = new Attributes();
                  InPipe->getAttributes( *Ap );
                  OutPipe->putAttributes( *Ap );
                  if ( diag ) cerr << "\n" << *Ap;
                  delete Ap;
                  break;

              case PKT_DESCRIPTOR:
                  SDp = new StructureDescriptor();
                  InPipe->getDescriptor( *SDp );
                  OutPipe->putDescriptor( *SDp, PH.tag );
                  RecordDict->insert( PH.tag, *SDp );
                  if ( diag ) cerr << "\n" << *SDp;
                  delete SDp;
                  break;

              case PKT_DATA:
                  InPipe->getData( RecordDict->fetch( PH.tag ) );
                  OutPipe->putData( RecordDict->fetch( PH.tag ) );
                  if ( diag ) cerr <<"\n"<< ( RecordDict->fetch( PH.tag ) );
                  break;

              case PKT_COMMAND:
                  OutPipe->putCommand( PH.tag );
                  if ( diag ) cerr << "\nCommand tag = " << PH.tag << "\n";
                  break;
            }
            PH = InPipe->getPacketHeader();

        }
}
// Database containing the tid ID number.
// For each new tid encountered, it will be assigned an int+1 from the
// previous one. 

struct tidname {
	char tidchar[20];
} tid_database[100];

int tid_count=0;

int
search_tid_DB(char *searchtid)
{
	int return_val;
	int Found=0;
	int i;
	for (i=0; i<tid_count; i++) {		
	//cout << "in DB: " << tid_database[i].tidchar << endl;
	//cout << "Tid: " << searchtid << endl;
	if (!strcmp(tid_database[i].tidchar, searchtid)) {
		return_val=i;
		Found=1;
		//cout << "Found: " << i << endl;
		break;
	}
	}
	// If Found is False, you want to put tid in tid_database.
	if (!Found) {
		//cout << "new entry" << endl;
		return_val=tid_count;
		strcpy(tid_database[tid_count].tidchar, searchtid);
		Found=0;
		tid_count++;
	}
	return return_val;
}

void simval(int newcode, double clock, char *tid, int lc,int code,char *msg, int rc) {

  int simtidval;
  Array *simmsgstr;
  int simmsgsize=20;

  RecordDict->setValue(newcode, "Time", clock);
  simtidval = search_tid_DB(tid);
  RecordDict->setValue(newcode, "Tid", simtidval);
  RecordDict->setValue(newcode, "Lc", lc);
  simmsgstr = RecordDict->getArrayP( newcode, "Msg" );
  RecordDict->setValue(newcode, "Code", newcode);
  simmsgstr->setDimSizes(&simmsgsize);
  simmsgstr->setCellString(msg,0);
  RecordDict->setValue(newcode, "Return code", rc);
  
}

void
processTraceFile( FILE *inFile, PipeWriter *OutPipe, int diag )
{
	/* 
	 * Keep some count of records and types processed....  
	 */
	int recordCount = 0;
	int typeCount[70];	
	//cout << "in ProcessTraceFile..." << endl;
//	for ( int i = 0; i <= 70 ; typeCount[i++] = 0 );
	int i;
	//cout << "done with init..." << endl;

	/* 
	 * Read the trace records 
	 */
	char	Buffer[2048];	// Buffer to hold record
	char	*Bufp;		// Buffer pointer
	long int	value[10];
	int	valueCnt;
	long double	clock;
	char	*dtid;
	char	string[128];
	int biggestString = 30;
        Array *tidstr, *msgstr, *dtidstr, *hoststr, *Typestr, *progstr;
	Array *ndtidstr;
	int dimSizes[2];
	int record[100];
	int errorFlag = 0;
	int tidsize = 10;
	int msgsize = 20;
	int dtidsize = 10;
	int hostsize = 20;
	int Typesize = 10;
	int flag = 0;
	int Seconds, uSeconds, lc, code, rc;
	char *tid, *msg, *hostname;
	char *Type, *prog;
	int length, mtag, ntids;
	int progsize = 20;
	int *dimSizes2 = new int[1];
	int tidval,dtidval;
	
	cout << "Currently processing record... \n";

	while ( ( Bufp = fgets( Buffer, 2048, inFile ) ) != NULL ) {

	Seconds = atoi(strtok(Bufp," "));
	uSeconds = atoi(strtok(0, " "));
	tid = strtok(0, " ");
	lc = atoi(strtok(0, " "));
	code = atoi(strtok(0, " "));
	msg = strtok(0," ");
	rc = atoi(strtok(0, " "));
	//printf("%d %d %s %d %d %s %d\n", Seconds, uSeconds, tid,
	//					lc, code, msg, rc);

	    /* 
	     * Update our counters and check that values were found 
	     */
	    if ( ( ++recordCount % 100 ) == 0 ) {
		//cout << recordCount << "... ";
		cout.flush();
	    }
	    if ( ( 0 < code ) && ( code <= 70 ) ) {
		++typeCount[code ];		// expected type
	    } else {
		++typeCount[ 0 ];		// unexpected type
	    }

	    /*
	     * All the records should have values for seconds and microseconds.
	     * Merge those into a single clock value.
	     */

	clock = Seconds + uSeconds*0.000001;

	switch ( code ) {
	case 9: // pvm_exit
	  //cout << "pvm_exit" << endl;
	  record[0]=9;	  // pvm_exit is relabelled as 1.
	  simval(PVM_EXIT, clock,tid,lc, code,msg,rc);
	  OutPipe->putData( RecordDict->fetch(PVM_EXIT));
	  if (diag) 
	    cerr << "\n****\n" << (RecordDict->fetch(PVM_EXIT));
	  break;

	case 23: // pvm_mcast
	  //cout << "pvm_mcast" << endl;
	  record[1]=23;

	  strtok(0," ");
	  ntids=atoi(strtok(0," "));
	  
	  tidval = search_tid_DB(tid);
	  for (i=0; i<ntids; i++) {
	    RecordDict->setValue(PVM_SEND, "Time", clock);

	    RecordDict->setValue(PVM_SEND, "Tid", tidval);
	    RecordDict->setValue(PVM_SEND, "Lc", lc);
	    RecordDict->setValue(PVM_SEND, "Code", PVM_SEND);
	    msgstr = RecordDict->getArrayP(PVM_SEND, "Msg" );
	    msgstr->setDimSizes(&msgsize);
	    msgstr->setCellString(msg,0);
	    RecordDict->setValue(PVM_SEND, "Return code", rc);

	    dtid = strtok(0, " ");
	    dtidval=search_tid_DB(dtid);
	    RecordDict->setValue(PVM_SEND,"Dtid",dtidval);


	    RecordDict->setValue(PVM_SEND, "Mtag", 0);
	    RecordDict->setValue(PVM_SEND, "Length", 0);
	    OutPipe->putData( RecordDict->fetch(PVM_SEND));
	    if (diag) 
	      cerr << "\n****\n" << (RecordDict->fetch(PVM_SEND));
	  }
	  break;
	case 26: // pvm_mytid
	  //cout << "pvm_mytid" << endl;
	  record[2]=26;
	  simval(PVM_MYTID, clock,tid,lc, code,msg,rc);
	  dtid=strtok(0," ");
	  dtidval = search_tid_DB(dtid);
	  RecordDict->setValue(PVM_MYTID, "Dtid",dtidval);
	  //cout << " Dtid : " << dtid<< endl;
	  hoststr=RecordDict->getArrayP(PVM_MYTID, "Hostname");
	  hostname=strtok(0," ");
	  hoststr->setDimSizes(&hostsize);
	  hoststr->setCellString(hostname,0);
	  cout << " hostname : " << hostname << endl;
	  Typestr=RecordDict->getArrayP(PVM_MYTID, "Type");
	  Type=strtok(0," ");
	  Typestr->setDimSizes(&Typesize);
	  Typestr->setCellString(Type,0);
	  //cout << "type : " << Type << endl;
	  OutPipe->putData( RecordDict->fetch(PVM_MYTID));
	  if (diag) 
	    cerr << "\n****\n" << (RecordDict->fetch(PVM_MYTID));
	  break;

	case 42:	// pvm_recv_entry
	  //cout << "pvm_recv_entry" << endl;
	  record[3]=42;
	  simval(PVM_RENTRY, clock,tid,lc, code,msg,rc);
	  dtid = strtok(0," ");
	  dtidval = search_tid_DB(dtid);
	  RecordDict->setValue(PVM_RENTRY, "Dtid", dtidval);
	  
	  mtag = atoi(strtok(0," "));
	  RecordDict->setValue(PVM_RENTRY, "Mtag", mtag);
	  OutPipe->putData( RecordDict->fetch(PVM_RENTRY));
	  if (diag) 
	    cerr << "\n****\n" << (RecordDict->fetch(PVM_RENTRY));
	  break;

	case 43:	// pvm_send
	  record[4]=43;
	  simval(PVM_SEND, clock,tid,lc, code,msg,rc);
	  //cout << "pvm_send" << endl;
	  dtid = strtok(0," ");
	  dtidval = search_tid_DB(dtid);
	  RecordDict->setValue(PVM_SEND, "Dtid", dtidval);

	  mtag = atoi(strtok(0," "));
	  RecordDict->setValue(PVM_SEND, "Mtag", mtag);
	  length = atoi(strtok(0, " "));
	  RecordDict->setValue(PVM_SEND, "Length", length);

	  OutPipe->putData( RecordDict->fetch(PVM_SEND));
	  if (diag) 
	    cerr << "\n****\n" << (RecordDict->fetch(PVM_SEND));
	  break;

	case 49:	// pvm_spawn
	  //cout << "pvm_spawn" << endl;
	  record[5]=49;
	  simval(PVM_SPAWN, clock,tid,lc, code,msg,rc);
	  progstr = RecordDict->getArrayP(PVM_SPAWN, "Program");
	  prog = strtok(0, " ");
	  progstr->setDimSizes(&progsize);
	  progstr->setCellString(prog,0);
	  strtok(0," ");
	  ntids=atoi(strtok(0," "));
	  //cout << "in pvm_spawn ntids = " << ntids << endl;
	  RecordDict->setValue(PVM_SPAWN, "Number tids", ntids);
	  ndtidstr = RecordDict->getArrayP(PVM_SPAWN, "ndtids");

	  dimSizes2[0]=ntids;
	  //cout << "initializing " <<endl;
	  ndtidstr->setDimSizes(dimSizes2);
	  cout << "initializing " <<endl;
	  for (i=0; i<ntids; i++) {
	    dtid = strtok(0, " ");
	    dtidval = search_tid_DB(dtid);
	    //cout << "Dtid = " << dtid << " dtidval = ";
	    //cout << dtidval << " i = " << i << endl;
	    ndtidstr->setCellValue(dtidval,i);
	  }
	  //cout << "done with pvm_spawn" << endl;

	  OutPipe->putData( RecordDict->fetch(PVM_SPAWN));
	  break;

	case 64:	// pvm_recv_done
	  //cout << "pvm_recv_done" << endl;
	  record[6]=64;
	  simval(PVM_RDONE, clock,tid,lc, code,msg,rc);
	  dtid = strtok(0," ");
	  //cout << "going to search with dtid = " << dtid << endl;
	  dtidval=search_tid_DB(dtid);
	  RecordDict->setValue(PVM_RDONE, "Dtid", dtidval);

	  mtag = atoi(strtok(0," "));
	  //cout << "Mtag = " << mtag << endl;
	  RecordDict->setValue(PVM_RDONE, "Mtag", mtag);
	  length = atoi(strtok(0, " "));
	  RecordDict->setValue(PVM_RDONE, "Length", length);

	  OutPipe->putData( RecordDict->fetch(PVM_RDONE));
	  if (diag) 
	    cerr << "\n****\n" << (RecordDict->fetch(PVM_RDONE));
	  break;

	default:
		errorFlag=-1;
		printf("Defaulted in switch!!!!");
	}

	    if ( errorFlag == -1 ) {
		cout << "\nWARNING: Record number " << recordCount 
		     << " had unknown type " << value[0] 
		     << " and was skipped.\n";
	    } else if ( errorFlag == 1 ) {
		cout << "WARNING: Record number " << recordCount << " shown "
		     << "had incorrect number of entries and was skipped.\n"
		     << Buffer << "\n";
	    }
	    errorFlag = 0;

	}

	Attributes	attributes;

	sprintf( Buffer, "Total Records Processed: " );
	sprintf( string, "%d", recordCount );

	cout << "\n\n" << Buffer << string << "\n";
	attributes.insert( Buffer, string );
	char *label;
	
	for (int t =0 ; t <= 6; t++) {
		switch (t) {
		  case 0:
			sprintf(Buffer,"Records of type %d <pvm_exit> :",
				record[t]);
			break;
		  case 1:
			sprintf(Buffer, "Records of type %d<pvm_mcast> :",
				record[t]);
			break;
		  case 2:
			sprintf(Buffer, "Records of type %d <pvm_mytid> :",
				record[t]);
		  	break;
		  case 3:
			sprintf(Buffer, "Records of type %d <pvm_send> :",
				record[t]);
			break;
		  case 4:
			sprintf(Buffer,"Records of type %d <pvm_recv_entry> :",
				record[t]);
			break;
		  case 5:
			sprintf(Buffer, "Records of type %d <pvm_spawn> :",
				record[t]);
		 	break;
		  case 6:
			sprintf(Buffer , "Records of type %d <pvm_recv_done>:",
				record[t]);
			break;
		}
			sprintf( string, "%d ", typeCount[record[t]]);
			cout << Buffer << string << "\n";
			attributes.insert(Buffer, string);

	}

	OutPipe->putAttributes( attributes );
}

			
