#!/usr/local/bin/perl
# Yet Another Code Harness Tool
# yacht.pl 1.5 96/05/21
# by Richard Beton
# (C) Roke Manor Research Limited (November 1995)
#{{{  Disclaimer
#
# This software is provided by Roke Manor Research Limited ``as is''
# and any express or implied warranties, including, but not limited to,
# the implied warranties of merchantability and fitness for a
# particular purpose are disclaimed.  In no event shall the
# regents or contributors be liable for any direct, indirect,
# incidental, special, exemplary, or consequential damages
# (including, but not limited to, procurement of substitute
# goods or services; loss of use, data, or profits; or business
# interruption) however caused and on any theory of liability,
# whether in contract, strict liability, or tort (including
# negligence or otherwise) arising in any way out of the use of
# this software, even if advised of the possibility of such
# damage.
#
#}}}  

@occam_primitive_types = ("BOOL", "BYTE",
                          "INT", "INT16", "INT32", "INT64",
                          "REAL32", "REAL64");

#{{{  $twk_tool_kit setup

if ($ENV{"TWK_TOOL_KIT"})
{
    $twk_tool_kit = $ENV{"TWK_TOOL_KIT"};
}
else
{
    $twk_tool_kit = '/cadre/tool_kit';
}

#}}}  

#{{{  sub abort_if_error

sub abort_if_error
{
    my ($value) = @_;

    if ($value != 0)
    {
        print "END\n";

        if ($opt_o ne "")
        {
            select (STDOUT);
            close (OUT);
            unlink ($opt_o);
        }
        exit ($value);
    }
}

#}}}  ------------------------------------------------------------
#{{{  sub debug

sub debug
{
    if ($opt_z)
    {
        printf STDERR @_;
    }
}

#}}}  ------------------------------------------------------------
#{{{  sub curly_arg

sub curly_arg
{
    local ($option) = @_;
    if ($option =~ /\{.*\}/)
    {
        $option =~ s/.*\{(.*)\}.*/$1/;
    }
    else
    { $option = ''; }
    $option;
}

#}}}  ------------------------------------------------------------
#{{{  sub square_arg

sub square_arg
{
    local ($option) = @_;
    if ($option =~ /\[.*\]/)
    {
        $option =~ s/.*(\[.*\]).*/$1/;
    }
    else
    { $option = ''; }
    $option;
}

#}}}  ------------------------------------------------------------
#{{{  sub round_arg

sub round_arg
{
    local ($option) = @_;
    if ($option =~ /\(.*\)/)
    {
        $option =~ s/[^\(]*\((.*)\)[^\)]*/$1/;
    }
    else
    { $option = ''; }
    $option;
}

#}}}  ------------------------------------------------------------
#{{{  sub chopchop
# removes first and last character from a string

sub chopchop
{
    local ($string) = @_;
    $string = substr ($string, 1, length ($string) - 2);
    $string;
}

#}}}  ------------------------------------------------------------

#{{{  sub twk_get_the_file
#
# sub twk_get_the_file runs the Teamwork twk_get utility to extract
# the specified DFD from the Teamwork database and read it, as CDIF,
# into the array @cdif
#

sub twk_get_the_file
{
    local $command;
    local $model;
    local $config;
    local $object;
    local $other;

    if (-x "$twk_tool_kit/twk_get")
    {
        $command = "$twk_tool_kit/twk_get";
        $model   = "-model $opt_m";
        $config  = "-config $opt_c";
        $object  = "-object $opt_d";
        $other   = "-type DFD -noheader -no_notes";
        print STDERR "$command $model $config $object $other\n";
        open (TWK, "$command $model $config $object $other|") ||
            die "yacht: failed to run twk_get\n";
        @cdif = <TWK>;
        close (TWK);
        if ($opt_l) { print STDERR @cdif; }
        chop (@cdif);
        if ($opt_z) { print STDERR $#cdif+1, " lines of CDIF.\n"; }
        debug "\n";
    }
    else
    {
        die "yacht: cannot execute twk_get.\
       Set TWK_TOOL_KIT to the directory containing twk_get.\n";
    }
}

#}}}  ------------------------------------------------------------
#{{{  sub read_the_file
#
# sub read_the_file reads the input file specified by $opt_f into the
# array @cdif
#

sub read_the_file
{
    open (TWK, $opt_f) || die "yacht: cannot read from $opt_f\n";
    @cdif = <TWK>;
    close (TWK);
    if ($opt_l) { print STDERR @cdif; }
    chop (@cdif);
    if ($opt_z) { print STDERR $#cdif+1, " lines of CDIF.\n"; }
    debug "\n";
}

#}}}  ------------------------------------------------------------
#{{{  sub parse_the_cdif
#
# sub parse_the_cdif does all the hard work of converting CDIF/EDIF
# text into internal data structures representing the OCM model
#

#{{{  sub find_dfd_extent
# defines $firstLine and $lastLine

sub find_dfd_extent
{
    for ($i = 0; $i <= $#cdif; $i++)
    {
        if ($cdif[$i] =~ m/ \(timeStamp /)
        #{{{  store timeStamp
        {
            $timeStamp = $cdif[$i];
            $timeStamp =~ s/ *\(timeStamp ([1-9][0-9][0-9][0-9]) ([0-9][0-9]) ([0-9][0-9]).*/$3.$2.$1/;
            print "TIME.STAMP $timeStamp\n";
        }
        #}}}  

        elsif ($cdif[$i] =~ m/ \(modelName /)
        #{{{  store modelName
        {
            $modelName = $cdif[$i];
            $modelName =~ s/ *\(modelName \"(.*)\"\)/$1/;
            print "MODEL.NAME $modelName\n";
        }
        #}}}  

        elsif ($cdif[$i] =~ m/ \(dfdName /)
        #{{{  store dfdName
        {
            $dfdName = $cdif[$i];
            $dfdName =~ s/ *\(dfdName \"(.*)\"\)/$1/;
            print "DFD.NAME   $dfdName\n";
        }
        #}}}  

        elsif ($cdif[$i] =~ m/ \(dfdID /)
        #{{{  store dfdID
        {
            $dfdID = $cdif[$i];
            $dfdID =~ s/ *\(dfdID \"(.*)\"\)/$1/;
            print "DFD.ID $dfdID\n";
        }
        #}}}  

        elsif ($cdif[$i] =~ m/ \(contents/)
        {
            $i++;
            if ($cdif[$i] =~ m/ \(dataFlowDiagram/)
            {
                $firstLine = $i;
            }
            elsif ($firstLine != -1 && $lastLine == -1)
            {
                $lastLine = $i;
            }
        }
    }

    debug "DFD firstLine = $firstLine, lastLine = $lastLine\n";

    if ($firstLine == -1)
    {
        die "Error - $opt_m object $opt_d is not a data flow diagram.\n";
    }
}

#}}}  
#{{{  sub parse_cdif_dfd
# store dfd name

sub parse_cdif_dfd
{
    local ($i) = @_;

    if ($cdif[$i] =~ m/ \(dfdName ""/)
    { }
    else
    {
        $dfdName = $cdif[$i];
        $dfdName =~ s/[ ]*\(dfdName "(.*)".*/$1/;
        print "DFD.NAME $dfdName\n";
    }

    $i; # returned
}

#}}}  
#{{{  sub parse_cdif_process

sub parse_cdif_process
{
    local ($i) = @_;
    local $processName;
    local $processOpt;
    local $instanceNumber;
    local $id;
    local $curly;
    local $square;

    $cdif[$i] =~ s/ +/ /g;
    $processName = $processOpt = $cdif[$i];
    $processName =~ s/.*"([A-Za-z][A-Za-z.0-9%]*).*/$1/;
    $processName =~ s/%010//g;
    $processOpt  =~ s/.*"[A-Za-z][A-Za-z.0-9]*(.*)".*/$1/;
    $processOpt  =~ s/%010/ /g;

    $i++;
    $id = $cdif[$i];
    $id =~ s/.*\(processID \"(.+)\"\).*/$1/;

    $i++;
    $instanceNumber = $cdif[$i];
    $instanceNumber =~ s/.*\(instanceNumber (.+)\).*/$1/;

    if ($largestInum < $instanceNumber)
    { $largestInum = $instanceNumber; }

    $curly  = curly_arg  ($processOpt);
    $square = chopchop (square_arg ($processOpt));
    $round  = round_arg  ($processOpt);
    $round  =~ s/%034/"/g;

    print "BUBBLE $processName\n";
    print "  ID         $instanceNumber\n";
    print "  LABEL      $id\n";
    if ($curly ne '')  { print "  FLAGS      $curly\n"; }
    if ($square ne '') { print "  REPLICATOR $square\n"; }
    if ($round ne '')  { print "  ARGS       $round\n"; }
    print ":\n";

    $processNameList{$processName} = 1; # define on a per-key basis

    $i; # returned
}

#}}}  
#{{{  sub parse_cdif_term

sub parse_cdif_term
{
    local ($i) = @_;
    local $termName;
    local $instanceNumber;

    $i += 2;
    $termName = $cdif[$i];
    $termName =~ s/.*"(.*)".*/$1/;
    $termName =~ s/%010/ /g;
    $termName =~ s/%034/"/g;
    $i += 1;

    $instanceNumber = $cdif[$i];
    $instanceNumber =~ s/.*\(instanceNumber (.+)\).*/$1/;

    if ($largestInum < $instanceNumber)
    { $largestInum = $instanceNumber; }

    print "TERM $termName\n";
    print "  ID         $instanceNumber\n";
    print ":\n";

    $i; # returned
}

#}}}  
#{{{  sub parse_cdif_flow

sub parse_cdif_flow
{
    local ($i) = @_;

    local $flowName = '';
    local $flowOpt  = '';
    local $instanceNumber = -1;
    local $thing = '';
    local $flowType;
    local $sobjInstanceNumber = -1;
    local $dobjInstanceNumber = -1;
    local $end = $cdif[$i];
    $end =~ s/( +)\(.*/$1\)/;
    # $end is the closing parenthesis with correct indentation

    $i++;

    while ($cdif[$i] ne $end && $cdif[$i] ne '')
    {
        if ($cdif[$i] =~ m/ \(flowName /)
        #{{{  
        {
            if ($cdif[$i] =~ m/ \(flowName ""\)/)
            { }
            else
            {
                $cdif[$i] =~ s/%010/ /g;
                $cdif[$i] =~ s/ +/ /g;
                $flowName = $flowOpt = $cdif[$i];
                $flowName =~ s/.*"([A-Za-z][A-Za-z.0-9]*).*/$1/;
                $flowOpt  =~ s/.*"[A-Za-z][A-Za-z.0-9]*(.*)".*/$1/;
                $flowName =~ s/%034/"/g;
                $flowOpt  =~ s/%034/"/g;
                $curly  = &curly_arg  ($flowOpt);
                $square = &square_arg ($flowOpt);
                $round  = &round_arg  ($flowOpt);

                #
                # could add a check that the protocol of a multi-used channel
                # is consistent between different places.
                #
            }
        }
        #}}}  
        elsif ($cdif[$i] =~ m/ \(arrowDirection /)
        #{{{  
        {
            $arrowDirection = $cdif[$i];
            $arrowDirection =~ s/.*"(.*)".*/$1/;
        }
        #}}}  
        elsif ($cdif[$i] =~ m/ \(instanceNumber /)
        #{{{  
        {
            $instanceNumber = $cdif[$i];
            $instanceNumber =~ s/.*\(instanceNumber (.+)\).*/$1/;

            if ($largestInum < $instanceNumber)
            { $largestInum = $instanceNumber; }
        }
        #}}}  
        elsif ($cdif[$i] =~ m/ \(flowType /)
        #{{{  
        {
            $flowType = $cdif[$i];
            $flowType =~ s/.*\(flowType "(.+)"\).*/$1/;
        }
        #}}}  
        elsif ($cdif[$i] =~ m/ \(flowSource/)
        #{{{  
        {
            $i += 2;
            $sobjInstanceNumber = $cdif[$i];
            $sobjInstanceNumber =~ s/.*\(instanceNumber (.+)\).*/$1/;

            $i--;
            $sThing = $cdif[$i];
            $sThing =~ s/.*\(flowObjectType "(.*)".*/$1/;

            if (($sThing !~ m/process/) &&
                ($sThing !~ m/term/))
            {
                $error++;
                print STDERR "Error - channel '$flowName' should not be connected to a ",
                    "$sThing object.\n";
            }

            $i += 2;
        }
        #}}}  
        elsif ($cdif[$i] =~ m/ \(flowDestination/)
        #{{{  
        {
            $i += 2;
            $dobjInstanceNumber = $cdif[$i];
            $dobjInstanceNumber =~ s/.*\(instanceNumber (.+)\).*/$1/;

            $i--;
            $dThing = $cdif[$i];
            $dThing =~ s/.*\(flowObjectType "(.*)".*/$1/;

            if (($dThing !~ m/process/) &&
                ($dThing !~ m/term/))
            {
                $error++;
                print STDERR "Error - channel '$flowName' should not be connected to a ",
                    "$dThing object.\n";
            }

            $i += 2;
        }
        #}}}  
        $i++;
    }

    #{{{  check curly syntax

    if ($curly eq '')
    {
        $curly = "BLOCK";
    }

    $curly =~ tr/ //d;

    if (($curly !~ m/BLOCK/i) &&
        ($curly !~ m/STOP/i)  &&
        ($curly !~ m/OWB/i))
    {
        $error++;
        print STDERR "Error - channel $flowName has invalid buffer mode \'$curly\'\n";
    }

    #}}}  
    #{{{  check consistent channel protocols
    #
    # if a channel name occurs on more than one flow, it is checked
    # for consistency of the channel protocol.
    #

    if ($flowType eq "control")
    {
        if ($cflowNameList{$flowName})
        {
            #
            # check this instance's $curly agains that of the previous instance
            #
            if ($cflowProtocol{$cflowNameList{$flowName}} ne $curly)
            {
                $error++;
                print STDERR "Error - channel $flowName has inconsistent channel protocols.\n";
            }
            if (($sThing =~ m/term/) ||
                ($dThing =~ m/term/))
            {
                $error++;
                print STDERR "Error - channel $flowName cannot connect to a $dThing.\n";
            }
        }
    }
    else
    {
        if ($dflowNameList{$flowName})
        {
            #
            # check this instance's $curly agains that of the previous instance
            #
            if ($dflowProtocol{$dflowNameList{$flowName}} ne $curly)
            {
                $error++;
                print STDERR "Error - channel $flowName has inconsistent channel protocols.\n";
            }
            if (($sThing =~ m/term/) ||
                ($dThing =~ m/term/))
            {
                $error++;
                print STDERR "Error - channel $flowName cannot connect to a $dThing.\n";
            }
        }
    }

    #}}}  
    #{{{  check flowName is defined

    if ($flowName eq '')
    {
        if ($sobjInstanceNumber >= 0 &&
            $dobjInstanceNumber >= 0)
        {
            $error++;
            print STDERR "Error - a channel between '",
                         $processName{$sobjInstanceNumber},
                         "' and '",
                         $processName{$dobjInstanceNumber},
                         "' is unnamed.\n";
        }
        elsif ($sobjInstanceNumber >= 0)
        {
            $error++;
            print STDERR "Error - a channel from '",
                         $processName{$sobjInstanceNumber},
                         "' is unnamed.\n";
        }
        else
        {
            $error++;
            print STDERR "Error - a channel to '",
                         $processName{$dobjInstanceNumber},
                         "' is unnamed.\n";
        }
    }

    #}}}  
    #{{{  check flowName is valid

    if ($processNameList{$flowName})
    {
        $error++;
        print STDERR "Error - $flowName is used both for channels and processes.\n";
    }

    #}}}  
    #{{{  check loop channel square syntax

    if ($square !~ /^\[\]$/)
    {
        $square = chopchop ($square);
    }

    #
    # in a loop channel, the square expression is required and must start
    # with ? (input) or ! (output)
    #

    if ($sobjInstanceNumber == $dobjInstanceNumber)
    {
        if ($square !~ m/^[\!\?]/)
        {
            $error++;
            print STDERR "Error - channel $flowName must have [!<expression>] or [?<expression>].\n";
        }
    }

    #}}}  
    #{{{  print results

    if ($flowType eq "control")
    {
        print "CFLOW $flowName\n";
    }
    else
    {
        print "DFLOW $flowName\n";
    }
    print "  ID         $instanceNumber\n";
    if ($round  ne '') { print "  PROTOCOL   $round\n"; }
    if ($square ne '') { print "  SUBSCRIPT  $square\n"; }
    if ($curly ne '')  { print "  MODE       $curly\n"; }
    print "  SOURCE     $sobjInstanceNumber\n";
    print "  DEST       $dobjInstanceNumber\n";
    print ":\n";

    #}}}  

    $i; # returned
}

#}}}  

sub parse_the_cdif
{
    # these are updated globally by the suprograms
    $firstLine = -1;
    $lastLine  = -1;
    $error     = 0;

    &find_dfd_extent;

    $largestInum = 1;

    for ($i = $firstLine; $i < $lastLine; $i++)
    {
        #if ($cdif[$i] =~ m/ \(dfdName /)
        #{
        #    $i = &parse_cdif_dfd ($i);
        #}
        if ($cdif[$i] =~ m/ \(processName /)
        {
            $i = &parse_cdif_process ($i);
        }
        elsif ($cdif[$i] =~ m/ \(term$/)
        {
            $i = &parse_cdif_term ($i);
        }
        elsif ($cdif[$i] =~ m/ \(flow$/)
        {
            $i = &parse_cdif_flow ($i);
        }
    }
    abort_if_error ($error);
}

#}}}  ------------------------------------------------------------
#{{{  sub scan_occam
#
# sub scan_occam searches occam source code for the PROC declarations
# named in the Teamwork model. If found, these declarations are output,
# along with their formal parameter list (reformatted to make
# subsequent parsing easier).
#

sub scan_occam
{
    my ($file) = @_;
    if (! open (IN, $file))
    {
        print STDERR "Cannot read from $file.\n";
        abort_if_error (1);
    }

    #{{{  main

    while ($line = <IN>)
    {
        if ($line =~ /^ *PROC /)
        {
            chop ($line);
            $name = $line;
            $name =~ s/^ *PROC +//;
            $name =~ s/\(.*//;
            $name =~ s/ +$//;

            if ($processNameList{$name})
            #{{{  do this process
            {
                my ($i, $in, $out, $going) = (0, 0, 0, 1);
                #print STDERR "[$line]\n";
                #print STDERR "PROC $name...\n";

                $line =~ s/\t/        /g;
                $line =~ s/ *, */,/g;
                $line =~ s/ *\( */(/;
                if ($line =~ /\)/) { $going = 0; }
                $line =~ s/ *\) *//;
                $line =~ s/\(/ (/g;
                if ($line =~ /-- in$/)  { $in  = 1; }
                if ($line =~ /-- out$/) { $out = 1; }
                $line =~ s/--.*//;

                $nest = $line;
                $nest =~ s/^( *)PROC.*/$1/;

                $indent = $line;
                $indent =~ s/\(.*//;
                $indent =~ tr/[A-Za-z0-9\.]/ /;

                $line   =~ s/^$nest//;
                $indent =~ s/^$nest//;

                @args = split (/,/, $line);
                print "$args[0]";
                if ($in)     { print ".in";  $in  = 0; }
                elsif ($out) { print ".out"; $out = 0; }
                $comma = ',';
                for ($i = 1; $i <= $#args; $i++)
                {
                    print "${comma}\n${indent} $args[$i]";
                    $comma = ',';
                }
                while (($going) && ($line = <IN>))
                {
                    chop ($line);
                    #print STDERR "[$line]\n";
                    $line =~ s/\t/        /g;
                    $line =~ s/ *, */,/g;
                    $line =~ s/ *\( */(/;
                    if ($line =~ /\)/) { $going = 0; }
                    $line =~ s/ *\) *//;
                    $line =~ s/^ +//;
                    if ($line =~ /-- in$/)  { $in  = 1; }
                    if ($line =~ /-- out$/) { $out = 1; }
                    $line =~ s/--.*//;
                    @args = split (/,/, $line);
                    for ($i = 0; $i <= $#args; $i++)
                    {
                        print "${comma}\n${indent} $args[$i]";
                        if ($in)     { print ".in";  $in  = 0; }
                        elsif ($out) { print ".out"; $out = 0; }
                        $comma = ',';
                    }
                }
                print ")\n:\n";
                $processNameList{$name} = 0;
            }
            #}}}  
        }
    }

    #}}}  

    close (IN);
    $scan_occam_called = 1;
}

#}}}  ------------------------------------------------------------
#{{{  sub check_bubbles_found

sub check_bubbles_found
{
    my $first = 1;

    if (! $opt_n)
    {
        foreach $proc (sort keys %processNameList)
        {
            if ($processNameList{$proc})
            {
                if ($first)
                {
                    print STDERR "Information - generating PROC stubs for\n";
                    $first = 0;
                }
                print STDERR "  PROC $proc\n";
            }
        }
    }
}

#}}}  
#{{{  sub error

sub error
{
    local ($format) = @_;

    select (STDERR);
    $~ = $format;
    write;
    exit (-1);
}

#}}}  ------------------------------------------------------------

#{{{  MAIN: read the arguments
@LIBDIRS = ();
@LIBS = ();

for ($i = 0; $i <= $#ARGV; $i++)
{
    $arg = $ARGV[$i];
    if    ($arg =~ /^-f/) { $i++; $opt_f = $ARGV[$i]; }
    elsif ($arg =~ /^-m/) { $i++; $opt_m = $ARGV[$i]; }
    elsif ($arg =~ /^-c/) { $i++; $opt_c = $ARGV[$i]; }
    elsif ($arg =~ /^-d/) { $i++; $opt_d = $ARGV[$i]; }
    elsif ($arg =~ /^-o/) { $i++; $opt_o = $ARGV[$i]; }
    elsif ($arg =~ /^-n/) { $opt_n = 1; }
    elsif ($arg =~ /^-z/) { $opt_z = 1; }
    elsif ($arg =~ /^-i/) { }
    elsif ($arg =~ /^-L/) { $arg =~ s/^-L//; push (@LIBDIRS, $arg); }
    elsif ($arg =~ /^-l/) { $arg =~ s/^-l//; push (@LIBS, $arg); }
    elsif ($arg =~ /^-/)  { &error ("USAGE"); $arg_ok = 0; }
}

#}}}  
#{{{  MAIN: tempfile name

$user = $ENV{"USER"};

$tempfile = "/tmp/yacht.$user.$$";

#}}}  
#{{{  MAIN: OS info

open (IN, "uname -sr|") || die "Broken pipe from uname\n";
$system = <IN>;
close (IN);
chop ($system);
$system =~ tr/ //d;

if ($system =~ /^SunOS/)
{
    $system =~ s/\..*//;
    open (IN, "uname -m|") || die "Broken pipe from uname\n";
    $machine = <IN>;
    close (IN);
    chop ($machine);
    if ($machine =~ /^sun4/)
    {
        $system .= '_sparc';
    }
    else
    {
        die "Sorry - yacht is not available for $system on $machine.\n";
    }
}
elsif ($system =~ /^OSF1/)
{
    $system =~ s/V.*//;
    $system .= '_alpha';
}

if ($opt_z)
{
    print STDERR "System=$system\n";
}

#}}}  
#{{{  MAIN: twk_get from database or read cdif files

if (! $opt_e)
{ $opt_e = ''; }

if ($opt_m ne "" && $opt_c ne "" && $opt_d ne "")
{
    $modelName = $opt_m;
    $dfdID     = $opt_d;
    &twk_get_the_file;
    $arg_ok = 1;
}
elsif ($opt_f ne "")
{
    &read_the_file;
    $arg_ok = 1;
}
else
{
    &error ("USAGE");
    $arg_ok = 0;
}

#}}}  
#{{{  MAIN: search for yacht binary

if ($ENV{"YACHT"})
{
    $dir = $ENV{"YACHT"};
}
else { $dir = ''; }

if (! -x "$dir/yacht.$system")
{
    @path = split (/:/, $ENV{"PATH"});
    foreach $tdir (@path)
    {
        if (-x "$tdir/yacht.$system")
        {
            $dir = $tdir;
            last;
        }
    }
}

if (! -x "$dir/yacht.$system")
{
    die "yacht.$system: command not found\nTry altering your PATH, or set the YACHT variable.\n";
}

#}}}  
#{{{  MAIN: run library lister

if ($#LIBS >= 0)
{
    my $command = "liblist";

    if (! -x "$dir/liblist")
    {
        die "liblist: command not found\nTry altering your PATH, or set the YACHT variable.\n";
    }

    push (@LIBDIRS, '.');

    foreach $lib (@LIBS)
    {
        my $found = 0;

        foreach $dir (@LIBDIRS)
        {
            if ($found) { }
            elsif (-f "$dir/$lib.lib")
            {
                $found = 1;
                $command = "$command $dir/$lib.lib";
            }
        }
        if (! $found)
        {
            die "Error - $lib.lib not found.\n";
        }
    }

    if (system ("$dir/$command > $tempfile") != 0)
    {
        die "Error - Cannot run command\n$command\n";
    }

    if ($opt_z) { print STDERR  "$command > $tempfile\n"; }
}


#}}}  
#{{{  MAIN: generate occam source code

if ($arg_ok)
{
    #{{{  open output pipe to yacht.$system

    if ($opt_n eq "")
    {
        if ($opt_o ne "")
        {
            open (OUT, "|$dir/yacht.$system >$opt_o") ||
                die "yacht: cannot pipe yacht_new & write $opt_o\n";
            select (OUT);
            $out = OUT;
        }
        else
        {
            open (OUT, "|$dir/yacht.$system") ||
                die "yacht: cannot pipe yacht_new\n";
            select (OUT);
            $out = OUT;
        }
    }

    #}}}  
    #{{{  pass on debug flag

    if ($opt_z)
    {
        print "DEBUG\n";
    }

    #}}}  
    #{{{  pass on date

    if (open (PIPE, "date +\"DATE       %a %h %d ('%y) %T\"|"))
    {
        $date = <PIPE>;
        close (PIPE);
        print $date;
    }

    #}}}  

    &parse_the_cdif;

    #{{{  read occam source files

    if (-f $tempfile)
    {
        scan_occam ($tempfile);
        unlink ($tempfile);
    }

    for ($i = 0; $i <= $#ARGV; $i++)
    {
        $arg = $ARGV[$i];
        if ($arg =~ /^-i/)
        {
            for ($i++; $i <= $#ARGV; $i++)
            {
                $arg = $ARGV[$i];
                if ($arg =~ /^-/) { last; }
                else { scan_occam ($arg); }
            }
        }
    }

    #}}}  

    print "END\n";

    #{{{  OK?

    if (! $scan_occam_called)
    {
        print STDERR "Warning - no occam input files specified.\n";
    }
    &check_bubbles_found;

    #}}}  
    #{{{  close output pipe

    if ($opt_n eq "")
    {
        select (STDOUT);
        close (OUT);
        if ($?)
        {
            if ($opt_o ne "")
            {
                die "yacht.$system unable to write $opt_o correctly.\n";
            }
            else
            {
                die "yacht.$system failed.\n";
            }
        }
    }

    #}}}  

    if ($opt_z) { exit (1); }
}

#}}}  

#{{{  Usage text
#
# The usage text is printed on stderr when command line arguments
# are incorrectly supplied.
#

format USAGE =
Usage: yacht -m model -c config -d dfd [-o op-file] [options]

       yacht -f cdif_file [-o op-file] [options]

Yet-Another-Code-Harness-Tool : A utility to convert a specialised
Teamwork(tm) data flow diagram (DFD) into an Occam process description.

Options:
    -n             no code generation, just parse the CDIF

    -i <file> ...  read occam source for PROC declarations

    -L<dir>        include <dir> on search path for libraries

    -l<lib>        read library for PROC declarations

    -z             enable diagnostic output (very verbose!)

Richard Beton, Version 1.1.5 (96/05/21)
(C) Roke Manor Research Limited 1995
.
#}}}  ------------------------------------------------------------
