
/*{{{  includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef _ICC
#include <ctype.h>
#endif
/*}}}   */
/*{{{  defines */
/*{{{  program name */
#define TOOL "srci"
/*}}}   */
/*{{{  os dependencies */
#ifdef __MSDOS__
#define SW        "/"
#define MAXCMD    128
#define MAXLINE   512
#define MAXLEAVES 2048
#else
#define SW        "-"
#define MAXCMD    1024
#define MAXLINE   512
#define MAXLEAVES 4096
#endif
/*}}}   */
/*{{{  debugging */
#define DBP 0
#define DBT 0
#define DBL 0
#define DBS 0
/*}}}   */
/*}}}   */
/*{{{  typedefs */
typedef struct _nd
{
    char       *name;
    struct _nd *next;
    struct _nd *depend;
}
Node;
/*}}}   */
/*{{{  constants */
const char *usage[] =
{
"",
"USAGE: " TOOL " [" SW "a] [" SW "e] <makefile> [" SW "X[q][f] <command line>]",
"",
"       Source inventory utility. Finds files that form the terminal leaf nodes",
"       of the dependency tree in <makefile>, sorts them (removing duplicates)",
"       and lists them to <stdout>, ie: lists the minimum set of source files",
"       that are needed to re-make all targets. Switches (case-sensitive)",
"       affect the sort ordering as follows:",
"",
"         " SW "a   Raw alphanumeric sort, do not list path-containing files at end.",
"",
"         " SW "e   Sort by file extension first, then by name within each group.",
"",
"       If the optional switch " SW "X is supplied the list is not written. Instead",
"       the remainder of the command line after the " SW "X switch will be executed",
"       once for each file in the list. To include the name of the file, or a",
"       part of it in the command use the placeholders =F= (full name), =X=",
"       (no extension), =P= (omit path when present), =D= (just the path),",
"       =B= (base name only) or =E= (extension only).",
"",
"       If the switch is given as " SW "Xq (quiet) the commands will not be echoed",
"       to <stdout> before being executed. If " SW "Xf or " SW "Xqf (force) is given the",
"       command processing loop will not be halted in the event of an error.",
};
/*}}}   */
/*{{{  globals */
char *infile = (char *)0;
long line_no = 0L;
int dirslast = 1;
int sortext = 0;
/*}}}   */
/*{{{  functions */
/*{{{  mknode */
Node *
mknode(char *str)
{
    Node *np;
    char *cp;

    if (((np = (Node *)malloc(sizeof(Node))) == (Node *)0)    ||
        ((cp = (char *)malloc(strlen(str) + 1)) == (char *)0) )
    {
        fprintf(stderr,
                "Fatal-" TOOL "-%s(%ld)- memory allocation failure.\n",
                infile,
                line_no);
        exit (2);
    }
    else
    {
        strcpy(cp,str);
        np->name = cp;
        np->next = (Node *)0;
        np->depend = (Node *)0;
    }
    return (np);
}
/*}}}   */
/*{{{  sort_name */
int
sort_name(const void *a, const void *b)
{
    char *ap = *((char **)a);
    char *bp = *((char **)b);
    int suba = strchr(ap,'/') || strchr(ap,'\\');
    int subb = strchr(bp,'/') || strchr(bp,'\\');

    if (dirslast && suba && !subb)
        return (1);
    else if (dirslast && !suba && subb)
        return (-1);
    else
    {
       if (sortext)
       {
           char *xa = strrchr(ap,'.');
           char *xb = strrchr(bp,'.');
           if (xa && !xb)
               return (1);
           else if (!xa && xb)
               return (-1);
           else if (!xa && !xb)
               return (strcmp(ap,bp));
           else
           {
               int xcmp = strcmp(xa,xb);
               if (xcmp)
                   return (xcmp);
               else
                   return (strcmp(ap,bp));
           }
       }
       else
           return (strcmp(ap,bp));
    }
}
/*}}}   */
/*{{{  show_help */
void
show_help(void)
{
    int i;
    for (i = 0; i < (sizeof(usage)/sizeof(char*)); ++i)
    {
        fprintf(stderr,"%s\n",usage[i]);
    }
    exit (0);
}
/*}}}   */
/*}}}   */
/*{{{  main */
int
main(int argc, char *argv[])
{
    /*{{{  declare */
    char cmdline[MAXCMD + 1];
    FILE *in;
    int docmd = 0;
    int doecho = 1;
    int force = 0;
    int i;
    /*}}}   */
    /*{{{  parse command line */
    if (argc < 2)
        show_help();
    i = 1;
    while (!docmd && (i < argc))
    {
        if (!strcmp(argv[i],"-h") ||
            !strcmp(argv[i],"/h") ||
            !strcmp(argv[i],"-?") ||
            !strcmp(argv[i],"/?") )
        {
            show_help();
        }
        else if (!strcmp(argv[i],SW "X"))
        {
            docmd = 1;
        }
        else if (!strcmp(argv[i],SW "Xf"))
        {
            docmd = 1;
            force = 1;
        }
        else if (!strcmp(argv[i],SW "Xq"))
        {
            docmd = 1;
            doecho = 0;
        }
        else if (!strcmp(argv[i],SW "Xqf") ||
                 !strcmp(argv[i],SW "Xfq") )
        {
            docmd = 1;
            doecho = 0;
            force = 1;
        }
        else if (!strcmp(argv[i],SW "a"))
        {
            dirslast = 0;
        }
        else if (!strcmp(argv[i],SW "e"))
        {
            sortext = 1;
        }
        else if (!strcmp(argv[i],SW "ae") ||
                 !strcmp(argv[i],SW "ea") )
        {
            dirslast = 0;
            sortext = 1;
        }
        else if (infile)
        {
            fprintf(stderr,"Severe-" TOOL "- Illegal second input file \'%s\' on command line.\n",argv[i]);
            exit (1);
        }
        else
        {
            infile = &(argv[i][0]);
        }
        i++;
    }
    *cmdline = '\0';
    while (docmd && (i < argc))
    {
        if ((strlen(cmdline) + strlen(argv[i])) >= MAXCMD)
        {
            fprintf(stderr,"Severe-" TOOL "- Command line too long.\n");
            exit (1);
        }
        else
        {
            strcat(cmdline,argv[i]);
            strcat(cmdline," ");
        }
        i++;
    }
    if (!infile)
    {
        fprintf(stderr,"Severe-" TOOL "- No input file specified.\n");
        exit (1);
    }
    /*}}}   */
    /*{{{  read in and process makefile */
    if ((in = fopen(infile,"rt")) == (FILE *)0)
    {
        fprintf(stderr,"Severe-" TOOL "-%s- Unable to open file for input.\n",infile);
        exit (1);
    }
    else
    {
        /*{{{  declare */
        char line[MAXLINE];
        char *leaf[MAXLEAVES];
        Node *iter, *root = (Node *)0;
        Node **nxtnode = &root;
        int i, nleaves = 0;
        int nrules = 0;
        /*}}}   */
        /*{{{  scan the input file */
        while (fgets(line,MAXLINE - 1,in))
        {
            char *tgt;
        
            line_no++;
            if (!isspace(line[0])                      &&
                (line[0] != '#')                       &&
                ((tgt = strchr(line,':')) != (char *)0))
            {
                if ((tgt == (line + 1)) &&
                    isalpha(line[0])    &&
                    (line[2] == '\\')   )
                {
                    tgt = strchr(line + 2,':');
                }
                if (tgt)
                {
                    /*{{{  declare */
                    Node **dpp;
                    Node *np;
                    char *deps, *dfile;
                    /*}}}   */
                    nrules++;
                    /*{{{  strip out the target and point to 1st dependency */
                    deps = tgt;
                    while (*deps == ':')
                        deps++;
                    tgt--;
                    while (isspace(*tgt) || (*tgt == ':')) tgt--;
                    tgt++;
                    *tgt = '\0';
                    tgt = line;
                    /*{{{  debug */
                    #if DBP
                    printf("TARGET: %s\n",tgt);
                    #endif
                    /*}}}   */
                    /*}}}   */
                    if (((*tgt == '.') &&
                         !(strchr(tgt,'\\') ||
                           strchr(tgt,'/')))  ||
                             strchr(tgt,'%'))
                    {
                        /*{{{  it's an implicit rule */
                        fprintf(stderr,
                                "Warning-" TOOL "-%s(%ld)- Implicit rule \"%s:\" ignored (unsupported).\n",
                                infile,
                                line_no,
                                tgt      );
                        /*}}}   */
                    }
                    else
                    {
                        /*{{{  make a node for it and link it */
                        np = mknode(tgt);
                        *nxtnode = np;
                        nxtnode = &(np->next);
                        dpp = &(np->depend);
                        /*}}}   */
                        /*{{{  find it's dependencies */
                        dfile = strtok(deps," \t\n");
                        while (dfile)
                        {
                            if (!strcmp(dfile,"\\"))
                            {
                                /*{{{  read another line */
                                if (!fgets(line,MAXLINE - 1,in))
                                {
                                    fprintf(stderr,
                                            "Severe-" TOOL "-%s(%ld)- Unexpected End-of-file.\n",
                                            infile,
                                            line_no );
                                    exit (1);
                                }
                                line_no++;
                                dfile = strtok(line," \t\n");
                                /*}}}   */
                            }
                            else if (*dfile == '#')
                            {
                                dfile = (char *)0;
                            }
                            else
                            {
                                /*{{{  link the dependency */
                                Node *dp = mknode(dfile);
                                *dpp = dp;
                                dpp = &(dp->depend);
                                /*{{{  debug */
                                #if DBP
                                printf("DEPENDS ON: %s\n",dfile);
                                #endif
                                /*}}}   */
                                /*}}}   */
                                /*{{{  advance to next one */
                                dfile = strtok((char *)0," \t\n");
                                /*}}}   */
                            }
                        }
                        /*}}}   */
                    }
                }
            }
        }
        fclose(in);
        /*{{{  debug */
        #if DBP
        printf("PARSING COMPLETE.\n");
        #endif
        /*}}}   */
        /*}}}   */
        if (nrules)
        {
            /*{{{  assemble the leaf node list */
            /*{{{  debug */
            #if DBT
            printf("TARGET LIST...\n");
            {
                Node *np = root;
                while (np)
                {
                    printf("Node   @ %lX\n  name @ %lX = %s\n  next @ %lX\n  dep  @ %lX\n",
                           (long)np,
                           (long)np->name,
                           np->name,
                           (long)np->next,
                           (long)np->depend);
                    if (*(np->name) == 4) exit (100);
                    np = np->next;
                }
            }
            #endif
            #if DBL
            printf("BUILDING LEAF NODE LIST...\n");
            #endif
            /*}}}   */
            iter = root;
            while (iter)
            {
                Node *dep = iter->depend;
                while (dep)
                {
                    /*{{{  test if it's a leaf node */
                    Node *seek = root;
                    while (seek && (strcmp(seek->name,dep->name)))
                       seek = seek->next;
                    if (!seek)
                    {
                        int i = 0;
                        while ((i < nleaves) && strcmp(leaf[i],dep->name))
                            i++;
                        if (i == nleaves)
                        {
                            leaf[nleaves] = dep->name;
                            nleaves++;
                        }
                        /*{{{  debug */
                        #if DBL
                        printf("LEAF: %s\n",dep->name);
                        #endif
                        /*}}}   */
                        if (nleaves == MAXLEAVES)
                        {
                            fprintf(stderr,
                                    "Fatal-" TOOL "-%s- Too many leaf nodes in dependency tree.\n",
                                    infile );
                            exit (2);
                        }
                    }
                    /*}}}   */
                    dep = dep->depend;
                }
                iter = iter->next;
            }
            /*}}}   */
            /*{{{  sort it and perform the output action */
            /*{{{  debug */
            #if DBS
            printf("UNSORTED LEAF LIST...\n");
            {
                int i;
                for (i = 0; i < nleaves; ++i)
                    printf("%s\n",leaf[i]);
                printf("------------------------\n");
            }
            #endif
            /*}}}   */
            qsort(&(leaf[0]),nleaves,sizeof(char *),sort_name);
            for (i = 0; i < nleaves; ++i)
            {
                if (docmd)
                {
                    /*{{{  execute the supplied command line */
                    char exec[MAXCMD + 1];
                    char *ip = cmdline;
                    char *op = exec;
                    int len = 0;
                    while (*ip && (len < (MAXCMD - 1)))
                    {
                        /*{{{  copy line, substituing filename or parts as appropriate */
                        if (!strncmp(ip,"=F=",3))
                        {
                            /*{{{  use full name */
                            int sl = strlen(leaf[i]);
                            ip += 3;
                            len += sl;
                            if (len < (MAXCMD - 1))
                            {
                                strcpy(op,leaf[i]);
                                op += sl;
                            }
                            /*}}}   */
                        }
                        else if (!strncmp(ip,"=X=",3))
                        {
                            /*{{{  drop extension */
                            char *xp = strrchr(leaf[i],'.');
                            int sl, j = 0;
                            ip += 3;
                            if (xp)
                                sl = (xp - leaf[i]);
                            else
                                sl = strlen(leaf[i]);
                            len += sl;
                            if (len < (MAXCMD - 1))
                            {
                                char *lp = leaf[i];
                                while (j < sl)
                                {
                                    *op = *lp;
                                    op++;
                                    lp++;
                                    j++;
                                }
                            }
                            /*}}}   */
                        }
                        else if (!strncmp(ip,"=P=",3))
                        {
                            /*{{{  drop path */
                            char *pp = strrchr(leaf[i],'\\');
                            char *lp = leaf[i];
                            int sl;
                            ip += 3;
                            if (!pp)
                                pp = strrchr(leaf[i],'/');
                            if (pp)
                                lp = pp + 1;
                            sl = strlen(lp);
                            len += sl;
                            if (len < (MAXCMD - 1))
                            {
                                strcpy(op,lp);
                                op += sl;
                            }
                            /*}}}   */
                        }
                        else if (!strncmp(ip,"=D=",3))
                        {
                            /*{{{  just the path */
                            char *pp = strrchr(leaf[i],'\\');
                            ip += 3;
                            if (!pp)
                                pp = strrchr(leaf[i],'/');
                            if (pp)
                            {
                                int sl = (pp - leaf[i] + 1);
                                len += sl;
                                if (len < (MAXCMD - 1))
                                {
                                    char *lp = leaf[i];
                                    int j = 0;
                                    while (j < sl)
                                    {
                                        *op = *lp;
                                        op++;
                                        lp++;
                                        j++;
                                    }
                                }
                            }
                            /*}}}   */
                        }
                        else if (!strncmp(ip,"=B=",3))
                        {
                            /*{{{  just use base name */
                            char *pp = strrchr(leaf[i],'\\');
                            char *xp = strrchr(leaf[i],'.');
                            char *lp = leaf[i];
                            int sl, j = 0;
                            ip += 3;
                            if (!pp)
                                pp = strrchr(leaf[i],'/');
                            if (pp)
                                lp = pp + 1;
                            if (xp)
                                sl = (xp - lp);
                            else
                                sl = strlen(lp);
                            len += sl;
                            if (len < (MAXCMD - 1))
                            {
                                while (j < sl)
                                {
                                    *op = *lp;
                                    op++;
                                    lp++;
                                    j++;
                                }
                            }
                            /*}}}   */
                        }
                        else if (!strncmp(ip,"=E=",3))
                        {
                            /*{{{  just use extension */
                            char *xp = strrchr(leaf[i],'.');
                            int sl;
                            ip += 3;
                            if (xp)
                            {
                                xp += 1;
                                sl = strlen(xp);
                                if (len < (MAXCMD - 1))
                                {
                                    strcpy(op,xp);
                                    op += sl;
                                }
                            }
                            /*}}}   */
                        }
                        else
                        {
                            *op = *ip;
                            op++;
                            ip++;
                            len++;
                        }
                        /*}}}   */
                    }
                    if (len >= (MAXCMD - 1))
                    {
                        fprintf(stderr,
                                "%s-" TOOL "-%s- Command line too long with file \'%s\'.\n",
                                force ? "Error" : "Severe",
                                infile,
                                leaf[i] );
                        if (!force)
                            i = MAXLEAVES + 1;
                    }
                    else
                    {
                        *op = '\n';
                        op++;
                        *op = '\0';
                        if (doecho)
                            printf("\t%s",exec);
                        if (system(exec) && !force)
                            i = MAXLEAVES + 1;
                    }
                    /*}}}   */
                }
                else
                    printf("%s\n",leaf[i]);
            }
            /*}}}   */
        }
        else
        {
            fprintf(stderr,
                    "Severe-" TOOL "-%s- No rules found; is this really a makefile ?",
                    infile);
            exit (1);
        }
    }
    /*}}}   */
    return (0);
}
/*}}}   */

