/*#define DEBUG*/
/****************************************************************************
 *
 *  Occam two virtual tree walk and copying routines
 *
 ****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include "includes.h"
# include "vtierror.h"
# include "syndef.h"
# include "chkdef.h"
# include "lexdef.h"
# include "genhdr.h"
/*}}}*/

/*{{{  tree copying*/
/*{{{  comment*/
/*
 *  These routines are used to make new copies of trees
 *  The standard routine is copytree which returns a copy of the
 *  tree whose root is passed to it as a parameter.
 *  Any declarations within the tree will be copied and references
 *  to those declarations changed to the new copy.
 *  It is assumed the global 'lexlevel' is set to the lexlevel of
 *  the new tree and will be used when declaring new copies of variables.
 *
 *  To make a copy of the tree and at the same time change
 *  some variable references to expressions (eg replace formal with
 *  actual parameters) first call initnametrans followed by addnametrance
 *  for each name to be translated and then call transcopytree to do
 *  the copy and translation.
 */
/*}}}*/

/*{{{  PRIVATE data*/
#ifdef TDS
#define MAXNODECOPY  500
#else
#define MAXNODECOPY 1000
#endif

/*#define MAXNODESTACK 100*/
#define MAXNODESTACK 10 /* changed to 10 16/1/91 */

struct nametrans
{
  treenode *old;
  treenode *new;
};

PRIVATE struct nametrans *nametranstable;
PRIVATE int numnametrans;
PRIVATE int nametransbase;
PRIVATE int nametransstackptr;
PRIVATE int *nametransstack;

PRIVATE SOURCEPOSN copylocn;   /* Locn of current treenode being copied */
PRIVATE SOURCEPOSN parentlocn; /* Locn of 'inliner' instance */
/*}}}*/

/*{{{  PRIVATE int nodesize(t)*/
PRIVATE int nodesize (const int t)
{
  switch(nodetypeoftag(t))
    {
      default:           badtag(copylocn, t, "nodesize"); return 0;
      case ACTIONNODE:   return TREENODEBASE + sizeof(struct actionnode_s);
      case ALTNODE:      return TREENODEBASE + sizeof(struct altnode_s);
      case ARRAYNODE:    return TREENODEBASE + sizeof(struct arraynode_s);
      case ARRAYSUBNODE: return TREENODEBASE + sizeof(struct arraysubnode_s);
      case CHANNODE:     return TREENODEBASE + sizeof(struct channode_s);
      case CNODE:        return TREENODEBASE + sizeof(struct cnode_s);
      case CONDNODE:     return TREENODEBASE + sizeof(struct condnode_s);
      #ifdef CONDEXP
      case CONDEXPNODE:  return TREENODEBASE + sizeof(struct condexpnode_s);
      #endif
      case CONSTEXPNODE: return TREENODEBASE + sizeof(struct constexpnode_s);
      case CONSTTABLENODE:return TREENODEBASE+ sizeof(struct consttablenode_s);
      case DECLNODE:     return TREENODEBASE + sizeof(struct declnode_s);
      case DOPNODE:      return TREENODEBASE + sizeof(struct dopnode_s);
      case HIDDENPARAMNODE:return TREENODEBASE+sizeof(struct hiddenparamnode_s);
      case INSTANCENODE: return TREENODEBASE + sizeof(struct instancenode_s);
      case LEAFNODE:     return TREENODEBASE + 0;
      case LISTNODE:     return TREENODEBASE + sizeof(struct listnode_s);
      case LITNODE:      return TREENODEBASE + sizeof(struct litnode_s);
      case MOPNODE:      return TREENODEBASE + sizeof(struct mopnode_s);
      case NAMENODE:     return TREENODEBASE + sizeof(struct namenode_s);
      case PROCESSORNODE:return TREENODEBASE + sizeof(struct processornode_s);
      case REPLCNODE:    return TREENODEBASE + sizeof(struct replcnode_s);
      case SEGMENTNODE:  return TREENODEBASE + sizeof(struct segmentnode_s);
      case SPACENODE:    return TREENODEBASE + sizeof(struct spacenode_s);
      case VALOFNODE:    return TREENODEBASE + sizeof(struct valofnode_s);
      case VARIANTNODE:  return TREENODEBASE + sizeof(struct variantnode_s);
      #ifdef CONFIG
      case CONFIGNODE:   return TREENODEBASE + sizeof(struct confignode_s);
      #endif
    }
}
/*}}}*/
/*{{{  PRIVATE treenode *lookupnametrans(tptr)*/
PRIVATE treenode *lookupnametrans ( treenode *tptr )
{
  int i;
  assert(tptr != NULL);
  for (i = nametransbase; i < numnametrans; i++)
    if (nametranstable[i].old == tptr)
      {
        DEBUG_MSG(("lookupnametrans: (0x%x to 0x%x) ", tptr, nametranstable[i].new));
        return nametranstable[i].new;
      }
  return NULL;
}
/*}}}*/
/*{{{  PRIVATE treenode *copyitem(t)*/
PRIVATE treenode *copyitem ( treenode *t )
{
  treenode *newnode = NULL;
  if (t != NULL)
    {
      const int n = nodesize(TagOf(t));
      newnode = (treenode *)newvec(n);
      memcpy(newnode, t, n);
      if (parentlocn != NOPOSN)
        SetLocn(newnode, parentlocn);
      if (nodetypeoftag(TagOf(t)) == NAMENODE)
        addnametrans(t, newnode);
    }
  return newnode;
}
/*}}}*/
/*{{{  PRIVATE void makecopy(tptr)*/
/*****************************************************************************
 *
 *  makecopy takes a tree and produces a new copy of it
 *
 *****************************************************************************/
PRIVATE void makecopy ( treenode **tptr )
{
  while (*tptr != NULL)
    {
      treenode *t = *tptr;
      SOURCEPOSN newlocn;
      copylocn = LocnOf(t);
      newlocn  = (parentlocn == NOPOSN) ? copylocn : parentlocn;
      switch(nodetypeoftag(TagOf(t)))
        /*{{{  cases*/
        {
          /*{{{  NAME: do not copy; return*/
          case WORDNODE:
            return;
          /*}}}*/
          /*{{{  namenode: look for copy; return*/
          case NAMENODE:
            /* This added 19/6/90 by CO'N for expanding known replicators */
            if ((TagOf(t) == N_REPL) && (NReplKnownOf(t)))
              *tptr = newconstant(NReplValueOf(t));
            else
              {
                t = lookupnametrans(t);
                if (t != NULL)
                  *tptr = t;
              }
            return;
          /*}}}*/
          default :
            /*{{{  copy children*/
            {
              t = copyitem(t);
              *tptr = t;
              switch(nodetypeoftag(TagOf(t)))
                {
                  /*{{{  cases*/
                  default:
                    badtag(copylocn, (BIT32)TagOf(t), "makecopy");
                  /*{{{  action*/
                  case ACTIONNODE:
                    makecopy(LHSAddr(t));
                    makecopy(RHSAddr(t));
                    return;
                  /*}}}*/
                  /*{{{  alt*/
                  case ALTNODE:
                    makecopy(AltGuardAddr(t));
                    makecopy(AltInputAddr(t));
                    tptr = AltBodyAddr(t);
                    break;
                  /*}}}*/
                  /*{{{  array*/
                  case ARRAYNODE:
                    makecopy(ARDimLengthAddr(t));
                    tptr = ARTypeAddr(t);
                    break;
                  /*}}}*/
                  /*{{{  arraysub*/
                  case ARRAYSUBNODE:
                    makecopy(ASBaseAddr(t));
                    makecopy(ASIndexAddr(t));
                    makecopy(ASExpAddr(t));
                    tptr = ASLengthAddr(t);
                    break;
                  /*}}}*/
                  #ifdef ARRAYCONSTRUCTOR
                  /*{{{  arrayconstructor*/
                  case ARRAYCONSTRUCTORNODE :
                    {
                      treenode *name = copyitem(ACNameOf(t));
                      SetReplCName(t, name);
                      SetNDecl(name, t);
                      SetNLexLevel(name, lexlevel);
                      makecopy(ACStartExpAddr(t));
                      makecopy(ACLengthExpAddr(t));
                      tptr = ACValExpAddr(t);
                      break;
                    }
                  /*}}}*/
                  #endif
                  /*{{{  channode*/
                  case CHANNODE:
                    tptr = ProtocolAddr(t);
                    break;
                  /*}}}*/
                  /*{{{  cnode*/
                  case CNODE:
                    tptr = CBodyAddr(t);
                    break;
                  /*}}}*/
                  /*{{{  cond*/
                  case CONDNODE:
                    makecopy(CondGuardAddr(t));
                    tptr = CondBodyAddr(t);
                    break;
                  /*}}}*/
                  #ifdef CONDEXP
                  /*{{{  condexp*/
                  case CONDEXPNODE:
                    makecopy(CondExpGuardAddr(t));
                    makecopy(CondExpTrueAddr(t));
                    tptr = CondExpFalseAddr(t);
                    break;
                  /*}}}*/
                  #endif
                  /*{{{  constexp*/
                  case CONSTEXPNODE :
                    tptr = CExpAddr(t);
                    break;
                  /*}}}*/
                  /*{{{  consttable*/
                  case CONSTTABLENODE:
                    tptr = CTExpAddr(t);
                    break;
                  /*}}}*/
                  /*{{{  decl*/
                  case DECLNODE:
                  switch(TagOf(t))
                  {
                  default:
                    badtag(copylocn, (BIT32)TagOf(t), "makecopy-decl");
                  /*{{{  S_DECL*/
                  case S_DECL:
                    {
                      treenode *name = DNameOf(t);
                      if (TagOf(name) == S_LIST)
                        {
                          /* They all share the same type tree so do this once */
                          treenode *typeptr = NTypeOf(ThisItem(name)),
                                   **lastitem = DNameAddr(t);
                          makecopy(&typeptr);
                          copylocn = LocnOf(t);
                          newlocn = (parentlocn == NOPOSN) ? copylocn : parentlocn;
                          while (!EndOfList(name))
                            {
                              treenode *item = copyitem(ThisItem(name));
                              treenode *l = newlistnode (S_LIST, newlocn, item, NULL);
                              SetNDecl(item, t);
                              SetNType(item, typeptr);
                              SetNLexLevel(item, lexlevel);
                              *lastitem = l;
                              lastitem = NextItemAddr(l);
                              name = NextItem(name);
                            }
                        }
                      else
                        {
                          treenode *nptr = copyitem(name);
                          makecopy(DValAddr(t));
                          SetDName(t, nptr);
                          SetNDecl(nptr, t);
                          SetNLexLevel(nptr, lexlevel);
                          makecopy(NTypeAddr(nptr));
                        }
                      tptr = DBodyAddr(t);
                    }
                    break;
                  /*}}}*/
                  /*{{{  procedure or function*/
                  case S_PROCDEF:
                  case S_SFUNCDEF: case S_LFUNCDEF:
                  CASE_CONFIG_SPEC
                    {
                      treenode *name = DNameOf(t);
                      treenode *nptr = copyitem(name);
                      treenode *dummy = NULL;
                      treenode **fparams = &dummy;
                      SetDName(t, nptr);
                      SetNDecl(nptr, t);
                      SetNLexLevel(nptr, lexlevel);
                      lexlevel++;
                      switch(TagOf(t))
                        {
                        case S_PROCDEF:
                          /*makecopy(NTypeAddr(nptr));*/ /* Parameter list */
                          fparams = NTypeAddr(nptr);
                          break;
                        case S_SFUNCDEF: case S_LFUNCDEF:
                          {
                            treenode *fntype = copyitem(NTypeOf(name));
                            SetNType(nptr, fntype);
                            makecopy(FnTypeListAddr(fntype));
                            /*makecopy(FnParamsAddr(fntype));*/
                            fparams = FnParamsAddr(fntype);
                          }
                          break;
                        default:
                          break;
                        }
                      {  /* copy formal parameter list */
                        treenode *params = *fparams;
                        while (!EndOfList(params))
                          {
                            treenode *item = copyitem(ThisItem(params));
                            treenode *l = newlistnode(S_LIST, newlocn, item, NULL);
                            if (nodetypeoftag(TagOf(item)) == NAMENODE)
                              {
                                makecopy(NTypeAddr(item));
                                SetNLexLevel(item, lexlevel);
                              }
                            /* else it might be VSP etc */
                            *fparams = l;
                            fparams  = NextItemAddr(l);
                            params   = NextItem(params);
                          }
                      }
                      makecopy(DValAddr(t)); /* Copy body of PROC/FUNCTION */
                      lexlevel--;
                      makecopy(DBodyAddr(t));
                    }
                    return;
                  /*}}}*/
                  /*{{{  others*/
                  case S_ABBR: case S_VALABBR: case S_RETYPE: case S_VALRETYPE:
                  case S_TPROTDEF: case S_SPROTDEF:
                  /*case S_LABELDEF:*/ /* removed for bug 1112 16/1/91 */
                    {
                      treenode *nptr = copyitem(DNameOf(t));
                      makecopy(DValAddr(t));
                      SetDName(t, nptr);
                      SetNDecl(nptr, t);
                      SetNLexLevel(nptr, lexlevel);
                      makecopy(NTypeAddr(nptr));
                      tptr = DBodyAddr(t);
                    }
                    break;
                  /*}}}*/
                  /*{{{  labels*/
                  case S_LABELDEF: /* added for bug 1112 16/1/91 */
                    {
                      makecopy(DNameAddr(t)); /* pick up the new name */
                      makecopy(DValAddr(t));  /* always NULL for S_LABELDEF? */
                      SetNDecl(DNameOf(t), t);
                      SetNLexLevel(DNameOf(t), lexlevel);
                      makecopy(NTypeAddr(DNameOf(t)));
                      tptr = DBodyAddr(t); /* body will always be NULL */
                    }
                    break;
                  /*}}}*/
                  /*{{{  PLACE*/
                  case S_PLACE: case S_WSPLACE: case S_VSPLACE:
                  #ifdef CONFIG
                  case S_PLACEON:
                  #endif
                    makecopy(DNameAddr(t));
                    makecopy(DValAddr(t));
                    tptr = DBodyAddr(t);
                    break;
                  /*}}}*/
                  }
                  break;
                  /*}}}*/
                  /*{{{  dopnode*/
                  case DOPNODE:
                    makecopy(LeftOpAddr(t));
                    tptr = RightOpAddr(t);
                    break;
                  /*}}}*/
                  /*{{{  instance*/
                  case INSTANCENODE:
                    makecopy(INameAddr(t));
                    tptr = IParamListAddr(t);
                    break;
                  /*}}}*/
                  /*{{{  leaf / lit*/
                  case LEAFNODE:
                  case LITNODE:
                    return;
                  /*}}}*/
                  /*{{{  list*/
                  case LISTNODE:
                    makecopy(ThisItemAddr(t));
                    tptr = NextItemAddr(t);
                    t = *tptr;
                    while (!EndOfList(t))
                      {
                        t = newlistnode (S_LIST, newlocn, ThisItem(t), NextItem(t));
                        *tptr = t;
                        makecopy(ThisItemAddr(t));
                        tptr = NextItemAddr(t);
                        t = *tptr;
                      }
                    return;
                  /*}}}*/
                  /*{{{  mop*/
                  case MOPNODE:
                    tptr = OpAddr(t);
                    break;
                  /*}}}*/
                  /*{{{  processor*/
                  case PROCESSORNODE:
                    makecopy(ProcessorExpAddr(t));
                    lexlevel++;
                    makecopy(ProcessorBodyAddr(t));
                    lexlevel--;
                    return;
                  /*}}}*/
                  /*{{{  replc*/
                  case REPLCNODE:
                    {
                      treenode *name = copyitem(ReplCNameOf(t));
                      SetReplCName(t, name);
                      SetNDecl(name, t);
                      SetNLexLevel(name, lexlevel);
                      makecopy(ReplCStartExpAddr(t));
                      makecopy(ReplCLengthExpAddr(t));
                      if (parrepl(TagOf(t)))
                        {
                          lexlevel++;
                          makecopy(ReplCBodyAddr(t));
                          lexlevel--;
                          return;
                        }
                      tptr = ReplCBodyAddr(t);
                    }
                    break;
                  /*}}}*/
                  /*{{{  segment*/
                  case SEGMENTNODE:
                    makecopy(SNameAddr(t));
                    makecopy(SStartExpAddr(t));
                    tptr = SLengthExpAddr(t);
                    break;
                  /*}}}*/
                  /*{{{  valof*/
                  case VALOFNODE:
                    makecopy(VLBodyAddr(t));
                    tptr = VLResultListAddr(t);
                    break;
                  /*}}}*/
                  /*{{{  variant*/
                  case VARIANTNODE:
                    makecopy(VRTaggedListAddr(t));
                    tptr = VRBodyAddr(t);
                    break;
                  /*}}}*/
                  #ifdef CONFIG                  
                  /*{{{  confignode*/
                  case CONFIGNODE :
                    makecopy(STDevAddr(t));
                    makecopy(STAttrNameAddr(t));
                    tptr = STAttrExpAddr(t);
                    break;
                  /*}}}*/
                  #endif
                  /*}}}*/
                }
            }
            break;
            /*}}}*/
        }
        /*}}}*/
    }
}
/*}}}*/

/*{{{  PUBLIC void marknametrans()*/
/* save current nametransbase on stack */
PUBLIC void marknametrans ( void )
{
  if (nametransstackptr < MAXNODESTACK)
    {
      nametransstack[nametransstackptr++] = nametransbase;
      nametransbase = numnametrans;
    }
  else
    vtiabort(VTI_NAMETRANS_OVERFLOW, 0);
}
/*}}}*/
/*{{{  PUBLIC void freenametrans()*/
/* restore nametransbase from stack */
PUBLIC void freenametrans ( void )
{
  if (nametransstackptr > 0)
    {
      numnametrans  = nametransbase;
      nametransbase = nametransstack[--nametransstackptr];
    }
  else
    vtiabort(VTI_NAMETRANS_UNDERFLOW, 0);
}
/*}}}*/
/*{{{  PUBLIC void addnametrans(o, n)*/
/* Add a name translation to the table of name translations */
PUBLIC void addnametrans ( treenode *old , treenode *new )
{
  assert(old != NULL);
  assert(new != NULL);
  if (numnametrans < MAXNODECOPY)
    {
      DEBUG_MSG(("addnametrans: (0x%x to 0x%x) ", old, new));
      nametranstable[numnametrans].old = old;
      nametranstable[numnametrans].new = new;
      numnametrans++;
    }
  else
    vtiabort(VTI_TOO_MANY_NAMETRANS, 0);
}
/*}}}*/
/*{{{  PRIVATEPARAM int stacklabeldefs*/
PRIVATEPARAM int stacklabeldefs(treenode *tptr)
/* This is used to scope ASM labels correctly when copying a tree */
{
  if (TagOf(tptr) == S_LABELDEF)
    {
      DEBUG_MSG(("stacklabeldefs: found label name (0x%x)\n", DNameOf(tptr)));
      /* only create a new name if the user hasn't created a new one already */
      if (lookupnametrans(DNameOf(tptr)) == NULL)
        {
          DEBUG_MSG(("stacklabeldefs: copying label name (0x%x)\n", DNameOf(tptr)));
          /* This adds the new name to the translation stack too */
          (void) copyitem(DNameOf(tptr));
        }
      return STOP_WALK;
    }
  return CONTINUE_WALK;
}
/*}}}*/
/*{{{  PUBLIC treenode *transcopytree(tptr)*/
/* Requires call to initnametrans to be done by the caller */
PUBLIC treenode *transcopytree ( treenode *tptr, SOURCEPOSN locn )
{
  /* If locn is NOPOSN, will copy all location information from the old tree,
     otherwise it marks all nodes with that locn.
     CON 9/10/90
  */

  treenode *newtree = tptr;
  SOURCEPOSN saved_parentlocn = parentlocn;
  parentlocn = locn;
  if (guyinserts != 0) /* added for bug 1112 16/1/91 */
    {
      DEBUG_MSG(("transcopytree: looking for labeldefs\n"));
      prewalktree(newtree, stacklabeldefs); /* scope ASM labels correctly */
    }
  makecopy(&newtree);
  parentlocn = saved_parentlocn;
  return newtree;
}
/*}}}*/
/*{{{  PUBLIC treenode *copytree(tptr)*/
/* Does NOT require call to initnametrans to be done by the caller
 * we save any current translations so we may call copytree while
 * a list of translations is being built up
 */

PUBLIC treenode *copytree ( treenode *tptr )
{
  treenode *newtree;
  marknametrans();
  newtree = transcopytree(tptr, NOPOSN);
  freenametrans();
  return newtree;
}
/*}}}*/

/*{{{  PUBLIC void initcopytree()*/
PUBLIC void initcopytree ( void )
{
  nametransstack = (int *)memalloc(MAXNODESTACK * sizeof(int));
  nametransstackptr = 0;
  nametransbase = 0;
  numnametrans = 0;
  nametranstable = (struct nametrans *)memalloc(MAXNODECOPY*
                                       sizeof(struct nametrans));
}
/*}}}*/
/*}}}*/
/*{{{  tree walking*/
/* None of these routines walk into the types in namenodes and
 * the expressions of temps and folded constants
 * Or the names of a declaration.
 */

/*{{{  PUBLIC void applytoexp(tptr, f1)*/
/*****************************************************************************
 *
 *  applytoexp   applies the function f1 to names & expressions or
 *               lists of expressions in the node tptr.
 *
 *****************************************************************************/

PUBLIC void applytoexp (treenode *tptr, void (*f1)(treenode *))
{
  if (tptr != NULL)
    switch(TagOf(tptr))
      /*{{{  cases*/
      {
        /*{{{  decl*/
        case S_ABBR: case S_VALABBR: case S_RETYPE: case S_VALRETYPE:
          (*f1)(DValOf(tptr));
          break;
        /*}}}*/
        /*{{{  REPLSEQ REPLIF REPLALT REPLPAR*/
        case S_REPLPAR: case S_REPLSEQ: case S_REPLIF: case S_REPLALT:
        case S_PRIREPLPAR: case S_PRIREPLALT: case S_PLACEDREPLPAR: case S_REPLDO:
          (*f1)(ReplCStartExpOf(tptr));
          (*f1)(ReplCLengthExpOf(tptr));
          break;
        /*}}}*/
        /*{{{  WHILE CHOICE*/
        case S_WHILE: case S_CHOICE:
          (*f1)(CondGuardOf(tptr));
          break;
        /*}}}*/
        #ifdef CONDEXP
        /*{{{  CONDEXP*/
        case S_CONDEXP :
          (*f1)(CondExpGuardOf(tptr));
          (*f1)(CondExpTrueOf(tptr));
          (*f1)(CondExpFalseOf(tptr));
          break;
        /*}}}*/
        #endif
        /*{{{  ALTERNATIVE*/
        case S_ALTERNATIVE:
          if (AltGuardOf(tptr) != NULL)
            (*f1)(AltGuardOf(tptr));
          break;
        /*}}}*/
        /*{{{  configuration*/
        /*{{{  PLACE*/
        case S_PLACE:
          (*f1)(DValOf(tptr));
          break;
        /*}}}*/
        /*{{{  PROCESSOR*/
        case S_PROCESSOR:
          (*f1)(ProcessorExpOf(tptr));
          break;
        /*}}}*/
        /*}}}*/
        /*{{{  PINSTANCE FINSTANCE*/
        case S_PINSTANCE: case S_FINSTANCE:
          walklist (f1, IParamListOf(tptr));
          return;
        /*}}}*/
        /*{{{  ASS*/
        case S_ASS:
          if (TagOf(LHSOf(tptr)) == S_LIST)
            walklist(f1, LHSOf(tptr));
          else
            (*f1)(LHSOf(tptr));
          if (TagOf(RHSOf(tptr)) == S_LIST)
            walklist (f1, RHSOf(tptr));
          else
            (*f1)(RHSOf(tptr));
          break;
        /*}}}*/
        /*{{{  OUTPUT INPUT TAGGED_INPUT DELAYED_INPUT*/
        case S_OUTPUT: case S_INPUT: case S_TAGGED_INPUT: case S_DELAYED_INPUT:
          (*f1)(LHSOf(tptr));
          if (TagOf(RHSOf(tptr)) == S_LIST)
            walklist(f1, RHSOf(tptr));
          else
            (*f1)(RHSOf(tptr));
          break;
        /*}}}*/
        /*{{{  CASE CASE_INPUT*/
        case S_CASE: case S_CASE_INPUT:
          (*f1)(LHSOf(tptr));
          return;
        /*}}}*/
        /*{{{  VARIANT*/
        case S_VARIANT:
          walklist(f1, VRTaggedListOf(tptr));
          break;
        /*}}}*/
        /*{{{  monadic*/
        case S_NEG: case S_BITNOT: case S_NOT: case S_SIZE: case S_EXACT: case S_UMINUS:
        case S_ROUND: case S_TRUNC: case S_ADDRESSOF:
          (*f1)(OpOf(tptr));
          break;
        /*}}}*/
        /*{{{  dyadic*/
        case S_AND: case S_OR:
        case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
        case S_BITAND: case S_BITOR: case S_XOR: case S_LSHIFT: case S_RSHIFT:
        case S_PLUS: case S_MINUS: case S_TIMES:
        case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE:
        case S_AFTER: case S_COLON2: case S_CSUB0: case S_CCNT1:
        case S_GUYCODE: case S_GUYSTEP :
        case S_OVERLAPCHECK: case S_FOR:
        /*case S_INTERVAL:*/
          (*f1)(LeftOpOf(tptr));
          (*f1)(RightOpOf(tptr));
          break;
        /*}}}*/
        /*{{{  constructor*/
        case S_CONSTRUCTOR:
        /* case S_STRUCTCONSTRUCTOR : */
          walklist(f1, OpOf(tptr));
          break;
        /*}}}*/
        #ifdef ARRAYCONSTRUCTOR
        /*{{{  arrayconstructor*/
        case S_ARRAYCONSTRUCTOR :
           (*f1)(ACStartExpOf(tptr));
           (*f1)(ACLengthExpOf(tptr));
           (*f1)(ACValExpOf(tptr));
           break;
        /*}}}*/
        #endif
        /*{{{  valof*/
        case S_VALOF:
          walklist(f1, VLResultListOf(tptr));
          break;
        /*}}}*/
        /*{{{  subscript*/
        case S_ARRAY:
          (*f1)(ARDimLengthOf(tptr));
          break;
        
        case S_ARRAYSUB:
          (*f1)(ASBaseOf(tptr));
          (*f1)(ASIndexOf(tptr));
          break;
        
        case S_ARRAYITEM:
          (*f1)(ASExpOf(tptr));
          break;
        /*}}}*/
        /*{{{  segment*/
        case S_SEGMENT: case S_SEGMENTITEM:
          (*f1)(SStartExpOf(tptr));
          (*f1)(SLengthExpOf(tptr));
          if (SSubscriptExpOf(tptr) != NULL)
            (*f1)(SSubscriptExpOf(tptr));
          (*f1)(SCheckExpOf(tptr));
          break;
        /*}}}*/
        #ifdef CONFIG
        /*{{{  confignode*/
        case S_SET: case S_CONNECT: case S_MAP:
          (*f1)(STDevOf(tptr));
          if (STAttrNameOf(tptr) != NULL)
            (*f1)(STAttrNameOf(tptr));
          if (STAttrExpOf(tptr) != NULL)
            (*f1)(STAttrExpOf(tptr));
          break;
        /*}}}*/
        #endif
      }
      /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC void prewalktree(tptr, f1)*/
/*****************************************************************************
 *
 *  prewalktree does a walk of the tree 'tptr' applying f1() to each node of
 *  the tree before traversing its children.
 *           if f1 returns FALSE
 *             prewalk_tree will walk down the children this node
 *           else
 *             prewalk_tree simply returns
 *
 *
 *****************************************************************************/

PUBLIC void prewalktree (treenode *tptr, int (*f1)(treenode *))
{
  while (tptr != NULL)
    {
      #if 0 /*def DEBUG*/
      if (assembly_output || disassemble)
        DEBUG_MSG(("prewalktree: %s\n", itagstring(TagOf(tptr))));
      #endif
      if ((*f1)(tptr) != CONTINUE_WALK) return;
      switch(TagOf(tptr))
        /*{{{  cases*/
        {
          default:
            return;
          /*{{{  decl*/
          case S_DECL: case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
          case S_ABBR: case S_VALABBR: case S_RETYPE: case S_VALRETYPE:
          case S_TPROTDEF: case S_SPROTDEF:
          case S_LABELDEF:
          case S_PLACE: case S_WSPLACE:case S_VSPLACE:
          #ifdef CONFIG
          case S_PLACEON: CASE_CONFIG_SPEC
          #endif
            prewalktree(DValOf(tptr), f1);
            tptr = DBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  SEQ IF ALT PAR GUY*/
          case S_PAR: case S_SEQ: case S_IF: case S_ALT:
          case S_PRIPAR: case S_PRIALT: case S_PLACEDPAR:
          case S_GUY: case S_ASM: case S_DO:
          /*case S_BASICBLOCK:*/
            tptr = CBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  REPLSEQ REPLIF REPLALT REPLPAR*/
          case S_REPLPAR: case S_REPLSEQ: case S_REPLIF: case S_REPLALT:
          case S_PRIREPLPAR: case S_PRIREPLALT: case S_PLACEDREPLPAR: case S_REPLDO:
            prewalktree (ReplCStartExpOf(tptr),f1);
            prewalktree (ReplCLengthExpOf(tptr),f1);
            tptr = ReplCBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  WHILE CHOICE*/
          case S_WHILE: case S_CHOICE: case S_SELECTION :
            prewalktree (CondGuardOf(tptr),f1);
            tptr = CondBodyOf(tptr);
            break;
          /*}}}*/
          #ifdef CONDEXP
          /*{{{  CONDEXP*/
          case S_CONDEXP:
            prewalktree (CondExpGuardOf(tptr),f1);
            prewalktree (CondExpTrueOf(tptr),f1);
            tptr = CondExpFalseOf(tptr);
            break;
          /*}}}*/
          #endif
          /*{{{  VARIANT*/
          case S_VARIANT:
            prewalktree (VRTaggedListOf(tptr),f1);
            tptr = VRBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  ALTERNATIVE*/
          case S_ALTERNATIVE:
            prewalktree (AltGuardOf(tptr),f1);
            prewalktree (AltInputOf(tptr),f1);
            tptr = AltBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  configuration*/
          #if 0 /* never used */
          /*{{{  PRI PLACED*/
          case S_PRI: case S_PLACED:
            tptr = ConfigBodyOf(tptr);
            break;
          /*}}}*/
          #endif
          /*{{{  PROCESSOR*/
          case S_PROCESSOR:
            prewalktree (ProcessorExpOf(tptr),f1);
            tptr = ProcessorBodyOf(tptr);
            break;
          /*}}}*/
          /*}}}*/
          /*{{{  instance*/
          /*{{{  PINSTANCE FINSTANCE*/
          case S_PINSTANCE: case S_FINSTANCE:
            prewalktree (INameOf(tptr),f1);
            prewalktree (IParamListOf(tptr),f1);
            return;
          /*}}}*/
          /*}}}*/
          /*{{{  action*/
          /*{{{  ASS OUTPUT INPUT TAGGED_INPUT DELAYED_INPUT CASE CASEINPUT*/
          case S_ASS: case S_OUTPUT: case S_INPUT:
          case S_TAGGED_INPUT: case S_DELAYED_INPUT: case S_CASE: case S_CASE_INPUT:
            prewalktree (LHSOf(tptr),f1);
            prewalktree (RHSOf(tptr),f1);
            return;
          /*}}}*/
          /*}}}*/
          /*{{{  monadic*/
          case S_NEG: case S_BITNOT: case S_NOT: case S_SIZE:
          case S_EXACT: case S_ROUND: case S_TRUNC:
          case S_ADDRESSOF: case S_MOSTPOS: case S_MOSTNEG:  case S_UMINUS:
            tptr = OpOf(tptr);
            break;
          /*}}}*/
          /*{{{  dyadic*/
          case S_AND: case S_OR:
          case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
          case S_BITAND: case S_BITOR: case S_XOR: case S_LSHIFT: case S_RSHIFT:
          case S_PLUS: case S_MINUS: case S_TIMES:
          case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE: case S_AFTER:
          case S_COLON2: case S_CSUB0: case S_CCNT1:
          case S_GUYCODE: case S_GUYSTEP :
          /*case S_INTERVAL:*/
          case S_OVERLAPCHECK: case S_FOR:
          case S_EVAL:
            prewalktree (LeftOpOf(tptr),f1);
            tptr = RightOpOf(tptr);
            break;
          /*}}}*/
          /*{{{  constructor*/
          case S_CONSTRUCTOR:
          /* case S_STRUCTCONSTRUCTOR: */
            prewalktree (OpOf(tptr),f1);
            return;
          /*}}}*/
          #ifdef ARRAYCONSTRUCTOR
          /*{{{  arrayconstructor*/
          case S_ARRAYCONSTRUCTOR :
              prewalktree(ACStartExpOf(tptr), f1);
              prewalktree(ACLengthExpOf(tptr), f1);
              tptr = ACValExpOf(tptr);
              break;
          /*}}}*/
          #endif
          /*{{{  valof*/
          case S_VALOF:
            prewalktree (VLBodyOf(tptr),f1);
            tptr = VLResultListOf(tptr);
            break;
          /*}}}*/
          /*{{{  subscript*/
          case S_ARRAY:
            prewalktree (ARDimLengthOf(tptr),f1);
            tptr = ARTypeOf(tptr);
            break;
          
          case S_ARRAYSUB:
            prewalktree (ASBaseOf(tptr),f1);
            tptr = ASIndexOf(tptr);
            break;
          
          case S_ARRAYITEM:
            prewalktree (ASBaseOf(tptr),f1);
            tptr = ASExpOf(tptr);
            break;
          /*}}}*/
          /*{{{  segment*/
          case S_SEGMENT: case S_SEGMENTITEM:
            prewalktree (SNameOf(tptr),f1);
            prewalktree (SStartExpOf(tptr),f1);
            prewalktree (SLengthExpOf(tptr),f1);
            prewalktree (SSubscriptExpOf(tptr), f1);
            tptr = SCheckExpOf(tptr);
            break;
          /*}}}*/
          /*{{{  space usage node*/
          case S_SPACEUSAGE:
            tptr = SpBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  hidden parameter node*/
          case S_HIDDEN_PARAM:
          case S_FNFORMALRESULT:
          case S_FNACTUALRESULT:
            tptr = HExpOf(tptr);
            break;
          /*}}}*/
          /*{{{  S_LIST*/
          case S_LIST:
            while (!EndOfList(tptr))
              {
                prewalktree (ThisItem(tptr),f1);
                tptr = NextItem(tptr);
              }
            return;
          /*}}}*/
          /*{{{  S_FNTYPE*/
          case S_FNTYPE:
            prewalktree(LeftOf(tptr), f1);
            tptr = RightOf(tptr);
            break;
          /*}}}*/
          #ifdef CONFIG
          /*{{{  connect*/
          case S_CONNECT:
            prewalktree(ConnectFromEdgeOf(tptr), f1);
            prewalktree(ConnectToEdgeOf(tptr), f1);
            tptr = ConnectArcOf(tptr);
            break;
          /*}}}*/
          /*{{{  set*/
          case S_SET :
            prewalktree(STDevOf(tptr), f1);
            prewalktree(STAttrNameOf(tptr), f1);
            tptr = STAttrExpOf(tptr);
            break;
          /*}}}*/
          /*{{{  map*/
          case S_MAP:
            prewalktree(MapSourceOf(tptr), f1);
            prewalktree(MapDestOf(tptr), f1);
            tptr = MapPriOf(tptr);
            break;
          /*}}}*/
          #endif
        }
        /*}}}*/
    }
}
/*}}}*/
/*{{{  PUBLIC void prewalkproctree(tptr, f1)*/
/*****************************************************************************
 *
 *  prewalkproctree does a walk of the tree 'tptr' applying f1() to each node
 *  of the tree before traversing its children.
 *  It does not traverse expressions !.
 *           if f1 returns FALSE
 *             prewalk_tree will walk down the children this node
 *           else
 *             prewalk_tree simply returns
 *
 *
 *****************************************************************************/

PUBLIC void prewalkproctree (treenode *tptr, int (*f1)(treenode *))
{
  while (tptr != NULL)
    {
      #if 0 /*def DEBUG*/
      if (assembly_output || disassemble)
        DEBUG_MSG(("prewalkproctree: %s\n", itagstring(TagOf(tptr))));
      #endif
      if ((*f1)(tptr) != CONTINUE_WALK) return;
      switch(TagOf(tptr))
        /*{{{  cases*/
        {
          default:
            return;
          /*{{{  decl*/
          case S_DECL:
            prewalkproctree(DValOf(tptr), f1);
            tptr = DBodyOf(tptr);
            break;
          case S_ABBR: case S_VALABBR: case S_RETYPE: case S_VALRETYPE:
          case S_TPROTDEF: case S_SPROTDEF:
          case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
          case S_LABELDEF:
          case S_PLACE: case S_WSPLACE: case S_VSPLACE:
          #ifdef CONFIG
          case S_PLACEON: CASE_CONFIG_SPEC
          #endif
            tptr = DBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  SEQ IF ALT PAR*/
          case S_PAR: case S_SEQ: case S_IF: case S_ALT:
          case S_PRIPAR : case S_PRIALT: case S_PLACEDPAR:
          case S_GUY: case S_ASM: /*case S_BASICBLOCK:*/ case S_DO:
            tptr = CBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  REPLSEQ REPLIF REPLALT REPLPAR*/
          case S_REPLPAR: case S_REPLSEQ: case S_REPLIF: case S_REPLALT:
          case S_PRIREPLPAR: case S_PRIREPLALT: case S_PLACEDREPLPAR: case S_REPLDO:
            tptr = ReplCBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  WHILE CHOICE*/
          case S_WHILE: case S_CHOICE: case S_SELECTION:
            tptr = CondBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  VARIANT*/
          case S_VARIANT:
            tptr = VRBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  ALTERNATIVE*/
          case S_ALTERNATIVE:
            prewalkproctree (AltInputOf(tptr),f1);
            tptr = AltBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  configuration*/
          #if 0 /* never used */
          /*{{{  PRI PLACED*/
          case S_PRI: case S_PLACED:
            tptr = ConfigBodyOf(tptr);
            break;
          /*}}}*/
          #endif
          /*{{{  PROCESSOR*/
          case S_PROCESSOR:
            tptr = ProcessorBodyOf(tptr);
            break;
          /*}}}*/
          /*}}}*/
          /*{{{  PINSTANCE FINSTANCE*/
          case S_PINSTANCE: case S_FINSTANCE:
            return;
          /*}}}*/
          /*{{{  ASS OUTPUT INPUT TAGGED_INPUT DELAYED_INPUT*/
          case S_ASS: case S_OUTPUT: case S_INPUT:
          case S_TAGGED_INPUT: case S_DELAYED_INPUT:
            return;
          /*}}}*/
          /*{{{  CASE CASE_INPUT*/
          case S_CASE: case S_CASE_INPUT:
            tptr = RHSOf(tptr);
            break;
          /*}}}*/
          /*{{{  valof*/
          case S_VALOF:
            tptr = VLBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  space usage node*/
          case S_SPACEUSAGE:
            tptr = SpBodyOf(tptr);
            break;
          /*}}}*/
          /*{{{  S_LIST*/
          case S_LIST:
            while (!EndOfList(tptr))
              {
                prewalkproctree (ThisItem(tptr),f1);
                tptr = NextItem(tptr);
              }
            return;
          /*}}}*/
        }
        /*}}}*/
    }
}
/*}}}*/
/*{{{  PUBLIC void modprewalktree(tptr, f1)*/
/*****************************************************************************
 *
 *  modprewalktree does a walk of the tree 'tptr' applying f1() to each node of
 *  the tree before traversing its children.
 *
 *
 *****************************************************************************/

PUBLIC void modprewalktree (treenode **tptr, int (*f1)(treenode **))
{
  while (*tptr != NULL)
    {
      treenode *t;
      #if 0 /*def DEBUG*/
      if (assembly_output || disassemble)
        DEBUG_MSG(("modprewalktree: %s\n", itagstring(TagOf(tptr))));
      #endif
      if ((*f1)(tptr) != CONTINUE_WALK) return;
      t = *tptr;
      if (t != NULL)
        switch(TagOf(t))
          /*{{{  cases*/
          {
            default:
              return;
            /*{{{  decl*/
            case S_DECL: case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
            case S_ABBR: case S_VALABBR: case S_RETYPE: case S_VALRETYPE:
            case S_TPROTDEF: case S_SPROTDEF:
            case S_LABELDEF:
            case S_PLACE: case S_WSPLACE: case S_VSPLACE:
            #ifdef CONFIG
            case S_PLACEON: CASE_CONFIG_SPEC
            #endif
              modprewalktree(DValAddr(t), f1);
              tptr = DBodyAddr(t);
              break;
            /*}}}*/
            /*{{{  SEQ IF ALT PAR*/
            case S_SEQ: case S_PAR: case S_IF: case S_ALT:
            case S_PRIPAR : case S_PRIALT: case S_PLACEDPAR:
            case S_GUY: case S_ASM: /*case S_BASICBLOCK:*/ case S_DO:
              tptr = CBodyAddr(t);
              break;
            /*}}}*/
            /*{{{  REPLSEQ REPLIF REPLALT REPLPAR*/
            case S_REPLPAR: case S_REPLSEQ: case S_REPLIF: case S_REPLALT:
            case S_PRIREPLPAR: case S_PRIREPLALT: case S_PLACEDREPLPAR: case S_REPLDO:
              modprewalktree (ReplCStartExpAddr(t),f1);
              modprewalktree (ReplCLengthExpAddr(t),f1);
              tptr = ReplCBodyAddr(t);
              break;
            /*}}}*/
            /*{{{  WHILE CHOICE*/
            case S_WHILE: case S_CHOICE: case S_SELECTION:
              modprewalktree (CondGuardAddr(t),f1);
              tptr = CondBodyAddr(t);
              break;
            /*}}}*/
            #ifdef CONDEXP
            /*{{{  CONDEXP*/
            case S_CONDEXP:
              modprewalktree (CondExpGuardAddr(t),f1);
              modprewalktree (CondExpTrueAddr(t),f1);
              tptr = CondExpFalseAddr(t);
              break;
            /*}}}*/
            #endif
            /*{{{  VARIANT*/
            case S_VARIANT:
              modprewalktree (VRTaggedListAddr(t),f1);
              tptr = VRBodyAddr(t);
              break;
            /*}}}*/
            /*{{{  ALTERNATIVE*/
            case S_ALTERNATIVE:
              modprewalktree (AltGuardAddr(t),f1);
              modprewalktree (AltInputAddr(t),f1);
              tptr = AltBodyAddr(t);
              break;
            /*}}}*/
            /*{{{  configuration*/
            #if 0 /* never used */
            /*{{{  PRI PLACED*/
            case S_PRI: case S_PLACED:
              tptr = ConfigBodyAddr(t);
              break;
            /*}}}*/
            #endif
            /*{{{  PROCESSOR*/
            case S_PROCESSOR:
              modprewalktree (ProcessorExpAddr(t),f1);
              tptr = ProcessorBodyAddr(t);
              break;
            /*}}}*/
            /*}}}*/
            /*{{{  instance*/
            /*{{{  PINSTANCE FINSTANCE*/
            case S_PINSTANCE: case S_FINSTANCE:
              modprewalktree (INameAddr(t),f1);
              modprewalktree (IParamListAddr(t),f1);
              return;
            /*}}}*/
            /*}}}*/
            /*{{{  action*/
            /*{{{  ASS OUTPUT INPUT TAGGED_INPUT DELAYED_INPUT CASE CASEINPUT*/
            case S_ASS: case S_OUTPUT: case S_INPUT:
            case S_TAGGED_INPUT: case S_DELAYED_INPUT:
            case S_CASE: case S_CASE_INPUT:
              modprewalktree (LHSAddr(t),f1);
              modprewalktree (RHSAddr(t),f1);
              return;
            /*}}}*/
            /*}}}*/
            /*{{{  monadic*/
            case S_NEG: case S_BITNOT: case S_NOT: case S_SIZE: case S_EXACT:
            case S_ROUND: case S_TRUNC: case S_ADDRESSOF:case S_MOSTPOS: case S_MOSTNEG: 
            case S_UMINUS:
              tptr = OpAddr(t);
              break;
            /*}}}*/
            /*{{{  dyadic*/
            case S_AND: case S_OR:
            case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
            case S_BITAND: case S_BITOR: case S_XOR: case S_LSHIFT: case S_RSHIFT:
            case S_PLUS: case S_MINUS: case S_TIMES:
            case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE: case S_AFTER:
            case S_COLON2: case S_CSUB0: case S_CCNT1:
            case S_GUYCODE: case S_GUYSTEP :
            case S_OVERLAPCHECK: case S_FOR:
            case S_EVAL:
            /*case S_INTERVAL :*/
              modprewalktree (LeftOpAddr(t),f1);
              tptr = RightOpAddr(t);
              break;
            /*}}}*/
            /*{{{  constructor*/
            case S_CONSTRUCTOR:
            /* case S_STRUCTCONSTRUCTOR: */
              modprewalktree (OpAddr(t),f1);
              return;
            /*}}}*/
            #ifdef ARRAYCONSTRUCTOR
            /*{{{  arrayconstructor*/
            case S_ARRAYCONSTRUCTOR :
                modprewalktree(ACStartExpAddr(t), f1);
                modprewalktree(ACLengthExpAddr(t), f1);
                tptr = ACValExpAddr(t);
                break;
            /*}}}*/
            #endif
            /*{{{  valof*/
            case S_VALOF:
              modprewalktree (VLBodyAddr(t),f1);
              tptr = VLResultListAddr(t);
              break;
            /*}}}*/
            /*{{{  subscript*/
            case S_ARRAY:
              modprewalktree (ARDimLengthAddr(t),f1);
              tptr = ARTypeAddr(t);
              break;
            
            case S_ARRAYSUB:
              modprewalktree (ASBaseAddr(t),f1);
              tptr = ASIndexAddr(t);
              break;
            
            case S_ARRAYITEM:
              modprewalktree (ASBaseAddr(t),f1);
              tptr = ASExpAddr(t);
              break;
            /*}}}*/
            /*{{{  segment*/
            case S_SEGMENT: case S_SEGMENTITEM:
              modprewalktree (SNameAddr(t),f1);
              modprewalktree (SStartExpAddr(t),f1);
              modprewalktree (SLengthExpAddr(t),f1);
              modprewalktree (SSubscriptExpAddr(t), f1);
              tptr = SCheckExpAddr(t);
              break;
            /*}}}*/
            /*{{{  space usage node*/
            case S_SPACEUSAGE:
              tptr = SpBodyAddr(t);
              break;
            /*}}}*/
            /*{{{  hidden parameter node*/
            case S_HIDDEN_PARAM:
              tptr = HExpAddr(t);
              break;
            case S_FNFORMALRESULT: case S_FNACTUALRESULT:
              tptr = HExpAddr(t);
              break;
            /*}}}*/
            /*{{{  S_LIST*/
            case S_LIST:
              while (!EndOfList(t))
                {
                  modprewalktree (LeftAddr(t),f1);
                  t = NextItem(t);
                }
              return;
            /*}}}*/
            /*{{{  S_FNTYPE*/
            case S_FNTYPE:
              modprewalktree(LeftAddr(t), f1);
              tptr = RightAddr(t);
              break;
            /*}}}*/
            #ifdef CONFIG
            /*{{{  CONNECT*/
            case S_CONNECT :
              modprewalktree(ConnectFromEdgeAddr(t), f1);
              modprewalktree(ConnectToEdgeAddr(t), f1);
              tptr = ConnectArcAddr(t);
              break;
            /*}}}*/
            /*{{{  SET*/
            case S_SET :
              modprewalktree(STDevAddr(t), f1);
              modprewalktree(STAttrNameAddr(t), f1);
              tptr = STAttrExpAddr(t);
              break;
            /*}}}*/
            /*{{{  map*/
            case S_MAP:
              modprewalktree(MapSourceAddr(t), f1);
              modprewalktree(MapDestAddr(t), f1);
              tptr = MapPriAddr(t);
              break;
            /*}}}*/
            #endif
          }
          /*}}}*/
    }
}
/*}}}*/
/*{{{  PUBLIC void modprewalkproctree(tptr, f1)*/
/*****************************************************************************
 *
 *  modprewalkproctree does a walk of the tree 'tptr' applying f1() to each node
 *  of the tree before traversing its children.
 *  It does not traverse expressions !.
 *           if f1 returns FALSE
 *             prewalk_tree will walk down the children this node
 *           else
 *             prewalk_tree simply returns
 *
 *
 *****************************************************************************/

PUBLIC void modprewalkproctree (treenode **tptr, int (*f1)(treenode **))
{
  while (*tptr != NULL)
    {
      treenode *t;
      #if 0 /*def DEBUG*/
      if (assembly_output || disassemble)
        DEBUG_MSG(("modprewalkproctree: %s\n", itagstring(TagOf(tptr))));
      #endif
      if ((*f1)(tptr) != CONTINUE_WALK) return;
      t = *tptr;
      switch(TagOf(t))
        /*{{{  cases*/
        {
          default:
            return;
          /*{{{  decl*/
          case S_DECL:
            modprewalkproctree(DValAddr(t), f1);
            tptr = DBodyAddr(t);
            break;
          case S_ABBR: case S_VALABBR: case S_RETYPE: case S_VALRETYPE:
          case S_TPROTDEF: case S_SPROTDEF:
          case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
          case S_LABELDEF:
          case S_PLACE: case S_WSPLACE: case S_VSPLACE:
          #ifdef CONFIG
          case S_PLACEON: CASE_CONFIG_SPEC
          #endif
            tptr = DBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  SEQ IF ALT PAR*/
          case S_PAR: case S_SEQ: case S_IF: case S_ALT:
          case S_PRIPAR : case S_PRIALT: case S_PLACEDPAR:
          case S_GUY: case S_ASM: /*case S_BASICBLOCK:*/ case S_DO:
            tptr = CBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  REPLSEQ REPLIF REPLALT REPLPAR*/
          case S_REPLPAR: case S_REPLSEQ: case S_REPLIF: case S_REPLALT:
          case S_PRIREPLPAR: case S_PRIREPLALT: case S_PLACEDREPLPAR: case S_REPLDO:
            tptr = ReplCBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  WHILE CHOICE*/
          case S_WHILE: case S_CHOICE: case S_SELECTION:
            tptr = CondBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  VARIANT*/
          case S_VARIANT:
            tptr = VRBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  ALTERNATIVE*/
          case S_ALTERNATIVE:
            modprewalkproctree (AltInputAddr(t),f1);
            tptr = AltBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  configuration*/
          #if 0 /* never used */
          /*{{{  PRI PLACED*/
          case S_PRI: case S_PLACED:
            tptr = ConfigBodyAddr(t);
            break;
          /*}}}*/
          #endif
          /*{{{  PROCESSOR*/
          case S_PROCESSOR:
            tptr = ProcessorBodyAddr(t);
            break;
          /*}}}*/
          /*}}}*/
          /*{{{  PINSTANCE FINSTANCE*/
          case S_PINSTANCE: case S_FINSTANCE:
            return;
          /*}}}*/
          /*{{{  ASS OUTPUT INPUT TAGGED_INPUT DELAYED_INPUT*/
          case S_ASS: case S_OUTPUT: case S_INPUT:
          case S_TAGGED_INPUT: case S_DELAYED_INPUT:
            return;
          /*}}}*/
          /*{{{  CASE CASE_INPUT*/
          case S_CASE: case S_CASE_INPUT:
            tptr = RHSAddr(t);
            break;
          /*}}}*/
          /*{{{  valof*/
          case S_VALOF:
            tptr = VLBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  space usage node*/
          case S_SPACEUSAGE:
            tptr = SpBodyAddr(t);
            break;
          /*}}}*/
          /*{{{  S_LIST*/
          case S_LIST:
            while (!EndOfList(t))
              {
                modprewalkproctree (ThisItemAddr(t),f1);
                t = NextItem(t);
              }
            return;
          /*}}}*/
        }
        /*}}}*/
    }
}
/*}}}*/

#ifdef CONFIG
/*{{{  PUBLIC void freenode*/
PUBLIC void freenode(treenode **tptr)
{
  DEBUG_MSG(("freenode:%s ", itagstring(TagOf(*tptr))));
  freevec(*tptr, nodesize(TagOf(*tptr)));
  *tptr = NULL;
}
/*}}}*/
/*{{{  PUBLIC void freetree*/
PUBLIC void freetree(treenode **tptr)
{
  treenode *const t = *tptr;
  if (t == NULL) return;
  DEBUG_MSG(("freetree:%s ", itagstring(TagOf(t))));
  switch (nodetypeoftag(TagOf(t)))
    {
      default:
        badtag(LocnOf(t), TagOf(t), "freetree");
        break;
      case ACTIONNODE:
        freetree(LHSAddr(t));
        freetree(RHSAddr(t));
        break;
      case ALTNODE:
        freetree(AltGuardAddr(t));
        freetree(AltInputAddr(t));
        freetree(AltBodyAddr(t));
        freetree(AltChanExpAddr(t));
        break;
      case ARRAYNODE:
        freetree(ARDimLengthAddr(t));
        freetree(ARTypeAddr(t));
        break;
      case ARRAYSUBNODE:
        freetree(ASBaseAddr(t));
        freetree(ASIndexAddr(t));
        freetree(ASExpAddr(t));
        freetree(ASLengthAddr(t));
        break;
      case CHANNODE:
        freetree(ProtocolAddr(t));
        break;
      case CNODE:
        freetree(CBodyAddr(t));
        freetree(CTempAddr(t));
        break;
      case CONDNODE:
        freetree(CondGuardAddr(t));
        freetree(CondBodyAddr(t));
        break;
      case CONSTEXPNODE:
        freetree(CExpAddr(t));
        break;
      case CONSTTABLENODE:
        freetree(CTExpAddr(t));
        break;
      case DECLNODE:
        /* The obvious thing to do is a simple tree walk.
           But to avoid excessive recursion on lists of specifications,
           we should process these specially.
           But I'm too lazy to finish writing this, cos it will involve
           pointer reversal and all that muck, to ensure that we never
           access a treenode which has already been freed.
           CON - 4/1/91.
        */
      #if 1 /* simple version */ /* but it works */
        freetree(DValAddr(t));  /* freeup rhs of abbreviation, or PROC body */
        freetree(DBodyAddr(t)); /* this causes deep recursion */
        /* now freeup this namenode and its type tree */
        if (TagOf(DNameOf(t)) == S_LIST) /* multiple variable declaration */
          {
            treenode *namelist = DNameOf(t);
            freetree(NTypeAddr(ThisItem(namelist))); /* They all share a type tree */
            while (!EndOfList(namelist))
              {
                treenode *this = namelist;
                freenode(ThisItemAddr(namelist)); /* freeup name node */
                namelist = NextItem(namelist);
                freenode(&this);                  /* freeup list node */
              }
          }
        else
          {
            /* we first want to free up formal parameter declarations */
            switch(TagOf(t))
              {
                default:
                  break;
                case S_PROCDEF:
                case S_LFUNCDEF: case S_SFUNCDEF:
                  {
                    treenode *param = NTypeOf(DNameOf(t));
                    if (TagOf(t) != S_PROCDEF)
                      {
                        /* We rely here on the fact that freetree and freenode
                           clear the pointer which is passed into them
                           after freeing up the space. This means that
                           the code later on which again clobbers the type
                           portion of this namenode doesn't clear this up twice.
                           CON - 14/2/91.
                           Hence the FnTypeList portion of the type is left NULL
                        */
                        freetree(FnTypeListAddr(param)); /* return type list */
                        param = FnParamsOf(param);       /* formal param list */
                      }
                    for (; !EndOfList(param); param = NextItem(param))
                      {
                        const int tag = TagOf(ThisItem(param));
                        if (tag == N_PARAM || tag == N_VALPARAM)
                          freetree(NTypeAddr(ThisItem(param))); /* formal var's type */
                        freenode(ThisItemAddr(param));          /* formal var */
                        /* see comment above about resetting all these pointers
                           to NULL; hence we simply get a list, where
                           each ThisItem is NULL */
                      }
                  }
                  break;
              }
            switch(TagOf(t))
              {
                default:
                  freetree(NTypeAddr(DNameOf(t))); /* Free up the type tree */
                  freenode(DNameAddr(t));          /* freeup name node */
                  break;
                case S_PLACE: case S_WSPLACE: case S_VSPLACE:
              #ifdef CONFIG
                case S_PLACEON:
              #endif
                  break;
              }
          }
        break;
      #else /* efficient version */ /* not finished */
        {
          /* We must ensure that we don't free up a namenode before any
             process which may refer to it.
          */
          treenode *temp = t;
          while (isspecification(temp))
            {
              freetree(DValAddr(temp));
              temp = DBodyOf(temp);
            }
          freetree(&temp); /* free up the process after the list of decls */
          we must now free up the names, and the declnodes,
          in the correct order.
          return;
        }
      #endif
      case DOPNODE:
        freetree(LeftOpAddr(t));
        freetree(RightOpAddr(t));
        break;
      case HIDDENPARAMNODE:
        /* actual results have a 'cross-pointer' to the result expression */
        /* formal results have a 'cross-pointer' to the result type */
        if (TagOf(t) != S_FNACTUALRESULT &&
            TagOf(t) != S_FNFORMALRESULT)
          freetree(HExpAddr(t));
        break;
      case INSTANCENODE:
        freetree(INameAddr(t));
        freetree(IParamListAddr(t));
        break;
      case LEAFNODE:
        /* don't freeup dummy expressions because they are shared */
        if (TagOf(t) == S_DUMMYEXP) return;
        break;
      case LISTNODE:
        {
          treenode *list = t;
          while (list != NULL)
            {
              treenode *this = list;
              freetree(LeftAddr(list));
              list = RightOf(list);
              freenode(&this);
            }
          *tptr = NULL; /* destroy the reference to the list */
        }
        return;
      case LITNODE:
        break;
      case MOPNODE:
        /* bug 1142 - treat S_ELSIZE and S_SEGSTART specially 7/2/91 */
        if ((TagOf(t) != S_ELSIZE) && (TagOf(t) != S_SEGSTART))
          freetree(OpAddr(t));
        break;
      case NAMENODE:
      #if 0 /* This breaks on ARRAYNODES (hidden parameters?) */
        if (TagOf(t) == T_TEMP || TagOf(t) == T_PREEVALTEMP)
          freetree(NTypeAddr(t)); /* freeup the expression for the temporary */
      #endif
        return; /* don't free the namenode here! */
      case REPLCNODE:
        freetree(ReplCStartExpAddr(t));
        freetree(ReplCLengthExpAddr(t));
        freetree(ReplCBodyAddr(t));
        freetree(ReplCTempAddr(t));
        freetree(NTypeAddr(ReplCNameOf(t))); /* The repl var's type tree */
        freenode(ReplCNameAddr(t)); /* freeup the name node explicitly */
        break;
      case SEGMENTNODE:
        freetree(SNameAddr(t));
        freetree(SStartExpAddr(t));
        freetree(SLengthExpAddr(t));
        freetree(SCheckExpAddr(t));
        freetree(SSubscriptExpAddr(t));
        freetree(SLengthAddr(t));
        break;
      case SPACENODE:
        freetree(SpBodyAddr(t));
        break;
      case VALOFNODE:
        freetree(VLBodyAddr(t));
        freetree(VLResultListAddr(t));
        break;
      case VARIANTNODE:
        freetree(VRBodyAddr(t));
        freetree(VRTaggedListAddr(t));
        break;
      case WORDNODE: /* This may happen for the body of a string */
        return;      /* Don't reclaim name table entries */
    #ifdef CONFIG
      case CONFIGNODE:
        freetree(STDevAddr(t));
        freetree(STAttrNameAddr(t));
        freetree(STAttrExpAddr(t));
        break;
      case PROCESSORNODE:
        freetree(ProcessorExpAddr(t));
        /*freetree(ProcessorTypeAddr(t));*/ /* That's a wordnode */
        freetree(ProcessorBodyAddr(t));
        break;
    #endif
    }
  freenode(tptr);
}
/*}}}*/
#endif
/*}}}*/
