/*#define DEBUG*/
/******************************************************************************
*
*  Virtual tree handler - pretty printer and node types
*
*
******************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include <string.h>
# include "includes.h"
# include "genhdr.h"
# include "lexconst.h"
# include "lex1def.h"
# include "lexdef.h"
# include "chkdef.h"
# include "usehdr.h"
/*}}}*/

/*{{{  keywords and all names for nodes and tags*/
PRIVATE struct
  {
    int   s_tag;       /* lexical tag and/or node tag */
    char *s_keyword;   /* keyword, or NULL if not a keyword */
    char *s_tagstr;    /* 'user friendly' name */
    char *s_itagstr;   /* internal name */
    char *s_ftagstr;   /* data for error messages */
    int   s_nodetype;  /* Type of that treenode */
  } tokens[] =
{
{ 000,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ S_ADD,       NULL,     "+",      "ADD",    "\"+\"", DOPNODE },
{ S_AFTER,     "AFTER",  NULL,     NULL,     NULL,    DOPNODE },
{ S_ALT,       "ALT",    NULL,     NULL,     NULL,    CNODE },
{ S_AMPERSAND, NULL,     "&",      "AMPERSAND", "\"&\"", NONODE },
{ S_AND,       "AND",    NULL,     NULL,     NULL,    DOPNODE },
{ S_ANY,       "ANY",    NULL,     NULL,     NULL,    LEAFNODE },
{ S_ASS,       NULL,     ":=",     "ASSIGN", "\":=\"",ACTIONNODE },
{ S_AT,        "AT",     NULL,     NULL,     NULL,    NONODE },
{ S_BITAND,    "BITAND", NULL,     NULL,     NULL,    DOPNODE },
{ S_BITNOT,    "BITNOT", NULL,     NULL,     NULL,    MOPNODE },
{ S_BITOR,     "BITOR",  NULL,     NULL,     NULL,    DOPNODE },
{ S_BOOL,      "BOOL",   NULL,     NULL,     NULL,    LEAFNODE },
{ S_BOX,       NULL,     "[]",     "BOX",    "\"[]\"",NONODE },
{ S_BYTE,      "BYTE",   NULL,     NULL,     NULL,    LEAFNODE },
{ S_BYTELIT,   NULL,     "BYTE literal", "BYTELIT", "BYTE literal %s", LITNODE },
{ S_CASE,      "CASE",   NULL,     NULL,     NULL,    ACTIONNODE },
{ S_CHAN,      "CHAN",   NULL,     NULL,     NULL,    CHANNODE },
{ S_COLON,     NULL,     ":",      "COLON",  "\":\"", NONODE },
{ S_COLON2,    NULL,     "::",     "COLON2", "\"::\"",DOPNODE },
{ S_COMMA,     NULL,     ",",      "COMMA",  "\",\"", NONODE },
{ S_DIV,       NULL,     "/",      "DIV",    "\"/\"", DOPNODE },
{ S_ELSE,      "ELSE",   NULL,     NULL,     NULL,    LEAFNODE },
{ S_EQ,        NULL,     "=",      "EQ",     "\"=\"", DOPNODE },
{ S_FALSE,     "FALSE",  NULL,     NULL,     NULL,    LEAFNODE },
{ S_FOR,       "FOR",    NULL,     NULL,     NULL,    DOPNODE },
{ S_FROM,      "FROM",   NULL,     NULL,     NULL,    NONODE },
{ S_FUNCTION,  "FUNCTION",NULL,    NULL,     NULL,    NONODE },
{ S_GE,        NULL,     ">=",     "GE",     "\">=\"",DOPNODE },
{ S_GR,        NULL,     ">",      "GR",     "\">\"", DOPNODE },
{ S_IF,        "IF",     NULL,     NULL,     NULL,    CNODE },
{ S_INPUT,     NULL,     "?",      "INPUT",  "\"?\"", ACTIONNODE },
{ S_INT,       "INT",    NULL,     NULL,     NULL,    LEAFNODE },
{ S_INT16,     "INT16",  NULL,     NULL,     NULL,    LEAFNODE },
{ S_INT32,     "INT32",  NULL,     NULL,     NULL,    LEAFNODE },
{ S_INT64,     "INT64",  NULL,     NULL,     NULL,    LEAFNODE },
{ S_INTLIT,    NULL,     "INT literal", "INTLIT", "INT literal \"%s\"", LITNODE },
{ S_INT16LIT,  NULL,     "INT16 literal", "INT16LIT", "INT16 literal \"%s\"", LITNODE },
{ S_INT32LIT,  NULL,     "INT32 literal", "INT32LIT", "INT32 literal \"%s\"", LITNODE },
{ S_INT64LIT,  NULL,     "INT64 literal", "INT64LIT", "INT64 literal \"%s\"", LITNODE },
{ S_IS,        "IS",     NULL,     NULL,     NULL,    NONODE },
{ S_LBOX,      NULL,     "[",      "LBOX",   "\"[\"", NONODE },
{ S_LE,        NULL,     "<=",     "LE",     "\"<=\"",DOPNODE },
{ S_LPAREN,    NULL,     "(",      "LPAREN", "\"(\"", NONODE },
{ S_LS,        NULL,     "<",      "LS",     "\"<\"", DOPNODE },
{ S_LSHIFT,    NULL,     "<<",     "LSHIFT", "\"<<\"",DOPNODE },
{ S_MINUS,     "MINUS",  NULL,     NULL,     NULL,    DOPNODE },
{ S_MOSTNEG,   "MOSTNEG",NULL,     NULL,     NULL,    MOPNODE },
{ S_MOSTPOS,   "MOSTPOS",NULL,     NULL,     NULL,    MOPNODE },
{ S_MULT,      NULL,     "*",      "MULT",   "\"*\"", DOPNODE },
{ S_NAME,      NULL,     "name",   "NAME",   "name \"%s\"",WORDNODE },
{ S_NE,        NULL,     "<>",     "NE",     "\"<>\"",DOPNODE },
{ S_NOT,       "NOT",    NULL,     NULL,     NULL,    MOPNODE },
{ S_OF,        "OF",     NULL,     NULL,     NULL,    NONODE },
{ S_OR,        "OR",     NULL,     NULL,     NULL,    DOPNODE },
{ S_OUTPUT,    NULL,     "!",      "OUTPUT", "\"!\"", ACTIONNODE },
{ S_PAR,       "PAR",    NULL,     NULL,     NULL,    CNODE },
{ S_PLACE,     "PLACE",  NULL,     NULL,     NULL,    DECLNODE },
{ S_PLACED,    "PLACED", NULL,     NULL,     NULL,    NONODE },
{ S_PLUS,      "PLUS",   NULL,     NULL,     NULL,    DOPNODE },
{ S_PORT,      "PORT",   NULL,     NULL,     NULL,    CHANNODE },
{ S_PRI,       "PRI",    NULL,     NULL,     NULL,    NONODE },
{ S_PROC,      "PROC",   NULL,     NULL,     NULL,    NONODE },
{ S_PROCESSOR, "PROCESSOR",NULL,   NULL,     NULL,    PROCESSORNODE },
{ S_PROTOCOL,  "PROTOCOL",NULL,    NULL,     NULL,    NONODE },
{ S_RBOX,      NULL,     "]",      "RBOX",   "\"]\"", NONODE },
{ S_REAL32,    "REAL32", NULL,     NULL,     NULL,    LEAFNODE },
{ S_REAL64,    "REAL64", NULL,     NULL,     NULL,    LEAFNODE },
{ S_UREALLIT,  NULL,     "real literal", "UREALLIT", "real literal \"%s\"", LITNODE },
{ S_REAL32LIT, NULL,     "REAL32 literal", "REAL32LIT", "REAL32 literal \"%s\"", LITNODE },
{ S_REAL64LIT, NULL,     "REAL64 literal", "REAL64LIT", "REAL64 literal \"%s\"", LITNODE },
{ S_REM,       "REM",    "\\",     "REM",    "\"\\\"",DOPNODE },
{ S_RESULT,    "RESULT", NULL,     NULL,     NULL,    NONODE },
{ S_RETYPES,   "RETYPES",NULL,     NULL,     NULL,    NONODE },
{ S_ROUND,     "ROUND",  NULL,     NULL,     NULL,    MOPNODE },
{ S_RPAREN,    NULL,     ")",      "RPAREN", "\")\"", NONODE },
{ S_RSHIFT,    NULL,     ">>",     "RSHIFT", "\">>\"",DOPNODE },
{ S_SEMICOLON, NULL,     ";",      "SEMICOLON","\";\"",NONODE },
{ S_SEQ,       "SEQ",    NULL,     NULL,     NULL,    CNODE },
{ S_SIZE,      "SIZE",   NULL,     NULL,     NULL,    MOPNODE },
{ S_SKIP,      "SKIP",   NULL,     NULL,     NULL,    LEAFNODE },
{ S_STOP,      "STOP",   NULL,     NULL,     NULL,    LEAFNODE },
{ S_STRING,    NULL,     "string", "STRING", "string \"%s\"", CONSTTABLENODE },
{ S_SUBTRACT,  NULL,     "-",      "SUBTRACT","\"-\"",DOPNODE },
{ S_TIMER,     "TIMER",  NULL,     NULL,     NULL,    LEAFNODE },
{ S_TIMES,     "TIMES",  NULL,     NULL,     NULL,    DOPNODE },
{ S_TRUE,      "TRUE",   NULL,     NULL,     NULL,    LEAFNODE },
{ S_TRUNC,     "TRUNC",  NULL,     NULL,     NULL,    MOPNODE },
{ S_VAL,       "VAL",    NULL,     NULL,     NULL,    NONODE },
{ S_VALOF,     "VALOF",  NULL,     NULL,     NULL,    VALOFNODE },
{ S_WHILE,     "WHILE",  NULL,     NULL,     NULL,    CONDNODE },
{ S_XOR,       NULL,     "><",     "XOR",    "\"><\"",DOPNODE },

{ S_NEWLINE,   NULL,     "new line",NULL,    "end of line", NONODE },
{ S_END,       NULL,     "end of file","END",NULL,    LEAFNODE },
{ S_COMMENT,   NULL,     "comment",NULL,     NULL,    NONODE },

{ S_UINTLIT,   NULL,     "integer literal", "UINTLIT", "integer literal \"%s\"", LITNODE },
{ S_STRINGCONT, NULL,    NULL,     "STRINGCONT",NULL, NONODE },
{ S_UBYTELIT,  NULL,     "byte literal", "UBYTELIT", "byte literal %s", LITNODE },

{ S_ILLEGALSYMB,NULL,    NULL,     "ILLEGALSYMB",NULL,NONODE },
{ S_WORKSPACE, "WORKSPACE",NULL,   NULL,     NULL,    NONODE },
{ S_VECSPACE,  "VECSPACE",NULL,    NULL,     NULL,    NONODE },
{ S_IN,        "IN",     NULL,     NULL,     NULL,    NONODE },

{ S_SC,        NULL,     "#SC",    NULL,     NULL,    NONODE },
{ S_INCLUDE,   NULL,     "#INCLUDE",NULL,    NULL,    NONODE },
{ S_USE,       NULL,     "#USE",   NULL,     NULL,    NONODE },
{ S_INLINE,    "INLINE", NULL,     NULL,     NULL,    NONODE },
{ S_GUY,       "GUY",    NULL,     NULL,     NULL,    CNODE },
{ S_DOT,       ".",      NULL,     NULL,     "\".\"", NONODE },

/*{ S_STEP,    "STEP",   NULL,     NULL,     NULL,    NONODE },*/
{ 108,         NULL,     NULL,     NULL,     NULL,    NONODE },

{ S_IMPORT,    NULL,     "#IMPORT",NULL,     NULL,    NONODE },
{ S_OPTION,    NULL,     "#OPTION",NULL,     NULL,    NONODE },
{ S_HCOMMENT,  NULL,     "#COMMENT",NULL,    NULL,    NONODE },

#ifdef ARRAYCONSTRUCTOR
{ S_ARRAYCONSTRUCTOR,"ARRAY",NULL,"ARRAYCONSTRUCTOR",NULL,NONODE },
#else
{ 112,         NULL,     NULL,     NULL,     NULL,    NONODE },
#endif
#ifdef CONDEXP
{ S_RIGHTARROW,NULL,     "->",     "RIGHTARROW", "\"->\"", NONODE },
#else
{ 113,         NULL,     NULL,     NULL,     NULL,    NONODE },
#endif

#ifdef CONFIG
{ S_SET,       "SET",    NULL,     NULL,     NULL,    CONFIGNODE },
{ S_ARC,       "ARC",    NULL,     NULL,     NULL,    LEAFNODE },
{ S_EDGE,      "EDGE",   NULL,     NULL,     NULL,    LEAFNODE },
{ S_NODE,      "NODE",   NULL,     NULL,     NULL,    LEAFNODE },
{ S_CONFIG,    "CONFIG", NULL,     NULL,     NULL,    DECLNODE },
{ S_NETWORK,   "NETWORK",NULL,     NULL,     NULL,    DECLNODE },
{ N_CONFIG,    NULL,     NULL,     "N_CONFIG",NULL,   NAMENODE },
{ N_NETWORK,   NULL,     NULL,     "N_NETWORK",NULL,  NAMENODE },
{ S_CONNECT,   "CONNECT",NULL,     NULL,     NULL,    CONFIGNODE },
{ S_TO,        "TO",     NULL,     NULL,     NULL,    NONODE },
{ S_WITH,      "WITH",   NULL,     NULL,     NULL,    NONODE },
{ S_MAPPING,   "MAPPING",NULL,     NULL,     NULL,    DECLNODE },
{ N_MAPPING,   NULL,     NULL,     "N_MAPPING",NULL,  NAMENODE },
{ S_MAP,       "MAP",    NULL,     NULL,     NULL,    CONFIGNODE },
{ S_ON,        "ON",     NULL,     NULL,     NULL,    NONODE },
{ S_ONTO,      "ONTO",   NULL,     NULL,     NULL,    NONODE },
#else
{ 114,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 115,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 116,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 117,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 118,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 119,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 120,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 121,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 122,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 123,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 124,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 125,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 126,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 127,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 128,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 129,         NULL,     NULL,     NULL,     NULL,    NONODE },
#endif

{ S_LIST,      NULL,     NULL,     "LIST",   NULL,    LISTNODE },
{ S_DELAYED_INPUT,NULL,  NULL,     "DELAYED_INPUT",NULL,ACTIONNODE },
{ S_CASE_INPUT,NULL,     NULL,     "CASE_INPUT",NULL, ACTIONNODE },
{ S_VALABBR,   NULL,     NULL,     "VALABBR",NULL,    DECLNODE },
{ S_VALRETYPE, NULL,     NULL,     "VALRETYPE",NULL,  DECLNODE },
{ S_DECL,      NULL,     NULL,     "DECL",   NULL,    DECLNODE },
{ S_ABBR,      NULL,     NULL,     "ABBR",   NULL,    DECLNODE },
{ S_RETYPE,    NULL,     NULL,     "RETYPE", NULL,    DECLNODE },
{ S_TPROTDEF,  NULL,     NULL,     "TPROTDEF",NULL,   DECLNODE },
{ S_SPROTDEF,  NULL,     NULL,     "SPROTDEF", NULL,  DECLNODE },
{ S_PROCDEF,   NULL,     NULL,     "PROCDEF",NULL,    DECLNODE },
{ S_SFUNCDEF,  NULL,     NULL,     "SFUNCDEF",NULL,   DECLNODE },
{ S_LFUNCDEF,  NULL,     NULL,     "LFUNCDEF", NULL,  DECLNODE },
{ S_ARRAYSUB,  NULL,     NULL,     "ARRAYSUB", NULL,  ARRAYSUBNODE },
{ S_NEG,       NULL,     "-",      "NEG",    NULL,    MOPNODE },

{ 145,         NULL,     NULL,     NULL,     NULL,    NONODE },

{ S_EXACT,     NULL,     NULL,     "EXACT",  NULL,    MOPNODE },
{ S_SEGMENT,   NULL,     NULL,     "SEGMENT",NULL,    SEGMENTNODE },
{ S_VARIANT,   NULL,     "variant","VARIANT",NULL,    VARIANTNODE },
{ S_ARRAY,     NULL,     "ARRAY",  NULL,     NULL,    ARRAYNODE },
{ S_FNTYPE,    NULL,     NULL,     "FNTYPE", NULL,    LISTNODE },

{ N_VALABBR,   NULL,     "VAL abbreviation", "N_VALABBR",   NULL, NAMENODE },
{ N_VALRETYPE, NULL,     "VAL retype",       "N_VALRETYPE", NULL, NAMENODE },
{ N_ABBR,      NULL,     "abbreviation", "N_ABBR",    NULL, NAMENODE },
{ N_RETYPE,    NULL,     "retype", "N_RETYPE",        NULL, NAMENODE },
{ N_DECL,      NULL,     "declaration", "N_DECL",     NULL, NAMENODE },
{ N_TAGDEF,    NULL,     "tag",    "N_TAGDEF",        NULL, NAMENODE },
{ N_SPROTDEF,  NULL,     "protocol", "N_SPROTDEF",    NULL, NAMENODE },
{ N_TPROTDEF,  NULL,     "protocol", "N_TPROTDEF",    NULL, NAMENODE },
{ N_PROCDEF,   NULL,     "procedure", "N_PROCDEF",    NULL, NAMENODE },
{ N_SFUNCDEF,  NULL,     "function", "N_SFUNCDEF",    NULL, NAMENODE },
{ N_LFUNCDEF,  NULL,     "function", "N_LFUNCDEF",    NULL, NAMENODE },
{ N_VALPARAM,  NULL,     "VAL parameter",    "N_VALPARAM",  NULL, NAMENODE },
{ N_REPL,      NULL,     "replicator", "N_REPL",      NULL, NAMENODE },

{ S_ALTERNATIVE, NULL,   NULL,     "ALTERNATIVE", NULL, ALTNODE },
{ S_CONSTRUCTOR,NULL,    NULL,     "CONSTRUCTOR",NULL,MOPNODE },
{ N_PARAM,     NULL,     "parameter", "N_PARAM",      NULL, NAMENODE },
{ S_CONSTEXP,  NULL,     NULL,     "CONSTEXP",NULL,   CONSTEXPNODE },

{ 168,         NULL,     NULL,     NULL,     NULL,    NONODE },

{ S_UNDECLARED,NULL,     NULL,     "UNDECLARED",NULL, LEAFNODE },
{ S_WSPLACE,   NULL,     NULL,     "WSPLACE",NULL,    DECLNODE },
{ S_VSPLACE,   NULL,     NULL,     "VSPLACE",NULL,    DECLNODE },
{ N_SCPROCDEF, NULL,     NULL,     "N_SCPROCDEF",     NULL, NAMENODE },
{ N_SCFUNCDEF, NULL,     NULL,     "N_SCFUNCDEF",     NULL, NAMENODE },
{ N_LIBPROCDEF,NULL,     NULL,     "N_LIBPROCDEF",    NULL, NAMENODE },
{ N_LIBFUNCDEF,NULL,     NULL,     "N_LIBFUNCDEF",    NULL, NAMENODE },
{ N_PREDEFFUNCTION,NULL, NULL,     "N_PREDEFFUNCTION",NULL, NAMENODE },
{ N_PREDEFPROC, NULL,    NULL,     "N_PREDEFPROC",    NULL, NAMENODE },
{ N_STDLIBPROCDEF,NULL,  NULL,     "N_STDLIBPROCDEF", NULL, NAMENODE },
{ N_STDLIBFUNCDEF,NULL,  NULL,     "N_STDLIBFUNCDEF", NULL, NAMENODE },
{ N_LABELDEF,  NULL,     NULL,     "N_LABELDEF",      NULL, NAMENODE },
{ S_LABELDEF,  NULL,     NULL,     "LABELDEF", NULL,  DECLNODE },

{ S_UMINUS,    NULL,     "MINUS",  "UMINUS", NULL,    MOPNODE },
{ S_GUYCODE,   NULL,     NULL,     "GUYCODE",NULL,    DOPNODE },
{ S_GUYSTEP,   NULL,     NULL,     "GUYSTEP",NULL,    DOPNODE },
{ S_LABEL,     NULL,     NULL,     "LABEL",  NULL,    LEAFNODE },
{ S_PRIPAR,    NULL,     NULL,     "PRIPAR", NULL,    CNODE },
{ S_PRIREPLPAR,NULL,     NULL,     "PRIREPLPAR",NULL, REPLCNODE },
{ S_PRIALT,    NULL,     NULL,     "PRIALT", NULL,    CNODE },
{ S_PRIREPLALT,NULL,     NULL,     "PRIREPLALT",NULL, REPLCNODE },
{ S_PLACEDPAR, NULL,     NULL,     "PLACEDPAR", NULL, CNODE },
{ S_PLACEDREPLPAR,NULL,  NULL,     "PLACEDREPLPAR",NULL,REPLCNODE },
{ S_TAGGED_INPUT,NULL,   NULL,     "TAGGED_INPUT",NULL,ACTIONNODE },
{ S_REPLSEQ,   NULL,     NULL,     "REPLSEQ",NULL,    REPLCNODE },
{ S_REPLPAR,   NULL,     NULL,     "REPLPAR",NULL,    REPLCNODE },
{ S_REPLIF,    NULL,     NULL,     "REPLIF", NULL,    REPLCNODE },
{ S_REPLALT,   NULL,     NULL,     "REPLALT",NULL,    REPLCNODE },
{ S_CHOICE,    NULL,     NULL,     "CHOICE", NULL,    CONDNODE },
{ S_SELECTION, NULL,     NULL,     "SELECTION",NULL,  CONDNODE },
{ S_FINSTANCE, NULL,     NULL,     "FINSTANCE", NULL, INSTANCENODE },
{ S_PINSTANCE, NULL,     NULL,     "PINSTANCE", NULL, INSTANCENODE },

#ifdef CONDEXP
{ S_CONDEXP,   NULL,     "conditional expression", "CONDEXP", NULL, CONDEXPNODE },
#else
{ 201,         NULL,     NULL,     NULL,     NULL,    NONODE },
#endif
{ 202,         NULL,     NULL,     NULL,     NULL,    NONODE },

{ N_INLINEPROCDEF, NULL, NULL,     "N_INLINEPROCDEF", NULL, NAMENODE },
{ N_INLINEFUNCDEF, NULL, NULL,     "N_INLINEFUNCDEF", NULL, NAMENODE },

{ S_RECORDSUB, NULL,     NULL,     "RECORDSUB", NULL, ARRAYSUBNODE },
{ N_FIELD,     NULL,     NULL,     "N_FIELD",         NULL, NAMENODE },

#ifdef CONFIG
{ S_DO,        "DO",     NULL,     NULL,     NULL,    CNODE },
{ S_REPLDO,    NULL,     NULL,     "REPLDO", NULL,    REPLCNODE },
{ S_PLACEON,   NULL,     NULL,     "PLACEON",NULL,    DECLNODE },
#else
{ 207,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 208,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 209,         NULL,     NULL,     NULL,     NULL,    NONODE },
#endif

{ S_UNKNOWN,   NULL,     "unknown","UNKNOWN",NULL,    NONODE },
{ S_CONSTCONSTRUCTOR,NULL,NULL,    "CONSTCONSTRUCTOR",NULL, CONSTTABLENODE },

{ 212,         NULL,     NULL,     NULL,     NULL,    NONODE },

{ S_CONSTPTR,  NULL,     NULL,     NULL,     NULL,    NONODE }, /* used in constant folding */
{ S_CSUB0,     NULL,     "csub0",  "CSUB0",  NULL,    DOPNODE },
{ T_TEMP,      NULL,     "$temp",  "T_TEMP", NULL,    NAMENODE },

{ S_DUMMYEXP,  NULL,     "dummy expression","DUMMYEXP",NULL,LEAFNODE },
{ S_ARRAYITEM, NULL,     NULL,     "ARRAYITEM", NULL, ARRAYSUBNODE },
{ S_SEGMENTITEM,NULL,    NULL,     "SEGMENTITEM",NULL,SEGMENTNODE },
{ T_PREEVALTEMP,NULL,    "$preevaltemp", "T_PREEVALTEMP", NULL, NAMENODE },
{ S_CCNT1,     NULL,     "ccnt1",  "CCNT1",  NULL,    DOPNODE },
{ S_FNFORMALRESULT,NULL, "$formal.result", "FORMALRESULT", NULL, HIDDENPARAMNODE },
{ S_FNACTUALRESULT,NULL, "$actual.result", "ACTUALRESULT", NULL, HIDDENPARAMNODE },
{ S_ELSIZE,    NULL,     "ELSIZE", NULL,     NULL,    MOPNODE },
{ S_SEGSTART,  NULL,     "SEGSTART", NULL,   NULL,    MOPNODE },
{ S_HIDDEN_PARAM,NULL,   NULL,     "HIDDENPARAM",NULL,HIDDENPARAMNODE },
{ S_PARAM_STATICLINK,NULL,NULL,    "STATICLINK", NULL,LEAFNODE },
{ S_PARAM_VSP, NULL,     NULL,     "VSP",    NULL,    /*LEAFNODE*/HIDDENPARAMNODE },
{ S_EVAL,      NULL,     "EVAL",   NULL,     NULL,    DOPNODE },
{ S_OVERLAPCHECK,NULL,   NULL,     "OVERLAPCHECK",NULL,DOPNODE },

{ 230,         NULL,     NULL,     NULL,     NULL,    NONODE },

{ S_ADDRESSOF, NULL,     "ADDRESSOF", NULL,  NULL,    MOPNODE },

{ 232,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 233,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 234,         NULL,     NULL,     NULL,     NULL,    NONODE },
{ 235,         NULL,     NULL,     NULL,     NULL,    NONODE },

{ S_SPACEUSAGE, NULL,    NULL,     "SPACEUSAGE",NULL, SPACENODE },
{ T_REGTEMP,   NULL,     NULL,     "T_REGTEMP",NULL,  NAMENODE },
{ T_RESERVEDWS, NULL,    NULL,     "T_RESERVEDWS",NULL,NAMENODE },
#if 0
{ S_BASICBLOCK,NULL,     NULL,     "BASICBLOCK", NULL,CNODE },
#else
{ 239,         NULL,     NULL,     NULL,     NULL,    NONODE },
#endif

{ S_PRAGMA,    NULL,     "#PRAGMA",NULL,     NULL,    NONODE },
{ S_ASM,       "ASM",    NULL,     NULL,     NULL,    CNODE },
{ S_ASMNAME,   NULL,     "asmname","ASMNAME", NULL,   WORDNODE }

};

#define MAX_TAG sizeof(tokens)/sizeof(tokens[0])

PRIVATE char *asmnames[ASMNAMES_COUNT] =
  { ".WSSIZE", ".VSPTR", ".STATIC" };
PUBLIC INT32 asmvalues[ASMNAMES_COUNT];
PUBLIC int   asmvalids[ASMNAMES_COUNT];

/*}}}*/

/*{{{  PRIVATE void dec (s, item)*/
PRIVATE void dec ( char *s , int item )
{
  wordnode *wptr = lookupword(s, strlen(s));
  SetWTag(wptr,item);
}
/*}}}*/
/*{{{  PUBLIC void declkeywords ()*/
/* Insert the occam reserved words in the symbol table */
PUBLIC void declkeywords ( void )
{
  int i;
  for (i = 0; i < MAX_TAG; i++)
    {
      assert(tokens[i].s_tag == i);
        
      if (tokens[i].s_keyword != NULL)
        dec(tokens[i].s_keyword, i);
    }

  for (i = 0; i < ASMNAMES_COUNT; i++)
    {
      dec(asmnames[i], S_ASMNAME);
      asmvalids[i] = ASMNAME_INVALID; /* currently unknown value */
    }
}
/*}}}*/
/*{{{  PUBLIC char *keywordstring (t)    occam keyword*/
PUBLIC char *keywordstring ( int t )
{
  if (t < 0 || t >= MAX_TAG) return NULL;
  return tokens[t].s_keyword;
}
/*}}}*/
/*{{{  PUBLIC int nodetypeoftag (t)*/
PUBLIC int nodetypeoftag ( int t )
{
  if (t < 0 || t >= MAX_TAG) return NONODE;
  return tokens[t].s_nodetype;
}
/*}}}*/
/*{{{  PUBLIC char *tagstring(t)    occam tag*/
PUBLIC char *tagstring ( int t )
{
  if (t >= 0 && t < MAX_TAG)
    {
      char *s;
      s = tokens[t].s_tagstr;
      if (s != NULL)
        return s;

      s = tokens[t].s_keyword;
      if (s != NULL)
        return s;
    }

  {
    static char a_tagstring[50];
    sprintf(a_tagstring, "tagstring: unknown tag %d", t);
    return a_tagstring;
  }
}
/*}}}*/
/*{{{  PUBLIC char *itagstring (t)   internal tag*/
PUBLIC char *itagstring ( int t )
{
  if (t >= 0 && t < MAX_TAG)
    {
      char *s = tokens[t].s_itagstr;
      if (s != NULL)
        return s;
    }
  return tagstring(t);
}
/*}}}*/
/*{{{  PRIVATE getstrchar*/
/* return a string for byte b */
PRIVATE char *getstrchar ( int b )
{
  char *s;
  static char string[5]; /* enough for *#nn with a NUL on the end */
  switch (b)
    {
      case '\r': s = "*c";  break;
      case '\n': s = "*n";  break;
      case '\t': s = "*t";  break;
      case '\'': s = "*'";  break;
      case '"':  s = "*\""; break;
      case '*':  s = "**";  break;
      default:
        if ((b >= 32) && (b <= 127))
          { string[0] = b; string[1] = '\0'; }
        else
          sprintf(string, "*#%02X", b & 0xff);
        return string;
    }
  return s;
}
/*}}}*/
/*{{{  PUBLIC void ftagstring (t)   more data for errors*/
PUBLIC void ftagstring (char *string, int t )
{
  if (t >= 0 && t < MAX_TAG)
    {
      char *s = tokens[t].s_ftagstr;
      if (s != NULL)
        {
        #if 0 /* bug 319 5/11/90 */
          /* Using sprintf breaks if the literal contains a NUL character*/
          /* However, the result of ftagstring is also processed by printf type
             functions, so we really need to correct them all! */
          /* NO: this is the only place where literalv is accessed.
             Just expand the non-printing chars here. CO'N 5/11/90 */
          sprintf(string, s, (t == S_NAME) ? lexword->w_name : literalv);
        #else
          char *dest = string;
          while ((*s) != '\0')
            {
              if ((*s) == '%')
                {
                  int len = (t == S_NAME) ? lexword->w_length : literalp;
                #if 1 /* can't use memcpy cos we need to expand stuff */
                  char *c = (t == S_NAME) ? lexword->w_name   : literalv;
                  int i;
                  for (i = 0; i < len; i++)
                    /* *dest++ = c[i]; */
                    {
                      char *expanded = getstrchar(c[i]);
                      int expanded_len = strlen(expanded);
                      memcpy(dest, expanded, expanded_len);
                      dest += expanded_len;
                    }
                #else
                  memcpy(dest, (t == S_NAME) ? lexword->w_name : literalv, len);
                  dest += len;
                #endif
                  s += 2; /* skip over the '%s' */
                }
              else
                *dest++ = *s++;
            }
          *dest = '\0';
        #endif
          return;
        }
    }
  strcpy (string, tagstring(t));
}
/*}}}*/
/*{{{  PRIVATE void ptag (t)          occam tag*/
PRIVATE void ptag ( int t )
{
  fprintf (outfile, "%s ", tagstring(t));
}
/*}}}*/
/*{{{  PRIVATE void pitag (t)         internal tag*/
PRIVATE void pitag ( int t )
{
  fprintf (outfile, "%s ", itagstring(t));
}
/*}}}*/
/*{{{  PUBLIC int which_asmname*/
PUBLIC int which_asmname ( treenode *tptr )
{
  int i;
  for (i = 0; i < ASMNAMES_COUNT; i++)
    if (strcmp(asmnames[i], WNameOf((wordnode *)tptr)) == 0)
      return i;
  return 0;  /* should never be reached */
}
/*}}}*/
/*{{{  PUBLIC int parrepl*/
PUBLIC int parrepl(int tag)
{
  switch(tag)
    {
      case S_REPLPAR: case S_PRIREPLPAR: case S_PLACEDREPLPAR:
      case S_REPLDO:
        return TRUE;
      default:
        return FALSE;
    }
}
/*}}}*/
#ifdef CONFIG
/*{{{  PUBLIC int network_datatype(int type)*/
PUBLIC int network_datatype(int type)
{
  switch(type)
    {
      CASE_CONFIG_TYPE
        return TRUE;
      default:
        return FALSE;
    }
}
/*}}}*/
#endif
/*{{{  PRIVATE printstrchar*/
/* Print out the byte b */
PRIVATE void printstrchar ( int b )
{
  fputs(getstrchar(b), outfile);
}
/*}}}*/
/*{{{  PUBLIC printbyte*/
/* Print out the byte b */
PUBLIC void printbyte ( int b )
{
  putc ('\'', outfile);
  printstrchar(b);
  fprintf (outfile, "\' ");
}
/*}}}*/
/*{{{  PUBLIC printstring*/
/* Print out the string in s[0..l] */
PUBLIC void printstring ( char *s , int l )
{
  int i;
  putc ('"', outfile);
  for (i = 0; i < l; i++)
    printstrchar(s[i]);
  fprintf (outfile, "\" ");
}
/*}}}*/
/*{{{  support for printtree*/
/*{{{  PRIVATE void setindent (i)*/
PRIVATE void setindent ( int i )
{
  int n;
  putc ('\n', outfile);
  for (n = 0; n < i; n++)
    putc (' ', outfile);
}
/*}}}*/
/*{{{  printdecllist  forward declaration*/
PRIVATE void printdecllist PARMS((int indent, treenode *n));
/*}}}*/
/*{{{  printmode()*/
PRIVATE void printmode ( int mode )
{
  char *s;
  switch (mode)
    {
      case NM_DEFAULT:   s = "NM_DEFAULT";   break;
      case NM_WORKSPACE: s = "NM_WORKSPACE"; break;
      case NM_VECSPACE:  s = "NM_VECSPACE";  break;
      case NM_PLACED:    s = "NM_PLACED";    break;
      case NM_POINTER:   s = "NM_POINTER";   break;
    /*case NM_INLINE:    s = "NM_INLINE";    break;*/ /* never used */
      case NM_WSPLACED:  s = "NM_WSPLACED";  break;
      default:           s = "unknown";      break;
    }
  fprintf(outfile, " Mode %s", s);
}
/*}}}*/
/*{{{  PRIVATE void printdecl (indent, n)*/
PRIVATE void printdecl ( int indent , treenode *n )
{
  int dtype = TagOf(n);
  if (nodetypeoftag(dtype) == NAMENODE)
    {
      wordnode *wptr = NNameOf(n);
      treenode *type = NTypeOf(n);
      setindent (indent);
      printstring (WNameOf(wptr), WLengthOf(wptr));
      DEBUG_MSG(("(0x%x)", n));
      fprintf(outfile, ", lexlevel = %d, scope = %d,",
              NLexLevelOf(n), NScopeOf(n));
      printmode(NModeOf(n));
      switch(dtype)
        {
          case N_TPROTDEF: case N_PROCDEF: case N_INLINEPROCDEF:
            printdecllist(indent + 2, type);             /* tags or parameters */
            break;
          case N_SFUNCDEF: case N_LFUNCDEF: case N_INLINEFUNCDEF:
            DEBUG_MSG((" %s:(0x%x) ", itagstring(TagOf(type)), type));
            printtree(indent + 2, FnTypeListOf(type));       /* result types */
            printdecllist(indent + 2, FnParamsOf(type));       /* parameters */
            break;
          default:
            printtree (indent + 2, type);
            break;
        }

      switch(dtype)
        {
          case N_PROCDEF: case N_SFUNCDEF: case N_LFUNCDEF:
          case N_INLINEPROCDEF: case N_INLINEFUNCDEF:
            /*{{{  print out the free variable list*/
            {
              varlist *vlist = NFreeVarsOf(n);
              setindent(indent + 2);
              if (vlist == (varlist *)(-1))
                vlist = NULL;
              fprintf(outfile, "%sFree variables:", (vlist == NULL) ? "No " : "");
              while (vlist != NULL)
                {
                  printtree(indent + 4, VLNameOf(vlist));
                  vlist = VLNextOf(vlist);
                }
            }
            break;
            /*}}}*/
          default: break;
        }
    }
  else
    printtree(indent, n);
}
/*}}}*/
/*{{{  PRIVATE void printdecllist (indent, n)*/
PRIVATE void printdecllist ( int indent , treenode *n )
{
  while (!EndOfList(n))
    /*{{{  print left, move to right*/
    {
      setindent (indent);
      pitag (S_LIST);
      DEBUG_MSG(("(0x%x) ", n));
      printdecl (indent + 2, ThisItem(n));
      n = NextItem(n);
    }
    /*}}}*/
  setindent(indent);
  fprintf(outfile, "NULL");
}
/*}}}*/
/*}}}*/

/*{{{  PUBLIC void printexp(tptr)*/
PUBLIC void printexp ( treenode *tptr )
{
  while (tptr != NULL)
    switch (TagOf(tptr))
      /*{{{  cases*/
      {
        /*{{{  monadic operator*/
        case S_MOSTPOS: case S_MOSTNEG:
          ptag (TagOf(tptr));
          ptag (TagOf(OpOf(tptr)));
          return;
        case S_NEG: case S_BITNOT: case S_NOT: case S_SIZE: case S_UMINUS:
        case S_ADDRESSOF:
          ptag (TagOf(tptr));
          tptr = OpOf(tptr);
          break;
        case S_ELSIZE:
          tptr = dimexpof(OpOf(tptr), 0);
          break;
        case S_SEGSTART:
          tptr = SStartExpOf(OpOf(tptr));
          break;
        /*}}}*/
        /*{{{  conversion*/
        case S_EXACT: case S_ROUND: case S_TRUNC:
          ptag (MOpTypeOf(tptr));
          if (TagOf(tptr) != S_EXACT)
            ptag (TagOf(tptr));
          tptr = OpOf(tptr);
          break;
        /*}}}*/
        /*{{{  constructor*/
        case S_CONSTRUCTOR:
          ptag (S_LBOX);
          printexp (OpOf(tptr));
          ptag (S_RBOX);
          return;
        /*}}}*/
        /*{{{  dyadic operator*/
        case S_AND: case S_OR:
        case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
        case S_BITAND: case S_BITOR: case S_XOR:
        case S_LSHIFT: case S_RSHIFT:
        case S_PLUS: case S_MINUS: case S_TIMES:
        case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE: case S_AFTER:
        case S_CSUB0: case S_CCNT1: case S_EVAL: /*case S_POSTEVAL:*/
        case S_OVERLAPCHECK: case S_FOR:
          ptag (S_LPAREN);
          printexp(LeftOpOf(tptr));
          ptag(TagOf(tptr));
          printexp(RightOpOf(tptr));
          ptag (S_RPAREN);
          return;
        /*}}}*/
        /*{{{  literal*/
        /*{{{  Bool literals*/
        case S_TRUE:
        case S_FALSE:
          ptag (TagOf(tptr));
          return;
        /*}}}*/
        /*{{{  byte int and real literals*/
        case S_INTLIT:
        case S_INT16LIT:
        case S_INT32LIT:
        case S_INT64LIT:
        case S_REAL32LIT:
        case S_REAL64LIT:
        case S_BYTELIT:
          {
            wordnode *nptr = (wordnode *)StringPtrOf(tptr);
            if (nptr != NULL) printstring (WNameOf(nptr), WLengthOf(nptr));
          }
          return;
        /*}}}*/
        /*{{{  string*/
        case S_STRING:
          {
            wordnode *nptr = (wordnode *)CTValOf(tptr);
            printstring (WNameOf(nptr), WLengthOf(nptr));
          }
          return;
        /*}}}*/
        /*{{{  constconstructor*/
        case S_CONSTCONSTRUCTOR:
          tptr = CTExpOf(tptr);
          break;
        /*}}}*/
        /*}}}*/
        /*{{{  function instance*/
        case S_FINSTANCE:
          printexp(INameOf(tptr));
          ptag(S_LPAREN);
          printexp (IParamListOf(tptr));
          ptag(S_RPAREN);
          return;
        /*}}}*/
        /*{{{  subscript*/
        case S_ARRAYSUB: case S_RECORDSUB:
          printexp(ASBaseOf(tptr));
          ptag (S_LBOX);
          printexp (ASIndexOf(tptr));
          ptag (S_RBOX);
          return;
        /*}}}*/
        /*{{{  transformed subscript*/
        case S_ARRAYITEM:
          printexp(ASBaseOf(tptr));
          ptag (S_LBOX);
          if (ASOffsetOf(tptr) != ZERO32) ptag(S_LPAREN);
          if (ASExpOf(tptr) != NULL)
            printexp(ASExpOf(tptr));
          if (ASOffsetOf(tptr) != ZERO32)
            {
              ptag(S_RPAREN);
              ptag(S_PLUS);
              fprintf (outfile, "%ld ", (long)ASOffsetOf(tptr));
            }
          ptag(S_RBOX);
          return;
        /*}}}*/
        /*{{{  segment*/
        case S_SEGMENT:
        case S_SEGMENTITEM:
          ptag (S_LBOX);
          printexp(SNameOf(tptr));
          ptag (S_FROM);
          printexp (SStartExpOf(tptr));
          ptag (S_FOR);
          printexp (SLengthExpOf(tptr));
          ptag (S_RBOX);
          return;
        /*}}}*/
        /*{{{  name*/
        case N_VALABBR:
        case N_VALRETYPE:
        case N_ABBR:
        case N_RETYPE:
        case N_DECL:
        case N_TAGDEF:
        case N_SPROTDEF:
        case N_TPROTDEF:
        case N_VALPARAM:
        case N_PARAM:
        case N_PROCDEF:
        case N_SFUNCDEF:
        case N_LFUNCDEF:
        case N_REPL:
        case N_SCFUNCDEF:
        case N_LIBFUNCDEF:
        case N_SCPROCDEF:
        case N_LIBPROCDEF:
        case N_STDLIBFUNCDEF:
        case N_STDLIBPROCDEF:
        case N_PREDEFFUNCTION:
        case N_PREDEFPROC:
        case N_FIELD:
          {
            wordnode *nptr = NNameOf(tptr);
            printstring (WNameOf(nptr), WLengthOf(nptr));
          }
          return;
        /*}}}*/
        /*{{{  list*/
        case S_LIST:
          {
            printexp(ThisItem(tptr));
            tptr = NextItem(tptr);
            if (tptr == NULL)
              return;
            ptag(S_COMMA);
          }
          break;
        /*}}}*/
        /*{{{  specification valof*/
        case S_VALABBR:
        case S_VALRETYPE:
        case S_DECL:
        case S_ABBR:
        case S_RETYPE:
        case S_TPROTDEF:
        case S_SPROTDEF:
        case S_PROCDEF:
        case S_SFUNCDEF:
        case S_LFUNCDEF:
        case S_VALOF:
          while (TagOf(tptr) != S_VALOF)
            tptr = DBodyOf(tptr);
          ptag (S_VALOF);
          fprintf (outfile, "... ");
          ptag (S_RESULT);
          tptr = VLResultListOf(tptr);
          break;
        /*}}}*/
        /*{{{  constant expression*/
        case S_CONSTEXP:
          if (CExpOf(tptr) != NULL && TagOf(CExpOf(tptr)) != S_DUMMYEXP)
            printexp(CExpOf(tptr));
          else
            fprintf (outfile, "%ld ", (long)LoValOf(tptr));
          return;
        /*}}}*/
        /*{{{  temp*/
        case T_TEMP:
        case T_PREEVALTEMP:
          fprintf(outfile, "$%stemp%ld ",
                  (TagOf(tptr) == T_TEMP) ? "" : "preeval", NVVarNumOf(tptr));
          return;
        /*}}}*/
        /*{{{  special parameters*/
        case S_HIDDEN_PARAM:
          fprintf(outfile, "hiddenparam");
          return;
        case S_PARAM_STATICLINK:
          fprintf(outfile, "staticlink");
          return;
        case S_PARAM_VSP:
          fprintf(outfile, "vsptr");
          return;
        /*}}}*/
        /*{{{  array dimension*/
        case S_ARRAY:
          fprintf(outfile, "arraydim");
          return;
        /*}}}*/
        default:
          ptag(TagOf(tptr));
          return;
      }
      /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC void printtree (indent, n)*/
PUBLIC void printtree ( int indent , treenode *n )
{
  while (n != NULL)
    {
      setindent (indent);
      pitag (TagOf(n));
      DEBUG_MSG(("(0x%x) ", n));
      switch (nodetypeoftag(TagOf(n)))
        {
          /*{{{  cases*/
          case LEAFNODE: return;
          /*{{{  construct node*/
          case CNODE:
            indent += 2;
            /*if (CTempOf(n) != NULL)
              printtree(indent, CTempOf(n));*/
            n = CBodyOf(n);
            break;
          /*}}}*/
          /*{{{  replicated construct node*/
          case REPLCNODE:
            indent += 2;
            printtree (indent, ReplCNameOf(n));
            printtree (indent, ReplCStartExpOf(n));
            printtree (indent, ReplCLengthExpOf(n));
            n = ReplCBodyOf(n);
            break;
          /*}}}*/
          /*{{{  condition node*/
          case CONDNODE:
            indent += 2;
            printtree (indent, CondGuardOf(n));
            n = CondBodyOf(n);
            break;
          /*}}}*/
          /*{{{  instance node*/
          case INSTANCENODE:
            if (inline(INameOf(n)))
              fprintf(outfile, "INLINE");
            indent += 2;
            printtree (indent, INameOf(n));
            n = IParamListOf(n);
            break;
          /*}}}*/
          /*{{{  action node*/
          case ACTIONNODE:
            indent += 2;
            printtree (indent, LHSOf(n));
            n = RHSOf(n);
            break;
          /*}}}*/
          /*{{{  declaration node*/
          case DECLNODE:
            if (inline(DNameOf(n)))
              fprintf(outfile, "INLINE");
            if (TagOf(DNameOf(n)) == S_LIST)
              printdecllist(indent + 2, DNameOf(n));
            else
              printdecl(indent + 2, DNameOf(n));
            if (DValOf(n) != NULL) printtree(indent + 2, DValOf(n));
            /* indent += 2; */
            n = DBodyOf(n);
            break;
          /*}}}*/
          /*{{{  list node*/
          case LISTNODE:
            if (TagOf(n) == S_FNTYPE)
              {
                indent += 2;
                printtree (indent, FnTypeListOf(n));
                n = FnParamsOf(n);
              }
            else
              {
                /*indent += 2;
                  printtree (indent, ThisItem(n));*/
                printtree (indent + 2, ThisItem(n));
                n = NextItem(n);
              }
            break;
          /*}}}*/
          /*{{{  monadic operator node*/
          case MOPNODE:
            indent += 2;
            switch(TagOf(n))
              {
                default:
                  setindent (indent);
                  pitag (MOpTypeOf(n));
                  n = OpOf(n);
                  break;
                case S_ELSIZE:
                  n = dimexpof(OpOf(n), 0);
                  break;
                case S_SEGSTART:
                  n = SStartExpOf(OpOf(n));
                  break;
              }
            break;
          /*}}}*/
          /*{{{  dyadic operator node*/
          case DOPNODE:
            indent += 2;
            if ((TagOf(n) == S_GUYCODE) || (TagOf(n) == S_GUYSTEP))
              {
                /*setindent (indent);*/
                fprintf (outfile, "Code: #%X, (%s)", DOpTypeOf(n),
                         WNameOf((wordnode *)LeftOpOf(n)));
              }
            else
              {
                setindent (indent);
                pitag (DOpTypeOf(n));
                printtree(indent, LeftOpOf(n));
              }
            n = RightOpOf(n);
            break;
          /*}}}*/
          /*{{{  literal node*/
          case LITNODE:
            {
              wordnode *nptr = StringPtrOf(n);
              printstring (WNameOf(nptr), WLengthOf(nptr));
              return;
            }
          /*}}}*/
          /*{{{  valof node*/
          case VALOFNODE:
            indent += 2;
            printtree (indent, VLBodyOf(n));
            n = VLResultListOf(n);
            break;
          /*}}}*/
          /*{{{  segment node*/
          case SEGMENTNODE:
            indent += 2;
            if (TagOf(n) == S_SEGMENTITEM)
              fprintf (outfile, "SOffsetOf: %d", (int)(SOffsetOf(n)));
            printtree (indent, SNameOf(n));
            printtree (indent, SStartExpOf(n));
            printtree (indent, SLengthExpOf(n));
            if (TagOf(n) == S_SEGMENTITEM)
              {
                printtree (indent, SSubscriptExpOf(n));
                printtree (indent, SLengthOf(n));
              }
            n = SCheckExpOf(n);
            break;
          /*}}}*/
          /*{{{  variant node*/
          case VARIANTNODE:
            indent += 2;
            printtree (indent, VRTaggedListOf(n));
            n = VRBodyOf(n);
            break;
          /*}}}*/
          /*{{{  name node*/
          case NAMENODE:
            if ((TagOf(n) == T_TEMP) || (TagOf(n) == T_REGTEMP) || (TagOf(n)==T_PREEVALTEMP))
            {
              fprintf(outfile, "%ld, lexlevel = %d ", NVVarNumOf(n), NLexLevelOf(n));
             /*fprintf(outfile, "lexlevel = %d ", NLexLevelOf(n));*/
              if ((TagOf(n) != T_REGTEMP) && (NVOffsetOf(n) != NO_SLOT))
                fprintf(outfile, "Offset = %ld ", NVOffsetOf(n));
              fprintf(outfile, "Type = "); pitag(TagOf(NTypeOf(n)));
              printmode(NModeOf(n));
              n = NDeclOf(n);
            }
            else
            {
              wordnode *nptr = NNameOf(n);
              printstring (WNameOf(nptr), WLengthOf(nptr));
              fprintf(outfile, "Type = ");
              if (NTypeOf(n) == NULL) 
                fprintf (outfile, "NULL ");
              else
                pitag(TagOf(NTypeOf(n)));
              printmode(NModeOf(n));
              return;
            }
            break;
          /*}}}*/
          /*{{{  word node*/
          case WORDNODE:
            {
              wordnode *w = (wordnode *)n;
              printstring(WNameOf(w), WLengthOf(w));
            }
            return;
          /*}}}*/
          /*{{{  array node*/
          case ARRAYNODE:
            indent += 2;
            fprintf (outfile, "ARDimOf: %d", (int)ARDimOf(n));
            printtree (indent, ARDimLengthOf(n));
            n = ARTypeOf(n);
            break;
          /*}}}*/
          /*{{{*/
          case ARRAYSUBNODE:
            indent += 2;
            if (TagOf(n) == S_ARRAYITEM)
              fprintf(outfile, "ASOffsetOf: %d", (int)(ASOffsetOf(n)));
            printtree(indent, ASBaseOf(n));
            if (TagOf(n) == S_ARRAYITEM)
              {
                printtree(indent, ASExpOf(n));
                n = ASLengthOf(n);
              }
            else
              n = ASIndexOf(n);
            break;
          /*}}}*/
          /*{{{  channel node*/
          case CHANNODE:
            indent += 2;
            n = ProtocolOf(n);
            break;
          /*}}}*/
          /*{{{  alternative node*/
          case ALTNODE:
            indent += 2;
            printtree (indent, AltGuardOf(n));
            printtree (indent, AltChanExpOf(n)); /* added 2/11/90 bug 779 */
            printtree (indent, AltInputOf(n));
            n = AltBodyOf(n);
            break;
          /*}}}*/
          /*{{{  constant expression node*/
          case CONSTEXPNODE:
            fprintf(outfile, "offset: %ld", CEOffsetOf(n));
            indent += 2;
            setindent (indent);
            fprintf (outfile, "%d (Lo:#%X, Hi:#%X)", (int)LoValOf(n), (int)LoValOf(n), (int)HiValOf(n));
            n = CExpOf(n);
            break;
          /*}}}*/
          /*{{{  constant table node*/
          case CONSTTABLENODE:
            if (TagOf(n) == S_STRING)
            {
              wordnode *nptr = (wordnode *)CTValOf(n);
              printstring (WNameOf(nptr), WLengthOf(nptr));
              return;
            }
            else /*S_CONSTCONSTRUCTOR*/
            {
              wordnode *nptr = (wordnode *)CTValOf(n);
              int i;
              BYTE *b = (BYTE *)WNameOf(nptr);
              for (i = 0; i< WLengthOf(nptr); i++)
                {
                  if ((i % 16) == 0) setindent(indent + 2);
                  fprintf(outfile, "#%-3x", *b++);
                }
              indent += 2;
              n = CTExpOf(n);
              break;
            }
          /*}}}*/
          /*{{{  hidden parameter node*/
          case HIDDENPARAMNODE:
            fprintf(outfile, "Dim/offset: %d", HDimensionOf(n));
            /* bug 1012 - if we try to see what's inside an actual result,
               we get stuck into an infinite loop */
            if (TagOf(n) == S_FNACTUALRESULT)
              return;
            indent += 2;
            n = HExpOf(n);
            break;
          /*}}}*/
          /*{{{   PROCESSOR*/
          case PROCESSORNODE:
            fprintf(outfile, "%s, scope: %d",
                             ProcessorTypeOf(n) == NULL ? "NULL"
                             : WNameOf(ProcessorTypeOf(n)),
                             (int)ProcessorScopeOf(n));
            indent += 2;
            printtree (indent, ProcessorExpOf(n));
            n = ProcessorBodyOf(n);
            break;
          /*}}}*/
          /*{{{  space usage node*/
          case SPACENODE:
            fprintf(outfile, "maxwsp = %ld, datasize = %ld", SpMaxwspOf(n),
                             SpDatasizeOf(n));
            n = SpBodyOf(n);
            break;
          /*}}}*/
          #ifdef CONDEXP
          /*{{{  conditional expression node*/
          case S_CONDEXPNODE:
            indent += 2;
            printtree (indent, CondExpGuardOf(n));
            printtree (indent, CondExpTrueOf(n));
            n = CondExpFalseOf(n);
            break;
          /*}}}*/
          #endif
          #ifdef CONFIG
          /*{{{  confignode*/
          case CONFIGNODE :
            indent += 2;
            printtree (indent, STDevOf(n));
            printtree (indent, STAttrNameOf(n));
            n = STAttrExpOf(n);
            break;
          /*}}}*/
          #endif
          /*}}}*/
          default:
            fprintf(outfile, "printtree: Unknown tag %d", TagOf(n));
            return;
        }
    }
  setindent (indent);
  fprintf (outfile, "NULL");
}
/*}}}*/
#if 0 /* not used */
/*{{{  PUBLIC void printstatement (indent, n)*/
PUBLIC void printstatement ( int indent , treenode *n )
{
  setindent (indent);
  switch (TagOf(n))
    {
      /*{{{  leafnode*/
      case S_SEQ:
      case S_IF:
      case S_PAR: case S_DO:
      case S_ALT:
      case S_GUY: case S_ASM:
      case S_STOP:
      case S_SKIP:
        ptag(TagOf(n));
        break;
      /*}}}*/
      /*{{{  pri and placed*/
      case S_PRIPAR:
      case S_PRIALT:
        ptag(S_PRI);
        putc (' ', outfile);
        ptag(TagOf(n) == S_PRIPAR ? S_PAR : S_ALT);
        break;
      case S_PLACEDPAR:
        ptag(S_PLACED);
        putc (' ', outfile);
        ptag(S_PAR);
        break;
      /*}}}*/
      /*{{{  replicated construct node*/
      case S_REPLSEQ:
      case S_REPLPAR:
      case S_REPLIF:
      case S_REPLALT:
      case S_PRIREPLPAR:
      case S_PRIREPLALT:
      case S_PLACEDREPLPAR: case S_REPLDO:
        {
          int tag1 = 0, tag2 = 0;
          switch(TagOf(n))
            /*{{{  set up tag1 and tag2*/
            {
              case S_REPLSEQ:                        tag2 = S_SEQ; break;
              case S_REPLPAR:                        tag2 = S_PAR; break;
              case S_REPLIF:                         tag2 = S_IF;  break;
              case S_REPLALT:                        tag2 = S_ALT; break;
              case S_REPLDO:                         tag2 = S_DO;  break;
              case S_PRIREPLPAR:    tag1 = S_PRI;    tag2 = S_PAR; break;
              case S_PRIREPLALT:    tag1 = S_PRI;    tag2 = S_ALT; break;
              case S_PLACEDREPLPAR: tag1 = S_PLACED; tag2 = S_PAR; break;
            }
            /*}}}*/
          if (tag1 != 0)
            /*{{{  print tag1*/
            {
              ptag(tag1);
              putc (' ', outfile);
            }
            /*}}}*/
          ptag(tag2);
          printexp (ReplCNameOf(n));
          fputs (" = ", outfile);
          printexp (ReplCStartExpOf(n));
          fputs (" FOR ", outfile);
          printexp (ReplCLengthExpOf(n));
        }
        break;
      /*}}}*/
      /*{{{  condition node*/
      case S_WHILE:
        ptag(TagOf(n));
        putc (' ', outfile);
        printexp (CondGuardOf(n));
        break;
      
      case S_CHOICE:
      case S_SELECTION:
        printexp (CondGuardOf(n));
        break;
      /*}}}*/
      /*{{{  instance node*/
      case S_FINSTANCE:
      case S_PINSTANCE:
        printexp (n);
        break;
      /*}}}*/
      /*{{{  action node*/
      /*{{{  S_ASS*/
      case S_ASS:
        printexp(LHSOf(n));
        fputs (" := ", outfile);
        printexp(RHSOf(n));
        break;
      /*}}}*/
      /*{{{  S_OUPUT*/
      case S_OUTPUT:
        printexp (LHSOf(n));
        fputs (" ! ", outfile);
        printexp(RHSOf(n));
        break;
      
      /*}}}*/
      /*{{{  S_INPUT S_TAGGED_INPUT  S_DELAYED_INPUT*/
      case S_INPUT:
      case S_TAGGED_INPUT:
      case S_DELAYED_INPUT:
        printexp (LHSOf(n));
        fputs (" ? ", outfile);
        printexp(RHSOf(n));
        break;
      
      /*}}}*/
      /*}}}*/
      /*{{{  case*/
      case S_CASE:
        ptag(TagOf(n));
        putc (' ', outfile);
        printexp (LHSOf(n));
        break;
      /*}}}*/
      /*{{{  declaration node*/
      case S_DECL:
        if (TagOf(DNameOf(n)) == S_LIST)
          printexp(NTypeOf(ThisItem(DNameOf(n))));
        else
          printexp(NTypeOf(DNameOf(n)));
        putc (' ', outfile);
        printexp(DNameOf(n));
        fputs (" :", outfile);
        break;
      case S_VALABBR:
      case S_ABBR:
        if (TagOf(n) == S_VALABBR)
          fputs ("VAL ", outfile);
        printexp(NTypeOf(DNameOf(n)));
        putc (' ', outfile);
        printexp(DNameOf(n));
        fputs (" IS ", outfile);
        printexp(DValOf(n));
        fputs (" :", outfile);
        break;
      /*}}}*/
      /*{{{  place node*/
      case S_PLACE:
        ptag(TagOf(n));
        putc (' ', outfile);
        printexp(DNameOf(n));
        fputs (" AT ", outfile);
        printexp(DValOf(n));
        fputs (" :", outfile);
        break;
      case S_WSPLACE:
      case S_VSPLACE:
        ptag(S_PLACE);
        putc (' ', outfile);
        printexp(DNameOf(n));
        fputs (" IN ", outfile);
        ptag(TagOf(n));
        fputs (" :", outfile);
        break;
      case S_PLACEON:
        something here.
      /*}}}*/
      /*{{{  valof node*/
      case S_VALOF:
        ptag(TagOf(n));
        break;
      /*}}}*/
      /*{{{  alternative node*/
      case S_ALTERNATIVE:
        printexp (AltGuardOf(n));
        fputs (" & ", outfile);
        printstatement(0, AltInputOf(n));
        break;
      /*}}}*/
      default:
        return;
    }
}
/*}}}*/
#endif
