/*#define DEBUG*/
/****************************************************************************
 *
 *  Occam two virtual tree interface routines
 *
 ****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>

# include "includes.h"

#ifdef ANSI
# include <stdlib.h>
# include <string.h>
#else
# include <malloc.h>
#endif

# include "vtierror.h"
# include "syndef.h"
# include "chkdef.h"
# include "lexdef.h"
# include "genhdr.h"

#if defined(DEBUG) && defined(IMS)
# include <misc.h>
#endif
/*}}}*/

#ifdef CONFIG
  #define FREEUP TRUE
#else
  #define FREEUP FALSE
#endif

/*{{{  private variables*/
#ifdef DEBUG
PRIVATE int node_count[256];
#endif

#if FREEUP
/* We keep a free list chain for all sizes up to 15 words long.
   This covers all the tree nodes.
   (Changed '15' to size of treenodes, plus a bit).
*/
/*#define MAXFREEUPSIZE (15 * sizeof(int))*/
#define MAXFREEUPSIZE (sizeof(treenode) + 2*sizeof(int))
PRIVATE void *freelist[MAXFREEUPSIZE]; /* will be initialised to NULLs */
#endif
/*}}}*/
/*{{{  space allocation*/
/*{{{  comment on workspace handling*/
/*
 * Two workspaces are maintained:
 *   the real workspace is where the parse tree and symbol table are built up.
 *   The real workspace is never reused.
 *
 *   the temporary workspace is used for building up temporary tree
 *     structures, for instance by the checker.
 *     This can be freed up at request, and this is done after each major
 *     phase of the compiler.
 *
 *   switch_to_temp_workspace causes the free space allocator, newvec to
 *     satisfy future space requests from the temporary workspace.
 *
 *   switch_to_real_workspace causes the free space allocator, newvec, to
 *     satisfy future space requests from the real workspace.
 *     This is the initial state.
 *
 *  There is also a mechanism for marking and freeing real workspace.
 */
/*}}}*/

/*{{{  definitions*/
#if defined(TDS) || defined(MSC)
#define WORKSPACESIZE     10000  /* Must be a multiple of HOST_ALIGN_SIZE */
#define TEMPWORKSPACESIZE  5000  /* Must be a multiple of HOST_ALIGN_SIZE */
#else
#define WORKSPACESIZE     50000  /* Must be a multiple of HOST_ALIGN_SIZE */
#define TEMPWORKSPACESIZE 10000  /* Must be a multiple of HOST_ALIGN_SIZE */
#endif

/* when this is enabled, we perform a simple check for malloc overruns
   every time newvec needs more memory
*/
#define CHECK_MALLOC

#ifdef CHECK_MALLOC
#define MAGIC1 12345678
#define MAGIC2 87654321
#endif

struct wsblock
{
#if 0
  struct wsblock *next;
#endif
#if defined(CHECK_MALLOC) || FREEUP
  struct wsblock *prev;
#endif
#ifdef CHECK_MALLOC
  INT32  magic1; /* check for under-runs */
#endif
  char   ws[WORKSPACESIZE];        /* This must align correctly */
#ifdef CHECK_MALLOC
  INT32  magic2; /* check for over-runs */
#endif
};
struct tempwsblock
{
  struct tempwsblock *prev;
#ifdef CHECK_MALLOC
  INT32  magic1; /* check for under-runs */
#endif
  char ws[TEMPWORKSPACESIZE];        /* This must align correctly */
#ifdef CHECK_MALLOC
  INT32  magic2; /* check for over-runs */
#endif
};

/*}}}*/
/*{{{  PRIVATE variables*/
PRIVATE int vecp, tempvecp, blocks, tempblocks;
PRIVATE struct wsblock *workspace = NULL;
/*PRIVATE char *tempworkspace;*/
PRIVATE struct tempwsblock *tempworkspace;
PRIVATE int tempworkspaceflag;

#ifdef DEBUG
PRIVATE size_t small_malloc = 0;
PRIVATE size_t large_malloc = 0;
PRIVATE size_t newvec_size  = 0;
PRIVATE size_t newvec_temp  = 0;
PRIVATE int malloc_count  = 0;
PRIVATE int free_count    = 0;
PRIVATE int newvec_count  = 0;
PRIVATE int newvec_malloc = 0;
#ifdef IMS
PRIVATE size_t total_malloc = 0;
#endif
#endif
/*}}}*/

/*{{{  PUBLIC void *memalloc(size)*/
PUBLIC void *memalloc ( const size_t size )
{
  void *p = malloc(size);
#ifdef DEBUG
  malloc_count++;
  if (size > 500 || size == 0)
    {
      large_malloc += size;
      DEBUG_MSG(("memalloc(%d), ", size));
    }
  else
    small_malloc += size;
#ifdef IMS
  /* the word below the pointer returned by malloc contains the
     length actually allocated
  */
  if (p != NULL)
    total_malloc += ((int *)p)[-1];
#endif
#endif
  assert(size != 0);
  if (p == NULL)
    {
      DEBUG_MSG(("memalloc: malloc of %d bytes returned NULL\n", size));
      vtiabort(VTI_OUT_OF_SPACE, flocn);
    }
  return p;
}
/*}}}*/
/*{{{  PUBLIC void memfree(ptr)*/
PUBLIC void memfree (void *const ptr )
{
#ifdef DEBUG
  free_count++;
#ifdef IMS
  /* the word below the pointer returned by malloc contains the
     length actually allocated
  */
  if (ptr != NULL)
    total_malloc -= ((int *)ptr)[-1];
#endif
#endif
  free(ptr);
}
/*}}}*/
#ifdef CHECK_MALLOC
/*{{{  PRIVATE void check_tempblocks*/
PRIVATE void check_tempblocks(const struct tempwsblock *t)
{
  while (t != NULL)
    {
      assert(t->magic1 == MAGIC1);
      assert(t->magic2 == MAGIC2);
      t = t->prev;
    }
}
/*}}}*/
#endif
/*{{{  newvec*/
/* Allocate n bytes of workspace and return a pointer to them */
PUBLIC void *newvec ( const size_t asked_n )
{
  size_t n;
  /* Make sure n is a multiple of HOST_ALIGN_SIZE */
#if HOST_ALIGN_SIZE != 1
  #if HOST_ALIGN_SIZE == 4
    n = (asked_n+3) & (~3);  /* round up to a multiple of 4 */
  #else
    {
      int tmp = asked_n % HOST_ALIGN_SIZE;
      if (tmp != 0)
        n = asked_n + (HOST_ALIGN_SIZE - tmp);
    }
  #endif
#else
  n = asked_n;
#endif
  if (tempworkspaceflag)
    {
    #ifdef DEBUG
      newvec_temp += n;
    #endif
      if (n > TEMPWORKSPACESIZE)
        {
      #if 0
          DEBUG_MSG(("newvec: out of temp space\n"));
          vtiabort(VTI_OUT_OF_SPACE, flocn);
      #endif
          return memalloc(n);
        }
      tempvecp -= n;
      if (tempvecp < 0)
        {
          struct tempwsblock *t = (struct tempwsblock *)
                              memalloc(sizeof(struct tempwsblock));
          DEBUG_MSG(("new temp block <%x>\n", (int)t));

          t->prev = tempworkspace;
        #ifdef CHECK_MALLOC
          t->magic1 = MAGIC1;
          t->magic2 = MAGIC2;
          check_tempblocks(tempworkspace);
        #endif
          tempworkspace=t;
          tempvecp = TEMPWORKSPACESIZE - n;
          tempblocks++;
        }
      return &(tempworkspace->ws[tempvecp]);
    }
  else
    {
    #ifdef DEBUG
      newvec_count++;
      newvec_size += n;
    #endif
      if (n > WORKSPACESIZE)
        {
        #if 0
          DEBUG_MSG(("newvec: out of normal space\n"));
          vtiabort(VTI_OUT_OF_SPACE, flocn);
        #endif
          return memalloc(n);
        }
      #if FREEUP
      if ((n < MAXFREEUPSIZE) && (freelist[n] != NULL))
        {
          void *ptr = freelist[n];
          freelist[n] = *(void **)ptr;
          /*DEBUG_MSG(("newvec(free):%u ", n));*/
          return ptr;
        }
      #endif
      DEBUG_MSG(("newvec:%u ", n));
      vecp -= n;
      if (vecp < 0)
        {
          struct wsblock *t = (struct wsblock *)
                              memalloc(sizeof(struct wsblock));
        #ifdef DEBUG
          newvec_malloc += sizeof(struct wsblock);
          DEBUG_MSG(("newblock <%x>\n", (int)t));
        #endif
        #if 0
          t->next = NULL;
        #endif
        #if defined(CHECK_MALLOC) || FREEUP
          t->prev = workspace;
        #endif
        #ifdef CHECK_MALLOC
          t->magic1 = MAGIC1;
          t->magic2 = MAGIC2;
          {
            struct wsblock *temp = workspace;
            while (temp != NULL)
              {
                assert(temp->magic1 == MAGIC1);
                assert(temp->magic2 == MAGIC2);
                temp = temp->prev;
              }
          }
        #endif
        #if 0
          if (workspace != NULL)
            workspace->next = t;
        #endif
          workspace=t;
          vecp = WORKSPACESIZE - n;
          blocks++;
        }
      return &(workspace->ws[vecp]);
    }
}
/*}}}*/
#if FREEUP
/*{{{  PUBLIC void freevec*/
PUBLIC void freevec(void *ptr, const size_t size)
{
  /*DEBUG_MSG(("freevec:%u ", size));*/
  if (size < MAXFREEUPSIZE)
    {
      *(void **)ptr = freelist[size];
      freelist[size] = ptr;
    }
}
/*}}}*/
#endif
/*{{{  markws() and freews*/
#if 0 /*defined(CONFIG)*/
#define MAXWSSTACK 10
struct wsmark
{
  struct wsblock *blocklist;
  int offset;
};
PRIVATE struct wsmark wsstack[MAXWSSTACK];
PRIVATE int wsstackptr;
PUBLIC void markws ( void )
{
  if (wsstackptr >= MAXWSSTACK)
    vtiabort(VTI_STACK_OVERFLOW, flocn);
  DEBUG_MSG(("Workspace mark <%x><%d>\n", (int)workspace, (int)vecp));
  wsstack[wsstackptr].blocklist = workspace;
  wsstack[wsstackptr].offset = vecp;
  wsstackptr++;
}
PUBLIC void freews ( void )
{
  wsstackptr--;
  if (wsstackptr < 0)
    vtiabort(VTI_STACK_UNDERFLOW, flocn);
  workspace = wsstack[wsstackptr].blocklist;
  vecp = wsstack[wsstackptr].offset;
  DEBUG_MSG(("freews: Workspace free <%x><%d>\n", (int)workspace, (int)vecp));
  if (workspace != NULL)
    {
      struct wsblock *wptr = workspace->next;
      while (wptr != NULL)
        {
          struct wsblock *t = wptr->next;
          DEBUG_MSG(("freed <%x>\n", (int)wptr));
          memfree(wptr);
          wptr = t;
        }
      workspace->next = NULL;
    }
}
#endif
/*}}}*/
/*{{{  switch_to_temp_workspace*/
PUBLIC int switch_to_temp_workspace ( void )
{
  const int old = tempworkspaceflag;
  tempworkspaceflag = TRUE;
  return old;
}
/*}}}*/
/*{{{  switch_to_real_workspace*/
PUBLIC int switch_to_real_workspace ( void )
{
  const int old = tempworkspaceflag;
  tempworkspaceflag = FALSE;
  return old;
}
/*}}}*/
/*{{{  switch_to_prev_workspace*/
PUBLIC void switch_to_prev_workspace ( const int old )
{
  tempworkspaceflag = old;
}
/*}}}*/
/*{{{  void freeup_temp_workspace*/
PUBLIC void freeup_temp_workspace (void)
{
  DEBUG_MSG(("freeup_temp_workspace"));
  switch_to_real_workspace();
#ifdef CHECK_MALLOC
  check_tempblocks(tempworkspace);
#endif
  while (tempworkspace != NULL)
    {
      struct tempwsblock *prev = tempworkspace->prev;
      DEBUG_MSG((", freeing <%lx>", (BIT32)tempworkspace));
      memfree(tempworkspace);
      tempworkspace = prev;
    }
  DEBUG_MSG(("\n"));
  tempvecp = 0; /* this forces another initial block to be created */
  tempblocks = 0;
  return;
}
/*}}}*/
#if FREEUP
/*{{{  void freeup_all_workspace*/
PUBLIC void freeup_all_workspace (void)
{
  int i;
  DEBUG_MSG(("freeup_all_workspace"));
  switch_to_real_workspace();
  while (workspace != NULL)
    {
      struct wsblock *prev = workspace->prev;
      DEBUG_MSG((", freeing <%lx>", (BIT32)workspace));
      memfree(workspace);
      workspace = prev;
    }
  DEBUG_MSG(("\n"));
  vecp = 0; /* this forces another initial block to be created */
  blocks = 0;

  for (i = 0; i < MAXFREEUPSIZE; i++)
    freelist[i] = NULL;

  return;
}
/*}}}*/
#endif
/*{{{  PUBLIC tablesize*/
PUBLIC long tablesize ( void )
{
  return (((long)WORKSPACESIZE     * (long)blocks    ) - (long)vecp    ) +
         (((long)TEMPWORKSPACESIZE * (long)tempblocks) - (long)tempvecp);
}
/*}}}*/

/*}}}*/

/*{{{  new treenode creation*/
/*{{{  PRIVATE void checknodetype*/
PRIVATE void checknodetype(const int tag, const int nodetype)
{
  if (nodetypeoftag(tag) != nodetype) badtag(NOPOSN, tag, "checknodetype");
}
/*}}}*/
/*{{{  treenode *newactionnode (t, ln, l, r)*/
treenode *newactionnode ( int t , SOURCEPOSN ln , treenode *l , treenode *r )
{
  treenode *nptr =  (treenode *)newvec(TREENODEBASE +
                                       sizeof (struct actionnode_s));
  checknodetype(t, ACTIONNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetLHS(nptr, l);
  SetRHS(nptr, r);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newaltnode (t, ln, g, i, b)*/
treenode *newaltnode ( int t , SOURCEPOSN ln , treenode *g , treenode *i , treenode *b )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct altnode_s));
  checknodetype(t, ALTNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetAltGuard(nptr, g);
  SetAltInput(nptr, i);
  SetAltBody(nptr, b);
  SetAltChanExp(nptr, NULL); /* added for bug 779 2/11/90 */
  SetAltLabel(nptr, -1);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newarraynode (t, ln, d, tp)*/
treenode *newarraynode ( int t , SOURCEPOSN ln , treenode *d , treenode *tp )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct arraynode_s));
  checknodetype(t, ARRAYNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetARDimLength(nptr, d);
  SetARType(nptr, tp);
  /* This field is filled in by chk if the dimension is known */
  SetARDim(nptr, -1);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newarraysubnode (t, ln, d, tp)*/
treenode *newarraysubnode ( int t , SOURCEPOSN ln , treenode *base , treenode *index )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct arraysubnode_s));
  checknodetype(t, ARRAYSUBNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetASBase(nptr, base);
  SetASIndex(nptr, index);
  SetASExp(nptr, NULL);
  SetASOffset(nptr, 0);
  SetASLength(nptr, NULL);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newchannode (t, ln, p)*/
treenode *newchannode ( int t , SOURCEPOSN ln , treenode *p )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct channode_s));
  checknodetype(t, CHANNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetProtocol(nptr, p);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newcnode (t, ln, b)*/
treenode *newcnode ( int t , SOURCEPOSN ln , treenode *b )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE + sizeof (struct cnode_s));
  checknodetype(t, CNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetCBody(nptr, b);
  SetCTemp(nptr, NULL);  /* Used in ALT constructors containing replicators */
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newcondnode (t, ln, condition, process)*/
treenode *newcondnode ( int t , SOURCEPOSN ln , treenode *condition , treenode *process )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct condnode_s));
  checknodetype(t, CONDNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetCondGuard(nptr, condition);
  SetCondBody(nptr, process);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
#ifdef CONDEXP
/*{{{  treenode *newcondexpnode (t, ln, condition, true, false)*/
treenode *newcondexpnode ( int t , SOURCEPOSN ln , treenode *condition , treenode *true , treenode *false )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct condexpnode_s));
  checknodetype(t, CONDEXPNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetCondExpGuard(nptr, condition);
  SetCondExpTrue(nptr, true);
  SetCondExpFalse(nptr, false);
  SetCondExpType(nptr, 0);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
#endif
#ifdef CONFIG
/*{{{  treenode *newconfignode*/
treenode *newconfignode ( int t , SOURCEPOSN ln , treenode *a, treenode *b, treenode *c)
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct confignode_s));
  checknodetype(t, CONFIGNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetSTDev(nptr, a);
  SetSTAttrName(nptr, b);
  SetSTAttrExp(nptr, c);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
#endif
/*{{{  treenode *newconstexpnode (t, ln, e, h, l)*/
treenode *newconstexpnode ( int t , SOURCEPOSN ln , treenode *e , BIT32 h , BIT32 l )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct constexpnode_s));
  checknodetype(t, CONSTEXPNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetCExp(nptr, e);
  SetHiVal(nptr, h);
  SetLoVal(nptr, l);
  SetCENext(nptr, NULL);
  SetCEOffset(nptr, -1);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *newconstant(BIT32 n)*/
PUBLIC treenode *newconstant(BIT32 n)
{
  return newconstexpnode(S_CONSTEXP, NOPOSN, dummyexp_p, ZERO32, n);
}
/*}}}*/
/*{{{  treenode *newconsttablenode (t, ln, v, e)*/
treenode *newconsttablenode ( int t , SOURCEPOSN ln , wordnode *v , treenode *e )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct consttablenode_s));
  checknodetype(t, CONSTTABLENODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetCTVal(nptr, v);
  SetCTExp(nptr, e);
  SetCTNext(nptr, NULL);
  SetCTLabel(nptr, -1);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newdeclnode (t, ln, n, v)*/
treenode *newdeclnode ( int t , SOURCEPOSN ln , treenode *n , treenode *val , treenode *b )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct declnode_s));
  checknodetype(t, DECLNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetDName(nptr, n);
  SetDVal(nptr, val);
  SetDBody(nptr, b);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newdopnode (t, ln, l, r)*/
treenode *newdopnode ( int t , SOURCEPOSN ln , treenode *l , treenode *r, int type )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct dopnode_s));
  checknodetype(t, DOPNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetLeftOp(nptr, l);
  SetRightOp(nptr, r);
  SetDOpType(nptr, type);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newhiddenparamnode (t, ln, e, d)*/
treenode *newhiddenparamnode ( int t , SOURCEPOSN ln , treenode *e , int d )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct hiddenparamnode_s));
  checknodetype(t, HIDDENPARAMNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetHExp(nptr, e);
  SetHDimension(nptr, d);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newinstancenode (t, ln, n, p)*/
treenode *newinstancenode ( int t , SOURCEPOSN ln , treenode *n , treenode *p )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct instancenode_s));
  checknodetype(t, INSTANCENODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetIName(nptr, n);
  SetIParamList(nptr, p);
  SetILoadSeq(nptr, 0);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newleafnode (t, ln)*/
treenode *newleafnode ( int t , SOURCEPOSN ln )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE);
  checknodetype(t, LEAFNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newlistnode (t, ln, l, r)*/
treenode *newlistnode ( int t , SOURCEPOSN ln , treenode *l , treenode *r )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct listnode_s));
  checknodetype(t, LISTNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetLeft(nptr, l);
  SetRight(nptr, r);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newlitnode (t, ln, w)*/
treenode *newlitnode ( int t , SOURCEPOSN ln , wordnode *w )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct litnode_s));
  checknodetype(t, LITNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetStringPtr(nptr, w);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newmopnode (t, ln, o)*/
treenode *newmopnode ( int t , SOURCEPOSN ln , treenode *o, int type )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct mopnode_s));
  checknodetype(t, MOPNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetOp(nptr, o);
  SetMOpType(nptr, type);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newnamenode(tag, ln, name, type, spec, lexlevel, scope, mode)*/
treenode *newnamenode ( int tag , SOURCEPOSN ln , wordnode *name , treenode *type , treenode *spec , int lexlevel , int scope , int mode )
{
  treenode *nptr = (treenode *)newvec (TREENODEBASE +
                                       sizeof(struct namenode_s));
  static struct namenode_s zeronode;
  checknodetype(tag, NAMENODE);
  nptr->n_u.n_s = zeronode;  /* Zero the miscellaneous bits just to be sure */
  SetTag(nptr, tag);
  SetLocn(nptr, ln);
  SetNName(nptr, name);
  SetNType(nptr, type);
  SetNDecl(nptr, spec);
  SetNLexLevel(nptr, lexlevel);
  SetNNestedPriPar(nptr, FALSE);
  SetNUsed(nptr, FALSE);
  SetNScope(nptr, scope);
  SetNMode(nptr, mode);
#ifdef DEBUG
  node_count[tag]++;
#endif
  return(nptr);
}
/*}}}*/
#if 0 /* never used */
/*{{{  treenode *newoverlapnode (t, ln, b1, c1, b2, c2)*/
PRIVATE treenode *newoverlapnode ( int t , SOURCEPOSN ln , treenode *b1 , treenode *c1 , treenode *b2 , treenode *c2 )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct overlapnode_s));
  checknodetype(t, OVERLAPNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetOBase1(nptr, b1);
  SetOCount1(nptr, c1);
  SetOBase2(nptr, b2);
  SetOCount2(nptr, c2);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
#endif /* 0 */
/*{{{  treenode *newprocessornode (t, ln, condition, process)*/
treenode *newprocessornode ( int t , SOURCEPOSN ln , treenode *exp , wordnode *type , treenode *process )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct processornode_s));
  checknodetype(t, PROCESSORNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetProcessorExp(nptr,exp);
  SetProcessorType(nptr,type);
  SetProcessorBody(nptr,process);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newreplcnode (t, ln, n, s, l, b)*/
treenode *newreplcnode ( int t , SOURCEPOSN ln , treenode *n , treenode *s , treenode *l , treenode *b )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct replcnode_s));
  checknodetype(t, REPLCNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetReplCName(nptr, n);
  SetReplCStartExp(nptr, s);
  SetReplCLengthExp(nptr, l);
  SetReplCBody(nptr, b);
  SetReplCTemp(nptr, NULL);  /* Used in ALT constructors containing replicators */
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newsegmentnode (t, ln, n, s, l)*/
treenode *newsegmentnode ( int t , SOURCEPOSN ln , treenode *n , treenode *s , treenode *l )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct segmentnode_s));
  checknodetype(t, SEGMENTNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetSName(nptr, n);
  SetSStartExp(nptr, s);
  SetSLengthExp(nptr, l);
  SetSSubscriptExp(nptr, NULL);
  SetSCheckExp(nptr, NULL);
  SetSLength(nptr, NULL);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newspacenode (t, ln, p, maxwsp, datasize,... )*/
treenode *newspacenode ( int t , SOURCEPOSN ln , treenode *p , BIT32 maxwsp , BIT32 datasize , BIT32 vsusage , BIT32 nestedvs , treenode *namechain , int cpoffset )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct spacenode_s));
  checknodetype(t, SPACENODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetSpBody(nptr, p);
  SetSpMaxwsp(nptr, maxwsp);
  SetSpDatasize(nptr, datasize);
  SetSpVSUsage(nptr, vsusage);
  SetSpNestedVS(nptr, nestedvs);
  SetSpNamechain(nptr, namechain);
  SetSpCPOffset(nptr, cpoffset);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newvalofnode (t, ln, b, r)*/
treenode *newvalofnode ( int t , SOURCEPOSN ln , treenode *b , treenode *r )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct valofnode_s));
  checknodetype(t, VALOFNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetVLBody(nptr, b);
  SetVLResultList(nptr, r);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/
/*{{{  treenode *newvariantnode (t, ln, tl, b)*/
treenode *newvariantnode ( int t , SOURCEPOSN ln , treenode *tl , treenode *b )
{
  treenode *nptr = (treenode *)newvec(TREENODEBASE +
                                      sizeof (struct variantnode_s));
  checknodetype(t, VARIANTNODE);
  SetTag(nptr, t);
  SetLocn(nptr, ln);
  SetVRTaggedList(nptr, tl);
  SetVRBody(nptr, b);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/

/*{{{  wordnode *newwordnode (t, name, len, next)*/
wordnode *newwordnode ( int t , char *name , int len , wordnode *next )
{
  wordnode *nptr = (wordnode *) newvec(sizeof(wordnode));
  checknodetype(t, WORDNODE);
  SetWTag(nptr, t);
  SetWName(nptr, name);
  SetWLength(nptr, len);
  SetWNext(nptr, next);
#ifdef DEBUG
  node_count[t]++;
#endif
  return (nptr);
}
/*}}}*/

/*{{{  PRIVATE void print_node_count()*/
#ifdef NOTHING_AT_ALL /*DEBUG*/
PRIVATE void print_node_count ( void )
{
  int i, tot=0;
  for (i=0; i < 256; i++)
    {
      tot += node_count[i];
      if (node_count[i]) printf("%-16s - %d\n", itagstring(i), node_count[i]);
    }
  printf("Total new nodes: %d\n", tot);
}
#endif /* DEBUG */
/*}}}*/
/*}}}*/
/*{{{  tree accessing*/
/*{{{  treenode *NParamListOf(n)*/
treenode *NParamListOf ( treenode *n )
{
  switch (TagOf(n))
    {
      case N_PROCDEF: case N_SCPROCDEF: case N_LIBPROCDEF:
      case N_STDLIBPROCDEF: case N_INLINEPROCDEF:
        return(NTypeOf(n));
      default:
        return(FnParamsOf(NTypeOf(n)));
    }
}
/*}}}*/
/*{{{  void SetNParamList(n, v)*/
void SetNParamList ( treenode *n , treenode *v )
{
  switch (TagOf(n))
    {
      case N_PROCDEF: case N_SCPROCDEF: case N_LIBPROCDEF:
      case N_STDLIBPROCDEF: case N_INLINEPROCDEF:
        SetNType(n, v);
        break;
      default:
        SetFnParams(NTypeOf(n), v);
        break;
    }
}
/*}}}*/
/*}}}*/


/*{{{  PUBLIC void vtiinit*/
PUBLIC void vtiinit ( void )
{
#if 0 /*defined(CONFIG)*/
  wsstackptr = 0;
#endif
  initcopytree();
  tempworkspace = NULL;
  tempvecp = 0; /* this forces an initial block to be created */
  tempblocks = 0;
  vecp = 0;     /* this forces an initial block to be created */
  blocks = 0;
  switch_to_real_workspace();
}
/*}}}*/
/*{{{  PUBLIC void vtifinish*/
PUBLIC void vtifinish ( void )
{
#ifdef DEBUG
  fprintf(outfile,"VTIP diagnostics:\n");
  fprintf(outfile,"Total number of calls to malloc: %d, total calls to free: %d",
         malloc_count, free_count);
  fprintf(outfile," (Diff is %d)\n", malloc_count - free_count);
  fprintf(outfile,"Total size of small mallocs (< 500): %d, total size of large mallocs: %d\n",
         small_malloc, large_malloc);
  fprintf(outfile,"Total malloc-ed size: %d, tree size: %ld",
                  small_malloc + large_malloc, tablesize());
#ifdef IMS
  fprintf(outfile," (actual left is %u)", total_malloc);
#endif
  fprintf(outfile,"\nThere were %d calls to newvec, for a total of %d bytes\n",
         newvec_count, newvec_size);
  fprintf(outfile,"(or %d from malloc). Plus %d temp bytes\n", newvec_malloc, newvec_temp);
#ifdef IMS
  fprintf(outfile,"Max stack usage is: %ld\n", max_stack_usage());
#endif /* IMS */
#if 0 /*defined(SUN) || defined(GNU)*/
  fprintf(outfile,"mallocmap:\n");
  mallocmap();
#endif
  /*print_node_count();*/
#endif /* DEBUG */

  print_memstats();

#if FREEUP && defined(DEBUG)
  {
    int i;
    for (i = 0; i < MAXFREEUPSIZE; i++)
      {
        int count = 0;
        void **p;
        for (p = (void **)freelist[i]; p != NULL; p = *p)
          count++;
        if (count != 0)
          fprintf(outfile, "freelist[%d]:%d(%d) ", i, count, i*count);
      }
    fputc('\n', outfile);
  }
#endif /* FREEUP */
}
/*}}}*/
