#define DEBUG
/*****************************************************************************
 *
 *  use1 - routines to support usage and alias checkers
 *
 *****************************************************************************/
/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987, 1988
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include "includes.h"
# include "useerror.h"
# include "lexconst.h"
# include "predefhd.h"
# include "usehdr.h"
# include "use1def.h"
# include "use2def.h"
# include "usedef.h"
# include "chkerror.h"
# include "chkdef.h"
/*}}}*/
/*{{{  static variables*/
PRIVATE subscrlist *subscr_free = NULL;
PRIVATE varlist    *vlist_free  = NULL;
PRIVATE abbrevlist *alist_free  = NULL;
/*}}}*/
/*{{{  global variables*/
PUBLIC errorlist *uerrors = NULL;
PUBLIC int warn_on_usage_error = FALSE;
/*}}}*/

/*{{{  DEBUG_MSG*/
#undef DEBUG_MSG
#ifdef DEBUG
  #define DEBUG_MSG(X) { if (debuguse) printf X; }
#else
  #define DEBUG_MSG(X)
#endif
/*}}}*/

/*{{{  PUBLIC void usereport (n, locn, p1)*/
/*****************************************************************************
 *
 *  usereport adds an error report to the error list
 *
 *****************************************************************************/
PUBLIC void usereport ( int n , SOURCEPOSN locn , treenode *p1)
{
  errorlist *ptr, **prev;
  prev = &uerrors;
  ptr = *prev;
  /*{{{  search for first error report on same line*/
  while ((ptr != NULL) && (ERLocnOf(ptr) < locn))
    {
      prev = &ERNextOf(ptr);
      ptr = *prev;
    }
  /*}}}*/
  /*{{{  check error has not been reported before - return if it has*/
  while ((ptr != NULL) && (ERLocnOf(ptr) == locn))
    {
      if ((ERCodeOf(ptr) == n) && (ERP1Of(ptr) == p1))
        return;   /* already reported */
      prev = &ERNextOf(ptr);
      ptr = *prev;
    }
  /*}}}*/
  /*{{{  insert new error record*/
  DEBUG_MSG(("Error is generated now\n"));
  *prev = (errorlist *) newvec(sizeof(errorlist));
  SetERNext(*prev, ptr);
  SetERCode(*prev, n);
  SetERLocn(*prev, locn);
  SetERP1(*prev, p1);
  /*}}}*/
}
/*}}}*/
#ifdef DEBUG
/*{{{  PUBLIC char *usetagstring (n)*/
/*****************************************************************************
 *
 *  return ascii string for specified tag
 *
 *****************************************************************************/
PUBLIC char *usetagstring ( treenode *n )
{
  static char tagstring_s[MAXSTRING_SIZE];
  treenode *name;
  switch (nodetypeoftag(TagOf(n)))
    {
      default: return itagstring(TagOf(n));
      /*{{{  cases*/
      case DECLNODE:
        if ((TagOf(n) == S_DECL) && (TagOf(DNameOf(n)) == S_LIST))
          name = ThisItem(DNameOf(n));
        else
          name = DNameOf(n);
        break;
      case INSTANCENODE:
        name = INameOf(n);
        break;
      case REPLCNODE:
        name = ReplCNameOf(n);
        break;
      case NAMENODE:
        name = n;
        break;
      /*}}}*/
    }
  sprintf(tagstring_s, "%s %s", itagstring(TagOf(n)), WNameOf(NNameOf(name)));
  return tagstring_s;
}
/*}}}*/
#endif
/*{{{  PRIVATE void printsublist (slist, name)*/
/*****************************************************************************
 *
 *  print list of subscripts accessed by free variable
 *
 *****************************************************************************/
PRIVATE void printsublist ( subscrlist *slist , char *name )
{
  if (slist != NULL)
    {
      fprintf(outfile, "%s: ", name);
      while (slist != NULL)
        {
          if ((SLFirstOf(slist) == 0) && (SLLastOf(slist) == MAXINDEX))
            fprintf(outfile, "all ");
          else
            fprintf(outfile, "%ld-%ld ", SLFirstOf(slist), SLLastOf(slist));
          slist = SLNextOf(slist);
        }
    }
}
/*}}}*/
/*{{{  PUBLIC void printfreelist (list)*/
/*****************************************************************************
 *
 *  Print the free variable list
 *
 *****************************************************************************/
PUBLIC void printfreelist ( varlist *list )
{
  if (list == NULL)
    fprintf(outfile, "(none)");
  while (list != NULL)
    {
      fprintf(outfile, "'%s' (", WNameOf(NNameOf(VLNameOf(list))) );
      printsublist(VLReadOf(list),    "read");
      printsublist(VLWrittenOf(list), "written");
      printsublist(VLInputOf(list),   "input");
      printsublist(VLOutputOf(list),  "output");
      fprintf(outfile, ")");
      list = VLNextOf(list);
    }
  fprintf(outfile, "\n");
}
/*}}}*/
/*{{{  PUBLIC int is_free (n, scope)*/
/*****************************************************************************
 *
 *  is_free returns TRUE if the name 'n' is declared outside of the
 *  scope 'scope'.
 *
 *****************************************************************************/
PUBLIC int is_free ( treenode *n , int scope )
{
  return((scope < 0) || (NScopeOf(n) < scope));
}
/*}}}*/
/*{{{  PUBLIC subscrlist *allocate_subscr ()*/
/*****************************************************************************
 *
 *  Allocate a record for storing subscript info
 *  Recycle any previously used records if available
 *
 *****************************************************************************/
PUBLIC subscrlist *allocate_subscr ( void )
{
  subscrlist *ptr;
  if (subscr_free == NULL)
    ptr = (subscrlist *) newvec(sizeof(subscrlist));
  else
    {
      ptr = subscr_free;
      subscr_free = SLNextOf(subscr_free);
    }
  return(ptr);
}
/*}}}*/
/*{{{  PUBLIC void release_subscr (subscr_node)*/
/*****************************************************************************
 *
 *  Return subscript info record to free list
 *
 *****************************************************************************/
PUBLIC void release_subscr ( subscrlist *subscr_node )
{
  SetSLNext(subscr_node, subscr_free);
  subscr_free = subscr_node;
}
/*}}}*/
/*{{{  PUBLIC subscrlist *add_subscr_rec (start, end, link)*/
/*****************************************************************************
 *
 *  Allocate and initialise a new subscript info record
 *
 *****************************************************************************/
PUBLIC subscrlist *add_subscr_rec ( INT32 start , INT32 end , subscrlist *link )
{
  subscrlist *p;
  p = allocate_subscr();
  SetSLFirst(p, start);
  SetSLLast(p, end);
  SetSLNext(p, link);
  return(p);
}
/*}}}*/
/*{{{  PUBLIC void add_subscr (vptr, start, end)*/
/*****************************************************************************
 *
 *  Add new accesses from 'start' to 'end' to subscript access list 'vptr'
 *
 *****************************************************************************/
PUBLIC void add_subscr ( subscrlist **vptr , INT32 start , INT32 end )
{
  subscrlist **prev, *ptr;
  prev = vptr;
  ptr = *prev;
  while (TRUE)
    {
      if (ptr == NULL)
        /*{{{  add new record and return*/
        {
          *prev = add_subscr_rec(start, end, NULL);
          return;
        }
        /*}}}*/
      else if (end < (SLFirstOf(ptr) - 1))
        /*{{{  add new record and return*/
        {
          *prev = add_subscr_rec(start, end, ptr);
          return;
        }
        /*}}}*/
      else if (end <= SLLastOf(ptr))
        /*{{{  update first and return*/
        {
          if (start < SLFirstOf(ptr))
            SetSLFirst(ptr, start);
          return;
        }
        /*}}}*/
      else if (start <= (SLLastOf(ptr) + 1))
        /*{{{  update list and return*/
        {
          if (start < SLFirstOf(ptr))
            SetSLFirst(ptr, start);
          SetSLLast(ptr, end);
          while ((SLNextOf(ptr) != NULL) &&
                (end >= (SLFirstOf(SLNextOf(ptr)) - 1)))
            {
              subscrlist *next;
              INT32 next_end;
              next = SLNextOf(ptr);
              next_end = SLLastOf(next);
              if (end < next_end)
                SetSLLast(ptr, next_end);
              SetSLNext(ptr, SLNextOf(next));
              release_subscr(next);
            }
          return;
        }
        /*}}}*/
      else
        /*{{{  loop*/
        {
          prev = &SLNextOf(ptr);
          ptr = *prev;
        }
        /*}}}*/
    }
}
/*}}}*/
/*{{{  PUBLIC int subscr_overlaps (sptr, start, end)*/
/*****************************************************************************
 *
 *  Test if the range 'start' to 'end' overlaps any subscript accesses in
 *  the list 'sptr'
 *
 *****************************************************************************/
PUBLIC int subscr_overlaps ( subscrlist *sptr , INT32 start , INT32 end )
{
  while (sptr != NULL)
    {
      if (end < SLFirstOf(sptr))
        return(FALSE);
      else if (start <= SLLastOf(sptr))
        return(TRUE);
      else
        sptr = SLNextOf(sptr);
    }
  return(FALSE);
}
/*}}}*/
/*{{{  PUBLIC subsclist *merge_subscr_info (from_list, to_list)*/
/*****************************************************************************
 *
 *  Merge the accesses in 'from_list' into 'to_list' and return the new list
 *  formed.  'from_list' is not affected.
 *
 *****************************************************************************/
PUBLIC subscrlist *merge_subscr_info ( subscrlist *from_list , subscrlist *to_list )
{
  while (from_list != NULL)
    {
      add_subscr(&to_list, SLFirstOf(from_list), SLLastOf(from_list));
      from_list = SLNextOf(from_list);
    }
  return(to_list);
}
/*}}}*/
/*{{{  PUBLIC void release_subscripts (subscr)*/
/*****************************************************************************
 *
 *  Return all subscript info records in the list 'subscr' to free list
 *
 *****************************************************************************/
PUBLIC void release_subscripts ( subscrlist *subscr )
{
  while (subscr != NULL)
    {
      subscrlist *next_subscr = SLNextOf(subscr);
      release_subscr(subscr);
      subscr = next_subscr;
    }
}
/*}}}*/
/*{{{  PUBLIC abbrevlist *new_abbrev_node (n, next, first, last, loc, ...)*/
/*****************************************************************************
 *
 *  Allocate a record for storing abbreviation info
 *  Recycle any previously used records if available
 *  Initialise the record to the values given
 *
 *****************************************************************************/
PUBLIC abbrevlist *new_abbrev_node ( treenode *n , abbrevlist *next , INT32 first , INT32 last , SOURCEPOSN locn , treenode *subscripts )
{
  abbrevlist *list;
  if (alist_free == NULL)
    list = (abbrevlist *)newvec(sizeof(abbrevlist));
  else
    {
      list = alist_free;
      alist_free = ALNextOf(alist_free);
    }
  SetALNext(list, next);
  SetALName(list, n);
  SetALLoc(list, locn);
  SetALFirst(list, first);
  SetALLast(list, last);
  SetALSubscripts(list, subscripts);
  return(list);
}
/*}}}*/
/*{{{  PUBLIC abbrevlist *release_headof_abbrevlist (alist)*/
/*****************************************************************************
 *
 *  Return the first abbreviation info record in 'alist' to the free list
 *  and return the next element in the list to the caller
 *
 *****************************************************************************/
PUBLIC abbrevlist *release_headof_abbrevlist ( abbrevlist *alist )
{
  abbrevlist *next;
  next = ALNextOf(alist);
  SetALNext(alist, alist_free);
  alist_free = alist;
  return(next);
}
/*}}}*/
/*{{{  PUBLIC int inabbrevlist (n, alist, start, end)*/
/*****************************************************************************
 *
 *  inabbrevlist returns TRUE if the namenode 'n' is already in the abbreviated
 *  variable list 'alist'.
 *
 *****************************************************************************/
PUBLIC int inabbrevlist ( treenode *n , abbrevlist *alist , INT32 start , INT32 end )
{
  while (alist != NULL)
    {
      if (ALNameOf(alist) == n)
        {
          if ((ALFirstOf(alist) < 0) ||
             (start < 0) ||
             (((start <= ALLastOf(alist)) && (end >= (ALFirstOf(alist))))))
            return(TRUE);
        }
      alist = ALNextOf(alist);
    }
  return(FALSE);
}
/*}}}*/
/*{{{  PUBLIC varlist *new_list_node (n, next)*/
/*****************************************************************************
 *
 *  Allocate a record for storing variable access info
 *  Recycle any previously used records if available
 *  Initialise the record
 *
 *****************************************************************************/
PUBLIC varlist *new_list_node ( treenode *n , varlist *next )
{
  varlist *list;
  if (vlist_free == NULL)
    list = (varlist *) newvec(sizeof(varlist));
  else
    {
      list = vlist_free;
      vlist_free = VLNextOf(vlist_free);
    }
  SetVLName(list, n);
  SetVLNext(list, next);
  SetVLRead(list, NULL);
  SetVLWritten(list, NULL);
  SetVLInput(list, NULL);
  SetVLOutput(list, NULL);
  return(list);
}
/*}}}*/
/*{{{  PUBLIC int invarlist (n, vlist, use_mode, start, end)*/
/*****************************************************************************
 *
 *  invarlist returns TRUE if the namenode 'n' is already in the free variable
 *  list 'vlist'.
 *
 *****************************************************************************/
PUBLIC int invarlist ( treenode *n , varlist *vlist , int use_mode , INT32 start , INT32 end )
{
  while (vlist != NULL)
    {
      if (VLNameOf(vlist) == n)
        {
          subscrlist *s;
          switch(use_mode)
            {
              case EXP_READ:    s = VLReadOf(vlist);    break;
              case EXP_WRITTEN: s = VLWrittenOf(vlist); break;
              case CHAN_INPUT:  s = VLInputOf(vlist);   break;
              case CHAN_OUTPUT: s = VLOutputOf(vlist);  break;
            }
          return (subscr_overlaps(s, start, end));
        }
      vlist = VLNextOf(vlist);
    }
  return(FALSE);
}
/*}}}*/
/*{{{  PUBLIC varlist *addtolist (n, vlist, use_mode, start_sub, end_sub)*/
/*****************************************************************************
 *
 *  addtolist adds the name 'n' with access range 'start_sub' to 'end_sub'
 *  and usage mode 'use_mode' to 'vlist' and returns the new list.
 *
 *****************************************************************************/
PUBLIC varlist *addtolist ( treenode *n , varlist *vlist , int use_mode , INT32 start_sub , INT32 end_sub )
{
  varlist *list;
#ifdef DEBUG
  if (debuguse)
    {
      fprintf(outfile, "Adding %s[", WNameOf(NNameOf(n)));
      if (start_sub == 0 && end_sub == MAXINDEX)
        fputs("all", outfile);
      else
        fprintf(outfile, "%ld-%ld", start_sub, end_sub);
      fputs("] to list ", outfile);
      switch (use_mode)
        {
          case EXP_READ:    fputs("EXP_READ\n", outfile); break;
          case EXP_WRITTEN: fputs("EXP_WRITTEN\n", outfile); break;
          case CHAN_INPUT:  fputs("CHAN_INPUT\n", outfile); break;
          case CHAN_OUTPUT: fputs("CHAN_OUTPUT\n", outfile); break;
        }
    }
#endif
  list = vlist;
  while (list != NULL)
    {
      if (VLNameOf(list) == n)
        {
          switch (use_mode)
            {
              case EXP_READ:
                add_subscr(&VLReadOf(list), start_sub, end_sub); break;
              case EXP_WRITTEN:
                add_subscr(&VLWrittenOf(list), start_sub, end_sub); break;
              case CHAN_INPUT:
                add_subscr(&VLInputOf(list), start_sub, end_sub); break;
              case CHAN_OUTPUT:
                add_subscr(&VLOutputOf(list), start_sub, end_sub); break;
            }
          return(vlist);
        }
      list = VLNextOf(list);
    }
  list = new_list_node(n, vlist);
  switch (use_mode)
    {
      case EXP_READ:
        add_subscr(&VLReadOf(list), start_sub, end_sub); break;
      case EXP_WRITTEN:
        add_subscr(&VLWrittenOf(list), start_sub, end_sub); break;
      case CHAN_INPUT:
        add_subscr(&VLInputOf(list), start_sub, end_sub); break;
      case CHAN_OUTPUT:
        add_subscr(&VLOutputOf(list), start_sub, end_sub); break;
    }
  return(list);
}
/*}}}*/
/*{{{  PUBLIC varlist *release_headof_varlist (vlist)*/
/*****************************************************************************
 *
 *  Return first record on variable access list 'vlist' to the free pool
 *  and return next item on list to the caller
 *
 *****************************************************************************/
PUBLIC varlist *release_headof_varlist ( varlist *vlist )
{
  varlist *next = VLNextOf(vlist);
  release_subscripts(VLReadOf(vlist));
  release_subscripts(VLWrittenOf(vlist));
  release_subscripts(VLInputOf(vlist));
  release_subscripts(VLOutputOf(vlist));
  SetVLNext(vlist, vlist_free);
  vlist_free = vlist;
  return(next);
}
/*}}}*/
/*{{{  PUBLIC void release_varlist (vlist)*/
/*****************************************************************************
 *
 *  Release all records in variable access list 'vlist'
 *
 *****************************************************************************/
PUBLIC void release_varlist ( varlist *vlist )
{
  while (vlist != NULL)
    vlist = release_headof_varlist(vlist);
}
/*}}}*/
/*{{{  PUBLIC varlist *merge (vlist1, scope, vlist2)*/
/*****************************************************************************
 *
 *  merge takes two lists of free variables, 'vlist1' and 'vlist2' and a scope
 *  'scope'.  It adds to the front of 'vlist2' all those variables in 'vlist1'
 *  which are not already in 'vlist2' and are outside 'scope'.
 *  The resultant list is returned.
 *  'vlist1' is not changed.
 *
 *****************************************************************************/
PUBLIC varlist *merge ( varlist *vlist1 , int scope , varlist *vlist2 )
{
  while (vlist1 != NULL)
    {
      treenode *n = VLNameOf(vlist1);
      if (is_free(n, scope))
        {
          varlist *list = vlist2;
          while ((list != NULL) && (VLNameOf(list) != n))
            list = VLNextOf(list);
          if (list == NULL)
            {
              vlist2 = new_list_node(n, vlist2);
              list = vlist2;
            }
          SetVLRead(list,    merge_subscr_info(VLReadOf(vlist1),
                                               VLReadOf(list)));
          SetVLWritten(list, merge_subscr_info(VLWrittenOf(vlist1),
                                               VLWrittenOf(list)));
          SetVLInput(list,   merge_subscr_info(VLInputOf(vlist1),
                                               VLInputOf(list)));
          SetVLOutput(list,  merge_subscr_info(VLOutputOf(vlist1),
                                               VLOutputOf(list)));
        }
      vlist1 = VLNextOf(vlist1);
    }
  return(vlist2);
}
/*}}}*/
/*{{{  PUBLIC void subscripts_accessed (n, first, last)*/
/*****************************************************************************
 *
 *  'n' points to an expression tree containing only subscripts and slices of
 *  an element with compile-time evaluable subscripts.
 *  'first' is set to the subscript of the first accessed element of the
 *  array treated as a single dimensional array and 'last' is set to the last.
 *  if check_subscripts is TRUE, it will report subscript errors
 *  (incase of expanded replicators, etc).
 *
 *****************************************************************************/
PUBLIC void subscripts_accessed ( treenode *n , INT32 *first , INT32 *last,
                                  int check_subscripts )
{
  switch (TagOf(n))
    {
      case S_ARRAYSUB:
        /*{{{  subscript*/
        {
          subscripts_accessed(ASBaseOf(n), first, last, check_subscripts);
          if (*first < 0)
            ; /* skip */
          else if (!is_evaluable(ASIndexOf(n)))
            *first = (-1);
          else
            {
              INT32 subscript = evaluate(ASIndexOf(n));
              treenode *type  = gettype(ASBaseOf(n));
              DEBUG_MSG(("subscripts_accessed: ARRAYSUB; first=%ld, last=%ld, subs=%ld, ARDIM=%ld\n",
                         *first, *last, subscript, ARDimOf(type)));
              if (check_subscripts && /* added 16/10/90 for bug 1017 */
                  ((subscript < 0) || (subscript >= ARDimOf(type))))
                {
                  *first = (-1);
                  if (check_subscripts)
                    chkerr_i(CHK_SUBSCRIPT_RANGE, LocnOf(n), subscript);
                }
              else
                {
                  INT32 size = elementsin(gettype(n));
                  if (size == (-1))
                    *first = (-1);
                  else if (ARDimOf(type) < 0) /* added 16/10/90 for bug 1017 */
                    {
                      *first += (size * subscript); *last = *first + size - 1;
                    }
                  else
                    {
                      INT32 start  = *first + (size * subscript);
                      INT32 length = ((*last + 1) - *first) / ARDimOf(type);
                      *first = start;
                      *last = start + (length - 1);
                    }
                }
            }
          break;
        }
        /*}}}*/
      case S_SEGMENT:
        /*{{{  segment*/
        {
          subscripts_accessed(SNameOf(n), first, last, check_subscripts);
          if (*first < 0)
            ; /* skip */
          else if (!is_evaluable(SStartExpOf(n)) || !is_evaluable(SLengthExpOf(n)))
            *first = (-1);
          else
            {
              INT32 start    = evaluate(SStartExpOf(n));
              INT32 slength  = evaluate(SLengthExpOf(n));
              treenode *type = gettype(SNameOf(n));
              if (check_subscripts && /* bug 1017 16/10/90 */
                  ((start < 0) || (slength < 0) || ((start + slength) > ARDimOf(type))))
                {
                  *first = (-1);
                  if (check_subscripts)
                    {
                      if ((start < 0) || (start > ARDimOf(type)))
                        chkerr_i(CHK_SEG_START_RANGE, LocnOf(n), start);
                      else if ((slength < 0) || (slength > ARDimOf(type)))
                        chkerr_i(CHK_SEG_LENGTH_RANGE, LocnOf(n), slength);
                      else
                        chkerr_i(CHK_SEG_RANGE, LocnOf(n), start+slength);
                    }
                }
              else if (ARDimOf(type) < 0) /* added 16/10/90 bug 1017 */
                {
                  *first += start;
                  *last = *first + slength - 1; /* quick and dirty approximation */
                }
              else
                {
                  INT32 size = elementsin(type);
                  if (size == (-1))
                    *first = (-1);
                  else
                    {
                      INT32 base   = *first + ((size / ARDimOf(type)) * start);
                      INT32 length = (((*last + 1) - *first) / ARDimOf(type)) * slength;
                      *first = base;
                      *last = base + (length - 1);
                    }
                }
            }
          break;
        }
        /*}}}*/
      #ifdef CONFIG
      case S_RECORDSUB:
        *first = 0;
        *last  = elementsin(NTypeOf(ASIndexOf(n))) - 1;
        break;
      #endif
      default:
        /*{{{  decl node*/
        *first = 0;
        *last  = elementsin(NTypeOf(n)) - 1;
        if (*last < 0)
          *last = MAXINDEX;
        break;
        /*}}}*/
    }
  DEBUG_MSG(("subscripts_accessed: %s, first:%ld, last:%ld\n", usetagstring(n), *first, *last));
}
/*}}}*/
/*{{{  PUBLIC int name_used_in_exp*/
PRIVATE int       name_used_in_exp_result;
PRIVATE treenode *name_used_in_exp_name;
PRIVATEPARAM int do_name_used_in_exp(treenode *tptr)
{
  if ((nodetypeoftag(TagOf(tptr)) == NAMENODE) &&
      (tptr == name_used_in_exp_name))
    {
      name_used_in_exp_result = TRUE;
      return STOP_WALK;
    }
  return CONTINUE_WALK;
}
PUBLIC int name_used_in_exp(treenode *name, treenode *tptr)
{
  name_used_in_exp_name = name;
  name_used_in_exp_result = FALSE;
  prewalktree(tptr, do_name_used_in_exp);
  return name_used_in_exp_result;
}
/*}}}*/
/*{{{  PUBLIC int chan_use (n, name)*/
PRIVATE treenode *chan_use_name;
PRIVATE int chan_use_result;
/*{{{  PRIVATEPARAM int dochan_use (n)*/
/*****************************************************************************
 *
 *  searches tree 'n' for uses of channel 'name' and returns a value indicating
 *  whether it is unused, used for input, used for output or both
 *
 *****************************************************************************/
PRIVATEPARAM int dochan_use ( treenode *n )
{
  DEBUG_MSG(("dochan_use: %s\n", usetagstring(n)));
  switch(TagOf(n))
    {
      /*{{{  PINSTANCE*/
      case S_PINSTANCE:
        {
          treenode *flist, *formal;
          /*{{{  check free vars*/
          if ((TagOf(INameOf(n)) == N_PROCDEF) || (TagOf(INameOf(n)) == N_INLINEPROCDEF))
            {
              varlist *freevars = NFreeVarsOf(INameOf(n));
              while ((freevars != NULL) && (VLNameOf(freevars) != chan_use_name))
                freevars = VLNextOf(freevars);
              if (freevars != NULL)
                {
                  if (VLInputOf(freevars) != NULL)
                    chan_use_result |= CHAN_USE_INPUT;
                  if (VLOutputOf(freevars) != NULL)
                    chan_use_result |= CHAN_USE_OUTPUT;
                }
            }
          /*}}}*/
          if (TagOf(INameOf(n)) == N_PREDEFPROC) /* bug 1035 1/11/90 */
            /*{{{  check params*/
            {
              int pdno = NModeOf(INameOf(n));
              if (pdno == PD_LOADINPUTCHANNEL  || pdno == PD_LOADINPUTCHANNELVECTOR ||
                  pdno == PD_LOADOUTPUTCHANNEL || pdno == PD_LOADOUTPUTCHANNELVECTOR)
                {
                  int usemode = (pdno == PD_LOADINPUTCHANNEL || pdno == PD_LOADINPUTCHANNELVECTOR) ?
                                 CHAN_USE_INPUT : CHAN_USE_OUTPUT;
                  flist = NTypeOf(INameOf(n));
                  n = IParamListOf(n);
                  while (n != NULL && chan_use_result != CHAN_USE_BIDIRECTIONAL)
                    {
                      formal = ThisItem(flist);
                      if ((TagOf(formal) == N_PARAM) && (nameof(ThisItem(n)) == chan_use_name))
                        chan_use_result |= usemode;
                      flist = NextItem(flist);
                      n = NextItem(n);
                    }
                }
             }
            /*}}}*/
          else
            /*{{{  check parameters*/
            {
              flist = NTypeOf(INameOf(n));
              n = IParamListOf(n);
              while (!EndOfList(n))
                {
                  formal = ThisItem(flist);
                  if ((TagOf(formal) == N_PARAM) && (nameof(ThisItem(n)) == chan_use_name))
                    {
                      if (ParamInputOn(formal))
                        chan_use_result |= CHAN_USE_INPUT;
                      if (ParamOutputOn(formal))
                        chan_use_result |= CHAN_USE_OUTPUT;
                    }
                  flist = NextItem(flist);
                  n = NextItem(n);
                }
            }
            /*}}}*/
          return STOP_WALK;
        }
      /*}}}*/
      /*{{{  output*/
      case S_OUTPUT:
        if (nameof(LHSOf(n)) == chan_use_name)
          chan_use_result |= CHAN_USE_OUTPUT;
        return STOP_WALK;
      /*}}}*/
      /*{{{  input, taggedinput, delayedinput*/
      case S_INPUT: case S_TAGGED_INPUT: case S_DELAYED_INPUT:
        if (nameof(LHSOf(n)) == chan_use_name)
            chan_use_result |= CHAN_USE_INPUT;
        return STOP_WALK;
      /*}}}*/
      /*{{{  caseinput*/
      case S_CASE_INPUT:
        if (nameof(LHSOf(n)) == chan_use_name)
          chan_use_result |= CHAN_USE_INPUT;
        break;
      /*}}}*/
      /*{{{  abbr*/
      case S_ABBR:
      #if 0 /* this doesn't work if rhs is a channel constructor: bug 685 3/9/90 */
        if (nameof(DValOf(n)) == chan_use_name)
          {
            treenode *oldname = chan_use_name;
            chan_use_name = DNameOf(n);
            prewalkproctree(DBodyOf(n), dochan_use);
            chan_use_name = oldname;
          }
      #else
        {
          int type = basetype(NTypeOf(DNameOf(n)));
          if (((type == S_CHAN) || (type == S_PORT)) &&
            name_used_in_exp(chan_use_name, DValOf(n)))
            {
              treenode *oldname = chan_use_name;
              chan_use_name = DNameOf(n);
              prewalkproctree(DBodyOf(n), dochan_use);
              chan_use_name = oldname;
            }
        }
      #endif
        n = DBodyOf(n);
        break;
      /*}}}*/
    }
  return (chan_use_result == CHAN_USE_BIDIRECTIONAL) ? STOP_WALK : CONTINUE_WALK;
}
/*}}}*/
/*{{{  PUBLIC int chan_use (n, name)*/
/*****************************************************************************
 *
 *  searches tree 'n' for uses of channel 'name' and returns a value indicating
 *  whether it is unused, used for input, used for output or both
 *
 *****************************************************************************/
PUBLIC int chan_use ( treenode *n , treenode *name )
{
  chan_use_name = name;
  chan_use_result = CHAN_UNUSED;
  prewalkproctree(n, dochan_use);
  DEBUG_MSG(("chan_use returns: in?:%d, out?:%d\n",
             (chan_use_result & CHAN_USE_INPUT) != 0,
             (chan_use_result & CHAN_USE_OUTPUT) != 0));
  return chan_use_result;
}
/*}}}*/
/*}}}*/
/*{{{  free vars procedures*/
FORWARD PRIVATEPARAM int do_freevarsin PARMS((treenode *n));
FORWARD PRIVATEPARAM void do_freevarsinexptree PARMS((treenode *n));

PRIVATE int freevarsin_usage_check;
PRIVATE int freevarsin_scope;
PRIVATE int freevarsin_use_mode;
PRIVATE varlist  *freevarsin_vlist;

/*{{{  PRIVATEPARAM int do_freevarsinexp (n)*/
/*****************************************************************************
 *
 *  freevarsinexp takes a treenode, 'n', a scope level, 'scope', a list of
 *  free variables, 'vlist', and a usage mode 'use_mode'.
 *  It adds to 'vlist' any variables used in tree 'n' of scope less than
 *  'scope', which aren't already in 'vlist'.
 *
 *****************************************************************************/
PRIVATEPARAM int do_freevarsinexp ( treenode *n )
{
  if (n != NULL)
    {
      DEBUG_MSG(("do_freevarsinexp: %s\n", usetagstring(n)));
      switch(TagOf(n))
        {
          /*{{{  FINSTANCE*/
          case S_FINSTANCE:
            {
              varlist *freevars = NFreeVarsOf(INameOf(n));
              freevarsin_vlist = merge(freevars, freevarsin_scope, freevarsin_vlist);
            }
            break;
          /*}}}*/
          /*{{{  subscripting / slicing*/
          case S_ARRAYSUB: case S_SEGMENT:
            {
              treenode *init_n = n;
              int all_subs_constant = TRUE;
              int old_mode = freevarsin_use_mode;
              freevarsin_use_mode = EXP_READ;
              while ((TagOf(n) == S_ARRAYSUB) || (TagOf(n) == S_SEGMENT))
                {
                  if (TagOf(n) == S_ARRAYSUB)
                    /*{{{  subscript*/
                    {
                      all_subs_constant = all_subs_constant && is_evaluable(ASIndexOf(n));
                      prewalktree(ASIndexOf(n), do_freevarsinexp);
                      n = ASBaseOf(n);
                    }
                    /*}}}*/
                  else
                    /*{{{  segment*/
                    {
                      all_subs_constant = all_subs_constant &&
                                          is_evaluable(SStartExpOf(n)) &&
                                          is_evaluable(SLengthExpOf(n));
                      prewalktree(SStartExpOf(n), do_freevarsinexp);
                      prewalktree(SLengthExpOf(n), do_freevarsinexp);
                      n = SNameOf(n);
                    }
                    /*}}}*/
                }
              freevarsin_use_mode = old_mode;
              switch (TagOf(n))
                {
                  /*{{{  default :*/
                  default :
                    /* This test changed back to 'OR' rather than 'AND' for bug 1017 16/10/90 */
                    if ((!freevarsin_usage_check) || (basetype(NTypeOf(n)) != S_TIMER))
                      {
                        INT32 start, end;
                        if (all_subs_constant && (elementsin(n) >= 0))
                          subscripts_accessed(init_n, &start, &end, FALSE);
                        else
                          start = (-1);
                        DEBUG_MSG(("do_freevarsinexp; arraysub, start:%ld, end:%ld, scope:%d\n", start, end, freevarsin_scope));
                        if (is_free(n, freevarsin_scope))
                          {
                            if (start >= 0)
                              freevarsin_vlist = addtolist(n, freevarsin_vlist, freevarsin_use_mode,
                                                           start, end);
                            else
                              freevarsin_vlist = addtolist(n, freevarsin_vlist, freevarsin_use_mode,
                                                           0, MAXINDEX);
                          }
                      }
                      break;
                  /*}}}*/
                  /*{{{  case S_CONSTRUCTOR : case S_ARRAYCONSTRUCTOR :*/
                  case S_CONSTRUCTOR :
                  #ifdef ARRAYCONSTRUCTOR
                  case S_ARRAYCONSTRUCTOR :
                  #endif
                    do_freevarsinexptree(n);
                    break;
                  /*}}}*/
                  /*{{{  case S_STRING : case S_CONSTCONSTRUCTOR*/
                  case S_STRING : case S_CONSTCONSTRUCTOR:
                    break;
                  /*}}}*/
                }
              return STOP_WALK;
            }
          /*}}}*/
          /*{{{  name node*/
          case N_VALABBR: case N_ABBR: case N_VALRETYPE: case N_RETYPE:
          case N_DECL: case N_VALPARAM: case N_PARAM: case N_REPL:
            /* Whole variable is being used */
            if (!freevarsin_usage_check || basetype(NTypeOf(n)) != S_TIMER)
              {
                if (is_free(n, freevarsin_scope))
                  freevarsin_vlist = addtolist(n, freevarsin_vlist,
                                               freevarsin_use_mode, 0, MAXINDEX);
              }
            return STOP_WALK;
          /*}}}*/
          /*{{{  specification / VALOF*/
          case S_ABBR: case S_RETYPE:
          case S_VALABBR: case S_VALRETYPE:
          case S_PROCDEF: case S_SFUNCDEF: case S_LFUNCDEF:
          case S_DECL: case S_TPROTDEF: case S_SPROTDEF: case S_VALOF:
          CASE_CONFIG_SPEC
            prewalkproctree(n, do_freevarsin);
            return STOP_WALK;
          /*}}}*/
          #ifdef ARRAYCONSTRUCTOR
          /*{{{  arrayconstructor*/
          case S_ARRAYCONSTRUCTOR :
            if (freevarsin_scope < 0)
              freevarsin_scope = NScopeOf(ACNameOf(n));
          
            freevarsin_vlist = freevarsinexp(ACStartExpOf(n),
                               freevarsin_scope, freevarsin_vlist, EXP_READ,
                               freevarsin_usage_check);
            freevarsin_vlist = freevarsinexp(ACLengthExpOf(n),
                               freevarsin_scope, freevarsin_vlist, EXP_READ,
                               freevarsin_usage_check);
          
            if (freevarsin_usage_check &&
                is_evaluable(ACStartExpOf(n)) &&
                is_evaluable(ACLengthExpOf(n)))
              {
                /*{{{  base & count both known*/
                {
                  INT32 base = evaluate(ACStartExpOf(n));
                  INT32 count = evaluate(ACLengthExpOf(n));
                  treenode *name = ACNameOf(n);
                  n = ACValExpOf(n);
                  SetNReplKnown(name, TRUE);
                  while (count > 0)
                    {
                      SetNReplValue(name, base);
                      prewalktree(n, do_freevarsinexp);
                      base++;
                      count--;
                    }
                  SetNReplKnown(name, FALSE);
                }
                /*}}}*/
                return STOP_WALK;
              }
            else
              {
                SetNReplKnown(ReplCNameOf(n), FALSE);
                return CONTINUE_WALK;
              }
          /*}}}*/
          #endif
        }
    }
  return CONTINUE_WALK;
}
/*}}}*/
/*{{{  PRIVATEPARAM void do_freevarsinexptree(n)*/
PRIVATEPARAM void do_freevarsinexptree ( treenode *n )
{
  prewalktree(n, do_freevarsinexp);
}
/*}}}*/
/*{{{  PRIVATE int do_freevarsin_if*/
PRIVATE int do_freevarsin_if(treenode *tptr)
/*
  Interprets an IF construct.
  Returns TRUE if a branch was chosen
*/

{
  DEBUG_MSG(("do_freevarsin_if: %s\n", usetagstring(tptr)));
  tptr = CBodyOf(tptr);
  while (!EndOfList(tptr))
    {
      treenode *t = ThisItem(tptr);
      while (isspecification(t))
        {
          if (do_freevarsin(t) != CONTINUE_WALK)
            {
              tptr = NextItem(tptr);  /* point at next guard */
              if (EndOfList(tptr))
                return FALSE;
              t = ThisItem(tptr);
            }
          else
            t = DBodyOf(t);
        }
      DEBUG_MSG(("do_freevarsin_if: (after spec): %s\n", usetagstring(t)));
      if (TagOf(t) == S_CHOICE)
        {
          prewalktree(CondGuardOf(t), do_freevarsinexp);
          if (is_evaluable(CondGuardOf(t))) /* we can check intelligently */
            {
              if (evaluate(CondGuardOf(t)) != 0)
                {
                  DEBUG_MSG(("do_freevarsin_if: evaluable guard (TRUE)\n"));
                  prewalkproctree(CondBodyOf(t), do_freevarsin);
                  return TRUE;
                }
              DEBUG_MSG(("do_freevarsin_if: evaluable guard (FALSE)\n"));
            }
          else /* process this guard anyway */
            {
              DEBUG_MSG(("do_freevarsin_if: unevaluable guard\n"));
              prewalkproctree(CondBodyOf(t), do_freevarsin);
              /* and drop through to next guard */
            }
        }
      else if (TagOf(t) == S_REPLIF)
        {
          /* don't attempt intelligent processing of a replicated if */
          prewalkproctree(t, do_freevarsin);
          /* and drop through to next guard */
        }
      else /* S_IF */
        {
          if (do_freevarsin_if(t))
            return TRUE;
        }
      tptr = NextItem(tptr);  /* point at next guard */
    }
  return FALSE;
}
/*}}}*/
/*{{{  PRIVATEPARAM int *do_freevarsin (n)*/
/*****************************************************************************
 *
 *  do_freevarsin takes a treenode, 'n', a scope level, 'freevarsin_scope',
 *  and a list of free variables, 'freevarsin_vlist',
 *  and adds to 'freevarsin_vlist' any variables used in tree
 *  'n' of freevarsin_scope less than 'scope',
 *  which aren't already in freevarsin_freevarsin_vlist.
 *
 *****************************************************************************/
PRIVATEPARAM int do_freevarsin ( treenode *n )
{
  /* Only used while checking which vars are written by a CASE_INPUT */
  static int vars_modified_by_case_input = FALSE; /* yes, I MEAN static */

  int old_mode  = freevarsin_use_mode;
  freevarsin_use_mode = EXP_READ;
  DEBUG_MSG(("do_freevarsin: %s\n", usetagstring(n)));
  switch(TagOf(n))
    {
      /*{{{  IF*/
      case S_IF:
        if (freevarsin_usage_check)
          {
            (void)do_freevarsin_if(n);
            return STOP_WALK;
          }
        return CONTINUE_WALK;
      /*}}}*/
      /*{{{  PAR PRIPAR PLACEDPAR*/
      case S_PAR: case S_PRIPAR: case S_PLACEDPAR: case S_DO:
        if (freevarsin_usage_check)
          {
            freevarsin_vlist = use_freevarsin(n, freevarsin_scope, freevarsin_vlist);
            return STOP_WALK;
          }
        return CONTINUE_WALK;
      /*}}}*/
      /*{{{  REPLPAR PRIREPLPAR PLACEDREPLPAR*/
      case S_REPLPAR: case S_PRIREPLPAR: case S_PLACEDREPLPAR: case S_REPLDO:
        if (freevarsin_scope < 0)
          freevarsin_scope = NScopeOf(ReplCNameOf(n));
        if (freevarsin_usage_check)
          {
            freevarsin_vlist = use_freevarsin(n, freevarsin_scope, freevarsin_vlist);
            return STOP_WALK;
          }
        break;
      /*}}}*/
      /*{{{  REPLSEQ REPLIF REPLALT PRIREPLALT*/
      case S_REPLSEQ: case S_REPLIF: case S_REPLALT: case S_PRIREPLALT:
        if (freevarsin_scope < 0)
          freevarsin_scope = NScopeOf(ReplCNameOf(n));
      
        freevarsin_vlist = freevarsinexp(ReplCStartExpOf(n),
                           freevarsin_scope, freevarsin_vlist, EXP_READ,
                           freevarsin_usage_check);
        freevarsin_vlist = freevarsinexp(ReplCLengthExpOf(n),
                           freevarsin_scope, freevarsin_vlist, EXP_READ,
                           freevarsin_usage_check);
      
        if (freevarsin_usage_check &&
            is_evaluable(ReplCStartExpOf(n)) &&
            is_evaluable(ReplCLengthExpOf(n))
            /* Added by CO'N 1/3/90 for bug 55 (ie to speed things up) */
            /* && goes_par(ReplCBodyOf(n))*/ /* removed again for bug 1025 */
           )
          {
            /*{{{  base & count both known & body goes PAR*/
            {
              INT32 base = evaluate(ReplCStartExpOf(n));
              INT32 count = evaluate(ReplCLengthExpOf(n));
              treenode *name = (treenode *) ReplCNameOf(n);
              n = ReplCBodyOf(n);
              SetNReplKnown(name, TRUE);
              while (count > 0)
                {
                  SetNReplValue(name, base);
                  prewalkproctree(n, do_freevarsin);
                  base++;
                  count--;
                }
              SetNReplKnown(name, FALSE);
            }
            /*}}}*/
            return STOP_WALK;
          }
        else
          {
            SetNReplKnown(ReplCNameOf(n), FALSE);
            return CONTINUE_WALK;
          }
      /*}}}*/
      /*{{{  PINSTANCE*/
      case S_PINSTANCE:
        {
          treenode *flist, *formal;
          varlist *freevars = NFreeVarsOf(INameOf(n));
          /* pre-defs, etc. don't have free variable lists so ... */
          if ((TagOf(INameOf(n)) == N_PROCDEF) || (TagOf(INameOf(n)) == N_INLINEPROCDEF))
            freevarsin_vlist = merge(freevars, freevarsin_scope, freevarsin_vlist);
          if (TagOf(INameOf(n)) == N_PREDEFPROC) /* bug 1035 1/11/90 */
            /*{{{  check params*/
            {
              int pdno = NModeOf(INameOf(n));
              if (pdno == PD_LOADINPUTCHANNEL  || pdno == PD_LOADINPUTCHANNELVECTOR ||
                  pdno == PD_LOADOUTPUTCHANNEL || pdno == PD_LOADOUTPUTCHANNELVECTOR)
                {
                  freevarsin_use_mode = (pdno == PD_LOADINPUTCHANNEL || pdno == PD_LOADINPUTCHANNELVECTOR) ?
                                        CHAN_INPUT : CHAN_OUTPUT;
                  flist = NTypeOf(INameOf(n));
                  n = IParamListOf(n);
                  while (n != NULL)
                    {
                      int type;
                      formal =ThisItem(flist);
                      type = basetype(NTypeOf(formal));
                      if ((type == S_CHAN) || (type == S_PORT))
                        prewalktree(ThisItem(n), do_freevarsinexp);
                      flist = NextItem(flist);
                      n = NextItem(n);
                    }
                }
             }
            /*}}}*/
          else
          /*{{{  check parameters*/
          {
          flist = NTypeOf(INameOf(n));
          n = IParamListOf(n);
          while (n != NULL)
            {
              int type;
              formal = ThisItem(flist);
              type = basetype(NTypeOf(formal));
              if ((type == S_CHAN) || (type == S_PORT))
                {
                  if (ParamInputOn(formal) || ParamOutputOn(formal))
                    {
                      if (ParamInputOn(formal))
                        {
                          freevarsin_use_mode = CHAN_INPUT;
                          prewalktree(ThisItem(n), do_freevarsinexp);
                        }
                      if (ParamOutputOn(formal))
                        {
                          freevarsin_use_mode = CHAN_OUTPUT;
                          prewalktree(ThisItem(n), do_freevarsinexp);
                        }
                    }
                  else
                    /*{{{  NOT USED add to EXP_READ list*/
                    {
                      freevarsin_use_mode = EXP_READ;
                      prewalktree(ThisItem(n), do_freevarsinexp);
                    }
                    /*}}}*/
                }
              else
                {
                  if (TagOf(formal) == N_VALPARAM)
                    freevarsin_use_mode = EXP_READ;
                  else
                    freevarsin_use_mode = EXP_WRITTEN;
                  prewalktree(ThisItem(n), do_freevarsinexp);
                }
              flist = NextItem(flist);
              n = NextItem(n);
            }
          }
          /*}}}*/
        }
        freevarsin_use_mode = old_mode;
        return CONTINUE_WALK;
      /*}}}*/
      /*{{{  ASS CASE OUTPUT INPUT TAGGED_INPUT DELAYED_INPUT*/
      case S_ASS: case S_OUTPUT: case S_INPUT: case S_TAGGED_INPUT:
      case S_DELAYED_INPUT: /* case S_CASE_INPUT: */
        {
          int rightmode = EXP_READ;
          switch (TagOf(n))
            {
              case S_ASS:           freevarsin_use_mode = EXP_WRITTEN;  break;
              case S_OUTPUT:        freevarsin_use_mode = CHAN_OUTPUT;  break;
              case S_DELAYED_INPUT: freevarsin_use_mode = CHAN_INPUT;   break;
              case S_INPUT: case S_TAGGED_INPUT: /* case S_CASE_INPUT: */
               freevarsin_use_mode = CHAN_INPUT;  rightmode = EXP_WRITTEN; break;
            }
          prewalktree(LHSOf(n), do_freevarsinexp);
          freevarsin_use_mode = rightmode;
          prewalktree(RHSOf(n), do_freevarsinexp);
          freevarsin_use_mode = old_mode;
        }
        return CONTINUE_WALK;
      /*}}}*/
      /*{{{  caseinput*/
      case S_CASE_INPUT:
        freevarsin_use_mode = CHAN_INPUT;
        prewalktree(LHSOf(n), do_freevarsinexp);

        /* Added a walk over rhs of the input, BUT NOT the 'bodies' */
        freevarsin_use_mode = EXP_WRITTEN;
        n = RHSOf(n);  /* point to a list of variants */
        while (n != NULL)
          {
            /* We allow for the fact that the variant may be preceded by
               specifications, by setting this flag.
               Inside here, we just want to look at the input part, not the
               attached processes. */

            int saved = vars_modified_by_case_input;
            vars_modified_by_case_input = TRUE;
            prewalktree(ThisItem(n), do_freevarsin);
            n = NextItem(n);
            vars_modified_by_case_input = saved;
          }

        freevarsin_use_mode = old_mode;
        return CONTINUE_WALK;
      /*}}}*/
      /*{{{  VARIANT*/
      case S_VARIANT:
        /* Added 24/4/90 by CO'N for bug 289 */
        walklist(do_freevarsinexptree, VRTaggedListOf(n));
        if (vars_modified_by_case_input)
          return STOP_WALK;  /* Don't look at the 'body' of the variant */
        return CONTINUE_WALK;
      /*}}}*/
      /*{{{  abbr / retype*/
      case S_ABBR: case S_RETYPE:
        {
          int type;
          if (freevarsin_scope < 0)
            freevarsin_scope = NScopeOf(DNameOf(n));
          type = basetype(NTypeOf(DNameOf(n)));
          if ((type != S_CHAN) && (type != S_PORT))
            /*{{{  WRITTEN*/
            {
              freevarsin_use_mode = EXP_WRITTEN;
              prewalktree(DValOf(n), do_freevarsinexp);
            }
            /*}}}*/
          else
            {
              int use = chan_use(DBodyOf(n), DNameOf(n));
              if ((use & CHAN_USE_INPUT) != 0)
                /*{{{  INPUT*/
                {
                  freevarsin_use_mode = CHAN_INPUT;
                  prewalktree(DValOf(n), do_freevarsinexp);
                }
                /*}}}*/
              if ((use & CHAN_USE_OUTPUT) != 0)
                /*{{{  OUTPUT*/
                {
                  freevarsin_use_mode = CHAN_OUTPUT;
                  prewalktree(DValOf(n), do_freevarsinexp);
                }
                /*}}}*/
              if (use == CHAN_UNUSED)
                /*{{{  NOT USED add to EXP_READ list*/
                {
                  freevarsin_use_mode = EXP_READ;
                  prewalktree(DValOf(n), do_freevarsinexp);
                }
                /*}}}*/
            }
          freevarsin_use_mode = old_mode;
        }
        return CONTINUE_WALK;
      /*}}}*/
      /*{{{  valabbr / valretype*/
      case S_VALABBR: case S_VALRETYPE:
        if (freevarsin_scope < 0)
          freevarsin_scope = NScopeOf(DNameOf(n));
        break;
      /*}}}*/
      /*{{{  others*/
      case S_DECL: case S_TPROTDEF:
      case S_SPROTDEF: case S_PROCDEF:
      case S_SFUNCDEF: case S_LFUNCDEF:
        if (freevarsin_scope < 0)
          {
            if (TagOf(DNameOf(n)) == S_LIST)
              freevarsin_scope = NScopeOf(ThisItem(DNameOf(n)));
            else
              freevarsin_scope = NScopeOf(DNameOf(n));
          }
        return CONTINUE_WALK;
      /*}}}*/
    }
  applytoexp(n, do_freevarsinexptree);
  freevarsin_use_mode = old_mode;
  return CONTINUE_WALK;
}
/*}}}*/

/*{{{  PUBLIC varlist *freevarsinexp (n, scope, vlist, use_mode, usage_check)*/
/*****************************************************************************
 *
 *  freevarsinexp takes a treenode, 'n', a scope level, 'scope', a list of
 *  free variables, 'vlist', and a usage mode 'use_mode'.
 *  It adds to 'vlist' any variables used in tree 'n' of scope less than
 *  'scope', which aren't already in 'vlist'.
 *
 *****************************************************************************/
PUBLIC varlist *freevarsinexp ( treenode *n , int scope , varlist *vlist , int use_mode , int usage_check )
{
  int       old_freevarsin_scope = freevarsin_scope;
  int       old_freevarsin_use_mode = freevarsin_use_mode;
  int       old_freevarsin_usage_check = freevarsin_usage_check;
  varlist  *old_freevarsin_vlist = freevarsin_vlist;

  freevarsin_scope = scope;
  freevarsin_vlist = vlist;
  freevarsin_use_mode = use_mode;
  freevarsin_usage_check = usage_check;

  DEBUG_MSG(("freevarsinexp\n"));
  prewalktree(n, do_freevarsinexp);

  vlist = freevarsin_vlist;
  freevarsin_scope = old_freevarsin_scope;
  freevarsin_vlist = old_freevarsin_vlist;
  freevarsin_use_mode = old_freevarsin_use_mode;
  freevarsin_usage_check = old_freevarsin_usage_check;
  return vlist;
}
/*}}}*/
/*{{{  PUBLIC varlist *freevarsin (n, scope, vlist, usage_check)*/
/*****************************************************************************
 *
 *  freevarsin takes a treenode, 'n', a scope level, 'scope', and a list of
 *  free variables, 'vlist', and adds to 'vlist' any variables used in tree
 *  'n' of scope less than 'scope', which aren't already in vlist.
 *
 *****************************************************************************/
PUBLIC varlist *freevarsin ( treenode *n , int scope , varlist *vlist , int usage_check )
{
  int       old_freevarsin_scope = freevarsin_scope;
  varlist  *old_freevarsin_vlist = freevarsin_vlist;
  int       old_freevarsin_usage_check = freevarsin_usage_check;
  freevarsin_scope = scope;
  freevarsin_vlist = vlist;
  freevarsin_usage_check = usage_check;

  DEBUG_MSG(("freevarsin\n"));
  prewalkproctree(n, do_freevarsin);

  vlist = freevarsin_vlist;
  freevarsin_scope = old_freevarsin_scope;
  freevarsin_vlist = old_freevarsin_vlist;
  freevarsin_usage_check = old_freevarsin_usage_check;
  return(vlist);
}
/*}}}*/
/*}}}*/
