/****************************************************************************
 *
 *  Parse tree structure
 *
 ****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/

/*{{{  SOURCEPOSN*/
typedef BIT32 SOURCEPOSN;
#define FileLineOf(L) ((int)((L) & 0xfffff))
#define FileNumOf(L)  ((int)(((L) >> 20) & 0xfff))

#define SetFileLine(L,M) ((L) = ((L) & 0xfff00000) | (((BIT32)M) & 0xfffff))
#define SetFileNum(L,M)  ((L) = ((L) & 0x000fffff) | ((((BIT32)M) & 0xfff) << 20))

#define NOPOSN ((SOURCEPOSN)0)
/*}}}*/

/*{{{  tagtype_s*/
/* Many structures must have their tag as the first part of the structure,
   and they must all be the same type. This enforces this requirement.
*/

/* This used to be an unsigned char, but since all the revelant structures
    have the next item word aligned, we can make it an int to speed access.
*/

/*typedef unsigned char tagtype_s;*/
typedef int tagtype_s;
/*}}}*/

/*{{{  wordnodestruct*/
struct wordnodestruct
  {
    tagtype_s w_tag;
    struct wordnodestruct *w_next;
    int w_length;
    char *w_name;
  };
typedef struct wordnodestruct wordnode;
/*}}}*/

struct tnode_s; /* Forward declaration */

/*{{{  parse tree nodes*/
struct treenodebase_s    /* This node must be EXACTLY the same as tnode_s */
                         /* except for the entry n_u */
  {
    tagtype_s tag;
    SOURCEPOSN lcn;
  };

#define TREENODEBASE  (sizeof(struct treenodebase_s))

/*{{{  structures used within tnode_s*/
/*{{{  struct actionnode_s           action*/
struct actionnode_s
  {
    struct tnode_s *act_lhs,             /* Left-hand side of the action */
                   *act_rhs;             /* Right-hand side of the action */
  };
/*}}}*/
/*{{{  struct altnode_s              alternative*/
struct altnode_s
  {
    struct tnode_s *alt_guard,            /* Guard expression */
                   *alt_input,            /* Channel input */
                   *alt_body;             /* Process in the ALT */
    struct tnode_s *alt_chanexp;          /* Channel expression */
    int             alt_label;            /* Used by backend */
  };
/*}}}*/
/*{{{  struct arraynode_s            array definition*/
struct arraynode_s
  {
    struct tnode_s *ar_dimlength,    /* Expression tree for dimension length */
                   *ar_type;         /* Type tree for the array */
    INT32           ar_dim;          /* Constant-folded dimension length */
      /* If dimension length is unknown, ar_dimlength == NULL, ar_dim == -1,
         until the workspace allocator reuses it to contain the workspace
         offset of the variable containing the array length */
  };
/*}}}*/
/*{{{  struct arraysubnode_s            array subscript*/
struct arraysubnode_s
  {
    struct tnode_s *as_base,         /* What is being subscripted */
                   *as_index,        /* the subscript */
                   *as_exp;          /* Transformed expression for backend */
    struct tnode_s *as_length;       /* Expression containing length, if not constant */
    INT32           as_offset;       /* Constant-folded dimension length */
#ifdef SIC
    int as_temp;                        /* Silicon compiler tempory */
#endif
  };
/*}}}*/
/*{{{  struct channode_s             channel definition*/
struct channode_s
  {
    struct tnode_s *ch_protptr;        /* Tree representing channel protocol */
  };
/*}}}*/
/*{{{  struct cnode_s                construct (SEQ, PAR, IF, ALT)*/
struct cnode_s
  {
    struct tnode_s *c_body;              /* Body of the construct */
    struct tnode_s *c_temp;              /* Space for a temporary */
  };
/*}}}*/
/*{{{  struct condnode_s             condition (guard)*/
struct condnode_s
  {
    struct tnode_s *cnd_guard,           /* Guard expression */
                   *cnd_body;            /* Guarded process */
  };
/*}}}*/
#ifdef CONDEXP
/*{{{  struct condexpnode_s          conditional expression*/
struct condexpnode_s
  {
    struct tnode_s *cndexp_guard;           /* Guard expression */
    struct tnode_s *cndexp_true;            /* true expression  */
    struct tnode_s *cndexp_false;           /* false expression */
    int cndexp_type;                        /* Type of operator */
  };
/*}}}*/
#endif
/*{{{  struct constexpnode_s         constant folded expression*/
struct constexpnode_s
  {
    struct tnode_s *ce_exp;              /* Unfolded expression tree */
    BIT32 ce_vhi, ce_vlo;                /* Folded value of expression */
    struct tnode_s *ce_next;             /* Link to next item in list,
                                            used by code generator */
    INT32 ce_offset;                     /* Offset from beginning of
                                            long constant table,
                                            used by code generator */
  };
/*}}}*/
/*{{{  struct consttablenode_s       constant table*/
struct consttablenode_s
  {
    wordnode       *ct_val;              /* Value of constant table */
    struct tnode_s *ct_exp,              /* Constant tree before folding */
                   *ct_next;             /* Link to next item in list,
                                            used by code generator */
    int ct_label;                        /* Label of constant table,
                                            used by code generator */
  };
/*}}}*/
/*{{{  struct declnode_s             specification*/
struct declnode_s
  {
    struct tnode_s *d_nameptr,           /* List of symbol table entries */
                   *d_v,                 /* Initializing value, if any */
                   *d_body;              /* Process following specification */
  };
/*}}}*/
/*{{{  struct dopnode_s              dyadic operator*/
struct dopnode_s
  {
    struct tnode_s *dop_leftop,          /* Left-hand operand */
                   *dop_rightop;         /* Right-hand operand */
    int dop_type;                  /* Type of operator */
#ifdef SIC
    int dop_temp;                        /* Silicon compiler tempory */
#endif
  };
/*}}}*/
/*{{{  struct hiddenparamnode_s      hidden parameter, used by backend only*/
struct hiddenparamnode_s
  {
    struct tnode_s *h_exp;       /* Expression to which hidden param applies */
    BIT32 h_dimension;           /* Dimension which hidden param represents */
  };
/*}}}*/
/*{{{  struct instancenode_s         function or procedure call*/
struct instancenode_s
  {
    struct tnode_s *i_nameptr;         /* Symbol table entry for called proc */
    struct tnode_s *i_paramlist;                /* List of actual parameters */
    unsigned char i_loadseq;                   /* Parameter loading sequence */
  };
/*}}}*/
/*{{{  struct listnode_s             list of items*/
struct listnode_s
  {
    struct tnode_s *ls_left,             /* This item on the list */
                   *ls_right;            /* Next item on the list, or another
                                            list node */
  };
/*}}}*/
/*{{{  struct litnode_s              literal*/
struct litnode_s
  {
    wordnode *l_stringptr;   /* Name table entry representing literal */
  };
/*}}}*/
/*{{{  struct mopnode_s              monadic operator*/
struct mopnode_s
  {
    struct tnode_s *mop_operand;         /* Operand */
    int mop_type;                        /* Type of operator */
#ifdef SIC
    int mop_temp;                        /* Silicon compiler temporary */
#endif
  };
/*}}}*/
#if 0 /* never used */
/*{{{  struct overlapnode_s          overlap check*/
struct overlapnode_s
  {
    struct tnode_s *o_base1,            /* base of first subscript     */
                   *o_count1,           /* length of first susbcript   */
                   *o_base2,            /* base of second subscript    */
                   *o_count2;           /* length of second subscript  */
  };
/*}}}*/
#endif
/*{{{  struct processornode_s        processor*/
struct processornode_s
  {
    struct tnode_s *p_exp;              /* processor number expression */
    wordnode       *p_type;      /* processor type              */
    struct tnode_s *p_body;             /* processor body              */
    BIT16           p_scope;            /* scope of PROCESSOR statement*/
  };
/*}}}*/
/*{{{  struct replcnode_s            replicated construct*/
struct replcnode_s
  {
    struct tnode_s *rc_nameptr;    /* Symbol table entry for replicator */
    struct tnode_s *rc_startexp,         /* Replicator start expression */
                   *rc_lengthexp,        /* Replicator count expression */
                   *rc_body;             /* Replicated process */
    struct tnode_s *rc_temp;             /* Temp for replicated ALTs    */
  };
/*}}}*/
/*{{{  struct segmentnode_s          segment instance*/
struct segmentnode_s
  {
    struct tnode_s *s_nptr,              /* Segmented element */
                   *s_startexp,          /* Segment start expression */
                   *s_lengthexp,         /* Segment length expression */
                   *s_subscriptexp,      /* Transformed subscript expression */
                   *s_checkexp;          /* Segment check expression */
    struct tnode_s *s_length;            /* Expression containing length, if not constant */
    INT32          s_offset;             /* Constant offset from subscript */

  };
/*}}}*/
/*{{{  struct spacenode_s            space usage information, used  by backend*/
struct spacenode_s
  {
    struct tnode_s *sp_body;             /* Sized process */
    INT32           sp_maxwsp,           /* space usage above wptr */
                    sp_datasize,         /* total space including below wptr */
                    sp_vsusage,          /* vector space usage */
                    sp_nestedvs;         /* vs usage for nested calls */
    struct tnode_s *sp_namechain;        /* names declared in body */
    BIT32           sp_cpoffset;         /* constant ptr offset in repl PAR workspace */
  };
/*}}}*/
#if 0 /* never used */
/*{{{  struct tempnode_s             temporary variable*/
struct tempnode_s
  {
    int t_number;                        /* Number of temporary */
    BIT32 t_offset;                      /* Workspace offset of variable */
    struct tnode_s *t_exp;               /* Pointer to expression temporary */
                                         /* represents */
    struct tnode_s *t_next;              /* Next node in allocation list */
    unsigned char t_lexlev;              /* Lex level at which temp declared */
  };
/*}}}*/
#endif
/*{{{  struct valofnode_s            valof instance*/
struct valofnode_s
  {
    struct tnode_s *vl_body,             /* VALOF body */
                   *vl_resultlist;       /* VALOF result list */
  };
/*}}}*/
/*{{{  struct variantnode_s          variant in case input*/
struct variantnode_s
  {
    struct tnode_s *vr_taggedlist,       /* Tag, followed by list of input items */
                   *vr_body;             /* Guarded process */
  };
/*}}}*/
#ifdef CONFIG
/*{{{  struct confignode_s           config node*/
struct confignode_s
  {                        /* CONNECT SET        MAP */
    struct tnode_s *cnf_a; /* From    Name       Source */
    struct tnode_s *cnf_b; /* To      Attr Names Dest   */
    struct tnode_s *cnf_c; /* Arc     Attr Exps  Pri    */
  };
/*}}}*/
#endif
/*}}}*/

/*{{{  struct namenode_s             symbol table entry*/
struct namenode_s
  {
    wordnode *n_name;           /* String representing name */
    struct tnode_s *n_type;     /* Type tree */
    struct tnode_s *n_decl;     /* Pointer to name declaration on parse tree */
    unsigned char n_mode;       /* Mode, ie. workspace/vecspace/pointer ... */
    unsigned char n_lexlev;     /* Lexical level of name declaration */
                                /* For a routine, top bit set if the routine
                                   contains a PRI PAR */
                                /* For any name, bit 6 set if it is used */
    BIT16 n_scope;              /* Scope of declaration */
    void *n_checker;            /* Used by usage checker */

    /* The elements of the following union are arranged in such a silly
       order, because when I separated out the PROC stuff from the VAR stuff,
       I wanted to be sure I didn't break any hidden dependencies of their
       ordering.
       CO'N 3/7/90
    */
    union{
      struct{ /* First we have the record for PROCs and FUNCTIONs */
        union{
          struct tnode_s *n_ctable;    /* pointer to constant table chain */
          struct tnode_s *n_nextlib;   /* pointer to next lib entry point */
        } n_un1;
        BIT32             n_label;     /* label of entry point */
        BIT32             n_maxwsp;    /* Max workspace size */
        BIT32             n_datasize;  /* Below workspace size */
        BIT32             n_vsusage;   /* VS Usage */
        void             *n_external;  /* pointer to lib file data (LIBs only)*/
        BIT32             n_params;    /* Parameter info */
        union{
          BIT32           n_cpoffset;  /* WS offset of constant table */
          BIT32           n_liboffset; /* Code offset for lib patch */
        } n_un2;
      } n_proc;

      struct{ /* This is for variables (ie most others) */
        BIT32             n_offset;    /* Workspace position*/
        union{
          BIT32           n_vsoffset;  /* VS offset if mode = NM_VECSPACE */
          int             n_replknown; /* whether value of N_REPL is known */
          int             n_chanmark;  /* whether chan used by ASM (if type is scalar chan) */
        } n_un3;
        union{
          BIT32           n_usecount;  /* Usage count for ws allocation */
          BIT32           n_replval;   /* Value of replicator N_REPL*/
          BIT32           n_rsize;     /* Size of T_RESERVEDWS */
        } n_un4;
        union{
          struct{ /* These are all frontend only */
            int             n_shared;    /* Whether variable may be shared */
          } n_front;
          struct{ /* These are all backend only */
            struct tnode_s *n_shares;    /* Chain of vars with same ws offset */
            BIT32           n_varnum;    /* Temporaries only */
            struct tnode_s *n_nexttemp;  /* Temporaries only */
            struct tnode_s *n_allocnext; /* used while allocating ws */
          } n_back;
        } n_un5;
      } n_var;

      struct{ /* This is for protocol tags */
        BIT32             n_tagval;    /* Tag value */
      } n_tag;

    } n_un;

  };
/*}}}*/

/*{{{  struct tnode_s*/
struct tnode_s
  {
    tagtype_s tag;     /* Node tag */
    SOURCEPOSN lcn;        /* Source location corresponding to node */
    /*{{{  union {...} n_u    union of possible node bodies*/
    union
      {
        struct actionnode_s      act_s;
        struct altnode_s         alt_s;
        struct arraynode_s       ar_s;
        struct arraysubnode_s    as_s;
        struct channode_s        ch_s;
        struct cnode_s           c_s;
        struct condnode_s        cnd_s;
        struct constexpnode_s    ce_s;
        struct consttablenode_s  ct_s;
        struct declnode_s        d_s;
        struct dopnode_s         dop_s;
        struct hiddenparamnode_s h_s;
        struct instancenode_s    i_s;
        struct listnode_s        ls_s;
        struct litnode_s         l_s;
        struct mopnode_s         mop_s;
        struct namenode_s        n_s;
      /*struct overlapnode_s     o_s;*/ /* never used */
        struct processornode_s   p_s;
        struct replcnode_s       rc_s;
        struct segmentnode_s     s_s;
        struct spacenode_s       sp_s;
      /*struct tempnode_s        t_s;*/ /* never used */
        struct valofnode_s       vl_s;
        struct variantnode_s     vr_s;
    #ifdef CONFIG
        struct confignode_s      cnf_s;
    #endif
    #ifdef CONDEXP
        struct condexpnode_s     cndexp_s;
    #endif
      } n_u;
    /*}}}*/
  };
/*}}}*/

typedef struct tnode_s treenode;


/* Node types */
#define NONODE            0
#define ACTIONNODE        1
#define ALTNODE           2
#define ARRAYNODE         3
#define ARRAYSUBNODE      4
#define CHANNODE          5
#define CNODE             6
#define CONDNODE          7
#define CONSTEXPNODE      8
#define CONSTTABLENODE    9
#define DECLNODE         10
#define DOPNODE          11
#define HIDDENPARAMNODE  12
#define INSTANCENODE     13
#define LEAFNODE         14
#define LISTNODE         15
#define LITNODE          16
#define MOPNODE          17
#define NAMENODE         18
#define PROCESSORNODE    19
#define REPLCNODE        20
#define SEGMENTNODE      21
#define SPACENODE        22
#define VALOFNODE        24
#define VARIANTNODE      25
#define WORDNODE         26

#ifdef CONFIG
#define CONFIGNODE       27
#endif

#ifdef CONDEXP
#define CONDEXPNODE      30
#endif

/*#define OVERLAPNODE      31*/
/*#define TEMPNODE         32*/

