/*#define DEBUG*/
/****************************************************************************
 *
 *  Occam two syntax analyser 2
 *
 ****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include "includes.h"
# include "lexconst.h"
# include "lexerror.h"
# include "synerror.h"
# include "lex1def.h"
# include "lexdef.h"
# include "syn1def.h"
# include "syn2def.h"
# include "syn3def.h"
#ifdef CONFIG
# include "syn4def.h"
#endif
# include "syndef.h"
# include "chkdef.h"
# include "desc1def.h"
/*}}}*/

/*{{{  PUBLIC vars*/
PUBLIC int allow_inlines = TRUE;
/*}}}*/

#define STRING_RECORD_SIZE 1000

/*{{{  forward declarations*/
PRIVATE treenode *rspecorexpr ( int *specflag , int pindent , int explist );
/*}}}*/

/*{{{  routines*/
/*{{{  support*/
/*{{{  PUBLIC int mustbespec(s)*/
PUBLIC int mustbespec ( int s )
{
  switch (s)
    {
      case S_VAL: case S_PROC: case S_PROTOCOL: case S_CHAN:
      case S_PORT: case S_TIMER: case S_BOX: case S_PLACE:
      case S_INLINE: CASE_CONFIG_SPEC
        return TRUE;
      default :
        return FALSE;
   }
}
/*}}}*/
/*{{{  PUBLIC int mustbeexp(s)*/
PUBLIC int mustbeexp ( int s )
{
  switch (s)
    {
     case S_MOSTPOS: case S_MOSTNEG: case S_SUBTRACT: case S_MINUS:
     case S_BITNOT: case S_NOT: case S_SIZE: case S_LPAREN:
     case S_TRUE: case S_FALSE:
     case S_UBYTELIT: case S_UINTLIT: case S_UREALLIT:
     case S_STRING: case S_STRINGCONT:
     case S_VALOF:
        return TRUE;
      default :
        return FALSE;
   }
}
/*}}}*/
/*}}}*/

/*{{{  lists*/
/*{{{  PUBLIC treenode *rproclist (readitem, indent)*/
/*****************************************************************************
 *
 *  rproclist reads a list of items. 'readitem' is a function called to read
 *            each item.  Each item of the list must have the same indentation
 *            as the first item: a smaller indentation terminates the list,
 *            a greater indentation is an error.
 *
 *****************************************************************************/
PUBLIC treenode *rproclist ( treenode *(*readitem)(void), int indent )
{
  treenode *root, **pptr;
  SOURCEPOSN locn = flocn;
  DEBUG_MSG(("rproclist... "));
  root = NULL;
  pptr = &root;
  while (symbindent >= indent)
    /*{{{  create a list node and read another item*/
    {
      DEBUG_MSG(("rproclist: looking for next process, symb:symbindent is %s:%d... ",
                 tagstring(symb), symbindent));
      if (symb == S_COMMENT)
        nextline ();
      else if (symbindent > indent)
        /*{{{  error so skip to next process if any*/
        {
          synerr(SYN_BAD_INDENT, flocn);
          skiplines (indent);
        }
        /*}}}*/
      else
        /*{{{  parse a process item, and append to list*/
        {
          treenode *item = (*readitem) ();
          if (item != NULL)
            /*{{{  make new node*/
            {
              *pptr = newlistnode (S_LIST, locn, item, NULL);
              pptr = NextItemAddr(*pptr);
            }
            /*}}}*/
        }
        /*}}}*/
      locn = flocn;
    }
    /*}}}*/
  return (root);
}
/*}}}*/
/*{{{  PUBLIC treenode *rlist (p, sep)*/
/*****************************************************************************
 *
 *  rlist reads a list of items.  'p' is a function which reads each item,
 *        and 'sep' is the token which separates each item.
 *        ie. parse   p { sep p }
 *
 *****************************************************************************/

/* may return NULL if error found */
PUBLIC treenode *rlist ( treenode *(*p)(void), int sep )
{
  treenode *tptr, *a, *item;
  SOURCEPOSN locn = flocn;
  DEBUG_MSG(("rlist... "));
  /*{{{  parse item*/
  if ((item = (*p)()) == NULL)
    return NULL;
  /*}}}*/
  tptr = newlistnode(S_LIST, locn, item, NULL);
  a = tptr;
  while (symb == sep)
    /*{{{  do another item*/
    {
      nextsymb ();
      /*{{{  ignore line break*/
      if (checklinebreak ())
        return NULL;
      /*}}}*/
      /*{{{  parse item*/
      if ((item = (*p)()) == NULL)
        return NULL;
      /*}}}*/
      NewNextItem(newlistnode (S_LIST, locn, item, NULL), a);
      a = NextItem(a);
      locn = flocn;
    }
    /*}}}*/
  return (tptr);
}
/*}}}*/
/*}}}*/

/* expression parsers may return NULL if an error is found , */
/* but will not skip to the end of a line,                   */
/* except when a VALOF is parsed.                            */
/*{{{  expressions*/
/*{{{  PUBLIC wordnode *rname ()*/
PUBLIC wordnode *rname ( void )
{
  wordnode *name;

  DEBUG_MSG(("rname... "));
  if ((symb == S_NAME) || (symb == S_ASMNAME))
    {
      name = lexword;
      nextsymb ();
      return (name);
    }
  else
    {
      synerr_e (SYN_E_NAME, flocn, symb);
      return NULL;
    }
}
/*}}}*/
/*{{{  PUBLIC treenode *rsegment(node)*/
PUBLIC treenode *rsegment ( treenode *node )
{
  /*{{{  parsing*/
  /*
    the '[' and element have already been parsed,
    node is the element we are taking the slice of :
  
      'FROM' exp 'FOR' exp ']'
  
  */
  /*}}}*/
  treenode *start, *length;
  if (checkfor(S_FROM))
    return NULL;

  if (checklinebreak () ||
      ((start = rexp ()) == NULL) ||
      checkfor (S_FOR) ||
      checklinebreak () ||
      ((length = rexp ()) == NULL) ||
      checkfor (S_RBOX) )
    return NULL;

  return newsegmentnode (S_SEGMENT, flocn, node, start, length);
}
/*}}}*/
/*{{{  PUBLIC treenode *rconstructor(node)*/
PUBLIC treenode *rconstructor ( treenode *node )
{
  /*{{{  parsing*/
  /*
    The opening box & first element have been parsed,
    node is is the first element.
    we do :
             {',' exp} ']'
  */
  /*}}}*/
  if (symb == S_COMMA)
    {                                                 /* read list of items */
      treenode *list;
      nextsymb ();
      if (checklinebreak ())
        return NULL;
      if ((list = rlist (rexp, S_COMMA)) == NULL)
        return NULL;
      node = newlistnode (S_LIST, flocn, node, list);
    }
  else
    node = newlistnode(S_LIST, flocn, node, NULL);     /* one item list      */
  if (checkfor (S_RBOX))
    return NULL;
  return newmopnode (S_CONSTRUCTOR, flocn, node, 0);
}
/*}}}*/
#ifdef ARRAYCONSTRUCTOR
/*{{{  PRIVATE treenode *rarrayconstructor()*/
PRIVATE treenode *rarrayconstructor ( void )
{
  /*{{{  parsing*/
  /*
    the '[' has been parsed,
  
      'ARRAY' name 'FROM' exp 'FOR' exp 'OF' exp ']'
  
  */
  /*}}}*/
  treenode *nptr, *rcptr, *start, *length, *val;

  if (checkfor(S_ARRAYCONSTRUCTOR) ||
      rrepl(&nptr, &start, &length) ||
      checkfor (S_OF) ||
      checklinebreak () ||
      ((val = rexp ()) == NULL) ||
      checkfor (S_RBOX))
    return NULL;

  rcptr = newreplcnode (S_ARRAYCONSTRUCTOR, flocn, nptr, start, length, val);
  SetNDecl(nptr, rcptr);
  return rcptr;
}
/*}}}*/
#endif
/*{{{  PUBLIC treenode *rsubscript (a)*/
/* The element built up so far */
/* may return NULL */
/* Parse
'[' expression ']' { '[' expression ']' }
*/
PUBLIC treenode *rsubscript ( treenode *a )
{
  SOURCEPOSN locn = flocn;
  treenode *exp;
  DEBUG_MSG(("rsubscript... "));
  if (symb != S_LBOX)
    {
      synetoken (S_LBOX);
      return NULL;
    }
  while (symb == S_LBOX)
    {
      nextsymb();
      if ((exp = rexp ()) == NULL)
        return NULL;
      if (checkfor (S_RBOX))
        return NULL;
      a = newarraysubnode (S_ARRAYSUB, locn, a, exp);
      /* if (symb == S_LBOX) a = rsubscript (a); */
    }
  return (a);
}
/*}}}*/
/*{{{  PUBLIC treenode *relement ()*/
/* may return NULL if error found */
PUBLIC treenode *relement ( void )
{
  SOURCEPOSN locn = flocn;
  treenode *a;

  DEBUG_MSG(("relement... "));
  if (symb == S_NAME)
    a = (treenode *)rname(); /* name or subscripted element */
  else if (symb == S_LBOX)
    /*{{{  segment*/
    {
      treenode *name;
      nextsymb ();
      if ((name = relement ()) == NULL)
        return NULL;
      a = rsegment (name);
    }
    /*}}}*/
  else
    {
      synerr_e (SYN_E_ELEMENT, locn, symb);
      return NULL;
    }

  if (a != NULL && symb == S_LBOX)
    a = rsubscript (a);

  return (a);
}
/*}}}*/
/*{{{  PUBLIC treenode *rinstance (tag, locn, name)*/
PUBLIC treenode *rinstance ( int tag , SOURCEPOSN locn , treenode *name )
{
  /*{{{  parsing*/
  /*
    Assume the name has already been read
    '(' exp {',' exp } ')'
   */
  /*}}}*/
  treenode *params;
  if (checkfor (S_LPAREN))
    return NULL;
  if (symb != S_RPAREN)
    {
      if ((params = rlist (rexp, S_COMMA)) == NULL)
        return NULL;
    }
  else
    params = NULL;
  if (checkfor (S_RPAREN))
    return NULL;
  return newinstancenode (tag, locn, name, params);
}
/*}}}*/
/*{{{  PRIVATE treenode *rvalof ()*/
/* Terminates with symb at start of new line */
/*{{{  what we are parsing*/
/*  Parse
valof = 'VALOF'
process
'RESULT' expression.list

including the newline following the expression list,
or
valof = specification
valof
*/
/*}}}*/
PRIVATE treenode *rvalof ( void )
{
  int indent = symbindent;              /* The indent of the 'VALOF' keyword */
  int savedlinebreakindent;
  SOURCEPOSN locn = flocn;
  treenode *p, *n, *e;

  DEBUG_MSG(("rvalof... "));
  /*{{{  VALOF*/
  if (checkfor (S_VALOF))
    goto error;
  if (checknlindent (indent + 2))
    goto error2;
  /*}}}*/
  /*{{{  PROCESS*/
  savedlinebreakindent = linebreakindent;
  linebreakindent = (-1);
  if (((p = rprocess()) == NULL) || (checkindent (indent + 2)))
    goto error2;
  ignorecomments(indent + 2);
  /*}}}*/
  /*{{{  RESULT list*/
  if (checkfor (S_RESULT) || ((e = rlist (rexp, S_COMMA)) == NULL))
    goto error;
  /*}}}*/
  n = newvalofnode (S_VALOF, locn, p, e);
  checknewline ();
  linebreakindent = savedlinebreakindent;
  return (n);

error:
  nextline ();
error2:
  skiplines (indent);
  return NULL;
}
/*}}}*/
#ifdef CONDEXP
/*{{{  PRIVATE treenode *rcondexp(node)*/
PRIVATE treenode *rcondexp ( treenode *node )
{
  /*{{{  parsing*/
  /*
      the boolean expression has been parsed already :
          '->' exp ',' exp
        | '->' exp ',' conditional.expression
   */
  /*}}}*/
  treenode *true, *false;
  if (checkfor (S_RIGHTARROW) ||
      checklinebreak () ||
      ((true = rexp ()) == NULL) ||
      checkfor (S_COMMA) ||
      checklinebreak () ||
      ((false = rexp ()) == NULL))
    return NULL;
  if (symb == S_RIGHTARROW)
    false = rcondexp(false);
  return newcondexpnode (S_CONDEXP, flocn, node, true, false);
}
/*}}}*/
#endif
/*{{{  PRIVATE treenode *rstring*/
PRIVATE treenode *rstring()
/* This routine reads a string, allowing for long strings.
   It does NOT advance the symbol to the next symbol after the string
*/
{
  treenode *node;
  SOURCEPOSN locn = flocn;
  if (symb == S_STRING)
    {
      literalv[literalp] = '\0';
      node = newconsttablenode(S_STRING, locn, lookupword(literalv, literalp), NULL);
    }
  else /* (symb == S_STRINGCONT) */
    {
      int more = TRUE;
      int len = literalp;
      int stringsize = STRING_RECORD_SIZE;
      char *string = memalloc(stringsize);
      memcpy(string, literalv, literalp);
      nextsymb();
      while (more)
        {
          if ((len + literalp) > stringsize)
            /*{{{  make string a bit bigger*/
            {
              char *newstring;
              stringsize += STRING_RECORD_SIZE;
              newstring = memalloc(stringsize);
              memcpy(newstring, string, len);
              memfree(string);
              string = newstring;
            }
            /*}}}*/
          memcpy(string + len, literalv, literalp);
          len += literalp;
          locn = flocn;
          if (symb == S_STRINGCONT)
            nextsymb();
          else /* symb == S_STRING */
            more = FALSE;
        }
      node = newconsttablenode(S_STRING, locn, lookupword(string, len), NULL);
      memfree(string);
    }
  return node;
}
/*}}}*/
/*{{{  PRIVATE treenode *roperand ()*/
/* On error, leave symb unchanged, except when VALOF found */
/*{{{  what we are parsing*/
/* Parse
'(' expression ')'
| '(' valof ')'
| '(' conditional.expression ')'
| literal
| element
| '[' expression { ',' expression } ']'
| name '(' expression { ',' expression } ')'
*/
/*}}}*/
PRIVATE treenode *roperand ( void )
{
  SOURCEPOSN locn = flocn;
  treenode *node = NULL;
  DEBUG_MSG(("roperand... "));
  switch (symb)
    {
      /*{{{  case S_LPAREN could be expression, specification or valof*/
      case S_LPAREN:
        {
          int specflag = FALSE;
          treenode **a = &node;
          int pindent = symbindent, indent;
          nextsymb ();
          indent = symbindent; /* The indent of first symbol after left paren */
          /*{{{  read spec or expression into *a*/
          if ((*a = rspecorexpr (&specflag, pindent, FALSE)) == NULL)
            if (!specflag)
              return NULL;  /* previous syntax error */
          /*}}}*/
          if (specflag)
            /*{{{  read further specifications, a valof, and check it*/
            {
              while (symb != S_VALOF && symbindent == indent)
                {
                  if (*a != NULL) a = DBodyAddr(*a);  /* move a along list */
                  while (symb == S_COMMENT)           /* skip comments    */
                    if (checknlindent (indent))
                      return NULL;
                  *a = rspecification ();             /* parse expression  */
                }
              if (symb != S_VALOF)                    /* Check for valof   */
                {
                  synetoken(S_VALOF);
                  return NULL;
                }
              if (checkindent (indent))
                return NULL;
              if (*a != NULL)
                a = DBodyAddr(*a);                       /* move a along list   */
              if ((*a = rvalof ()) == NULL)
                return NULL;                             /* parse VALOF         */
              ignorecomments(pindent);                   /* throw away comments */
            }
            /*}}}*/
          #ifdef CONDEXP
          if (symb == S_RIGHTARROW)
            *a = rcondexp(*a);
          #endif
          if (checkfor (S_RPAREN)) return NULL;
        }
        break;
      /*}}}*/
      /*{{{  case S_TRUE FALSE*/
      case S_TRUE: case S_FALSE:
        node = newleafnode(symb, locn);
        nextsymb ();
        break;
      /*}}}*/
      /*{{{  S_UINTLIT S_UREALLIT S_UBYTELIT*/
      case S_UINTLIT: case S_UREALLIT: case S_UBYTELIT:
        {
          int s = symb;
          wordnode *w = lookupword(literalv, literalp);
          nextsymb ();
          if (symb == S_LPAREN)
            /*{{{  we have a type specifier*/
            { /* '(' simple type ')' */
              nextsymb ();
              if (((s == S_UINTLIT || s == S_UBYTELIT) &&
                     (symb == S_INT   || symb == S_BYTE || symb == S_INT64 ||
                        symb == S_INT32 || symb == S_INT16)) ||
                  (s == S_UREALLIT && (symb == S_REAL32 || symb == S_REAL64)))
                s = littag(symb);
              else
                {
                  synerr(SYN_ILLEGAL_CONVERSION, locn);
                  return NULL;
                }
              nextsymb ();
              if (checkfor (S_RPAREN))
                return NULL;
            }
            /*}}}*/
          node = newlitnode (s, locn, w);
        }
        break;
      /*}}}*/
      /*{{{  case S_STRING, S_STRINGCONT*/
      case S_STRING: case S_STRINGCONT:
        node = rstring();
        nextsymb();
        if (symb == S_LBOX) node = rsubscript(node);
        break;
      /*}}}*/
      /*{{{  case S_NAME*/
      case S_NAME: case S_ASMNAME:
        node = (treenode *)rname();
        if (symb == S_LBOX)
          node = rsubscript (node);
        else if (symb == S_LPAREN)
          node = rinstance (S_FINSTANCE, locn, node);
        break;
      /*}}}*/
      /*{{{  case S_LBOX*/
      case S_LBOX:
        nextsymb ();
      #ifdef ARRAYCONSTRUCTOR
        if (symb == S_ARRAYCONSTRUCTOR)
          node = rarrayconstructor();
        else
      #endif
          {
            if ((node = rexp ()) == NULL)
              return NULL;
            if (symb == S_FROM)
              node = rsegment(node);
            else
              node = rconstructor(node);
          }
        if (node != NULL && symb == S_LBOX)
          node = rsubscript (node);
        break;
      /*}}}*/
      default:
        synerr_e (SYN_E_OPERAND, flocn, symb);
        break;
    }
  return node;
}
/*}}}*/
/*{{{  PRIVATE treenode *rrestofexp (a, locn)*/
/* May return NULL if error found */
/* Parse
{ dyadic.op operand }
*/
PRIVATE treenode *rrestofexp ( treenode *a , SOURCEPOSN locn )
{
  int assoc = TRUE;
  /*{{{  build up expression*/
  while (assoc)
    {
      treenode *operand;
      int operator;
      switch (symb)
        {
          /*{{{  S_AND S_OR*/
          case S_AND:
          case S_OR:
            assoc = TRUE;
            break;
          /*}}}*/
          /*{{{  ADD SUBTRACT MULT DIV REM BITAND BITOR XOR LSHIFT RSHIFT PLUS... */
          case S_ADD: case S_SUBTRACT: case S_MULT: case S_DIV: case S_REM:
          case S_BITAND: case S_BITOR: case S_XOR:
          case S_LSHIFT: case S_RSHIFT:
          case S_PLUS: case S_MINUS: case S_TIMES:
          case S_EQ: case S_NE: case S_LS: case S_LE: case S_GR: case S_GE: case S_AFTER:
            assoc = FALSE;
            break;
          /*}}}*/
          default:
            return (a);
        };
      operator = symb;
      nextsymb ();
      if (checklinebreak ())
        return NULL;
      if ((operand = roperand ()) == NULL)
        return NULL;
      a = newdopnode (operator, locn, a, operand, 0);
    };
  /*}}}*/
  return (a);
}
/*}}}*/
/*{{{  PUBLIC treenode *rexp ()*/
/* Expressions may be broken after an expression operator - the indentation
of the first symbol on the continuation line must be at least as great
as that of the first line.
*/
PUBLIC treenode *rexp ( void )
{
  SOURCEPOSN locn = flocn;
  DEBUG_MSG(("rexp... "));

  switch (symb)
    {
      /*{{{  case S_MOSTPOS S_MOSTNEG*/
      case S_MOSTPOS:
      case S_MOSTNEG:
        {
          int s = symb;
          nextsymb ();
          switch (symb)
            {
              case S_BYTE: case S_INT: case S_INT16: case S_INT32: case S_INT64:
                {
                  int t = symb;
                  nextsymb ();
                  return (newmopnode (s, locn, newleafnode(t, locn), 0));
                }
              default:
                synerr_e (SYN_E_INTTYPE, locn, symb);
                return NULL;
            };
        }
      /*}}}*/
      /*{{{  case S_BOOL S_BYTE S_INT S_INTn S_REALn*/
      case S_BOOL: case S_BYTE:
      case S_INT: case S_INT16: case S_INT32: case S_INT64:
      case S_REAL32: case S_REAL64:
        {
          int s = symb;
          treenode *op;
          treenode *conversion;
          int t = S_EXACT;
      
          nextsymb ();
          if ((symb == S_ROUND) || (symb == S_TRUNC))
            {
              t = symb;
              nextsymb ();
              /*{{{  ignore line break*/
              if (checklinebreak ())
                return NULL;
              /*}}}*/
            }
          if ((op = roperand ()) == NULL)
            return NULL;
          conversion = newmopnode(t, locn, op, s);
          return (conversion);
        }
      /*}}}*/
      /*{{{  LBOX NAME TRUE FALSE BYTELIT UINTLIT UREALLIT STRING STRINGCONT LPAREN*/
      case S_LBOX: case S_NAME: case S_TRUE: case S_FALSE: case S_ASMNAME:
      case S_UBYTELIT: case S_UINTLIT: case S_UREALLIT:
      case S_STRING: case S_STRINGCONT: case S_LPAREN:
        {
          treenode *op = roperand ();
          return (op == NULL ? NULL : rrestofexp (op, locn));
        }
      /*}}}*/
      /*{{{  case S_SUBTRACT S_MINUS S_BITNOT S_NOT S_SIZE*/
      case S_SUBTRACT: case S_MINUS: case S_BITNOT: case S_NOT: case S_SIZE:
        {
          treenode *op;
          int s = symb;
          if (s == S_SUBTRACT)
            s = S_NEG;
          else if (s == S_MINUS)
            s = S_UMINUS;
          nextsymb ();
          if (checklinebreak ())
            return NULL;
          op = roperand ();
          return (op == NULL ? NULL : newmopnode (s, locn, op, 0));
        }
      /*}}}*/
      #if 0  /* #ifdef CONFIG */ /* This is all obsolete */
      /*{{{  case S_NODE*/
      case S_NODE :
        {
          treenode *e;
          nextsymb();
          if (checkfor (S_LPAREN))
            return NULL;
          if ((e = rlist (rexp, S_COMMA)) == NULL)
            return NULL;
          if (checkfor (S_RPAREN))
            return NULL;
          e = newmopnode (S_STRUCTCONSTRUCTOR, locn, e, S_NODE);
          return(e);
        }
      /*}}}*/
      #endif
      default:
        synerr_e (SYN_E_EXPR, locn, symb);
        return NULL;
    }
}
/*}}}*/
/*{{{  PRIVATE treenode *rspecorexpr (specflag, pindent, explist)*/
/* If (specflag), leaves symb on next line, otherwise leaves symb unchanged */
/*{{{  note on pindent*/
/* Set to the indent of left parenthesis if we have been called
from roperand, negative if called from anywhere else
(rchoice or ralternative). pindent is only used if we find
a VALOF: if it is negative it means that VALOF is not legal in
this context.
*/
/*}}}*/
/* If TRUE we may have an expression list (option) */
/*{{{  what we are parsing*/
/* Parse
expression
| specification
VALOF
| VALOF
and return the parse tree.
*specflag is TRUE  if a specification was found,
FALSE if an expression was found.
*/
/*}}}*/
PRIVATE treenode *rspecorexpr ( int *specflag , int pindent , int explist )
{
  SOURCEPOSN locn = flocn;
  int indent = symbindent;
  *specflag = FALSE;
  switch (symb)
    {
      /*{{{  S_VALOF                          valof*/
      case S_VALOF:
        {
          treenode *valof;
          if ((valof = rvalof ()) == NULL)
            goto error2;
          if (pindent >= 0)
            {
              if (checkindent (pindent)) goto error2;
              while (symb == S_COMMENT)      /* Comments after result statement but*/
                if (checknlindent (pindent)) /* before right bracket               */
                  goto error2;
              return (valof);
            }
          else
            {
              synerr_e (SYN_E_EXPR, locn, S_VALOF);
              goto error2;
            }
        }
      /*}}}*/
      /*{{{  S_BOOL S_BYTE S_INTn S_REALn     expression or specification*/
      case S_BOOL: case S_BYTE: case S_INT:
      case S_INT16: case S_INT32: case S_INT64: case S_REAL32: case S_REAL64:
        {
          int atag = symb;
          nextsymb ();
          if ((symb == S_TRUNC) || (symb == S_ROUND))
            /*{{{  conversion  so expression*/
            {
              treenode *op, *conversion;
              int ct = symb;
              nextsymb ();
              /*{{{  ignore line break*/
              if (checklinebreak ())
                return NULL;
              /*}}}*/
              if ((op = roperand ()) == NULL)
                goto error;
              conversion = newmopnode(ct, locn, op, atag);
              return (conversion);
            }
            /*}}}*/
          else
            /*{{{  could still be conversion or specification*/
            {
              if ((symb == S_COMMA) || (symb == S_FUNCTION) || (symb == S_INLINE))
                /*{{{  it must be a function definition*/
                {
                  treenode *a = newleafnode(atag, locn);
                  treenode *list;
                  *specflag = TRUE;
                
                  if (symb == S_COMMA)
                    /*{{{  read in the rest of the specifiers*/
                    {
                      nextsymb ();
                      if (checklinebreak ())
                        goto error2;
                      if ((list = rlist (rspecifier, S_COMMA)) == NULL)
                        goto error;
                      a = newlistnode (S_LIST, locn, a, list);
                    }
                    /*}}}*/
                  return (rfunctiondef (a, locn, indent));
                }
                /*}}}*/
              else
                {
                  if (symb != S_NAME)
                    /*{{{  it must be a conversion*/
                      {
                        treenode *op, *conversion;
                        if ((op = roperand ()) == NULL)
                          goto error;
                        conversion = newmopnode(S_EXACT, locn, op, atag);
                        return (conversion);
                      }
                    /*}}}*/
                  else
                    /*{{{  could still be conversion or specification*/
                    {
                      treenode *list = NULL;
                      wordnode *name = rname ();
                      if (symb == S_COMMA)
                        {
                          if (explist)
                            /*{{{  still don't know*/
                            {
                              nextsymb ();
                              if (checklinebreak ())
                                goto error2;
                              if ((list = rlist (rexp, S_COMMA)) == NULL)
                                goto error;
                              /* we have type in atag, first name in name, expression list in list */
                            }
                            /*}}}*/
                          else
                            /*{{{  it must be a declaration*/
                            {
                              treenode *list;
                              treenode *nptr;
                              *specflag = TRUE;
                              nextsymb ();
                              if (checklinebreak ())
                                goto error2;
                              if ((list = rlist ((treenode *(*)())rname, S_COMMA)) == NULL)
                                goto error;
                              nptr = newlistnode (S_LIST, locn, (treenode *)name, list);
                              return (rrestofspec (newleafnode(atag, locn), nptr, locn, indent));
                            }
                            /*}}}*/
                        }
                      /* We have type in atag, name in 'name' */
                      /*{{{  we have reached the IS COLON or RETYPES moment of truth for spec*/
                      if
                        /*{{{  next symbol is IS, RETYPES or COLON => specification*/
                        ((symb == S_IS) || (symb == S_RETYPES) || (symb == S_COLON))
                          {
                            *specflag = TRUE;
                            if (list != NULL)
                              list = newlistnode(S_LIST, locn, (treenode *)name, list);
                            else
                              list = (treenode *)name;
                            return (rrestofspec (newleafnode(atag, locn), list, locn, indent));
                          }
                        /*}}}*/
                      else
                        /*{{{  it was an expression after all*/
                        {
                          treenode *b;
                          if (list == NULL)
                            /*{{{  read the rest of this expression*/
                            {
                              if (symb == S_LBOX)
                                {
                                  if ((b = rsubscript ((treenode *)name)) == NULL) goto error;
                                }
                              else if (symb == S_LPAREN)
                                b = rinstance (S_FINSTANCE, locn, (treenode *)name);
                              else
                                b = (treenode *)name;
                              b = newmopnode(S_EXACT, locn, b, atag);
                            }
                            /*}}}*/
                          else
                            {
                              b = newlistnode(S_LIST, locn,
                                   newmopnode(S_EXACT, locn, (treenode *)name, atag),
                                   list);
                            }
                          return(b);
                        }
                        /*}}}*/
                      /*}}}*/
                    }
                    /*}}}*/
                }
            }
            /*}}}*/
        }
      /*}}}*/
      /*{{{  S_LBOX                           specification or expression*/
      case S_LBOX:
        /* Could be start of specifier in a specification,
           or segment in an expression, or constructor in an expression */
        {
          treenode *a;
          nextsymb ();
      #ifdef ARRAYCONSTRUCTOR
          if (symb == S_ARRAYCONSTRUCTOR)
            a = rarrayconstructor();
          else
      #endif
            {
              if ((a = rexp ()) != NULL)
                {
                  if (symb == S_RBOX)
                    /*{{{  constructor with one item, or specifier in specification*/
                    {
                      nextsymb ();
                      switch (symb)
                        {
                          /*{{{  S_LBOX or primitive type specifier in specification*/
                          case S_LBOX:
                          case S_CHAN: case S_PORT: case S_TIMER: case S_BOOL: case S_BYTE: case S_INT:
                          case S_INT16: case S_INT32: case S_INT64: case S_REAL32: case S_REAL64:
                            {
                              treenode *spec;
                              *specflag = TRUE;
                              if ((spec = rspecifier ()) == NULL)
                                goto error2;
                              a = newarraynode (S_ARRAY, locn, a, spec);
                              switch (symb)
                                {
                                  /*{{{  case S_NAME*/
                                  case S_NAME:
                                    {
                                      wordnode *name = rname ();
                                      treenode *nptr;
                                  
                                      if (symb == S_COMMA)
                                        {
                                          treenode *list;
                                          nextsymb ();
                                          if (checklinebreak ())
                                            goto error2;
                                          if ((list = rlist ((treenode *(*)())rname, S_COMMA)) == NULL)
                                            goto error;
                                          nptr = newlistnode (S_LIST, locn, (treenode *)name, list);
                                        }
                                      else
                                        nptr = (treenode *)name;
                                      return (rrestofspec (a, nptr, locn, indent));
                                    }
                                  /*}}}*/
                                  /*{{{  case S_COMMA*/
                                  case S_COMMA:
                                    {
                                      treenode *list;
                                      nextsymb ();
                                      if (checklinebreak ())
                                        goto error2;
                                      if ((list = rlist (rspecifier, S_COMMA)) == NULL)
                                        goto error2;
                                      a = newlistnode (S_LIST, locn, a, list);
                                      return (rfunctiondef(a, locn, indent));
                                    }
                                  /*}}}*/
                                  /*{{{  case S_FUNCTION*/
                                  case S_INLINE: case S_FUNCTION:
                                    return (rfunctiondef(a, locn, indent));
                                  /*}}}*/
                                  default:
                                    synerr_e (SYN_E_SPEC, locn, symb);
                                    goto error;
                                };
                            }
                          /*}}}*/
                          default:
                            a = newmopnode (S_CONSTRUCTOR, locn,
                                            newlistnode(S_LIST, locn, a, NULL), 0);
                        }
                    }
                    /*}}}*/
                  else if (symb == S_COMMA)
                    a = rconstructor(a);
                  else if (symb == S_FROM)
                    a = rsegment(a);
                  else
                    {
                      synerr_e(SYN_E_EXPR_OR_SPEC, locn, symb);
                      goto error;
                    }
                }
            }
          if (a == NULL)
            goto error;
          if (symb == S_LBOX)
            a = rsubscript (a);
          return (rrestofexp (a, locn));
        }
      /*}}}*/
      /*{{{  S_NAME                           specification or expression*/
      case S_NAME:
        /* Could be an abbreviation without a specifier, or an expression */
        {
          treenode *name = (treenode *)rname ();
          if (symb == S_IS)
            /*{{{  it was an abbreviation*/
            {
              *specflag = TRUE;
              return (rrestofspec (NULL, name, locn, indent));
            }
            /*}}}*/
          else
            /*{{{  it was an expression after all*/
            {
              treenode *a;
            
              if (symb == S_LBOX)
                {
                  if ((a = rsubscript (name)) == NULL)
                    goto error;
                }
              else if (symb == S_LPAREN)
                a = rinstance (S_FINSTANCE, locn, name);
              else
                a = name;
            
              return (rrestofexp (a, locn));
            }
            /*}}}*/
        }
      /*}}}*/
      default:
        if (mustbespec(symb))
          { *specflag = TRUE; return (rspecification()); }
        if (mustbeexp(symb))
          { return (rexp()); }
        synerr_e (SYN_E_EXPR_OR_SPEC, locn, symb);
      error:
        if (*specflag)
          nextline ();
      error2:
        if (*specflag)
          skiplines (indent);
        return NULL;
    }
}
/*}}}*/
/*}}}*/

/* These may return NULL, but will always move onto the next line */
/*{{{  declarations, abbreviations and retypes*/
/* All these functions actually declare items */
/*{{{  PUBLIC treenode *rvalabbr (t, name, locn, indent)*/

/* Pointer to a tree representing type */
/* Pointer to node representing name */
/* File location of start of abbreviation */
/* Indent of the specifier statement */
PUBLIC treenode *rvalabbr ( treenode *t , wordnode *name , SOURCEPOSN locn , int indent )
{
  treenode *e, *retptr;
  DEBUG_MSG(("rvalabbr... "));

  if ((e = rexp ()) == NULL)
    goto error;
  if (checkfor (S_COLON))
    goto error;
  if (checknlindent (indent))
    return NULL;
  retptr = declare (S_VALABBR, locn, t, name, e);
  return retptr;

error:
  nextline ();
  skiplines (indent);
  if (e != NULL)
    retptr = declare (S_VALABBR, locn, t, name, e);
  return NULL;
}
/*}}}*/
/*{{{  PRIVATE treenode *rvalretype (t, name, locn, indent)*/

/* Pointer to a tree representing type */
/* Pointer to node representing name */
/* File location of start of retype */
/* Indent of the retypes statement */
PRIVATE treenode *rvalretype ( treenode *t , wordnode *name , SOURCEPOSN locn , int indent )
{
  treenode *e, *retptr;

  DEBUG_MSG(("rvalretype... "));

  if ((e = rexp ()) == NULL)
    goto error;
  if (checkfor (S_COLON))
    goto error;
  if (checknlindent (indent))
    return NULL;

  retptr = declare (S_VALRETYPE, locn, t, name, e);
  return retptr;

error:
  nextline ();
  skiplines (indent);
  if (e != NULL)
    retptr = declare (S_VALRETYPE, locn, t, name, e);
  return NULL;
}
/*}}}*/
/*{{{  PUBLIC treenode *rrestofspec (t, namelist, locn, indent)*/
/* Tree structure representing specifier */
/* Tree structure representing name or namelist */
/* May (illegally)be an expression list, so check it */
/* File location of start of specification */
PUBLIC treenode *rrestofspec ( treenode *t , treenode *namelist , SOURCEPOSN locn , int indent )
{
  treenode *a;

  DEBUG_MSG(("rrestofspec... "));

  switch (symb)
    {
      /*{{{  case S_COLON   declaration*/
      case S_COLON:                                                 /* Declaration */
      /* Lets through unsized arrays, eg. [] INT b :
         These are thrown out by chk */
        {
          treenode *tptr;
      
          nextsymb ();
          if (checknlindent (indent))
            goto error2;
          tptr = newdeclnode (S_DECL, locn, NULL, NULL, NULL);
          if (TagOf(namelist) == S_LIST)
            {
              a = namelist;
              while (!EndOfList(a))
                /*{{{  declare name on left, move to right*/
                {
                  treenode *nptr = ThisItem(a);
                  if (TagOf(nptr) == S_NAME)
                    NewItem(declname(N_DECL, locn,
                                    (wordnode *)nptr, t, tptr), a);
                  else
                    {
                      synerr_e(SYN_E_SPEC, locn, TagOf(nptr));
                      goto error;
                    }
                  a = NextItem(a);
                }
                /*}}}*/
            }
          else
            namelist = declname(N_DECL, locn,
                               (wordnode *)namelist, t, tptr);
          SetDName(tptr, namelist);
          return (tptr);
        }
      /*}}}*/
      /*{{{  case S_IS case S_RETYPES :*/
      case S_IS: case S_RETYPES:
      #ifdef DECL_EQ      
      case S_EQ:
      #endif
        {
          int nodetype = (symb == S_IS ? S_ABBR :
      #ifdef DECL_EQ
                          symb == S_EQ ? S_DECL :
      #endif
                          S_RETYPE);
          if (TagOf(namelist) == S_NAME)
            {
              treenode *e;
              nextsymb ();
              if (checklinebreak () ||
                  ((e = roperand ()) == NULL) ||
                  checkfor (S_COLON))
                goto error;
              if (checknlindent (indent))
                goto error2;
              return declare (nodetype, locn, t,
                              (wordnode *)namelist, e);
            }
          else
            {
              synerr(SYN_TOO_MANY_NAMES, locn);
              goto error;
            }
        }
      /*}}}*/
      default:
        synerr_e(SYN_E_COLON_IS_RETYPES, flocn, symb);
      error:
        nextline ();
      error2:
        skiplines (indent);
        return NULL;
    }
}
/*}}}*/
/*{{{  PUBLIC treenode *rspecnameandbody(t, locn, indent)*/
/* Tree structure representing specifier */
/* File location of start of specification */
PUBLIC treenode *rspecnameandbody ( treenode *t , SOURCEPOSN locn , int indent )
{
  switch (symb)
    {
      /*{{{  case S_NAME*/
      case S_NAME:
        {
          wordnode *name = rname();
          treenode *nptr;
          if (symb == S_COMMA)
            {
              treenode *list;
              nextsymb ();
              if (checklinebreak ())
                goto error2;
              if ((list = rlist ((treenode *(*)())rname, S_COMMA)) == NULL)
                goto error;
              nptr = newlistnode(S_LIST, locn, (treenode *)name, list);
            }
          else
            nptr = (treenode *)name;
          return (rrestofspec(t, nptr, locn, indent));
        }
      /*}}}*/
      /*{{{  case S_COMMA*/
      case S_COMMA:
        {
          treenode *list;
          nextsymb ();
          if (checklinebreak ())
            goto error2;
          if ((list = rlist (rspecifier, S_COMMA)) == NULL)
            goto error;
          t = newlistnode(S_LIST, locn, t, list);
          return (rfunctiondef(t, locn, indent));
        }
      /*}}}*/
      /*{{{  case S_FUNCTION*/
      case S_FUNCTION: case S_INLINE :
        return (rfunctiondef(t, locn, indent));
      /*}}}*/
      default:
        synerr_e(SYN_E_NAME, locn, symb);
      error:
        nextline ();
      error2:
        skiplines (indent);
        return NULL;
    }
}
/*}}}*/
/*}}}*/

/*{{{  protocol definition and instance*/
/* Only protdef actually declares items */
/*{{{  PRIVATE treenode *rpsub ()*/
/* Read a subscripted protocol type */
/* On error, leave symb unchanged */
PRIVATE treenode *rpsub ( void )
{
  SOURCEPOSN locn = flocn;
  DEBUG_MSG(("rpsub... "));

  switch (symb)
    {
      /*{{{  case S_LBOX*/
      case S_LBOX:
        {
          treenode *e, *sub;
          nextsymb ();
          if ((e = rexp ()) == NULL) return NULL;
          if (checkfor (S_RBOX)) return NULL;
          if ((sub = rpsub ()) == NULL) return NULL;
          return (newarraynode (S_ARRAY, locn, e, sub));
        }
      /*}}}*/
      /*{{{  case S_BOOL S_BYTE S_INTn S_REALn*/
      case S_BOOL:case S_BYTE: case S_INT:
      case S_INT16: case S_INT32: case S_INT64: case S_REAL32: case S_REAL64:
        {
          int s = symb;
          nextsymb ();
          return (newleafnode (s, locn));
        }
      /*}}}*/
      default:
        synerr_e(SYN_E_PTYPE, flocn, symb);
        return NULL;
    }
}
/*}}}*/
/*{{{  PUBLIC treenode *rsimpleprotocol ()*/
/* On error, leave symb unchanged */
PUBLIC treenode *rsimpleprotocol ( void )
{
  treenode *s, *t;
  SOURCEPOSN locn = flocn;

  DEBUG_MSG(("rsimpleprotocol... "));
  switch (symb)
    {
      /*{{{  case S_REAL32 S_REAL64 S_BOOL*/
      case S_REAL32: case S_REAL64: case S_BOOL:
        s = newleafnode (symb, locn);
        nextsymb ();
        return (s);
      /*}}}*/
      /*{{{  case S_BYTE S_INT S_INT16 S_INT32 S_INT64*/
      case S_BYTE: case S_INT: case S_INT16: case S_INT32: case S_INT64:
        s = newleafnode (symb, locn);
        nextsymb ();
        if (symb == S_COLON2)
          /*{{{  read in array following*/
          {
            SOURCEPOSN alocn;
            nextsymb ();
            alocn = flocn;
            if (checkfor (S_BOX)) return NULL;
            switch (symb)
              {
                /*{{{  case S_BOOL S_BYTE S_INT S_INTn S_REALn*/
                case S_BOOL: case S_BYTE: case S_INT:
                case S_INT16: case S_INT32: case S_INT64: case S_REAL32: case S_REAL64:
                  t = newleafnode (symb, alocn);
                  nextsymb ();
                  break;
                /*}}}*/
                /*{{{  case S_LBOX*/
                case S_LBOX:
                  if ((t = rpsub ()) == NULL) return NULL;
                  break;
                /*}}}*/
                default:
                  synerr_e(SYN_E_PTYPE, flocn, symb);
                  return NULL;
              }
            s = newdopnode (S_COLON2, locn, s, newarraynode (S_ARRAY, alocn, NULL, t), 0);
          }
          /*}}}*/
        return (s);
      /*}}}*/
      /*{{{  case S_LBOX*/
      case S_LBOX:
        return (rpsub ());
      /*}}}*/
      default:
        synerr_e(SYN_E_PTYPE, locn, symb);
        return NULL;
    }
}
/*}}}*/
/*{{{  PUBLIC treenode *rtaggedprotocol ()*/
/* Terminates with symb at start of next line */
/* Parse
tagged.protocol { tagged.protocol }
where
tagged.protocol = tag { ',' simple.protocol }
*/
PUBLIC treenode *rtaggedprotocol ( void )
{
  wordnode *name;
  treenode *taglist;
  int indent = symbindent;
  SOURCEPOSN locn;

  linebreakindent = (-1);
  DEBUG_MSG(("rtaggedprotocol... "));
  while (symb == S_COMMENT)
    if (checknlindent (indent))
      goto error2;
  locn = flocn;
  /*{{{  read tagged list*/
  if ((name = rname ()) == NULL) goto error;
  if (symb == S_SEMICOLON)
    {
      nextsymb ();
      if (checklinebreak ()) goto error2;
      taglist = rlist (rsimpleprotocol, S_SEMICOLON);
      if (taglist == NULL) goto error;
    }
  else
    taglist = NULL;
  /*}}}*/
  checknewline ();
  return (declname (N_TAGDEF, locn, name, taglist, NULL));

error:
  nextline ();
error2:
  skiplines (indent);
  return NULL;
}
/*}}}*/
/*{{{  PRIVATE treenode *rprotdef ()*/
/* Terminates with symb at start of next line */
PRIVATE treenode *rprotdef ( void )
{
  int indent = symbindent;
  SOURCEPOSN locn = flocn;
  wordnode *name;

  DEBUG_MSG(("rprotdef... "));
  if (checkfor (S_PROTOCOL)) goto error;
  name = rname ();
  ignore (S_COMMENT);
  if (symb == S_NEWLINE)
    /*{{{  we have a tagged protocol*/
    {
      treenode *tlist;                    /* Pointer to list of tagged protocols */
      treenode *retptr;
    
      nextsymb ();
      if (checkindent (indent + 2))
        goto error2;
      while (symb == S_COMMENT)
        if (checknlindent (indent + 2))
          goto error2;
      if (checkfor (S_CASE))
        goto error;
      checknewline ();
    
      tlist = rproclist (rtaggedprotocol, indent + 4);
    
      if (checkindent (indent))
        goto error2;
      /*{{{  check for : but continue if not found*/
      if (checkfor (S_COLON))
        {
          skiplines (indent);
          goto cont;
        }
      /*}}}*/
      if (checknlindent (indent))
        goto error2;
    
    cont:
      retptr = declare (S_TPROTDEF, locn, tlist, name, NULL);
      return retptr;
    }
    /*}}}*/
  else
    /*{{{  we have a sequential protocol*/
    {
      treenode *p;                             /* Pointer to sequential protocol */
    
      if (checkfor (S_IS) ||
          checklinebreak () ||
          ((p = rlist (rsimpleprotocol, S_SEMICOLON)) == NULL) ||
          checkfor (S_COLON))
            goto error;
      if (checknlindent (indent))  goto error2;
    
      return declare (S_SPROTDEF, locn, p, name, NULL);
    }
    /*}}}*/

error:
  nextline ();
error2:
  skiplines (indent);
  if (symb == S_COLON)
    nextline ();
  return NULL;
}
/*}}}*/
/*{{{  PRIVATE treenode *rprotocol ()*/
/* On error, leaves protocol unchanged */
PRIVATE treenode *rprotocol ( void )
{
  DEBUG_MSG(("rprotocol... "));
  switch (symb)
    {
      case S_ANY:
        {
          SOURCEPOSN locn = flocn;
          nextsymb ();
          return (newleafnode (S_ANY, locn));
        }
      case S_NAME:
        return((treenode *)rname());
      default:
        return (rsimpleprotocol());
    }
}
/*}}}*/
/*}}}*/

/*{{{  PUBLIC treenode *rspecifier ()*/
/* On error, leaves symb unchanged */
/* Actually allows PORTs, although they are strictly illegal */
PUBLIC treenode *rspecifier ( void )
{
  SOURCEPOSN locn = flocn;
  treenode *s, *e;
  DEBUG_MSG(("rspecifier... "));
  switch (symb)
    {
      /*{{{  BOOL BYTE INT INT16 INT32 INT64 REAL32 REAL64*/
      case S_BOOL: case S_BYTE:
      case S_INT: case S_INT16: case S_INT32: case S_INT64:
      case S_REAL32: case S_REAL64: case S_TIMER:
      CASE_CONFIG_TYPE
        s = newleafnode (symb, locn);
        nextsymb ();
        return (s);
      /*}}}*/
      /*{{{  S_CHAN S_PORT*/
      case S_CHAN: case S_PORT:
        {
          treenode *prot;
          int s = symb;
          nextsymb ();
          if (checkfor (S_OF)) return NULL;
          if ((prot = rprotocol ()) == NULL) return NULL;
          return (newchannode (s, locn, prot));
        }
      /*}}}*/
      /*{{{  S_LBOX*/
      case S_LBOX:
        {
          treenode *spec;
          nextsymb ();
          if ((e = rexp ()) == NULL ||
              checkfor (S_RBOX) ||
              (spec = rspecifier ()) == NULL)
            return NULL;
          return (newarraynode (S_ARRAY, locn, e, spec));
        }
      /*}}}*/
      /*{{{  S_BOX*/
      case S_BOX:
        {
          treenode *spec;
          nextsymb ();
          if ((spec = rspecifier ()) == NULL) return NULL;
          return (newarraynode(S_ARRAY, locn, NULL, spec));
        }
      /*}}}*/
      default:
        synerr_e(SYN_E_SPECIFIER, flocn, symb);
        return NULL;
    };
}
/*}}}*/

/*{{{  procedure and function definitions*/
/* Only rprocdef and rfunctiondef actually declare items */
/*{{{  PRIVATE treenode *rparam(t, valparam)*/
/*****************************************************************************
 *
 *  rparam reads a parameter specification, ie. it parses
 *         [ [ 'VAL' ] specifier ] name
 *         t is the specifier of the previous parameter in the list (if this
 *         specifier is omitted) valparam is the tag of the previous parameter
 *         in the list.
 *
 *****************************************************************************/

/* On error, leaves symb unchanged */
PRIVATE treenode *rparam ( treenode *t , int valparam )
{
  SOURCEPOSN locn = flocn;
  wordnode *name;
  treenode *nptr;
  /*{{{  check for a new specifier*/
  switch (symb)
    {
      /*{{{  case S_VAL*/
      case S_VAL:                               /* Read in new VAL specifier */
        valparam = N_VALPARAM;
        nextsymb ();
        /*{{{  parse specifier*/
        if ((t = rspecifier ()) == NULL)
          return NULL;
        /*}}}*/
        break;
      /*}}}*/
      /*{{{  case S_LBOX BOOL BYTE INT INT16 INT32 INT64 REAL32 REAL64 CHAN PORT TIMER*/
      case S_LBOX: case S_BOX:
      case S_BOOL: case S_BYTE: case S_INT: case S_INT16: case S_INT32: case S_INT64:
      case S_REAL32: case S_REAL64:
      case S_CHAN: case S_PORT:
      case S_TIMER:                          /* Read in new var specifier */
        valparam = N_PARAM;
        /*{{{  parse specifier*/
        if ((t = rspecifier()) == NULL)
          return NULL;
        /*}}}*/
        break;
      /*}}}*/
      default:
        if (t == NULL)                           /* We have had no specifier */
          {
            synerr_e(SYN_E_SPECIFIER, locn, symb);
            return NULL;
          }
        else /* make a new copy of t */
          t = copytree(t);
    }
  /*}}}*/
  if ((name = rname()) == NULL) return NULL;
  nptr = declname(valparam, locn, name, t, NULL);
  return(newlistnode(S_LIST, locn, nptr, NULL));
}
/*}}}*/
/*{{{  PRIVATE treenode *rparmlist()*/
/*****************************************************************************
 *
 *  rparmlist reads in a formal parameter list,
 *            ie. it parses
 *              [ 'VAL' ] specifier name { ',' [ [ 'VAL' ] specifier ] name }
 *
 *****************************************************************************/
/* On error, leaves symb unchanged */
PRIVATE treenode *rparmlist ( void )
{
  treenode *tptr, *a;
  treenode *t = NULL;
  int valparam = N_PARAM;

  if ((tptr = rparam(t, valparam)) == NULL) return NULL;
  a = tptr;
  while (symb == S_COMMA)
    /*{{{  read another parameter*/
    {
      treenode *b;
      valparam = TagOf(ThisItem(a));
      t = NTypeOf(ThisItem(a));
      nextsymb();
      if (checklinebreak ()) return NULL;
      if ((b = rparam(t, valparam)) == NULL) return NULL;
      NewNextItem(b, a);
      a = b;
    }
    /*}}}*/
  return(tptr);
}
/*}}}*/
/*{{{  PRIVATE treenode *rdescriptorline ()*/
PRIVATE treenode *rdescriptorline ( void )
{
  treenode *n = NULL;
  wordnode *name = rname();
  if (name != NULL)
    {
      if (symb == S_INPUT || symb == S_OUTPUT)
        {
          n = newactionnode(symb, flocn, (treenode *)name, NULL);
          nextsymb();
          checknewline();
        }
      else
        {
          synerr(SYN_BAD_DESCRIPTOR, flocn);
          nextline();
        }
    }
  else
    nextline();
  return n;
}
/*}}}*/
/*{{{  PRIVATE treenode *rdescriptorbody (indent)*/
PRIVATE treenode *rdescriptorbody ( int indent )
{
  treenode *pbody;
  int tag;
  SOURCEPOSN locn;

  if (checknlindent(indent + 2)) goto error2;

  locn = flocn;
  if (symb == S_PRI)
    /*{{{  PRI PAR*/
    {
      tag = S_PRIPAR;
      nextsymb();
      if (checkfor(S_PAR)) goto error2;
    }
    /*}}}*/
  else
    /*{{{  SEQ*/
    {
      if (checkfor(S_SEQ)) goto error2;
      tag = S_SEQ;
    }
    /*}}}*/

  checknewline();
  pbody = rproclist(rdescriptorline, indent + 4);

  checkindent(indent);
  if (checkfor (S_COLON))
    return(NULL);
  pbody = newcnode(tag, locn, pbody);
  return(pbody);

error2:
  skiplines(indent);
  return NULL;
}
/*}}}*/
/*{{{  PRIVATE treenode *rprocdef ()*/

/* Terminates with symb at start of next line */
PRIVATE treenode *rprocdef ( void )
{
  int indent = symbindent;
  SOURCEPOSN locn = flocn;
  wordnode *name;
  treenode *params = NULL, *pbody;
  treenode *retptr;
  int inline = FALSE;
  int parsedheader = FALSE;
  int oldlexlevel = lexlevel;
  DEBUG_MSG(("rprocdef... "));
  foundroutine = TRUE;
  lexlevel++;

  /*{{{  parse and declare procedure*/
  /*{{{  read proc header*/
  /*{{{  inline ?*/
  if (symb == S_INLINE)
    {
      inline = allow_inlines;
      nextsymb();
    }
  /*}}}*/
  if (checkfor (S_PROC))
    goto error;
  if ((name = rname ()) == NULL)
    goto error;
  if (checkfor (S_LPAREN))
    goto error;
  if (symb != S_RPAREN)
    {
      if ((params = rparmlist()) == NULL)
        goto error;
    }
  if (checkfor (S_RPAREN))
    goto error;
  /*}}}*/
  parsedheader = TRUE;
  
  /*{{{  read body*/
  if (lexmode == LEX_SOURCE)
    {
      if (checknlindent (indent + 2)) goto error2;
      pbody = rprocess();
      if (checkindent (indent)) goto error2;
      while (symb == S_COMMENT)
        if (checknlindent (indent)) goto error2;
      if (checkfor (S_COLON)) goto error2;
    }
  else
    {
      pbody = rdescriptorbody(indent);
      if (pbody == NULL) goto error2;
    }
  /*}}}*/
  
  lexlevel = oldlexlevel;
  retptr = declare(S_PROCDEF, locn, params, name, pbody);
  if (inline)
    SetTag(DNameOf(retptr), N_INLINEPROCDEF);
  if (checknlindent (indent))
    {
      skiplines(indent);
      return(NULL);
    }
  return retptr;
  /*}}}*/

error:
  nextline ();
error2:
  msg_out_s(SEV_INFO, SYN, SYN_SKIPPING_DEFN, 0, "PROC");
  skiplines (indent);
  if (symb == S_COLON)
    nextline ();
  lexlevel = oldlexlevel;
  if (parsedheader && params != NULL)
    retptr = declare(S_PROCDEF, locn, params, name, skipp);
  return NULL;
}
/*}}}*/
/*{{{  PUBLIC treenode *rfunctiondef (typelist, locn, indent)*/
/* Terminates with symb at start of next line */
/* The return types of the function */
/* The file location of the function header */
/* The indent of the function header */
PUBLIC treenode *rfunctiondef ( treenode *typelist , SOURCEPOSN locn , int indent )
{
  int t;                                            /* Tag used for treenode */
  wordnode *name;                       /* Name of the function */
  treenode *params = NULL, *fbody;       /* Parameter list and function body */
  treenode *retptr;                              /* Parse tree node returned */
  treenode *fntype;                                     /* Funtion type node */
  int inline = FALSE;
  int oldlexlevel = lexlevel;
  DEBUG_MSG(("rfunctiondef... "));

  foundroutine = TRUE;
  lexlevel++;
  /*{{{  read function header*/
  /*{{{  inline ?*/
  if (symb == S_INLINE)
    {
      inline = allow_inlines;
      nextsymb();
    }
  t = S_LFUNCDEF;
  /*}}}*/
  if (checkfor (S_FUNCTION)     ||
      (name = rname ()) == NULL ||
      checkfor (S_LPAREN)       )
    goto error;
  if (symb != S_RPAREN)
    /*{{{  parse and declare the parameter list*/
    {
      if ((params = rparmlist()) == NULL)
        goto error;
    }
    /*}}}*/
  /*{{{  check for )*/
  if (checkfor (S_RPAREN))
    goto error;
  /*}}}*/
  /* convert typelist to a list, if not one already */
  if (TagOf(typelist) != S_LIST)
    typelist = newlistnode(S_LIST, locn, typelist, NULL);
  fntype = newlistnode (S_FNTYPE, locn, typelist, params);
  /*}}}*/
  if (symb == S_IS)
    /*{{{  short function definition*/
    {
      t = S_SFUNCDEF;
      nextsymb ();
      /*{{{  ignore line break*/
      if (checklinebreak ())
        goto error;
      /*}}}*/
      /*{{{  parse expression list*/
      if ((fbody = rlist (rexp, S_COMMA)) == NULL)
        {
          nextline ();
          goto error;
        }
      /*}}}*/
      fbody = newvalofnode (S_VALOF, locn, newleafnode (S_SKIP, locn),fbody);
      /*{{{  check for :*/
      if (checkfor (S_COLON))
        {
          nextline ();
          goto error;
        }
      /*}}}*/
    }
    /*}}}*/
  else
    /*{{{  long or descriptor function definition*/
    {
      t = S_LFUNCDEF;
      if (lexmode == LEX_SOURCE)
        /*{{{  long function definition*/
        {
          treenode **fbodyptr = &fbody;
          /*{{{  check nl, indent*/
          if (checknlindent (indent + 2))
            goto error;
          /*}}}*/
        
          while (ignorecomments (indent + 2),
                     symb != S_VALOF && symbindent == indent + 2)
            /*{{{  parse leading specification*/
            {
              *fbodyptr = rspecification ();
              if (*fbodyptr != NULL)
                fbodyptr = DBodyAddr(*fbodyptr);
            }
            /*}}}*/
        
          /*{{{  parse VALOF*/
          if ((*fbodyptr = rvalof ()) == NULL)
            goto error;
          /*}}}*/
          /*{{{  check indent*/
          if (checkindent (indent))
            goto error;
          /*}}}*/
          while (symb == S_COMMENT)
            /*{{{  check nl, indent*/
            if (checknlindent (indent))
              goto error;
            /*}}}*/
          /*{{{  check for :*/
          if (checkfor (S_COLON))
            goto error;
          /*}}}*/
        }
        /*}}}*/
      else
        /*{{{  descriptor function definition*/
        {
          fbody = rdescriptorbody(indent);
          if (fbody == NULL) goto error;
        }
        /*}}}*/
    }
    /*}}}*/
  lexlevel = oldlexlevel;

  retptr = declare (t, locn, fntype, name, fbody);
  if (inline)
    SetTag(DNameOf(retptr), N_INLINEFUNCDEF);
  if (checknlindent (indent))
    { skiplines(indent); return(NULL);}
  return retptr;

error:
  /*{{{  discard function definition*/
  msg_out_s(SEV_INFO, SYN, SYN_SKIPPING_DEFN, 0, "FUNCTION");
  skiplines (indent);
  if (symb == S_COLON)
    nextline ();
  lexlevel = oldlexlevel;
  return NULL;
  /*}}}*/

}
/*}}}*/
/*}}}*/


/*{{{  PUBLIC int rfile ()*/
PUBLIC int rfile ( void )
{
  int entered = FALSE;
  int s = symb;
  int indent = lineindent;
  nextsymb();
  if (symb != S_STRING)
    /*{{{  report error, skip to next line*/
    {
      synerr(SYN_MISSING_FILENAME, flocn);
      nextline();
    }
    /*}}}*/
  else if (s == S_SC)
    {
      synerr_s(SYN_SC_IS_OBSOLETE, flocn, literalv);
      nextline();
    }
  else
    /*{{{  try and open the file*/
    {
      int mode;
      switch (s)
        {
          case S_INCLUDE :
                mode = LEX_SOURCE;  /* #INCLUDE MUST specify the extension */
                break;
          default :
                /* S_SC, S_IMPORT or S_USE, all are descriptor files */
          mode = ((s == S_SC) ? LEX_SC : LEX_LIB);
                add_default_extension(literalv);
                break;
        }
      entered = open_file(literalv, mode, indent);
      if (!entered)
        /*{{{  report error, skip to next line*/
        {
          synerr_s(SYN_FILE_OPEN_ERROR, flocn, literalv);
          nextline();
        }
        /*}}}*/
      else
        {
          if (information)
            fprintf(outfile, "%s \"%s\"\n", tagstring(s), literalv);
          nextsymb();
          while (symb == S_NEWLINE)    /* Throw away leading blank lines */
            nextsymb();
          ignorecomments(0);           /* Throw away leading comments */
        }
    }
    /*}}}*/
  return entered;
}
/*}}}*/
/*{{{  PUBLIC int rpragma ()*/
PUBLIC int rpragma ( void )
{
  int ok = FALSE;
  wordnode *wptr;
#ifndef CONFIG
  int indent = lineindent;
#endif
  nextsymb();         /* skip past the #PRAGMA */
  wptr = rname();     /* if ok, this also skips past the name */
  if (wptr == NULL)
    nextline();       /* we just supplied an error message */
#ifndef CONFIG
  /* No #PRAGMA EXTERNAL when configuring, because that requires linking */
  else if (strcmp(WNameOf(wptr), "EXTERNAL") == 0)
    /*{{{  EXTERNAL*/
    { 
      if ((symb == S_STRING) || (symb == S_STRINGCONT))
        {
          treenode *str = rstring();
          ok = open_file(WNameOf(CTValOf(str)), LEX_EXTERNAL, indent);
          if (ok)
            {
              if (information)
                fprintf(outfile, "%s EXTERNAL \"%s\"\n", tagstring(S_PRAGMA), WNameOf(CTValOf(str)));
              nextsymb();
            }
          else
            nextline();
        }
      else
        {
          synwarn_s(SYN_BAD_PRAGMA_DIRECTIVE, flocn, "EXTERNAL");
          nextline();
        }
    }
    /*}}}*/
#endif
  else if (strcmp(WNameOf(wptr), "TRANSLATE") == 0)
    /*{{{  TRANSLATE*/
    {
      wptr = rname();  /* if ok, this also skips past the name */
      if (wptr != NULL)
        {
          if (symb == S_STRING)
            {
              ok = setup_translation(wptr, literalv, literalp);
              if (ok && information)
                fprintf(outfile, "%s TRANSLATE %s \"%s\"\n",
                  tagstring(S_PRAGMA), WNameOf(wptr), literalv);
            }
          else if (symb == S_STRINGCONT)
            synerr_i(SYN_STRING_TOO_LONG, flocn, MAXSTRING_SIZE);
          else
            synwarn_s(SYN_BAD_PRAGMA_DIRECTIVE, flocn, "TRANSLATE");
        }
      nextline();
    }
    /*}}}*/
#ifndef CONFIG
  /* No point in having LINKAGE pragma when configuring, cos we don't link
     after configuring - CON 11/10/90 */
  else if (strcmp(WNameOf(wptr), "LINKAGE") == 0)
    /*{{{  LINKAGE*/
    {
      switch (symb)
        {
          case S_STRING:  ok = setup_text_name(literalv, literalp);     break;
          case S_STRINGCONT:
            synerr_i(SYN_STRING_TOO_LONG, flocn, MAXSTRING_SIZE);
            ok = FALSE;
            break;
          /*case S_NEWLINE:*/   /* Now we allow anything, (eg comments) */
          default:        ok = setup_text_name(NULL, 0);                break;
          /*default:
                fwarningprint(SYN, SYN_BAD_PRAGMA_DIRECTIVE, flocn, (BIT32)"LINKAGE", ZERO32);
                break;
          */
        }
      if (ok && information)
        {
          fprintf(outfile, "%s LINKAGE", tagstring(S_PRAGMA));
          if (symb == S_STRING)
            fprintf(outfile, " \"%s\"", literalv);
          fprintf(outfile, "\n");
        }
      nextline();
    }
    /*}}}*/
#endif
  else
    {
      /*synerror(SYN_BAD_PRAGMA_NAME, flocn, (BIT32)symb);*/
      synwarn_s(SYN_BAD_PRAGMA_NAME, flocn, WNameOf(wptr));
      nextline();
    }
  return ok;
}
/*}}}*/
/* Skips to new line on error */
/*{{{  PUBLIC treenode *rspecification ()*/
/* Terminates with symb at start of next line */
PUBLIC treenode *rspecification ( void )
{
  int indent = symbindent;
  SOURCEPOSN locn = flocn;

  DEBUG_MSG(("rspecification... "));

  switch (symb)
    {
      /*{{{  case S_VAL*/
      case S_VAL:
        /* Parse
              'VAL' [specifier] name 'IS' expression ':'
           or 'VAL' specifier name 'RETYPES' expression ':'
        */
      {
        treenode *spec;
        wordnode *name;
        nextsymb ();
        /*{{{  read in specifier, if present*/
        switch (symb)
          {
            case S_BOOL: case S_BYTE:
            case S_INT: case S_INT16: case S_INT32: case S_INT64:
            case S_REAL32: case S_REAL64:
            case S_CHAN: case S_PORT: case S_LBOX:
            case S_TIMER: case S_BOX:
            CASE_CONFIG_TYPE
              /*{{{  parse specifier*/
              if ((spec = rspecifier ()) == NULL)
                goto error;
              /*}}}*/
              break;
            default:
              spec = NULL;
          };
        /*}}}*/
        /*{{{  parse name*/
        if ((name = rname ()) == NULL)
          goto error;
        /*}}}*/
        if (symb == S_IS)
          /*{{{  parse val abbreviation*/
          {
            nextsymb ();
            /*{{{  ignore line break*/
            if (checklinebreak ())
              goto error;
            /*}}}*/
            return (rvalabbr (spec, name, locn, indent));
          }
          /*}}}*/
        else if (symb == S_RETYPES)
          /*{{{  parse val retype*/
          {
            nextsymb ();
            /*{{{  ignore line break*/
            if (checklinebreak ())
              goto error;
            /*}}}*/
            return (rvalretype (spec, name, locn, indent));
          }
          /*}}}*/
        else
          /*{{{  error*/
          {
            synerr_e(SYN_E_IS_OR_RETYPES, locn, symb);
            goto error;
          }
          /*}}}*/
      }
      /*}}}*/
      /*{{{  case S_PROC*/
      case S_INLINE: case S_PROC:
        return (rprocdef ());
      /*}}}*/
      /*{{{  case S_PROTOCOL*/
      case S_PROTOCOL:
        return (rprotdef ());
      /*}}}*/
      /*{{{  case S_BOOL S_BYTE S_INT S_INTn S_REALn S_TIMER S_CHAN S_PORT ...*/
      case S_BOOL: case S_BYTE: case S_INT: case S_INT16:
      case S_INT32: case S_INT64: case S_REAL32: case S_REAL64:
      case S_CHAN: case S_PORT: case S_TIMER:
      case S_LBOX: case S_BOX:
      CASE_CONFIG_TYPE
        {
          /*{{{  COMMENT what we are parsing*/
          /* Parse
                 specifier name 'IS' element ':'                         |
                 specifier name 'RETYPES' element ':'                    |
                 type {1 ',' name } ':'                                  |
                 type name                                               |
                   init.body
                 ':'
                 {1 ',' type } 'FUNCTION' name '(' {0 ',' formal } ')'
                   function.body
                 ':'                                                     |
                 {1 ',' type } 'FUNCTION' name '(' {0 ',' formal } ')' 'IS' exprlist ':'
          
             types and specifiers are treated as equivalent
          */
          /*}}}*/
          treenode *t;
          /*{{{  parse specifier*/
          if ((t = rspecifier()) == NULL)
            goto error;
          /*}}}*/
          return rspecnameandbody(t, locn, indent);
        }
      /*}}}*/
      /*{{{  case S_NAME*/
      case S_NAME:
        {
          wordnode *name = rname();
          treenode *spec = NULL;     /* Null specifier */
          if (symb == S_IS)
            return (rrestofspec (spec, (treenode *)name, locn, indent));
          else
            {
              synetoken (S_IS);
              goto error;
            }
        }
      /*}}}*/
      /*{{{        case S_NETWORK:*/
      #ifdef CONFIG
      CASE_CONFIG_SPEC
        return (rconfigdef());
      #endif
      /*}}}*/
      /*{{{  S_PLACE*/
      case S_PLACE:
        {
          treenode *n;
          treenode *e = NULL;
          int tag;
          nextsymb();
          /*{{{  parse name*/
          if ((n = relement()) == NULL)
            goto error;
          /*}}}*/
          if (symb == S_AT)
            /*{{{  PLACE name AT [ WORKSPACE ] expression :*/
            {
              if (TagOf(n) != S_NAME)
                {
                  synerr(SYN_PLACE_NAME, flocn);
                  goto error;
                }
              nextsymb();
              if (symb == S_WORKSPACE)
                {
                  tag = S_WSPLACE;
                  nextsymb();
                }
              else
                tag = S_PLACE;
              if ((e = rexp()) == NULL)
                goto error;
            }
            /*}}}*/
          else if (symb == S_IN)
            /*{{{  PLACE name IN ( WORKSPACE | VECSPACE ) :*/
            {
              if (TagOf(n) != S_NAME)
                {
                  synerr(SYN_PLACE_NAME, flocn);
                  goto error;
                }
              nextsymb();
              if (symb == S_WORKSPACE)
                tag = S_WSPLACE;
              else if (symb == S_VECSPACE)
                tag = S_VSPLACE;
            #if 1 /*SHARED_VARS*/
              /* we allow "PLACE name IN PAR :" to indicate that it is
                 not to be usage checked.
                 We mark this by setting the 'expression' field of a S_VSPLACE
                 to be non-NULL
                 EXPERIMENTAL - CON 5/2/91
              */
              else if (symb == S_PAR)
                { tag = S_VSPLACE; e = dummyexp_p; }
            #endif
              else
                {
                  synerr_e(SYN_E_WS_OR_VS, flocn, symb);
                  goto error;
                }
              nextsymb();
            }
            /*}}}*/
      #ifdef CONFIG
          else if ((symb == S_ON) || (symb == S_COMMA))
            /*{{{  PLACE {1 ',' element} ON element :*/
            {
              treenode *list = NULL;
              if (symb == S_COMMA)
                {
                  nextsymb();
                  if (checklinebreak())
                    goto error2;
                  if ((list = rlist(relement, S_COMMA)) == NULL)
                    goto error;
                }
              n = newlistnode(S_LIST, flocn, n, list);
              if (checkfor(S_ON))
                goto error;
              if ((e = relement()) == NULL)
                goto error;
              tag = S_PLACEON;
            }
            /*}}}*/
      #endif
          else
            /*{{{  report error*/
            {
              synerr_e(SYN_E_AT_OR_IN, flocn, symb);
              goto error;
            }
            /*}}}*/
          /*{{{  check for :*/
          if (checkfor(S_COLON))
            goto error;
          /*}}}*/
          /*{{{  check nl, indent*/
          if (checknlindent(indent))
            goto error2;
          /*}}}*/
          return newdeclnode(tag, locn, n, e, NULL);
        }
      /*}}}*/
      /*{{{  S_INCLUDE S_SC S_USE etc*/
      case S_INCLUDE: case S_SC: case S_USE: case S_IMPORT:
        rfile ();
        return rspecification();
      /*}}}*/
      default:
        synerr_e(SYN_E_SPEC, locn, symb);
      error:
        nextline ();
      error2:
        skiplines (indent);
        return NULL;
    }
}
/*}}}*/
/*{{{  PUBLIC int rleadingspecs(specroot, exp)*/
/* Reads a specifications before a selection, alternative or choice
* return the first expression found.
* specs is left pointing to the root of the spec list.
* readerror is set true if error reading expression
*/
PUBLIC int rleadingspecs ( treenode **specroot , treenode **exp )
{
  treenode *s, **lasts = specroot;
  int indent = symbindent;
  DEBUG_MSG(("rleadingspecs... "));
  while (TRUE)
    switch (symb)
      {
        /*{{{  S_INCLUDE S_SC S_USE S_IMPORT*/
        case S_INCLUDE: case S_SC: case S_USE: case S_IMPORT:
          if (!rfile ())
            {
              skiplines(indent);
              if (lineindent < indent)
                { *exp = NULL; *lasts = NULL; return FALSE; }
            }
          break;
        /*}}}*/
        /*{{{  S_PRAGMA*/
        case S_PRAGMA:
          synerr(SYN_BAD_PRAGMA_POSN, flocn);
          nextline();
          break;
        /*}}}*/
        /*{{{  S_HCOMMENT*/
        case S_HCOMMENT:
          nextsymb();  /* string after comment */
          if (symb != S_STRING)
            {
              synerr(SYN_E_HCOMMENT, flocn);
              nextline();
            }
          else
            {
              process_hcomment(literalv, literalp);
              nextsymb (); /* should be end of line */
              if (checknlindent(indent))
                { skiplines(indent);
                  if (lineindent < indent)
                    { *exp = NULL; *lasts = NULL; return FALSE; }
                }
            }
          break;
        /*}}}*/
        /*{{{  primitive type S_NAME S_LBOX type = specification or expression*/
        case S_NAME: case S_LBOX: case S_BOOL: case S_BYTE:
        case S_INT: case S_INT16: case S_INT32: case S_INT64:
        case S_REAL32: case S_REAL64:
          {
            int specflag = FALSE; /* Assume no leading specification */
            s = rspecorexpr (&specflag, -1, FALSE);
            if (specflag)
              {
                *lasts = s;
                if (*lasts != NULL) lasts = DBodyAddr(*lasts);
              }
            else
              { /* exp or error reading exp */
                *lasts = NULL;
                *exp = s;
                return (s==NULL);
              }
            break;
          }
        /*}}}*/
        default :
          if (mustbespec(symb))
            /*{{{  read spec; break*/
            {
              *lasts = rspecification ();
              if (*lasts != NULL) lasts = DBodyAddr(*lasts);
              break;
            }
            /*}}}*/
          else if (mustbeexp(symb))
            /*{{{  read expression; return*/
            {
              *lasts = NULL;
              *exp = rexp ();
              return (*exp == NULL);
            }
            /*}}}*/
          DEBUG_MSG(("rleadingspecs: returning FALSE... "));
          *lasts = NULL;
          *exp = NULL;
          return FALSE;
      }
}
/*}}}*/
/*}}}*/

