/*#define DEBUG*/
/****************************************************************************
 *
 *  Error reporting
 *
 ****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include "includes.h"
# include "lexerror.h" /**/
# include "synerror.h" /**/
# include "chkerror.h" /**/
# include "useerror.h" /**/
# include "vtierror.h" /**/
# include "generror.h"
# include "lexconst.h" /**/
# include "lexdef.h" /**/
# include "lex1def.h" /**/
# include "syn1def.h" /**/
# include "popen.h"
/*}}}*/

/*{{{  global variables*/
PUBLIC int errors = 0;
PUBLIC int comp_error;
PUBLIC jmp_buf env;
/*}}}*/

#define MAX_ERRORS 100
#define TABSIZE 8

/*{{{  local variables*/
PRIVATE char *expectedstring = "Expected %s, found %s";
/*}}}*/

/*{{{  routines*/
/*{{{  PRIVATE char *anymessagestring*/
PRIVATE char *anymessagestring(const int n)
{
  switch(n)
    {
      case ANY_BAD_TAG:
        return "Unknown tag %s in %s";
      default:
        return NULL;
    }
}
/*}}}*/
/*{{{  PRIVATE char *synmessagestring (n)*/
PRIVATE char *synmessagestring ( const int n)
{
  switch (n)
    {      
      /*{{{  LEX errors*/
      case LEX_E_QUOTE:
        return ("Expected single quote, found \"%c\"");
      case LEX_E_DOUBLEQUOTE:
        return ("Expected double quotes, found \"%c\"");
      case LEX_E_HEXDIGIT:
        return ("Expected hex digit, found \"%c\"");
      case LEX_ILLEGAL_CHAR:
        return ("Illegal character (ASCII %d)");
      /*case LEX_FORMAT_ERROR:
        return ("Format error");*/
      case LEX_NUMBER_ERROR:
        /*return("In number");*/
        return("Expected digit, found \"%c\"");
      case LEX_SIGN_ERROR:
        return("Exponent must be signed");
      case LEX_EOF:
        return ("Unexpected end of file");
      case LEX_ILLEGAL_STRING_CHAR:
        return("Illegal character \"%c\" following \"*\" in string");
      case LEX_BAD_STRING_LENGTH:
        return("String too long to insert \"*l\" value");
      case LEX_BAD_STRING_CONTINUATION:
        return("Illegal string continuation");
      case LEX_BAD_INDENT:
        return("Incorrect indentation of string continuation");
      case LEX_BAD_STRING_LENGTH_POSN:
        return("'*l' must be at start of string");
      case LEX_READ_ERROR:
        return("Error code %d when reading file");
      case LEX_TOO_MANY_FILES:
        return("Files nested too deeply: maximum depth is %d");
      case LEX_FILE_OPEN_ERROR:
        return("Cannot open file \"%s\"");
      case LEX_FILE_LENGTH_ERROR:
        return("Maximum filename length is %d characters");
      /*
      case LEX_BAD_SCDESC_FORMAT:
        return("Bad descriptor file tag %d");
      case LEX_BAD_SC_COMPATIBILITY:
        return("Object code not compatible with current version (%s)");
      case LEX_BAD_SC_PROCTYPE:
        return("Object code compiled for incompatible target (%s)");
      case LEX_BAD_SC_ERRORMODE:
        return("Object code compiled in incompatible error mode (%s)");
      case LEX_TOO_MANY_NESTED_SCS:
        return("Cannot open \"%s\" - too many nested SCs");
      */
      case LEX_MISSING_ENTRYPOINT:
        return("Missing entrypoint \"%s\"");
      case LEX_OBJFILE_WRITE_ERROR:
        /*return("%d writing to object file");*/
        return("Cannot write to object file: error code %d");
      /* No longer relevant cos there's now no hard limit
      case LEX_TOO_MANY_ENTRYPOINTS:
        return("Too many entry points in file \"%s\"");
      */
      case LEX_BAD_DESC_FORMAT:
        return("Bad object file format");
      case LEX_LIBRARY_NOT_SC:
        return("\"%s\" is a library, not an SC");
      case LEX_HASHWORD_ERROR:
        return("Badly formed compiler directive");
      case LEX_TCOFF_IOERR :
        return ("TCOFF IO ERROR: 64 bit number!");
      case LEX_TCOFF_BAD_LCHAR :
        return ("Characters 'l%c' unknown in record string");
      case LEX_TCOFF_BAD_CHAR :
        return ("Character %c unknown in record string");
      case LEX_TCOFF_BAD_DESC :
        return ("Peculiar descriptor : \"%s\"");
      /*
      case LEX_TCOFF_BAD_PROCESSOR :
        return ("Unknown processor type in object file");
      case LEX_TCOFF_BAD_MODE :
        return ("Unknown error mode in object file");
      */
      case LEX_TCOFF_UNMATCHED_ENDMODULE :
        return ("Unmatched END_MODULE");
      case LEX_TCOFF_DESC_OVERFLOW :
        return ("Descriptor overflow");
      #ifdef CONFIG
      case LEX_TCOFF_NOT_LINKED:
        return ("\"%s\" is not a linked object file");
      #else
      case LEX_TCOFF_WRONG_FORMAT:
        return ("\"%s\" is not a valid object file");
      #endif
      case LEX_NAME_TRUNCATED:
        return("Name too long - maximum length is %d characters");
      case LEX_NUMBER_TRUNCATED:
        return("Number too long - maximum length is %d characters");
      /*}}}*/
#ifndef BACKEND      
      /*{{{  SYN errors*/
      case SYN_TOO_MANY_NAMES:
        return("One name only allowed in specification or retype");
      case LEX_STRING_ERROR:
        return("Illegal character (ASCII %d) in string");
      case LEX_CHAR_ERROR:
        return("Illegal character constant");
      /*case LEX_LINE_TRUNCATED:
        return("Line too long - maximum length is %d characters");*/
      case SYN_PROTOCOL_NOT_DECLARED:
        return ("Protocol is not declared");
      case SYN_ILLEGAL_CONVERSION:
        return ("Illegal conversion");
      case SYN_INVALID_PROCESS:
        return ("Invalid process");
      case SYN_BAD_INDENT:
        return ("Incorrect indentation");
      /*case SYN_ILLEGAL_TAG:
        return ("Illegal treenode tag");*/
      case SYN_ILLEGAL_DECL_TYPE:
        return ("Illegal type %d for declaration");
      case SYN_LIST_IN_IO:
        return ("Only one element allowed on left-hand side of input or output");
      case SYN_SC_ERROR:
        return ("Item not allowed at outermost level of compilation unit");
      case SYN_STRING_TOO_LONG:
        return ("String is too long for buffer (max %d bytes)");
      case SYN_FILE_OPEN_ERROR:
        return("Cannot open file \"%s\"");
      case SYN_MISSING_FILENAME:
        return("Missing filename");
      case SYN_BAD_GUYLINE:
        return("Badly formed code insert line");
      case SYN_GUY_NOT_ENABLED:
        return("Code insertion is not enabled");
      case SYN_OPTION_IN_BAD_POS:
        return("Option in illegal position");
      case SYN_BAD_OPTION:
        return("Unrecognised option \"%c\" in option string");
      case SYN_E_HCOMMENT:
        return("Expected string after #COMMENT");
      case SYN_E_OPTION:
        return("Expected string after #OPTION");
      case SYN_PROG_ERROR:
        return ("Item not allowed in PROGRAM unit");
      case SYN_BAD_DESCRIPTOR:
        return ("Descriptor has incorrect format");
      case SYN_SC_EMPTY:
        return ("No PROC or FUNCTION declared in SC module");
      case SYN_BAD_PRAGMA_NAME :
        return ("Unknown #PRAGMA name: %s");
      case SYN_BAD_PRAGMA_POSN:
        return ("#PRAGMA not allowed in this position");
      case SYN_BAD_PRAGMA_DIRECTIVE :
        return ("Badly formed #PRAGMA %s directive");
      /*case SYN_STDLIB_HAS_VS :
        return ("Standard library routine %s requires vectorspace");*/
      /*case SYN_DUPLICATE_IN_LIB :
        return ("Routine %s imported by multiple #USEs");*/
      case SYN_NOTHING_IN_LIB :
        return ("No compatible entrypoints found in \"%s\"");
      case SYN_TRANSLATE_NULL :
      /*return ("TRANSLATE ignored: String for %s contains NULL character");*/
        return ("TRANSLATE ignored: String contains NULL character");
      case SYN_TRANSLATE_DUPLICATE_IN :
        return ("TRANSLATE ignored: Name %s has already been used");
      case SYN_TRANSLATE_DUPLICATE_EX :
        return ("TRANSLATE ignored: String \"%s\" has already been used");
      case SYN_TRANSLATE_SEQUENCE :
        return ("TRANSLATE ignored: Module containing \"%s\" has already been loaded");
      case SYN_SC_IS_OBSOLETE :
        return ("#SC \"%s\" is obsolete; use #USE instead");
      case SYN_DIFFERENT_DESCRIPTORS:
        return ("Differing definitions of %s in library");
      case SYN_PLACE_NAME:
        return ("You may only PLACE a name");
      case SYN_SKIPPING_DEFN:
        return ("Skipping over %s definition");
      /*}}}*/
      /*{{{  expected "%s" found "%s" messages*/
      case SYN_E_TOKEN:  assert(FALSE); /* this is processed externally */
      case SYN_E_EXPR_OR_SPEC:    return "expression or specification";
      case SYN_E_GUARD_OR_SPEC:   return "guard or specification";
      case SYN_E_ELEMENT_OR_SPEC: return "element or specification";
      case SYN_E_SPEC:            return "specification";
      case SYN_E_PTYPE:           return "primitive type";
      case SYN_E_SPECIFIER:       return "specifier";
      case SYN_E_ELEMENT:         return "element";
      case SYN_E_OPERAND:         return "operand";
      case SYN_E_INTTYPE:         return "integer type";
      case SYN_E_EXPR:            return "expression";
      case SYN_E_IS_OR_RETYPES:   return "IS or RETYPES";
      case SYN_E_ACTION:          return "\"?\", \"!\" or \":=\"";
      case SYN_E_PAR_OR_ALT:      return "PAR or ALT";
      case SYN_E_RBOX_OR_FROM:    return "\"[\" or FROM";
      case SYN_E_AT_OR_IN:        return "AT or IN";
      case SYN_E_NAME:            return "name";
      case SYN_E_NEWLINE:         return "new line";
      case SYN_E_COLON_IS_RETYPES:return "\":\", IS or RETYPES";
      case SYN_E_WS_OR_VS:        return "WORKSPACE or VECSPACE";
      #ifndef CONFIG
      case SYN_E_PROCESSOR_TYPE:  return "processor type";
      #endif
      #if 0
      case SYN_E_LINK_DEL:        return "INTERVAL or DELETING";
      #endif
      /*}}}*/
      default:
        return NULL;
#endif
    }
}
/*}}}*/
/*{{{  PRIVATE char *chkmessagestring*/
PRIVATE char *chkmessagestring ( const int n )
{
  switch (n)
    {
      /*{{{  cases*/
      case CHK_UNKNOWN_TYPE:
        return ("Unknown type %d");
      case CHK_TYPES_DIFF:
        return ("Operands have different types for operator %s");
      case CHK_INVTYPE:
        return ("Type of operand invalid for operator %s");
      case CHK_CONSTRUCTOR_TYPE_MISMATCH:
        return ("Table elements are of different types");
      case CHK_INVCOUNT:
        return ("Shift count must be of type integer");
      case CHK_INV_EXACT:
        return ("Type of operand invalid for exact type conversion %s");
      case CHK_INV_ROUND_TRUNC:
        return ("Type of operand invalid for \'ROUND\' or \'TRUNC\'");
      case CHK_ASS_UNBALANCED:
        return ("Number of variables does not match number of expressions");
      case CHK_INV_SUBSCRIPT:
        return ("%s subscripted too many times");
      case CHK_INVTYPE_ASS:
        return ("Variable type does not match assigned expression %s");
      case CHK_ABBR_TYPE_MISMATCH:
        return ("Expression type does not match abbreviation specifier");
      /*case CHK_UNKNOWN_NODE:
        return ("Unknown node tag %d encountered");*/
      case CHK_INVTYPE_FRESULT:
       return ("Expression %s does not match type of corresponding function result");
      case CHK_TOOMANYEXPS:
        return ("Function result list contains too many expressions");
      case CHK_TOOFEWEXPS:
        return ("Function result list contains too few expressions");
      case CHK_RETYPE_TYPE_MISMATCH:
        return ("Size of retype does not match the expression size");
      case CHK_INV_FUNC_LIST:
        return ("Multi-valued function may not be used as expression operand");
      case CHK_INV_VALOF_LIST:
        return ("Multi-valued valof may not be used as expression operand");
      case CHK_ADIM_NOT_CONST:
        return ("Array dimension is not a constant expression");
      case CHK_ADIM_NOT_INT:
        return ("Array dimension is not of type INT");
      case CHK_INV_PROT:
        return ("Invalid channel protocol");
      /*case CHK_INVTYPETAG:
        return ("Invalid type tag %d");*/
      case CHK_TOO_FEW_VARS:
        return ("There are more expressions than variables in multiple assignment");
      case CHK_TOO_MANY_VARS:
        return ("There are more variables than expressions in multiple assignment");
      /*case CHK_INV_SIZE_OPERAND:
        return ("\'SIZE\' operand has invalid type");*/
      case CHK_CFOLD_OFLOW:
        return ("Overflow when evaluating constant expression");
      case CHK_NAMENOTCONST:
        return ("Constant expression contains non-constant name");
      case CHK_NAMENOTARRAY:
        /*return ("Subscripted operand is not an array");*/
        return ("Item may not be subscripted");
      case CHK_SUBSCRIPT_NOTCONST:
        return ("Constant expression contains non-constant subscript");
      case CHK_SUBSCRIPT_OUT_OF_RANGE:
        return ("Subscript out of range");
      case CHK_INV_DECL_TAG:
        return ("Invalid tag %d in decl node");
      case CHK_ADIM_NEGATIVE:
        return ("Array dimension must be greater than zero");
      case CHK_ADIM_TOOBIG:
        return ("Array dimension is too large");
      case CHK_INV_CONSTRUCT:
        return ("Invalid construct tag %d");
      case CHK_INV_LIST:
        return ("Invalid list tag %d");
      case CHK_INV_SEGMENT_OPERAND:
        return ("Segmented element is not an array");
      /*case CHK_SLENGTH_NOT_CONST:
        return ("Segment length not constant");*/
      case CHK_ADIM_MISSING:
        return ("Array type does not have dimension");
      case CHK_INVTYPE_SPEC:
        return ("Illegal specifier");
      /*case CHK_INV_EXP:
        return ("Invalid expression %d");*/
      case CHK_INVTYPE_CONSTEXP:
        return ("Constant expression has invalid type %d");
      case CHK_EXP_NOT_CONST:
        return ("Expression is not constant");
      case CHK_TOO_MANY_ELSES:
        return ("More than one ELSE in CASE process");
      case CHK_MULTIPLE_CASE:
        return ("Option is used more than once in this CASE");
      case CHK_INVTYPE_SELECTOR:
        return ("Invalid selector type");
      case CHK_SELN_NOTCONST:
        return ("Option must be a constant expression");
      case CHK_INVTYPE_SELN:
        return ("Type mismatch for CASE option");
      case CHK_UNRESOLVED_INTTYPE:
        return ("Integer number value must have a type specifier");
      case CHK_UNRESOLVED_REALTYPE:
        return ("Real number value must have a type specifier");
      case CHK_TYPE_MISMATCH:
        return ("Type mismatch");
      case CHK_ASUB_NOT_INT:
        return ("Array index expression must be of type INT");
      case CHK_FN_VAR_PARAM:
        return ("Formal parameter to function must be VAL");
      case CHK_PTYPE_MISMATCH:
        return ("I/O list item %d does not match protocol");
      case CHK_TOO_FEW_PITEMS:
        return ("Not enough items in i/o list");
      case CHK_TOO_MANY_PITEMS:
        return ("Too many items in i/o list");
      case CHK_BAD_PTAG:
        return ("Illegal tag in i/o list");
      case CHK_BAD_ANY_PROT:
        return ("Cannot send zero length segment on CHAN OF ANY");
      case CHK_INVTYPE_STARTEXP:
        return ("Replicator start expression not of type INT");
      case CHK_INVTYPE_LENGTHEXP:
        return ("Replicator count expression not of type INT");
      case CHK_REPL_NOT_CONST:
        return ("Implementation restriction: PAR replicator must have a constant count value");
      case CHK_REPL_NEGATIVE:
        return ("Negative replicator counts are invalid");
      case CHK_INVTYPE_GUARD:
        return ("Guard must be of type BOOL");
      case CHK_INV_VAL:
      /*return ("Invalid type specification for VAL parameter or abbreviation");*/
      #ifdef CONFIG
        return ("Cannot use VAL on a CHAN, PORT, TIMER, or hardware item");
      #else
        return ("Cannot use VAL on a CHAN, PORT, or TIMER");
      #endif
      case CHK_NAME_NOT_PROC:
        return ("%s is not a procedure");
      case CHK_INVVARIABLE_PARAM:
        return ("Parameter %d is not a variable");
      case CHK_INVTYPE_PARAM:
        return ("Type mismatch for parameter %d in call of %s");
      case CHK_TOO_FEW_ACTUALS:
        return ("Not enough actual parameters in call of %s");
      case CHK_TOO_MANY_ACTUALS:
        return ("Too many actual parameters in call of %s");
      case CHK_INV_CINDEX:
        return ("Invalid subscript into table");
      case CHK_NOT_CHANNEL:
        return ("Left-hand side must be of type CHAN or PORT");
      case CHK_INV_TIMER_INPUT:
        return ("Invalid timer input");
      case CHK_TIME_TYPE_MISMATCH:
        return ("Time expression must be of type INT");
      case CHK_NOT_CHANNEL_OR_TIMER:
        return ("Left-hand side must be of type CHAN, PORT, or TIMER");
      case CHK_NOT_TIMER:
        return ("Left-hand side must be of type TIMER");
      case CHK_INV_PORT_PROT:
        return ("Invalid port protocol");
      case CHK_INDISTINCT_TAG:
        return ("Tag name %s conflicts with another tag name");
      case CHK_NOT_A_TAG:
        return ("I/O list does not begin with a tag");
      case CHK_LOST_VARIANT:
        return ("Cannot find variant");
      case CHK_NAME_NOT_FUNCTION:
        return ("%s is not a function");
      case CHK_LOST_VALOF:
        return ("Lost valof, found %d");
      case CHK_VALOF_SCOPE:
        return ("%s is not assignable within VALOF");
      case CHK_BAD_VALOF:
        return ("%s is not allowed within a VALOF");
      case CHK_VALOF_CALL_SCOPE:
       return ("Instanced PROC assigns to free variable %s");
      case CHK_SUBSCRIPT_RANGE:
        return ("Subscript value %d is out of range");
      case CHK_SEG_START_RANGE:
        return ("Segment start value %d is out of range");
      case CHK_SEG_LENGTH_RANGE:
        return ("Segment length %d is longer than array");
      case CHK_SEG_RANGE:
        return ("Segment end value %d is out of range");
      /*case CHK_CONSTANT_SEGMENT_ERROR:
        return ("Error in constant segment folding, tag = %d,");*/
      case CHK_BAD_PLACE_NAME:
        return ("%s may not be placed");
      case CHK_INVTYPE_PLACEMENT:
        return("Placement address must be of type INT");
      case CHK_TOO_MANY_MULTIVALUES:
        return("Only one expression list allowed on right-hand side of assignment");
      case CHK_NAME_NOT_DECLARED:
        return("%s is not declared");
      case CHK_TAG_CONFLICT:
        return ("Tag name %s conflicts with protocol name");
      case CHK_BAD_FUNCTION_TYPE:
        return ("Function types must be scalar");
      case CHK_BAD_INPUT_PROTOCOL:
        return ("Simple input is not allowed on a channel with a tagged protocol");
      case CHK_BAD_CASE_INPUT_PROTOCOL:
      case CHK_BAD_TAGGED_INPUT_PROTOCOL:
        return("Case input is not allowed on a channel without a tagged protocol");
      case CHK_ALT_INPUT_PORT:
        return("PORT input is not allowed in an alternative");
      case CHK_BAD_WSPLACE:
        return("Only vector declarations may be placed");
      case CHK_NAME_MULTIPLY_PLACED:
        return("%s has already been placed");
      case CHK_MULTIPLE_VARIANT:
        return ("Tag %s is used more than once in case input");
      case CHK_TOO_MANY_TAGS:
        return ("Too many tags in protocol definition - %d allowed");
      case CHK_BAD_DEST:
        return ("Cannot write to %s");
      case CHK_BAD_COUNT:
        return ("counted array protocol not allowed on anarchic channel");
      case CHK_COUNT_OUT_OF_RANGE:
        return ("Count is out of range for I/O list item %d");
      /*case CHK_ENTRYPOINT_NOT_FOUND:
        return("Cannot find entry point %s in entry point list");*/
      /*case CHK_NAME_STACK_OVERFLOW:
        return("Implementation limit: maximum of %d names in scope");*/
      case CHK_UNIMPLEMENTED_ASMCODE:
        return("Implementation restriction: instruction %s not available");
      case CHK_BAD_GUY_OPERAND:
        return("Instruction operand must be a constant expression or a name");
      case CHK_DISABLED_GUYCODE:
        return("Instruction %s is not available in current code insertion mode");
      case CHK_INVALID_GUYCODE:
        return("Instruction %s is not available on target processor");
      case CHK_BAD_GUYCODE:
        return("Instruction %s does not exist");
      case CHK_BAD_STEPCODE:
        return("STEP instruction must be a secondary");
      case CHK_ASM_BAD_STORE:
        return("Operand to a store must be an element");
      case CHK_ASM_TOO_BIG:
        return ("Operand does not fit into a word");
      case CHK_MISSING_OPERAND:
        /*return("Instruction %s takes an operand");*/
        return("Instruction %s has too few operands");
      case CHK_EXTRANEOUS_OPERAND:
        /*return("Instruction %s does not take an operand");*/
        return("Instruction %s has too many operands");
      case CHK_BAD_VALOF_CALL:
        return ("Instanced PROC contains %s, not allowed within a VALOF");
      case CHK_INVTYPE_PROCEXP:
        return("Processor expression must be of type %s");
      #if 0  /* #ifndef CONFIG */
      case CHK_INVPROCTYPE :
        return("Unrecognised processor type '%s'\n");
      #endif
      case CHK_NESTED_PRI_PAR:
        return("PRI PAR is nested inside another PRI PAR");
      case CHK_NESTED_PRI_PROC:
        return("%s contains a nested PRI PAR");
      case CHK_INV_PRI_PAR:
        return("Implementation restriction: A PRI PAR must have two component processes");
      case CHK_NO_PRI_REPL_PAR:
        return("Implementation restriction: replicated PRI PAR not implemented");
      case CHK_UNKNOWN_CONSTRUCTOR_SIZE:
        return("Implementation restriction: table size must be known");
      case CHK_ASS_NAME_USED_MULTIPLY:
        return ("%s appears more than once on left-hand side of assignment");
      #ifdef CONDEXP
      case CHK_CONDEXP_NOT_BOOL :
        return("Condition in expression must be of type BOOL");
      #endif
      case CHK_INV_ABBR :
        return ("Right hand side of abbreviation must be assignable");
      case CHK_INV_RETYPE :
        /*return ("Retyping an item of type CHAN, PORT, or TIMER is not allowed");*/
        /*return ("Retyping an item of type TIMER is not allowed");*/
        return ("You may only RETYPE data items");
      case CHK_NAME_NOT_USED:
        return ("%s is not used");
      case CHK_PARAM_NOT_USED:
        return ("Parameter %s is not used");
      case CHK_ROUTINE_NOT_USED:
        return ("Routine %s is not used");
      /*}}}*/
      case CHK_BAD_VALRETYPE:
        return ("RETYPE cannot be VAL for CHAN, PORT or TIMER");
      case CHK_BAD_CHANRETYPE:
        /*return ("CHAN, PORT or TIMER may only be RETYPEd to CHAN etc");*/
        return ("RETYPE of CHAN or PORT must be VAL");
      case CHK_RETYPE_NOTYPE:
        return ("RETYPE must have a type specifier");
      case CHK_BAD_DEST_CONSTRUCTOR:
        return ("Cannot write to table");
      case CHK_BAD_DEST_STRING:
        return ("Cannot write to string");
      case CHK_INV_SHIFT:
        return ("Shift count is out of range");
      case CHK_INV_MULTIASSIGN:
        return ("Implementation restriction: arrays may not be unknown size");
      /*case CHK_LABEL_INSIDE_INLINE:
        return ("Implementation restriction: no labels inside INLINE routines");*/
      case CHK_NAME_IS_DESCOPED:
        return ("Name %s descopes a previous declaration");
      case CHK_ARRAY_SIZE_OVERFLOW:
        return ("Implementation restriction: array size is too large");
      case CHK_RSUB_NOT_FIELD:
      #ifdef CONFIG
        return ("Subscript expression is not a NODE attribute");
      #else
        return ("Subscript expression is not a field name of %s");
      #endif
      case CHK_LONELY_FIELD:
        return ("Field selector %s may not appear on its own");
      case CHK_LIB_WRONG_TARGET:
        return ("%s %s is not compiled for a compatible processor type or mode");
      case CHK_INV_ASSERT:
        return ("Parameter to ASSERT is FALSE");

#ifdef CONFIG
      case CHK_ILLEGAL_CONFIG_CONSTRUCT:
        return ("Illegal item in configuration code");
      case CHK_ILLEGAL_CONSTRUCT:
        return ("Illegal item inside %s construct");
      case CHK_DUPLICATE_CONSTRUCT:
        return ("Multiple %s constructs not permitted");
      case CHK_ZERO_CONSTRUCT:
        return ("No %s construct");

      case CHK_INVTYPE_EDGE:
        return("Edge expression must be of type %s");
      case CHK_INVTYPE_ARC:
        return("Arc expression must be of type %s");
      case CHK_INVTYPE_DEVICE:
        return("Node expression must be of type %s");
      case CHK_INVTYPE_ATTR:
        return("Unknown attribute name %s");
      case CHK_INV_MAPPING_RHS:
        return("Right hand side of mapping must be of type NODE or ARC");
      case CHK_INV_MAPPING_LHS:
        return("Left hand side of mapping must be of type %s");
      case CHK_INV_MAPPING_PRI:
        return("Priority expression of mapping must be of type INT");
      case CHK_INV_MAPPING_NOPRI:
        return("No priority expression permitted when mapping CHANs");
      case CHK_INV_PLACEON_RHS:
        return("Right hand side of mapping must be of type ARC");
      case CHK_ILLEGAL_PLACE:
        return("Cannot PLACE %s which was declared outside a PROCESSOR construct");
      case CHK_CALLING_CONVENTION:
        return("FUNCTION %s returns a REAL result but is compiled for wrong calling convention");

      case CONF_CANNOT_EVAL_LINKNUM:
        return("Cannot evaluate link number");
      case CONF_CANNOT_EVAL_EXP:
        return("Cannot evaluate expression");
      case CONF_EVAL_STOP:
        return("Process evaluates to STOP");
      case CONF_NAME_LINK_CONNECTED:
      /*return ("Link %s[link][%d] has already been connected");*/
        return ("Link %s[link][%s] has already been connected"); /* bug 1087 20/12/90 */
      case CONF_EDGE_CONNECTED:
        return ("EDGE %s has already been connected");
      case CONF_ARC_ALREADY_USED:
        return ("ARC %s has already been connected");
      case CONF_BAD_LINKNUM:
        return ("Link number %d is illegal");
      case CONF_CHAN_MULTIPLY_USED:
        return ("Channel %s used for %s by more than one process");
      case CONF_ILLEGAL_PROCTYPE:
        return ("Illegal processor type \"%s\"");
      case CONF_DUPLICATE_PROCTYPE:
        return ("Processor type has already been set for NODE %s");
      case CONF_NO_PROCTYPE:
        return ("Processor type has not been set for NODE %s");
      case CONF_NOT_CONNECTED:
        return ("%s %s is not connected to rest of network");
      case CONF_ILLEGAL_MEMVAL:
        return ("Illegal memory size: %d bytes");
      case CONF_DUPLICATE_MEMVAL:
        return ("Memory size has already been set for NODE %s");
      case CONF_NO_MEMVAL:
        return ("Memory size has not been set for NODE %s");
      case CONF_ILLEGAL_ROMSIZE:
        return ("Illegal ROM memory size: %d bytes");
      case CONF_DUPLICATE_ROMSIZE:
        return ("ROM memory size has already been set");
      case CONF_NO_ROMSIZE:
        return ("ROM memory size has not been set for root NODE");
      case CONF_DUPLICATE_ORDERINFO:
        return ("Ordering attribute has already been set");
      case CONF_INVALID_PROTOCOL:
        return ("Channel %s with protocol containing INT used across different wordlengths");
      case CONF_NO_ROUTE:
        return ("No hardware route exists from processor %s for channel %s");
      case CONF_FULL_ROUTE:
        return ("Not enough links from processor %s for channel %s");
      case CONF_ILLEGAL_PRIVAL:
        return ("PRI expression %d must evaluate to 0 or 1");
      case CONF_REQUIRE_PHYSICAL_NODE:
        return ("Right hand side of mapping must be a physical NODE");
      case CONF_NAME_ALREADY_MAPPED:
        return ("%s has already been mapped");
      case CONF_LOGICAL_IS_PHYSICAL:
        return ("%s has already been used as a physical NODE");
      case CONF_LOGICAL_NOT_MAPPED:
        return ("NODE %s has not been mapped");
      case CONF_ARC_NOT_CONNECTED:
        return("Channel %s mapped onto unconnected ARC %s");
      case CONF_ARC_WRONGLY_CONNECTED:
        return("Channel %s is mapped onto ARC %s, not connected to this processor");
      case CONF_ARC_FULL_INPUT:
        return("Too many channels inputting on ARC %s");
      case CONF_ARC_FULL_OUTPUT:
        return("Too many channels outputting on ARC %s");
      case CONF_ARC_NO_EDGE:
        return("Channel %s is used for both input and output, but ARC %s connects to an EDGE");
      case CONF_NO_ROOT:
        return("No NODE has been specified as root");
      case CONF_DUPLICATE_ROOT:
        return("Cannot set NODE %s as root; another has already been set");
      case CONF_ROOT_NOT_ARRAY:
        return("Implementation restriction; root NODE %s must not be an array element");
      case CONF_CHAN_NO_DIRECTION:
        return("No direction known for channel %s on PROCESSOR %s");
      case CONF_CHAN_ABBR_REPL:
        return("Implementation restriction: Cannot use replicator %s in channel abbreviation outside a PROCESSOR construct");
      case CONF_DECL_REPL:
        return("Implementation restriction: Cannot declare %s %s inside a replicator");
      case CONF_ORDER_IGNORED:
        return("Ordering attributes ignored because re-ordering isn't enabled");
      case CONF_NESTED_PRI_PAR:
        return("Process mapped at high priority may not contain a PRI PAR");
      case CONF_NO_HOST_CHAN:
        return("No channel has been placed onto the host connection");
      case CONF_SUBSCRIPT_OUT_OF_RANGE:
        return("Subscript out of range for channel %s");

      #if 0
      case CHK_INV_CONFIG:
        return ("Invalid configuration");

      case CHK_BAD_NETDECL :
        return("Invalid declaration in network description");
      case CHK_BAD_ATTRIBUTE :
        return("Invalid %s attribute");
      case CHK_MISSING_ATTRIBUTE :
        return("Missing %s attribute");
      case CHK_TOOMANY_ATTRIBUTES :
        return("Too many attributes for node");
      case CHK_MISSING_INIT:
        return ("Declaration must have initialisation");      
      #endif

#endif
      default:
        return (NULL);
    }
}
/*}}}*/
#ifndef BACKEND
/*{{{  PRIVATE char *usemessagestring (n, p1, p2)*/
PRIVATE char *usemessagestring ( const int n)
{
  switch (n)
    {
      /*{{{  cases*/
      /*case USE_BAD_TAG:
        sprintf(errorstring, "Illegal tag %d in routine %s",
                (long int)p1, (char *)p2);
        return(errorstring);*/
      /*case USE_GLOBAL_FREE_VARS:
        return("There are global free variables");*/
      case USE_VAR_VAR_ABBREVIATED:
        return ("'%s' cannot be used (it has been abbreviated)");
      case USE_VAR_VAL_ABBREVIATED:
        return ("'%s' cannot be altered (it has been VAL abbreviated)");
      case USE_BAD_CHAN_PARAM:
        return ("Channel parameter '%s' cannot be used for both input and output");
      case USE_BAD_FREE_CHAN:
        return ("Free channel '%s' cannot be used for both input and output");
      case USE_WRITTEN_IN_PAR:
        return ("Variable '%s' is assigned to in parallel");
      case USE_READ_AND_WRITTEN_IN_PAR:
        return ("Variable '%s' is read and assigned to in parallel");
      case USE_OUTPUT_IN_PAR:
      /*return ("Channel '%s' is output on in parallel");*/
        return ("Parallel outputs on channel '%s'");
      case USE_INPUT_IN_PAR:
      /*return ("Channel '%s' is input on in parallel");*/
        return ("Parallel inputs on channel '%s'");
      /* I've removed the fixed limit - CO'N 17/2/90
      case USE_TOO_MANY_OVERLAP_CHECKS:
        return("Too many overlap checks required: maximum is %d");
      */
      /*case USE_VAR_TABLE_SUBSCRIPTS :
        return "All subscripts in channel tables must be constant";*/
      case USE_VAR_FREEVAR:
      case USE_VAL_FREEVAR:
        return ("Aliasing error, '%s' is a free variable");
      /*}}}*/
      default:
        return (NULL);
    }
}
/*}}}*/
#endif
/*{{{  PRIVATE char *vtimessagestring (n, p1, p2)*/
PRIVATE char *vtimessagestring ( const int n)
{
  switch (n)
    {
      /*{{{  cases*/
      case VTI_OUT_OF_SPACE:
      /*return("Run out of symbol space");*/
      #ifdef CONFIG
        return("Not enough memory to configure this program");
      #else
        return("Not enough memory to compile this program");
      #endif
      case VTI_STACK_OVERFLOW:
        return("Workspace stack overflow");
      case VTI_STACK_UNDERFLOW:
        return("Workspace stack underflow");
      /*case VTI_BAD_TAG:
        return("Illegal tag %d in routine %s");*/
      case VTI_NAMETRANS_OVERFLOW :
        return("Name translation stack overflow");
      case VTI_NAMETRANS_UNDERFLOW :
        return("Name translation stack underflow");
      case VTI_TOO_MANY_NAMETRANS :
        return("Too many names for name translation");
      /*}}}*/
      default:
        return (NULL);
    }
}
/*}}}*/
/*{{{  PRIVATE char *genmessagestring (n)*/
PRIVATE char *genmessagestring ( const int n )
{
  switch (n)
    {
      /*{{{  cases*/
      case GEN_BAD_OPD:
        return("Bad operand mode %d in routine %s");
      case GEN_BAD_MOVEELEMENT_DIRN:
        return ("Bad dirn parameter %d to routine moveelement");
      case GEN_CODE_BUFFER_FULL:
        return ("Code buffer full (%d)");
      case GEN_LABEL_BUFFER_FULL:
        return ("Label buffer full (%d)");
      case GEN_MULTIPLY_DEFINED_LABEL:
        return ("Multiply defined label L%d");
      case GEN_ADJUST_ERROR:
        return ("Reference to undeclared label L%d");
      case GEN_ADJUST_TAG_ERROR:
        return ("Bad tag %d in routine adjust");
      case GEN_CANNOT_OPEN_OBJECT_FILE:
        return ("Cannot open object file: %s");
      /*case GEN_TOO_MANY_SEGMENTS:
        return("Too many entry points (> %d) declared");*/
      case GEN_BAD_CONSTANT_LOAD:
        return("Bad direction %d for moving a constant table in moveelement");
      /*case GEN_BAD_TAG:
        return ("Unknown tag %d in routine %s");*/
      case GEN_PLACE_OVERFLOW:
        return ("Placement expression for %s wraps around memory");
      /*case GEN_TOO_MANY_NESTED_ALTS:
        return("Implementation limit: ALT is nested too deeply, limit is %d");*/
      case GEN_TOO_MANY_VARS :
        return("Implementation limit: Too many variables (max %d) in routine %s");
      /*case GEN_TOO_MANY_STDLIB_NAMES:
        return("Too many entries in standard library, limit is %d");*/
      case GEN_MISSING_STDLIB_ENTRY:
        return("%s is not declared in the standard library");
      case GEN_BAD_PRIMARY_OPERAND:
        return("%s may only take a constant or label operand");
      case GEN_KERNEL_RUN_ERROR:
        return("Fourth parameter to KERNEL.RUN must be a constant >= 3");
      case GEN_CSE:
        return("Common subexpression elimination");
      case GEN_CANNOT_CLOSE_OBJECT_FILE:
        return ("Cannot close object file: %s");
      case GEN_INDEX_HASH_TABLE_ERR:
        return("Address not found in index hash table for node with tag %d");
      case GEN_UNSIZED_ABBREV:
        return("%s has unknown size");
      case GEN_RUN_TIME_OVERLAP_CHECK:
        return("Run-time disjointness check inserted");
      case GEN_RUN_TIME_OVERLAP_CHECKS:
        return("%d run-time disjointness checks inserted");
      case GEN_LINEMARK_LIST_ERR:
        return("Index not found in linemark list");
      case GEN_UNEXPECTED_EOF:
        return("Unexpected EOF when reading source file for source output");
      /*case GEN_LINE_NOT_IN_LIST:
        return("Line with line number %d and file number %d is not in lines list");*/
      case GEN_BAD_USECOUNT :
        return("Bad usecount for %s");
      case GEN_ERROR_IN_ROUTINE:
        return("Error %d in routine %s");
      case GEN_STDLIBS_NOT_ENABLED:
        return("Routine %s required from the standard library");
      case GEN_STDLIB_HAS_VS :
        return("Standard library routine %s requires vectorspace");
      case GEN_WORKSPACE_CLASH :
        return("Workspace clashes with variable PLACED AT WORKSPACE %d");
      case GEN_GUY_NO_CHANS :
        return("No access to channels permitted in GUY code; use ASM");
      case GEN_BAD_ASMNAME:
        return("Special name %s is not permitted at this position");
      case GEN_PLACE_ON_LINKS:
        return("Placement expression for %s clashes with interactive debugger");
      /*}}}*/
      #ifdef CONFIG
      case GEN_CONF_PATCH_TOO_SMALL:
        return("Cannot insert a patch of %d bytes");
      case GEN_CONF_CANT_FIND_PATCH:
        return("Cannot find patch at address %d");
      #endif
      default:
        return (NULL);
    }
}
/*}}}*/
#if 0  /* VCONF is no longer used */
/*{{{  PRIVATE char *vconfmessagestring (n)*/
PRIVATE char *vconfmessagestring ( const int n )
{
  switch (n)
    {
      /*{{{  cases*/
      case VCONF_DUPLICATE_PROCESSOR :
        return "Duplicate PROCESSOR %d";
      case VCONF_INPUT_OUPUT :
        return "Free channel '%s' cannot be used for both input and output";
      case VCONF_CHAN_INPUT_NOTOUTPUT :
        return "Free channel '%s' has no corresponding output";
      case VCONF_CHAN_OUTPUT_NOTINPUT :
        return "Free channel '%s' has no corresponding input";
      case VCONF_UNUSED_CHANNEL :
        return "Free channel '%s' is not used for input or output.";
      case VCONF_BAD_PROCESSOR_EXP :
        return "PROCESSOR expression is not constant";
      case VCONF_BAD_TAG :
        return "Illegal tag %d in routine %s";
      case VCONF_FAILED_OPEN :
        return "Failed to open routing data file %s";
      case VCONF_EOF :
        return "Unexpected end of routing data file";
      case VCONF_BAD_DATA :
        return "Bad data in routing data file";
      case VCONF_MISSING_ENTRY_POINT:
        return "Missing entry point %s in %s";
      case VCONF_TOO_MANY_VLIB_NAMES :
        return "There are more than %d entries in virtual channel library";
      case VCONF_NOCONFIG :
        return "No configuration after network description";
      case VCONF_STOP :
        return "Found a STOP in network description";
      case VCONF_SUBSCRIPT:
        return "Subscript out of range in network description";
      case VCONF_BAD_DEVICE:
        return "Unknown device %s";
      case VCONF_BAD_ATTRIBUTE:
        return "Bad or missing %s attribute";
      case VCONF_EXP_NOT_CONST:
        return "Expression is not constant in network description";
      case VCONF_BADLINK :
        return "Bad link number %d for this device";
      case VCONF_LINKCONNECTED :
        return "Link %d is already connected";
      case VCONF_BADINTERVAL :
        return "Interval for link %d is not distinct";
      case VCONF_DEADLOCK :
        return "Link %d on node %d may cause a deadlock";
      case VCONF_BAD_BLOCKSIZE :
        return "Bad or missing block size (%d)";
      /*}}}*/
      default:
        return (NULL);
    }
}
/*}}}*/
#endif
/*}}}*/
/*{{{  PRIVATE void printbuf (file, locn, indent, lines)*/
PRIVATE void printbuf ( FILE *const file, const SOURCEPOSN locn, const int indent)
{
  int filenum = FileNumOf(locn);
  int fileline = FileLineOf(locn);

  char line[LINEMAX];
  char full_name[MAX_FILENAME_LENGTH];
  FILE *fptr = popen_read(lookupfilename(filenum), pathname,
                          &full_name[0], POPEN_MODE_TEXT);
  if (fptr != NULL)
    {
      int i,j;
      for (i = 0; i < fileline - 5; i++)
        fgets (line, LINEMAX, fptr);
      j = (fileline < 5) ? fileline : 5;
      for (i = 1; i <= 9 && !feof(fptr); i++)
        /*{{{  print a line*/
        {
          fgets (line, LINEMAX, fptr);
          if (!feof(fptr))
            {
              char *s = line;
              int column = 0;
              int nomoretabs = FALSE;
              fprintf(file, (i == j) ? "%4d:" : "    :", fileline);
              if (strchr(line, '\t') == NULL)
                fputs(line, file);
              else
              while (*s != '\0')
                {
                  if (*s == ' ')
                    fputc(' ', file);
                  else if (*s != '\t')
                    { fputc(*s, file); nomoretabs = TRUE; }
                  else if (nomoretabs)
                    fputc(' ', file);
                  else
                    {
                      int j = (TABSIZE - (column % TABSIZE));
                      column += (j-1);
                      while (j > 0)
                        { fputc(' ', file); j--; }
                    }
                  column++; s++;
                }
              /* the following added for bug 1020 18/10/90 */
              if (strchr(line, '\n') == NULL)
                { /* it was a very long line; keep going! */
                  int c;
                  while ( (c = fgetc(fptr)) != EOF && c != '\n')
                    fputc(c, file);
                  fputc('\n', file);
                }

              if ((i == j) && (indent >= 0))
                /*{{{  print pointer to lexeme giving error*/
                {
                  int k;
                  for (k = 0; k < indent + 5; ++k)
                    fputc ('-', file);
                  fputs("^\n", file);
                }
                /*}}}*/
            }
        }
        /*}}}*/
      fclose (fptr);
    }
}
/*}}}*/
/*{{{  PUBLIC SOURCEPOSN sourcefileof*/
PUBLIC SOURCEPOSN sourcefileof(const SOURCEPOSN old_locn)
{
  SOURCEPOSN locn = old_locn;
  if (locn != NOPOSN)
    {
      int filenum = FileNumOf(locn);
      while (filemodeoffile(filenum) != LEX_SOURCE)
        {
          SetFileLine(locn, parentposnoffile(filenum));
          filenum = parentoffile(filenum);
          SetFileNum(locn, filenum);
        }
    }
  return locn;
}
/*}}}*/
/*{{{  PRIVATE char *get_msg*/
PRIVATE char *get_msg(const int module, const int n)
{
  char *s;
  /*fflush(outfile);*/ /* This is done in process_msg */
  switch (module)
    {
      case ANY_MODULE: s = anymessagestring(n); break;
      case SYN:        s = synmessagestring(n); break;
      case CHK:        s = chkmessagestring(n); break;
      case USE:        s = usemessagestring(n); break;
      case VTI:        s = vtimessagestring(n); break;
#if 0
      case VCONF:      s = vconfmessagestring(n); break;
#endif
      case GEN:        s = genmessagestring(n); break;
      default:         s = NULL;                break;
    }
  return (s == NULL) ? "" : s;
}
/*}}}*/
/*{{{  PRIVATE void process_msg*/
PRIVATE void process_msg(const int severity, const int module,
                         SOURCEPOSN locn, const char *const msg)
{
  int filenum  = FileNumOf(locn);
  int fileline = FileLineOf(locn);

  if ((module == SYN) && (symb == S_ILLEGALSYMB) && (severity != SEV_ERR_JMP))
    return;

  fflush(outfile); /* flush any normal output */

  if ((locn != NOPOSN) &&
      (severity > SEV_WARN) && (module != GEN) && (module != ANY_MODULE))
    {
      locn     = sourcefileof(locn);
      filenum  = FileNumOf(locn);
      fileline = FileLineOf(locn);

      if (!brieferrors)
        /*{{{  print out an extract of the offending file*/
        {
          fputc    ('\n', errfile);
          printbuf (errfile, locn, (module == SYN) ? symbindent : -1);
        }
        /*}}}*/
    }
  /*{{{  print heading*/
  {
    char *s;
    switch(severity)
      {
        case SEV_INFO:                          s = "Information"; break;
        case SEV_WARN:                          s = "Warning"; break;
        case SEV_ERR:      case SEV_ERR_JMP:    s = "Error"; break;
        case SEV_FATAL:    case SEV_ABORT:      s = "Fatal"; break;
        default:         /*case SEV_INTERNAL:*/ s = "Internal-error"; break;
      }
    fprintf(errfile, "%s-%s-", s, compilername);
    if (locn != NOPOSN)
      {
        fputs(lookupfilename(filenum), errfile);
        if (fileline != 0)
          fprintf(errfile, "(%d)", fileline);
        fputc('-', errfile);
      }
    fputc(' ', errfile);
  }
  /*}}}*/

  fputs(msg, errfile); fputc('\n', errfile);

  if (severity > SEV_WARN)
    {
      comp_error = TRUE;
      /*{{{  check to see if too many errors*/
      errors++;
      if (errors >= MAX_ERRORS)
        {
          fprintf (errfile, "Fatal-%s-Too many errors, maximum is %d\n", compilername, MAX_ERRORS);
          end_compiler(EXIT_FAILURE);
        }
      /*}}}*/
      if (module == GEN && !assembly_output && !disassemble && objfile != NULL)
        {
          fclose(objfile);
          remove(objfilename);
        }
    }
  /*{{{  Finish off */
  switch(severity)
    {
      case SEV_INFO: case SEV_WARN: case SEV_ERR:
        return;
      case SEV_ERR_JMP: case SEV_FATAL:
        switch_to_real_workspace();
        longjmp(env, TRUE);
        break;
      case SEV_ABORT:
        end_compiler(EXIT_FAILURE);
        break;
      case SEV_INTERNAL:
        fprintf(errfile, "Fatal-%s-Internal error - forcing stack dump for debugging\n", compilername);
        fclose(outfile);  /* Flush any diagnostics */
        assert(FALSE);
        break;
    }
  /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC void msg_out*/
PUBLIC void msg_out(const int severity, const int module, const int n, const SOURCEPOSN locn)
{
  process_msg(severity, module, locn, get_msg(module, n));
}
/*}}}*/
/*{{{  PUBLIC void msg_out_c*/
PUBLIC void msg_out_c(const int severity, const int module, const int n, const SOURCEPOSN locn, const char c)
{
  char msg[MAXSTRING_SIZE];
  sprintf(msg, get_msg(module, n), c);
  process_msg(severity, module, locn, msg);
}
/*}}}*/
/*{{{  PUBLIC void msg_out_i*/
PUBLIC void msg_out_i(const int severity, const int module, const int n, const SOURCEPOSN locn, const BIT32 d)
{
  char msg[MAXSTRING_SIZE];
  sprintf(msg, get_msg(module, n), d);
  process_msg(severity, module, locn, msg);
}
/*}}}*/
/*{{{  PUBLIC void msg_out_s*/
PUBLIC void msg_out_s(const int severity, const int module, const int n, const SOURCEPOSN locn, const char *const s)
{
  char msg[MAXSTRING_SIZE];
  sprintf(msg, get_msg(module, n), s);
  process_msg(severity, module, locn, msg);
}
/*}}}*/
#if 0 /* never used */
/*{{{  PUBLIC void msg_out_ii*/
PUBLIC void msg_out_ii(const int severity, const int module, const int n, const SOURCEPOSN locn,
                       const BIT32 d1, const BIT32 d2)
{
  char msg[MAXSTRING_SIZE];
  sprintf(msg, get_msg(module, n), d1, d2);
  process_msg(severity, module, locn, msg);
}
/*}}}*/
#endif
/*{{{  PUBLIC void msg_out_is*/
PUBLIC void msg_out_is(const int severity, const int module, const int n, const SOURCEPOSN locn,
                       const BIT32 d, const char *const s)
{
  char msg[MAXSTRING_SIZE];
  sprintf(msg, get_msg(module, n), d, s);
  process_msg(severity, module, locn, msg);
}
/*}}}*/
/*{{{  PUBLIC void msg_out_ss*/
PUBLIC void msg_out_ss(const int severity, const int module, const int n, const SOURCEPOSN locn,
                       const char *const s1, const char *const s2)
{
  char msg[MAXSTRING_SIZE];
  sprintf(msg, get_msg(module, n), s1, s2);
  process_msg(severity, module, locn, msg);
}
/*}}}*/
/*{{{  PUBLIC void msg_out_e*/
PUBLIC void msg_out_e(const int severity, const int module, const int n, const SOURCEPOSN locn,
                      const int tag)
{
  char found[MAXSTRING_SIZE];
  char msg[MAXSTRING_SIZE];
  ftagstring(found, tag);
  sprintf(msg, expectedstring, get_msg(module, n), found);
  process_msg(severity, module, locn, msg);
}
/*}}}*/
/*{{{  PUBLIC void synetoken*/
PUBLIC void synetoken(const int token)
{
  /* This performs the error for SYN_E_TOKEN */
  if (symb != S_ILLEGALSYMB)
    {
      char expected[MAXSTRING_SIZE], found[MAXSTRING_SIZE];
      char msg[MAXSTRING_SIZE];
      ftagstring(expected, token);
      ftagstring(found, symb);
      sprintf(msg, expectedstring, expected, found);
      process_msg(SEV_ERR, SYN, flocn, msg);
    }
}
/*}}}*/
/*{{{  PUBLIC void chk_invtype*/
PUBLIC void chk_invtype(const SOURCEPOSN locn, const int tag)
{
  char expected[MAXSTRING_SIZE];
  ftagstring(expected, tag);
  chkreport_s(CHK_INVTYPE, locn, expected);
}
/*}}}*/
