/*#define DEBUG*/
/******************************************************************************
*
*  configurer - hardware graph stuff
*
******************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1990
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
# include <string.h>
# include "includes.h"
# include "lexconst.h"
# include "lex1def.h"
# include "chkdef.h"
# include "chkerror.h"
# include "usehdr.h"
# include "usedef.h"
# include "use1def.h"
# include "confhdr.h"
# include "confdef.h"
# include "conf1def.h"
# include "conf2def.h"
# include "predefhd.h"
# include "syn3def.h"
/*}}}*/

/*{{{  global variables*/
PUBLIC int no_of_links;     /* used for checking the connections */
PUBLIC int processor_class; /* TRUE if current processor is a class */

PUBLIC int config_hw_dump     = FALSE;
PUBLIC int config_reorderable = FALSE;

PUBLIC treenode *proclist = NULL; /* list of all processors */
PUBLIC treenode *chanlist = NULL; /* list of all channels */
PUBLIC treenode *edgelist = NULL; /* list of all edges */
PUBLIC treenode *arclist  = NULL; /* list of all arcs */
PUBLIC connection_t *connectionchain = NULL;
 /* all connections chained together */
PUBLIC treenode *chanabbrlist = NULL; /* list of all chan abbreviations */

PUBLIC  prochwnode_t *root_processor = NULL;
PUBLIC  INT32 romsize = PROCNODE_INIT_MEMVAL;
PRIVATE prochwnode_t *romsize_processor;

PRIVATE int processing_mapping; /* TRUE while inside the MAPPING */

PRIVATE SOURCEPOSN order_flags_locn = NOPOSN;
/*}}}*/

/*{{{  c_decl abstractions */
/****************************************************************************/

/*{{{  new graph nodes*/
PRIVATE void init_name(namedesc_t *const name)
{
  name->n_name  = NULL;
  name->n_index = (-1);
}

PRIVATE void set_name(namedesc_t *const name, treenode *tptr, const int i)
{
  name->n_name  = tptr;
  name->n_index = i;
}

PUBLIC int empty_name(const namedesc_t *const name)
/*****************************************************************************
*
* empty_name returns TRUE if the name has not been set to anything
*
*****************************************************************************/
{
  return name->n_name == NULL;
}

PUBLIC int eq_names(const namedesc_t *const name1, const namedesc_t *const name2)
{
  return (name1->n_name  == name2->n_name) &&
         (name1->n_index == name2->n_index);
}

PRIVATE void init_edge(edgehwnode_t *const edge)
{
  edge->e_conn = NULL;
}

PRIVATE void set_edge(edgehwnode_t *const edge, connection_t *const conn)
/*****************************************************************************
*
* set_edge sets the edge as connected to the indicated name and link
*
*****************************************************************************/
{
  edge->e_conn = conn;
}

#if 0
PRIVATE int empty_edge(const edgehwnode_t *const edge)
/*****************************************************************************
*
* empty_edge returns TRUE if the edge has not been connected
*
*****************************************************************************/
{
  return edge->e_conn == NULL;
}
#else
#define empty_edge(e) ((e)->e_conn == NULL)
#endif

PUBLIC int empty_proc(const prochwnode_t *const proc)
/*****************************************************************************
*
* empty_proc returns TRUE if the processor hw description has not been used at all
*
*****************************************************************************/
{
  const int empty =    (proc->p_type      == NULL)
                    && (proc->p_mem       == PROCNODE_INIT_MEMVAL)
                    /* added bug 1145 5/2/91: */
                    && (proc->p_ordercode == DEFAULT_ORDER_VALUE)
                    && (proc->p_ordervs   == DEFAULT_ORDER_VALUE);
  if (empty)
    {
      int i;
      for (i = 0; i < MAX_PROC_LINKS; i++)
        if (!empty_edge(&(proc->p_edge[i])))
          return FALSE;
      return TRUE;  /* unused processor */
    }
  return FALSE;
}
PRIVATE void init_prochwnode(prochwnode_t *const p, treenode *const tptr, const int subscript)
{
  static prochwnode_t initnode; /* initialised to NULL etc */
  int i;
  *p = initnode;
  for (i = 0; i < MAX_PROC_LINKS; i++)
    init_edge(&(p->p_edge[i]));
  p->p_mem       = PROCNODE_INIT_MEMVAL;
  p->p_ordercode = DEFAULT_ORDER_VALUE;
  p->p_ordervs   = DEFAULT_ORDER_VALUE;
  set_name(&(p->p_name), tptr, subscript);
}
PRIVATE void init_procswnode(procswnode_t *const p, treenode *const tptr, const int subscript)
{
  /* this will be initialised to all zeroes and NULLs */
  static procswnode_t initnode;
  (*p) = initnode;
  set_name(&(p->p_name), tptr, subscript);
}
PRIVATE c_declnode_t *newc_declnode(const c_declnode_tag tag, void *const hwnodes, void *const swnodes)
/*****************************************************************************
*
* newc_declnode sets up a new c_declnode
*
*****************************************************************************/
{
  c_declnode_t *ptr =
 (c_declnode_t *)newvec(sizeof(c_declnode_t));
  ptr->c_nodetype = tag;
  ptr->c_hwnodes  = hwnodes;
  ptr->c_swnodes  = swnodes;
  return ptr;
}
PRIVATE c_declnode_t *newprocnode(treenode *const tptr, const INT32 elements)
/*****************************************************************************
*
* newprocnode sets up an [elements]NODE array
*
*****************************************************************************/
{
  /* These are left using memalloc, because they may be freed if the
     node turns out to be a logical node */
  prochwnode_t *const hwptr =
 (prochwnode_t *)memalloc((size_t)(sizeof(prochwnode_t)*elements));
  procswnode_t *const swptr =
 (procswnode_t *)memalloc((size_t)(sizeof(procswnode_t)*elements));
  int subscript;
  for (subscript = 0; subscript < elements; subscript++)
    {
      init_prochwnode(&(hwptr[subscript]), tptr, subscript);
      init_procswnode(&(swptr[subscript]), tptr, subscript);
    }
  return newc_declnode(C_DECLNODE_DEVICE, hwptr, swptr);
}
PRIVATE c_declnode_t *newedgehwnode(const INT32 elements)
/*****************************************************************************
*
* newedgehwnode sets up an [elements]EDGE array
*
*****************************************************************************/
{
  edgehwnode_t *const ptr =
 (edgehwnode_t *)newvec((size_t)(sizeof(edgehwnode_t)*elements));
  int subscript;
  for (subscript = 0; subscript < elements; subscript++)
    init_edge(&ptr[subscript]);
  return newc_declnode(C_DECLNODE_EDGE, ptr, NULL);
}
PRIVATE c_declnode_t *newarchwnode(treenode *const tptr, const INT32 elements)
/*****************************************************************************
*
* newarchwnode sets up an [elements]ARC array
*
*****************************************************************************/
{
  static archwnode_t initnode; /* initialised to NULL pointers, etc */
  archwnode_t *const ptr =
 (archwnode_t *)newvec((size_t)(sizeof(archwnode_t)*elements));
  int subscript;
  for (subscript = 0; subscript < elements; subscript++)
    {
      ptr[subscript] = initnode;
      set_name(&(ptr[subscript].a_name), tptr, subscript);
    }
  return newc_declnode(C_DECLNODE_ARC, ptr, NULL);
}
PRIVATE c_declnode_t *newchanswnode(treenode *const tptr, const INT32 elements)
/*****************************************************************************
*
* newchanswnode sets up an [elements]CHAN array
*
*****************************************************************************/
{
  static chanswnode_t initnode; /* initialised to NULL pointers, etc */
  chanswnode_t *const ptr =
 (chanswnode_t *)newvec((size_t)(sizeof(chanswnode_t)*elements));
  int subscript;
  for (subscript = 0; subscript < elements; subscript++)
    {
      chanswnode_t *thisnode = &(ptr[subscript]);
      *thisnode = initnode;
      set_name(&(thisnode->c_name), tptr, subscript);
      init_name(&(thisnode->c_dirn[DIRN_IN ].c_id));
      init_name(&(thisnode->c_dirn[DIRN_OUT].c_id));
    }
  return newc_declnode(C_DECLNODE_CHAN, NULL, ptr);
}
PRIVATE connection_t *newconnection(void)
{
  connection_t *const ptr =  (connection_t *)newvec(sizeof(connection_t));
  ptr->c_next = connectionchain;
  connectionchain = ptr;
  return ptr;
}
/*}}}*/
/*{{{  PRIVATE void removefromlist*/
PRIVATE void removefromlist(treenode **list, treenode *const t)
{
  while (!EndOfList(*list))
    {
      if (ThisItem(*list) == t)
        {
          *list = NextItem(*list);
          return;
        }
      list = NextItemAddr(*list);
    }
}
/*}}}*/
/*{{{  PRIVATE void convert_to_logical*/
PRIVATE void convert_to_logical(treenode *const name, const SOURCEPOSN locn)
/*
  convert a NODE declaration from a physical node structure into a
  logical node structure.
*/
{
  int i;
  const INT32 elements = elementsin(NTypeOf(name));
  c_declnode_t *const hwnode   = NVGraphOf(name);
  prochwnode_t *procnode = (prochwnode_t *)hwnode->c_hwnodes;
  logicalprocnode_t *logicals;

  if (config_info)
    fprintf(outfile, "Converting %s to logical NODEs\n", WNameOf(NNameOf(name)));

  for (i = 0; i < elements; i++)
    if (!empty_proc(&procnode[i]))
      {
        char str[MAXSTRING_SIZE];
        name_to_str(str, &procnode[i].p_name);
        chkerr_s(CONF_LOGICAL_IS_PHYSICAL, locn, str);
      }

  memfree(procnode);
  hwnode->c_hwnodes = NULL;
  memfree(hwnode->c_swnodes);

  /* The resulting logical can never be freed, so we can 'newvec' it rather
     than use memalloc */
  logicals = (logicalprocnode_t *)newvec((size_t)(sizeof(logicalprocnode_t)*elements));

  for (i = 0; i < elements; i++)
    logicals[i].p_translation = NULL;

  hwnode->c_swnodes = logicals;
  hwnode->c_nodetype = C_DECLNODE_LOGICAL;

  removefromlist(&proclist, name);
}
/*}}}*/
/*{{{  PRIVATE chanabbrnode_t *setup_abbreviations*/
PRIVATE chanabbrnode_t *setup_abbreviations(treenode *tptr, chanabbrnode_t *abbrs)
{
  if (abbrs != NULL)
  switch(TagOf(tptr))
    {
      case S_CONSTRUCTOR:
        DEBUG_MSG(("setup_abbreviations: constructor\n"));
        tptr = OpOf(tptr);
        while (!EndOfList(tptr))
          {
            abbrs = setup_abbreviations(ThisItem(tptr), abbrs);
            tptr = NextItem(tptr);
          }
        break;
      default:
        {
          int len;
          namedesc_t name;
          map_to_basename(tptr, &name);
          if (name.n_index < 0) return NULL;
          if (TagOf(NTypeOf(name.n_name)) == S_ARRAY)
            {
              INT32 first, last;
              subscripts_accessed(tptr, &first, &last, FALSE);
              len = last - first + 1;
            }
          else
            {
              len = 1;
            }
          DEBUG_MSG(("setup_abbreviations: element [%s FROM %ld FOR %ld]\n",
                     WNameOf(NNameOf(name.n_name)), name.n_index, len));
          while (len > 0)
            {
              abbrs->c_abbr = channodeof(&name);
              name.n_index++;
              abbrs++;
              len--;
            }
        }
        break;
    }
  return abbrs;
}
/*}}}*/
/*{{{  PRIVATE void abbreviate_chan*/
PRIVATE void abbreviate_chan(treenode *name, treenode *rhs)
{
  INT32 elements = elementsin(NTypeOf(name));
  chanabbrnode_t *abbrs =
 (chanabbrnode_t *)newvec((size_t)(sizeof(chanabbrnode_t)*elements));
  SetNVGraph(name, newc_declnode(C_DECLNODE_CHANABBR, NULL, abbrs));
  {
    treenode *list = expand_repllist;
    while (!EndOfList(list))
      {
        DEBUG_MSG(("abbreviate_chan: channame: %s, looking for repl %s\n",
             WNameOf(NNameOf(name)), WNameOf(NNameOf(ThisItem(list)))));
        if (name_used_in_exp(ThisItem(list), rhs))
          chkerr_s(CONF_CHAN_ABBR_REPL, LocnOf(rhs), WNameOf(NNameOf(ThisItem(list))));
        list = NextItem(list);
      }
  }
  chanabbrlist = addtofront(name, chanabbrlist);
  (void) setup_abbreviations(rhs, abbrs);
}
/*}}}*/
/*{{{  selecting nodes*/
/*{{{  PUBLIC prochwnode_t *prochwnodeof*/
PUBLIC prochwnode_t *prochwnodeof(const namedesc_t *const name)
/*****************************************************************************
*
* prochwnodeof returns the hw procnode for a given name descriptor
*
*****************************************************************************/
{
  c_declnode_t *hwnode   = NVGraphOf(name->n_name);
  prochwnode_t *procnode = (prochwnode_t *)hwnode->c_hwnodes;
  return &(procnode[name->n_index]);
}
/*}}}*/
/*{{{  PUBLIC int valid_procswnode*/
PUBLIC int valid_procswnode(const namedesc_t *const name)
{
  c_declnode_t *hwnode = NVGraphOf(name->n_name);

  if (hwnode->c_nodetype == C_DECLNODE_LOGICAL)
    {
      logicalprocnode_t *logicalnode = (logicalprocnode_t *)hwnode->c_swnodes;
      return (logicalnode[name->n_index].p_translation != NULL);
    }
  else
    return TRUE;
}
/*}}}*/
/*{{{  PUBLIC INT32 procswnode_priority*/
PUBLIC INT32 procswnode_priority(const namedesc_t *const name)
{
  c_declnode_t *hwnode = NVGraphOf(name->n_name);

  if (hwnode->c_nodetype == C_DECLNODE_LOGICAL)
    {
      logicalprocnode_t *logicalnode = (logicalprocnode_t *)hwnode->c_swnodes;
      return logicalnode[name->n_index].p_priority;
    }
  else
    return PRIORITY_LO;
}
/*}}}*/
/*{{{  PUBLIC procswnode_t *procswnodeof*/
PUBLIC procswnode_t *procswnodeof(const namedesc_t *const name)
/*****************************************************************************
*
* procswnodeof returns the sw procnode for a given name descriptor
*
*****************************************************************************/
{
  c_declnode_t *hwnode = NVGraphOf(name->n_name);
  if (hwnode->c_nodetype == C_DECLNODE_LOGICAL)
    {
      logicalprocnode_t *logicalnode = (logicalprocnode_t *)hwnode->c_swnodes;
      procswnode_t *procnode = logicalnode[name->n_index].p_translation;
      if (procnode == NULL)
        {
          char str[MAXSTRING_SIZE];
          name_to_str(str, name);
          msg_out_s(SEV_FATAL, CHK, CONF_LOGICAL_NOT_MAPPED, NOPOSN, str);
        }
      return procnode;
    }
  else
    {
      procswnode_t *procnode = (procswnode_t *)hwnode->c_swnodes;
      return &(procnode[name->n_index]);
    }
}
/*}}}*/
/*{{{  PRIVATE edgehwnode_t *edgenodeof*/
PRIVATE edgehwnode_t *edgenodeof(const namedesc_t *const name)
/*****************************************************************************
*
* edgenodeof returns the edgenode for a given name descriptor
*
*****************************************************************************/
{
  c_declnode_t *hwnode   = NVGraphOf(name->n_name);
  edgehwnode_t *edgenode = (edgehwnode_t *)hwnode->c_hwnodes;
  return &(edgenode[name->n_index]);
}
/*}}}*/
/*{{{  PRIVATE archwnode_t *arcnodeof*/
PRIVATE archwnode_t *arcnodeof(const namedesc_t *const name)
/*****************************************************************************
*
* arcnodeof returns the arcnode for a given name descriptor
*
*****************************************************************************/
{
  c_declnode_t *hwnode  = NVGraphOf(name->n_name);
  archwnode_t  *arcnode = (archwnode_t *)hwnode->c_hwnodes;
  return &(arcnode[name->n_index]);
}
/*}}}*/
/*{{{  PUBLIC chanswnode_t *channodeof*/
PUBLIC chanswnode_t *channodeof(const namedesc_t *const name)
/*****************************************************************************
*
* channodeof returns the channode for a given name descriptor
*
*****************************************************************************/
{
  c_declnode_t *hwnode = NVGraphOf(name->n_name);
  if (hwnode->c_nodetype == C_DECLNODE_CHANABBR)
    {
      chanabbrnode_t *abbrnode = (chanabbrnode_t *)hwnode->c_swnodes;
      return abbrnode[name->n_index].c_abbr;
    }
  else
    {
      chanswnode_t *channode = (chanswnode_t *)hwnode->c_swnodes;
      return &(channode[name->n_index]);
    }
}
/*}}}*/
/*{{{  PRIVATE edgehwnode_t *edgenodeofnamelink*/
PRIVATE edgehwnode_t *edgenodeofnamelink(const namedesc_t *const name, const INT32 link)
/*****************************************************************************
*
* edgenodeofnamelink looks for either an edge namedesc, or a device and linknum,
*                    and returns the appropriate edge
*
*****************************************************************************/
{
  edgehwnode_t *edge;
  if (link == EDGENODE_NO_LINK)
    edge = edgenodeof(name);
  else
    edge = &(prochwnodeof(name)->p_edge[link]);
  return edge;
}
/*}}}*/
/*{{{  PRIVATE void find_edge*/
PRIVATE void find_edge(treenode *edge, namedesc_t *const name, int *const link)
/*****************************************************************************
*
* find_edge takes an element who's type is EDGE, and returns either the
*           edge's name descriptor, or a device's name descriptor plus
*           linknum
*
*****************************************************************************/
{
  if (TagOf(edge) == S_ARRAYSUB && TagOf(ASBaseOf(edge)) == S_RECORDSUB)
    {
      treenode *device  = ASBaseOf(ASBaseOf(edge));
      treenode *linknum = ASIndexOf(edge);
      if (is_evaluable(linknum))
        {
          map_to_basename(device, name);
          *link = (int)evaluate(linknum);
          if ((*link < 0) || (*link >= MAX_PROC_LINKS))
            chkreport_i(CONF_BAD_LINKNUM, LocnOf(linknum), *link);
        }
      else
        chkreport(CONF_CANNOT_EVAL_LINKNUM, LocnOf(linknum));
    }
  else
    {
      map_to_basename(edge, name);
      *link = EDGENODE_NO_LINK;

      /* if the edge we've got is an edge like EDGE e IS d[link][0] :
         then recurse onto the processor/link */
      if (NVGraphOf(name->n_name)->c_nodetype == C_DECLNODE_EDGEABBR)
        {
          find_edge(DValOf(NDeclOf(name->n_name)), name, link);
        }
    }
}
/*}}}*/
/*{{{  PUBLIC int proc_wordlength(namedesc_t *name)*/
PUBLIC int proc_wordlength(const namedesc_t *const name)
{
  return prochwnodeof(name)->p_bpw;
}
/*}}}*/
/*}}}*/
/*{{{  dumping graph nodes*/
/*{{{  PRIVATE void dump_edgedesc*/
PRIVATE void dump_edgedesc(const edgedesc_t *const edge)
/*****************************************************************************
*
* dump_edgedesc displays the contents of an edgedesc
*
*****************************************************************************/
{
  if (edge->e_link == EDGENODE_NO_LINK)
    print_name_to_str(&(edge->e_un.e_edge));
  else
    {
      print_name_to_str(&(edge->e_un.e_proc->p_name));
      fprintf(outfile, ":%d", edge->e_link);
    }
}
/*}}}*/
/*{{{  PRIVATE void dump_connection*/
PRIVATE void dump_connection(const connection_t *const conn, const int display_arc)
/*****************************************************************************
*
* dump_connection displays the contents of a connection
*
*****************************************************************************/
{
  fputs("connects ", outfile);
  if (conn == NULL)
    fputs("nothing", outfile);
  else
    {
      dump_edgedesc(&(conn->c_edge[0]));
      fputs(" to ", outfile);
      dump_edgedesc(&(conn->c_edge[1]));
      if (display_arc && (conn->c_arc != NULL))
        {
          char str[MAXSTRING_SIZE];
          name_to_str(str, &(conn->c_arc->a_name));
          fprintf(outfile, " (by %s)", str);
        }
    }
}
/*}}}*/
/*{{{  PRIVATE void dump_edge_data*/
PRIVATE void dump_edge_data(const edgehwnode_t *const edge)
/*****************************************************************************
*
* dump_edge_data displays the contents of an edge connection
*
*****************************************************************************/
{
  if (empty_edge(edge))
    fputs("empty", outfile);
  else
    dump_connection(edge->e_conn, TRUE);
}
/*}}}*/
/*{{{  PRIVATEPARAM void dump_proc*/
PRIVATEPARAM void dump_proc(const namedesc_t *const name)
/*****************************************************************************
*
* dump_proc displays the processor's data structures
*
*****************************************************************************/
{
  prochwnode_t *hwnode = prochwnodeof(name);
  procswnode_t *swnode = procswnodeof(name);
  int i;
  char str[MAXSTRING_SIZE];

  name_to_str(str, name);
  fprintf(outfile, "NODE %s type \"%s\", memory %ld, freelist: ", str,
                   (hwnode->p_type == NULL) ? "NULL" : WNameOf(hwnode->p_type),
                   hwnode->p_mem);
  printfreelist(swnode->p_varlist);
  for (i = 0; i < MAX_PROC_LINKS; i++)
    if (!empty_edge(&(hwnode->p_edge[i])))
      {
        fprintf(outfile, "  Edge %d, ", i);
        dump_edge_data(&(hwnode->p_edge[i]));
        fputc('\n', outfile);
      }
}
/*}}}*/
/*{{{  PRIVATEPARAM void dump_edge*/
PRIVATEPARAM void dump_edge(const namedesc_t *const name)
/*****************************************************************************
*
* dump_edge displays an EDGE's data structures
*
*****************************************************************************/
{
  edgehwnode_t *node = edgenodeof(name);
  char str[MAXSTRING_SIZE];
  name_to_str(str, name);

  fprintf(outfile, "EDGE %s ", str);
  dump_edge_data(node);
  fputc('\n', outfile);
}
/*}}}*/
/*{{{  PRIVATEPARAM void dump_arc*/
PRIVATEPARAM void dump_arc(const namedesc_t *const name)
/*****************************************************************************
*
* dump_arc displays an ARC's data structures
*
*****************************************************************************/
{
  archwnode_t *node = arcnodeof(name);
  char str[MAXSTRING_SIZE];
  int edge, dirn;

  name_to_str(str, name);
  fprintf(outfile, "ARC %s ", str);
  dump_connection(node->a_conn, FALSE);
  fputs("\n  ", outfile);
  for (edge = 0; edge < 2; edge++)
    for (dirn = DIRN_START; dirn < DIRN_END; dirn++)
      {
        const chanswnode_t *const chan = node->a_usage[edge][dirn];
        if (chan == NULL)
          strcpy(str, "NULL");
        else
          name_to_str(str, &(chan->c_name));
        fprintf(outfile, "%s: %s, ", (dirn == DIRN_IN) ? "in" : "out", str);
      }
  fputc('\n', outfile);
}
/*}}}*/
/*{{{  PRIVATEPARAM void dump_chan*/
PRIVATEPARAM void dump_chan(const namedesc_t *const name)
/*****************************************************************************
*
* dump_chan displays a CHAN's data structures
*
*****************************************************************************/
{
  chanswnode_t *node = channodeof(name);
  int i;

  fputs("CHAN ", outfile);
  print_name_to_str(name);

  for (i = DIRN_START; i < DIRN_END; i++)
    {
      char str[MAXSTRING_SIZE];
      fputs((i == 0) ? " from " : " to ", outfile);
      if (node->c_dirn[i].c_proc == NULL)
        fputs("NULL", outfile);
      else
        print_name_to_str(&(node->c_dirn[i].c_proc->p_name));
      name_to_str(str, &(node->c_dirn[i].c_id));
      fprintf(outfile, " (id: %s)", str);
    }
  fputc('\n', outfile);
}
/*}}}*/
/*}}}*/

/****************************************************************************/
/*}}}*/

/*{{{  PUBLIC INT32 setproctypeto(namedesc_t *name)*/
PUBLIC INT32 setproctypeto(const namedesc_t *const name)
{
  const INT32 i = setprocessor(WNameOf(prochwnodeof(name)->p_type));
  setprocessorattr();
  return i;
}
/*}}}*/
/*{{{  PUBLIC void walk_config_list*/
PUBLIC void walk_config_list(treenode *list, void (*f)(const namedesc_t *const))
/*****************************************************************************
*
* walk_config_list walks a list, applying the function to the namedesc
*                  of every treenode in the list
*
*****************************************************************************/
{
  for (; !EndOfList(list); list = NextItem(list))
    {
      treenode *t = ThisItem(list);
      namedesc_t name;
      const INT32 elements = elementsin(NTypeOf(t));
      INT32 i;
      name.n_name = t;
      for (i = 0; i < elements; i++)
        {
          name.n_index = i;
          (*f)(&name);
        }
    }
}
/*}}}*/

/*{{{  declare_hwnode*/
PRIVATE void declare_hwnode(treenode *const name)
/*****************************************************************************
*
* declare_hwnode processes the declaration of a hardware item.
*
*****************************************************************************/
{
  const int type = basetype(NTypeOf(name));
  const INT32 size = elementsin(NTypeOf(name));
  c_declnode_t *node;
  DEBUG_MSG(("declare_hwnode\n"));
  if (!EndOfList(expand_repllist)) /* cannot declare inside a replicator */
    chkerr_ss(CONF_DECL_REPL, LocnOf(name), itagstring(type), WNameOf(NNameOf(name)));
  switch (type)
    {
      case S_NODE:
        node = newprocnode(name, size);
        proclist = addtofront(name, proclist);
        break;
      case S_ARC:
        node = newarchwnode(name, size);
        arclist = addtofront(name, arclist);
        break;
      case S_EDGE:
        node = newedgehwnode(size);
        edgelist = addtofront(name, edgelist);
        break;
      case S_CHAN:
        node = newchanswnode(name, size);
        chanlist = addtofront(name, chanlist);
        break;
      default:
        badtag(LocnOf(name), type, "declare_hwnode");
        break;
    }
  SetNVGraph(name, node);

  if (config_info)
    fprintf(outfile, "Declared [%ld]%s %s\n",
            size, itagstring(type), WNameOf(NNameOf(name)));
}
/*}}}*/
/*{{{  PRIVATE void abbreviate_hwnode*/
PRIVATE void abbreviate_hwnode(treenode *const tptr)
/*****************************************************************************
*
* abbreviate_hwnode processes the abbreviation of a hardware item
*
*****************************************************************************/
{
  treenode *const name = DNameOf(tptr);
  const int type = basetype(NTypeOf(name));
  treenode *const rhs = DValOf(tptr);
  namedesc_t rhsname;
  c_declnode_t *node;
  DEBUG_MSG(("abbreviate_hwnode\n"));
  switch (type)
    {
      case S_NODE:
        {
          map_to_basename(rhs, &rhsname);
          if (rhsname.n_index < 0) return;
          node = newc_declnode(C_DECLNODE_DEVICE, prochwnodeof(&rhsname),
                               procswnodeof(&rhsname));
          SetNVGraph(name, node);
        }
        break;
      case S_ARC:
        {
          map_to_basename(rhs, &rhsname);
          if (rhsname.n_index < 0) return;
          node = newc_declnode(C_DECLNODE_ARC, arcnodeof(&rhsname), NULL);
          SetNVGraph(name, node);
        }
        break;
      case S_EDGE:
        {
          int link;
          find_edge(rhs, &rhsname, &link);
          if (rhsname.n_index < 0) return;
          if (link == EDGENODE_NO_LINK)
            node = newc_declnode(C_DECLNODE_EDGE, edgenodeofnamelink(&rhsname, link), NULL);
          else /* special incase of EDGE e IS p[link][i]: */
            node = newc_declnode(C_DECLNODE_EDGEABBR, NULL, NULL);
          SetNVGraph(name, node);
        }
        break;
      case S_CHAN:
        abbreviate_chan(name, rhs);
        break;
      default:
        break;
    }
  if (config_info && (network_datatype(type) || (type == S_CHAN)))
    fprintf(outfile, "Abbreviated [%ld]%s %s\n",
            elementsin(NTypeOf(name)), itagstring(type), WNameOf(NNameOf(name)));
}
/*}}}*/
/*{{{  PRIVATE void set_connection*/
PRIVATE void set_connection(const SOURCEPOSN locn,
                            const namedesc_t *const name, const int link,
                            connection_t *const conn,
                            edgedesc_t *const edgedesc)
/*****************************************************************************
*
* set_connection marks this (processor & link) or edge, to be connected
*                via that connection. It also sets up the fields in the
*                connection record.
*
*****************************************************************************/
{
  edgehwnode_t *edge = edgenodeofnamelink(name, link);
  if (!empty_edge(edge))
    {
      char str[MAXSTRING_SIZE];
      name_to_str(str, name);
      if (link == EDGENODE_NO_LINK)
        chkerr_s(CONF_EDGE_CONNECTED, locn, str);
      else
        {
          char buf[50];
          sprintf(buf, "%d", link);
          chkerr_ss(CONF_NAME_LINK_CONNECTED, locn, str, buf);
        }
    }

  edgedesc->e_link = link;
  if (link == EDGENODE_NO_LINK)
    edgedesc->e_un.e_edge = *name;
  else
    edgedesc->e_un.e_proc = prochwnodeof(name);

  set_edge(edge, conn);
}
/*}}}*/
/*{{{  PRIVATE void set_arc*/
PRIVATE void set_arc(const SOURCEPOSN locn, const namedesc_t *const arcname,
                     connection_t *const conn)
/*****************************************************************************
*
* set_arc sets the arc as attached to a particular connection,
*         and marks the connection record
*
*****************************************************************************/
{
  archwnode_t *arcnode = arcnodeof(arcname);

  if (arcnode->a_conn != NULL)
    {
      char str[MAXSTRING_SIZE];
      name_to_str(str, arcname);
      chkerr_s(CONF_ARC_ALREADY_USED, locn, str);
    }

  conn->c_arc = arcnode;

  arcnode->a_conn = conn;
}
/*}}}*/
/*{{{  PRIVATE void make_connection*/
PRIVATE void make_connection(treenode *const tptr)
/*****************************************************************************
*
* make_connection processes a CONNECT statement
*
*****************************************************************************/
{
  treenode *const arc = ConnectArcOf(tptr);
  namedesc_t arcname, proc[2];
  int link[2];
  connection_t *conn;

  DEBUG_MSG(("make_connection\n"));

  conn = newconnection();

  find_edge(ConnectFromEdgeOf(tptr), &proc[0], &link[0]);
  find_edge(ConnectToEdgeOf(tptr),   &proc[1], &link[1]);

  if (arc != NULL)
    {
      map_to_basename(arc, &arcname);
      set_arc(LocnOf(tptr), &arcname, conn);
    }
  else
    conn->c_arc = NULL;
  if ((proc[0].n_index < 0) || (proc[1].n_index < 0) ||
      ((arc != NULL) && (arcname.n_index < 0)))
    return;

  set_connection(LocnOf(tptr), &proc[0], link[0], conn, &(conn->c_edge[0]));
  set_connection(LocnOf(tptr), &proc[1], link[1], conn, &(conn->c_edge[1]));

  if (config_info)
    {
      char str[MAXSTRING_SIZE];
      int i;
      for (i = DIRN_START; i < DIRN_END; i++)
        {
          name_to_str(str, &proc[i]);
          fprintf(outfile, "%s %s", (i == DIRN_START) ? "Connected" : " to", str);
          if (link[i] != EDGENODE_NO_LINK)
            fprintf(outfile, "[link][%d]", link[i]);
        }
      if (arc != NULL)
        {
          fputs(" with arc ", outfile);
          print_name_to_str(&arcname);
        }
      fputc('\n', outfile);
    }
}
/*}}}*/
/*{{{  PRIVATE void set_dev_attr*/
PRIVATE void set_dev_attr(prochwnode_t *const procnode, const SOURCEPOSN locn,
                          treenode *const expr, const int attr)
/*****************************************************************************
*
* set_dev_attr sets an attribute of a processor
*
*****************************************************************************/
{
  int allowed_inside_mapping = FALSE;
  char procstr[MAXSTRING_SIZE];
  name_to_str(procstr, &(procnode->p_name));

  if (config_info)
    fprintf(outfile, "Processor %s, setting ", procstr);
  switch (attr)
    {
      /*{{{  processor type*/
      case PD_ATTR_TYPE:
        {
          treenode *type_str = newconstexp(expr);
          wordnode *s = (wordnode *)CTValOf(type_str);
          if ((setprocessor(WNameOf(s)) == ZERO32) || (processor_class != 0))
            chkerr_s(CONF_ILLEGAL_PROCTYPE, locn, WNameOf(s));
          else if (procnode->p_type != NULL)
            chkerr_s(CONF_DUPLICATE_PROCTYPE, locn, procstr);
          else
            {
              if (config_info)
                fprintf(outfile, "type   to \"%s\"\n", WNameOf(s));
              procnode->p_type  = s;
              procnode->p_links = no_of_links;
              procnode->p_bpw   = bytesperword;
            }
          setprocessordefault(); /* go back to normal */
        }
        break;
      /*}}}*/
      /*{{{  memory size*/
      case PD_ATTR_MEM:
        {
          const INT32 val = evaluate(expr);
          if (val < 0)
            chkerr_i(CONF_ILLEGAL_MEMVAL, locn, val);
          else if (procnode->p_mem != PROCNODE_INIT_MEMVAL)
            chkerr_s(CONF_DUPLICATE_MEMVAL, locn, procstr);
          else
            {
              if (config_info)
                fprintf(outfile, "memory to %ld\n", val);
              procnode->p_mem = val;
            }
        }
        break;
      /*}}}*/
      /*{{{  root processor*/
      case PD_ATTR_ROOT:
        if (evaluate(expr) != 0)
          {
            if ((root_processor != NULL) && (root_processor != procnode))
              chkerr_s(CONF_DUPLICATE_ROOT, locn, procstr);
            else if (TagOf(NTypeOf(procnode->p_name.n_name)) == S_ARRAY)
              chkerr_s(CONF_ROOT_NOT_ARRAY, locn, procstr);
            root_processor = procnode;
            if (config_info)
              fprintf(outfile, "root   to TRUE\n");
          }
        break;
      /*}}}*/
      /*{{{  romsize*/
      case PD_ATTR_ROMSIZE:
        {
          const INT32 val = evaluate(expr);
          if (val < 0)
            chkerr_i(CONF_ILLEGAL_ROMSIZE, locn, val);
          else if (romsize != PROCNODE_INIT_MEMVAL)
            chkerr(CONF_DUPLICATE_ROMSIZE, locn);
          else
            {
              romsize = val;
              romsize_processor = procnode;
              if (config_info)
                fprintf(outfile, "romsize to %ld\n", val);
            }
        }
        break;
      /*}}}*/
      /*{{{  ORDER*/
      case PD_ATTR_ORDERCODE: case PD_ATTR_ORDERVS:
        {
          const INT32 val = evaluate(expr);
          INT32 *const oldval = (attr == PD_ATTR_ORDERCODE) ? &procnode->p_ordercode
                                                            : &procnode->p_ordervs;
          if (*oldval != DEFAULT_ORDER_VALUE)
            chkerr(CONF_DUPLICATE_ORDERINFO, locn);
          *oldval = val;
          if (config_info)
            fprintf(outfile, "order.%s to %ld\n",
                    (attr == PD_ATTR_ORDERCODE) ? "code" : "vs", val);
          allowed_inside_mapping = TRUE;
          if (order_flags_locn == NOPOSN) order_flags_locn = locn;
        }
        break;
      /*}}}*/
      default:
        break;
    }
  if (processing_mapping != allowed_inside_mapping)
    chkerr_s(CHK_ILLEGAL_CONSTRUCT, locn,
            tagstring(processing_mapping ? S_MAPPING : S_NETWORK));
}
/*}}}*/
/*{{{  PRIVATE void set_dev_attributes*/
PRIVATE void set_dev_attributes(treenode *const tptr)
/*****************************************************************************
*
* set_dev_attributes processes a SET statement
*
*****************************************************************************/
{
  treenode *dev   = STDevOf(tptr);
  treenode *attrs = STAttrNameOf(tptr);
  treenode *exps  = STAttrExpOf(tptr);
  namedesc_t devname;
  prochwnode_t *procnode;
  DEBUG_MSG(("set_attributes\n"));

  map_to_basename(dev, &devname);
  if (devname.n_index < 0) return;

  /*procnode = prochwnodeof(&devname);*/
  /* bug 1145 - 5/2/91 - translate logicals */
  procnode = prochwnodeof(&(procswnodeof(&devname)->p_name));
  while (!EndOfList(attrs) && !EndOfList(exps))
    {
      set_dev_attr(procnode, LocnOf(tptr), ThisItem(exps), NModeOf(ThisItem(attrs)));
      attrs = NextItem(attrs);
      exps  = NextItem(exps);
    }
}
/*}}}*/
/*{{{  PRIVATE void map_logical_to_physical*/
PRIVATE void map_logical_to_physical(treenode *const source, const SOURCEPOSN locn,
         const namedesc_t *const destname, const INT32 privalue)
/* maps logical processor source onto physical processor dest */
{
  c_declnode_t *hwnode;
  namedesc_t sourcename;
  logicalprocnode_t *procnode;

  map_to_basename(source, &sourcename);
  if (sourcename.n_index < 0) return;

  if (config_info)
    {
      fputs("Mapping ", outfile);
      print_name_to_str(&sourcename);
      fputs(" onto ", outfile);
      print_name_to_str(destname);
      fputc('\n', outfile);
    }

  if (NVGraphOf(destname->n_name)->c_nodetype != C_DECLNODE_DEVICE)
    chkerr(CONF_REQUIRE_PHYSICAL_NODE, locn);

  hwnode = NVGraphOf(sourcename.n_name);
  if (hwnode->c_nodetype != C_DECLNODE_LOGICAL)
    convert_to_logical(sourcename.n_name, locn);

  procnode = (logicalprocnode_t *)hwnode->c_swnodes;
  procnode = &(procnode[sourcename.n_index]);

  if (procnode->p_translation != NULL)
    {
      char str[MAXSTRING_SIZE];
      name_to_str(str, &sourcename);
      chkerr_s(CONF_NAME_ALREADY_MAPPED, locn, str);
    }
  procnode->p_translation = procswnodeof(destname);
  procnode->p_priority    = privalue;
  if (procnode->p_translation->p_procname == NULL)
    {
      char str[MAXSTRING_SIZE];
      parseable_str(str, &sourcename);
      procnode->p_translation->p_procname = lookupword(str, strlen(str));
    }
}
/*}}}*/
/*{{{  PRIVATE void map_channel*/
PRIVATE void map_channel(treenode *const source, const SOURCEPOSN locn,
                         const namedesc_t *const destname)
/* maps channel source onto arc dest */
{
  namedesc_t sourcename;
  chanswnode_t *channode;

  map_to_basename(source, &sourcename);
  if (sourcename.n_index < 0) return;

  if (config_info)
    {
      fputs("Mapping ", outfile);
      print_name_to_str(&sourcename);
      fputs(" onto ", outfile);
      print_name_to_str(destname);
      fputc('\n', outfile);
    }
  channode = channodeof(&sourcename);
  if (channode->c_arc != NULL)
    {
      char str[MAXSTRING_SIZE];
      name_to_str(str, &sourcename);
      chkerr_s(CONF_NAME_ALREADY_MAPPED, locn, str);
    }
  channode->c_arc = arcnodeof(destname);
  /* channode->c_arc->a_used = TRUE; */
}
/*}}}*/
/*{{{  PRIVATE void process_map*/
PRIVATE void process_map(treenode *const tptr)
/*****************************************************************************
*
* process_map processes a MAP statement
*
*****************************************************************************/
{
  treenode *sourcelist = MapSourceOf(tptr);
  treenode *dest       = MapDestOf(tptr);
  treenode *pri        = MapPriOf(tptr);
  INT32 privalue;
  namedesc_t destname;
  int type;
  DEBUG_MSG(("process_map\n"));

  map_to_basename(dest, &destname);
  if (destname.n_index < 0) return;

  type = basetype(NTypeOf(destname.n_name));
  if (pri == NULL)
    privalue = PRIORITY_LO;
  else
    {
      privalue = evaluate(pri);
      if (privalue != PRIORITY_HI && privalue != PRIORITY_LO)
        chkerr_i(CONF_ILLEGAL_PRIVAL, LocnOf(tptr), privalue);
    }

  while (!EndOfList(sourcelist))
    {
      if (type == S_NODE)
        map_logical_to_physical(ThisItem(sourcelist), LocnOf(tptr), &destname, privalue);
      else
        map_channel(ThisItem(sourcelist), LocnOf(tptr), &destname);
      sourcelist = NextItem(sourcelist);
    }
}
/*}}}*/
/*{{{  PRIVATE void process_placeon*/
PRIVATE void process_placeon(treenode *const tptr)
{
  treenode *sourcelist = DNameOf(tptr);
  namedesc_t destname;

  DEBUG_MSG(("process_placeon\n"));

  map_to_basename(DValOf(tptr), &destname);
  if (destname.n_index < 0) return;

  while (!EndOfList(sourcelist))
    {
      map_channel(ThisItem(sourcelist), LocnOf(tptr), &destname);
      sourcelist = NextItem(sourcelist);
    }
}
/*}}}*/
/*{{{  process_config*/
PRIVATEPARAM int do_process_config(treenode *const tptr)
/*****************************************************************************
*
* do_process_config 
*
*****************************************************************************/
{
  switch(TagOf(tptr))
    {
      case S_DECL:  /* will be a device, arc, or edge, or maybe a chan */
        {
          treenode *name = DNameOf(tptr);
          if (TagOf(name) == S_LIST)
            {
              while (!EndOfList(name))
                {
                  declare_hwnode(ThisItem(name));
                  name = NextItem(name);
                }
            }
          else
            declare_hwnode(name);
        }
        break;
      case S_ABBR: case S_VALABBR:
        abbreviate_hwnode(tptr);
        break;
      case S_PLACEON:
        process_placeon(tptr);
        break;
      /*{{{  Hardware pass only*/
      case S_CONNECT:
        make_connection(tptr);
        break;
      case S_SET:
        set_dev_attributes(tptr);
        break;
      case S_NETWORK:
        /* the tree walk doesn't go inside networks, so we do it explicitly */
        if (config_info)
          fprintf(outfile, "Entering NETWORK description\n");
        process_config(DValOf(tptr), FALSE);
        if (config_info)
          fprintf(outfile, "Leaving NETWORK description\n");
        break;
      /*}}}*/
      /*{{{  Software pass only*/
      case S_PROCESSOR:
        declare_process(tptr);
        return STOP_WALK;
      /*}}}*/
    }
  return CONTINUE_WALK;
}

PUBLIC void process_config(treenode *const tptr, const int hardware)
/*****************************************************************************
*
* process_config this walks the tree, expanding replicators, 
*                and then does the right stuff for the 'outer level'
*                software configuration (ie down to a PROCESSOR statement).
*
*                While the hardware description is being written in occam,
*                we may aswell use the same routine for the hardware
*                description too.
*                Once the H/W description is abstracted out, this can deal
*                purely with the S/W description.
*
*****************************************************************************/
{
  processing_mapping = FALSE;
  if (hardware)
    declare_hwnode(hostedgeptr);
  expandwalkproctree(tptr, do_process_config);
}
/*}}}*/
/*{{{  process_mapping*/
PRIVATEPARAM int do_process_mapping(treenode *const tptr)
/*****************************************************************************
*
* do_process_mapping
*
*****************************************************************************/
{
  switch(TagOf(tptr))
    {
      case S_MAP:
        process_map(tptr);
        break;
      case S_SET:
        set_dev_attributes(tptr);
        break;
    }
  return CONTINUE_WALK;
}

PUBLIC void process_mapping(treenode *const tptr)
{
  processing_mapping = TRUE;
  expandwalkproctree(tptr, do_process_mapping);
}
/*}}}*/

/*{{{  PUBLIC void check_arc_status*/
PUBLIC void check_arc_status(const procswnode_t *const procnode, const SOURCEPOSN locn,
                             chanswnode_t *const chan, const int dirn_index)
/*****************************************************************************
*
* check_arc_status checks whether a channel may be
*            placed onto that arc in that direction from that processor.
*            if arc isn't connected to anything
*              error CONF_ARC_NOT_CONNECTED
*            if arc isn't connected to processor
*              error CONF_ARC_WRONG_CONNECTED
*            if arc is already used in that direction
*              error CONF_ARC_FULL_*
*
*****************************************************************************/
{
  connection_t *connection;
  prochwnode_t *prochwnode = prochwnodeof(&procnode->p_name);
  archwnode_t  *arc = chan->c_arc;
  int i;
  char chan_str[MAXSTRING_SIZE], arc_str[MAXSTRING_SIZE];
  int end_index = (-1), found_an_end = FALSE;

  if (arc == NULL)
    return;

  connection = arc->a_conn;
  name_to_str(chan_str, &chan->c_name);
  name_to_str(arc_str,  &arc->a_name);

  if (connection == NULL)
    {
      chkerr_ss(CONF_ARC_NOT_CONNECTED, locn, chan_str, arc_str);
      return;
    }
  for (i = 0; (i < 2) && (end_index < 0); i++)
    if (connection->c_edge[i].e_link != EDGENODE_NO_LINK &&
        connection->c_edge[i].e_un.e_proc == prochwnode)
      {
        /* Check incase both ends of this arc are connected to same processor */
        if ((i == 0) &&
            ((arc->a_usage[0][dirn_index] != NULL) || /* other channel has already input */
             (arc->a_usage[0][1-dirn_index] == chan))) /* this channel uses that end */
          found_an_end = TRUE; /* this end has already been used for this dirn */
        else
          end_index = i;
      }
  if ((end_index < 0) && !found_an_end)
    {
      chkerr_ss(CONF_ARC_WRONGLY_CONNECTED, locn, chan_str, arc_str);
      return;
    }
  if ((chan->c_dirn[DIRN_IN].c_proc != NULL) && (chan->c_dirn[DIRN_OUT].c_proc != NULL) &&
      ((connection->c_edge[0].e_link == EDGENODE_NO_LINK) ||
       (connection->c_edge[1].e_link == EDGENODE_NO_LINK) )  )
    /* both ends of the channel are used, but arc is mapped onto an EDGE */
    {
      chkerr_ss(CONF_ARC_NO_EDGE, locn, chan_str, arc_str);
      return;
    }
  if ((end_index < 0) || (arc->a_usage[end_index][dirn_index] != NULL))
    chkerr_s((dirn_index == DIRN_OUT) ? CONF_ARC_FULL_OUTPUT : CONF_ARC_FULL_INPUT,
            locn, arc_str);
  else
    arc->a_usage[end_index][dirn_index] = chan;

  return;
}
/*}}}*/
/*{{{  PRIVATE int edge_connects_to_proc()*/
PRIVATE int edge_connects_to_proc(const edgedesc_t *const edge, const prochwnode_t *const proc)
/*****************************************************************************
*
* edge_connects_to_proc returns TRUE if the edge is attached to that processor
*
*****************************************************************************/
{
  return (edge->e_link != EDGENODE_NO_LINK) && (edge->e_un.e_proc == proc);
}
/*}}}*/
/*{{{  PUBLIC int connection_status*/
PUBLIC int connection_status(const namedesc_t *const fromproc,
                             const namedesc_t *const toproc)
/*****************************************************************************
*
* connection_status returns an error code according to whether a connection
*                   exists between fromproc and toproc.
*                   we already know that fromproc and toproc are different.
*                   if one exists, and hasn't already been used,
*                     return CHK_OK
*                   if at least one exists, but all have been used,
*                     return CONF_FULL_ROUTE
*                   if no route exists,
*                     return CONF_NO_ROUTE
*
*****************************************************************************/
{
  prochwnode_t *from = prochwnodeof(fromproc);
  prochwnode_t *to   = prochwnodeof(toproc);
  int found = FALSE;
  int i;
  for (i = 0; i < from->p_links; i++)
    {
      connection_t *conn = from->p_edge[i].e_conn;
      if ( (conn != NULL) && (edge_connects_to_proc(&(conn->c_edge[0]), to) ||
                              edge_connects_to_proc(&(conn->c_edge[1]), to)   ) )
        {
          linkbits_t bit       = (1 << i);
          linkbits_t *usedbits = &(from->p_linkbits);
          int unused = ((*usedbits) & bit) == 0;
          *usedbits |= bit;  /* mark this link as used */
          if (unused)
            return CHK_OK;
          found = TRUE;
        }
    }
  if (found)
    return CONF_FULL_ROUTE;
  return CONF_NO_ROUTE;
}
/*}}}*/
/*{{{  PRIVATE void mark_hw_connectivity()*/
 /* forward decl */
PRIVATE void mark_edge_connections(prochwnode_t *const procnode, const connection_t *const conn);

PRIVATE void mark_hw_connectivity(prochwnode_t *const procnode)
/*****************************************************************************
*
* mark_hw_connectivity marks all the processors attached to this one as
*                      connected
*
*****************************************************************************/
{
  if (!procnode->p_connected)
    {
      int i;
      procnode->p_connected = TRUE;

      for (i=0; i < MAX_PROC_LINKS; i++)
        {
          connection_t *conn = procnode->p_edge[i].e_conn;
          if (conn != NULL)
            mark_edge_connections(procnode, conn);
        }
    }
}
/*}}}*/
/*{{{  PRIVATE void mark_edge_connection*/
PRIVATE void mark_edge_connection(prochwnode_t *const procnode, const edgedesc_t *const edge)
{
  if (edge->e_link != EDGENODE_NO_LINK)
    {
      prochwnode_t *p = edge->e_un.e_proc;
      if (p != procnode)  /* this speeds things up */
        mark_hw_connectivity(p);
    }
}
/*}}}*/
/*{{{  PRIVATE void mark_edge_connections*/
PRIVATE void mark_edge_connections(prochwnode_t *const procnode, const connection_t *const conn)
{
  mark_edge_connection(procnode, &(conn->c_edge[0]));
  mark_edge_connection(procnode, &(conn->c_edge[1]));
}
/*}}}*/
/*{{{  PRIVATE void check_root_processor*/
PRIVATE void check_root_processor(const int check_connection)
{
  if (config_rom)
    {
      if (root_processor == NULL)
        chkerr(CONF_NO_ROOT, NOPOSN);
      else
        {
          if ((romsize == PROCNODE_INIT_MEMVAL) ||
              (romsize_processor != root_processor))
            chkerr(CONF_NO_ROMSIZE, NOPOSN);
          mark_hw_connectivity(root_processor);
        }
    }
  else
    {
      /* This checks that the HOST edge is connected to a root processor,
         and marks all processors which are connected to that */
      namedesc_t hostname;
      connection_t *hostconn;

      hostname.n_name = hostedgeptr;
      hostname.n_index = 0;

      hostconn = edgenodeof(&hostname)->e_conn;
      if (hostconn == NULL)
        chkerr_ss(CONF_NOT_CONNECTED, NOPOSN, tagstring(S_EDGE),
                  /*"HOST"*/ WNameOf(NNameOf(hostedgeptr)));
      else
        {
          if (check_connection)
            {
              /* Check that a channel is placed onto the host link */
              /* added for bug 1070 - 2/1/91 */
              /* Note that we have to do the check after the software stuff
                 has been processed */
              const archwnode_t *const hostarc = hostconn->c_arc;
              int connected = FALSE;
              if (hostarc != NULL)
                {
                  int i, dirn;
                  /*connected = hostarc->a_used;*/
                  for (i = 0; i < 2; i++) /* which end of the connection */
                    for (dirn = DIRN_START; dirn < DIRN_END; dirn++)
                      {
                        connected = connected || (hostarc->a_usage[i][dirn] != NULL);
                        DEBUG_MSG(("check_root_processor; usage[%d][%d] ?:%d\n", i, dirn,
                                   hostarc->a_usage[i][dirn] != NULL));
                      }
                }
              if (!connected)
                /*chkwarn(CONF_NO_HOST_CHAN, NOPOSN);*/
                msg_out(SEV_WARN, CHK, CONF_NO_HOST_CHAN, NOPOSN);
            }
          else
            mark_edge_connections(NULL, hostconn);
        }
    }
}
/*}}}*/
/*{{{  PRIVATEPARAM void check_proc_consistency*/
PRIVATEPARAM void check_proc_consistency(const namedesc_t *const name)
/*****************************************************************************
*
* check_proc_consistency checks that all the attributes of a processor
*                        have been set correctly
*
*****************************************************************************/
{
  int i;
  prochwnode_t *procnode = prochwnodeof(name);
  char str[MAXSTRING_SIZE];
  name_to_str(str, name);

  if (empty_proc(procnode))
    return;

  if (procnode->p_type == NULL)
    chkerr_s(CONF_NO_PROCTYPE, LocnOf(name->n_name), str);
  else
    {
      for (i = procnode->p_links; i < MAX_PROC_LINKS; i++)
        if (!empty_edge(&(procnode->p_edge[i])))
          chkerr_i(CONF_BAD_LINKNUM, LocnOf(name->n_name), i);
    }
  if (procnode->p_mem == PROCNODE_INIT_MEMVAL)
    chkerr_s(CONF_NO_MEMVAL, LocnOf(name->n_name), str);

  if (!procnode->p_connected)
    {
      chkerr_ss(CONF_NOT_CONNECTED, LocnOf(name->n_name),
              tagstring(S_NODE), str);
      mark_hw_connectivity(procnode);
    }
}
/*}}}*/
/*{{{  PUBLIC check_hw_consistency()*/
PUBLIC void check_hw_consistency(void)
/*****************************************************************************
*
* check_hw_consistency checks that all the hardware description is ok
*
*****************************************************************************/
{
  /* These are only reversed so that we look at them in the same
     order as they were declared; it isn't really necessary */
  proclist = reverselist(proclist);
  edgelist = reverselist(edgelist);
  arclist  = reverselist(arclist);
  chanlist = reverselist(chanlist);

  if ((order_flags_locn != NOPOSN) && !config_reorderable)
    msg_out(SEV_WARN, CHK, CONF_ORDER_IGNORED, order_flags_locn);

  {
    connection_t *reverser = NULL;
    while (connectionchain != NULL)
      {
        connection_t *next = connectionchain->c_next;
        connectionchain->c_next = reverser;
        reverser = connectionchain;
        connectionchain = next;
      }
    connectionchain = reverser;
  }

  check_root_processor(FALSE);
  walk_config_list(proclist, check_proc_consistency);
}
/*}}}*/
/*{{{  PUBLIC void check_host_connection*/
PUBLIC void check_host_connection(void)
{
  /* it doesn't matter if we perform a few of the checks twice, so
     we can simply call check_root_processor again */
  check_root_processor(TRUE);
}
/*}}}*/
/*{{{  PUBLIC void check_proc_hw*/
PUBLIC void check_proc_hw(const namedesc_t *const procname)
/*****************************************************************************
*
* check_proc_hw checks that a device has been initialised ready to put
*               software onto it
*
*****************************************************************************/
{
  if (empty_proc(prochwnodeof(procname)))
    {
      char str[MAXSTRING_SIZE];
      name_to_str(str, procname);
      chkerr_s(CONF_NO_PROCTYPE, LocnOf(procname->n_name), str);
    }
}
/*}}}*/

/*{{{  PUBLIC void dump_config*/
PUBLIC void dump_config(void)
/*****************************************************************************
*
* dump_config displays all the hardware data strutures
*
*****************************************************************************/
{
  if (config_hw_dump)
    {
      fputs("dump of configuration information\n", outfile);
      walk_config_list(proclist, dump_proc);
      walk_config_list(edgelist, dump_edge);
      walk_config_list(arclist, dump_arc);
      walk_config_list(chanlist, dump_chan);
    }
}
/*}}}*/
