/*#define DEBUG*/
/*****************************************************************************
 *
 *  bind3 - workspace allocation
 *
 *****************************************************************************/

/*{{{  copyright*/
/******************************************************************************
*
*  occam 2 compiler
*
*  copyright Inmos Limited 1987
*
*
******************************************************************************/
/*}}}*/

/*{{{  include files*/
# include <stdio.h>
#ifdef ANSI
# include <stdlib.h>
#endif
# include "includes.h"
# include "generror.h"
# include "genhdr.h"
# include "syndef.h"
# include "chkdef.h"
# include "bind3def.h"
# include "gen1def.h"
# include "gen2def.h"
/*# include "code1def.h"*/
/*}}}*/

#ifdef CONFIG
  #define MAXVAR 512
#else
  #define MAXVAR 2048 /* changed for bug 1075 13/12/90 */
#endif
#define VARROW (MAXVAR/(sizeof(int) * 8))

#define   SORT_USAGE_CUTOFF 16 /* first 16 vars are sorted by usage, rest by scope */
#define RESORT_SLOTS_CUTOFF 16 /* first 16 slots are sorted by usage, rest by scope */

/*{{{  global variables*/
PUBLIC BIT32 datasize, vsp, maxvsp;
PUBLIC int staticlinkused;
PUBLIC int var_base, num_var;
PUBLIC INT32 loop_weight;

PUBLIC treenode *current_proc;

PUBLIC int alloc_strategy = ALLOC_DEFAULT;
/*}}}*/

/*{{{  local variables*/
PRIVATE int nextlabel;
PRIVATE int live_var[VARROW];
PRIVATE int *var_map = NULL;
PRIVATE treenode *var_list[MAXVAR];
PRIVATE treenode *fixedws_list;

PRIVATE treenode *vars_mapped_onto_vsptr;
/*}}}*/

/*{{{  vectorspace allocation*/
/*{{{  PUBLIC void initvsp (n)*/
/*****************************************************************************
 *
 *  initvsp initialises the vector space and maximum vector space usage to
 *          n slots.
 *
 *****************************************************************************/
PUBLIC void initvsp ( BIT32 n )
{
  vsp = n;
  maxvsp = n;
}
/*}}}*/
/*{{{  PUBLIC void setvsp(n)*/
/*****************************************************************************
 *
 *  setvsp sets the vector space to be n words, adjusting maxvsp if neccessary.
 *
 *****************************************************************************/
PUBLIC void setvsp ( BIT32 n )
{
  vsp = n;
  maxvsp = max(maxvsp, vsp);
}
/*}}}*/
/*{{{  PUBLIC BIT32 newvs (n)*/
/*****************************************************************************
 *
 *  newvs reserves n words of vector space and adjusts maxvs if neccessary.
 *        Returns the starting slot number of the reserved space.
 *
 *****************************************************************************/
PUBLIC BIT32 newvs ( BIT32 n )
{
  BIT32 result = vsp;
  setvsp(vsp + n);
  return(result);
}
/*}}}*/
/*}}}*/
/*{{{  workspace allocation*/
/*{{{  comment*/
/* allocation:
 *
 * A bit map showing the interference between variables is built up
 * during the register mapping phase of this pass.
 * The map is an  n x n bit var showing which are 'live'
 * at the same time:
 *    For example the map for
 *
 *    a1:                            a1 a2 a3 a4
 *    SEQ                         a1 1  1  1  1
 *      : :                       a2 1  1  1  0
 *      a2:           =           a3 1  1  1  0
 *      SEQ                       a4 1  0  0  0
 *        : :
 *        a3:
 *        SEQ
 *          : :
 *      a4:
 *      SEQ
 *        : :
 *
 * After the map has been built it is sorted by usecount.
 * Those variables requiring fixed positions are allocated first.
 * The allocation is done by scanning the list of variables looking for the
 * first slot large enough to take the variable which is not used by any
 * variable live at the same time as it.
 *
 * Variables which overlay each other are chained onto the original varible.
 */
/*}}}*/
/*{{{  PUBLIC INT32 allocsize(nptr)*/
PUBLIC INT32 allocsize ( treenode *nptr )
{
  treenode *type = NTypeOf(nptr);
  INT32 size = wordsin(type);
  if (size < 0)
    geninternal_is(GEN_ERROR_IN_ROUTINE, size, "allocsize");
  if (chanaspointer &&
      (TagOf(nptr) == N_DECL) &&  /* added 17/9/90 for bug 222 */
      (basetype(type) == S_CHAN))
    {
#if OPTIMISE_LOCAL_CHANS
      if ((TagOf(type) == S_ARRAY) || NChanMarkOf(nptr))
        size *= 2;
      /* this leaves (unmarked) scalar channels as using a single word */
#else
      size *= 2;
#endif
    }
  return size;
}
/*}}}*/
/*{{{  support*/
/*{{{  magic macros*/
/* These macros are used for setting and testing bits in arrays of words */
#define WORDSHIFT 5     /* such that 2^WORDSHIFT = (MAXVAR/VARROW) */
#define BYTEMASK  0x1f  /* (2^WORDSHIFT) - 1 */

#define wordnum(n) ((n) >> WORDSHIFT)
#define bitnum(n)  ((n) &  BYTEMASK)

#define setbit(array,n)   ((array)[wordnum(n)] |=  (1 << (bitnum(n))))
#define clearbit(array,n) ((array)[wordnum(n)] &= ~(1 << (bitnum(n))))
#define testbit(array,n)  ((array)[wordnum(n)] &   (1 << (bitnum(n))))

/* Because the bit map is symetrical (var_map[i][j] == var_map[j][i])
 * we can store it in half the space.
 *
 * This macro is used to map a row number to its position in a linear array
 */

#define startofrow(n)    ((((n)+1)*(n)) >> 1) /* 1/2 n(n+1) */
/*}}}*/

/*{{{  PRIVATE void setbitmap (map, i, j)*/
PRIVATE void setbitmap ( int *map , int i , int j )
{
  int bitpsn;
  if (i > j)
    bitpsn = startofrow(i) + j;
  else
    bitpsn = startofrow(j) + i;
  setbit(map, bitpsn);
}
/*}}}*/
/*{{{  PRIVATE void clearbitmap (map, i, j)*/
#if 0 /* clearbitmap is never used */
PRIVATE void clearbitmap ( int *map , int i , int j )
{
  int bitpsn;
  if (i > j)
    bitpsn = startofrow(i) + j;
  else
    bitpsn = startofrow(j) + i;
  clearbit(map, bitpsn);
}
#endif /* 0 */
/*}}}*/
/*{{{  PRIVATE int testbitmap (map, i, j)*/
PRIVATE int testbitmap ( int *map , int i , int j )
{
  int bitpsn;
  if (i > j)
    bitpsn = startofrow(i) + j;
  else
    bitpsn = startofrow(j) + i;
  return (testbit(map, bitpsn) != 0);
}
/*}}}*/

/*{{{  PRIVATE int markbitmap(map, mask, from, to, n)*/

/* Allocation map             */
/* map of live variables      */
/* range of variables to mark */
/* bit to be set              */
PRIVATE int markbitmap ( int *map , int *mask , int from , int to , int n )
{
  int i;
  int count = 0;
  for (i = from; i<=to; i++)
    if (testbit(mask, i))
      {
        setbitmap(map, i, n);
        count++;
      }
  return(count);
}
/*}}}*/

/*{{{  PRIVATE int vars_overlap (b1, l1, b2, l2)*/
/* Test for overlap of two vars,
* a variable of length 0 never overlaps
*/
PRIVATE int vars_overlap ( int b1 , int l1 , int b2 , int l2 )
{
  return ((l1 != 0 && l2 != 0) &&
          (((b1 >= b2) && (b1 < b2+l2)) || ((b2 >= b1) && (b2 < b1+l1))));
}
/*}}}*/
/*{{{  PRIVATE int numslots (nptr)*/
/* Give number of slots needed to store name nptr */
PRIVATE int numslots ( treenode *nptr )
{
  int length;
  switch (TagOf(nptr))
    {
      /*{{{  T_RESERVEDWS*/
      case T_RESERVEDWS:
        length = NVRSizeOf(nptr);
        break;
      /*}}}*/
      /*{{{  T_TEMP, N_DECL, T_PREEVALTEMP*/
      case T_TEMP:
      case N_DECL:
      case T_PREEVALTEMP:
        if (isplaced(nptr))
          length = 0;
        else if (isinvectorspace(nptr) || (NModeOf(nptr) == NM_POINTER))
          length = 1;
        else
          length = allocsize(nptr);
        break;
      /*}}}*/
      /*{{{  N_ABBR N_VALABBR N_RETYPE N_VALRETYPE*/
      case N_VALABBR: case N_ABBR:
      case N_VALRETYPE: case N_RETYPE:
        {
          treenode *spec = NDeclOf(nptr);
          const int am = abbrevmode(spec);
          if ((am == AM_CONST) || (am == AM_ISRHS))
            length = 0;
          else if ((am == AM_PTR) || isinvectorspace(nptr))
            length = 1;
          else
            length = allocsize(nptr);
        }
        break;
      /*}}}*/
      /*{{{  N_REPL*/
      case N_REPL:
        /* Replicator occupies two words, one for base, one for count */
        {
          /* This was used before the change in the way replicated ALT was done: */
          /*
            int t = TagOf(NDeclOf(nptr));
            length = (t == S_REPLALT || t == S_PRIREPLALT) ? 3 : 2;
          */
          length = 2;
        }
        break;
      /*}}}*/
      default:
        badtag(genlocn, TagOf(nptr), "numslots");
    }
  return (length);
}
/*}}}*/
/*{{{  PRIVATE int nextvarnum()*/
PRIVATE int nextvarnum ( void )
{
  if (num_var >= MAXVAR)
    msg_out_is(SEV_FATAL, GEN, GEN_TOO_MANY_VARS, genlocn,
               MAXVAR, WNameOf(NNameOf(current_proc)));
  return num_var++;
}
/*}}}*/
/*{{{  PRIVATE int clashcount(first_var, n)*/
#if 0  /* clashcount is never called */
PRIVATE int clashcount ( int first_var , int n )
{
  int i;
  int total = 0;
  for (i=first_var; i < num_var; i++)
    if (testbitmap(var_map, n, i))
      total++;
  return (total);
}
#endif /* 0 */
/*}}}*/

/*{{{  compare routines*/
/*{{{  PUBLIC int compare_scope (v1, v2) ** passed as a parameter*/
/*{{{  comment*/
/* Compare two variables c1 and c2,
   The ordering used to compare vars is :

   T_RESERVEDWS <  all other vars

   all other vars are then compared on scope
*/
/*}}}*/
PRIVATEPARAM int compare_scope ( int *v1 , int *v2 )
{
  treenode  *var1 = var_list[*v1];
  treenode  *var2 = var_list[*v2];
  int tag1 = TagOf(var1); int tag2 = TagOf(var2);
  if (tag1 == T_RESERVEDWS || tag2 == T_RESERVEDWS)
    /*{{{  */
    {
      if (tag1 == tag2)
        return 0;
      else if (tag1 == T_RESERVEDWS)
        return -1;
      else
        return 1;
    }
    /*}}}*/
  else if (tag1 == T_TEMP || tag2 == T_TEMP)
    /*{{{  */
    {
      if (tag1 == tag2)
        return 0;
      else if (tag1 == T_TEMP)
        return -1;
      else
        return 1;
    }
    /*}}}*/
  else if (NScopeOf(var1) > NScopeOf(var2))
    return -1;
  else if (NScopeOf(var1) == NScopeOf(var2))
    return 0;
  else
    return 1;
}
/*}}}*/
/*{{{  PUBLIC int compare_use (v1, v2) ** passed as a parameter*/
/*{{{  comment*/
/* Compare two variables c1 and c2,
   The ordering used to compare vars is :

   T_RESERVEDWS <  all other vars

   all other vars are then compared on use count
*/
/*}}}*/
PRIVATEPARAM int compare_use ( int *v1 , int *v2 )
{
  treenode  *var1 = var_list[*v1];
  treenode  *var2 = var_list[*v2];
  int tag1 = TagOf(var1); int tag2 = TagOf(var2);
  if (tag1 == T_RESERVEDWS || tag2 == T_RESERVEDWS)
    /*{{{  */
    {
      if (tag1 == tag2)
        return 0;
      else if (tag1 == T_RESERVEDWS)
        return -1;
      else
        return 1;
    }
    /*}}}*/
  else if (NVUseCountOf(var1) > NVUseCountOf(var2))
    return -1;
  else if (NVUseCountOf(var1) == NVUseCountOf(var2))
    #if 0
    /*{{{  compare clashes*/
    {
      if (NVClashCountOf(var1) < NVClashCountOf(var2))
        return -1;
      else if (NVClashCountOf(var1) == NVClashCountOf(var2))
        return 0;
      else
        return 1;
    }
    /*}}}*/
    #endif
    /*return 0;*/
    return compare_scope(v1, v2);
  else
    return 1;
}
/*}}}*/
#if 0  /* never used */
/*{{{  PUBLIC int compare_clash (v1, v2) ** passed as a parameter*/
/*{{{  comment*/
/* Compare two variables c1 and c2,
   The ordering used to compare vars is :

   T_RESERVEDWS <  all other vars

   all other vars are then compared on use count
*/
/*}}}*/
PRIVATE int compare_clash ( int *v1 , int *v2 )
{
  treenode  *var1 = var_list[*v1];
  treenode  *var2 = var_list[*v2];
  int tag1 = TagOf(var1); int tag2 = TagOf(var2);
  if (tag1 == T_RESERVEDWS || tag2 == T_RESERVEDWS)
    /*{{{  */
    {
      if (tag1 == tag2)
        return 0;
      else if (tag1 == T_RESERVEDWS)
        return -1;
      else
        return 1;
    }
    /*}}}*/
  else if (NVClashCountOf(var1) < NVClashCountOf(var2))
    return -1;
  else if (NVClashCountOf(var1) == NVClashCountOf(var2))
    /*{{{  compare use count*/
    {
      if (NVUseCountOf(var1) > NVUseCountOf(var2))
        return -1;
      else if (NVUseCountOf(var1) == NVUseCountOf(var2))
        return 0;
      else
        return 1;
    }
    /*}}}*/
  else
    return 1;
}
/*}}}*/
#endif /* 0 */
#if 0 /* never used */
/*{{{  PUBLIC int compare_offset (v1, v2) ** passed as a parameter*/
/*{{{  comment*/
/* Compare two variables c1 and c2,
   The ordering used to compare vars is :

   T_RESERVEDWS <  all other vars

   all other vars are then compared on use count
*/
/*}}}*/
PRIVATEPARAM int compare_offset ( int *v1 , int *v2 )
{
  treenode  *var1 = var_list[*v1];
  treenode  *var2 = var_list[*v2];
  if (NVOffsetOf(var1) < NVOffsetOf(var2))
    return -1;
  else if (NVOffsetOf(var1) == NVOffsetOf(var2))
    return 0;
  else
    return 1;
}
/*}}}*/
#endif /* 0 */
/*}}}*/
/*}}}*/
/*{{{  PRIVATE void printname(nptr)*/
PRIVATE void printname ( treenode *nptr )
{
  if (TagOf(nptr) == T_RESERVEDWS)
    fprintf(outfile, "$Reserved%-6ld", NVVarNumOf(nptr));
  else if (TagOf(nptr) == T_TEMP)
    fprintf(outfile, "$temp%-10ld", NVVarNumOf(nptr));
  else
    fprintf(outfile, "%-15s", WNameOf(NNameOf(nptr)));
}
/*}}}*/
#if 0 /* never used */
/*{{{  PRIVATE void copyrow (dest, source)*/
/* Copy the contents of source to dest */
PRIVATE void copyrow ( int *dest , int *source )
{
  memcpy (dest, source, sizeof(int) * VARROW);
}
/*}}}*/
#endif
/*{{{  PUBLIC int *save_scope()*/
/* Copy the contents of source to dest */
PUBLIC int *save_scope ( void )
{
  int *scope = (int *)memalloc(sizeof(int) * VARROW);
  memcpy (scope, live_var, sizeof(int) * VARROW);
  return scope;
}
/*}}}*/
/*{{{  PUBLIC void restore_scope(scope)*/
/* Copy the contents of source to dest */
PUBLIC void restore_scope ( int *scope )
{
  memcpy (live_var, scope, sizeof(int) * VARROW);
  memfree(scope);
}
/*}}}*/
/*{{{  PRIVATE void initrow (row)*/
/* Set all elements of row zero */
PRIVATE void initrow ( int *row )
{
  int i;
  for (i = 0; i < VARROW; i ++)
    row[i] = 0;
}
/*}}}*/
/*{{{  PRIVATE int numwords_of()*/
PRIVATE int numwords_of(const int n)
{
  const int i = startofrow(n);
  int numwords = wordnum(i);
  if (bitnum(i) > 0)
    numwords++;
  return numwords;
}
/*}}}*/
/*{{{  PRIVATE void init_varmap(map, n)*/
PRIVATE void init_varmap ( int map[], const int n )
{
  const int numwords = numwords_of(n);
  int i;
  for (i = 0; i < numwords; i++)
    map[i] = 0;
}
/*}}}*/
/*{{{  PUBLIC void init_mapping()*/
PUBLIC void init_mapping ( void )
{
  initrow(live_var);
  init_varmap(var_map, MAXVAR);
}
/*}}}*/
/*{{{  PRIVATE void alloc_varmap (n)*/
/* allocate new bitmap */
PRIVATE int *new_varmap ( const int n )
{
  /* n is never small enough for numwords_of(n) to be zero */
  int *var_map = (int *)memalloc(sizeof(int) * numwords_of(n));

  init_varmap(var_map, n);
  return var_map;
}
/*}}}*/
/*{{{  PRIVATE void printbitmap()*/
PRIVATE void printbitmap ( int sorted_var[] )
{
  int i, j;
  fputs ("********** Variable allocation map **********\n", outfile);
  fputs ("      WS SIZE USAGE                ", outfile);
  for (i = var_base; i < num_var; i++) fputc('0'+(i%10), outfile);
  fputc('\n',outfile);
  for (i = var_base; i < num_var; i++)
    {
      const int varnum = sorted_var[i];
      treenode *nptr = var_list[varnum];
      if (NVOffsetOf(nptr) == NO_SLOT)
        fprintf(outfile,"%-3d:NONE %3d %4ld : ", i,
                numslots(nptr), NVUseCountOf(nptr));
      else
        fprintf(outfile,"%-3d:%4ld %3d %4ld : ", i,
                NVOffsetOf(nptr), numslots(nptr), NVUseCountOf(nptr));
      printname(nptr);
      for (j=var_base; j < num_var; j++)
        fputs(testbitmap(var_map, varnum, sorted_var[j]) ? "*" : ".", outfile);
      fputc('\n', outfile);
    }
  fprintf(outfile, "*********************************************\n");
}
/*}}}*/
/*{{{  PUBLIC void create_var (nptr)*/
/* Make a new variable, and mark all live variables */
PUBLIC void create_var ( treenode *nptr )
{
  const int n = nextvarnum();
  setbit (live_var, n);
  markbitmap(var_map, live_var, var_base, num_var-1, n);
  SetNVVarNum(nptr, n);
  SetNVOffset(nptr, NO_SLOT); /* bug 1135 31/1/91 */
  if (TagOf(nptr) != T_RESERVEDWS)
    SetNVNext(nptr, NULL); /* Clear list of overlayed vars */
  var_list [n] = nptr;
  DEBUG_MSG(("create_var: %s, n=%d, RESERVED?=%d\n",
             WNameOf(NNameOf(nptr)), n, (TagOf(nptr) == T_RESERVEDWS)));
}
/*}}}*/
/*{{{  PUBLIC void create_var_subsidiary*/
PUBLIC void create_var_subsidiary (treenode *nptr)
{
  /* This creates the vectorspace and other temporaries required for this var */
  if (isinvectorspace(nptr))
    SetNVVSOffset(nptr, newvs(allocsize(nptr)));
  if (complexinitialise(NTypeOf(nptr)))
    /* Pointer and count held in workspace */
    reservelowworkspace(2);
}
/*}}}*/
/*{{{  PUBLIC treenode *get_var()*/
PUBLIC treenode *get_var(const int n)
{
  return var_list[n];
}
/*}}}*/
/*{{{  PUBLIC void kill_var (n)*/
/* variable no longer in use */
PUBLIC void kill_var ( treenode *nptr )
{
  const int n = NVVarNumOf(nptr);
  DEBUG_MSG(("kill_var: n=%d\n", n));
  clearbit(live_var, n);
}
/*}}}*/
/*{{{  PUBLIC void resurrect_var(nptr)*/
/* Bring a variable back from the dead */
PUBLIC void resurrect_var ( treenode *nptr )
{
  int n = NVVarNumOf(nptr);
  DEBUG_MSG(("resurrect_var: n=%d, RESERVED?=%d\n", n, (TagOf(nptr) == T_RESERVEDWS)));
  if (!testbit(live_var, n))
    {
      DEBUG_MSG(("resurrect_var: testbit was false\n"));
      setbit(live_var, n);
      markbitmap(var_map, live_var, var_base, num_var-1, n);
      if (TagOf(nptr) == N_DECL)
        create_var_subsidiary(nptr);  /* resurrect the vectorspace etc */
    }
}
/*}}}*/
/*{{{  PUBLIC treenode *resurrect_specs(tptr)*/
/* ressurect list of specs returning a pointer to the process
* following the specs.
*/
PUBLIC treenode *resurrect_specs ( treenode *tptr )
{
  while (isspecification(tptr))
    /*{{{  resurect vars*/
    {
      switch (TagOf(tptr))
        {
          case S_DECL: case S_ABBR: case S_VALABBR:
          case S_RETYPE: case S_VALRETYPE :
            {
              treenode *n = DNameOf(tptr);
              if (TagOf(n) == S_LIST)
                while (!EndOfList(n))
                  {
                    resurrect_var(ThisItem(n));
                    n = NextItem(n);
                  }
              else
                resurrect_var(n);
            }
            break;
        }
      tptr = DBodyOf(tptr);
    }
    /*}}}*/
  return tptr;
}
/*}}}*/
/*{{{  PUBLIC int create_immortalvar (nptr)*/
/* Create a new variable which is live for all varibles */
PUBLIC int create_immortalvar ( treenode *nptr )
{
  int n = nextvarnum(), i;
  /* mark all those in scope as interfering with the new var */
  for (i = var_base; i<=n; i++)
    setbitmap(var_map, i, n);
  var_list [n] = nptr;
  SetNVVarNum(nptr, n);
  SetNVNext(nptr, NULL); /* Clear list of overlayed vars */
  return n;
}
/*}}}*/

/*{{{  PUBLIC treenode *alloc_fixedws(slot, size)*/
/*{{{  comment*/
/* Reserve workspace from slot to (slot+size) - 1.
* Mark all live variables as interfering with this area of workspace
*
* We keep a list of nodes already created in the past.
*/
/*}}}*/
PUBLIC treenode *alloc_fixedws ( int slot , int size )
{
  /* Look down the list of slots for one which is
     the same start, the same size, not allocated at all or
     is allocated within our block.
  */
  treenode *nptr = fixedws_list;
  DEBUG_MSG(("alloc_fixedws; slot:%d, size:%d\n", slot, size));
  {
    int i;
    for (i = var_base; i < num_var; i++)
      {
        treenode *ptr = var_list[i];
        if (testbit(live_var, i) && (TagOf(ptr) == T_RESERVEDWS))
          {
            int ov = vars_overlap(slot, size, NVOffsetOf(ptr), NVRSizeOf(ptr));
            DEBUG_MSG(("alloc_fixedws: Live with reserved %d, clashes? %d\n", i, ov));
            if (ov)
              genwarn_i(GEN_WORKSPACE_CLASH, NVOffsetOf(ptr));
          }
      }
  }
  while ((nptr!=NULL) &&
         !((NVOffsetOf(nptr)==slot && NVRSizeOf(nptr)==size) &&
           ((NVVarNumOf(nptr) == -1) ||
            (NVVarNumOf(nptr) >= var_base && NVVarNumOf(nptr) < num_var))))
    nptr = NVNextOf(nptr);
  if (nptr == NULL)
    /*{{{  Make new fixed workspace node*/
    {
      switch_to_real_workspace();
      nptr = newnamenode(T_RESERVEDWS, genlocn, tempname_p, NULL,
                         NULL, lexlevel, (BIT16)(-1), NM_DEFAULT);
      SetNVRSize(nptr, size);
      SetNVNext(nptr, fixedws_list);  fixedws_list = nptr;
      create_var(nptr);  /* Make a new var */
      DEBUG_MSG(("alloc_fixedws; creating new node\n"));
    }
    /*}}}*/
  else
    /*{{{  Use existing record*/
    {
      DEBUG_MSG(("alloc_fixedws; resurrecting?:%d\n", NVVarNumOf(nptr) != -1));
      if (NVVarNumOf(nptr) == -1)
        create_var(nptr); /* Not used in this block yet */
      else
        resurrect_var(nptr);    /* re-use this node */
    }
    /*}}}*/
  SetNVOffset(nptr, slot); /* leave this set correctly */
  return nptr;
}
/*}}}*/
/*{{{  PUBLIC void free_fixedwslist()*/
PUBLIC void free_fixedwslist ( void )
{
  /* Mark fixed workspace node as being unused */
  int i;
  DEBUG_MSG(("free_fixedwslist:\n"));
  for (i = var_base; i < num_var; i++)
    if (TagOf(var_list[i]) == T_RESERVEDWS)
      SetNVVarNum(var_list[i], -1);
}
/*}}}*/

/*{{{  PUBLIC void offsetvars (first_var, offset)*/
/* Add offset to all vars in the var list from first_var onwards */
PUBLIC void offsetvars ( int first_var , int offset )
{
  int i;
  for (i = first_var; i < num_var; i++)
    {
      treenode *var = var_list[i];
      if (TagOf(var) != T_RESERVEDWS)
        while (var != NULL)
          {
            SetNVOffset(var, NVOffsetOf(var) + offset);
            if (diagnostics)
              /*{{{  print some diagnostics*/
              {
                printname(var);
                fprintf (outfile, " moved to workspace offset %ld\n", NVOffsetOf(var));
              }
              /*}}}*/
            var = NVNextOf(var);
          }
    }
}
/*}}}*/
/*{{{  PUBLIC void addnamechain (chain, item)*/
/* Add a new name onto a names namechain */
PUBLIC void addnamechain ( treenode *chain , treenode *item )
{
  SetNVNext(item, NVNextOf(chain));
  SetNVNext(chain, item);
  SetNVVarNum(item, NVVarNumOf(chain));
}
/*}}}*/
/*{{{  PUBLIC void reservelowworkspace (n)*/
/*****************************************************************************
 *
 *  reservelowworkspace reserves the bottom n slots of workspace.
 *  only those variable currently live will be effected.
 *
 *****************************************************************************/
PUBLIC void reservelowworkspace ( INT32 n )
{
  kill_var(alloc_fixedws(0, n));
}
/*}}}*/

/*{{{  PRIVATE void printdiagnostic*/
PRIVATE void printdiagnostic(treenode *var, int show_ws, int mapped_onto_vsptr)
{
  printname(var);
  if (show_ws)
    fprintf (outfile, " at workspace offset %ld", NVOffsetOf(var));
  else if (mapped_onto_vsptr)
    fputs(" mapped onto vsptr", outfile);
  if (isinvectorspace(var))
    {
      INT32 vsoffset = NVVSOffsetOf(var);
      fprintf(outfile, "  vector space [%ld..%ld]",
              vsoffset, vsoffset + allocsize(var) - 1);
    }
  fputc('\n', outfile);
}
/*}}}*/
/*{{{  PRIVATE void allocate_overlays*/
PRIVATE void allocate_overlays(treenode *nptr, int base)
{
  while (nptr != NULL)
    {
      SetNVOffset(nptr, base + NVOffsetOf(nptr));
      if (diagnostics && (TagOf(nptr) != T_TEMP))
        {
          fputs(" - ", outfile);
          printdiagnostic(nptr, TRUE, FALSE);
        }
      nptr = NVNextOf(nptr);
    }
}
/*}}}*/
/*{{{  PUBLIC void initallocvars*/
PUBLIC void initallocvars(void)
{
  vars_mapped_onto_vsptr = NULL; /* added for bug 1061 30/11/90 */
}
/*}}}*/
/*{{{  PUBLIC int allocvars (first_var, free_paramslots)*/
/* Allocate the vars using bit map built up during first pass.
* Only first_var and above are allocated.
* Returns size of workspace allocated.
* If free_paramslots > 0 then they will be used for least used vars
* which will fit.
*/
PUBLIC int allocvars ( int first_var, int free_paramslots, int inside_par )
{
  int sorted_var[MAXVAR];   /* sorted order of vars */
  int wssize = 0;
  /*{{{  debug message*/
  if (diagnostics)
    fprintf(outfile, "Allocating vars from %d upto %d\n", var_base, num_var);
  /*}}}*/
  if (num_var > first_var)
    {
      int i;
      treenode *alloclist = NULL;
      /*{{{  initialise names*/
      for (i=first_var; i < num_var; i++)
        {
          if (TagOf(var_list[i]) != T_RESERVEDWS)
            {
               treenode *next;
               INT32 total = 0;
               const int size = numslots(var_list[i]);
               for (next = var_list[i]; next != NULL; next = NVNextOf(next))
                 total += NVUseCountOf(next);
               /* IF the object is greater than a word long (changed to 2 words)
                  then we scale down the usage count:
               */
               if (((alloc_strategy & ALLOC_NODIVBYSIZE) == 0) && (size > /*1*/ 2))
                 {
                   /* bug 1153 - 11/2/91 - take the sizes of objects into account */
                   total /= size;
                   if (total == 0) total = 1; /* in case of `underflow' */
                 }
               SetNVUseCount(var_list[i], total);
               SetNVOffset(var_list[i], NO_SLOT);
               /* SetNVClashCount(var_list[i], clashcount(first_var, i)); */
            }
          sorted_var[i] = i;
        }
      /*}}}*/
      /*{{{  sort names*/
      {
        const int size = num_var-first_var;
        if (alloc_strategy & ALLOC_BYSCOPE)
          local_qsort(&sorted_var[first_var], size, sizeof(int), (int(*)())compare_scope);
        else
          {
            local_qsort(&sorted_var[first_var],size,sizeof(int), (int(*)())compare_use);
          #if 0 /* now done after allocating the first 'n' variables */
            if (size > (SORT_USAGE_CUTOFF+1))
              local_qsort(&sorted_var[first_var+SORT_USAGE_CUTOFF],size-SORT_USAGE_CUTOFF,sizeof(int),
                          (int(*)())compare_scope);
          #endif
          }
      }
      /*}}}*/
      /*{{{  allocate vars*/
      {
        /* alloclist is the list of allocated variables kept in the order:
         *     (NVOffsetOf(n) < NVOffset(NVAllocNextOf(n))) ||
         *     ((NVOffsetOf(n) == NVOffset(NVAllocNextOf(n)) &&
         *       ((numslots(n) >= numslots(NVAllocNextOf(n)))
         */
        int re_sorted = FALSE;
        for (i=first_var; i < num_var; i++)
          /*{{{  allocate next var*/
          {
            int varnum = sorted_var[i];
            treenode *var = var_list[varnum];
            int length = numslots(var);
            if (length == 0 || NVUseCountOf(var) == 0)
              {
                if (diagnostics)
                  { printname(var); fputs(" not allocated\n", outfile); }
              }
            else if (!inside_par && ((alloc_strategy & ALLOC_NOVSOPT) == 0) &&
                     isinvectorspace(var) && (NVVSOffsetOf(var) == 0))
              {
                SetNVAllocNext(var, vars_mapped_onto_vsptr);
                vars_mapped_onto_vsptr = var;
                if (diagnostics)
                  printdiagnostic(var, FALSE, TRUE);
              }
            else
              {
                int base = 0;
                treenode *this = alloclist, *last = NULL;
                if (TagOf(var) != T_RESERVEDWS && NVOffsetOf(var) == NO_SLOT)
                  /*{{{  allocate var*/
                  {
                    int slotfound = FALSE;
                    /* Only look for overlap with vars already allocated */
                    while ((this != NULL) && !slotfound)
                      /*{{{  look for ws slot*/
                      {
                        /* Base is the next guess at the position to allocate variable */
                        int end = base + length;
                        if (end <= NVOffsetOf(this)) /* bug 709 changed to '<=' from '<' 14/9/90 */
                          slotfound = TRUE; /* position found */
                        else
                          {
                            treenode *firstclash = this;
                            /*{{{  look for end of gap*/
                            while ((firstclash != NULL) &&
                                    !testbitmap(var_map, varnum, NVVarNumOf(firstclash)))
                              firstclash = NVAllocNextOf(firstclash);
                            /*}}}*/
                            if ((firstclash == NULL) ||
                                (end <= NVOffsetOf(firstclash))) /* bug 709 changed to '<=' from '<' 14/9/90 */
                              slotfound = TRUE;
                            else
                              /*{{{  look for start of next gap*/
                              {
                                base = NVOffsetOf(firstclash) + numslots(firstclash);
                                last = firstclash;
                                this = NVAllocNextOf(firstclash);
                                while ((this != NULL) && (NVOffsetOf(this) < base))
                                  {
                                    int thisend = NVOffsetOf(this) + numslots(this);
                                    if (testbitmap(var_map, varnum, NVVarNumOf(this)) &&
                                        (base < thisend))
                                      base = thisend;
                                    last = this;
                                    this = NVAllocNextOf(this);
                                  }
                              }
                              /*}}}*/
                          }
                      }
                    /*{{{  Allocate Offset for this var*/
                    SetNVOffset(var, base);
                    if (diagnostics)
                      printdiagnostic(var, TRUE, FALSE);
                    allocate_overlays(NVNextOf(var), base);
                    /*}}}*/
                  }
                  /*}}}*/
                else
                  /*{{{  find position in alloc list where this should go*/
                  {
                    while ((this != NULL) && (NVOffsetOf(this) < NVOffsetOf(var)))
                      {
                        last = this;
                        this = NVAllocNextOf(this);
                      }
                    base = NVOffsetOf(var);
                    if (diagnostics)
                      printdiagnostic(var, TRUE, FALSE);
                  }
                  /*}}}*/
                wssize = max (wssize, base+length);
                /*{{{  Add var to alloc list*/
                /* Assume :
                 *   (last == NULL || NVOffsetOf(last) < base)
                 *   (this == NULL || NVOffsetOf(this) >= base)
                 */
                while ((this != NULL) &&
                       ((NVOffsetOf(this) == base) && (numslots(this) > length)))
                  {
                    last = this;
                    this = NVAllocNextOf(this);
                  }
                if (last == NULL)
                  {                                           /* insert on front    */
                    SetNVAllocNext(var, alloclist);
                    alloclist = var;
                  }
                else
                  {
                    SetNVAllocNext(last, var);
                    SetNVAllocNext(var, this);                /* insert after last  */
                  }
                /*}}}*/
                /*{{{  debugging */
                #if 0 /*#ifdef DEBUG*/
                {
                  treenode *tptr = alloclist;
                  fputs("alloclist: ", outfile);
                  while (tptr != NULL)
                    {
                      fprintf(outfile, "%s @:%ld-%ld, ",
                       (TagOf(tptr) == T_RESERVEDWS) ? "reserved" : WNameOf(NNameOf(tptr)),
                       NVOffsetOf(tptr), NVOffsetOf(tptr) + numslots(tptr) - 1);
                      tptr = NVAllocNextOf(tptr);
                    }
                  fputc('\n', outfile);
                }
                #endif
                /*}}}*/
                if (!re_sorted && (wssize >= RESORT_SLOTS_CUTOFF) && (i < num_var - 2))
                  {
                    if (diagnostics)
                      fprintf(outfile, "re-sorting %d variables by scope\n", num_var-(i+1));
                    if ((alloc_strategy & ALLOC_BYSCOPE) == 0)
                      local_qsort(&sorted_var[i+1],num_var-(i+1),sizeof(int),
                                  (int(*)())compare_scope);
                    re_sorted = TRUE;
                  }
              }
          }
          /*}}}*/
      }
      /*}}}*/
      /*{{{  try and squeeze top variables into parameter slots*/
      if (free_paramslots > 0)
        {
          treenode *movelist[MAXREGS];
          int free_slot = free_paramslots-1, move_vars = TRUE;
          while (free_slot > 0 && move_vars && wssize > 0)
            {
              int j = first_var;
              movelist[free_slot] = NULL;
              while (j < num_var && move_vars)
                /*{{{  check we can move all vars at top of workspace*/
                {
                  treenode *var = var_list[j];
                  int length = numslots(var);
                  if ((NVOffsetOf(var) + length) == wssize)
                    {
                      if ((TagOf(var) == T_RESERVEDWS) || (length > (free_slot+1)))
                        move_vars = FALSE; /* No point in doing more moves */
                      else
                        {
                          SetNVAllocNext(var, movelist[free_slot]);
                          movelist[free_slot] = var;
                        }
                    }
                  j++;
                }
                /*}}}*/
              if (move_vars) /* We can move top variables into unused param slot */
                {
                  free_slot--;
                  wssize--;
                }
            }
          /*{{{  Allocate to vars to their new slots*/
          {
            int top;
            free_slot++;
            top = wssize + INS_EXTRA + (REG_PARAMS-free_paramslots) + free_slot;
            while (free_slot < free_paramslots)
              {
                treenode *var = movelist[free_slot];
                while (var != NULL)
                  {
                    int base = top - (numslots(var)-1);
                    /*{{{  Allocate Offset for this var*/
                    SetNVOffset(var, base);
                    if (diagnostics)
                      /*{{{  print some diagnostics*/
                      {
                        printname(var);
                        fprintf (outfile, " moved to param slot, ws offset = %ld", NVOffsetOf(var));
                        fputc('\n', outfile);
                      }
                      /*}}}*/
                    /*}}}*/
                    /*{{{  allocate offsets for overlayed vars*/
                    {
                      treenode *nextvar = NVNextOf(var);
                      while (nextvar != NULL)
                        {
                          SetNVOffset(nextvar, base);
                          if (diagnostics)
                            /*{{{  print some diagnostics*/
                            {
                              fputs(" + ", outfile);
                              printname(nextvar);
                              fputc('\n', outfile);
                            }
                            /*}}}*/
                          nextvar = NVNextOf(nextvar);
                        }
                    }
                    /*}}}*/
                    var = NVAllocNextOf(var);
                  }
                top++;
                free_slot++;
              }
          }
          /*}}}*/
        }
      /*}}}*/
      /*{{{  debug message*/
      if (diagnostics)
        printbitmap(sorted_var);
      /*}}}*/
      /*{{{  check allocation -- commented out*/
      #if 0
      {
        int i,j;
        for (i=first_var; i < num_var; i++)
        {
          treenode *v1 = var_list[sorted_var[i]];
          for (j=first_var; j < num_var; j++)
            if (i != j)
            {
              treenode *v2 = var_list[sorted_var[j]];
              if (TagOf(v1) == T_RESERVEDWS && TagOf(v2) == T_RESERVEDWS)
                ;
              else if (NVOffsetOf(v1) == NO_SLOT || NVOffsetOf(v2) == NO_SLOT)
                ;
              else if (testbitmap(var_map, sorted_var[i], sorted_var[j]) &&
                  vars_overlap (NVOffsetOf(v1), numslots(v1),
                                NVOffsetOf(v2), numslots(v2)))
              {
                fprintf(outfile, "(%d)%s, (%d)%s\n",
                        i, WNameOf(NNameOf(v1)), j, WNameOf(NNameOf(v2)));
                fflush(outfile);
                badtag(genlocn,TagOf(v1),
                          "allocvars, var allocation failed");
              }
            }
        }
      }
      #endif
      /*}}}*/
    }
  return (wssize);
}
/*}}}*/
/*}}}*/
/*{{{  label allocation*/
/*{{{  PUBLIC int newlab ()*/
PUBLIC int newlab ( void )
{
  return (nextlabel++);
}
/*}}}*/
/*{{{  PUBLIC void initlabels()*/
PUBLIC void initlabels ( void )
{
  nextlabel = 0;
}
/*}}}*/
/*}}}*/
/*{{{  use count handling*/
#define HUGEVAL ((INT32)0x7fffffff)
/*{{{  PUBLIC void addusecount(nptr, n)*/
/* up the use count of a variable by n, where n has already been loop weighted
 * If the use count would overflow it is set to HUGEVAL
 */
PUBLIC void addusecount ( treenode *nptr , INT32 n )
{
  const INT32 oldcount = NVUseCountOf(nptr);
  if (n > (HUGEVAL - oldcount))
    SetNVUseCount(nptr, HUGEVAL);
  else
    SetNVUseCount(nptr, oldcount + n);
}
/*}}}*/
/*{{{  PUBLIC void upusecount(nptr, n)*/
/* up the use count of a variable n times.
 * If the use count would overflow it is set to HUGEVAL
 */
PUBLIC void upusecount ( treenode *nptr , INT32 n )
{
  if (loop_weight == 0)
    ; /* skip */
  else if (n > (HUGEVAL / loop_weight))
    SetNVUseCount(nptr, HUGEVAL);
  else
    addusecount(nptr, loop_weight * n);
}
/*}}}*/
/*{{{  PUBLIC void uploop_weight(n)*/
/* up loop_weight n times
 * If the loop_weight would overflow it is set to HUGEVAL
 */
PUBLIC void uploop_weight ( INT32 n )
{
  if (n == 0 || loop_weight == 0)
    loop_weight = 0;
  else if (n > (HUGEVAL / loop_weight))
    loop_weight = HUGEVAL;
  else
    loop_weight *= n;
}
/*}}}*/
/*}}}*/
/*{{{  temporary slot allocation*/
/*{{{  comment*/
/*
 *  To allocate a temporary we create a new temporary node then look through
 *  the current variable table looking for a temporary of the same size
 *  which is unused.
 *  If an unused temporary is found we add the new temporary node  to the
 *    variable chain of the unused temporary. This gives the new temporary
 *    the same variable number as the unused temporary so they will share the
 *    same slot in workspace .
 *    The unused temporary is then resurrected and a pointer to the new
 *    temporary node returned.
 *  If an unused temporary is not found the new temporary node is added
 *    to the variable table.
 *
 * To free a temporary we simply kill the entry in the variable table given by
 * the tempories variable number.
 */
/*}}}*/

/*{{{  PUBLIC void alloctemp(treenode *tempptr)*/
PUBLIC void alloctemp ( treenode *tempptr )
{
  int i = var_base;
  /*INT32 size = wordsin(NTypeOf(tempptr));*/
  const INT32 size = numslots(tempptr);
  /* look for an unused temp of the same size */
  while ((i < num_var) &&
         (TagOf(var_list[i]) != T_TEMP ||
          testbit(live_var, i) != 0 || size != numslots(var_list[i]) ))
    i++;
  if (i < num_var)
    {
      /* Add the new temp node to the unused tempories variable chain
       * and resurrect the old temporary
       */
      DEBUG_MSG(("alloctemp: Re-using temp%d\n", i));
      addnamechain(var_list[i], tempptr);
      SetNVOffset(tempptr, 0);
      resurrect_var(var_list[i]);
    }
  else
    {
      /* Allocate a new variable */
      DEBUG_MSG(("alloctemp: New temp%d\n", i));
      create_var(tempptr);
    }
}
/*}}}*/
/*{{{  PUBLIC treenode *newtypedtempnode(t, tptr, mode, type)*/
PUBLIC treenode *newtypedtempnode ( int t , treenode *tptr , int mode , treenode *type )
{
  treenode *tempptr;
  DEBUG_MSG(("newtypedtempnode\n"));
  tempptr = newnamenode(t, genlocn, tempname_p, type, tptr,
                                  lexlevel, 0, mode);
  SetNVOffset(tempptr, NO_SLOT);
  SetNVNext(tempptr, NULL);
  SetNVUseCount(tempptr, 1); /* make sure it always gets a slot allocated */
  return(tempptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *newtempnode(t, tptr, mode)*/
PUBLIC treenode *newtempnode ( int t , treenode *tptr , int mode )
{
  /* We call gettype_main so that any type tree fabricated is created
     in real workspace, not temporary workspace */
  return (newtypedtempnode(t, tptr, mode, gettype_main(tptr)));
}
/*}}}*/
/*{{{  PUBLIC treenode *gettemp(tptr, mode)*/
PUBLIC treenode *gettemp ( treenode *tptr , int mode )
{
  treenode *tempptr = newtempnode(T_TEMP, tptr, mode);
  alloctemp(tempptr);
  return(tempptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *gettypedtemp(tptr, mode, type)*/
PUBLIC treenode *gettypedtemp ( treenode *tptr , int mode , treenode *type )
{
  treenode *tempptr = newtypedtempnode(T_TEMP, tptr, mode, type);
  alloctemp(tempptr);
  return(tempptr);
}
/*}}}*/
/*{{{  PUBLIC treenode *getopdtemp(int opdmode, treenode *opd)*/
/****************************************************************************
 *
 *  getopdtemp takes an operand (opdmode, opd) and returns a suitable temp.
 *             node for it.
 *             This is a bit of a kludge as it pretends a pointer is an
 *             integer in workspace; ideally, we should have proper pointer
 *             types in the backend.
 *
 ****************************************************************************/
PUBLIC treenode *getopdtemp ( int opdmode , treenode *opd )
{
  return(gettemp(opd, (opdmode == P_PTR) ? NM_POINTER : NM_WORKSPACE));
}
/*}}}*/
/*{{{  PUBLIC void freetemp(temp)*/
PUBLIC void freetemp ( treenode *temp )
{
  kill_var(temp);
}
/*}}}*/
/*{{{  PUBLIC void freeiftemp(treenode *tptr)*/
/*{{{  comment*/
/*****************************************************************************
 *
 *  freeiftemp takes a treenode 'tptr', and if it is a temporary, or
 *             evaluated temporary, frees it.
 *
 *****************************************************************************/
/*}}}*/
PUBLIC void freeiftemp ( treenode *tptr )
{
  if (TagOf(tptr) == T_TEMP || TagOf(tptr) == T_PREEVALTEMP)
    freetemp(tptr);
}
/*}}}*/
/*}}}*/
/*{{{  PUBLIC void allocparams(nptr)*/
/*****************************************************************************
 *
 *  allocparams allocates workspace offsets to the formal parameters.
 *
 *****************************************************************************/
PUBLIC void allocparams ( treenode *nptr )
{
  treenode *fparams = NParamListOf(nptr);
  /*{{{  allocate workspace positions to the parameters*/
  {
    /*int paramposn = 1;*/
    INT32 paramposn = NPMaxwspOf(nptr) + 1;
    int olddiagnostics = diagnostics;
    if (separatelycompiled(nptr)) diagnostics = FALSE;
    if (diagnostics)
      fprintf(outfile, "******* Parameter workspace allocation ******\n");
    while (!EndOfList(fparams))
      {
        treenode *thisfparam = ThisItem(fparams);
        switch(TagOf(thisfparam))
          {
            /*{{{  formal parameter*/
            case N_PARAM:
            case N_VALPARAM:
              if (basetype(gettype(thisfparam)) != S_TIMER)
                /*{{{  allocate the parameter*/
                {
                  SetNVOffset(thisfparam, paramposn);
                  if (diagnostics)
                    fprintf (outfile, "Parameter %s at workspace offset %ld, use count = %ld\n",
                             WNameOf(NNameOf(thisfparam)), NVOffsetOf(thisfparam),
                             NVUseCountOf(thisfparam));
                  allocate_overlays(NVNextOf(thisfparam), paramposn);
                }
                /*}}}*/
              else
                /* decrement paramposn to cancel out the increment below */
                paramposn--;
              break;
            /*}}}*/
            /*{{{  hidden array dimension*/
            case S_HIDDEN_PARAM:
              SetNVOffset(HExpOf(thisfparam), paramposn);
              if (diagnostics)
                fputs("Hidden param", outfile);
              break;
            /*}}}*/
            /*{{{  static link*/
            case S_PARAM_STATICLINK:
              if (diagnostics)
                fputs("Static link", outfile);
              break;
            /*}}}*/
            /*{{{  vector space pointer*/
            case S_PARAM_VSP:
              {
                treenode *nptr = vars_mapped_onto_vsptr;
                while (nptr != NULL)
                  {
                    SetNVOffset(nptr, paramposn);
                    if (diagnostics)
                      printdiagnostic(nptr, TRUE, FALSE);
                    allocate_overlays(NVNextOf(nptr), paramposn);
                    nptr = NVAllocNextOf(nptr);
                  }
                if (diagnostics)
                  fputs("Vectorspace pointer", outfile);
              }
              break;
            /*}}}*/
            /*{{{  function result pointer*/
            case S_FNFORMALRESULT:
              SetHOffset(thisfparam, paramposn);
              if (diagnostics)
                fputs("Function result", outfile);
              break;
            /*}}}*/
            default:
              badtag(genlocn, TagOf(thisfparam), "allocparams");
          }
        if (diagnostics &&
            TagOf(thisfparam) != N_PARAM && TagOf(thisfparam) != N_VALPARAM)
          fprintf(outfile, " at workspace offset %ld\n", paramposn);
        paramposn++;
        fparams = NextItem(fparams);
      }
    if (diagnostics)
      fprintf(outfile, "*********************************************\n");
    diagnostics = olddiagnostics;
  }
  /*}}}*/
}
/*}}}*/
/*{{{  PUBLIC void initalloc ()*/
PUBLIC void initalloc ( void )
{
  /* Here we have some code to check the assumptions made in the
     live variable maps. IE that sizeof(int) == 32
  */

  /* we can't #if this, because it contains a sizeof() */
#if 0
  /* we can't assert() this, cos it evaluates to assert(TRUE);
     and gcc gives a warning: statement with no effect
  */
  assert((MAXVAR/VARROW) ==  (1 << WORDSHIFT));
#else
  if    ((MAXVAR/VARROW) !=  (1 << WORDSHIFT))
    assert(FALSE);
#endif

#if     (BYTEMASK        != ((1 << WORDSHIFT) - 1))
  /* we have to #if this, cos otherwise gcc gives a warning:
     statement with no effect
  */
  assert(BYTEMASK        == ((1 << WORDSHIFT) - 1));
#endif

  staticlinkused = 0;
  initvsp(0);
  datasize = DS_MIN;
  fixedws_list = NULL;
  if (var_map == NULL) /* incase this is called more than once (in configurer) */
    var_map = new_varmap(MAXVAR);
}
/*}}}*/
#ifdef CONFIG
/*{{{  PUBLIC void freevarmap*/
PUBLIC void freevarmap(void)
{
  if (var_map != NULL) { memfree(var_map); var_map = NULL; }
}
/*}}}*/
#endif
/*}}}*/

